"""Tests for the base Serializer functionality."""

from __future__ import annotations

import pytest
from msgspec import ValidationError

from django_bolt.serializers import Serializer, field_validator, model_validator


class TestSerializerBasics:
    """Test basic Serializer functionality."""

    def test_serializer_creation(self):
        """Test creating a simple serializer."""

        class UserSerializer(Serializer):
            username: str
            email: str

        user = UserSerializer(username="alice", email="alice@example.com")
        assert user.username == "alice"
        assert user.email == "alice@example.com"

    def test_serializer_with_defaults(self):
        """Test serializer with default values."""

        class UserSerializer(Serializer):
            username: str
            email: str = "no-email@example.com"

        user = UserSerializer(username="bob")
        assert user.username == "bob"
        assert user.email == "no-email@example.com"

    def test_serializer_to_dict(self):
        """Test converting serializer to dict."""

        class UserSerializer(Serializer):
            username: str
            email: str

        user = UserSerializer(username="charlie", email="charlie@example.com")
        data = user.to_dict()

        assert data == {"username": "charlie", "email": "charlie@example.com"}


class TestFieldValidators:
    """Test field-level validation."""

    def test_field_validator_basic(self):
        """Test basic field validator."""

        class UserSerializer(Serializer):
            email: str

            @field_validator("email")
            def validate_email(cls, value):
                if "@" not in value:
                    raise ValueError("Invalid email")
                return value

        # Valid email should work
        user = UserSerializer(email="valid@example.com")
        assert user.email == "valid@example.com"

        # Invalid email should raise
        with pytest.raises(ValidationError) as exc_info:
            UserSerializer(email="invalid")
        assert "Invalid email" in str(exc_info.value)

    def test_field_validator_transformation(self):
        """Test that field validators can transform values."""

        class UserSerializer(Serializer):
            email: str

            @field_validator("email")
            def normalize_email(cls, value):
                return value.lower().strip()

        user = UserSerializer(email="  ALICE@EXAMPLE.COM  ")
        assert user.email == "alice@example.com"

    def test_multiple_field_validators(self):
        """Test multiple validators on the same field."""

        class UserSerializer(Serializer):
            password: str

            @field_validator("password")
            def check_length(cls, value):
                if len(value) < 8:
                    raise ValueError("Password too short")
                return value

            @field_validator("password")
            def check_complexity(cls, value):
                if not any(c.isupper() for c in value):
                    raise ValueError("Password must have uppercase")
                return value

        # Valid password
        user = UserSerializer(password="SecurePass123")
        assert user.password == "SecurePass123"

        # Too short
        with pytest.raises(ValidationError):
            UserSerializer(password="Short1")

        # No uppercase
        with pytest.raises(ValidationError):
            UserSerializer(password="longpassword1")


class TestModelValidators:
    """Test model-level validation."""

    def test_model_validator_basic(self):
        """Test basic model validator."""

        class PasswordSerializer(Serializer):
            password: str
            password_confirm: str

            @model_validator
            def check_passwords_match(self):
                if self.password != self.password_confirm:
                    raise ValueError("Passwords don't match")

        # Matching passwords
        data = PasswordSerializer(password="Secret123", password_confirm="Secret123")
        assert data.password == "Secret123"

        # Non-matching passwords
        with pytest.raises(ValidationError) as exc_info:
            PasswordSerializer(password="Secret123", password_confirm="Different456")
        assert "don't match" in str(exc_info.value)

    def test_model_validator_cross_field(self):
        """Test model validator with cross-field logic."""

        class RangeSerializer(Serializer):
            min_value: int
            max_value: int

            @model_validator
            def validate_range(self):
                if self.min_value >= self.max_value:
                    raise ValueError("min_value must be less than max_value")

        # Valid range
        data = RangeSerializer(min_value=1, max_value=10)
        assert data.min_value == 1
        assert data.max_value == 10

        # Invalid range
        with pytest.raises(ValidationError):
            RangeSerializer(min_value=10, max_value=1)


class TestValidatorExecution:
    """Test validator execution order and behavior."""

    def test_field_validators_run_before_model_validators(self):
        """Test that field validators run before model validators."""
        execution_order = []

        class TestSerializer(Serializer):
            value: str

            @field_validator("value")
            def field_val(cls, v):
                execution_order.append("field")
                return v

            @model_validator
            def model_val(self):
                execution_order.append("model")

        TestSerializer(value="test")

        # Field validator should run before model validator
        assert execution_order[0] == "field"
        assert execution_order[1] == "model"


class TestOptionalFields:
    """Test optional fields with None values."""

    def test_optional_field(self):
        """Test optional field handling."""

        class UserSerializer(Serializer):
            username: str
            email: str | None = None

        user = UserSerializer(username="alice")
        assert user.username == "alice"
        assert user.email is None

        user2 = UserSerializer(username="bob", email="bob@example.com")
        assert user2.email == "bob@example.com"


class TestSerializerInheritance:
    """Test Serializer inheritance and composition."""

    def test_serializer_inheritance(self):
        """Test inheriting from a base serializer."""

        class BaseUserSerializer(Serializer):
            username: str
            email: str

            @field_validator("email")
            def validate_email(cls, value):
                if "@" not in value:
                    raise ValueError("Invalid email")
                return value

        class AdminSerializer(BaseUserSerializer):
            is_admin: bool = False

        # Child serializer should inherit validators
        with pytest.raises(ValidationError):
            AdminSerializer(username="alice", email="invalid", is_admin=True)

        # Valid data should work
        admin = AdminSerializer(username="alice", email="alice@example.com", is_admin=True)
        assert admin.is_admin is True


class TestMetaConfig:
    """Test Meta class configuration."""

    def test_meta_model_reference(self):
        """Test Meta.model reference."""

        class DummyModel:
            pass

        class UserSerializer(Serializer):
            username: str

            class Meta:
                model = DummyModel

        assert UserSerializer.Meta.model == DummyModel
