/*
 * Decompiled with CFR 0.152.
 */
package neqsim.process.equipment.flare;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import neqsim.process.equipment.ProcessEquipmentBaseClass;
import neqsim.process.equipment.stream.StreamInterface;
import neqsim.thermo.component.ComponentInterface;
import neqsim.thermo.system.SystemInterface;

public class FlareStack
extends ProcessEquipmentBaseClass
implements Serializable {
    private static final long serialVersionUID = 1L;
    private StreamInterface reliefInlet;
    private StreamInterface airAssist;
    private StreamInterface steamAssist;
    private double tipDiameter_m = 0.3;
    private double tipElevation_m = 40.0;
    private double ambientTempK = 288.15;
    private double ambientPressBar = 1.01325;
    private double windSpeed10m = 2.0;
    private double burningEfficiency = 0.985;
    private double radiantFraction = 0.18;
    private double so2Conversion = 1.0;
    private double unburnedTHCFraction = 0.005;
    private double coFraction = 0.003;
    private double excessAirFrac = 0.0;
    private double tipLossK = 1.0;
    private double tipBackpressureBar = 1.03;
    private RadiationModel radiationModel = RadiationModel.POINT_SOURCE;
    private double ch_epC_kWm2 = 160.0;
    private double ch_epA = 0.12;
    private double ch_lfC = 12.0;
    private double ch_lfA = 0.45;
    private double ch_lfB = 0.2;
    private double ch_kTilt = 1.3;
    private double ch_kAtten = 0.008;
    private int ch_lineSegments = 30;
    private double heatReleaseMW = 0.0;
    private Map<String, Double> emissionsKgPerHr = new HashMap<String, Double>();

    public FlareStack(String name) {
        super(name);
    }

    public void setReliefInlet(StreamInterface s) {
        this.reliefInlet = s;
    }

    public void setAirAssist(StreamInterface s) {
        this.airAssist = s;
    }

    public void setSteamAssist(StreamInterface s) {
        this.steamAssist = s;
    }

    public void setTipDiameter(double m) {
        this.tipDiameter_m = m;
    }

    public void setTipElevation(double m) {
        this.tipElevation_m = m;
    }

    public void setBurningEfficiency(double eff) {
        this.burningEfficiency = FlareStack.clamp01(eff);
    }

    public void setRadiantFraction(double f) {
        this.radiantFraction = FlareStack.clamp01(f);
    }

    public void setSO2Conversion(double f) {
        this.so2Conversion = FlareStack.clamp01(f);
    }

    public void setUnburnedTHCFraction(double f) {
        this.unburnedTHCFraction = Math.max(0.0, f);
    }

    public void setCOFraction(double f) {
        this.coFraction = Math.max(0.0, f);
    }

    public void setExcessAirFrac(double f) {
        this.excessAirFrac = Math.max(0.0, f);
    }

    public void setAmbient(double tempK, double pressBar) {
        this.ambientTempK = tempK;
        this.ambientPressBar = pressBar;
    }

    public void setWindSpeed10m(double u) {
        this.windSpeed10m = Math.max(0.0, u);
    }

    public void setTipLossK(double k) {
        this.tipLossK = Math.max(0.0, k);
    }

    public void setRadiationModel(RadiationModel m) {
        this.radiationModel = m;
    }

    public void setChamberlainEmissivePower(double epC_kWm2, double epA) {
        this.ch_epC_kWm2 = epC_kWm2;
        this.ch_epA = epA;
    }

    public void setChamberlainFlameLength(double lfC, double lfA, double lfB) {
        this.ch_lfC = lfC;
        this.ch_lfA = lfA;
        this.ch_lfB = lfB;
    }

    public void setChamberlainTilt(double kTilt) {
        this.ch_kTilt = kTilt;
    }

    public void setChamberlainAttenuation(double kAtten_1_per_m) {
        this.ch_kAtten = Math.max(0.0, kAtten_1_per_m);
    }

    public void setChamberlainSegments(int n) {
        this.ch_lineSegments = Math.max(5, n);
    }

    public double getHeatReleaseMW() {
        return this.heatReleaseMW;
    }

    public Map<String, Double> getEmissionsKgPerHr() {
        return this.emissionsKgPerHr;
    }

    public double getTipBackpressureBar() {
        return this.tipBackpressureBar;
    }

    @Override
    public void run(UUID id) {
        if (this.reliefInlet == null) {
            throw new RuntimeException("FlareStack: relief inlet not connected.");
        }
        SystemInterface mix = this.reliefInlet.getThermoSystem().clone();
        mix.setTemperature(this.ambientTempK);
        mix.setPressure(this.ambientPressBar);
        mix.init(0);
        double mdot_kg_s = FlareStack.safeRate(this.reliefInlet.getFlowRate("kg/sec"));
        CombustionMixProps props = CombustionMixProps.fromSystem(mix);
        double LHV_J_per_kg = props.LHV_J_per_kg;
        double O2_stoich_kmol_s = props.O2Stoich_kmol_per_kg * mdot_kg_s;
        double O2_avail_kmol_s = 0.0;
        if (this.airAssist != null) {
            double kmol_s = FlareStack.safeRate(this.airAssist.getMolarRate() / 1000.0);
            double yO2 = FlareStack.getMoleFrac(this.airAssist.getThermoSystem(), "oxygen");
            O2_avail_kmol_s += kmol_s * yO2 * (1.0 + this.excessAirFrac);
        }
        double extent = O2_stoich_kmol_s > 1.0E-12 ? Math.min(1.0, O2_avail_kmol_s / O2_stoich_kmol_s) : 1.0;
        double etaEff = this.burningEfficiency * extent;
        double Qdot_W = etaEff * mdot_kg_s * LHV_J_per_kg;
        this.heatReleaseMW = Qdot_W * 1.0E-6;
        EmissionResult er = EmissionResult.compute(mix, mdot_kg_s, etaEff, this.unburnedTHCFraction, this.coFraction, this.so2Conversion);
        this.emissionsKgPerHr = er.toMapKgPerHr();
        double rho_exit = Math.max(0.1, this.reliefInlet.getThermoSystem().getDensity("kg/m3"));
        double area = Math.PI * this.tipDiameter_m * this.tipDiameter_m / 4.0;
        double u_exit = rho_exit > 1.0E-9 && area > 1.0E-9 ? mdot_kg_s / (rho_exit * area) : 0.0;
        double deltaP_bar = this.tipLossK * 0.5 * rho_exit * u_exit * u_exit / 100000.0;
        this.tipBackpressureBar = this.ambientPressBar + deltaP_bar;
        this.setCalculationIdentifier(id);
    }

    public double heatFlux_W_m2(double groundRange_m) {
        switch (this.radiationModel) {
            case CHAMBERLAIN: {
                return this.chamberlainHeatFlux(groundRange_m);
            }
        }
        return this.pointSourceHeatFlux(groundRange_m);
    }

    public double pointSourceHeatFlux(double groundRange_m) {
        double Qdot = this.heatReleaseMW * 1000000.0;
        double R = Math.sqrt(groundRange_m * groundRange_m + this.tipElevation_m * this.tipElevation_m);
        return this.radiantFraction * Qdot / (Math.PI * 4 * R * R);
    }

    public double chamberlainHeatFlux(double groundRange_m) {
        if (this.heatReleaseMW <= 1.0E-9) {
            return 0.0;
        }
        double EP = this.ch_epC_kWm2 * 1000.0 * Math.pow(Math.max(0.001, this.heatReleaseMW), this.ch_epA);
        double mdot_kg_s = FlareStack.safeRate(this.reliefInlet.getFlowRate("kg/sec"));
        double rho_exit = Math.max(0.1, this.reliefInlet.getThermoSystem().getDensity("kg/m3"));
        double area = Math.PI * this.tipDiameter_m * this.tipDiameter_m / 4.0;
        double Vexit = rho_exit > 1.0E-9 && area > 1.0E-9 ? mdot_kg_s / (rho_exit * area) : 0.0;
        double Lf = this.ch_lfC * Math.pow(Math.max(0.001, this.heatReleaseMW), this.ch_lfA) / Math.pow(Math.max(1.0E-6, this.tipDiameter_m), this.ch_lfB);
        double theta = Math.atan(this.ch_kTilt * this.windSpeed10m / Math.max(0.5, Vexit));
        theta = Math.min(theta, Math.toRadians(65.0));
        int N = this.ch_lineSegments;
        double ds = Lf / (double)N;
        double q_sum = 0.0;
        double x0 = 0.0;
        double z0 = this.tipElevation_m;
        for (int i = 0; i < N; ++i) {
            double s_mid = ((double)i + 0.5) * ds;
            double x = x0 + s_mid * Math.sin(theta);
            double z = z0 + s_mid * Math.cos(theta);
            double dx = groundRange_m - x;
            double dz = z;
            double R = Math.sqrt(dx * dx + dz * dz);
            double view = ds / (Math.PI * 4 * R * R);
            double cosTerm = dz / R;
            if (cosTerm < 0.0) {
                cosTerm = 0.0;
            }
            double tau = Math.exp(-this.ch_kAtten * R);
            q_sum += EP * view * cosTerm * tau;
        }
        return q_sum;
    }

    private static double getMoleFrac(SystemInterface s, String comp) {
        try {
            return s.getPhase(0).getComponent(comp).getz();
        }
        catch (Exception e) {
            return 0.0;
        }
    }

    private static double clamp01(double v) {
        return Math.max(0.0, Math.min(1.0, v));
    }

    private static double safeRate(double v) {
        return Double.isFinite(v) ? Math.max(0.0, v) : 0.0;
    }

    static class ElementStoich {
        final double C;
        final double H;
        final double S;

        private ElementStoich(double C, double H, double S) {
            this.C = C;
            this.H = H;
            this.S = S;
        }

        static ElementStoich of(String name) {
            switch (name) {
                case "methane": {
                    return new ElementStoich(1.0, 4.0, 0.0);
                }
                case "ethane": {
                    return new ElementStoich(2.0, 6.0, 0.0);
                }
                case "propane": {
                    return new ElementStoich(3.0, 8.0, 0.0);
                }
                case "n-butane": 
                case "butane": {
                    return new ElementStoich(4.0, 10.0, 0.0);
                }
                case "n-pentane": 
                case "pentane": {
                    return new ElementStoich(5.0, 12.0, 0.0);
                }
                case "hydrogen": {
                    return new ElementStoich(0.0, 2.0, 0.0);
                }
                case "carbon monoxide": 
                case "co": {
                    return new ElementStoich(1.0, 0.0, 0.0);
                }
                case "h2s": {
                    return new ElementStoich(0.0, 2.0, 1.0);
                }
            }
            return new ElementStoich(1.0, 1.8, 0.0);
        }
    }

    static class SpeciesData {
        final double LHV_J_per_kmol;
        final double O2_stoich_per_kmolFuel;

        private SpeciesData(double LHV_J_per_kmol, double O2) {
            this.LHV_J_per_kmol = LHV_J_per_kmol;
            this.O2_stoich_per_kmolFuel = O2;
        }

        static SpeciesData of(String name) {
            switch (name) {
                case "methane": {
                    return new SpeciesData(8.023E8, 2.0);
                }
                case "ethane": {
                    return new SpeciesData(1.429E9, 3.5);
                }
                case "propane": {
                    return new SpeciesData(2.043E9, 5.0);
                }
                case "n-butane": 
                case "butane": {
                    return new SpeciesData(2.658E9, 6.5);
                }
                case "n-pentane": 
                case "pentane": {
                    return new SpeciesData(3.273E9, 8.0);
                }
                case "hydrogen": {
                    return new SpeciesData(2.42E8, 0.5);
                }
                case "carbon monoxide": 
                case "co": {
                    return new SpeciesData(2.83E8, 0.5);
                }
                case "h2s": {
                    return new SpeciesData(5.18E8, 1.5);
                }
            }
            return new SpeciesData(1.2E9, 3.0);
        }

        static double mixLHV_J_per_kg(SystemInterface s) {
            CombustionMixProps p = CombustionMixProps.fromSystem(s);
            return p.LHV_J_per_kg;
        }
    }

    static class EmissionResult {
        double co2_kg_s;
        double h2o_kg_s;
        double so2_kg_s;
        double nox_kg_s;
        double co_kg_s;
        double thc_kg_s;

        EmissionResult() {
        }

        Map<String, Double> toMapKgPerHr() {
            HashMap<String, Double> m = new HashMap<String, Double>();
            m.put("CO2_kg_h", this.co2_kg_s * 3600.0);
            m.put("H2O_kg_h", this.h2o_kg_s * 3600.0);
            m.put("SO2_kg_h", this.so2_kg_s * 3600.0);
            m.put("NOx_kg_h", this.nox_kg_s * 3600.0);
            m.put("CO_kg_h", this.co_kg_s * 3600.0);
            m.put("THC_kg_h", this.thc_kg_s * 3600.0);
            return m;
        }

        static EmissionResult compute(SystemInterface s, double mdot_kg_s, double etaEff, double fracTHC, double fracCO, double fracSO2) {
            EmissionResult r = new EmissionResult();
            double MWmix = s.getMolarMass();
            double kmolFuel_s = MWmix > 1.0E-12 ? mdot_kg_s / MWmix : 0.0;
            double C_kmol = 0.0;
            double H_kmol = 0.0;
            double S_kmol = 0.0;
            int nc = s.getPhase(0).getNumberOfComponents();
            for (int i = 0; i < nc; ++i) {
                ComponentInterface c = s.getPhase(0).getComponent(i);
                double z = c.getz() * kmolFuel_s;
                ElementStoich es = ElementStoich.of(c.getComponentName().toLowerCase());
                C_kmol += z * es.C;
                H_kmol += z * es.H;
                S_kmol += z * es.S;
            }
            double C_to_CO2 = etaEff * (1.0 - fracCO - fracTHC) * C_kmol;
            double C_to_CO = etaEff * fracCO * C_kmol;
            double C_to_THC = (1.0 - etaEff) * C_kmol + etaEff * fracTHC * C_kmol;
            double H_to_H2O = etaEff * H_kmol / 2.0;
            double S_to_SO2 = etaEff * fracSO2 * S_kmol;
            r.co2_kg_s = C_to_CO2 * 44.0095;
            r.co_kg_s = C_to_CO * 28.01;
            r.thc_kg_s = C_to_THC * 16.043;
            r.h2o_kg_s = H_to_H2O * 18.015;
            r.so2_kg_s = S_to_SO2 * 64.066;
            double EF_NOx_g_per_MJ = 0.3;
            double LHV_J_per_kg = SpeciesData.mixLHV_J_per_kg(s);
            r.nox_kg_s = etaEff * mdot_kg_s * (LHV_J_per_kg * 1.0E-6) * (EF_NOx_g_per_MJ / 1000000.0);
            return r;
        }
    }

    static class CombustionMixProps {
        double LHV_J_per_kg;
        double O2Stoich_kmol_per_kg;

        CombustionMixProps() {
        }

        static CombustionMixProps fromSystem(SystemInterface s) {
            CombustionMixProps p = new CombustionMixProps();
            double MWmix = s.getMolarMass();
            double LHV_J_per_kmol = 0.0;
            double O2_kmol_per_kmolFuel = 0.0;
            int nc = s.getPhase(0).getNumberOfComponents();
            for (int i = 0; i < nc; ++i) {
                ComponentInterface comp = s.getPhase(0).getComponent(i);
                String name = comp.getComponentName().toLowerCase();
                double zi = comp.getz();
                SpeciesData sd = SpeciesData.of(name);
                LHV_J_per_kmol += zi * sd.LHV_J_per_kmol;
                O2_kmol_per_kmolFuel += zi * sd.O2_stoich_per_kmolFuel;
            }
            p.LHV_J_per_kg = LHV_J_per_kmol / MWmix;
            p.O2Stoich_kmol_per_kg = O2_kmol_per_kmolFuel / MWmix;
            return p;
        }
    }

    public static enum RadiationModel {
        POINT_SOURCE,
        CHAMBERLAIN;

    }
}

