/*
 * Decompiled with CFR 0.152.
 */
package neqsim.process.equipment.reactor;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.UUID;
import neqsim.process.equipment.TwoPortEquipment;
import neqsim.process.equipment.stream.StreamInterface;
import neqsim.thermo.system.SystemInterface;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.ejml.simple.SimpleMatrix;

public class GibbsReactor
extends TwoPortEquipment {
    private final ThreadLocal<SystemInterface> tempFugacitySystem = new ThreadLocal();
    private EnergyMode energyMode = EnergyMode.ADIABATIC;
    private static final long serialVersionUID = 1000L;
    static Logger logger = LogManager.getLogger(GibbsReactor.class);
    private String method = "DirectGibbsMinimization";
    private boolean useAllDatabaseSpecies = false;
    private List<GibbsComponent> gibbsDatabase = new ArrayList<GibbsComponent>();
    private Map<String, GibbsComponent> componentMap = new HashMap<String, GibbsComponent>();
    private double[] lambda = new double[6];
    private Map<String, Double> lagrangeContributions = new HashMap<String, Double>();
    private String[] elementNames = new String[]{"O", "N", "C", "H", "S", "Ar"};
    private List<String> processedComponents = new ArrayList<String>();
    private Map<String, Double> objectiveFunctionValues = new HashMap<String, Double>();
    private Map<String, Double> initialMoles = new HashMap<String, Double>();
    private Map<String, Double> finalMoles = new HashMap<String, Double>();
    private double[] elementMoleBalanceIn = new double[6];
    private double[] elementMoleBalanceOut = new double[6];
    private double[] elementMoleBalanceDiff = new double[6];
    private List<Double> inlet_mole = new ArrayList<Double>();
    private List<Double> outlet_mole = new ArrayList<Double>();
    private Map<String, Integer> processedComponentIndexMap = new HashMap<String, Integer>();
    private double[] objectiveMinimizationVector;
    private List<String> objectiveMinimizationVectorLabels = new ArrayList<String>();
    private double[][] jacobianMatrix;
    private double[][] jacobianInverse;
    private List<String> jacobianRowLabels = new ArrayList<String>();
    private List<String> jacobianColLabels = new ArrayList<String>();
    private int maxIterations = 5000;
    private double convergenceTolerance = 1.0E-6;
    private double dampingComposition = 0.001;
    private int actualIterations = 0;
    private boolean converged = false;
    private double finalConvergenceError = 0.0;
    private SystemInterface system;
    private double inletEnthalpy;
    private double outletEnthalpy;
    private double dT = 0.0;
    private int tempUpdateIter = 0;
    double enthalpyOld = 0.0;
    private double enthalpyOfReactions = 0.0;
    private double temperatureChange = 0.0;
    private double deltaNorm = 0.0;
    private double GOLD = 0.0;
    private double G = 0.0;
    private double dG = 0.0;

    public double getMassBalanceError() {
        try {
            double inletMass = this.getInletStream().getThermoSystem().getFlowRate("kg/sec");
            double outletMass = this.getOutletStream().getThermoSystem().getFlowRate("kg/sec");
            return Math.abs(inletMass - outletMass);
        }
        catch (Exception e) {
            logger.debug("WARNING: Could not calculate mass balance error: {}", (Object)e.getMessage());
            return Double.NaN;
        }
    }

    public boolean getMassBalanceConverged() {
        double error = this.getMassBalanceError();
        return !Double.isNaN(error) && error < 0.001;
    }

    public double getEnthalpyOfReactions() {
        return this.enthalpyOfReactions;
    }

    public double getTemperatureChange() {
        return this.temperatureChange;
    }

    public double getPower() {
        return -this.enthalpyOfReactions * 1000.0;
    }

    public double getPower(String unit) {
        if (unit == null) {
            return this.getPower();
        }
        switch (unit.trim().toLowerCase()) {
            case "kw": {
                return -this.enthalpyOfReactions;
            }
            case "mw": {
                return -this.enthalpyOfReactions / 1000.0;
            }
        }
        return -this.enthalpyOfReactions * 1000.0;
    }

    public double calculateMixtureEnthalpy(List<String> componentNames, List<Double> n, double T, Map<String, GibbsComponent> componentMap) {
        double totalH = 0.0;
        for (int i = 0; i < componentNames.size(); ++i) {
            String compName = componentNames.get(i);
            GibbsComponent comp = componentMap.get(compName.toLowerCase());
            if (comp == null) {
                throw new IllegalArgumentException("Component '" + compName + "' not found in gibbsReactDatabase.");
            }
            totalH += n.get(i) * comp.calculateEnthalpy(T, i);
        }
        logger.info("Mixture enthalpy at T=" + T + " K: " + totalH + " kJ");
        return totalH;
    }

    public double calculateMixtureGibbsEnergy(List<String> componentNames, List<Double> n, Map<String, GibbsComponent> componentMap, double T) {
        double totalG = 0.0;
        for (int i = 0; i < componentNames.size(); ++i) {
            String compName = componentNames.get(i);
            GibbsComponent comp = componentMap.get(compName.toLowerCase());
            if (comp == null) {
                throw new IllegalArgumentException("Component '" + compName + "' not found in gibbsReactDatabase.");
            }
            totalG += n.get(i) * comp.calculateGibbsEnergy(T, i);
        }
        logger.info("Mixture Gibbs energy at T=" + T + " K: " + totalG + " kJ");
        return totalG;
    }

    public double getMixtureEnthalpy() {
        double T = this.system != null ? this.system.getTemperature() : 298.15;
        return this.calculateMixtureEnthalpy(this.processedComponents, this.inlet_mole, T, this.componentMap);
    }

    public double getMixtureGibbsEnergy() {
        double T = this.system != null ? this.system.getTemperature() : 298.15;
        return this.calculateMixtureGibbsEnergy(this.processedComponents, this.inlet_mole, this.componentMap, T);
    }

    public double calculateMixtureEnthalpyStandard(List<String> componentNames, List<Double> n, Map<String, GibbsComponent> componentMap) {
        double totalH = 0.0;
        for (int i = 0; i < componentNames.size(); ++i) {
            String compName = componentNames.get(i);
            GibbsComponent comp = componentMap.get(compName.toLowerCase());
            if (comp == null) {
                throw new IllegalArgumentException("Component '" + compName + "' not found in gibbsReactDatabase.");
            }
            totalH += n.get(i) * comp.calculateEnthalpy(298.15, i);
        }
        return totalH;
    }

    public double calculateMixtureEnthalpy(List<String> componentNames, List<Double> n, Map<String, GibbsComponent> componentMap, double T) {
        double totalH = 0.0;
        for (int i = 0; i < componentNames.size(); ++i) {
            String compName = componentNames.get(i);
            GibbsComponent comp = componentMap.get(compName.toLowerCase());
            if (comp == null) {
                throw new IllegalArgumentException("Component '" + compName + "' not found in gibbsReactDatabase.");
            }
            totalH += n.get(i) * comp.calculateEnthalpy(T, i);
        }
        return totalH;
    }

    public void setEnergyMode(EnergyMode mode) {
        this.energyMode = mode;
    }

    public void setEnergyMode(String mode) {
        if (mode == null) {
            throw new IllegalArgumentException("Energy mode string cannot be null");
        }
        switch (mode.trim().toLowerCase()) {
            case "adiabatic": {
                this.setEnergyMode(EnergyMode.ADIABATIC);
                break;
            }
            case "isothermal": {
                this.setEnergyMode(EnergyMode.ISOTHERMAL);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown energy mode: " + mode);
            }
        }
    }

    public EnergyMode getEnergyMode() {
        return this.energyMode;
    }

    public GibbsReactor(String name) {
        super(name);
        this.loadGibbsDatabase();
    }

    public GibbsReactor(String name, StreamInterface stream) {
        super(name, stream);
        this.loadGibbsDatabase();
    }

    private void loadGibbsDatabase() {
        try {
            InputStream inputStream = this.getClass().getResourceAsStream("/data/GibbsReactDatabase/GibbsReactDatabase.csv");
            if (inputStream == null) {
                inputStream = this.getClass().getResourceAsStream("/neqsim/data/GibbsReactDatabase/GibbsReactDatabase.csv");
            }
            if (inputStream == null) {
                inputStream = this.getClass().getResourceAsStream("/neqsim/data/GibbsReactDatabase.csv");
            }
            if (inputStream == null) {
                logger.warn("Could not find GibbsReactDatabase.csv in resources");
                return;
            }
            HashMap<String, double[]> extraCoeffMap = new HashMap<String, double[]>();
            try {
                InputStream coeffStream = this.getClass().getResourceAsStream("/data/GibbsReactDatabase/DatabaseGibbsFreeEnergyCoeff.csv");
                if (coeffStream != null) {
                    Scanner coeffScanner = new Scanner(coeffStream);
                    if (coeffScanner.hasNextLine()) {
                        coeffScanner.nextLine();
                    }
                    while (coeffScanner.hasNextLine()) {
                        String[] parts;
                        String line = coeffScanner.nextLine().trim();
                        if (line.isEmpty() || line.startsWith("#") || (parts = line.split(";")).length != 13) continue;
                        String compName = parts[0].trim().toLowerCase();
                        double[] coeffs = new double[12];
                        for (int i = 0; i < 12; ++i) {
                            coeffs[i] = Double.parseDouble(parts[i + 1].replace(",", "."));
                        }
                        extraCoeffMap.put(compName, coeffs);
                    }
                    coeffScanner.close();
                }
            }
            catch (Exception e) {
                logger.warn("Could not load extra Gibbs coefficients: " + e.getMessage());
            }
            Scanner scanner = new Scanner(inputStream);
            if (scanner.hasNextLine()) {
                scanner.nextLine();
            }
            while (scanner.hasNextLine()) {
                String[] parts;
                String line = scanner.nextLine().trim();
                if (line.isEmpty() || line.startsWith("#") || (parts = line.split(";")).length < 14) continue;
                try {
                    String molecule = parts[0].trim();
                    double[] elements = new double[6];
                    for (int i = 0; i < 6; ++i) {
                        String value = parts[i + 1].trim().replace(",", ".");
                        elements[i] = Double.parseDouble(value);
                    }
                    double[] heatCapCoeffs = new double[4];
                    for (int i = 0; i < 4; ++i) {
                        String value = parts[i + 7].trim().replace(",", ".");
                        heatCapCoeffs[i] = Double.parseDouble(value);
                    }
                    String deltaHf298Str = parts[11].trim().replace(",", ".");
                    String deltaGf298Str = parts[12].trim().replace(",", ".");
                    String deltaSf298Str = parts[13].trim().replace(",", ".");
                    double deltaHf298 = Double.parseDouble(deltaHf298Str);
                    double deltaGf298 = Double.parseDouble(deltaGf298Str);
                    double deltaSf298 = Double.parseDouble(deltaSf298Str);
                    double[] coeffs = extraCoeffMap.getOrDefault(molecule.toLowerCase(), new double[12]);
                    GibbsComponent component = new GibbsComponent(molecule, elements, heatCapCoeffs, deltaHf298, deltaGf298, deltaSf298, coeffs.length > 0 ? coeffs[0] : Double.NaN, coeffs.length > 1 ? coeffs[1] : Double.NaN, coeffs.length > 2 ? coeffs[2] : Double.NaN, coeffs.length > 3 ? coeffs[3] : Double.NaN, coeffs.length > 4 ? coeffs[4] : Double.NaN, coeffs.length > 5 ? coeffs[5] : Double.NaN, coeffs.length > 6 ? coeffs[6] : Double.NaN, coeffs.length > 7 ? coeffs[7] : Double.NaN, coeffs.length > 8 ? coeffs[8] : Double.NaN, coeffs.length > 9 ? coeffs[9] : Double.NaN, coeffs.length > 10 ? coeffs[10] : Double.NaN, coeffs.length > 11 ? coeffs[11] : Double.NaN);
                    this.gibbsDatabase.add(component);
                    this.componentMap.put(molecule.toLowerCase(), component);
                    logger.debug("Loaded component: " + molecule);
                }
                catch (NumberFormatException e) {
                    logger.warn("Error parsing line: " + line + " - " + e.getMessage());
                }
            }
            scanner.close();
            logger.info("Loaded " + this.gibbsDatabase.size() + " components from Gibbs database");
        }
        catch (Exception e) {
            logger.error("Error loading Gibbs database: " + e.getMessage());
        }
    }

    public void setUseAllDatabaseSpecies(boolean useAllDatabaseSpecies) {
        this.useAllDatabaseSpecies = useAllDatabaseSpecies;
    }

    public boolean getUseAllDatabaseSpecies() {
        return this.useAllDatabaseSpecies;
    }

    public String getMethod() {
        return this.method;
    }

    public void setMethod(String method) {
        this.method = method;
    }

    public Map<String, Double> getLagrangeContributions() {
        return new HashMap<String, Double>(this.lagrangeContributions);
    }

    @Override
    public void run(UUID id) {
        String compName;
        int i;
        this.tempFugacitySystem.remove();
        this.system = this.getInletStream().getThermoSystem().clone();
        this.initialMoles.clear();
        this.inlet_mole.clear();
        this.outlet_mole.clear();
        for (i = 0; i < this.system.getNumberOfComponents(); ++i) {
            compName = this.system.getComponent(i).getComponentName();
            double moles = this.system.getComponent(i).getNumberOfMolesInPhase();
            this.initialMoles.put(compName, moles);
            this.inlet_mole.add(moles);
        }
        this.calculateElementMoleBalance(this.system, this.elementMoleBalanceIn, true);
        if (this.useAllDatabaseSpecies) {
            for (GibbsComponent component : this.gibbsDatabase) {
                try {
                    this.system.addComponent(component.getMolecule(), 1.0E-6);
                }
                catch (Exception e) {
                    logger.debug("Could not add component " + component.getMolecule() + ": " + e.getMessage());
                }
            }
        }
        this.performGibbsMinimization(this.system);
        this.enforceMinimumConcentrations(this.system);
        this.finalMoles.clear();
        this.processedComponents.clear();
        this.outlet_mole.clear();
        this.processedComponentIndexMap.clear();
        for (i = 0; i < this.system.getNumberOfComponents(); ++i) {
            compName = this.system.getComponent(i).getComponentName();
            double moles = this.system.getComponent(i).getNumberOfMolesInPhase();
            this.finalMoles.put(compName, moles);
            this.processedComponents.add(compName);
            this.processedComponentIndexMap.put(compName, i);
            double outletMoles = Math.max(moles, 1.0E-6);
            this.outlet_mole.add(outletMoles);
        }
        this.calculateElementMoleBalance(this.system, this.elementMoleBalanceOut, false);
        for (i = 0; i < this.elementNames.length; ++i) {
            this.elementMoleBalanceDiff[i] = this.elementMoleBalanceOut[i] - this.elementMoleBalanceIn[i];
        }
        this.calculateObjectiveFunctionValues(this.system);
        this.solveGibbsEquilibrium();
        this.getOutletStream().run(id);
        if (!this.getMassBalanceConverged()) {
            logger.debug("WARNING: Mass balance not converged in GibbsReactor. Consider decreasing the iteration step (damping factor) for better closure.");
        }
    }

    private void performGibbsMinimization(SystemInterface system) {
        String compName;
        int i;
        boolean iteration = true;
        HashMap<String, Double> initialGuess = new HashMap<String, Double>();
        for (i = 0; i < system.getNumberOfComponents(); ++i) {
            compName = system.getComponent(i).getComponentName();
            initialGuess.put(compName, system.getComponent(i).getNumberOfMolesInPhase());
        }
        for (i = 0; i < system.getNumberOfComponents(); ++i) {
            double currentMoles;
            compName = system.getComponent(i).getComponentName();
            if (!initialGuess.containsKey(compName) || !((currentMoles = system.getComponent(i).getNumberOfMolesInPhase()) < 1.0E-6)) continue;
            system.addComponent(i, 1.0E-6 - currentMoles, 0);
        }
        logger.info("Gibbs minimization completed for iteration 1");
    }

    private void calculateElementMoleBalance(SystemInterface system, double[] elementBalance, boolean isInput) {
        int i;
        for (i = 0; i < elementBalance.length; ++i) {
            elementBalance[i] = 0.0;
        }
        for (i = 0; i < system.getNumberOfComponents(); ++i) {
            String compName = system.getComponent(i).getComponentName();
            double moles = isInput ? (i < this.inlet_mole.size() ? this.inlet_mole.get(i) : 0.0) : (i < this.outlet_mole.size() ? this.outlet_mole.get(i) : 0.0);
            GibbsComponent comp = this.componentMap.get(compName.toLowerCase());
            if (comp == null) continue;
            double[] elements = comp.getElements();
            for (int j = 0; j < this.elementNames.length; ++j) {
                int n = j;
                elementBalance[n] = elementBalance[n] + elements[j] * moles;
            }
        }
    }

    private void calculateObjectiveFunctionValues(SystemInterface system) {
        this.objectiveFunctionValues.clear();
        double T = system.getTemperature();
        double RT = 0.008314462618 * T;
        double totalMoles = 0.0;
        for (Double moles : this.outlet_mole) {
            totalMoles += moles.doubleValue();
        }
        for (int i = 0; i < system.getNumberOfComponents(); ++i) {
            String compName = system.getComponent(i).getComponentName();
            double moles = i < this.outlet_mole.size() ? this.outlet_mole.get(i) : 1.0E-6;
            GibbsComponent comp = this.componentMap.get(compName.toLowerCase());
            if (comp == null) continue;
            double Gf0 = comp.calculateGibbsEnergy(T, i);
            double[] phi = this.getFugacityCoefficient(0);
            double yi = moles / totalMoles;
            double lagrangeSum = 0.0;
            double[] elements = comp.getElements();
            for (int j = 0; j < this.lambda.length; ++j) {
                lagrangeSum += this.lambda[j] * elements[j];
            }
            double F = Gf0 + RT * Math.log(phi[i]) + RT * Math.log(yi) - lagrangeSum;
            this.objectiveFunctionValues.put(compName, F);
        }
    }

    public void setLagrangeMultiplier(int index, double value) {
        if (index >= 0 && index < this.lambda.length) {
            this.lambda[index] = value;
        }
    }

    public double[] getLagrangianMultipliers() {
        return (double[])this.lambda.clone();
    }

    public String[] getElementNames() {
        return (String[])this.elementNames.clone();
    }

    public Map<String, Map<String, Double>> getLagrangeMultiplierContributions() {
        HashMap<String, Map<String, Double>> contributions = new HashMap<String, Map<String, Double>>();
        List<Object> componentsToProcess = this.processedComponents.isEmpty() ? new ArrayList<String>(this.finalMoles.keySet()) : this.processedComponents;
        for (String string : componentsToProcess) {
            HashMap<String, Double> compContributions = new HashMap<String, Double>();
            GibbsComponent comp = this.componentMap.get(string.toLowerCase());
            if (comp == null) continue;
            double[] elements = comp.getElements();
            double totalContribution = 0.0;
            for (int i = 0; i < this.elementNames.length; ++i) {
                double contribution = this.lambda[i] * elements[i];
                compContributions.put(this.elementNames[i], contribution);
                totalContribution += contribution;
            }
            compContributions.put("TOTAL", totalContribution);
            contributions.put(string, compContributions);
        }
        return contributions;
    }

    public Map<String, Double> getObjectiveFunctionValues() {
        return new HashMap<String, Double>(this.objectiveFunctionValues);
    }

    public double[] getElementMoleBalanceIn() {
        return (double[])this.elementMoleBalanceIn.clone();
    }

    public double[] getElementMoleBalanceOut() {
        return (double[])this.elementMoleBalanceOut.clone();
    }

    public double[] getElementMoleBalanceDiff() {
        return (double[])this.elementMoleBalanceDiff.clone();
    }

    public Map<String, Map<String, Double>> getDetailedMoleBalance() {
        HashMap<String, Map<String, Double>> detailedBalance = new HashMap<String, Map<String, Double>>();
        for (int i = 0; i < this.processedComponents.size(); ++i) {
            Double molesOut;
            String compName = this.processedComponents.get(i);
            GibbsComponent comp = this.componentMap.get(compName.toLowerCase());
            Double molesIn = i < this.inlet_mole.size() ? this.inlet_mole.get(i) : Double.valueOf(0.0);
            Double d = molesOut = i < this.outlet_mole.size() ? this.outlet_mole.get(i) : Double.valueOf(1.0E-6);
            if (comp == null) continue;
            HashMap<String, Double> componentBalance = new HashMap<String, Double>();
            double[] elements = comp.getElements();
            componentBalance.put("MOLES_IN", molesIn);
            componentBalance.put("MOLES_OUT", molesOut);
            componentBalance.put("MOLES_DIFF", molesOut - molesIn);
            for (int j = 0; j < this.elementNames.length; ++j) {
                double elementIn = elements[j] * molesIn;
                double elementOut = elements[j] * molesOut;
                double elementDiff = elementOut - elementIn;
                componentBalance.put(this.elementNames[j] + "_IN", elementIn);
                componentBalance.put(this.elementNames[j] + "_OUT", elementOut);
                componentBalance.put(this.elementNames[j] + "_DIFF", elementDiff);
            }
            detailedBalance.put(compName, componentBalance);
        }
        return detailedBalance;
    }

    private void calculateObjectiveMinimizationVector() {
        int i;
        List<Integer> activeElements = this.findActiveElements();
        int numComponents = this.processedComponents.size();
        int numActiveElements = activeElements.size();
        this.objectiveMinimizationVector = new double[numComponents + numActiveElements];
        this.objectiveMinimizationVectorLabels.clear();
        for (i = 0; i < numComponents; ++i) {
            String compName = this.processedComponents.get(i);
            Double fValue = this.objectiveFunctionValues.get(compName);
            this.objectiveMinimizationVector[i] = fValue != null ? fValue : 0.0;
            this.objectiveMinimizationVectorLabels.add("F_" + compName);
        }
        for (i = 0; i < numActiveElements; ++i) {
            int elementIndex = activeElements.get(i);
            this.objectiveMinimizationVector[numComponents + i] = this.elementMoleBalanceDiff[elementIndex];
            this.objectiveMinimizationVectorLabels.add("Balance_" + this.elementNames[elementIndex]);
        }
    }

    public double[] getObjectiveMinimizationVector() {
        this.calculateObjectiveMinimizationVector();
        return (double[])this.objectiveMinimizationVector.clone();
    }

    public List<String> getObjectiveMinimizationVectorLabels() {
        return new ArrayList<String>(this.objectiveMinimizationVectorLabels);
    }

    private void calculateJacobian() {
        int i;
        int i2;
        if (this.processedComponents.isEmpty()) {
            return;
        }
        List<Integer> activeElements = this.findActiveElements();
        int numComponents = this.processedComponents.size();
        int numActiveElements = activeElements.size();
        int totalVars = numComponents + numActiveElements;
        this.jacobianMatrix = new double[totalVars][totalVars];
        this.jacobianRowLabels.clear();
        this.jacobianColLabels.clear();
        for (i2 = 0; i2 < numComponents; ++i2) {
            this.jacobianRowLabels.add("F_" + this.processedComponents.get(i2));
            this.jacobianColLabels.add("n_" + this.processedComponents.get(i2));
        }
        for (i2 = 0; i2 < numActiveElements; ++i2) {
            int elementIndex = activeElements.get(i2);
            this.jacobianRowLabels.add("Balance_" + this.elementNames[elementIndex]);
            this.jacobianColLabels.add("lambda_" + this.elementNames[elementIndex]);
        }
        SystemInterface system = this.getOutletStream().getThermoSystem();
        double T = system.getTemperature();
        double RT = 0.008314462618 * T;
        double totalMoles = 0.0;
        for (Double moles : this.outlet_mole) {
            totalMoles += moles.doubleValue();
        }
        for (i = 0; i < numComponents; ++i) {
            String compI = this.processedComponents.get(i);
            double ni = i < this.outlet_mole.size() ? this.outlet_mole.get(i) : 1.0E-6;
            double niForJacobian = Math.max(ni, 1.0E-6);
            system.init(3);
            for (int j = 0; j < numComponents; ++j) {
                this.jacobianMatrix[i][j] = i == j ? RT * (1.0 / niForJacobian - 1.0 / totalMoles + system.getPhase(0).getComponent(i).getdfugdn(j)) : -RT / totalMoles + system.getPhase(0).getComponent(i).getdfugdn(j);
            }
            GibbsComponent gibbsComp = this.componentMap.get(compI.toLowerCase());
            if (gibbsComp == null) continue;
            double[] elements = gibbsComp.getElements();
            for (int k = 0; k < numActiveElements; ++k) {
                int elementIndex = activeElements.get(k);
                this.jacobianMatrix[i][numComponents + k] = -elements[elementIndex];
            }
        }
        for (i = 0; i < numActiveElements; ++i) {
            int elementIndex = activeElements.get(i);
            for (int j = 0; j < numComponents; ++j) {
                String compName = this.processedComponents.get(j);
                GibbsComponent gibbsComp = this.componentMap.get(compName.toLowerCase());
                if (gibbsComp == null) continue;
                double[] elements = gibbsComp.getElements();
                this.jacobianMatrix[numComponents + i][j] = elements[elementIndex];
            }
            for (int k = 0; k < numActiveElements; ++k) {
                this.jacobianMatrix[numComponents + i][numComponents + k] = 0.0;
            }
        }
        this.jacobianInverse = this.calculateJacobianInverse();
    }

    public double[][] getJacobianMatrix() {
        this.calculateJacobian();
        if (this.jacobianMatrix == null) {
            return null;
        }
        double[][] copy = new double[this.jacobianMatrix.length][];
        for (int i = 0; i < this.jacobianMatrix.length; ++i) {
            copy[i] = (double[])this.jacobianMatrix[i].clone();
        }
        return copy;
    }

    public List<String> getJacobianRowLabels() {
        return new ArrayList<String>(this.jacobianRowLabels);
    }

    public List<String> getJacobianColLabels() {
        return new ArrayList<String>(this.jacobianColLabels);
    }

    public double[][] getJacobianInverse() {
        if (this.jacobianInverse == null) {
            return null;
        }
        double[][] copy = new double[this.jacobianInverse.length][];
        for (int i = 0; i < this.jacobianInverse.length; ++i) {
            copy[i] = (double[])this.jacobianInverse[i].clone();
        }
        return copy;
    }

    private double[][] calculateJacobianInverse() {
        if (this.jacobianMatrix == null) {
            return null;
        }
        try {
            SimpleMatrix ejmlMatrix = new SimpleMatrix(this.jacobianMatrix);
            SimpleMatrix inverseMatrix = (SimpleMatrix)ejmlMatrix.invert();
            int nRows = inverseMatrix.numRows();
            int nCols = inverseMatrix.numCols();
            double[][] result = new double[nRows][nCols];
            double[] data = inverseMatrix.getDDRM().getData();
            for (int i = 0; i < nRows; ++i) {
                for (int j = 0; j < nCols; ++j) {
                    result[i][j] = data[i * nCols + j];
                }
            }
            return result;
        }
        catch (RuntimeException e) {
            logger.warn("Jacobian matrix is singular or nearly singular: " + e.getMessage());
            return null;
        }
    }

    private void enforceMinimumConcentrations(SystemInterface system) {
        double minConcentration = 1.0E-6;
        boolean modified = false;
        for (int i = 0; i < system.getNumberOfComponents(); ++i) {
            double currentMoles;
            String compName = system.getComponent(i).getComponentName();
            if (this.componentMap.get(compName.toLowerCase()) == null || !((currentMoles = system.getComponent(i).getNumberOfMolesInPhase()) < minConcentration)) continue;
            logger.info("Component " + compName + " has very low concentration (" + currentMoles + "), setting to minimum: " + minConcentration);
            system.addComponent(i, minConcentration - currentMoles, 0);
            modified = true;
        }
        if (modified) {
            logger.info("System reinitialized after enforcing minimum concentrations");
        }
    }

    public List<Double> getInletMole() {
        return Collections.unmodifiableList(this.inlet_mole);
    }

    public List<Double> getOutletMole() {
        return Collections.unmodifiableList(this.outlet_mole);
    }

    public List<Double> getInletMoles() {
        return Collections.unmodifiableList(this.inlet_mole);
    }

    public List<Double> getOutletMoles() {
        return Collections.unmodifiableList(this.outlet_mole);
    }

    public void printDatabaseComponents() {
        for (GibbsComponent comp : this.gibbsDatabase) {
            String molecule = comp.getMolecule();
            double[] dArray = comp.getElements();
        }
        for (String string : this.componentMap.keySet()) {
        }
    }

    private List<Integer> findActiveElements() {
        ArrayList<Integer> activeElements = new ArrayList<Integer>();
        for (int elementIndex = 0; elementIndex < this.elementNames.length; ++elementIndex) {
            boolean elementPresent = false;
            for (String compName : this.processedComponents) {
                double[] elements;
                GibbsComponent comp = this.componentMap.get(compName.toLowerCase());
                if (comp == null || !(Math.abs((elements = comp.getElements())[elementIndex]) > 1.0E-6)) continue;
                elementPresent = true;
                break;
            }
            if (!elementPresent) continue;
            activeElements.add(elementIndex);
        }
        return activeElements;
    }

    private List<Integer> getActiveElementIndices() {
        ArrayList<Integer> activeIndices = new ArrayList<Integer>();
        for (int elementIndex = 0; elementIndex < this.elementNames.length; ++elementIndex) {
            boolean hasNonZero = false;
            for (String compName : this.processedComponents) {
                double[] elements;
                GibbsComponent comp = this.componentMap.get(compName.toLowerCase());
                if (comp == null || !(Math.abs((elements = comp.getElements())[elementIndex]) > 1.0E-10)) continue;
                hasNonZero = true;
                break;
            }
            if (!hasNonZero) continue;
            activeIndices.add(elementIndex);
        }
        return activeIndices;
    }

    public boolean verifyJacobianInverse() {
        if (this.jacobianMatrix == null || this.jacobianInverse == null) {
            return false;
        }
        try {
            SimpleMatrix jacobianMatrixEJML = new SimpleMatrix(this.jacobianMatrix);
            SimpleMatrix jacobianInverseEJML = new SimpleMatrix(this.jacobianInverse);
            SimpleMatrix resultMatrix = jacobianMatrixEJML.mult(jacobianInverseEJML);
            double[] resultData = resultMatrix.getDDRM().getData();
            double tolerance = 1.0E-10;
            int n = this.jacobianMatrix.length;
            for (int i = 0; i < n; ++i) {
                for (int j = 0; j < n; ++j) {
                    double expected;
                    double value = resultData[i * n + j];
                    double d = expected = i == j ? 1.0 : 0.0;
                    if (!(Math.abs(value - expected) > tolerance)) continue;
                    logger.warn("Jacobian inverse verification failed at [" + i + "," + j + "]: expected " + expected + ", got " + value);
                    return false;
                }
            }
            return true;
        }
        catch (RuntimeException e) {
            logger.warn("Error during Jacobian inverse verification: " + e.getMessage());
            return false;
        }
    }

    public double[] performNewtonRaphsonIteration() {
        this.calculateJacobian();
        if (this.jacobianMatrix == null || this.jacobianInverse == null) {
            logger.warn("Cannot perform Newton-Raphson iteration: Jacobian or its inverse is null");
            return null;
        }
        double[] objectiveVector = this.getObjectiveMinimizationVector();
        if (objectiveVector == null || objectiveVector.length != this.jacobianInverse.length) {
            logger.warn("Objective vector size mismatch with Jacobian matrix");
            return null;
        }
        try {
            SimpleMatrix jacobianInverseEJML = new SimpleMatrix(this.jacobianInverse);
            SimpleMatrix objectiveVectorEJML = new SimpleMatrix(objectiveVector.length, 1, true, objectiveVector);
            SimpleMatrix deltaXMatrix = (SimpleMatrix)jacobianInverseEJML.mult(objectiveVectorEJML).scale(-1.0);
            int nRows = deltaXMatrix.numRows();
            double[] result = new double[nRows];
            double[] data = deltaXMatrix.getDDRM().getData();
            for (int i = 0; i < nRows; ++i) {
                result[i] = data[i];
            }
            return result;
        }
        catch (RuntimeException e) {
            logger.warn("Error during Newton-Raphson iteration calculation: " + e.getMessage());
            return null;
        }
    }

    public boolean performIterationUpdate(double[] deltaX, double alphaComposition) {
        double newValue;
        double oldValue;
        int i;
        List<Integer> activeElementIndices;
        int numActiveElements;
        if (deltaX == null || this.outlet_mole.isEmpty()) {
            logger.warn("Cannot perform iteration update: deltaX or outlet_mole is null/empty");
            return false;
        }
        int numComponents = this.processedComponents.size();
        if (deltaX.length != numComponents + (numActiveElements = (activeElementIndices = this.getActiveElementIndices()).size())) {
            logger.warn("Delta vector size mismatch: expected " + (numComponents + numActiveElements) + ", got " + deltaX.length);
            return false;
        }
        this.deltaNorm = 0.0;
        for (i = 0; i < numComponents; ++i) {
            String compName = this.processedComponents.get(i);
            if (this.componentMap.get(compName.toLowerCase()) == null) continue;
            oldValue = this.outlet_mole.get(i);
            double deltaComposition = deltaX[i];
            newValue = oldValue + deltaComposition * alphaComposition;
            newValue = Math.max(newValue, 1.0E-15);
            this.outlet_mole.set(i, newValue);
            this.deltaNorm += Math.pow(deltaComposition * alphaComposition, 2.0);
        }
        this.deltaNorm = Math.sqrt(this.deltaNorm);
        for (i = 0; i < numActiveElements; ++i) {
            int elementIndex = activeElementIndices.get(i);
            oldValue = this.lambda[elementIndex];
            double deltaLambda = deltaX[numComponents + i];
            this.lambda[elementIndex] = newValue = oldValue + deltaLambda;
        }
        this.deltaNorm = Math.sqrt(this.deltaNorm);
        return this.updateSystemWithNewCompositions();
    }

    private boolean updateSystemWithNewCompositions() {
        try {
            int i;
            SystemInterface system = this.getOutletStream().getThermoSystem();
            for (i = 0; i < this.processedComponents.size(); ++i) {
                double currentMoles;
                double molesToAdd;
                String compName = this.processedComponents.get(i);
                if (this.componentMap.get(compName.toLowerCase()) == null) continue;
                double newMoles = this.outlet_mole.get(i);
                int compIndex = -1;
                for (int j = 0; j < system.getNumberOfComponents(); ++j) {
                    if (!compName.equals(system.getComponent(j).getComponentName())) continue;
                    compIndex = j;
                    break;
                }
                if (compIndex < 0 || !(Math.abs(molesToAdd = newMoles - (currentMoles = system.getComponent(compIndex).getNumberOfMolesInPhase())) > 1.0E-15)) continue;
                system.addComponent(compIndex, molesToAdd, 0);
            }
            this.calculateObjectiveFunctionValues(system);
            this.calculateElementMoleBalance(system, this.elementMoleBalanceOut, false);
            for (i = 0; i < this.elementNames.length; ++i) {
                this.elementMoleBalanceDiff[i] = this.elementMoleBalanceOut[i] - this.elementMoleBalanceIn[i];
            }
            return true;
        }
        catch (Exception e) {
            logger.error("Error updating system with new compositions: " + e.getMessage());
            return false;
        }
    }

    public double[] getFugacityCoefficient(Object phaseNameOrIndex) {
        int phaseIndex = 0;
        if (phaseNameOrIndex instanceof Integer) {
            phaseIndex = (Integer)phaseNameOrIndex;
        } else if (phaseNameOrIndex instanceof String) {
            String phaseName = ((String)phaseNameOrIndex).toLowerCase();
            for (int i = 0; i < this.system.getNumberOfPhases(); ++i) {
                String name = this.system.getPhase(i).getPhaseTypeName().toLowerCase();
                if (!name.contains(phaseName)) continue;
                phaseIndex = i;
                break;
            }
        }
        int numComponents = this.system.getNumberOfComponents();
        double[] phiArray = new double[numComponents];
        for (int i = 0; i < numComponents; ++i) {
            phiArray[i] = this.system.getPhase(phaseIndex).getComponent(i).getFugacityCoefficient();
        }
        return phiArray;
    }

    public void setMaxIterations(int maxIterations) {
        this.maxIterations = maxIterations;
    }

    public int getMaxIterations() {
        return this.maxIterations;
    }

    public void setConvergenceTolerance(double convergenceTolerance) {
        this.convergenceTolerance = convergenceTolerance;
    }

    public double getConvergenceTolerance() {
        return this.convergenceTolerance;
    }

    public void setDampingComposition(double dampingComposition) {
        this.dampingComposition = dampingComposition;
    }

    public double getDampingComposition() {
        return this.dampingComposition;
    }

    public int getActualIterations() {
        return this.actualIterations;
    }

    public boolean hasConverged() {
        return this.converged;
    }

    public double getFinalConvergenceError() {
        return this.finalConvergenceError;
    }

    public boolean solveGibbsEquilibrium(double alphaComposition) {
        this.converged = false;
        this.actualIterations = 0;
        this.finalConvergenceError = Double.MAX_VALUE;
        logger.info("Starting Gibbs equilibrium solution with Newton-Raphson iterations");
        logger.info("Maximum iterations: " + this.maxIterations);
        logger.info("Convergence tolerance: " + this.convergenceTolerance);
        logger.info("Composition step size: " + alphaComposition);
        for (int iteration = 1; iteration <= this.maxIterations; ++iteration) {
            this.actualIterations = iteration;
            SystemInterface outletSystem = this.getOutletStream().getThermoSystem();
            this.calculateObjectiveFunctionValues(outletSystem);
            logger.debug("Iteration {} component properties:", (Object)iteration);
            for (int i = 0; i < outletSystem.getNumberOfComponents(); ++i) {
                String compName = outletSystem.getComponent(i).getComponentName();
                GibbsComponent comp = this.componentMap.get(compName.toLowerCase());
                if (comp == null) continue;
                double T = outletSystem.getTemperature();
                double gibbs = comp.calculateGibbsEnergy(T, i);
                double enthalpy = comp.calculateEnthalpy(T, i);
                double entropy = comp.calculateEntropy(T, i);
                logger.debug(String.format("Component: %s, GibbsEnergy: %.2f kJ/mol, Enthalpy: %.2f kJ/mol, Entropy: %.2f kJ/(mol\u00b7K)", compName, gibbs, enthalpy, entropy));
            }
            Map<String, Double> fValues = this.getObjectiveFunctionValues();
            double fNorm = 0.0;
            for (Double value : fValues.values()) {
                fNorm += value * value;
            }
            fNorm = Math.sqrt(fNorm);
            logger.debug("Iteration " + iteration + ": F vector norm = " + fNorm);
            double[] deltaX = this.performNewtonRaphsonIteration();
            if (deltaX == null) {
                logger.warn("Newton-Raphson iteration failed at iteration " + iteration);
                this.finalConvergenceError = fNorm;
                return false;
            }
            double deltaXNorm = 0.0;
            for (double value : deltaX) {
                deltaXNorm += value * value;
            }
            deltaXNorm = Math.sqrt(deltaXNorm);
            logger.debug("Iteration " + iteration + ": Delta vector norm = " + deltaXNorm);
            logger.debug("deltaXNorm (full update vector): {}", (Object)deltaXNorm);
            if (iteration == 1) {
                this.G = this.calculateMixtureGibbsEnergy(this.processedComponents, this.outlet_mole, this.componentMap, this.system.getTemperature());
                logger.debug("Initial Gibbs energy G = " + this.G);
                this.GOLD = this.G;
            } else {
                this.GOLD = this.G;
                this.G = this.calculateMixtureGibbsEnergy(this.processedComponents, this.outlet_mole, this.componentMap, this.system.getTemperature());
                this.dG = this.G - this.GOLD;
                logger.debug("Gibbs energy change dG = " + this.dG);
            }
            if (this.energyMode == EnergyMode.ADIABATIC) {
                if (iteration == 1) {
                    this.enthalpyOld = this.inletEnthalpy = this.calculateMixtureEnthalpy(this.processedComponents, this.outlet_mole, this.componentMap, this.system.getTemperature());
                } else {
                    this.outletEnthalpy = this.calculateMixtureEnthalpy(this.processedComponents, this.outlet_mole, this.componentMap, this.system.getTemperature());
                    double dH = this.outletEnthalpy - this.enthalpyOld;
                    this.enthalpyOfReactions += dH;
                    this.enthalpyOld = this.outletEnthalpy;
                    double T_out = this.system.getTemperature() - dH * 1000.0 / this.system.getCp("J/K");
                    this.dT = Math.abs(T_out - this.system.getTemperature());
                    if (this.dT > 1000.0) {
                        throw new RuntimeException("Temperature change per iteration (dT) exceeded 1000 K. Please reduce the step of iteration (alphaComposition or damping factor).");
                    }
                    this.temperatureChange += this.dT;
                    this.system.setTemperature(T_out);
                    this.system.init(3);
                    this.getOutletStream().getThermoSystem().setTemperature(this.system.getTemperature());
                }
            }
            if (deltaXNorm < this.convergenceTolerance && iteration >= 100 || iteration == this.maxIterations) {
                logger.info((deltaXNorm < this.convergenceTolerance ? "Converged" : "Max iterations reached") + " at iteration " + iteration + " with delta norm = " + deltaXNorm);
                this.converged = deltaXNorm < this.convergenceTolerance;
                this.finalConvergenceError = deltaXNorm;
                this.updateSystemWithNewCompositions();
                this.getOutletStream().getThermoSystem().setTemperature(this.system.getTemperature());
                if (iteration == this.maxIterations) {
                    logger.warn("Maximum number of iterations reached without convergence. Please increase the maximum number of iterations (maxIterations) and try again.");
                }
                return true;
            }
            boolean updateSuccess = this.performIterationUpdate(deltaX, alphaComposition);
            if (!updateSuccess) {
                logger.warn("Iteration update failed at iteration " + iteration);
                this.finalConvergenceError = deltaXNorm;
                return false;
            }
            this.finalConvergenceError = deltaXNorm;
        }
        logger.warn("Maximum iterations (" + this.maxIterations + ") reached without convergence");
        logger.warn("Final convergence error: " + this.finalConvergenceError);
        return false;
    }

    public boolean solveGibbsEquilibrium() {
        return this.solveGibbsEquilibrium(this.dampingComposition);
    }

    public class GibbsComponent {
        private String molecule;
        private double[] elements = new double[6];
        private double[] heatCapacityCoeffs = new double[4];
        private double deltaHf298;
        private double deltaGf298;
        private double deltaSf298;
        private double coeffAg;
        private double coeffBg;
        private double coeffCg;
        private double coeffDg;
        private double coeffEg;
        private double coeffFg;
        private double coeffAh;
        private double coeffBh;
        private double coeffCh;
        private double coeffDh;
        private double coeffEh;
        private double coeffGh;

        public GibbsComponent(String molecule, double[] elements, double[] heatCapacityCoeffs, double deltaHf298, double deltaGf298, double deltaSf298, double coeffAg, double coeffBg, double coeffCg, double coeffDg, double coeffEg, double coeffFg, double coeffAh, double coeffBh, double coeffCh, double coeffDh, double coeffEh, double coeffGh) {
            this.molecule = molecule;
            this.elements = (double[])elements.clone();
            this.heatCapacityCoeffs = (double[])heatCapacityCoeffs.clone();
            this.deltaHf298 = deltaHf298;
            this.deltaGf298 = deltaGf298;
            this.deltaSf298 = deltaSf298;
            this.coeffAg = coeffAg;
            this.coeffBg = coeffBg;
            this.coeffCg = coeffCg;
            this.coeffDg = coeffDg;
            this.coeffEg = coeffEg;
            this.coeffFg = coeffFg;
            this.coeffAh = coeffAh;
            this.coeffBh = coeffBh;
            this.coeffCh = coeffCh;
            this.coeffDh = coeffDh;
            this.coeffEh = coeffEh;
            this.coeffGh = coeffGh;
        }

        public double getDeltaSf298() {
            return this.deltaSf298;
        }

        public String getMolecule() {
            return this.molecule;
        }

        public double[] getElements() {
            return (double[])this.elements.clone();
        }

        public double[] getHeatCapacityCoeffs() {
            return (double[])this.heatCapacityCoeffs.clone();
        }

        public double getDeltaHf298() {
            return this.deltaHf298;
        }

        public double getDeltaGf298() {
            return this.deltaGf298;
        }

        public double calculateGibbsEnergy(double temperature, int compNumber) {
            double T = temperature;
            if (!(Double.isNaN(this.coeffAg) && Double.isNaN(this.coeffBg) && Double.isNaN(this.coeffCg) && Double.isNaN(this.coeffDg) && Double.isNaN(this.coeffEg) && Double.isNaN(this.coeffFg))) {
                double gibbsPoly = (Double.isNaN(this.coeffAg) ? 0.0 : this.coeffAg) * Math.pow(T, 5.0) + (Double.isNaN(this.coeffBg) ? 0.0 : this.coeffBg) * Math.pow(T, 4.0) + (Double.isNaN(this.coeffCg) ? 0.0 : this.coeffCg) * Math.pow(T, 3.0) + (Double.isNaN(this.coeffDg) ? 0.0 : this.coeffDg) * Math.pow(T, 2.0) + (Double.isNaN(this.coeffEg) ? 0.0 : this.coeffEg) * T + (Double.isNaN(this.coeffFg) ? 0.0 : this.coeffFg);
                return gibbsPoly;
            }
            double H_T = this.calculateEnthalpy(T, compNumber);
            double S_T = this.calculateEntropy(T, compNumber);
            double dG_T = H_T - this.deltaHf298 - T * (S_T - this.deltaSf298 / 1000.0);
            return this.deltaGf298 + dG_T;
        }

        public double calculateEnthalpy(double temperature, int compNumber) {
            double T = temperature;
            if (!(Double.isNaN(this.coeffAh) && Double.isNaN(this.coeffBh) && Double.isNaN(this.coeffCh) && Double.isNaN(this.coeffDh) && Double.isNaN(this.coeffEh) && Double.isNaN(this.coeffGh))) {
                double enthalpyPoly = (Double.isNaN(this.coeffAh) ? 0.0 : this.coeffAh) * Math.pow(T, 5.0) + (Double.isNaN(this.coeffBh) ? 0.0 : this.coeffBh) * Math.pow(T, 4.0) + (Double.isNaN(this.coeffCh) ? 0.0 : this.coeffCh) * Math.pow(T, 3.0) + (Double.isNaN(this.coeffDh) ? 0.0 : this.coeffDh) * Math.pow(T, 2.0) + (Double.isNaN(this.coeffEh) ? 0.0 : this.coeffEh) * Math.pow(T, 1.0) + (Double.isNaN(this.coeffGh) ? 0.0 : this.coeffGh);
                return enthalpyPoly;
            }
            double T0 = 298.15;
            double Cp = this.calculateHeatCapacity(T, compNumber);
            return this.deltaHf298 + Cp * (T - T0) / 1000.0;
        }

        public double calculateEntropy(double temperature, int compNumber) {
            double T = temperature;
            double T0 = 298.15;
            double Cp = this.calculateHeatCapacity(T, compNumber);
            return (this.deltaSf298 + Cp * Math.log(T / T0)) / 1000.0;
        }

        public double calculateHeatCapacity(double temperature, int compNumber) {
            try {
                double cp0 = GibbsReactor.this.system.getComponent(compNumber).getCp0(temperature);
                return cp0;
            }
            catch (Exception e) {
                double T = temperature;
                double A = this.heatCapacityCoeffs[0];
                double B = this.heatCapacityCoeffs[1];
                double C = this.heatCapacityCoeffs[2];
                double D = this.heatCapacityCoeffs[3];
                return A + B * T + C * T * T + D * T * T * T;
            }
        }
    }

    public static enum EnergyMode {
        ISOTHERMAL,
        ADIABATIC;

    }
}

