# Data Processing

Data are a first-class citizen of Constellation. They are transmitted as records with one or more blocks, and mechanisms
are in place which ensure that no data is lost in transmission.
This document provides an overview of the concepts of data handling in Constellation and details some of the features
available.

## Transmitters & Receivers

Data in Constellation are handled exclusively by [satellites](./satellite.md).
Two types of satellites exits, transmitter satellites which generate data and send it off through the Constellation network,
and receiver satellites which receive data from one or more transmitter satellites and process it, e.g. by storing it to
disk.

```{seealso}
Details about how to implement data transmission in a satellite can be found in the
[application development guide](../../application_development/functionality/data_transmission.md).
```

## Structure

Data consist of individual messages that are grouped to so-called runs. A run represents a set of data recorded in known
conditions and with a fixed set of parameters.
A new run starts upon entering the `RUN` state of the satellite [finite state machine](./satellite.md#the-finite-state-machine)
and ends with the departure from this state.
Each run is labeled by its _run identifier_, which is provided along with the `start` command sent to the satellite, and
consists of individual data messages.
The first message of a run is the so-called begin-of-run (BOR) message, the last one the end-of-run (EOR) message.
This structure enables checks on sending and receiving side, e.g. that the connection is up and running or that all data have
been transmitted correctly.

```{seealso}
A detailed technical description, including protocol sequence diagrams, can be found in the
[protocol description chapter](../../framework_reference/protocols.md#data-transmission) in the framework development guide.
```

## Messages

Messages transmitted over the data communication channel of Constellation consist of a header with auxiliary information, and
a payload. The content of header as well as payload differs for the different message types and is detailed below.

### Data Messages

Regular messages are data messages. They contain the sender's name, the message type and any number of data records.
A data record contains contains a sequence number, a dictionary of tags and any number of data blocks.

The sequence number is automatically incremented for each generated record. This sequence is continuously checked at the
receiving end and therefore allows to directly detect missing messages.

Tags may be added to the dictionary by the sending code. If such information is available at the time of creating the
message, it is recommended to encode the begin and end of the data record contained in the respective message, as integer
values counting picoseconds since the start of the measurement run, in the tags `timestamp_begin` and `timestamp_end`.

The data blocks contain binary data. This data is not interpreted by the Constellation framework but e.g. directly written
to file without further processing. Whether a single block or multiple blocks per record are used is left to the sending
code.

### Begin-of-Run & End-of-Run Messages

The tags of the BOR and EOR messages can be added by the satellite implementations in their respective `starting` and
`stopping` functions. This can be used e.g. to record additional information such as a firmware version read from attached
instrument hardware at the time of starting, or to indicate an accumulated metric at the time of stopping.

The BOR message contains the configuration of the transmitting satellite at the time of entering the `RUN` state.
Depending on the actual implementation of the satellite itself, this allows to unequivocally infer the setup, state
and configuration of the satellite and its attached hardware from the recorded data.

The EOR message contains a dictionary with run metadata generated by the Constellation framework.
Typical run metadata entries are:

* `run_id`: the run identifier of this run
* `condition`: the run condition flags at the end of the run
* `condition_code`: the run condition flags code
* `time_start`: the starting time of the run, i.e. the time of sending the BOR message
* `time_end`: the ending time of the run, i.e. the time of sending the EOR message

The run condition flags allows to assess the condition during the `RUN` state of the Constellation and thereby the quality of
the data recorded. Runs are marked with any of the following run condition flags:

* `GOOD` (code: `0x00`, no flag set): The run has concluded normally, no other information has been provided by the sender
* `TAINTED` (code: `0x01`): The data has been marked as tainted by the sender
* `INCOMPLETE` (code: `0x02`): The receiver has noticed missing messages in the sequence
* `INTERRUPTED` (code: `0x04`): The run has been interrupted by this sender because of a failure condition elsewhere in the Constellation
* `ABORTED` (code: `0x08`): The run has been aborted by the sender and the EOR message has been appended by the receiver
* `DEGRADED` (code: `0x10`): The run has been marked as degraded because not all satellites were present during the entire time of the run


## Receiving and Storing Data

Receiver satellites accept data form one or more satellites within the same Constellation. In order to assign transmitter
satellites to a specific receiver, the receiver can be given the `_data_transmitters` parameter which holds a list of all
canonical names of transmitters it should connect to. If not given, receiver satellites connect to all transmitters.

Usually, receiver satellites will take great care that the data are stored properly, e.g. by testing access to the storage
location, and by checking sufficient available disk space. They also emit telemetry data on the remaining disk space at the
selected storage location via the `DISKSPACE_FREE` metric, so monitoring of this parameter on receiving satellites can be
set up e.g. with [Grafana](../howtos/setup_influxdb_grafana.md):

```{figure} diskspace_grafana.png
Grafana Dashboard detail showing the available disk space on a receiver satellite. Several sections of data reception are
visible, corresponding to individual runs, and times of constant available disk space, indicating pauses between consecutive
runs.
```

In addition, receiver satellites will emit `WARNING` or `CRITICAL` log messages when the available disk space at the
selected storage location falls below 10 GB or 3 GB, respectively. This provides yet another way to notify the operator to
take action, free disk space or select a different storage location:

```{figure} diskspace_observatory.png
Detail of the Observatory logging interface displaying `WARNING` messages about low disk space every 10 seconds, until the
available space falls below 3GB, when the message severity changes to `CRITICAL`.
```

```{seealso}
More details about receiving telemetry information or log messages are provided in the respective sections on
[telemetry](../concepts/telemetry.md) and [logging](../concepts/logging.md).
```

## Data Licensing

Constellation provides the possibility to store a data license identifier in the run metadata in order to explicitly
define reuse policy and attribution requirements as suggested by the [FAIR principles](https://www.go-fair.org/fair-principles/).
The data license can be configured for every transmitter satellite by adding the `_data_license` key to its configuration:

```toml
[satellites.MySatellite.Name]
_data_license = "ODC-By-1.0"
```

If no such key is provided, the [Open Data Commons Attribution License](https://opendatacommons.org/licenses/by/) is used as
default license for the respective data set.
