from typing import overload
import abc
import typing

import System
import System.Runtime.Intrinsics
import System.Runtime.Intrinsics.Arm


class ArmBase(System.Object, metaclass=abc.ABCMeta):
    """This class provides access to the ARM base hardware instructions via intrinsics"""

    class Arm64(System.Object, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

        @staticmethod
        @overload
        def LeadingSignCount(value: int) -> int:
            """A64: CLS Wd, Wn"""
            ...

        @staticmethod
        @overload
        def LeadingSignCount(value: int) -> int:
            """A64: CLS Xd, Xn"""
            ...

        @staticmethod
        @overload
        def LeadingZeroCount(value: int) -> int:
            """A64: CLZ Xd, Xn"""
            ...

        @staticmethod
        @overload
        def LeadingZeroCount(value: int) -> int:
            """A64: CLZ Xd, Xn"""
            ...

        @staticmethod
        @overload
        def MultiplyHigh(left: int, right: int) -> int:
            """A64: SMULH Xd, Xn, Xm"""
            ...

        @staticmethod
        @overload
        def MultiplyHigh(left: int, right: int) -> int:
            """A64: UMULH Xd, Xn, Xm"""
            ...

        @staticmethod
        @overload
        def ReverseElementBits(value: int) -> int:
            """A64: RBIT Xd, Xn"""
            ...

        @staticmethod
        @overload
        def ReverseElementBits(value: int) -> int:
            """A64: RBIT Xd, Xn"""
            ...

    IsSupported: bool

    @staticmethod
    @overload
    def LeadingZeroCount(value: int) -> int:
        """
        A32: CLZ Rd, Rm
          A64: CLZ Wd, Wn
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: int) -> int:
        """
        A32: CLZ Rd, Rm
          A64: CLZ Wd, Wn
        """
        ...

    @staticmethod
    @overload
    def ReverseElementBits(value: int) -> int:
        """
        A32: RBIT Rd, Rm
          A64: RBIT Wd, Wn
        """
        ...

    @staticmethod
    @overload
    def ReverseElementBits(value: int) -> int:
        """
        A32: RBIT Rd, Rm
          A64: RBIT Wd, Wn
        """
        ...

    @staticmethod
    def Yield() -> None:
        """
        A32: YIELD
          A64: YIELD
        """
        ...


class Sha256(System.Runtime.Intrinsics.Arm.ArmBase, metaclass=abc.ABCMeta):
    """This class provides access to the ARM SHA256 hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.ArmBase.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

    IsSupported: bool

    @staticmethod
    @overload
    def HashUpdate1(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_efgh: System.Runtime.Intrinsics.Vector128[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256hq_u32 (uint32x4_t hash_abcd, uint32x4_t hash_efgh, uint32x4_t wk)
          A32: SHA256H.32 Qd, Qn, Qm
          A64: SHA256H Qd, Qn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdate1(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_efgh: System.Runtime.Intrinsics.Vector128[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256hq_u32 (uint32x4_t hash_abcd, uint32x4_t hash_efgh, uint32x4_t wk)
          A32: SHA256H.32 Qd, Qn, Qm
          A64: SHA256H Qd, Qn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdate2(hash_efgh: System.Runtime.Intrinsics.Vector128[int], hash_abcd: System.Runtime.Intrinsics.Vector128[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256h2q_u32 (uint32x4_t hash_efgh, uint32x4_t hash_abcd, uint32x4_t wk)
          A32: SHA256H2.32 Qd, Qn, Qm
          A64: SHA256H2 Qd, Qn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdate2(hash_efgh: System.Runtime.Intrinsics.Vector128[int], hash_abcd: System.Runtime.Intrinsics.Vector128[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256h2q_u32 (uint32x4_t hash_efgh, uint32x4_t hash_abcd, uint32x4_t wk)
          A32: SHA256H2.32 Qd, Qn, Qm
          A64: SHA256H2 Qd, Qn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate0(w0_3: System.Runtime.Intrinsics.Vector128[int], w4_7: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256su0q_u32 (uint32x4_t w0_3, uint32x4_t w4_7)
          A32: SHA256SU0.32 Qd, Qm
          A64: SHA256SU0 Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate0(w0_3: System.Runtime.Intrinsics.Vector128[int], w4_7: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256su0q_u32 (uint32x4_t w0_3, uint32x4_t w4_7)
          A32: SHA256SU0.32 Qd, Qm
          A64: SHA256SU0 Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate1(w0_3: System.Runtime.Intrinsics.Vector128[int], w8_11: System.Runtime.Intrinsics.Vector128[int], w12_15: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256su1q_u32 (uint32x4_t w0_3, uint32x4_t w8_11, uint32x4_t w12_15)
          A32: SHA256SU1.32 Qd, Qn, Qm
          A64: SHA256SU1 Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate1(w0_3: System.Runtime.Intrinsics.Vector128[int], w8_11: System.Runtime.Intrinsics.Vector128[int], w12_15: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha256su1q_u32 (uint32x4_t w0_3, uint32x4_t w8_11, uint32x4_t w12_15)
          A32: SHA256SU1.32 Qd, Qn, Qm
          A64: SHA256SU1 Vd.4S, Vn.4S, Vm.4S
        """
        ...


class AdvSimd(System.Runtime.Intrinsics.Arm.ArmBase, metaclass=abc.ABCMeta):
    """This class provides access to the ARM AdvSIMD hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.ArmBase.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

        @staticmethod
        @overload
        def Abs(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vabsq_f64 (float64x2_t a)
              A64: FABS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def Abs(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vabsq_s64 (int64x2_t a)
              A64: ABS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def AbsoluteCompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcagtq_f64 (float64x2_t a, float64x2_t b)
              A64: FACGT Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        def AbsoluteCompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcageq_f64 (float64x2_t a, float64x2_t b)
              A64: FACGE Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareGreaterThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcage_f64 (float64x1_t a, float64x1_t b)
              A64: FACGE Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareGreaterThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcages_f32 (float32_t a, float32_t b)
              A64: FACGE Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareGreaterThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcagt_f64 (float64x1_t a, float64x1_t b)
              A64: FACGT Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareGreaterThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcagts_f32 (float32_t a, float32_t b)
              A64: FACGT Sd, Sn, Sm
            """
            ...

        @staticmethod
        def AbsoluteCompareLessThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcaltq_f64 (float64x2_t a, float64x2_t b)
              A64: FACGT Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        def AbsoluteCompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcaleq_f64 (float64x2_t a, float64x2_t b)
              A64: FACGE Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareLessThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcale_f64 (float64x1_t a, float64x1_t b)
              A64: FACGE Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareLessThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcales_f32 (float32_t a, float32_t b)
              A64: FACGE Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareLessThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcalt_f64 (float64x1_t a, float64x1_t b)
              A64: FACGT Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteCompareLessThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcalts_f32 (float32_t a, float32_t b)
              A64: FACGT Sd, Sn, Sm
            """
            ...

        @staticmethod
        def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vabdq_f64 (float64x2_t a, float64x2_t b)
              A64: FABD Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AbsoluteDifferenceScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vabd_f64 (float64x1_t a, float64x1_t b)
              A64: FABD Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def AbsoluteDifferenceScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vabds_f32 (float32_t a, float32_t b)
              A64: FABD Sd, Sn, Sm
            """
            ...

        @staticmethod
        def AbsSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vqabsq_s64 (int64x2_t a)
              A64: SQABS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def AbsSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqabsh_s16 (int16_t a)
              A64: SQABS Hd, Hn
            """
            ...

        @staticmethod
        @overload
        def AbsSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqabss_s32 (int32_t a)
              A64: SQABS Sd, Sn
            """
            ...

        @staticmethod
        @overload
        def AbsSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqabsd_s64 (int64_t a)
              A64: SQABS Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def AbsSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqabsb_s8 (int8_t a)
              A64: SQABS Bd, Bn
            """
            ...

        @staticmethod
        def AbsScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vabs_s64 (int64x1_t a)
              A64: ABS Dd, Dn
            """
            ...

        @staticmethod
        def Add(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vaddq_f64 (float64x2_t a, float64x2_t b)
              A64: FADD Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vaddv_u8 (uint8x8_t a)
              A64: ADDV Bd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vaddv_s16 (int16x4_t a)
              A64: ADDV Hd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vaddv_s8 (int8x8_t a)
              A64: ADDV Bd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vaddv_u16 (uint16x4_t a)
              A64: ADDV Hd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vaddvq_u8 (uint8x16_t a)
              A64: ADDV Bd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vaddvq_s16 (int16x8_t a)
              A64: ADDV Hd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vaddvq_s32 (int32x4_t a)
              A64: ADDV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vaddvq_s8 (int8x16_t a)
              A64: ADDV Bd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vaddvq_u16 (uint16x8_t a)
              A64: ADDV Hd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def AddAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vaddvq_u32 (uint32x4_t a)
              A64: ADDV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vaddlv_u8 (uint8x8_t a)
              A64: UADDLV Hd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vaddlv_s16 (int16x4_t a)
              A64: SADDLV Sd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vaddlv_s8 (int8x8_t a)
              A64: SADDLV Hd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vaddlv_u16 (uint16x4_t a)
              A64: UADDLV Sd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vaddlvq_u8 (uint8x16_t a)
              A64: UADDLV Hd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vaddlvq_s16 (int16x8_t a)
              A64: SADDLV Sd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vaddlvq_s32 (int32x4_t a)
              A64: SADDLV Dd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vaddlvq_s8 (int8x16_t a)
              A64: SADDLV Hd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vaddlvq_u16 (uint16x8_t a)
              A64: UADDLV Sd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def AddAcrossWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64_t vaddlvq_u32 (uint32x4_t a)
              A64: UADDLV Dd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vpaddq_u8 (uint8x16_t a, uint8x16_t b)
              A64: ADDP Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vpaddq_f64 (float64x2_t a, float64x2_t b)
              A64: FADDP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vpaddq_s16 (int16x8_t a, int16x8_t b)
              A64: ADDP Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vpaddq_s32 (int32x4_t a, int32x4_t b)
              A64: ADDP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vpaddq_s64 (int64x2_t a, int64x2_t b)
              A64: ADDP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vpaddq_s8 (int8x16_t a, int8x16_t b)
              A64: ADDP Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vpaddq_f32 (float32x4_t a, float32x4_t b)
              A64: FADDP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vpaddq_u16 (uint16x8_t a, uint16x8_t b)
              A64: ADDP Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vpaddq_u32 (uint32x4_t a, uint32x4_t b)
              A64: ADDP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def AddPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vpaddq_u64 (uint64x2_t a, uint64x2_t b)
              A64: ADDP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def AddPairwiseScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vpadds_f32 (float32x2_t a)
              A64: FADDP Sd, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def AddPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vpaddd_f64 (float64x2_t a)
              A64: FADDP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def AddPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vpaddd_s64 (int64x2_t a)
              A64: ADDP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def AddPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64_t vpaddd_u64 (uint64x2_t a)
              A64: ADDP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vsqadd_u8 (uint8x8_t a, int8x8_t b)
              A64: USQADD Vd.8B, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vuqadd_s16 (int16x4_t a, uint16x4_t b)
              A64: SUQADD Vd.4H, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vuqadd_s32 (int32x2_t a, uint32x2_t b)
              A64: SUQADD Vd.2S, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vuqadd_s8 (int8x8_t a, uint8x8_t b)
              A64: SUQADD Vd.8B, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vsqadd_u16 (uint16x4_t a, int16x4_t b)
              A64: USQADD Vd.4H, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vsqadd_u32 (uint32x2_t a, int32x2_t b)
              A64: USQADD Vd.2S, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vsqaddq_u8 (uint8x16_t a, int8x16_t b)
              A64: USQADD Vd.16B, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vuqaddq_s16 (int16x8_t a, uint16x8_t b)
              A64: SUQADD Vd.8H, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vuqaddq_s32 (int32x4_t a, uint32x4_t b)
              A64: SUQADD Vd.4S, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vuqaddq_s64 (int64x2_t a, uint64x2_t b)
              A64: SUQADD Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vuqaddq_s8 (int8x16_t a, uint8x16_t b)
              A64: SUQADD Vd.16B, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vsqaddq_u16 (uint16x8_t a, int16x8_t b)
              A64: USQADD Vd.8H, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vsqaddq_u32 (uint32x4_t a, int32x4_t b)
              A64: USQADD Vd.4S, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vsqaddq_u64 (uint64x2_t a, int64x2_t b)
              A64: USQADD Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqaddb_u8 (uint8_t a, uint8_t b)
              A64: UQADD Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vsqaddb_u8 (uint8_t a, int8_t b)
              A64: USQADD Bd, Bn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqaddh_s16 (int16_t a, int16_t b)
              A64: SQADD Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vuqaddh_s16 (int16_t a, uint16_t b)
              A64: SUQADD Hd, Hn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqadds_s32 (int32_t a, int32_t b)
              A64: SQADD Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vuqadds_s32 (int32_t a, uint32_t b)
              A64: SUQADD Sd, Sn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vuqadd_s64 (int64x1_t a, uint64x1_t b)
              A64: SUQADD Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqaddb_s8 (int8_t a, int8_t b)
              A64: SQADD Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vuqaddb_s8 (int8_t a, uint8_t b)
              A64: SUQADD Bd, Bn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqaddh_u16 (uint16_t a, uint16_t b)
              A64: UQADD Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vsqaddh_u16 (uint16_t a, int16_t b)
              A64: USQADD Hd, Hn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqadds_u32 (uint32_t a, uint32_t b)
              A64: UQADD Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vsqadds_u32 (uint32_t a, int32_t b)
              A64: USQADD Sd, Sn
            """
            ...

        @staticmethod
        @overload
        def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vsqadd_u64 (uint64x1_t a, int64x1_t b)
              A64: USQADD Dd, Dn
            """
            ...

        @staticmethod
        def Ceiling(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndpq_f64 (float64x2_t a)
              A64: FRINTP Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def CompareEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vceqq_f64 (float64x2_t a, float64x2_t b)
              A64: FCMEQ Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vceqq_s64 (int64x2_t a, int64x2_t b)
              A64: CMEQ Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vceqq_u64 (uint64x2_t a, uint64x2_t b)
              A64: CMEQ Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vceq_f64 (float64x1_t a, float64x1_t b)
              A64: FCMEQ Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareEqualScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vceq_s64 (int64x1_t a, int64x1_t b)
              A64: CMEQ Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vceqs_f32 (float32_t a, float32_t b)
              A64: FCMEQ Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def CompareEqualScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vceq_u64 (uint64x1_t a, uint64x1_t b)
              A64: CMEQ Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcgtq_f64 (float64x2_t a, float64x2_t b)
              A64: FCMGT Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcgtq_s64 (int64x2_t a, int64x2_t b)
              A64: CMGT Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcgtq_u64 (uint64x2_t a, uint64x2_t b)
              A64: CMHI Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcgeq_f64 (float64x2_t a, float64x2_t b)
              A64: FCMGE Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcgeq_s64 (int64x2_t a, int64x2_t b)
              A64: CMGE Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcgeq_u64 (uint64x2_t a, uint64x2_t b)
              A64: CMHS Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcge_f64 (float64x1_t a, float64x1_t b)
              A64: FCMGE Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcge_s64 (int64x1_t a, int64x1_t b)
              A64: CMGE Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcges_f32 (float32_t a, float32_t b)
              A64: FCMGE Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcge_u64 (uint64x1_t a, uint64x1_t b)
              A64: CMHS Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcgt_f64 (float64x1_t a, float64x1_t b)
              A64: FCMGT Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcgt_s64 (int64x1_t a, int64x1_t b)
              A64: CMGT Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcgts_f32 (float32_t a, float32_t b)
              A64: FCMGT Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def CompareGreaterThanScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcgt_u64 (uint64x1_t a, uint64x1_t b)
              A64: CMHI Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcltq_f64 (float64x2_t a, float64x2_t b)
              A64: FCMGT Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcltq_s64 (int64x2_t a, int64x2_t b)
              A64: CMGT Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcltq_u64 (uint64x2_t a, uint64x2_t b)
              A64: CMHI Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vcleq_f64 (float64x2_t a, float64x2_t b)
              A64: FCMGE Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcleq_s64 (int64x2_t a, int64x2_t b)
              A64: CMGE Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcleq_u64 (uint64x2_t a, uint64x2_t b)
              A64: CMHS Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vcle_f64 (float64x1_t a, float64x1_t b)
              A64: FCMGE Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcle_s64 (int64x1_t a, int64x1_t b)
              A64: CMGE Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vcles_f32 (float32_t a, float32_t b)
              A64: FCMGE Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanOrEqualScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcle_u64 (uint64x1_t a, uint64x1_t b)
              A64: CMHS Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vclt_f64 (float64x1_t a, float64x1_t b)
              A64: FCMGT Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vclt_s64 (int64x1_t a, int64x1_t b)
              A64: CMGT Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint32_t vclts_f32 (float32_t a, float32_t b)
              A64: FCMGT Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def CompareLessThanScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vclt_u64 (uint64x1_t a, uint64x1_t b)
              A64: CMHI Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareTest(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            uint64x2_t vtstq_f64 (float64x2_t a, float64x2_t b)
              A64: CMTST Vd.2D, Vn.2D, Vm.2D
            The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
            """
            ...

        @staticmethod
        @overload
        def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vtstq_s64 (int64x2_t a, int64x2_t b)
              A64: CMTST Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vtstq_u64 (uint64x2_t a, uint64x2_t b)
              A64: CMTST Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def CompareTestScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            uint64x1_t vtst_f64 (float64x1_t a, float64x1_t b)
              A64: CMTST Dd, Dn, Dm
            The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
            """
            ...

        @staticmethod
        @overload
        def CompareTestScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vtst_s64 (int64x1_t a, int64x1_t b)
              A64: CMTST Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def CompareTestScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vtst_u64 (uint64x1_t a, uint64x1_t b)
              A64: CMTST Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def ConvertToDouble(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vcvt_f64_f32 (float32x2_t a)
              A64: FCVTL Vd.2D, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def ConvertToDouble(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vcvtq_f64_s64 (int64x2_t a)
              A64: SCVTF Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def ConvertToDouble(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vcvtq_f64_u64 (uint64x2_t a)
              A64: UCVTF Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def ConvertToDoubleScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vcvt_f64_s64 (int64x1_t a)
              A64: SCVTF Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def ConvertToDoubleScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vcvt_f64_u64 (uint64x1_t a)
              A64: UCVTF Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToDoubleUpper(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vcvt_high_f64_f32 (float32x4_t a)
              A64: FCVTL2 Vd.2D, Vn.4S
            """
            ...

        @staticmethod
        def ConvertToInt64RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vcvtaq_s64_f64 (float64x2_t a)
              A64: FCVTAS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToInt64RoundAwayFromZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vcvta_s64_f64 (float64x1_t a)
              A64: FCVTAS Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToEven(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vcvtnq_s64_f64 (float64x2_t a)
              A64: FCVTNS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToEvenScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vcvtn_s64_f64 (float64x1_t a)
              A64: FCVTNS Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vcvtmq_s64_f64 (float64x2_t a)
              A64: FCVTMS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToNegativeInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vcvtm_s64_f64 (float64x1_t a)
              A64: FCVTMS Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vcvtpq_s64_f64 (float64x2_t a)
              A64: FCVTPS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToPositiveInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vcvtp_s64_f64 (float64x1_t a)
              A64: FCVTPS Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vcvtq_s64_f64 (float64x2_t a)
              A64: FCVTZS Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToInt64RoundToZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vcvt_s64_f64 (float64x1_t a)
              A64: FCVTZS Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToSingleLower(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vcvt_f32_f64 (float64x2_t a)
              A64: FCVTN Vd.2S, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToSingleRoundToOddLower(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vcvtx_f32_f64 (float64x2_t a)
              A64: FCVTXN Vd.2S, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToSingleRoundToOddUpper(lower: System.Runtime.Intrinsics.Vector64[float], value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vcvtx_high_f32_f64 (float32x2_t r, float64x2_t a)
              A64: FCVTXN2 Vd.4S, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToSingleUpper(lower: System.Runtime.Intrinsics.Vector64[float], value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vcvt_high_f32_f64 (float32x2_t r, float64x2_t a)
              A64: FCVTN2 Vd.4S, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcvtaq_u64_f64 (float64x2_t a)
              A64: FCVTAU Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundAwayFromZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcvta_u64_f64 (float64x1_t a)
              A64: FCVTAU Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToEven(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcvtnq_u64_f64 (float64x2_t a)
              A64: FCVTNU Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToEvenScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcvtn_u64_f64 (float64x1_t a)
              A64: FCVTNU Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcvtmq_u64_f64 (float64x2_t a)
              A64: FCVTMU Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToNegativeInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcvtm_u64_f64 (float64x1_t a)
              A64: FCVTMU Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcvtpq_u64_f64 (float64x2_t a)
              A64: FCVTPU Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToPositiveInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcvtp_u64_f64 (float64x1_t a)
              A64: FCVTPU Dd, Dn
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcvtq_u64_f64 (float64x2_t a)
              A64: FCVTZU Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def ConvertToUInt64RoundToZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint64x1_t vcvt_u64_f64 (float64x1_t a)
              A64: FCVTZU Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def Divide(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vdiv_f32 (float32x2_t a, float32x2_t b)
              A64: FDIV Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def Divide(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vdivq_f64 (float64x2_t a, float64x2_t b)
              A64: FDIV Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def Divide(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vdivq_f32 (float32x4_t a, float32x4_t b)
              A64: FDIV Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[float], index: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vdupq_laneq_f64 (float64x2_t vec, const int lane)
              A64: DUP Vd.2D, Vn.D[index]
            """
            ...

        @staticmethod
        @overload
        def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vdupq_laneq_s64 (int64x2_t vec, const int lane)
              A64: DUP Vd.2D, Vn.D[index]
            """
            ...

        @staticmethod
        @overload
        def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vdupq_laneq_u64 (uint64x2_t vec, const int lane)
              A64: DUP Vd.2D, Vn.D[index]
            """
            ...

        @staticmethod
        @overload
        def DuplicateToVector128(value: float) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vdupq_n_f64 (float64_t value)
              A64: DUP Vd.2D, Vn.D[0]
            """
            ...

        @staticmethod
        @overload
        def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vdupq_n_s64 (int64_t value)
              A64: DUP Vd.2D, Rn
            """
            ...

        @staticmethod
        @overload
        def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vdupq_n_s64 (uint64_t value)
              A64: DUP Vd.2D, Rn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqmovnh_u16 (uint16_t a)
              A64: UQXTN Bd, Hn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqmovns_s32 (int32_t a)
              A64: SQXTN Hd, Sn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqmovnd_s64 (int64_t a)
              A64: SQXTN Sd, Dn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqmovnh_s16 (int16_t a)
              A64: SQXTN Bd, Hn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqmovns_u32 (uint32_t a)
              A64: UQXTN Hd, Sn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqmovnd_u64 (uint64_t a)
              A64: UQXTN Sd, Dn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqmovunh_s16 (int16_t a)
              A64: SQXTUN Bd, Hn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqmovuns_s32 (int32_t a)
              A64: SQXTUN Hd, Sn
            """
            ...

        @staticmethod
        @overload
        def ExtractNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqmovund_s64 (int64_t a)
              A64: SQXTUN Sd, Dn
            """
            ...

        @staticmethod
        def Floor(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndmq_f64 (float64x2_t a)
              A64: FRINTM Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def FusedMultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vfmaq_f64 (float64x2_t a, float64x2_t b, float64x2_t c)
              A64: FMLA Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vfma_n_f32 (float32x2_t a, float32x2_t b, float32_t n)
              A64: FMLA Vd.2S, Vn.2S, Vm.S[0]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vfmaq_n_f64 (float64x2_t a, float64x2_t b, float64_t n)
              A64: FMLA Vd.2D, Vn.2D, Vm.D[0]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vfmaq_n_f32 (float32x4_t a, float32x4_t b, float32_t n)
              A64: FMLA Vd.4S, Vn.4S, Vm.S[0]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vfma_lane_f32 (float32x2_t a, float32x2_t b, float32x2_t v, const int lane)
              A64: FMLA Vd.2S, Vn.2S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vfma_laneq_f32 (float32x2_t a, float32x2_t b, float32x4_t v, const int lane)
              A64: FMLA Vd.2S, Vn.2S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vfmaq_laneq_f64 (float64x2_t a, float64x2_t b, float64x2_t v, const int lane)
              A64: FMLA Vd.2D, Vn.2D, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vfmaq_lane_f32 (float32x4_t a, float32x4_t b, float32x2_t v, const int lane)
              A64: FMLA Vd.4S, Vn.4S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vfmaq_laneq_f32 (float32x4_t a, float32x4_t b, float32x4_t v, const int lane)
              A64: FMLA Vd.4S, Vn.4S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddScalarBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vfmad_laneq_f64 (float64_t a, float64_t b, float64x2_t v, const int lane)
              A64: FMLA Dd, Dn, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddScalarBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vfmas_lane_f32 (float32_t a, float32_t b, float32x2_t v, const int lane)
              A64: FMLA Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplyAddScalarBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vfmas_laneq_f32 (float32_t a, float32_t b, float32x4_t v, const int lane)
              A64: FMLA Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        def FusedMultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vfmsq_f64 (float64x2_t a, float64x2_t b, float64x2_t c)
              A64: FMLS Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vfms_n_f32 (float32x2_t a, float32x2_t b, float32_t n)
              A64: FMLS Vd.2S, Vn.2S, Vm.S[0]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vfmsq_n_f64 (float64x2_t a, float64x2_t b, float64_t n)
              A64: FMLS Vd.2D, Vn.2D, Vm.D[0]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vfmsq_n_f32 (float32x4_t a, float32x4_t b, float32_t n)
              A64: FMLS Vd.4S, Vn.4S, Vm.S[0]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vfms_lane_f32 (float32x2_t a, float32x2_t b, float32x2_t v, const int lane)
              A64: FMLS Vd.2S, Vn.2S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vfms_laneq_f32 (float32x2_t a, float32x2_t b, float32x4_t v, const int lane)
              A64: FMLS Vd.2S, Vn.2S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vfmsq_laneq_f64 (float64x2_t a, float64x2_t b, float64x2_t v, const int lane)
              A64: FMLS Vd.2D, Vn.2D, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vfmsq_lane_f32 (float32x4_t a, float32x4_t b, float32x2_t v, const int lane)
              A64: FMLS Vd.4S, Vn.4S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vfmsq_laneq_f32 (float32x4_t a, float32x4_t b, float32x4_t v, const int lane)
              A64: FMLS Vd.4S, Vn.4S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractScalarBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vfmsd_laneq_f64 (float64_t a, float64_t b, float64x2_t v, const int lane)
              A64: FMLS Dd, Dn, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractScalarBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vfmss_lane_f32 (float32_t a, float32_t b, float32x2_t v, const int lane)
              A64: FMLS Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def FusedMultiplySubtractScalarBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vfmss_laneq_f32 (float32_t a, float32_t b, float32x4_t v, const int lane)
              A64: FMLS Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vcopy_lane_u8 (uint8x8_t a, const int lane1, uint8x8_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vcopy_laneq_u8 (uint8x8_t a, const int lane1, uint8x16_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vcopy_lane_s16 (int16x4_t a, const int lane1, int16x4_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vcopy_laneq_s16 (int16x4_t a, const int lane1, int16x8_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vcopy_lane_s32 (int32x2_t a, const int lane1, int32x2_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vcopy_laneq_s32 (int32x2_t a, const int lane1, int32x4_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vcopy_lane_s8 (int8x8_t a, const int lane1, int8x8_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vcopy_laneq_s8 (int8x8_t a, const int lane1, int8x16_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[float], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[float], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vcopy_lane_f32 (float32x2_t a, const int lane1, float32x2_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[float], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[float], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vcopy_laneq_f32 (float32x2_t a, const int lane1, float32x4_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vcopy_lane_u16 (uint16x4_t a, const int lane1, uint16x4_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vcopy_laneq_u16 (uint16x4_t a, const int lane1, uint16x8_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vcopy_lane_u32 (uint32x2_t a, const int lane1, uint32x2_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector64[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vcopy_laneq_u32 (uint32x2_t a, const int lane1, uint32x4_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vcopyq_lane_u8 (uint8x16_t a, const int lane1, uint8x8_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vcopyq_laneq_u8 (uint8x16_t a, const int lane1, uint8x16_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[float], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[float], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vcopyq_laneq_f64 (float64x2_t a, const int lane1, float64x2_t b, const int lane2)
              A64: INS Vd.D[lane1], Vn.D[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vcopyq_lane_s16 (int16x8_t a, const int lane1, int16x4_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vcopyq_laneq_s16 (int16x8_t a, const int lane1, int16x8_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vcopyq_lane_s32 (int32x4_t a, const int lane1, int32x2_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vcopyq_laneq_s32 (int32x4_t a, const int lane1, int32x4_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vcopyq_laneq_s64 (int64x2_t a, const int lane1, int64x2_t b, const int lane2)
              A64: INS Vd.D[lane1], Vn.D[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vcopyq_lane_s8 (int8x16_t a, const int lane1, int8x8_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vcopyq_laneq_s8 (int8x16_t a, const int lane1, int8x16_t b, const int lane2)
              A64: INS Vd.B[lane1], Vn.B[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[float], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[float], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vcopyq_lane_f32 (float32x4_t a, const int lane1, float32x2_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[float], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[float], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vcopyq_laneq_f32 (float32x4_t a, const int lane1, float32x4_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vcopyq_lane_u16 (uint16x8_t a, const int lane1, uint16x4_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vcopyq_laneq_u16 (uint16x8_t a, const int lane1, uint16x8_t b, const int lane2)
              A64: INS Vd.H[lane1], Vn.H[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vcopyq_lane_u32 (uint32x4_t a, const int lane1, uint32x2_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vcopyq_laneq_u32 (uint32x4_t a, const int lane1, uint32x4_t b, const int lane2)
              A64: INS Vd.S[lane1], Vn.S[lane2]
            """
            ...

        @staticmethod
        @overload
        def InsertSelectedScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector128[int], valueIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vcopyq_laneq_u64 (uint64x2_t a, const int lane1, uint64x2_t b, const int lane2)
              A64: INS Vd.D[lane1], Vn.D[lane2]
            """
            ...

        @staticmethod
        @overload
        def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vld1q_dup_f64 (float64_t const * ptr)
              A64: LD1R { Vt.2D }, [Xn]
            """
            ...

        @staticmethod
        @overload
        def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vld1q_dup_s64 (int64_t const * ptr)
              A64: LD1R { Vt.2D }, [Xn]
            """
            ...

        @staticmethod
        @overload
        def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vld1q_dup_u64 (uint64_t const * ptr)
              A64: LD1R { Vt.2D }, [Xn]
            """
            ...

        @staticmethod
        @overload
        def LoadPairScalarVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairScalarVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[float], System.Runtime.Intrinsics.Vector64[float]]:
            """A64: LDP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairScalarVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairScalarVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairScalarVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[float], System.Runtime.Intrinsics.Vector64[float]]:
            """A64: LDNP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairScalarVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[float], System.Runtime.Intrinsics.Vector128[float]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[float], System.Runtime.Intrinsics.Vector128[float]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[float], System.Runtime.Intrinsics.Vector128[float]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[float], System.Runtime.Intrinsics.Vector128[float]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector128NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]]:
            """A64: LDNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[float], System.Runtime.Intrinsics.Vector64[float]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[float], System.Runtime.Intrinsics.Vector64[float]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[float], System.Runtime.Intrinsics.Vector64[float]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[float], System.Runtime.Intrinsics.Vector64[float]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def LoadPairVector64NonTemporal(address: typing.Any) -> System.ValueTuple[System.Runtime.Intrinsics.Vector64[int], System.Runtime.Intrinsics.Vector64[int]]:
            """A64: LDNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        def Max(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmaxq_f64 (float64x2_t a, float64x2_t b)
              A64: FMAX Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vmaxv_u8 (uint8x8_t a)
              A64: UMAXV Bd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vmaxv_s16 (int16x4_t a)
              A64: SMAXV Hd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vmaxv_s8 (int8x8_t a)
              A64: SMAXV Bd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vmaxv_u16 (uint16x4_t a)
              A64: UMAXV Hd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vmaxvq_u8 (uint8x16_t a)
              A64: UMAXV Bd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vmaxvq_s16 (int16x8_t a)
              A64: SMAXV Hd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vmaxvq_s32 (int32x4_t a)
              A64: SMAXV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vmaxvq_s8 (int8x16_t a)
              A64: SMAXV Bd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmaxvq_f32 (float32x4_t a)
              A64: FMAXV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vmaxvq_u16 (uint16x8_t a)
              A64: UMAXV Hd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def MaxAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vmaxvq_u32 (uint32x4_t a)
              A64: UMAXV Sd, Vn.4S
            """
            ...

        @staticmethod
        def MaxNumber(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmaxnmq_f64 (float64x2_t a, float64x2_t b)
              A64: FMAXNM Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        def MaxNumberAcross(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmaxnmvq_f32 (float32x4_t a)
              A64: FMAXNMV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def MaxNumberPairwise(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vpmaxnm_f32 (float32x2_t a, float32x2_t b)
              A64: FMAXNMP Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def MaxNumberPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vpmaxnmq_f64 (float64x2_t a, float64x2_t b)
              A64: FMAXNMP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MaxNumberPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vpmaxnmq_f32 (float32x4_t a, float32x4_t b)
              A64: FMAXNMP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MaxNumberPairwiseScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vpmaxnms_f32 (float32x2_t a)
              A64: FMAXNMP Sd, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def MaxNumberPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vpmaxnmqd_f64 (float64x2_t a)
              A64: FMAXNMP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vpmaxq_u8 (uint8x16_t a, uint8x16_t b)
              A64: UMAXP Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vpmaxq_f64 (float64x2_t a, float64x2_t b)
              A64: FMAXP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vpmaxq_s16 (int16x8_t a, int16x8_t b)
              A64: SMAXP Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vpmaxq_s32 (int32x4_t a, int32x4_t b)
              A64: SMAXP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vpmaxq_s8 (int8x16_t a, int8x16_t b)
              A64: SMAXP Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vpmaxq_f32 (float32x4_t a, float32x4_t b)
              A64: FMAXP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vpmaxq_u16 (uint16x8_t a, uint16x8_t b)
              A64: UMAXP Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def MaxPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vpmaxq_u32 (uint32x4_t a, uint32x4_t b)
              A64: UMAXP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MaxPairwiseScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vpmaxs_f32 (float32x2_t a)
              A64: FMAXP Sd, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def MaxPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vpmaxqd_f64 (float64x2_t a)
              A64: FMAXP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def MaxScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vmax_f64 (float64x1_t a, float64x1_t b)
              A64: FMAX Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def MaxScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmaxs_f32 (float32_t a, float32_t b)
              A64: FMAX Sd, Sn, Sm
            The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
            """
            ...

        @staticmethod
        def Min(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vminq_f64 (float64x2_t a, float64x2_t b)
              A64: FMIN Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vminv_u8 (uint8x8_t a)
              A64: UMINV Bd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vminv_s16 (int16x4_t a)
              A64: SMINV Hd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vminv_s8 (int8x8_t a)
              A64: SMINV Bd, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vminv_u16 (uint16x4_t a)
              A64: UMINV Hd, Vn.4H
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vminvq_u8 (uint8x16_t a)
              A64: UMINV Bd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vminvq_s16 (int16x8_t a)
              A64: SMINV Hd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vaddvq_s32 (int32x4_t a)
              A64: SMINV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vminvq_s8 (int8x16_t a)
              A64: SMINV Bd, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vminvq_f32 (float32x4_t a)
              A64: FMINV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vminvq_u16 (uint16x8_t a)
              A64: UMINV Hd, Vn.8H
            """
            ...

        @staticmethod
        @overload
        def MinAcross(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vminvq_u32 (uint32x4_t a)
              A64: UMINV Sd, Vn.4S
            """
            ...

        @staticmethod
        def MinNumber(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vminnmq_f64 (float64x2_t a, float64x2_t b)
              A64: FMINNM Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        def MinNumberAcross(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vminnmvq_f32 (float32x4_t a)
              A64: FMINNMV Sd, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def MinNumberPairwise(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vpminnm_f32 (float32x2_t a, float32x2_t b)
              A64: FMINNMP Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def MinNumberPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vpminnmq_f64 (float64x2_t a, float64x2_t b)
              A64: FMINNMP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MinNumberPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vpminnmq_f32 (float32x4_t a, float32x4_t b)
              A64: FMINNMP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MinNumberPairwiseScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vpminnms_f32 (float32x2_t a)
              A64: FMINNMP Sd, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def MinNumberPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vpminnmqd_f64 (float64x2_t a)
              A64: FMINNMP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vpminq_u8 (uint8x16_t a, uint8x16_t b)
              A64: UMINP Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vpminq_f64 (float64x2_t a, float64x2_t b)
              A64: FMINP Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vpminq_s16 (int16x8_t a, int16x8_t b)
              A64: SMINP Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vpminq_s32 (int32x4_t a, int32x4_t b)
              A64: SMINP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vpminq_s8 (int8x16_t a, int8x16_t b)
              A64: SMINP Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vpminq_f32 (float32x4_t a, float32x4_t b)
              A64: FMINP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vpminq_u16 (uint16x8_t a, uint16x8_t b)
              A64: UMINP Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def MinPairwise(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vpminq_u32 (uint32x4_t a, uint32x4_t b)
              A64: UMINP Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def MinPairwiseScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vpmins_f32 (float32x2_t a)
              A64: FMINP Sd, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def MinPairwiseScalar(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vpminqd_f64 (float64x2_t a)
              A64: FMINP Dd, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def MinScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vmin_f64 (float64x1_t a, float64x1_t b)
              A64: FMIN Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def MinScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmins_f32 (float32_t a, float32_t b)
              A64: FMIN Sd, Sn, Sm
            The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
            """
            ...

        @staticmethod
        def Multiply(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmulq_f64 (float64x2_t a, float64x2_t b)
              A64: FMUL Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmulq_n_f64 (float64x2_t a, float64_t b)
              A64: FMUL Vd.2D, Vn.2D, Vm.D[0]
            """
            ...

        @staticmethod
        def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmulq_laneq_f64 (float64x2_t a, float64x2_t v, const int lane)
              A64: FMUL Vd.2D, Vn.2D, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingSaturateHighScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqdmulhh_s16 (int16_t a, int16_t b)
              A64: SQDMULH Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingSaturateHighScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmulhs_s32 (int32_t a, int32_t b)
              A64: SQDMULH Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqdmulhh_lane_s16 (int16_t a, int16x4_t v, const int lane)
              A64: SQDMULH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqdmulhh_laneq_s16 (int16_t a, int16x8_t v, const int lane)
              A64: SQDMULH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmulhs_lane_s32 (int32_t a, int32x2_t v, const int lane)
              A64: SQDMULH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmulhs_laneq_s32 (int32_t a, int32x4_t v, const int lane)
              A64: SQDMULH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningAndAddSaturateScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmlalh_s16 (int32_t a, int16_t b, int16_t c)
              A64: SQDMLAL Sd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningAndAddSaturateScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmlals_s32 (int64_t a, int32_t b, int32_t c)
              A64: SQDMLAL Dd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningAndSubtractSaturateScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmlslh_s16 (int32_t a, int16_t b, int16_t c)
              A64: SQDMLSL Sd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningAndSubtractSaturateScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmlsls_s32 (int64_t a, int32_t b, int32_t c)
              A64: SQDMLSL Dd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmullh_s16 (int16_t a, int16_t b)
              A64: SQDMULL Sd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmulls_s32 (int32_t a, int32_t b)
              A64: SQDMULL Dd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningSaturateScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmullh_lane_s16 (int16_t a, int16x4_t v, const int lane)
              A64: SQDMULL Sd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningSaturateScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmullh_laneq_s16 (int16_t a, int16x8_t v, const int lane)
              A64: SQDMULL Sd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningSaturateScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmulls_lane_s32 (int32_t a, int32x2_t v, const int lane)
              A64: SQDMULL Dd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningSaturateScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmulls_laneq_s32 (int32_t a, int32x4_t v, const int lane)
              A64: SQDMULL Dd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmlalh_lane_s16 (int32_t a, int16_t b, int16x4_t v, const int lane)
              A64: SQDMLAL Sd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmlalh_laneq_s16 (int32_t a, int16_t b, int16x8_t v, const int lane)
              A64: SQDMLAL Sd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmlals_lane_s32 (int64_t a, int32_t b, int32x2_t v, const int lane)
              A64: SQDMLAL Dd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmlals_laneq_s32 (int64_t a, int32_t b, int32x4_t v, const int lane)
              A64: SQDMLAL Dd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmlslh_lane_s16 (int32_t a, int16_t b, int16x4_t v, const int lane)
              A64: SQDMLSL Sd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqdmlslh_laneq_s16 (int32_t a, int16_t b, int16x8_t v, const int lane)
              A64: SQDMLSL Sd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmlsls_lane_s32 (int64_t a, int32_t b, int32x2_t v, const int lane)
              A64: SQDMLSL Dd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyDoublingWideningScalarBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqdmlsls_laneq_s32 (int64_t a, int32_t b, int32x4_t v, const int lane)
              A64: SQDMLSL Dd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtended(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vmulx_f32 (float32x2_t a, float32x2_t b)
              A64: FMULX Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtended(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmulxq_f64 (float64x2_t a, float64x2_t b)
              A64: FMULX Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtended(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vmulxq_f32 (float32x4_t a, float32x4_t b)
              A64: FMULX Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        def MultiplyExtendedByScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmulxq_lane_f64 (float64x2_t a, float64x1_t v, const int lane)
              A64: FMULX Vd.2D, Vn.2D, Vm.D[0]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vmulx_lane_f32 (float32x2_t a, float32x2_t v, const int lane)
              A64: FMULX Vd.2S, Vn.2S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vmulx_laneq_f32 (float32x2_t a, float32x4_t v, const int lane)
              A64: FMULX Vd.2S, Vn.2S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vmulxq_laneq_f64 (float64x2_t a, float64x2_t v, const int lane)
              A64: FMULX Vd.2D, Vn.2D, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vmulxq_lane_f32 (float32x4_t a, float32x2_t v, const int lane)
              A64: FMULX Vd.4S, Vn.4S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vmulxq_laneq_f32 (float32x4_t a, float32x4_t v, const int lane)
              A64: FMULX Vd.4S, Vn.4S, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vmulx_f64 (float64x1_t a, float64x1_t b)
              A64: FMULX Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmulxs_f32 (float32_t a, float32_t b)
              A64: FMULX Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vmulxd_laneq_f64 (float64_t a, float64x2_t v, const int lane)
              A64: FMULX Dd, Dn, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmulxs_lane_f32 (float32_t a, float32x2_t v, const int lane)
              A64: FMULX Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyExtendedScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vmulxs_laneq_f32 (float32_t a, float32x4_t v, const int lane)
              A64: FMULX Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingSaturateHighScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmulhh_s16 (int16_t a, int16_t b)
              A64: SQRDMULH Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingSaturateHighScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmulhs_s32 (int32_t a, int32_t b)
              A64: SQRDMULH Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmulhh_lane_s16 (int16_t a, int16x4_t v, const int lane)
              A64: SQRDMULH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmulhh_laneq_s16 (int16_t a, int16x8_t v, const int lane)
              A64: SQRDMULH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmulhs_lane_s32 (int32_t a, int32x2_t v, const int lane)
              A64: SQRDMULH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmulhs_laneq_s32 (int32_t a, int32x4_t v, const int lane)
              A64: SQRDMULH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        def MultiplyScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vmuld_laneq_f64 (float64_t a, float64x2_t v, const int lane)
              A64: FMUL Dd, Dn, Vm.D[lane]
            """
            ...

        @staticmethod
        @overload
        def Negate(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vnegq_f64 (float64x2_t a)
              A64: FNEG Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def Negate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vnegq_s64 (int64x2_t a)
              A64: NEG Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def NegateSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vqnegq_s64 (int64x2_t a)
              A64: SQNEG Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def NegateSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqnegh_s16 (int16_t a)
              A64: SQNEG Hd, Hn
            """
            ...

        @staticmethod
        @overload
        def NegateSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqnegs_s32 (int32_t a)
              A64: SQNEG Sd, Sn
            """
            ...

        @staticmethod
        @overload
        def NegateSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64_t vqnegd_s64 (int64_t a)
              A64: SQNEG Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def NegateSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqnegb_s8 (int8_t a)
              A64: SQNEG Bd, Bn
            """
            ...

        @staticmethod
        def NegateScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int64x1_t vneg_s64 (int64x1_t a)
              A64: NEG Dd, Dn
            """
            ...

        @staticmethod
        def ReciprocalEstimate(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrecpeq_f64 (float64x2_t a)
              A64: FRECPE Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def ReciprocalEstimateScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vrecpe_f64 (float64x1_t a)
              A64: FRECPE Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def ReciprocalEstimateScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vrecpes_f32 (float32_t a)
              A64: FRECPE Sd, Sn
            """
            ...

        @staticmethod
        @overload
        def ReciprocalExponentScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64_t vrecpxd_f64 (float64_t a)
              A64: FRECPX Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def ReciprocalExponentScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vrecpxs_f32 (float32_t a)
              A64: FRECPX Sd, Sn
            """
            ...

        @staticmethod
        def ReciprocalSquareRootEstimate(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrsqrteq_f64 (float64x2_t a)
              A64: FRSQRTE Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def ReciprocalSquareRootEstimateScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vrsqrte_f64 (float64x1_t a)
              A64: FRSQRTE Dd, Dn
            """
            ...

        @staticmethod
        @overload
        def ReciprocalSquareRootEstimateScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vrsqrtes_f32 (float32_t a)
              A64: FRSQRTE Sd, Sn
            """
            ...

        @staticmethod
        def ReciprocalSquareRootStep(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrsqrtsq_f64 (float64x2_t a, float64x2_t b)
              A64: FRSQRTS Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ReciprocalSquareRootStepScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vrsqrts_f64 (float64x1_t a, float64x1_t b)
              A64: FRSQRTS Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def ReciprocalSquareRootStepScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vrsqrtss_f32 (float32_t a, float32_t b)
              A64: FRSQRTS Sd, Sn, Sm
            """
            ...

        @staticmethod
        def ReciprocalStep(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrecpsq_f64 (float64x2_t a, float64x2_t b)
              A64: FRECPS Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ReciprocalStepScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float64x1_t vrecps_f64 (float64x1_t a, float64x1_t b)
              A64: FRECPS Dd, Dn, Dm
            """
            ...

        @staticmethod
        @overload
        def ReciprocalStepScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32_t vrecpss_f32 (float32_t a, float32_t b)
              A64: FRECPS Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ReverseElementBits(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vrbit_u8 (uint8x8_t a)
              A64: RBIT Vd.8B, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def ReverseElementBits(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vrbit_s8 (int8x8_t a)
              A64: RBIT Vd.8B, Vn.8B
            """
            ...

        @staticmethod
        @overload
        def ReverseElementBits(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vrbitq_u8 (uint8x16_t a)
              A64: RBIT Vd.16B, Vn.16B
            """
            ...

        @staticmethod
        @overload
        def ReverseElementBits(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vrbitq_s8 (int8x16_t a)
              A64: RBIT Vd.16B, Vn.16B
            """
            ...

        @staticmethod
        def RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndaq_f64 (float64x2_t a)
              A64: FRINTA Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def RoundToNearest(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndnq_f64 (float64x2_t a)
              A64: FRINTN Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndmq_f64 (float64x2_t a)
              A64: FRINTM Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndpq_f64 (float64x2_t a)
              A64: FRINTP Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        def RoundToZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vrndq_f64 (float64x2_t a)
              A64: FRINTZ Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def ShiftArithmeticRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrshlh_s16 (int16_t a, int16_t b)
              A64: SQRSHL Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def ShiftArithmeticRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrshls_s32 (int32_t a, int32_t b)
              A64: SQRSHL Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ShiftArithmeticRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqrshlb_s8 (int8_t a, int8_t b)
              A64: SQRSHL Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def ShiftArithmeticSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqshlh_s16 (int16_t a, int16_t b)
              A64: SQSHL Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def ShiftArithmeticSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqshls_s32 (int32_t a, int32_t b)
              A64: SQSHL Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ShiftArithmeticSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqshlb_s8 (int8_t a, int8_t b)
              A64: SQSHL Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshlb_n_u8 (uint8_t a, const int n)
              A64: UQSHL Bd, Bn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqshlh_n_s16 (int16_t a, const int n)
              A64: SQSHL Hd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqshls_n_s32 (int32_t a, const int n)
              A64: SQSHL Sd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqshlb_n_s8 (int8_t a, const int n)
              A64: SQSHL Bd, Bn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshlh_n_u16 (uint16_t a, const int n)
              A64: UQSHL Hd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshls_n_u32 (uint32_t a, const int n)
              A64: UQSHL Sd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshluh_n_s16 (int16_t a, const int n)
              A64: SQSHLU Hd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshlus_n_s32 (int32_t a, const int n)
              A64: SQSHLU Sd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLeftLogicalSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshlub_n_s8 (int8_t a, const int n)
              A64: SQSHLU Bd, Bn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqrshlb_u8 (uint8_t a, int8_t b)
              A64: UQRSHL Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqrshlh_u16 (uint16_t a, int16_t b)
              A64: UQRSHL Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqrshls_u32 (uint32_t a, int32_t b)
              A64: UQRSHL Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqrshlb_u8 (uint8_t a, int8_t b)
              A64: UQRSHL Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqrshlh_u16 (uint16_t a, int16_t b)
              A64: UQRSHL Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqrshls_u32 (uint32_t a, int32_t b)
              A64: UQRSHL Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshlb_u8 (uint8_t a, int8_t b)
              A64: UQSHL Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshlh_u16 (uint16_t a, int16_t b)
              A64: UQSHL Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshls_u32 (uint32_t a, int32_t b)
              A64: UQSHL Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshlb_u8 (uint8_t a, int8_t b)
              A64: UQSHL Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshlh_u16 (uint16_t a, int16_t b)
              A64: UQSHL Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshls_u32 (uint32_t a, int32_t b)
              A64: UQSHL Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqshrns_n_s32 (int32_t a, const int n)
              A64: SQSHRN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqshrnd_n_s64 (int64_t a, const int n)
              A64: SQSHRN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqshrnh_n_s16 (int16_t a, const int n)
              A64: SQSHRN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshrunh_n_s16 (int16_t a, const int n)
              A64: SQSHRUN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshruns_n_s32 (int32_t a, const int n)
              A64: SQSHRUN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshrund_n_s64 (int64_t a, const int n)
              A64: SQSHRUN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrshrns_n_s32 (int32_t a, const int n)
              A64: SQRSHRN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrshrnd_n_s64 (int64_t a, const int n)
              A64: SQRSHRN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqrshrnh_n_s16 (int16_t a, const int n)
              A64: SQRSHRN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqrshrunh_n_s16 (int16_t a, const int n)
              A64: SQRSHRUN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqrshruns_n_s32 (int32_t a, const int n)
              A64: SQRSHRUN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqrshrund_n_s64 (int64_t a, const int n)
              A64: SQRSHRUN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshrnh_n_u16 (uint16_t a, const int n)
              A64: UQSHRN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshrns_n_u32 (uint32_t a, const int n)
              A64: UQSHRN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshrnd_n_u64 (uint64_t a, const int n)
              A64: UQSHRN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqshrnh_n_u16 (uint16_t a, const int n)
              A64: UQSHRN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqshrns_n_u32 (uint32_t a, const int n)
              A64: UQSHRN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqshrnd_n_u64 (uint64_t a, const int n)
              A64: UQSHRN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqrshrnh_n_u16 (uint16_t a, const int n)
              A64: UQRSHRN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqrshrns_n_u32 (uint32_t a, const int n)
              A64: UQRSHRN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqrshrnd_n_u64 (uint64_t a, const int n)
              A64: UQRSHRN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqrshrnh_n_u16 (uint16_t a, const int n)
              A64: UQRSHRN Bd, Hn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqrshrns_n_u32 (uint32_t a, const int n)
              A64: UQRSHRN Hd, Sn, #n
            """
            ...

        @staticmethod
        @overload
        def ShiftRightLogicalRoundedNarrowingSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqrshrnd_n_u64 (uint64_t a, const int n)
              A64: UQRSHRN Sd, Dn, #n
            """
            ...

        @staticmethod
        @overload
        def Sqrt(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vsqrt_f32 (float32x2_t a)
              A64: FSQRT Vd.2S, Vn.2S
            """
            ...

        @staticmethod
        @overload
        def Sqrt(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vsqrtq_f64 (float64x2_t a)
              A64: FSQRT Vd.2D, Vn.2D
            """
            ...

        @staticmethod
        @overload
        def Sqrt(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vsqrtq_f32 (float32x4_t a)
              A64: FSQRT Vd.4S, Vn.4S
            """
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[float], value2: System.Runtime.Intrinsics.Vector64[float]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[float], value2: System.Runtime.Intrinsics.Vector64[float]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[float], value2: System.Runtime.Intrinsics.Vector128[float]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[float], value2: System.Runtime.Intrinsics.Vector128[float]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePair(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[float], value2: System.Runtime.Intrinsics.Vector64[float]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[float], value2: System.Runtime.Intrinsics.Vector64[float]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP Dt1, Dt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[float], value2: System.Runtime.Intrinsics.Vector128[float]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[float], value2: System.Runtime.Intrinsics.Vector128[float]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector128[int], value2: System.Runtime.Intrinsics.Vector128[int]) -> None:
            """A64: STNP Qt1, Qt2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairScalar(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairScalar(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[float], value2: System.Runtime.Intrinsics.Vector64[float]) -> None:
            """A64: STP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairScalar(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairScalarNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairScalarNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[float], value2: System.Runtime.Intrinsics.Vector64[float]) -> None:
            """A64: STNP St1, St2, [Xn]"""
            ...

        @staticmethod
        @overload
        def StorePairScalarNonTemporal(address: typing.Any, value1: System.Runtime.Intrinsics.Vector64[int], value2: System.Runtime.Intrinsics.Vector64[int]) -> None:
            """A64: STNP St1, St2, [Xn]"""
            ...

        @staticmethod
        def Subtract(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vsubq_f64 (float64x2_t a, float64x2_t b)
              A64: FSUB Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8_t vqsubb_u8 (uint8_t a, uint8_t b)
              A64: UQSUB Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqsubh_s16 (int16_t a, int16_t b)
              A64: SQSUB Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqsubs_s32 (int32_t a, int32_t b)
              A64: SQSUB Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8_t vqsubb_s8 (int8_t a, int8_t b)
              A64: SQSUB Bd, Bn, Bm
            """
            ...

        @staticmethod
        @overload
        def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16_t vqsubh_u16 (uint16_t a, uint16_t b)
              A64: UQSUB Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32_t vqsubs_u32 (uint32_t a, uint32_t b)
              A64: UQSUB Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vtrn1_u8(uint8x8_t a, uint8x8_t b)
              A64: TRN1 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vtrn1_s16(int16x4_t a, int16x4_t b)
              A64: TRN1 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vtrn1_s32(int32x2_t a, int32x2_t b)
              A64: TRN1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vtrn1_s8(int8x8_t a, int8x8_t b)
              A64: TRN1 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vtrn1_f32(float32x2_t a, float32x2_t b)
              A64: TRN1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vtrn1_u16(uint16x4_t a, uint16x4_t b)
              A64: TRN1 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vtrn1_u32(uint32x2_t a, uint32x2_t b)
              A64: TRN1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vtrn1q_u8(uint8x16_t a, uint8x16_t b)
              A64: TRN1 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vtrn1q_f64(float64x2_t a, float64x2_t b)
              A64: TRN1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vtrn1q_s16(int16x8_t a, int16x8_t b)
              A64: TRN1 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vtrn1q_s32(int32x4_t a, int32x4_t b)
              A64: TRN1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vtrn1q_s64(int64x2_t a, int64x2_t b)
              A64: TRN1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vtrn1q_u8(int8x16_t a, int8x16_t b)
              A64: TRN1 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vtrn1q_f32(float32x4_t a, float32x4_t b)
              A64: TRN1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vtrn1q_u16(uint16x8_t a, uint16x8_t b)
              A64: TRN1 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vtrn1q_u32(uint32x4_t a, uint32x4_t b)
              A64: TRN1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def TransposeEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vtrn1q_u64(uint64x2_t a, uint64x2_t b)
              A64: TRN1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vtrn2_u8(uint8x8_t a, uint8x8_t b)
              A64: TRN2 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vtrn2_s16(int16x4_t a, int16x4_t b)
              A64: TRN2 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vtrn2_s32(int32x2_t a, int32x2_t b)
              A64: TRN2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vtrn2_s8(int8x8_t a, int8x8_t b)
              A64: TRN2 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vtrn2_f32(float32x2_t a, float32x2_t b)
              A64: TRN2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vtrn2_u16(uint16x4_t a, uint16x4_t b)
              A64: TRN2 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vtrn2_u32(uint32x2_t a, uint32x2_t b)
              A64: TRN2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vtrn2q_u8(uint8x16_t a, uint8x16_t b)
              A64: TRN2 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vtrn2q_f64(float64x2_t a, float64x2_t b)
              A64: TRN2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vtrn2q_s16(int16x8_t a, int16x8_t b)
              A64: TRN2 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vtrn2q_s32(int32x4_t a, int32x4_t b)
              A64: TRN2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vtrn2q_s64(int64x2_t a, int64x2_t b)
              A64: TRN2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vtrn2q_u8(int8x16_t a, int8x16_t b)
              A64: TRN2 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vtrn2q_f32(float32x4_t a, float32x4_t b)
              A64: TRN2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vtrn2q_u16(uint16x8_t a, uint16x8_t b)
              A64: TRN2 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vtrn2q_u32(uint32x4_t a, uint32x4_t b)
              A64: TRN2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def TransposeOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vtrn2q_u64(uint64x2_t a, uint64x2_t b)
              A64: TRN2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vuzp1_u8(uint8x8_t a, uint8x8_t b)
              A64: UZP1 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vuzp1_s16(int16x4_t a, int16x4_t b)
              A64: UZP1 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vuzp1_s32(int32x2_t a, int32x2_t b)
              A64: UZP1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vuzp1_s8(int8x8_t a, int8x8_t b)
              A64: UZP1 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vuzp1_f32(float32x2_t a, float32x2_t b)
              A64: UZP1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vuzp1_u16(uint16x4_t a, uint16x4_t b)
              A64: UZP1 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vuzp1_u32(uint32x2_t a, uint32x2_t b)
              A64: UZP1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vuzp1q_u8(uint8x16_t a, uint8x16_t b)
              A64: UZP1 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vuzp1q_f64(float64x2_t a, float64x2_t b)
              A64: UZP1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vuzp1q_s16(int16x8_t a, int16x8_t b)
              A64: UZP1 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vuzp1q_s32(int32x4_t a, int32x4_t b)
              A64: UZP1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vuzp1q_s64(int64x2_t a, int64x2_t b)
              A64: UZP1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vuzp1q_u8(int8x16_t a, int8x16_t b)
              A64: UZP1 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vuzp1q_f32(float32x4_t a, float32x4_t b)
              A64: UZP1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vuzp1q_u16(uint16x8_t a, uint16x8_t b)
              A64: UZP1 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vuzp1q_u32(uint32x4_t a, uint32x4_t b)
              A64: UZP1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def UnzipEven(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vuzp1q_u64(uint64x2_t a, uint64x2_t b)
              A64: UZP1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vuzp2_u8(uint8x8_t a, uint8x8_t b)
              A64: UZP2 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vuzp2_s16(int16x4_t a, int16x4_t b)
              A64: UZP2 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vuzp2_s32(int32x2_t a, int32x2_t b)
              A64: UZP2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vuzp2_s8(int8x8_t a, int8x8_t b)
              A64: UZP2 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vuzp2_f32(float32x2_t a, float32x2_t b)
              A64: UZP2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vuzp2_u16(uint16x4_t a, uint16x4_t b)
              A64: UZP2 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vuzp2_u32(uint32x2_t a, uint32x2_t b)
              A64: UZP2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vuzp2q_u8(uint8x16_t a, uint8x16_t b)
              A64: UZP2 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vuzp2q_f64(float64x2_t a, float64x2_t b)
              A64: UZP2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vuzp2q_s16(int16x8_t a, int16x8_t b)
              A64: UZP2 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vuzp2q_s32(int32x4_t a, int32x4_t b)
              A64: UZP2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vuzp2q_s64(int64x2_t a, int64x2_t b)
              A64: UZP2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vuzp2q_u8(int8x16_t a, int8x16_t b)
              A64: UZP2 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vuzp2q_f32(float32x4_t a, float32x4_t b)
              A64: UZP2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vuzp2q_u16(uint16x8_t a, uint16x8_t b)
              A64: UZP2 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vuzp2q_u32(uint32x4_t a, uint32x4_t b)
              A64: UZP2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def UnzipOdd(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vuzp2q_u64(uint64x2_t a, uint64x2_t b)
              A64: UZP2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqvtbl1q_u8(uint8x16_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqvtbl1q_s8(int8x16_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqtbl2q_u8(uint8x16x2_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B, Vn+1.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqtbl2q_s8(int8x16x2_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B, Vn+1.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqtbl3q_u8(uint8x16x3_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqtbl3q_s8(int8x16x3_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqtbl4q_u8(uint8x16x4_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqtbl4q_s8(int8x16x4_t t, uint8x16_t idx)
              A64: TBL Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqvtbx1q_u8(uint8x16_t r, int8x16_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqvtbx1q_s8(int8x16_t r, int8x16_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqtbx2q_u8(uint8x16x2_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B, Vn+1.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqtbx2q_s8(int8x16x2_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B, Vn+1.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqtbx3q_u8(uint8x16x3_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqtbx3q_s8(int8x16x3_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vqtbx4q_u8(uint8x16x4_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector128[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vqtbx4q_s8(int8x16x4_t t, uint8x16_t idx)
              A64: TBX Vd.16B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vzip2_u8(uint8x8_t a, uint8x8_t b)
              A64: ZIP2 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vzip2_s16(int16x4_t a, int16x4_t b)
              A64: ZIP2 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vzip2_s32(int32x2_t a, int32x2_t b)
              A64: ZIP2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vzip2_s8(int8x8_t a, int8x8_t b)
              A64: ZIP2 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vzip2_f32(float32x2_t a, float32x2_t b)
              A64: ZIP2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vzip2_u16(uint16x4_t a, uint16x4_t b)
              A64: ZIP2 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vzip2_u32(uint32x2_t a, uint32x2_t b)
              A64: ZIP2 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vzip2q_u8(uint8x16_t a, uint8x16_t b)
              A64: ZIP2 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vzip2q_f64(float64x2_t a, float64x2_t b)
              A64: ZIP2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vzip2q_s16(int16x8_t a, int16x8_t b)
              A64: ZIP2 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vzip2q_s32(int32x4_t a, int32x4_t b)
              A64: ZIP2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vzip2q_s64(int64x2_t a, int64x2_t b)
              A64: ZIP2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vzip2q_u8(int8x16_t a, int8x16_t b)
              A64: ZIP2 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vzip2q_f32(float32x4_t a, float32x4_t b)
              A64: ZIP2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vzip2q_u16(uint16x8_t a, uint16x8_t b)
              A64: ZIP2 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vzip2q_u32(uint32x4_t a, uint32x4_t b)
              A64: ZIP2 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def ZipHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vzip2q_u64(uint64x2_t a, uint64x2_t b)
              A64: ZIP2 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint8x8_t vzip1_u8(uint8x8_t a, uint8x8_t b)
              A64: ZIP1 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16x4_t vzip1_s16(int16x4_t a, int16x4_t b)
              A64: ZIP1 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32x2_t vzip1_s32(int32x2_t a, int32x2_t b)
              A64: ZIP1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int8x8_t vzip1_s8(int8x8_t a, int8x8_t b)
              A64: ZIP1 Vd.8B, Vn.8B, Vm.8B
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
            """
            float32x2_t vzip1_f32(float32x2_t a, float32x2_t b)
              A64: ZIP1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint16x4_t vzip1_u16(uint16x4_t a, uint16x4_t b)
              A64: ZIP1 Vd.4H, Vn.4H, Vm.4H
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            uint32x2_t vzip1_u32(uint32x2_t a, uint32x2_t b)
              A64: ZIP1 Vd.2S, Vn.2S, Vm.2S
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint8x16_t vzip1q_u8(uint8x16_t a, uint8x16_t b)
              A64: ZIP1 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float64x2_t vzip1q_f64(float64x2_t a, float64x2_t b)
              A64: ZIP1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int16x8_t vzip1q_s16(int16x8_t a, int16x8_t b)
              A64: ZIP1 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int32x4_t vzip1q_s32(int32x4_t a, int32x4_t b)
              A64: ZIP1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int64x2_t vzip1q_s64(int64x2_t a, int64x2_t b)
              A64: ZIP1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            int8x16_t vzip1q_u8(int8x16_t a, int8x16_t b)
              A64: ZIP1 Vd.16B, Vn.16B, Vm.16B
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
            """
            float32x4_t vzip1q_f32(float32x4_t a, float32x4_t b)
              A64: ZIP1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint16x8_t vzip1q_u16(uint16x8_t a, uint16x8_t b)
              A64: ZIP1 Vd.8H, Vn.8H, Vm.8H
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint32x4_t vzip1q_u32(uint32x4_t a, uint32x4_t b)
              A64: ZIP1 Vd.4S, Vn.4S, Vm.4S
            """
            ...

        @staticmethod
        @overload
        def ZipLow(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
            """
            uint64x2_t vzip1q_u64(uint64x2_t a, uint64x2_t b)
              A64: ZIP1 Vd.2D, Vn.2D, Vm.2D
            """
            ...

    IsSupported: bool

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vabs_s16 (int16x4_t a)
          A32: VABS.S16 Dd, Dm
          A64: ABS Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vabs_s32 (int32x2_t a)
          A32: VABS.S32 Dd, Dm
          A64: ABS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vabs_s8 (int8x8_t a)
          A32: VABS.S8 Dd, Dm
          A64: ABS Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vabs_f32 (float32x2_t a)
          A32: VABS.F32 Dd, Dm
          A64: FABS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabsq_s16 (int16x8_t a)
          A32: VABS.S16 Qd, Qm
          A64: ABS Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabsq_s32 (int32x4_t a)
          A32: VABS.S32 Qd, Qm
          A64: ABS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vabsq_s8 (int8x16_t a)
          A32: VABS.S8 Qd, Qm
          A64: ABS Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Abs(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vabsq_f32 (float32x4_t a)
          A32: VABS.F32 Qd, Qm
          A64: FABS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcagt_f32 (float32x2_t a, float32x2_t b)
          A32: VACGT.F32 Dd, Dn, Dm
          A64: FACGT Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcagtq_f32 (float32x4_t a, float32x4_t b)
          A32: VACGT.F32 Qd, Qn, Qm
          A64: FACGT Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcage_f32 (float32x2_t a, float32x2_t b)
          A32: VACGE.F32 Dd, Dn, Dm
          A64: FACGE Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcageq_f32 (float32x4_t a, float32x4_t b)
          A32: VACGE.F32 Qd, Qn, Qm
          A64: FACGE Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareLessThan(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcalt_f32 (float32x2_t a, float32x2_t b)
          A32: VACLT.F32 Dd, Dn, Dm
          A64: FACGT Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareLessThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcaltq_f32 (float32x4_t a, float32x4_t b)
          A32: VACLT.F32 Qd, Qn, Qm
          A64: FACGT Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcale_f32 (float32x2_t a, float32x2_t b)
          A32: VACLE.F32 Dd, Dn, Dm
          A64: FACGE Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteCompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcaleq_f32 (float32x4_t a, float32x4_t b)
          A32: VACLE.F32 Qd, Qn, Qm
          A64: FACGE Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vabd_u8 (uint8x8_t a, uint8x8_t b)
          A32: VABD.U8 Dd, Dn, Dm
          A64: UABD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vabd_s16 (int16x4_t a, int16x4_t b)
          A32: VABD.S16 Dd, Dn, Dm
          A64: SABD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vabd_s32 (int32x2_t a, int32x2_t b)
          A32: VABD.S32 Dd, Dn, Dm
          A64: SABD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vabd_s8 (int8x8_t a, int8x8_t b)
          A32: VABD.S8 Dd, Dn, Dm
          A64: SABD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vabd_f32 (float32x2_t a, float32x2_t b)
          A32: VABD.F32 Dd, Dn, Dm
          A64: FABD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vabd_u16 (uint16x4_t a, uint16x4_t b)
          A32: VABD.U16 Dd, Dn, Dm
          A64: UABD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vabd_u32 (uint32x2_t a, uint32x2_t b)
          A32: VABD.U32 Dd, Dn, Dm
          A64: UABD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vabdq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VABD.U8 Qd, Qn, Qm
          A64: UABD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabdq_s16 (int16x8_t a, int16x8_t b)
          A32: VABD.S16 Qd, Qn, Qm
          A64: SABD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabdq_s32 (int32x4_t a, int32x4_t b)
          A32: VABD.S32 Qd, Qn, Qm
          A64: SABD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vabdq_s8 (int8x16_t a, int8x16_t b)
          A32: VABD.S8 Qd, Qn, Qm
          A64: SABD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vabdq_f32 (float32x4_t a, float32x4_t b)
          A32: VABD.F32 Qd, Qn, Qm
          A64: FABD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vabdq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VABD.U16 Qd, Qn, Qm
          A64: UABD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifference(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vabdq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VABD.U32 Qd, Qn, Qm
          A64: UABD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vaba_u8 (uint8x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VABA.U8 Dd, Dn, Dm
          A64: UABA Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vaba_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VABA.S16 Dd, Dn, Dm
          A64: SABA Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vaba_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VABA.S32 Dd, Dn, Dm
          A64: SABA Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vaba_s8 (int8x8_t a, int8x8_t b, int8x8_t c)
          A32: VABA.S8 Dd, Dn, Dm
          A64: SABA Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vaba_u16 (uint16x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VABA.U16 Dd, Dn, Dm
          A64: UABA Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vaba_u32 (uint32x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VABA.U32 Dd, Dn, Dm
          A64: UABA Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vabaq_u8 (uint8x16_t a, uint8x16_t b, uint8x16_t c)
          A32: VABA.U8 Qd, Qn, Qm
          A64: UABA Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabaq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VABA.S16 Qd, Qn, Qm
          A64: SABA Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabaq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VABA.S32 Qd, Qn, Qm
          A64: SABA Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vabaq_s8 (int8x16_t a, int8x16_t b, int8x16_t c)
          A32: VABA.S8 Qd, Qn, Qm
          A64: SABA Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vabaq_u16 (uint16x8_t a, uint16x8_t b, uint16x8_t c)
          A32: VABA.U16 Qd, Qn, Qm
          A64: UABA Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vabaq_u32 (uint32x4_t a, uint32x4_t b, uint32x4_t c)
          A32: VABA.U32 Qd, Qn, Qm
          A64: UABA Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vabdl_u8 (uint8x8_t a, uint8x8_t b)
          A32: VABDL.U8 Qd, Dn, Dm
          A64: UABDL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabdl_s16 (int16x4_t a, int16x4_t b)
          A32: VABDL.S16 Qd, Dn, Dm
          A64: SABDL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vabdl_s32 (int32x2_t a, int32x2_t b)
          A32: VABDL.S32 Qd, Dn, Dm
          A64: SABDL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabdl_s8 (int8x8_t a, int8x8_t b)
          A32: VABDL.S8 Qd, Dn, Dm
          A64: SABDL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vabdl_u16 (uint16x4_t a, uint16x4_t b)
          A32: VABDL.U16 Qd, Dn, Dm
          A64: UABDL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vabdl_u32 (uint32x2_t a, uint32x2_t b)
          A32: VABDL.U32 Qd, Dn, Dm
          A64: UABDL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vabal_u8 (uint16x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VABAL.U8 Qd, Dn, Dm
          A64: UABAL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabal_s16 (int32x4_t a, int16x4_t b, int16x4_t c)
          A32: VABAL.S16 Qd, Dn, Dm
          A64: SABAL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vabal_s32 (int64x2_t a, int32x2_t b, int32x2_t c)
          A32: VABAL.S32 Qd, Dn, Dm
          A64: SABAL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabal_s8 (int16x8_t a, int8x8_t b, int8x8_t c)
          A32: VABAL.S8 Qd, Dn, Dm
          A64: SABAL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vabal_u16 (uint32x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VABAL.U16 Qd, Dn, Dm
          A64: UABAL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vabal_u32 (uint64x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VABAL.U32 Qd, Dn, Dm
          A64: UABAL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vabdl_high_u8 (uint8x16_t a, uint8x16_t b)
          A32: VABDL.U8 Qd, Dn+1, Dm+1
          A64: UABDL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabdl_high_s16 (int16x8_t a, int16x8_t b)
          A32: VABDL.S16 Qd, Dn+1, Dm+1
          A64: SABDL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vabdl_high_s32 (int32x4_t a, int32x4_t b)
          A32: VABDL.S32 Qd, Dn+1, Dm+1
          A64: SABDL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabdl_high_s8 (int8x16_t a, int8x16_t b)
          A32: VABDL.S8 Qd, Dn+1, Dm+1
          A64: SABDL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vabdl_high_u16 (uint16x8_t a, uint16x8_t b)
          A32: VABDL.U16 Qd, Dn+1, Dm+1
          A64: UABDL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vabdl_high_u32 (uint32x4_t a, uint32x4_t b)
          A32: VABDL.U32 Qd, Dn+1, Dm+1
          A64: UABDL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vabal_high_u8 (uint16x8_t a, uint8x16_t b, uint8x16_t c)
          A32: VABAL.U8 Qd, Dn+1, Dm+1
          A64: UABAL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vabal_high_s16 (int32x4_t a, int16x8_t b, int16x8_t c)
          A32: VABAL.S16 Qd, Dn+1, Dm+1
          A64: SABAL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vabal_high_s32 (int64x2_t a, int32x4_t b, int32x4_t c)
          A32: VABAL.S32 Qd, Dn+1, Dm+1
          A64: SABAL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vabal_high_s8 (int16x8_t a, int8x16_t b, int8x16_t c)
          A32: VABAL.S8 Qd, Dn+1, Dm+1
          A64: SABAL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vabal_high_u16 (uint32x4_t a, uint16x8_t b, uint16x8_t c)
          A32: VABAL.U16 Qd, Dn+1, Dm+1
          A64: UABAL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AbsoluteDifferenceWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vabal_high_u32 (uint64x2_t a, uint32x4_t b, uint32x4_t c)
          A32: VABAL.U32 Qd, Dn+1, Dm+1
          A64: UABAL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AbsSaturate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqabs_s16 (int16x4_t a)
          A32: VQABS.S16 Dd, Dm
          A64: SQABS Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def AbsSaturate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqabs_s32 (int32x2_t a)
          A32: VQABS.S32 Dd, Dm
          A64: SQABS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def AbsSaturate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqabs_s8 (int8x8_t a)
          A32: VQABS.S8 Dd, Dm
          A64: SQABS Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def AbsSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqabsq_s16 (int16x8_t a)
          A32: VQABS.S16 Qd, Qm
          A64: SQABS Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def AbsSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqabsq_s32 (int32x4_t a)
          A32: VQABS.S32 Qd, Qm
          A64: SQABS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def AbsSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqabsq_s8 (int8x16_t a)
          A32: VQABS.S8 Qd, Qm
          A64: SQABS Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def AbsScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vabs_f64 (float64x1_t a)
          A32: VABS.F64 Dd, Dm
          A64: FABS Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def AbsScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vabss_f32 (float32_t a)
          A32: VABS.F32 Sd, Sm
          A64: FABS Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vadd_u8 (uint8x8_t a, uint8x8_t b)
          A32: VADD.I8 Dd, Dn, Dm
          A64: ADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vadd_s16 (int16x4_t a, int16x4_t b)
          A32: VADD.I16 Dd, Dn, Dm
          A64: ADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vadd_s32 (int32x2_t a, int32x2_t b)
          A32: VADD.I32 Dd, Dn, Dm
          A64: ADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vadd_s8 (int8x8_t a, int8x8_t b)
          A32: VADD.I8 Dd, Dn, Dm
          A64: ADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vadd_f32 (float32x2_t a, float32x2_t b)
          A32: VADD.F32 Dd, Dn, Dm
          A64: FADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vadd_u16 (uint16x4_t a, uint16x4_t b)
          A32: VADD.I16 Dd, Dn, Dm
          A64: ADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vadd_u32 (uint32x2_t a, uint32x2_t b)
          A32: VADD.I32 Dd, Dn, Dm
          A64: ADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaddq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VADD.I8 Qd, Qn, Qm
          A64: ADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vaddq_s16 (int16x8_t a, int16x8_t b)
          A32: VADD.I16 Qd, Qn, Qm
          A64: ADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vaddq_s32 (int32x4_t a, int32x4_t b)
          A32: VADD.I32 Qd, Qn, Qm
          A64: ADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vaddq_s64 (int64x2_t a, int64x2_t b)
          A32: VADD.I64 Qd, Qn, Qm
          A64: ADD Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vaddq_s8 (int8x16_t a, int8x16_t b)
          A32: VADD.I8 Qd, Qn, Qm
          A64: ADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vaddq_f32 (float32x4_t a, float32x4_t b)
          A32: VADD.F32 Qd, Qn, Qm
          A64: FADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vaddq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VADD.I16 Qd, Qn, Qm
          A64: ADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vaddq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VADD.I32 Qd, Qn, Qm
          A64: ADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Add(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vaddq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VADD.I64 Qd, Qn, Qm
          A64: ADD Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vaddhn_u16 (uint16x8_t a, uint16x8_t b)
          A32: VADDHN.I16 Dd, Qn, Qm
          A64: ADDHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vaddhn_s32 (int32x4_t a, int32x4_t b)
          A32: VADDHN.I32 Dd, Qn, Qm
          A64: ADDHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vaddhn_s64 (int64x2_t a, int64x2_t b)
          A32: VADDHN.I64 Dd, Qn, Qm
          A64: ADDHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vaddhn_s16 (int16x8_t a, int16x8_t b)
          A32: VADDHN.I16 Dd, Qn, Qm
          A64: ADDHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vaddhn_u32 (uint32x4_t a, uint32x4_t b)
          A32: VADDHN.I32 Dd, Qn, Qm
          A64: ADDHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vaddhn_u64 (uint64x2_t a, uint64x2_t b)
          A32: VADDHN.I64 Dd, Qn, Qm
          A64: ADDHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaddhn_high_u16 (uint8x8_t r, uint16x8_t a, uint16x8_t b)
          A32: VADDHN.I16 Dd+1, Qn, Qm
          A64: ADDHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vaddhn_high_s32 (int16x4_t r, int32x4_t a, int32x4_t b)
          A32: VADDHN.I32 Dd+1, Qn, Qm
          A64: ADDHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vaddhn_high_s64 (int32x2_t r, int64x2_t a, int64x2_t b)
          A32: VADDHN.I64 Dd+1, Qn, Qm
          A64: ADDHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vaddhn_high_s16 (int8x8_t r, int16x8_t a, int16x8_t b)
          A32: VADDHN.I16 Dd+1, Qn, Qm
          A64: ADDHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vaddhn_high_u32 (uint16x4_t r, uint32x4_t a, uint32x4_t b)
          A32: VADDHN.I32 Dd+1, Qn, Qm
          A64: ADDHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vaddhn_high_u64 (uint32x2_t r, uint64x2_t a, uint64x2_t b)
          A32: VADDHN.I64 Dd+1, Qn, Qm
          A64: ADDHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vpadd_u8 (uint8x8_t a, uint8x8_t b)
          A32: VPADD.I8 Dd, Dn, Dm
          A64: ADDP Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vpadd_s16 (int16x4_t a, int16x4_t b)
          A32: VPADD.I16 Dd, Dn, Dm
          A64: ADDP Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vpadd_s32 (int32x2_t a, int32x2_t b)
          A32: VPADD.I32 Dd, Dn, Dm
          A64: ADDP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vpadd_s8 (int8x8_t a, int8x8_t b)
          A32: VPADD.I8 Dd, Dn, Dm
          A64: ADDP Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vpadd_f32 (float32x2_t a, float32x2_t b)
          A32: VPADD.F32 Dd, Dn, Dm
          A64: FADDP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vpadd_u16 (uint16x4_t a, uint16x4_t b)
          A32: VPADD.I16 Dd, Dn, Dm
          A64: ADDP Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vpadd_u32 (uint32x2_t a, uint32x2_t b)
          A32: VPADD.I32 Dd, Dn, Dm
          A64: ADDP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vpaddl_u8 (uint8x8_t a)
          A32: VPADDL.U8 Dd, Dm
          A64: UADDLP Vd.4H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vpaddl_s16 (int16x4_t a)
          A32: VPADDL.S16 Dd, Dm
          A64: SADDLP Vd.2S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vpaddl_s8 (int8x8_t a)
          A32: VPADDL.S8 Dd, Dm
          A64: SADDLP Vd.4H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vpaddl_u16 (uint16x4_t a)
          A32: VPADDL.U16 Dd, Dm
          A64: UADDLP Vd.2S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vpaddlq_u8 (uint8x16_t a)
          A32: VPADDL.U8 Qd, Qm
          A64: UADDLP Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vpaddlq_s16 (int16x8_t a)
          A32: VPADDL.S16 Qd, Qm
          A64: SADDLP Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vpaddlq_s32 (int32x4_t a)
          A32: VPADDL.S32 Qd, Qm
          A64: SADDLP Vd.2D, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vpaddlq_s8 (int8x16_t a)
          A32: VPADDL.S8 Qd, Qm
          A64: SADDLP Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vpaddlq_u16 (uint16x8_t a)
          A32: VPADDL.U16 Qd, Qm
          A64: UADDLP Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWidening(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vpaddlq_u32 (uint32x4_t a)
          A32: VPADDL.U32 Qd, Qm
          A64: UADDLP Vd.2D, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vpadal_u8 (uint16x4_t a, uint8x8_t b)
          A32: VPADAL.U8 Dd, Dm
          A64: UADALP Vd.4H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vpadal_s16 (int32x2_t a, int16x4_t b)
          A32: VPADAL.S16 Dd, Dm
          A64: SADALP Vd.2S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vpadal_s8 (int16x4_t a, int8x8_t b)
          A32: VPADAL.S8 Dd, Dm
          A64: SADALP Vd.4H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vpadal_u16 (uint32x2_t a, uint16x4_t b)
          A32: VPADAL.U16 Dd, Dm
          A64: UADALP Vd.2S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vpadalq_u8 (uint16x8_t a, uint8x16_t b)
          A32: VPADAL.U8 Qd, Qm
          A64: UADALP Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vpadalq_s16 (int32x4_t a, int16x8_t b)
          A32: VPADAL.S16 Qd, Qm
          A64: SADALP Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vpadalq_s32 (int64x2_t a, int32x4_t b)
          A32: VPADAL.S32 Qd, Qm
          A64: SADALP Vd.2D, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vpadalq_s8 (int16x8_t a, int8x16_t b)
          A32: VPADAL.S8 Qd, Qm
          A64: SADALP Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vpadalq_u16 (uint32x4_t a, uint16x8_t b)
          A32: VPADAL.U16 Qd, Qm
          A64: UADALP Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vpadalq_u32 (uint64x2_t a, uint32x4_t b)
          A32: VPADAL.U32 Qd, Qm
          A64: UADALP Vd.2D, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vpadal_s32 (int64x1_t a, int32x2_t b)
          A32: VPADAL.S32 Dd, Dm
          A64: SADALP Vd.1D, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningAndAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vpadal_u32 (uint64x1_t a, uint32x2_t b)
          A32: VPADAL.U32 Dd, Dm
          A64: UADALP Vd.1D, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vpaddl_s32 (int32x2_t a)
          A32: VPADDL.S32 Dd, Dm
          A64: SADDLP Dd, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def AddPairwiseWideningScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vpaddl_u32 (uint32x2_t a)
          A32: VPADDL.U32 Dd, Dm
          A64: UADDLP Dd, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vraddhn_u16 (uint16x8_t a, uint16x8_t b)
          A32: VRADDHN.I16 Dd, Qn, Qm
          A64: RADDHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vraddhn_s32 (int32x4_t a, int32x4_t b)
          A32: VRADDHN.I32 Dd, Qn, Qm
          A64: RADDHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vraddhn_s64 (int64x2_t a, int64x2_t b)
          A32: VRADDHN.I64 Dd, Qn, Qm
          A64: RADDHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vraddhn_s16 (int16x8_t a, int16x8_t b)
          A32: VRADDHN.I16 Dd, Qn, Qm
          A64: RADDHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vraddhn_u32 (uint32x4_t a, uint32x4_t b)
          A32: VRADDHN.I32 Dd, Qn, Qm
          A64: RADDHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vraddhn_u64 (uint64x2_t a, uint64x2_t b)
          A32: VRADDHN.I64 Dd, Qn, Qm
          A64: RADDHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vraddhn_high_u16 (uint8x8_t r, uint16x8_t a, uint16x8_t b)
          A32: VRADDHN.I16 Dd+1, Qn, Qm
          A64: RADDHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vraddhn_high_s32 (int16x4_t r, int32x4_t a, int32x4_t b)
          A32: VRADDHN.I32 Dd+1, Qn, Qm
          A64: RADDHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vraddhn_high_s64 (int32x2_t r, int64x2_t a, int64x2_t b)
          A32: VRADDHN.I64 Dd+1, Qn, Qm
          A64: RADDHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vraddhn_high_s16 (int8x8_t r, int16x8_t a, int16x8_t b)
          A32: VRADDHN.I16 Dd+1, Qn, Qm
          A64: RADDHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vraddhn_high_u32 (uint16x4_t r, uint32x4_t a, uint32x4_t b)
          A32: VRADDHN.I32 Dd+1, Qn, Qm
          A64: RADDHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vraddhn_high_u64 (uint32x2_t r, uint64x2_t a, uint64x2_t b)
          A32: VRADDHN.I64 Dd+1, Qn, Qm
          A64: RADDHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqadd_u8 (uint8x8_t a, uint8x8_t b)
          A32: VQADD.U8 Dd, Dn, Dm
          A64: UQADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqadd_s16 (int16x4_t a, int16x4_t b)
          A32: VQADD.S16 Dd, Dn, Dm
          A64: SQADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqadd_s32 (int32x2_t a, int32x2_t b)
          A32: VQADD.S32 Dd, Dn, Dm
          A64: SQADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqadd_s8 (int8x8_t a, int8x8_t b)
          A32: VQADD.S8 Dd, Dn, Dm
          A64: SQADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqadd_u16 (uint16x4_t a, uint16x4_t b)
          A32: VQADD.U16 Dd, Dn, Dm
          A64: UQADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqadd_u32 (uint32x2_t a, uint32x2_t b)
          A32: VQADD.U32 Dd, Dn, Dm
          A64: UQADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqaddq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VQADD.U8 Qd, Qn, Qm
          A64: UQADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqaddq_s16 (int16x8_t a, int16x8_t b)
          A32: VQADD.S16 Qd, Qn, Qm
          A64: SQADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqaddq_s32 (int32x4_t a, int32x4_t b)
          A32: VQADD.S32 Qd, Qn, Qm
          A64: SQADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqaddq_s64 (int64x2_t a, int64x2_t b)
          A32: VQADD.S64 Qd, Qn, Qm
          A64: SQADD Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqaddq_s8 (int8x16_t a, int8x16_t b)
          A32: VQADD.S8 Qd, Qn, Qm
          A64: SQADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqaddq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VQADD.U16 Qd, Qn, Qm
          A64: UQADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqaddq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VQADD.U32 Qd, Qn, Qm
          A64: UQADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqaddq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VQADD.U64 Qd, Qn, Qm
          A64: UQADD Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vqadd_s64 (int64x1_t a, int64x1_t b)
          A32: VQADD.S64 Dd, Dn, Dm
          A64: SQADD Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def AddSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqadd_u64 (uint64x1_t a, uint64x1_t b)
          A32: VQADD.U64 Dd, Dn, Dm
          A64: UQADD Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def AddScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vadd_f64 (float64x1_t a, float64x1_t b)
          A32: VADD.F64 Dd, Dn, Dm
          A64: FADD Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def AddScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vadd_s64 (int64x1_t a, int64x1_t b)
          A32: VADD.I64 Dd, Dn, Dm
          A64: ADD Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def AddScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vadds_f32 (float32_t a, float32_t b)
          A32: VADD.F32 Sd, Sn, Sm
          A64: FADD Sd, Sn, Sm
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def AddScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vadd_u64 (uint64x1_t a, uint64x1_t b)
          A32: VADD.I64 Dd, Dn, Dm
          A64: ADD Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vaddl_u8 (uint8x8_t a, uint8x8_t b)
          A32: VADDL.U8 Qd, Dn, Dm
          A64: UADDL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vaddl_s16 (int16x4_t a, int16x4_t b)
          A32: VADDL.S16 Qd, Dn, Dm
          A64: SADDL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vaddl_s32 (int32x2_t a, int32x2_t b)
          A32: VADDL.S32 Qd, Dn, Dm
          A64: SADDL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vaddl_s8 (int8x8_t a, int8x8_t b)
          A32: VADDL.S8 Qd, Dn, Dm
          A64: SADDL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vaddl_u16 (uint16x4_t a, uint16x4_t b)
          A32: VADDL.U16 Qd, Dn, Dm
          A64: UADDL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vaddl_u32 (uint32x2_t a, uint32x2_t b)
          A32: VADDL.U32 Qd, Dn, Dm
          A64: UADDL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vaddw_s8 (int16x8_t a, int8x8_t b)
          A32: VADDW.S8 Qd, Qn, Dm
          A64: SADDW Vd.8H, Vn.8H, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vaddw_s16 (int32x4_t a, int16x4_t b)
          A32: VADDW.S16 Qd, Qn, Dm
          A64: SADDW Vd.4S, Vn.4S, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vaddw_s32 (int64x2_t a, int32x2_t b)
          A32: VADDW.S32 Qd, Qn, Dm
          A64: SADDW Vd.2D, Vn.2D, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vaddw_u8 (uint16x8_t a, uint8x8_t b)
          A32: VADDW.U8 Qd, Qn, Dm
          A64: UADDW Vd.8H, Vn.8H, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vaddw_u16 (uint32x4_t a, uint16x4_t b)
          A32: VADDW.U16 Qd, Qn, Dm
          A64: UADDW Vd.4S, Vn.4S, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def AddWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vaddw_u32 (uint64x2_t a, uint32x2_t b)
          A32: VADDW.U32 Qd, Qn, Dm
          A64: UADDW Vd.2D, Vn.2D, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vaddl_high_u8 (uint8x16_t a, uint8x16_t b)
          A32: VADDL.U8 Qd, Dn+1, Dm+1
          A64: UADDL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vaddl_high_s16 (int16x8_t a, int16x8_t b)
          A32: VADDL.S16 Qd, Dn+1, Dm+1
          A64: SADDL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vaddw_high_s8 (int16x8_t a, int8x16_t b)
          A32: VADDW.S8 Qd, Qn, Dm+1
          A64: SADDW2 Vd.8H, Vn.8H, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vaddw_high_s16 (int32x4_t a, int16x8_t b)
          A32: VADDW.S16 Qd, Qn, Dm+1
          A64: SADDW2 Vd.4S, Vn.4S, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vaddl_high_s32 (int32x4_t a, int32x4_t b)
          A32: VADDL.S32 Qd, Dn+1, Dm+1
          A64: SADDL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vaddw_high_s32 (int64x2_t a, int32x4_t b)
          A32: VADDW.S32 Qd, Qn, Dm+1
          A64: SADDW2 Vd.2D, Vn.2D, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vaddl_high_s8 (int8x16_t a, int8x16_t b)
          A32: VADDL.S8 Qd, Dn+1, Dm+1
          A64: SADDL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vaddw_high_u8 (uint16x8_t a, uint8x16_t b)
          A32: VADDW.U8 Qd, Qn, Dm+1
          A64: UADDW2 Vd.8H, Vn.8H, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vaddl_high_u16 (uint16x8_t a, uint16x8_t b)
          A32: VADDL.U16 Qd, Dn+1, Dm+1
          A64: UADDL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vaddw_high_u16 (uint32x4_t a, uint16x8_t b)
          A32: VADDW.U16 Qd, Qn, Dm+1
          A64: UADDW2 Vd.4S, Vn.4S, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vaddl_high_u32 (uint32x4_t a, uint32x4_t b)
          A32: VADDL.U32 Qd, Dn+1, Dm+1
          A64: UADDL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def AddWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vaddw_high_u32 (uint64x2_t a, uint32x4_t b)
          A32: VADDW.U32 Qd, Qn, Dm+1
          A64: UADDW2 Vd.2D, Vn.2D, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vand_u8 (uint8x8_t a, uint8x8_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vand_f64 (float64x1_t a, float64x1_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vand_s16 (int16x4_t a, int16x4_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vand_s32 (int32x2_t a, int32x2_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vand_s64 (int64x1_t a, int64x1_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vand_s8 (int8x8_t a, int8x8_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vand_f32 (float32x2_t a, float32x2_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vand_u16 (uint16x4_t a, uint16x4_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vand_u32 (uint32x2_t a, uint32x2_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vand_u64 (uint64x1_t a, uint64x1_t b)
          A32: VAND Dd, Dn, Dm
          A64: AND Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vandq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vandq_f64 (float64x2_t a, float64x2_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vandq_s16 (int16x8_t a, int16x8_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vandq_s32 (int32x4_t a, int32x4_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vandq_s64 (int64x2_t a, int64x2_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vandq_s8 (int8x16_t a, int8x16_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vandq_f32 (float32x4_t a, float32x4_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vandq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vandq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def And(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vandq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VAND Qd, Qn, Qm
          A64: AND Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vbic_u8 (uint8x8_t a, uint8x8_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[float], mask: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vbic_f64 (float64x1_t a, float64x1_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vbic_s16 (int16x4_t a, int16x4_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vbic_s32 (int32x2_t a, int32x2_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vbic_s64 (int64x1_t a, int64x1_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vbic_s8 (int8x8_t a, int8x8_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[float], mask: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vbic_f32 (float32x2_t a, float32x2_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vbic_u16 (uint16x4_t a, uint16x4_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vbic_u32 (uint32x2_t a, uint32x2_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector64[int], mask: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vbic_u64 (uint64x1_t a, uint64x1_t b)
          A32: VBIC Dd, Dn, Dm
          A64: BIC Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vbicq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[float], mask: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vbicq_f64 (float64x2_t a, float64x2_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vbicq_s16 (int16x8_t a, int16x8_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vbicq_s32 (int32x4_t a, int32x4_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vbicq_s64 (int64x2_t a, int64x2_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vbicq_s8 (int8x16_t a, int8x16_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[float], mask: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vbicq_f32 (float32x4_t a, float32x4_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vbicq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vbicq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseClear(value: System.Runtime.Intrinsics.Vector128[int], mask: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vbicq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VBIC Qd, Qn, Qm
          A64: BIC Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vbsl_u8 (uint8x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vbsl_f64 (uint64x1_t a, float64x1_t b, float64x1_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vbsl_s16 (uint16x4_t a, int16x4_t b, int16x4_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vbsl_s32 (uint32x2_t a, int32x2_t b, int32x2_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vbsl_s64 (uint64x1_t a, int64x1_t b, int64x1_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vbsl_s8 (uint8x8_t a, int8x8_t b, int8x8_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vbsl_f32 (uint32x2_t a, float32x2_t b, float32x2_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vbsl_u16 (uint16x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vbsl_u32 (uint32x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vbsl_u64 (uint64x1_t a, uint64x1_t b, uint64x1_t c)
          A32: VBSL Dd, Dn, Dm
          A64: BSL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vbslq_u8 (uint8x16_t a, uint8x16_t b, uint8x16_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vbslq_f64 (uint64x2_t a, float64x2_t b, float64x2_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vbslq_s16 (uint16x8_t a, int16x8_t b, int16x8_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vbslq_s32 (uint32x4_t a, int32x4_t b, int32x4_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vbslq_s64 (uint64x2_t a, int64x2_t b, int64x2_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vbslq_s8 (uint8x16_t a, int8x16_t b, int8x16_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vbslq_f32 (uint32x4_t a, float32x4_t b, float32x4_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vbslq_u16 (uint16x8_t a, uint16x8_t b, uint16x8_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vbslq_u32 (uint32x4_t a, uint32x4_t b, uint32x4_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def BitwiseSelect(select: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vbslq_u64 (uint64x2_t a, uint64x2_t b, uint64x2_t c)
          A32: VBSL Qd, Qn, Qm
          A64: BSL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Ceiling(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrndp_f32 (float32x2_t a)
          A32: VRINTP.F32 Dd, Dm
          A64: FRINTP Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def Ceiling(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndpq_f32 (float32x4_t a)
          A32: VRINTP.F32 Qd, Qm
          A64: FRINTP Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def CeilingScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrndp_f64 (float64x1_t a)
          A32: VRINTP.F64 Dd, Dm
          A64: FRINTP Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def CeilingScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrndps_f32 (float32_t a)
          A32: VRINTP.F32 Sd, Sm
          A64: FRINTP Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vceq_u8 (uint8x8_t a, uint8x8_t b)
          A32: VCEQ.I8 Dd, Dn, Dm
          A64: CMEQ Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vceq_s16 (int16x4_t a, int16x4_t b)
          A32: VCEQ.I16 Dd, Dn, Dm
          A64: CMEQ Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vceq_s32 (int32x2_t a, int32x2_t b)
          A32: VCEQ.I32 Dd, Dn, Dm
          A64: CMEQ Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vceq_s8 (int8x8_t a, int8x8_t b)
          A32: VCEQ.I8 Dd, Dn, Dm
          A64: CMEQ Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vceq_f32 (float32x2_t a, float32x2_t b)
          A32: VCEQ.F32 Dd, Dn, Dm
          A64: FCMEQ Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vceq_u16 (uint16x4_t a, uint16x4_t b)
          A32: VCEQ.I16 Dd, Dn, Dm
          A64: CMEQ Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vceq_u32 (uint32x2_t a, uint32x2_t b)
          A32: VCEQ.I32 Dd, Dn, Dm
          A64: CMEQ Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vceqq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VCEQ.I8 Qd, Qn, Qm
          A64: CMEQ Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vceqq_s16 (int16x8_t a, int16x8_t b)
          A32: VCEQ.I16 Qd, Qn, Qm
          A64: CMEQ Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vceqq_s32 (int32x4_t a, int32x4_t b)
          A32: VCEQ.I32 Qd, Qn, Qm
          A64: CMEQ Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vceqq_s8 (int8x16_t a, int8x16_t b)
          A32: VCEQ.I8 Qd, Qn, Qm
          A64: CMEQ Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vceqq_f32 (float32x4_t a, float32x4_t b)
          A32: VCEQ.F32 Qd, Qn, Qm
          A64: FCMEQ Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vceqq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VCEQ.I16 Qd, Qn, Qm
          A64: CMEQ Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vceqq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VCEQ.I32 Qd, Qn, Qm
          A64: CMEQ Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcgt_u8 (uint8x8_t a, uint8x8_t b)
          A32: VCGT.U8 Dd, Dn, Dm
          A64: CMHI Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vcgt_s16 (int16x4_t a, int16x4_t b)
          A32: VCGT.S16 Dd, Dn, Dm
          A64: CMGT Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcgt_s32 (int32x2_t a, int32x2_t b)
          A32: VCGT.S32 Dd, Dn, Dm
          A64: CMGT Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcgt_s8 (int8x8_t a, int8x8_t b)
          A32: VCGT.S8 Dd, Dn, Dm
          A64: CMGT Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcgt_f32 (float32x2_t a, float32x2_t b)
          A32: VCGT.F32 Dd, Dn, Dm
          A64: FCMGT Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vcgt_u16 (uint16x4_t a, uint16x4_t b)
          A32: VCGT.U16 Dd, Dn, Dm
          A64: CMHI Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcgt_u32 (uint32x2_t a, uint32x2_t b)
          A32: VCGT.U32 Dd, Dn, Dm
          A64: CMHI Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcgtq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VCGT.U8 Qd, Qn, Qm
          A64: CMHI Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcgtq_s16 (int16x8_t a, int16x8_t b)
          A32: VCGT.S16 Qd, Qn, Qm
          A64: CMGT Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcgtq_s32 (int32x4_t a, int32x4_t b)
          A32: VCGT.S32 Qd, Qn, Qm
          A64: CMGT Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcgtq_s8 (int8x16_t a, int8x16_t b)
          A32: VCGT.S8 Qd, Qn, Qm
          A64: CMGT Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcgtq_f32 (float32x4_t a, float32x4_t b)
          A32: VCGT.F32 Qd, Qn, Qm
          A64: FCMGT Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcgtq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VCGT.U16 Qd, Qn, Qm
          A64: CMHI Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcgtq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VCGT.U32 Qd, Qn, Qm
          A64: CMHI Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcge_u8 (uint8x8_t a, uint8x8_t b)
          A32: VCGE.U8 Dd, Dn, Dm
          A64: CMHS Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vcge_s16 (int16x4_t a, int16x4_t b)
          A32: VCGE.S16 Dd, Dn, Dm
          A64: CMGE Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcge_s32 (int32x2_t a, int32x2_t b)
          A32: VCGE.S32 Dd, Dn, Dm
          A64: CMGE Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcge_s8 (int8x8_t a, int8x8_t b)
          A32: VCGE.S8 Dd, Dn, Dm
          A64: CMGE Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcge_f32 (float32x2_t a, float32x2_t b)
          A32: VCGE.F32 Dd, Dn, Dm
          A64: FCMGE Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vcge_u16 (uint16x4_t a, uint16x4_t b)
          A32: VCGE.U16 Dd, Dn, Dm
          A64: CMHS Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcge_u32 (uint32x2_t a, uint32x2_t b)
          A32: VCGE.U32 Dd, Dn, Dm
          A64: CMHS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcgeq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VCGE.U8 Qd, Qn, Qm
          A64: CMHS Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcgeq_s16 (int16x8_t a, int16x8_t b)
          A32: VCGE.S16 Qd, Qn, Qm
          A64: CMGE Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcgeq_s32 (int32x4_t a, int32x4_t b)
          A32: VCGE.S32 Qd, Qn, Qm
          A64: CMGE Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcgeq_s8 (int8x16_t a, int8x16_t b)
          A32: VCGE.S8 Qd, Qn, Qm
          A64: CMGE Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcgeq_f32 (float32x4_t a, float32x4_t b)
          A32: VCGE.F32 Qd, Qn, Qm
          A64: FCMGE Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcgeq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VCGE.U16 Qd, Qn, Qm
          A64: CMHS Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareGreaterThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcgeq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VCGE.U32 Qd, Qn, Qm
          A64: CMHS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vclt_u8 (uint8x8_t a, uint8x8_t b)
          A32: VCLT.U8 Dd, Dn, Dm
          A64: CMHI Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vclt_s16 (int16x4_t a, int16x4_t b)
          A32: VCLT.S16 Dd, Dn, Dm
          A64: CMGT Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vclt_s32 (int32x2_t a, int32x2_t b)
          A32: VCLT.S32 Dd, Dn, Dm
          A64: CMGT Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vclt_s8 (int8x8_t a, int8x8_t b)
          A32: VCLT.S8 Dd, Dn, Dm
          A64: CMGT Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vclt_f32 (float32x2_t a, float32x2_t b)
          A32: VCLT.F32 Dd, Dn, Dm
          A64: FCMGT Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vclt_u16 (uint16x4_t a, uint16x4_t b)
          A32: VCLT.U16 Dd, Dn, Dm
          A64: CMHI Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vclt_u32 (uint32x2_t a, uint32x2_t b)
          A32: VCLT.U32 Dd, Dn, Dm
          A64: CMHI Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcltq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VCLT.U8 Qd, Qn, Qm
          A64: CMHI Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcltq_s16 (int16x8_t a, int16x8_t b)
          A32: VCLT.S16 Qd, Qn, Qm
          A64: CMGT Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcltq_s32 (int32x4_t a, int32x4_t b)
          A32: VCLT.S32 Qd, Qn, Qm
          A64: CMGT Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcltq_s8 (int8x16_t a, int8x16_t b)
          A32: VCLT.S8 Qd, Qn, Qm
          A64: CMGT Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcltq_f32 (float32x4_t a, float32x4_t b)
          A32: VCLT.F32 Qd, Qn, Qm
          A64: FCMGT Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcltq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VCLT.U16 Qd, Qn, Qm
          A64: CMHI Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThan(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcltq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VCLT.U32 Qd, Qn, Qm
          A64: CMHI Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcle_u8 (uint8x8_t a, uint8x8_t b)
          A32: VCLE.U8 Dd, Dn, Dm
          A64: CMHS Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vcle_s16 (int16x4_t a, int16x4_t b)
          A32: VCLE.S16 Dd, Dn, Dm
          A64: CMGE Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcle_s32 (int32x2_t a, int32x2_t b)
          A32: VCLE.S32 Dd, Dn, Dm
          A64: CMGE Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcle_s8 (int8x8_t a, int8x8_t b)
          A32: VCLE.S8 Dd, Dn, Dm
          A64: CMGE Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vcle_f32 (float32x2_t a, float32x2_t b)
          A32: VCLE.F32 Dd, Dn, Dm
          A64: FCMGE Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vcle_u16 (uint16x4_t a, uint16x4_t b)
          A32: VCLE.U16 Dd, Dn, Dm
          A64: CMHS Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcle_u32 (uint32x2_t a, uint32x2_t b)
          A32: VCLE.U32 Dd, Dn, Dm
          A64: CMHS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcleq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VCLE.U8 Qd, Qn, Qm
          A64: CMHS Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcleq_s16 (int16x8_t a, int16x8_t b)
          A32: VCLE.S16 Qd, Qn, Qm
          A64: CMGE Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcleq_s32 (int32x4_t a, int32x4_t b)
          A32: VCLE.S32 Qd, Qn, Qm
          A64: CMGE Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcleq_s8 (int8x16_t a, int8x16_t b)
          A32: VCLE.S8 Qd, Qn, Qm
          A64: CMGE Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vcleq_f32 (float32x4_t a, float32x4_t b)
          A32: VCLE.F32 Qd, Qn, Qm
          A64: FCMGE Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vcleq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VCLE.U16 Qd, Qn, Qm
          A64: CMHS Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareLessThanOrEqual(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcleq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VCLE.U32 Qd, Qn, Qm
          A64: CMHS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vtst_u8 (uint8x8_t a, uint8x8_t b)
          A32: VTST.8 Dd, Dn, Dm
          A64: CMTST Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vtst_s16 (int16x4_t a, int16x4_t b)
          A32: VTST.16 Dd, Dn, Dm
          A64: CMTST Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vtst_s32 (int32x2_t a, int32x2_t b)
          A32: VTST.32 Dd, Dn, Dm
          A64: CMTST Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vtst_s8 (int8x8_t a, int8x8_t b)
          A32: VTST.8 Dd, Dn, Dm
          A64: CMTST Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        uint32x2_t vtst_f32 (float32x2_t a, float32x2_t b)
          A32: VTST.32 Dd, Dn, Dm
          A64: CMTST Vd.2S, Vn.2S, Vm.2S
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vtst_u16 (uint16x4_t a, uint16x4_t b)
          A32: VTST.16 Dd, Dn, Dm
          A64: CMTST Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vtst_u32 (uint32x2_t a, uint32x2_t b)
          A32: VTST.32 Dd, Dn, Dm
          A64: CMTST Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vtstq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VTST.8 Qd, Qn, Qm
          A64: CMTST Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vtstq_s16 (int16x8_t a, int16x8_t b)
          A32: VTST.16 Qd, Qn, Qm
          A64: CMTST Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vtstq_s32 (int32x4_t a, int32x4_t b)
          A32: VTST.32 Qd, Qn, Qm
          A64: CMTST Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vtstq_s8 (int8x16_t a, int8x16_t b)
          A32: VTST.8 Qd, Qn, Qm
          A64: CMTST Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        uint32x4_t vtstq_f32 (float32x4_t a, float32x4_t b)
          A32: VTST.32 Qd, Qn, Qm
          A64: CMTST Vd.4S, Vn.4S, Vm.4S
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vtstq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VTST.16 Qd, Qn, Qm
          A64: CMTST Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def CompareTest(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vtstq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VTST.32 Qd, Qn, Qm
          A64: CMTST Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vcvta_s32_f32 (float32x2_t a)
          A32: VCVTA.S32.F32 Dd, Dm
          A64: FCVTAS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vcvtaq_s32_f32 (float32x4_t a)
          A32: VCVTA.S32.F32 Qd, Qm
          A64: FCVTAS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToInt32RoundAwayFromZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32_t vcvtas_s32_f32 (float32_t a)
          A32: VCVTA.S32.F32 Sd, Sm
          A64: FCVTAS Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToEven(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vcvtn_s32_f32 (float32x2_t a)
          A32: VCVTN.S32.F32 Dd, Dm
          A64: FCVTNS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToEven(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vcvtnq_s32_f32 (float32x4_t a)
          A32: VCVTN.S32.F32 Qd, Qm
          A64: FCVTNS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToInt32RoundToEvenScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32_t vcvtns_s32_f32 (float32_t a)
          A32: VCVTN.S32.F32 Sd, Sm
          A64: FCVTNS Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vcvtm_s32_f32 (float32x2_t a)
          A32: VCVTM.S32.F32 Dd, Dm
          A64: FCVTMS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vcvtmq_s32_f32 (float32x4_t a)
          A32: VCVTM.S32.F32 Qd, Qm
          A64: FCVTMS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToInt32RoundToNegativeInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32_t vcvtms_s32_f32 (float32_t a)
          A32: VCVTM.S32.F32 Sd, Sm
          A64: FCVTMS Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vcvtp_s32_f32 (float32x2_t a)
          A32: VCVTP.S32.F32 Dd, Dm
          A64: FCVTPS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vcvtpq_s32_f32 (float32x4_t a)
          A32: VCVTP.S32.F32 Qd, Qm
          A64: FCVTPS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToInt32RoundToPositiveInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32_t vcvtps_s32_f32 (float32_t a)
          A32: VCVTP.S32.F32 Sd, Sm
          A64: FCVTPS Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToZero(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vcvt_s32_f32 (float32x2_t a)
          A32: VCVT.S32.F32 Dd, Dm
          A64: FCVTZS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToInt32RoundToZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vcvtq_s32_f32 (float32x4_t a)
          A32: VCVT.S32.F32 Qd, Qm
          A64: FCVTZS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToInt32RoundToZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32_t vcvts_s32_f32 (float32_t a)
          A32: VCVT.S32.F32 Sd, Sm
          A64: FCVTZS Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToSingle(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vcvt_f32_s32 (int32x2_t a)
          A32: VCVT.F32.S32 Dd, Dm
          A64: SCVTF Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToSingle(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vcvt_f32_u32 (uint32x2_t a)
          A32: VCVT.F32.U32 Dd, Dm
          A64: UCVTF Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToSingle(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vcvtq_f32_s32 (int32x4_t a)
          A32: VCVT.F32.S32 Qd, Qm
          A64: SCVTF Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ConvertToSingle(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vcvtq_f32_u32 (uint32x4_t a)
          A32: VCVT.F32.U32 Qd, Qm
          A64: UCVTF Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ConvertToSingleScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vcvts_f32_s32 (int32_t a)
          A32: VCVT.F32.S32 Sd, Sm
          A64: SCVTF Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToSingleScalar(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vcvts_f32_u32 (uint32_t a)
          A32: VCVT.F32.U32 Sd, Sm
          A64: UCVTF Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcvta_u32_f32 (float32x2_t a)
          A32: VCVTA.U32.F32 Dd, Dm
          A64: FCVTAU Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcvtaq_u32_f32 (float32x4_t a)
          A32: VCVTA.U32.F32 Qd, Qm
          A64: FCVTAU Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToUInt32RoundAwayFromZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vcvtas_u32_f32 (float32_t a)
          A32: VCVTA.U32.F32 Sd, Sm
          A64: FCVTAU Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToEven(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcvtn_u32_f32 (float32x2_t a)
          A32: VCVTN.U32.F32 Dd, Dm
          A64: FCVTNU Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToEven(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcvtnq_u32_f32 (float32x4_t a)
          A32: VCVTN.U32.F32 Qd, Qm
          A64: FCVTNU Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToUInt32RoundToEvenScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vcvtns_u32_f32 (float32_t a)
          A32: VCVTN.U32.F32 Sd, Sm
          A64: FCVTNU Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcvtm_u32_f32 (float32x2_t a)
          A32: VCVTM.U32.F32 Dd, Dm
          A64: FCVTMU Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcvtmq_u32_f32 (float32x4_t a)
          A32: VCVTM.U32.F32 Qd, Qm
          A64: FCVTMU Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToUInt32RoundToNegativeInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vcvtms_u32_f32 (float32_t a)
          A32: VCVTM.U32.F32 Sd, Sm
          A64: FCVTMU Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcvtp_u32_f32 (float32x2_t a)
          A32: VCVTP.U32.F32 Dd, Dm
          A64: FCVTPU Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcvtpq_u32_f32 (float32x4_t a)
          A32: VCVTP.U32.F32 Qd, Qm
          A64: FCVTPU Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToUInt32RoundToPositiveInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vcvtps_u32_f32 (float32_t a)
          A32: VCVTP.U32.F32 Sd, Sm
          A64: FCVTPU Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToZero(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vcvt_u32_f32 (float32x2_t a)
          A32: VCVT.U32.F32 Dd, Dm
          A64: FCVTZU Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ConvertToUInt32RoundToZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vcvtq_u32_f32 (float32x4_t a)
          A32: VCVT.U32.F32 Qd, Qm
          A64: FCVTZU Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    def ConvertToUInt32RoundToZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vcvts_u32_f32 (float32_t a)
          A32: VCVT.U32.F32 Sd, Sm
          A64: FCVTZU Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def DivideScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vdiv_f64 (float64x1_t a, float64x1_t b)
          A32: VDIV.F64 Dd, Dn, Dm
          A64: FDIV Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def DivideScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vdivs_f32 (float32_t a, float32_t b)
          A32: VDIV.F32 Sd, Sn, Sm
          A64: FDIV Sd, Sn, Sm
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vdupq_lane_u8 (uint8x8_t vec, const int lane)
          A32: VDUP.8 Qd, Dm[index]
          A64: DUP Vd.16B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vdupq_lane_s16 (int16x4_t vec, const int lane)
          A32: VDUP.16 Qd, Dm[index]
          A64: DUP Vd.8H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdupq_lane_s32 (int32x2_t vec, const int lane)
          A32: VDUP.32 Qd, Dm[index]
          A64: DUP Vd.4S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[float], index: int) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vdupq_lane_f32 (float32x2_t vec, const int lane)
          A32: VDUP.32 Qd, Dm[index]
          A64: DUP Vd.4S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vdupq_lane_s8 (int8x8_t vec, const int lane)
          A32: VDUP.8 Qd, Dm[index]
          A64: DUP Vd.16B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vdupq_lane_u16 (uint16x4_t vec, const int lane)
          A32: VDUP.16 Qd, Dm[index]
          A64: DUP Vd.8H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdupq_lane_u32 (uint32x2_t vec, const int lane)
          A32: VDUP.32 Qd, Dm[index]
          A64: DUP Vd.4S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vdupq_lane_u8 (uint8x16_t vec, const int lane)
          A32: VDUP.8 Qd, Dm[index]
          A64: DUP Vd.16B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vdupq_lane_s16 (int16x8_t vec, const int lane)
          A32: VDUP.16 Qd, Dm[index]
          A64: DUP Vd.8H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdupq_lane_s32 (int32x4_t vec, const int lane)
          A32: VDUP.32 Qd, Dm[index]
          A64: DUP Vd.4S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[float], index: int) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vdupq_lane_f32 (float32x4_t vec, const int lane)
          A32: VDUP.32 Qd, Dm[index]
          A64: DUP Vd.4S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vdupq_lane_s8 (int8x16_t vec, const int lane)
          A32: VDUP.8 Qd, Dm[index]
          A64: DUP Vd.16B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vdupq_lane_u16 (uint16x8_t vec, const int lane)
          A32: VDUP.16 Qd, Dm[index]
          A64: DUP Vd.8H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector128(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdupq_lane_u32 (uint32x4_t vec, const int lane)
          A32: VDUP.32 Qd, Dm[index]
          A64: DUP Vd.4S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vdup_lane_u8 (uint8x8_t vec, const int lane)
          A32: VDUP.8 Dd, Dm[index]
          A64: DUP Vd.8B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vdup_lane_s16 (int16x4_t vec, const int lane)
          A32: VDUP.16 Dd, Dm[index]
          A64: DUP Vd.4H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdup_lane_s32 (int32x2_t vec, const int lane)
          A32: VDUP.32 Dd, Dm[index]
          A64: DUP Vd.2S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[float], index: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vdup_lane_f32 (float32x2_t vec, const int lane)
          A32: VDUP.32 Dd, Dm[index]
          A64: DUP Vd.2S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vdup_lane_s8 (int8x8_t vec, const int lane)
          A32: VDUP.8 Dd, Dm[index]
          A64: DUP Vd.8B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vdup_lane_u16 (uint16x4_t vec, const int lane)
          A32: VDUP.16 Dd, Dm[index]
          A64: DUP Vd.4H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdup_lane_u32 (uint32x2_t vec, const int lane)
          A32: VDUP.32 Dd, Dm[index]
          A64: DUP Vd.2S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vdup_laneq_u8 (uint8x16_t vec, const int lane)
          A32: VDUP.8 Dd, Dm[index]
          A64: DUP Vd.8B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vdup_laneq_s16 (int16x8_t vec, const int lane)
          A32: VDUP.16 Dd, Dm[index]
          A64: DUP Vd.4H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdup_laneq_s32 (int32x4_t vec, const int lane)
          A32: VDUP.32 Dd, Dm[index]
          A64: DUP Vd.2S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[float], index: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vdup_laneq_f32 (float32x4_t vec, const int lane)
          A32: VDUP.32 Dd, Dm[index]
          A64: DUP Vd.2S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vdup_laneq_s8 (int8x16_t vec, const int lane)
          A32: VDUP.8 Dd, Dm[index]
          A64: DUP Vd.8B, Vn.B[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vdup_laneq_u16 (uint16x8_t vec, const int lane)
          A32: VDUP.16 Dd, Dm[index]
          A64: DUP Vd.4H, Vn.H[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateSelectedScalarToVector64(value: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdup_laneq_u32 (uint32x4_t vec, const int lane)
          A32: VDUP.32 Dd, Dm[index]
          A64: DUP Vd.2S, Vn.S[index]
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vdupq_n_u8 (uint8_t value)
          A32: VDUP.8 Qd, Rt
          A64: DUP Vd.16B, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vdupq_n_s16 (int16_t value)
          A32: VDUP.16 Qd, Rt
          A64: DUP Vd.8H, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdupq_n_s32 (int32_t value)
          A32: VDUP.32 Qd, Rt
          A64: DUP Vd.4S, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vdupq_n_s8 (int8_t value)
          A32: VDUP.8 Qd, Rt
          A64: DUP Vd.16B, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: float) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vdupq_n_f32 (float32_t value)
          A32: VDUP Qd, Dm[0]
          A64: DUP Vd.4S, Vn.S[0]
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vdupq_n_u16 (uint16_t value)
          A32: VDUP.16 Qd, Rt
          A64: DUP Vd.8H, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector128(value: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdupq_n_u32 (uint32_t value)
          A32: VDUP.32 Qd, Rt
          A64: DUP Vd.4S, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vdup_n_u8 (uint8_t value)
          A32: VDUP.8 Dd, Rt
          A64: DUP Vd.8B, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vdup_n_s16 (int16_t value)
          A32: VDUP.16 Dd, Rt
          A64: DUP Vd.4H, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdup_n_s32 (int32_t value)
          A32: VDUP.32 Dd, Rt
          A64: DUP Vd.2S, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vdup_n_s8 (int8_t value)
          A32: VDUP.8 Dd, Rt
          A64: DUP Vd.8B, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: float) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vdup_n_f32 (float32_t value)
          A32: VDUP Dd, Dm[0]
          A64: DUP Vd.2S, Vn.S[0]
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vdup_n_u16 (uint16_t value)
          A32: VDUP.16 Dd, Rt
          A64: DUP Vd.4H, Rn
        """
        ...

    @staticmethod
    @overload
    def DuplicateToVector64(value: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdup_n_u32 (uint32_t value)
          A32: VDUP.32 Dd, Rt
          A64: DUP Vd.2S, Rn
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[int], index: int) -> int:
        """
        uint8_t vget_lane_u8 (uint8x8_t v, const int lane)
          A32: VMOV.U8 Rt, Dn[lane]
          A64: UMOV Wd, Vn.B[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[int], index: int) -> int:
        """
        int16_t vget_lane_s16 (int16x4_t v, const int lane)
          A32: VMOV.S16 Rt, Dn[lane]
          A64: SMOV Wd, Vn.H[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[int], index: int) -> int:
        """
        int32_t vget_lane_s32 (int32x2_t v, const int lane)
          A32: VMOV.32 Rt, Dn[lane]
          A64: SMOV Wd, Vn.S[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[int], index: int) -> int:
        """
        int8_t vget_lane_s8 (int8x8_t v, const int lane)
          A32: VMOV.S8 Rt, Dn[lane]
          A64: SMOV Wd, Vn.B[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[float], index: int) -> float:
        """
        float32_t vget_lane_f32 (float32x2_t v, const int lane)
          A32: VMOV.F32 Sd, Sm
          A64: DUP Sd, Vn.S[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[int], index: int) -> int:
        """
        uint16_t vget_lane_u16 (uint16x4_t v, const int lane)
          A32: VMOV.U16 Rt, Dn[lane]
          A64: UMOV Wd, Vn.H[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector64[int], index: int) -> int:
        """
        uint32_t vget_lane_u32 (uint32x2_t v, const int lane)
          A32: VMOV.32 Rt, Dn[lane]
          A64: UMOV Wd, Vn.S[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        uint8_t vgetq_lane_u8 (uint8x16_t v, const int lane)
          A32: VMOV.U8 Rt, Dn[lane]
          A64: UMOV Wd, Vn.B[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[float], index: int) -> float:
        """
        float64_t vgetq_lane_f64 (float64x2_t v, const int lane)
          A32: VMOV.F64 Dd, Dm
          A64: DUP Dd, Vn.D[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        int16_t vgetq_lane_s16 (int16x8_t v, const int lane)
          A32: VMOV.S16 Rt, Dn[lane]
          A64: SMOV Wd, Vn.H[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        int32_t vgetq_lane_s32 (int32x4_t v, const int lane)
          A32: VMOV.32 Rt, Dn[lane]
          A64: SMOV Wd, Vn.S[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        int64_t vgetq_lane_s64 (int64x2_t v, const int lane)
          A32: VMOV Rt, Rt2, Dm
          A64: UMOV Xd, Vn.D[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        int8_t vgetq_lane_s8 (int8x16_t v, const int lane)
          A32: VMOV.S8 Rt, Dn[lane]
          A64: SMOV Wd, Vn.B[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[float], index: int) -> float:
        """
        float32_t vgetq_lane_f32 (float32x4_t v, const int lane)
          A32: VMOV.F32 Sd, Sm
          A64: DUP Sd, Vn.S[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        uint16_t vgetq_lane_u16 (uint16x8_t v, const int lane)
          A32: VMOV.U16 Rt, Dn[lane]
          A64: UMOV Wd, Vn.H[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        uint32_t vgetq_lane_u32 (uint32x4_t v, const int lane)
          A32: VMOV.32 Rt, Dn[lane]
          A64: UMOV Wd, Vn.S[lane]
        """
        ...

    @staticmethod
    @overload
    def Extract(vector: System.Runtime.Intrinsics.Vector128[int], index: int) -> int:
        """
        uint64_t vgetq_lane_u64 (uint64x2_t v, const int lane)
          A32: VMOV Rt, Rt2, Dm
          A64: UMOV Xd, Vn.D[lane]
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmovn_u16 (uint16x8_t a)
          A32: VMOVN.I16 Dd, Qm
          A64: XTN Vd.8B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmovn_s32 (int32x4_t a)
          A32: VMOVN.I32 Dd, Qm
          A64: XTN Vd.4H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmovn_s64 (int64x2_t a)
          A32: VMOVN.I64 Dd, Qm
          A64: XTN Vd.2S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmovn_s16 (int16x8_t a)
          A32: VMOVN.I16 Dd, Qm
          A64: XTN Vd.8B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmovn_u32 (uint32x4_t a)
          A32: VMOVN.I32 Dd, Qm
          A64: XTN Vd.4H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmovn_u64 (uint64x2_t a)
          A32: VMOVN.I64 Dd, Qm
          A64: XTN Vd.2S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqmovn_u16 (uint16x8_t a)
          A32: VQMOVN.U16 Dd, Qm
          A64: UQXTN Vd.8B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqmovn_s32 (int32x4_t a)
          A32: VQMOVN.S32 Dd, Qm
          A64: SQXTN Vd.4H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqmovn_s64 (int64x2_t a)
          A32: VQMOVN.S64 Dd, Qm
          A64: SQXTN Vd.2S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqmovn_s16 (int16x8_t a)
          A32: VQMOVN.S16 Dd, Qm
          A64: SQXTN Vd.8B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqmovn_u32 (uint32x4_t a)
          A32: VQMOVN.U32 Dd, Qm
          A64: UQXTN Vd.4H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqmovn_u64 (uint64x2_t a)
          A32: VQMOVN.U64 Dd, Qm
          A64: UQXTN Vd.2S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqmovun_s16 (int16x8_t a)
          A32: VQMOVUN.S16 Dd, Qm
          A64: SQXTUN Vd.8B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqmovun_s32 (int32x4_t a)
          A32: VQMOVUN.S32 Dd, Qm
          A64: SQXTUN Vd.4H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqmovun_s64 (int64x2_t a)
          A32: VQMOVUN.S64 Dd, Qm
          A64: SQXTUN Vd.2S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqmovun_high_s16 (uint8x8_t r, int16x8_t a)
          A32: VQMOVUN.S16 Dd+1, Qm
          A64: SQXTUN2 Vd.16B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqmovun_high_s32 (uint16x4_t r, int32x4_t a)
          A32: VQMOVUN.S32 Dd+1, Qm
          A64: SQXTUN2 Vd.8H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqmovun_high_s64 (uint32x2_t r, int64x2_t a)
          A32: VQMOVUN.S64 Dd+1, Qm
          A64: SQXTUN2 Vd.4S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqmovn_high_u16 (uint8x8_t r, uint16x8_t a)
          A32: VQMOVN.U16 Dd+1, Qm
          A64: UQXTN2 Vd.16B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqmovn_high_s32 (int16x4_t r, int32x4_t a)
          A32: VQMOVN.S32 Dd+1, Qm
          A64: SQXTN2 Vd.8H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqmovn_high_s64 (int32x2_t r, int64x2_t a)
          A32: VQMOVN.S64 Dd+1, Qm
          A64: SQXTN2 Vd.4S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqmovn_high_s16 (int8x8_t r, int16x8_t a)
          A32: VQMOVN.S16 Dd+1, Qm
          A64: SQXTN2 Vd.16B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqmovn_high_u32 (uint16x4_t r, uint32x4_t a)
          A32: VQMOVN.U32 Dd+1, Qm
          A64: UQXTN2 Vd.8H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqmovn_high_u64 (uint32x2_t r, uint64x2_t a)
          A32: VQMOVN.U64 Dd+1, Qm
          A64: UQXTN2 Vd.4S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vmovn_high_u16 (uint8x8_t r, uint16x8_t a)
          A32: VMOVN.I16 Dd+1, Qm
          A64: XTN2 Vd.16B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmovn_high_s32 (int16x4_t r, int32x4_t a)
          A32: VMOVN.I32 Dd+1, Qm
          A64: XTN2 Vd.8H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmovn_high_s64 (int32x2_t r, int64x2_t a)
          A32: VMOVN.I64 Dd+1, Qm
          A64: XTN2 Vd.4S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vmovn_high_s16 (int8x8_t r, int16x8_t a)
          A32: VMOVN.I16 Dd+1, Qm
          A64: XTN2 Vd.16B, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmovn_high_u32 (uint16x4_t r, uint32x4_t a)
          A32: VMOVN.I32 Dd+1, Qm
          A64: XTN2 Vd.8H, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ExtractNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmovn_high_u64 (uint32x2_t r, uint64x2_t a)
          A32: VMOVN.I64 Dd+1, Qm
          A64: XTN2 Vd.4S, Vn.2D
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vextq_s8 (uint8x16_t a, uint8x16_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #n
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[float], lower: System.Runtime.Intrinsics.Vector128[float], index: int) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vextq_f64 (float64x2_t a, float64x2_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*8)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*8)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vextq_s16 (int16x8_t a, int16x8_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*2)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*2)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vextq_s32 (int32x4_t a, int32x4_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*4)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*4)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vextq_s64 (int64x2_t a, int64x2_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*8)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*8)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vextq_s8 (int8x16_t a, int8x16_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #n
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[float], lower: System.Runtime.Intrinsics.Vector128[float], index: int) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vextq_f32 (float32x4_t a, float32x4_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*4)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*4)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vextq_s16 (uint16x8_t a, uint16x8_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*2)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*2)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vextq_s32 (uint32x4_t a, uint32x4_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*4)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*4)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector128(upper: System.Runtime.Intrinsics.Vector128[int], lower: System.Runtime.Intrinsics.Vector128[int], index: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vextq_s64 (uint64x2_t a, uint64x2_t b, const int n)
          A32: VEXT.8 Qd, Qn, Qm, #(n*8)
          A64: EXT Vd.16B, Vn.16B, Vm.16B, #(n*8)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[int], lower: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vext_s8 (uint8x8_t a, uint8x8_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #n
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[int], lower: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vext_s16 (int16x4_t a, int16x4_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #(n*2)
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #(n*2)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[int], lower: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vext_s32 (int32x2_t a, int32x2_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #(n*4)
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #(n*4)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[int], lower: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vext_s8 (int8x8_t a, int8x8_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #n
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[float], lower: System.Runtime.Intrinsics.Vector64[float], index: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vext_f32 (float32x2_t a, float32x2_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #(n*4)
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #(n*4)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[int], lower: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vext_s16 (uint16x4_t a, uint16x4_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #(n*2)
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #(n*2)
        """
        ...

    @staticmethod
    @overload
    def ExtractVector64(upper: System.Runtime.Intrinsics.Vector64[int], lower: System.Runtime.Intrinsics.Vector64[int], index: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vext_s32 (uint32x2_t a, uint32x2_t b, const int n)
          A32: VEXT.8 Dd, Dn, Dm, #(n*4)
          A64: EXT Vd.8B, Vn.8B, Vm.8B, #(n*4)
        """
        ...

    @staticmethod
    @overload
    def Floor(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrndm_f32 (float32x2_t a)
          A32: VRINTM.F32 Dd, Dm
          A64: FRINTM Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def Floor(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndmq_f32 (float32x4_t a)
          A32: VRINTM.F32 Qd, Qm
          A64: FRINTM Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def FloorScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrndm_f64 (float64x1_t a)
          A32: VRINTM.F64 Dd, Dm
          A64: FRINTM Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def FloorScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrndms_f32 (float32_t a)
          A32: VRINTM.F32 Sd, Sm
          A64: FRINTM Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vhadd_u8 (uint8x8_t a, uint8x8_t b)
          A32: VHADD.U8 Dd, Dn, Dm
          A64: UHADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vhadd_s16 (int16x4_t a, int16x4_t b)
          A32: VHADD.S16 Dd, Dn, Dm
          A64: SHADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vhadd_s32 (int32x2_t a, int32x2_t b)
          A32: VHADD.S32 Dd, Dn, Dm
          A64: SHADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vhadd_s8 (int8x8_t a, int8x8_t b)
          A32: VHADD.S8 Dd, Dn, Dm
          A64: SHADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vhadd_u16 (uint16x4_t a, uint16x4_t b)
          A32: VHADD.U16 Dd, Dn, Dm
          A64: UHADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vhadd_u32 (uint32x2_t a, uint32x2_t b)
          A32: VHADD.U32 Dd, Dn, Dm
          A64: UHADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vhaddq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VHADD.U8 Qd, Qn, Qm
          A64: UHADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vhaddq_s16 (int16x8_t a, int16x8_t b)
          A32: VHADD.S16 Qd, Qn, Qm
          A64: SHADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vhaddq_s32 (int32x4_t a, int32x4_t b)
          A32: VHADD.S32 Qd, Qn, Qm
          A64: SHADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vhaddq_s8 (int8x16_t a, int8x16_t b)
          A32: VHADD.S8 Qd, Qn, Qm
          A64: SHADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vhaddq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VHADD.U16 Qd, Qn, Qm
          A64: UHADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def FusedAddHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vhaddq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VHADD.U32 Qd, Qn, Qm
          A64: UHADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrhadd_u8 (uint8x8_t a, uint8x8_t b)
          A32: VRHADD.U8 Dd, Dn, Dm
          A64: URHADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrhadd_s16 (int16x4_t a, int16x4_t b)
          A32: VRHADD.S16 Dd, Dn, Dm
          A64: SRHADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrhadd_s32 (int32x2_t a, int32x2_t b)
          A32: VRHADD.S32 Dd, Dn, Dm
          A64: SRHADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrhadd_s8 (int8x8_t a, int8x8_t b)
          A32: VRHADD.S8 Dd, Dn, Dm
          A64: SRHADD Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrhadd_u16 (uint16x4_t a, uint16x4_t b)
          A32: VRHADD.U16 Dd, Dn, Dm
          A64: URHADD Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrhadd_u32 (uint32x2_t a, uint32x2_t b)
          A32: VRHADD.U32 Dd, Dn, Dm
          A64: URHADD Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrhaddq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VRHADD.U8 Qd, Qn, Qm
          A64: URHADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrhaddq_s16 (int16x8_t a, int16x8_t b)
          A32: VRHADD.S16 Qd, Qn, Qm
          A64: SRHADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrhaddq_s32 (int32x4_t a, int32x4_t b)
          A32: VRHADD.S32 Qd, Qn, Qm
          A64: SRHADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrhaddq_s8 (int8x16_t a, int8x16_t b)
          A32: VRHADD.S8 Qd, Qn, Qm
          A64: SRHADD Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrhaddq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VRHADD.U16 Qd, Qn, Qm
          A64: URHADD Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def FusedAddRoundedHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrhaddq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VRHADD.U32 Qd, Qn, Qm
          A64: URHADD Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vfma_f32 (float32x2_t a, float32x2_t b, float32x2_t c)
          A32: VFMA.F32 Dd, Dn, Dm
          A64: FMLA Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vfmaq_f32 (float32x4_t a, float32x4_t b, float32x4_t c)
          A32: VFMA.F32 Qd, Qn, Qm
          A64: FMLA Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplyAddNegatedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vfnma_f64 (float64x1_t a, float64x1_t b, float64x1_t c)
          A32: VFNMA.F64 Dd, Dn, Dm
          A64: FNMADD Dd, Dn, Dm, Da
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplyAddNegatedScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vfnmas_f32 (float32_t a, float32_t b, float32_t c)
          A32: VFNMA.F32 Sd, Sn, Sm
          A64: FNMADD Sd, Sn, Sm, Sa
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplyAddScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vfma_f64 (float64x1_t a, float64x1_t b, float64x1_t c)
          A32: VFMA.F64 Dd, Dn, Dm
          A64: FMADD Dd, Dn, Dm, Da
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplyAddScalar(addend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vfmas_f32 (float32_t a, float32_t b, float32_t c)
          A32: VFMA.F32 Sd, Sn, Sm
          A64: FMADD Sd, Sn, Sm, Sa
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vfms_f32 (float32x2_t a, float32x2_t b, float32x2_t c)
          A32: VFMS.F32 Dd, Dn, Dm
          A64: FMLS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[float], left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vfmsq_f32 (float32x4_t a, float32x4_t b, float32x4_t c)
          A32: VFMS.F32 Qd, Qn, Qm
          A64: FMLS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplySubtractNegatedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vfnms_f64 (float64x1_t a, float64x1_t b, float64x1_t c)
          A32: VFNMS.F64 Dd, Dn, Dm
          A64: FNMSUB Dd, Dn, Dm, Da
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplySubtractNegatedScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vfnmss_f32 (float32_t a, float32_t b, float32_t c)
          A32: VFNMS.F32 Sd, Sn, Sm
          A64: FNMSUB Sd, Sn, Sm, Sa
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplySubtractScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vfms_f64 (float64x1_t a, float64x1_t b, float64x1_t c)
          A32: VFMS.F64 Dd, Dn, Dm
          A64: FMSUB Dd, Dn, Dm, Da
        """
        ...

    @staticmethod
    @overload
    def FusedMultiplySubtractScalar(minuend: System.Runtime.Intrinsics.Vector64[float], left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vfmss_f32 (float32_t a, float32_t b, float32_t c)
          A32: VFMS.F32 Sd, Sn, Sm
          A64: FMSUB Sd, Sn, Sm, Sa
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vhsub_u8 (uint8x8_t a, uint8x8_t b)
          A32: VHSUB.U8 Dd, Dn, Dm
          A64: UHSUB Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vhsub_s16 (int16x4_t a, int16x4_t b)
          A32: VHSUB.S16 Dd, Dn, Dm
          A64: SHSUB Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vhsub_s32 (int32x2_t a, int32x2_t b)
          A32: VHSUB.S32 Dd, Dn, Dm
          A64: SHSUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vhsub_s8 (int8x8_t a, int8x8_t b)
          A32: VHSUB.S8 Dd, Dn, Dm
          A64: SHSUB Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vhsub_u16 (uint16x4_t a, uint16x4_t b)
          A32: VHSUB.U16 Dd, Dn, Dm
          A64: UHSUB Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vhsub_u32 (uint32x2_t a, uint32x2_t b)
          A32: VHSUB.U32 Dd, Dn, Dm
          A64: UHSUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vhsubq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VHSUB.U8 Qd, Qn, Qm
          A64: UHSUB Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vhsubq_s16 (int16x8_t a, int16x8_t b)
          A32: VHSUB.S16 Qd, Qn, Qm
          A64: SHSUB Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vhsubq_s32 (int32x4_t a, int32x4_t b)
          A32: VHSUB.S32 Qd, Qn, Qm
          A64: SHSUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vhsubq_s8 (int8x16_t a, int8x16_t b)
          A32: VHSUB.S8 Qd, Qn, Qm
          A64: SHSUB Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vhsubq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VHSUB.U16 Qd, Qn, Qm
          A64: UHSUB Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def FusedSubtractHalving(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vhsubq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VHSUB.U32 Qd, Qn, Qm
          A64: UHSUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vset_lane_u8 (uint8_t a, uint8x8_t v, const int lane)
          A32: VMOV.8 Dd[lane], Rt
          A64: INS Vd.B[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vset_lane_s16 (int16_t a, int16x4_t v, const int lane)
          A32: VMOV.16 Dd[lane], Rt
          A64: INS Vd.H[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vset_lane_s32 (int32_t a, int32x2_t v, const int lane)
          A32: VMOV.32 Dd[lane], Rt
          A64: INS Vd.S[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vset_lane_s8 (int8_t a, int8x8_t v, const int lane)
          A32: VMOV.8 Dd[lane], Rt
          A64: INS Vd.B[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[float], index: int, data: float) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vset_lane_f32 (float32_t a, float32x2_t v, const int lane)
          A32: VMOV.F32 Sd, Sm
          A64: INS Vd.S[lane], Vn.S[0]
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vset_lane_u16 (uint16_t a, uint16x4_t v, const int lane)
          A32: VMOV.16 Dd[lane], Rt
          A64: INS Vd.H[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector64[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vset_lane_u32 (uint32_t a, uint32x2_t v, const int lane)
          A32: VMOV.32 Dd[lane], Rt
          A64: INS Vd.S[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsetq_lane_u8 (uint8_t a, uint8x16_t v, const int lane)
          A32: VMOV.8 Dd[lane], Rt
          A64: INS Vd.B[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[float], index: int, data: float) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vsetq_lane_f64 (float64_t a, float64x2_t v, const int lane)
          A32: VMOV.F64 Dd, Dm
          A64: INS Vd.D[lane], Vn.D[0]
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsetq_lane_s16 (int16_t a, int16x8_t v, const int lane)
          A32: VMOV.16 Dd[lane], Rt
          A64: INS Vd.H[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsetq_lane_s32 (int32_t a, int32x4_t v, const int lane)
          A32: VMOV.32 Dd[lane], Rt
          A64: INS Vd.S[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsetq_lane_s64 (int64_t a, int64x2_t v, const int lane)
          A32: VMOV.64 Dd, Rt, Rt2
          A64: INS Vd.D[lane], Xn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vsetq_lane_s8 (int8_t a, int8x16_t v, const int lane)
          A32: VMOV.8 Dd[lane], Rt
          A64: INS Vd.B[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[float], index: int, data: float) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vsetq_lane_f32 (float32_t a, float32x4_t v, const int lane)
          A32: VMOV.F32 Sd, Sm
          A64: INS Vd.S[lane], Vn.S[0]
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsetq_lane_u16 (uint16_t a, uint16x8_t v, const int lane)
          A32: VMOV.16 Dd[lane], Rt
          A64: INS Vd.H[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsetq_lane_u32 (uint32_t a, uint32x4_t v, const int lane)
          A32: VMOV.32 Dd[lane], Rt
          A64: INS Vd.S[lane], Wn
        """
        ...

    @staticmethod
    @overload
    def Insert(vector: System.Runtime.Intrinsics.Vector128[int], index: int, data: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsetq_lane_u64 (uint64_t a, uint64x2_t v, const int lane)
          A32: VMOV.64 Dd, Rt, Rt2
          A64: INS Vd.D[lane], Xn
        """
        ...

    @staticmethod
    @overload
    def InsertScalar(result: System.Runtime.Intrinsics.Vector128[float], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vcopyq_lane_f64 (float64x2_t a, const int lane1, float64x1_t b, const int lane2)
          A32: VMOV.F64 Dd, Dm
          A64: INS Vd.D[lane1], Vn.D[0]
        """
        ...

    @staticmethod
    @overload
    def InsertScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vcopyq_lane_s64 (int64x2_t a, const int lane1, int64x1_t b, const int lane2)
          A32: VMOV Dd, Dm
          A64: INS Vd.D[lane1], Vn.D[0]
        """
        ...

    @staticmethod
    @overload
    def InsertScalar(result: System.Runtime.Intrinsics.Vector128[int], resultIndex: int, value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vcopyq_lane_u64 (uint64x2_t a, const int lane1, uint64x1_t b, const int lane2)
          A32: VMOV Dd, Dm
          A64: INS Vd.D[lane1], Vn.D[0]
        """
        ...

    @staticmethod
    @overload
    def LeadingSignCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vcls_s16 (int16x4_t a)
          A32: VCLS.S16 Dd, Dm
          A64: CLS Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def LeadingSignCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vcls_s32 (int32x2_t a)
          A32: VCLS.S32 Dd, Dm
          A64: CLS Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def LeadingSignCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vcls_s8 (int8x8_t a)
          A32: VCLS.S8 Dd, Dm
          A64: CLS Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def LeadingSignCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vclsq_s16 (int16x8_t a)
          A32: VCLS.S16 Qd, Qm
          A64: CLS Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def LeadingSignCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vclsq_s32 (int32x4_t a)
          A32: VCLS.S32 Qd, Qm
          A64: CLS Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def LeadingSignCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vclsq_s8 (int8x16_t a)
          A32: VCLS.S8 Qd, Qm
          A64: CLS Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vclz_u8 (uint8x8_t a)
          A32: VCLZ.I8 Dd, Dm
          A64: CLZ Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vclz_s16 (int16x4_t a)
          A32: VCLZ.I16 Dd, Dm
          A64: CLZ Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vclz_s32 (int32x2_t a)
          A32: VCLZ.I32 Dd, Dm
          A64: CLZ Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vclz_s8 (int8x8_t a)
          A32: VCLZ.I8 Dd, Dm
          A64: CLZ Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vclz_u16 (uint16x4_t a)
          A32: VCLZ.I16 Dd, Dm
          A64: CLZ Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vclz_u32 (uint32x2_t a)
          A32: VCLZ.I32 Dd, Dm
          A64: CLZ Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vclzq_u8 (uint8x16_t a)
          A32: VCLZ.I8 Qd, Qm
          A64: CLZ Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vclzq_s16 (int16x8_t a)
          A32: VCLZ.I16 Qd, Qm
          A64: CLZ Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vclzq_s32 (int32x4_t a)
          A32: VCLZ.I32 Qd, Qm
          A64: CLZ Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vclzq_s8 (int8x16_t a)
          A32: VCLZ.I8 Qd, Qm
          A64: CLZ Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vclzq_u16 (uint16x8_t a)
          A32: VCLZ.I16 Qd, Qm
          A64: CLZ Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def LeadingZeroCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vclzq_u32 (uint32x4_t a)
          A32: VCLZ.I32 Qd, Qm
          A64: CLZ Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vld1_lane_u8 (uint8_t const * ptr, uint8x8_t src, const int lane)
          A32: VLD1.8 { Dd[index] }, [Rn]
          A64: LD1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vld1_lane_s16 (int16_t const * ptr, int16x4_t src, const int lane)
          A32: VLD1.16 { Dd[index] }, [Rn]
          A64: LD1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vld1_lane_s32 (int32_t const * ptr, int32x2_t src, const int lane)
          A32: VLD1.32 { Dd[index] }, [Rn]
          A64: LD1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vld1_lane_s8 (int8_t const * ptr, int8x8_t src, const int lane)
          A32: VLD1.8 { Dd[index] }, [Rn]
          A64: LD1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[float], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vld1_lane_f32 (float32_t const * ptr, float32x2_t src, const int lane)
          A32: VLD1.32 { Dd[index] }, [Rn]
          A64: LD1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vld1_lane_u16 (uint16_t const * ptr, uint16x4_t src, const int lane)
          A32: VLD1.16 { Dd[index] }, [Rn]
          A64: LD1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector64[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vld1_lane_u32 (uint32_t const * ptr, uint32x2_t src, const int lane)
          A32: VLD1.32 { Dd[index] }, [Rn]
          A64: LD1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vld1q_lane_u8 (uint8_t const * ptr, uint8x16_t src, const int lane)
          A32: VLD1.8 { Dd[index] }, [Rn]
          A64: LD1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[float], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vld1q_lane_f64 (float64_t const * ptr, float64x2_t src, const int lane)
          A32: VLDR.64 Dd, [Rn]
          A64: LD1 { Vt.D }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vld1q_lane_s16 (int16_t const * ptr, int16x8_t src, const int lane)
          A32: VLD1.16 { Dd[index] }, [Rn]
          A64: LD1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vld1q_lane_s32 (int32_t const * ptr, int32x4_t src, const int lane)
          A32: VLD1.32 { Dd[index] }, [Rn]
          A64: LD1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vld1q_lane_s64 (int64_t const * ptr, int64x2_t src, const int lane)
          A32: VLDR.64 Dd, [Rn]
          A64: LD1 { Vt.D }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vld1q_lane_s8 (int8_t const * ptr, int8x16_t src, const int lane)
          A32: VLD1.8 { Dd[index] }, [Rn]
          A64: LD1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[float], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vld1q_lane_f32 (float32_t const * ptr, float32x4_t src, const int lane)
          A32: VLD1.32 { Dd[index] }, [Rn]
          A64: LD1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vld1q_lane_u16 (uint16_t const * ptr, uint16x8_t src, const int lane)
          A32: VLD1.16 { Dd[index] }, [Rn]
          A64: LD1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vld1q_lane_u32 (uint32_t const * ptr, uint32x4_t src, const int lane)
          A32: VLD1.32 { Dd[index] }, [Rn]
          A64: LD1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndInsertScalar(value: System.Runtime.Intrinsics.Vector128[int], index: int, address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vld1q_lane_u64 (uint64_t const * ptr, uint64x2_t src, const int lane)
          A32: VLDR.64 Dd, [Rn]
          A64: LD1 { Vt.D }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vld1q_dup_u8 (uint8_t const * ptr)
          A32: VLD1.8 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.16B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vld1q_dup_s16 (int16_t const * ptr)
          A32: VLD1.16 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.8H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vld1q_dup_s32 (int32_t const * ptr)
          A32: VLD1.32 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.4S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vld1q_dup_s8 (int8_t const * ptr)
          A32: VLD1.8 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.16B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vld1q_dup_f32 (float32_t const * ptr)
          A32: VLD1.32 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.4S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vld1q_dup_u16 (uint16_t const * ptr)
          A32: VLD1.16 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.8H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vld1q_dup_u32 (uint32_t const * ptr)
          A32: VLD1.32 { Dd[], Dd+1[] }, [Rn]
          A64: LD1R { Vt.4S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vld1_dup_u8 (uint8_t const * ptr)
          A32: VLD1.8 { Dd[] }, [Rn]
          A64: LD1R { Vt.8B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vld1_dup_s16 (int16_t const * ptr)
          A32: VLD1.16 { Dd[] }, [Rn]
          A64: LD1R { Vt.4H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vld1_dup_s32 (int32_t const * ptr)
          A32: VLD1.32 { Dd[] }, [Rn]
          A64: LD1R { Vt.2S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vld1_dup_s8 (int8_t const * ptr)
          A32: VLD1.8 { Dd[] }, [Rn]
          A64: LD1R { Vt.8B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vld1_dup_f32 (float32_t const * ptr)
          A32: VLD1.32 { Dd[] }, [Rn]
          A64: LD1R { Vt.2S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vld1_dup_u16 (uint16_t const * ptr)
          A32: VLD1.16 { Dd[] }, [Rn]
          A64: LD1R { Vt.4H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadAndReplicateToVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vld1_dup_u32 (uint32_t const * ptr)
          A32: VLD1.32 { Dd[] }, [Rn]
          A64: LD1R { Vt.2S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vld1q_u8 (uint8_t const * ptr)
          A32: VLD1.8 Dd, Dd+1, [Rn]
          A64: LD1 Vt.16B, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vld1q_f64 (float64_t const * ptr)
          A32: VLD1.64 Dd, Dd+1, [Rn]
          A64: LD1 Vt.2D, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vld1q_s16 (int16_t const * ptr)
          A32: VLD1.16 Dd, Dd+1, [Rn]
          A64: LD1 Vt.8H, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vld1q_s32 (int32_t const * ptr)
          A32: VLD1.32 Dd, Dd+1, [Rn]
          A64: LD1 Vt.4S, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vld1q_s64 (int64_t const * ptr)
          A32: VLD1.64 Dd, Dd+1, [Rn]
          A64: LD1 Vt.2D, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vld1q_s8 (int8_t const * ptr)
          A32: VLD1.8 Dd, Dd+1, [Rn]
          A64: LD1 Vt.16B, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vld1q_f32 (float32_t const * ptr)
          A32: VLD1.32 Dd, Dd+1, [Rn]
          A64: LD1 Vt.4S, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vld1q_s16 (uint16_t const * ptr)
          A32: VLD1.16 Dd, Dd+1, [Rn]
          A64: LD1 Vt.8H, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vld1q_s32 (uint32_t const * ptr)
          A32: VLD1.32 Dd, Dd+1, [Rn]
          A64: LD1 Vt.4S, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector128(address: typing.Any) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vld1q_u64 (uint64_t const * ptr)
          A32: VLD1.64 Dd, Dd+1, [Rn]
          A64: LD1 Vt.2D, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vld1_u8 (uint8_t const * ptr)
          A32: VLD1.8 Dd, [Rn]
          A64: LD1 Vt.8B, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vld1_f64 (float64_t const * ptr)
          A32: VLD1.64 Dd, [Rn]
          A64: LD1 Vt.1D, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vld1_s16 (int16_t const * ptr)
          A32: VLD1.16 Dd, [Rn]
          A64: LD1 Vt.4H, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vld1_s32 (int32_t const * ptr)
          A32: VLD1.32 Dd, [Rn]
          A64: LD1 Vt.2S, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vld1_s64 (int64_t const * ptr)
          A32: VLD1.64 Dd, [Rn]
          A64: LD1 Vt.1D, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vld1_s8 (int8_t const * ptr)
          A32: VLD1.8 Dd, [Rn]
          A64: LD1 Vt.8B, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vld1_f32 (float32_t const * ptr)
          A32: VLD1.32 Dd, [Rn]
          A64: LD1 Vt.2S, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vld1_u16 (uint16_t const * ptr)
          A32: VLD1.16 Dd, [Rn]
          A64: LD1 Vt.4H, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vld1_u32 (uint32_t const * ptr)
          A32: VLD1.32 Dd, [Rn]
          A64: LD1 Vt.2S, [Xn]
        """
        ...

    @staticmethod
    @overload
    def LoadVector64(address: typing.Any) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vld1_u64 (uint64_t const * ptr)
          A32: VLD1.64 Dd, [Rn]
          A64: LD1 Vt.1D, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmax_u8 (uint8x8_t a, uint8x8_t b)
          A32: VMAX.U8 Dd, Dn, Dm
          A64: UMAX Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmax_s16 (int16x4_t a, int16x4_t b)
          A32: VMAX.S16 Dd, Dn, Dm
          A64: SMAX Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmax_s32 (int32x2_t a, int32x2_t b)
          A32: VMAX.S32 Dd, Dn, Dm
          A64: SMAX Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmax_s8 (int8x8_t a, int8x8_t b)
          A32: VMAX.S8 Dd, Dn, Dm
          A64: SMAX Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmax_f32 (float32x2_t a, float32x2_t b)
          A32: VMAX.F32 Dd, Dn, Dm
          A64: FMAX Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmax_u16 (uint16x4_t a, uint16x4_t b)
          A32: VMAX.U16 Dd, Dn, Dm
          A64: UMAX Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmax_u32 (uint32x2_t a, uint32x2_t b)
          A32: VMAX.U32 Dd, Dn, Dm
          A64: UMAX Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vmaxq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VMAX.U8 Qd, Qn, Qm
          A64: UMAX Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmaxq_s16 (int16x8_t a, int16x8_t b)
          A32: VMAX.S16 Qd, Qn, Qm
          A64: SMAX Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmaxq_s32 (int32x4_t a, int32x4_t b)
          A32: VMAX.S32 Qd, Qn, Qm
          A64: SMAX Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vmaxq_s8 (int8x16_t a, int8x16_t b)
          A32: VMAX.S8 Qd, Qn, Qm
          A64: SMAX Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmaxq_f32 (float32x4_t a, float32x4_t b)
          A32: VMAX.F32 Qd, Qn, Qm
          A64: FMAX Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmaxq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VMAX.U16 Qd, Qn, Qm
          A64: UMAX Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Max(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmaxq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VMAX.U32 Qd, Qn, Qm
          A64: UMAX Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MaxNumber(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmaxnm_f32 (float32x2_t a, float32x2_t b)
          A32: VMAXNM.F32 Dd, Dn, Dm
          A64: FMAXNM Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MaxNumber(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmaxnmq_f32 (float32x4_t a, float32x4_t b)
          A32: VMAXNM.F32 Qd, Qn, Qm
          A64: FMAXNM Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MaxNumberScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vmaxnm_f64 (float64x1_t a, float64x1_t b)
          A32: VMAXNM.F64 Dd, Dn, Dm
          A64: FMAXNM Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def MaxNumberScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vmaxnms_f32 (float32_t a, float32_t b)
          A32: VMAXNM.F32 Sd, Sn, Sm
          A64: FMAXNM Sd, Sn, Sm
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vpmax_u8 (uint8x8_t a, uint8x8_t b)
          A32: VPMAX.U8 Dd, Dn, Dm
          A64: UMAXP Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vpmax_s16 (int16x4_t a, int16x4_t b)
          A32: VPMAX.S16 Dd, Dn, Dm
          A64: SMAXP Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vpmax_s32 (int32x2_t a, int32x2_t b)
          A32: VPMAX.S32 Dd, Dn, Dm
          A64: SMAXP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vpmax_s8 (int8x8_t a, int8x8_t b)
          A32: VPMAX.S8 Dd, Dn, Dm
          A64: SMAXP Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vpmax_f32 (float32x2_t a, float32x2_t b)
          A32: VPMAX.F32 Dd, Dn, Dm
          A64: FMAXP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vpmax_u16 (uint16x4_t a, uint16x4_t b)
          A32: VPMAX.U16 Dd, Dn, Dm
          A64: UMAXP Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MaxPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vpmax_u32 (uint32x2_t a, uint32x2_t b)
          A32: VPMAX.U32 Dd, Dn, Dm
          A64: UMAXP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmin_u8 (uint8x8_t a, uint8x8_t b)
          A32: VMIN.U8 Dd, Dn, Dm
          A64: UMIN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmin_s16 (int16x4_t a, int16x4_t b)
          A32: VMIN.S16 Dd, Dn, Dm
          A64: SMIN Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmin_s32 (int32x2_t a, int32x2_t b)
          A32: VMIN.S32 Dd, Dn, Dm
          A64: SMIN Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmin_s8 (int8x8_t a, int8x8_t b)
          A32: VMIN.S8 Dd, Dn, Dm
          A64: SMIN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmin_f32 (float32x2_t a, float32x2_t b)
          A32: VMIN.F32 Dd, Dn, Dm
          A64: FMIN Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmin_u16 (uint16x4_t a, uint16x4_t b)
          A32: VMIN.U16 Dd, Dn, Dm
          A64: UMIN Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmin_u32 (uint32x2_t a, uint32x2_t b)
          A32: VMIN.U32 Dd, Dn, Dm
          A64: UMIN Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vminq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VMIN.U8 Qd, Qn, Qm
          A64: UMIN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vminq_s16 (int16x8_t a, int16x8_t b)
          A32: VMIN.S16 Qd, Qn, Qm
          A64: SMIN Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vminq_s32 (int32x4_t a, int32x4_t b)
          A32: VMIN.S32 Qd, Qn, Qm
          A64: SMIN Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vminq_s8 (int8x16_t a, int8x16_t b)
          A32: VMIN.S8 Qd, Qn, Qm
          A64: SMIN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vminq_f32 (float32x4_t a, float32x4_t b)
          A32: VMIN.F32 Qd, Qn, Qm
          A64: FMIN Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vminq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VMIN.U16 Qd, Qn, Qm
          A64: UMIN Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Min(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vminq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VMIN.U32 Qd, Qn, Qm
          A64: UMIN Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MinNumber(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vminnm_f32 (float32x2_t a, float32x2_t b)
          A32: VMINNM.F32 Dd, Dn, Dm
          A64: FMINNM Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MinNumber(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vminnmq_f32 (float32x4_t a, float32x4_t b)
          A32: VMINNM.F32 Qd, Qn, Qm
          A64: FMINNM Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MinNumberScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vminnm_f64 (float64x1_t a, float64x1_t b)
          A32: VMINNM.F64 Dd, Dn, Dm
          A64: FMINNM Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def MinNumberScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vminnms_f32 (float32_t a, float32_t b)
          A32: VMINNM.F32 Sd, Sn, Sm
          A64: FMINNM Sd, Sn, Sm
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vpmin_u8 (uint8x8_t a, uint8x8_t b)
          A32: VPMIN.U8 Dd, Dn, Dm
          A64: UMINP Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vpmin_s16 (int16x4_t a, int16x4_t b)
          A32: VPMIN.S16 Dd, Dn, Dm
          A64: SMINP Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vpmin_s32 (int32x2_t a, int32x2_t b)
          A32: VPMIN.S32 Dd, Dn, Dm
          A64: SMINP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vpmin_s8 (int8x8_t a, int8x8_t b)
          A32: VPMIN.S8 Dd, Dn, Dm
          A64: SMINP Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vpmin_f32 (float32x2_t a, float32x2_t b)
          A32: VPMIN.F32 Dd, Dn, Dm
          A64: FMINP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vpmin_u16 (uint16x4_t a, uint16x4_t b)
          A32: VPMIN.U16 Dd, Dn, Dm
          A64: UMINP Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MinPairwise(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vpmin_u32 (uint32x2_t a, uint32x2_t b)
          A32: VPMIN.U32 Dd, Dn, Dm
          A64: UMINP Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmul_u8 (uint8x8_t a, uint8x8_t b)
          A32: VMUL.I8 Dd, Dn, Dm
          A64: MUL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmul_s16 (int16x4_t a, int16x4_t b)
          A32: VMUL.I16 Dd, Dn, Dm
          A64: MUL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmul_s32 (int32x2_t a, int32x2_t b)
          A32: VMUL.I32 Dd, Dn, Dm
          A64: MUL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmul_s8 (int8x8_t a, int8x8_t b)
          A32: VMUL.I8 Dd, Dn, Dm
          A64: MUL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmul_f32 (float32x2_t a, float32x2_t b)
          A32: VMUL.F32 Dd, Dn, Dm
          A64: FMUL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmul_u16 (uint16x4_t a, uint16x4_t b)
          A32: VMUL.I16 Dd, Dn, Dm
          A64: MUL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmul_u32 (uint32x2_t a, uint32x2_t b)
          A32: VMUL.I32 Dd, Dn, Dm
          A64: MUL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vmulq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VMUL.I8 Qd, Qn, Qm
          A64: MUL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmulq_s16 (int16x8_t a, int16x8_t b)
          A32: VMUL.I16 Qd, Qn, Qm
          A64: MUL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmulq_s32 (int32x4_t a, int32x4_t b)
          A32: VMUL.I32 Qd, Qn, Qm
          A64: MUL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vmulq_s8 (int8x16_t a, int8x16_t b)
          A32: VMUL.I8 Qd, Qn, Qm
          A64: MUL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmulq_f32 (float32x4_t a, float32x4_t b)
          A32: VMUL.F32 Qd, Qn, Qm
          A64: FMUL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmulq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VMUL.I16 Qd, Qn, Qm
          A64: MUL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Multiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmulq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VMUL.I32 Qd, Qn, Qm
          A64: MUL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmla_u8 (uint8x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VMLA.I8 Dd, Dn, Dm
          A64: MLA Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmla_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VMLA.I16 Dd, Dn, Dm
          A64: MLA Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmla_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VMLA.I32 Dd, Dn, Dm
          A64: MLA Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmla_s8 (int8x8_t a, int8x8_t b, int8x8_t c)
          A32: VMLA.I8 Dd, Dn, Dm
          A64: MLA Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmla_u16 (uint16x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VMLA.I16 Dd, Dn, Dm
          A64: MLA Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmla_u32 (uint32x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VMLA.I32 Dd, Dn, Dm
          A64: MLA Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vmlaq_u8 (uint8x16_t a, uint8x16_t b, uint8x16_t c)
          A32: VMLA.I8 Qd, Qn, Qm
          A64: MLA Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlaq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VMLA.I16 Qd, Qn, Qm
          A64: MLA Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlaq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VMLA.I32 Qd, Qn, Qm
          A64: MLA Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vmlaq_s8 (int8x16_t a, int8x16_t b, int8x16_t c)
          A32: VMLA.I8 Qd, Qn, Qm
          A64: MLA Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlaq_u16 (uint16x8_t a, uint16x8_t b, uint16x8_t c)
          A32: VMLA.I16 Qd, Qn, Qm
          A64: MLA Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlaq_u32 (uint32x4_t a, uint32x4_t b, uint32x4_t c)
          A32: VMLA.I32 Qd, Qn, Qm
          A64: MLA Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmla_n_s16 (int16x4_t a, int16x4_t b, int16_t c)
          A32: VMLA.I16 Dd, Dn, Dm[0]
          A64: MLA Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmla_n_s32 (int32x2_t a, int32x2_t b, int32_t c)
          A32: VMLA.I32 Dd, Dn, Dm[0]
          A64: MLA Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmla_n_u16 (uint16x4_t a, uint16x4_t b, uint16_t c)
          A32: VMLA.I16 Dd, Dn, Dm[0]
          A64: MLA Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmla_n_u32 (uint32x2_t a, uint32x2_t b, uint32_t c)
          A32: VMLA.I32 Dd, Dn, Dm[0]
          A64: MLA Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlaq_n_s16 (int16x8_t a, int16x8_t b, int16_t c)
          A32: VMLA.I16 Qd, Qn, Dm[0]
          A64: MLA Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlaq_n_s32 (int32x4_t a, int32x4_t b, int32_t c)
          A32: VMLA.I32 Qd, Qn, Dm[0]
          A64: MLA Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlaq_n_u16 (uint16x8_t a, uint16x8_t b, uint16_t c)
          A32: VMLA.I16 Qd, Qn, Dm[0]
          A64: MLA Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddByScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlaq_n_u32 (uint32x4_t a, uint32x4_t b, uint32_t c)
          A32: VMLA.I32 Qd, Qn, Dm[0]
          A64: MLA Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmla_lane_s16 (int16x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VMLA.I16 Dd, Dn, Dm[lane]
          A64: MLA Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmla_laneq_s16 (int16x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VMLA.I16 Dd, Dn, Dm[lane]
          A64: MLA Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmla_lane_s32 (int32x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VMLA.I32 Dd, Dn, Dm[lane]
          A64: MLA Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmla_laneq_s32 (int32x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VMLA.I32 Dd, Dn, Dm[lane]
          A64: MLA Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmla_lane_u16 (uint16x4_t a, uint16x4_t b, uint16x4_t v, const int lane)
          A32: VMLA.I16 Dd, Dn, Dm[lane]
          A64: MLA Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmla_laneq_u16 (uint16x4_t a, uint16x4_t b, uint16x8_t v, const int lane)
          A32: VMLA.I16 Dd, Dn, Dm[lane]
          A64: MLA Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmla_lane_u32 (uint32x2_t a, uint32x2_t b, uint32x2_t v, const int lane)
          A32: VMLA.I32 Dd, Dn, Dm[lane]
          A64: MLA Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmla_laneq_u32 (uint32x2_t a, uint32x2_t b, uint32x4_t v, const int lane)
          A32: VMLA.I32 Dd, Dn, Dm[lane]
          A64: MLA Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlaq_lane_s16 (int16x8_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VMLA.I16 Qd, Qn, Dm[lane]
          A64: MLA Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlaq_laneq_s16 (int16x8_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VMLA.I16 Qd, Qn, Dm[lane]
          A64: MLA Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlaq_lane_s32 (int32x4_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VMLA.I32 Qd, Qn, Dm[lane]
          A64: MLA Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlaq_laneq_s32 (int32x4_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VMLA.I32 Qd, Qn, Dm[lane]
          A64: MLA Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlaq_lane_u16 (uint16x8_t a, uint16x8_t b, uint16x4_t v, const int lane)
          A32: VMLA.I16 Qd, Qn, Dm[lane]
          A64: MLA Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlaq_laneq_u16 (uint16x8_t a, uint16x8_t b, uint16x8_t v, const int lane)
          A32: VMLA.I16 Qd, Qn, Dm[lane]
          A64: MLA Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlaq_lane_u32 (uint32x4_t a, uint32x4_t b, uint32x2_t v, const int lane)
          A32: VMLA.I32 Qd, Qn, Dm[lane]
          A64: MLA Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyAddBySelectedScalar(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlaq_laneq_u32 (uint32x4_t a, uint32x4_t b, uint32x4_t v, const int lane)
          A32: VMLA.I32 Qd, Qn, Dm[lane]
          A64: MLA Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmul_n_s16 (int16x4_t a, int16_t b)
          A32: VMUL.I16 Dd, Dn, Dm[0]
          A64: MUL Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmul_n_s32 (int32x2_t a, int32_t b)
          A32: VMUL.I32 Dd, Dn, Dm[0]
          A64: MUL Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmul_n_f32 (float32x2_t a, float32_t b)
          A32: VMUL.F32 Dd, Dn, Dm[0]
          A64: FMUL Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmul_n_u16 (uint16x4_t a, uint16_t b)
          A32: VMUL.I16 Dd, Dn, Dm[0]
          A64: MUL Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmul_n_u32 (uint32x2_t a, uint32_t b)
          A32: VMUL.I32 Dd, Dn, Dm[0]
          A64: MUL Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmulq_n_s16 (int16x8_t a, int16_t b)
          A32: VMUL.I16 Qd, Qn, Dm[0]
          A64: MUL Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmulq_n_s32 (int32x4_t a, int32_t b)
          A32: VMUL.I32 Qd, Qn, Dm[0]
          A64: MUL Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmulq_n_f32 (float32x4_t a, float32_t b)
          A32: VMUL.F32 Qd, Qn, Dm[0]
          A64: FMUL Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmulq_n_u16 (uint16x8_t a, uint16_t b)
          A32: VMUL.I16 Qd, Qn, Dm[0]
          A64: MUL Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyByScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmulq_n_u32 (uint32x4_t a, uint32_t b)
          A32: VMUL.I32 Qd, Qn, Dm[0]
          A64: MUL Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmul_lane_s16 (int16x4_t a, int16x4_t v, const int lane)
          A32: VMUL.I16 Dd, Dn, Dm[lane]
          A64: MUL Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmul_laneq_s16 (int16x4_t a, int16x8_t v, const int lane)
          A32: VMUL.I16 Dd, Dn, Dm[lane]
          A64: MUL Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmul_lane_s32 (int32x2_t a, int32x2_t v, const int lane)
          A32: VMUL.I32 Dd, Dn, Dm[lane]
          A64: MUL Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmul_laneq_s32 (int32x2_t a, int32x4_t v, const int lane)
          A32: VMUL.I32 Dd, Dn, Dm[lane]
          A64: MUL Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmul_lane_f32 (float32x2_t a, float32x2_t v, const int lane)
          A32: VMUL.F32 Dd, Dn, Dm[lane]
          A64: FMUL Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmul_laneq_f32 (float32x2_t a, float32x4_t v, const int lane)
          A32: VMUL.F32 Dd, Dn, Dm[lane]
          A64: FMUL Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmul_lane_u16 (uint16x4_t a, uint16x4_t v, const int lane)
          A32: VMUL.I16 Dd, Dn, Dm[lane]
          A64: MUL Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmul_laneq_u16 (uint16x4_t a, uint16x8_t v, const int lane)
          A32: VMUL.I16 Dd, Dn, Dm[lane]
          A64: MUL Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmul_lane_u32 (uint32x2_t a, uint32x2_t v, const int lane)
          A32: VMUL.I32 Dd, Dn, Dm[lane]
          A64: MUL Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmul_laneq_u32 (uint32x2_t a, uint32x4_t v, const int lane)
          A32: VMUL.I32 Dd, Dn, Dm[lane]
          A64: MUL Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmulq_lane_s16 (int16x8_t a, int16x4_t v, const int lane)
          A32: VMUL.I16 Qd, Qn, Dm[lane]
          A64: MUL Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmulq_laneq_s16 (int16x8_t a, int16x8_t v, const int lane)
          A32: VMUL.I16 Qd, Qn, Dm[lane]
          A64: MUL Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmulq_lane_s32 (int32x4_t a, int32x2_t v, const int lane)
          A32: VMUL.I32 Qd, Qn, Dm[lane]
          A64: MUL Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmulq_laneq_s32 (int32x4_t a, int32x4_t v, const int lane)
          A32: VMUL.I32 Qd, Qn, Dm[lane]
          A64: MUL Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmulq_lane_f32 (float32x4_t a, float32x2_t v, const int lane)
          A32: VMUL.F32 Qd, Qn, Dm[lane]
          A64: FMUL Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmulq_laneq_f32 (float32x4_t a, float32x4_t v, const int lane)
          A32: VMUL.F32 Qd, Qn, Dm[lane]
          A64: FMUL Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmulq_lane_u16 (uint16x8_t a, uint16x4_t v, const int lane)
          A32: VMUL.I16 Qd, Qn, Dm[lane]
          A64: MUL Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmulq_laneq_u16 (uint16x8_t a, uint16x8_t v, const int lane)
          A32: VMUL.I16 Qd, Qn, Dm[lane]
          A64: MUL Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmulq_lane_u32 (uint32x4_t a, uint32x2_t v, const int lane)
          A32: VMUL.I32 Qd, Qn, Dm[lane]
          A64: MUL Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmulq_laneq_u32 (uint32x4_t a, uint32x4_t v, const int lane)
          A32: VMUL.I32 Qd, Qn, Dm[lane]
          A64: MUL Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmull_lane_s16 (int16x4_t a, int16x4_t v, const int lane)
          A32: VMULL.S16 Qd, Dn, Dm[lane]
          A64: SMULL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmull_laneq_s16 (int16x4_t a, int16x8_t v, const int lane)
          A32: VMULL.S16 Qd, Dn, Dm[lane]
          A64: SMULL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmull_lane_s32 (int32x2_t a, int32x2_t v, const int lane)
          A32: VMULL.S32 Qd, Dn, Dm[lane]
          A64: SMULL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmull_laneq_s32 (int32x2_t a, int32x4_t v, const int lane)
          A32: VMULL.S32 Qd, Dn, Dm[lane]
          A64: SMULL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmull_lane_u16 (uint16x4_t a, uint16x4_t v, const int lane)
          A32: VMULL.U16 Qd, Dn, Dm[lane]
          A64: UMULL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmull_laneq_u16 (uint16x4_t a, uint16x8_t v, const int lane)
          A32: VMULL.U16 Qd, Dn, Dm[lane]
          A64: UMULL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmull_lane_u32 (uint32x2_t a, uint32x2_t v, const int lane)
          A32: VMULL.U32 Qd, Dn, Dm[lane]
          A64: UMULL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmull_laneq_u32 (uint32x2_t a, uint32x4_t v, const int lane)
          A32: VMULL.U32 Qd, Dn, Dm[lane]
          A64: UMULL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlal_lane_s16 (int32x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VMLAL.S16 Qd, Dn, Dm[lane]
          A64: SMLAL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlal_laneq_s16 (int32x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VMLAL.S16 Qd, Dn, Dm[lane]
          A64: SMLAL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlal_lane_s32 (int64x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VMLAL.S32 Qd, Dn, Dm[lane]
          A64: SMLAL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlal_laneq_s32 (int64x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VMLAL.S32 Qd, Dn, Dm[lane]
          A64: SMLAL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlal_lane_u16 (uint32x4_t a, uint16x4_t b, uint16x4_t v, const int lane)
          A32: VMLAL.U16 Qd, Dn, Dm[lane]
          A64: UMLAL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlal_laneq_u16 (uint32x4_t a, uint16x4_t b, uint16x8_t v, const int lane)
          A32: VMLAL.U16 Qd, Dn, Dm[lane]
          A64: UMLAL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlal_lane_u32 (uint64x2_t a, uint32x2_t b, uint32x2_t v, const int lane)
          A32: VMLAL.U32 Qd, Dn, Dm[lane]
          A64: UMLAL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlal_laneq_u32 (uint64x2_t a, uint32x2_t b, uint32x4_t v, const int lane)
          A32: VMLAL.U32 Qd, Dn, Dm[lane]
          A64: UMLAL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsl_lane_s16 (int32x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VMLSL.S16 Qd, Dn, Dm[lane]
          A64: SMLSL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsl_laneq_s16 (int32x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VMLSL.S16 Qd, Dn, Dm[lane]
          A64: SMLSL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlsl_lane_s32 (int64x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VMLSL.S32 Qd, Dn, Dm[lane]
          A64: SMLSL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlsl_laneq_s32 (int64x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VMLSL.S32 Qd, Dn, Dm[lane]
          A64: SMLSL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsl_lane_u16 (uint32x4_t a, uint16x4_t b, uint16x4_t v, const int lane)
          A32: VMLSL.U16 Qd, Dn, Dm[lane]
          A64: UMLSL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsl_laneq_u16 (uint32x4_t a, uint16x4_t b, uint16x8_t v, const int lane)
          A32: VMLSL.U16 Qd, Dn, Dm[lane]
          A64: UMLSL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlsl_lane_u32 (uint64x2_t a, uint32x2_t b, uint32x2_t v, const int lane)
          A32: VMLSL.U32 Qd, Dn, Dm[lane]
          A64: UMLSL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlsl_laneq_u32 (uint64x2_t a, uint32x2_t b, uint32x4_t v, const int lane)
          A32: VMLSL.U32 Qd, Dn, Dm[lane]
          A64: UMLSL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmull_high_lane_s16 (int16x8_t a, int16x4_t v, const int lane)
          A32: VMULL.S16 Qd, Dn+1, Dm[lane]
          A64: SMULL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmull_high_laneq_s16 (int16x8_t a, int16x8_t v, const int lane)
          A32: VMULL.S16 Qd, Dn+1, Dm[lane]
          A64: SMULL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmull_high_lane_s32 (int32x4_t a, int32x2_t v, const int lane)
          A32: VMULL.S32 Qd, Dn+1, Dm[lane]
          A64: SMULL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmull_high_laneq_s32 (int32x4_t a, int32x4_t v, const int lane)
          A32: VMULL.S32 Qd, Dn+1, Dm[lane]
          A64: SMULL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmull_high_lane_u16 (uint16x8_t a, uint16x4_t v, const int lane)
          A32: VMULL.U16 Qd, Dn+1, Dm[lane]
          A64: UMULL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmull_high_laneq_u16 (uint16x8_t a, uint16x8_t v, const int lane)
          A32: VMULL.U16 Qd, Dn+1, Dm[lane]
          A64: UMULL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmull_high_lane_u32 (uint32x4_t a, uint32x2_t v, const int lane)
          A32: VMULL.U32 Qd, Dn+1, Dm[lane]
          A64: UMULL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmull_high_laneq_u32 (uint32x4_t a, uint32x4_t v, const int lane)
          A32: VMULL.U32 Qd, Dn+1, Dm[lane]
          A64: UMULL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlal_high_lane_s16 (int32x4_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VMLAL.S16 Qd, Dn+1, Dm[lane]
          A64: SMLAL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlal_high_laneq_s16 (int32x4_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VMLAL.S16 Qd, Dn+1, Dm[lane]
          A64: SMLAL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlal_high_lane_s32 (int64x2_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VMLAL.S32 Qd, Dn+1, Dm[lane]
          A64: SMLAL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlal_high_laneq_s32 (int64x2_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VMLAL.S32 Qd, Dn+1, Dm[lane]
          A64: SMLAL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlal_high_lane_u16 (uint32x4_t a, uint16x8_t b, uint16x4_t v, const int lane)
          A32: VMLAL.U16 Qd, Dn+1, Dm[lane]
          A64: UMLAL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlal_high_laneq_u16 (uint32x4_t a, uint16x8_t b, uint16x8_t v, const int lane)
          A32: VMLAL.U16 Qd, Dn+1, Dm[lane]
          A64: UMLAL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlal_high_lane_u32 (uint64x2_t a, uint32x4_t b, uint32x2_t v, const int lane)
          A32: VMLAL.U32 Qd, Dn+1, Dm[lane]
          A64: UMLAL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlal_high_laneq_u32 (uint64x2_t a, uint32x4_t b, uint32x4_t v, const int lane)
          A32: VMLAL.U32 Qd, Dn+1, Dm[lane]
          A64: UMLAL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsl_high_lane_s16 (int32x4_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VMLSL.S16 Qd, Dn+1, Dm[lane]
          A64: SMLSL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsl_high_laneq_s16 (int32x4_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VMLSL.S16 Qd, Dn+1, Dm[lane]
          A64: SMLSL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlsl_high_lane_s32 (int64x2_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VMLSL.S32 Qd, Dn+1, Dm[lane]
          A64: SMLSL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlsl_high_laneq_s32 (int64x2_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VMLSL.S32 Qd, Dn+1, Dm[lane]
          A64: SMLSL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsl_high_lane_u16 (uint32x4_t a, uint16x8_t b, uint16x4_t v, const int lane)
          A32: VMLSL.U16 Qd, Dn+1, Dm[lane]
          A64: UMLSL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsl_high_laneq_u16 (uint32x4_t a, uint16x8_t b, uint16x8_t v, const int lane)
          A32: VMLSL.U16 Qd, Dn+1, Dm[lane]
          A64: UMLSL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlsl_high_lane_u32 (uint64x2_t a, uint32x4_t b, uint32x2_t v, const int lane)
          A32: VMLSL.U32 Qd, Dn+1, Dm[lane]
          A64: UMLSL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyBySelectedScalarWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlsl_high_laneq_u32 (uint64x2_t a, uint32x4_t b, uint32x4_t v, const int lane)
          A32: VMLSL.U32 Qd, Dn+1, Dm[lane]
          A64: UMLSL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqdmulh_n_s16 (int16x4_t a, int16_t b)
          A32: VQDMULH.S16 Dd, Dn, Dm[0]
          A64: SQDMULH Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqdmulh_n_s32 (int32x2_t a, int32_t b)
          A32: VQDMULH.S32 Dd, Dn, Dm[0]
          A64: SQDMULH Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqdmulhq_n_s16 (int16x8_t a, int16_t b)
          A32: VQDMULH.S16 Qd, Qn, Dm[0]
          A64: SQDMULH Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmulhq_n_s32 (int32x4_t a, int32_t b)
          A32: VQDMULH.S32 Qd, Qn, Dm[0]
          A64: SQDMULH Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqdmulh_lane_s16 (int16x4_t a, int16x4_t v, const int lane)
          A32: VQDMULH.S16 Dd, Dn, Dm[lane]
          A64: SQDMULH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqdmulh_laneq_s16 (int16x4_t a, int16x8_t v, const int lane)
          A32: VQDMULH.S16 Dd, Dn, Dm[lane]
          A64: SQDMULH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqdmulh_lane_s32 (int32x2_t a, int32x2_t v, const int lane)
          A32: VQDMULH.S32 Dd, Dn, Dm[lane]
          A64: SQDMULH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqdmulh_laneq_s32 (int32x2_t a, int32x4_t v, const int lane)
          A32: VQDMULH.S32 Dd, Dn, Dm[lane]
          A64: SQDMULH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqdmulhq_lane_s16 (int16x8_t a, int16x4_t v, const int lane)
          A32: VQDMULH.S16 Qd, Qn, Dm[lane]
          A64: SQDMULH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqdmulhq_laneq_s16 (int16x8_t a, int16x8_t v, const int lane)
          A32: VQDMULH.S16 Qd, Qn, Dm[lane]
          A64: SQDMULH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmulhq_lane_s32 (int32x4_t a, int32x2_t v, const int lane)
          A32: VQDMULH.S32 Qd, Qn, Dm[lane]
          A64: SQDMULH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmulhq_laneq_s32 (int32x4_t a, int32x4_t v, const int lane)
          A32: VQDMULH.S32 Qd, Qn, Dm[lane]
          A64: SQDMULH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqdmulh_s16 (int16x4_t a, int16x4_t b)
          A32: VQDMULH.S16 Dd, Dn, Dm
          A64: SQDMULH Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqdmulh_s32 (int32x2_t a, int32x2_t b)
          A32: VQDMULH.S32 Dd, Dn, Dm
          A64: SQDMULH Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqdmulhq_s16 (int16x8_t a, int16x8_t b)
          A32: VQDMULH.S16 Qd, Qn, Qm
          A64: SQDMULH Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmulhq_s32 (int32x4_t a, int32x4_t b)
          A32: VQDMULH.S32 Qd, Qn, Qm
          A64: SQDMULH Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_s16 (int32x4_t a, int16x4_t b, int16x4_t c)
          A32: VQDMLAL.S16 Qd, Dn, Dm
          A64: SQDMLAL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_s32 (int64x2_t a, int32x2_t b, int32x2_t c)
          A32: VQDMLAL.S32 Qd, Dn, Dm
          A64: SQDMLAL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_s16 (int32x4_t a, int16x4_t b, int16x4_t c)
          A32: VQDMLSL.S16 Qd, Dn, Dm
          A64: SQDMLSL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_s32 (int64x2_t a, int32x2_t b, int32x2_t c)
          A32: VQDMLSL.S32 Qd, Dn, Dm
          A64: SQDMLSL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerByScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_n_s16 (int32x4_t a, int16x4_t b, int16_t c)
          A32: VQDMLAL.S16 Qd, Dn, Dm[0]
          A64: SQDMLAL Vd.4S, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerByScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_n_s32 (int64x2_t a, int32x2_t b, int32_t c)
          A32: VQDMLAL.S32 Qd, Dn, Dm[0]
          A64: SQDMLAL Vd.2D, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerByScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_n_s16 (int32x4_t a, int16x4_t b, int16_t c)
          A32: VQDMLSL.S16 Qd, Dn, Dm[0]
          A64: SQDMLSL Vd.4S, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerByScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_n_s32 (int64x2_t a, int32x2_t b, int32_t c)
          A32: VQDMLSL.S32 Qd, Dn, Dm[0]
          A64: SQDMLSL Vd.2D, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_lane_s16 (int32x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VQDMLAL.S16 Qd, Dn, Dm[lane]
          A64: SQDMLAL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_laneq_s16 (int32x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VQDMLAL.S16 Qd, Dn, Dm[lane]
          A64: SQDMLAL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_lane_s32 (int64x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VQDMLAL.S32 Qd, Dn, Dm[lane]
          A64: SQDMLAL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_laneq_s32 (int64x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VQDMLAL.S32 Qd, Dn, Dm[lane]
          A64: SQDMLAL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_lane_s16 (int32x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VQDMLSL.S16 Qd, Dn, Dm[lane]
          A64: SQDMLSL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_laneq_s16 (int32x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VQDMLSL.S16 Qd, Dn, Dm[lane]
          A64: SQDMLSL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_lane_s32 (int64x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VQDMLSL.S32 Qd, Dn, Dm[lane]
          A64: SQDMLSL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningLowerBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_laneq_s32 (int64x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VQDMLSL.S32 Qd, Dn, Dm[lane]
          A64: SQDMLSL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_s16 (int16x4_t a, int16x4_t b)
          A32: VQDMULL.S16 Qd, Dn, Dm
          A64: SQDMULL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_s32 (int32x2_t a, int32x2_t b)
          A32: VQDMULL.S32 Qd, Dn, Dm
          A64: SQDMULL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLowerByScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_n_s16 (int16x4_t a, int16_t b)
          A32: VQDMULL.S16 Qd, Dn, Dm[0]
          A64: SQDMULL Vd.4S, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLowerByScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_n_s32 (int32x2_t a, int32_t b)
          A32: VQDMULL.S32 Qd, Dn, Dm[0]
          A64: SQDMULL Vd.2D, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLowerBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_lane_s16 (int16x4_t a, int16x4_t v, const int lane)
          A32: VQDMULL.S16 Qd, Dn, Dm[lane]
          A64: SQDMULL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLowerBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_laneq_s16 (int16x4_t a, int16x8_t v, const int lane)
          A32: VQDMULL.S16 Qd, Dn, Dm[lane]
          A64: SQDMULL Vd.4S, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLowerBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_lane_s32 (int32x2_t a, int32x2_t v, const int lane)
          A32: VQDMULL.S32 Qd, Dn, Dm[lane]
          A64: SQDMULL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateLowerBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_laneq_s32 (int32x2_t a, int32x4_t v, const int lane)
          A32: VQDMULL.S32 Qd, Dn, Dm[lane]
          A64: SQDMULL Vd.2D, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_high_s16 (int16x8_t a, int16x8_t b)
          A32: VQDMULL.S16 Qd, Dn+1, Dm+1
          A64: SQDMULL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_high_s32 (int32x4_t a, int32x4_t b)
          A32: VQDMULL.S32 Qd, Dn+1, Dm+1
          A64: SQDMULL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpperByScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_high_n_s16 (int16x8_t a, int16_t b)
          A32: VQDMULL.S16 Qd, Dn+1, Dm[0]
          A64: SQDMULL2 Vd.4S, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpperByScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_high_n_s32 (int32x4_t a, int32_t b)
          A32: VQDMULL.S32 Qd, Dn+1, Dm[0]
          A64: SQDMULL2 Vd.2D, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpperBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_high_lane_s16 (int16x8_t a, int16x4_t v, const int lane)
          A32: VQDMULL.S16 Qd, Dn+1, Dm[lane]
          A64: SQDMULL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpperBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmull_high_laneq_s16 (int16x8_t a, int16x8_t v, const int lane)
          A32: VQDMULL.S16 Qd, Dn+1, Dm[lane]
          A64: SQDMULL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpperBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_high_lane_s32 (int32x4_t a, int32x2_t v, const int lane)
          A32: VQDMULL.S32 Qd, Dn+1, Dm[lane]
          A64: SQDMULL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningSaturateUpperBySelectedScalar(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmull_high_laneq_s32 (int32x4_t a, int32x4_t v, const int lane)
          A32: VQDMULL.S32 Qd, Dn+1, Dm[lane]
          A64: SQDMULL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_high_s16 (int32x4_t a, int16x8_t b, int16x8_t c)
          A32: VQDMLAL.S16 Qd, Dn+1, Dm+1
          A64: SQDMLAL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_high_s32 (int64x2_t a, int32x4_t b, int32x4_t c)
          A32: VQDMLAL.S32 Qd, Dn+1, Dm+1
          A64: SQDMLAL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_high_s16 (int32x4_t a, int16x8_t b, int16x8_t c)
          A32: VQDMLSL.S16 Qd, Dn+1, Dm+1
          A64: SQDMLSL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_high_s32 (int64x2_t a, int32x4_t b, int32x4_t c)
          A32: VQDMLSL.S32 Qd, Dn+1, Dm+1
          A64: SQDMLSL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperByScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_high_n_s16 (int32x4_t a, int16x8_t b, int16_t c)
          A32: VQDMLAL.S16 Qd, Dn+1, Dm[0]
          A64: SQDMLAL2 Vd.4S, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperByScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_high_n_s32 (int64x2_t a, int32x4_t b, int32_t c)
          A32: VQDMLAL.S32 Qd, Dn+1, Dm[0]
          A64: SQDMLAL2 Vd.2D, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperByScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_high_n_s16 (int32x4_t a, int16x8_t b, int16_t c)
          A32: VQDMLSL.S16 Qd, Dn+1, Dm[0]
          A64: SQDMLSL2 Vd.4S, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperByScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_high_n_s32 (int64x2_t a, int32x4_t b, int32_t c)
          A32: VQDMLSL.S32 Qd, Dn+1, Dm[0]
          A64: SQDMLSL2 Vd.2D, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_high_lane_s16 (int32x4_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VQDMLAL.S16 Qd, Dn+1, Dm[lane]
          A64: SQDMLAL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlal_high_laneq_s16 (int32x4_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VQDMLAL.S16 Qd, Dn+1, Dm[lane]
          A64: SQDMLAL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_high_lane_s32 (int64x2_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VQDMLAL.S32 Qd, Dn+1, Dm[lane]
          A64: SQDMLAL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndAddSaturate(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlal_high_laneq_s32 (int64x2_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VQDMLAL.S32 Qd, Dn+1, Dm[lane]
          A64: SQDMLAL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_high_lane_s16 (int32x4_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VQDMLSL.S16 Qd, Dn+1, Dm[lane]
          A64: SQDMLSL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqdmlsl_high_laneq_s16 (int32x4_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VQDMLSL.S16 Qd, Dn+1, Dm[lane]
          A64: SQDMLSL2 Vd.4S, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_high_lane_s32 (int64x2_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VQDMLSL.S32 Qd, Dn+1, Dm[lane]
          A64: SQDMLSL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyDoublingWideningUpperBySelectedScalarAndSubtractSaturate(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqdmlsl_high_laneq_s32 (int64x2_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VQDMLSL.S32 Qd, Dn+1, Dm[lane]
          A64: SQDMLSL2 Vd.2D, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmulh_n_s16 (int16x4_t a, int16_t b)
          A32: VQRDMULH.S16 Dd, Dn, Dm[0]
          A64: SQRDMULH Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmulh_n_s32 (int32x2_t a, int32_t b)
          A32: VQRDMULH.S32 Dd, Dn, Dm[0]
          A64: SQRDMULH Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmulhq_n_s16 (int16x8_t a, int16_t b)
          A32: VQRDMULH.S16 Qd, Qn, Dm[0]
          A64: SQRDMULH Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingByScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmulhq_n_s32 (int32x4_t a, int32_t b)
          A32: VQRDMULH.S32 Qd, Qn, Dm[0]
          A64: SQRDMULH Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmulh_lane_s16 (int16x4_t a, int16x4_t v, const int lane)
          A32: VQRDMULH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMULH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmulh_laneq_s16 (int16x4_t a, int16x8_t v, const int lane)
          A32: VQRDMULH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMULH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmulh_lane_s32 (int32x2_t a, int32x2_t v, const int lane)
          A32: VQRDMULH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMULH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmulh_laneq_s32 (int32x2_t a, int32x4_t v, const int lane)
          A32: VQRDMULH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMULH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmulhq_lane_s16 (int16x8_t a, int16x4_t v, const int lane)
          A32: VQRDMULH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMULH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmulhq_laneq_s16 (int16x8_t a, int16x8_t v, const int lane)
          A32: VQRDMULH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMULH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmulhq_lane_s32 (int32x4_t a, int32x2_t v, const int lane)
          A32: VQRDMULH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMULH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmulhq_laneq_s32 (int32x4_t a, int32x4_t v, const int lane)
          A32: VQRDMULH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMULH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmulh_s16 (int16x4_t a, int16x4_t b)
          A32: VQRDMULH.S16 Dd, Dn, Dm
          A64: SQRDMULH Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmulh_s32 (int32x2_t a, int32x2_t b)
          A32: VQRDMULH.S32 Dd, Dn, Dm
          A64: SQRDMULH Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmulhq_s16 (int16x8_t a, int16x8_t b)
          A32: VQRDMULH.S16 Qd, Qn, Qm
          A64: SQRDMULH Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingSaturateHigh(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmulhq_s32 (int32x4_t a, int32x4_t b)
          A32: VQRDMULH.S32 Qd, Qn, Qm
          A64: SQRDMULH Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vmul_f64 (float64x1_t a, float64x1_t b)
          A32: VMUL.F64 Dd, Dn, Dm
          A64: FMUL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def MultiplyScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vmuls_f32 (float32_t a, float32_t b)
          A32: VMUL.F32 Sd, Sn, Sm
          A64: FMUL Sd, Sn, Sm
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def MultiplyScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vmuls_lane_f32 (float32_t a, float32x2_t v, const int lane)
          A32: VMUL.F32 Sd, Sn, Dm[lane]
          A64: FMUL Sd, Sn, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyScalarBySelectedScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector128[float], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vmuls_laneq_f32 (float32_t a, float32x4_t v, const int lane)
          A32: VMUL.F32 Sd, Sn, Dm[lane]
          A64: FMUL Sd, Sn, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmls_u8 (uint8x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VMLS.I8 Dd, Dn, Dm
          A64: MLS Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmls_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VMLS.I16 Dd, Dn, Dm
          A64: MLS Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmls_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VMLS.I32 Dd, Dn, Dm
          A64: MLS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmls_s8 (int8x8_t a, int8x8_t b, int8x8_t c)
          A32: VMLS.I8 Dd, Dn, Dm
          A64: MLS Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmls_u16 (uint16x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VMLS.I16 Dd, Dn, Dm
          A64: MLS Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmls_u32 (uint32x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VMLS.I32 Dd, Dn, Dm
          A64: MLS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vmlsq_u8 (uint8x16_t a, uint8x16_t b, uint8x16_t c)
          A32: VMLS.I8 Qd, Qn, Qm
          A64: MLS Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlsq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VMLS.I16 Qd, Qn, Qm
          A64: MLS Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VMLS.I32 Qd, Qn, Qm
          A64: MLS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vmlsq_s8 (int8x16_t a, int8x16_t b, int8x16_t c)
          A32: VMLS.I8 Qd, Qn, Qm
          A64: MLS Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlsq_u16 (uint16x8_t a, uint16x8_t b, uint16x8_t c)
          A32: VMLS.I16 Qd, Qn, Qm
          A64: MLS Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsq_u32 (uint32x4_t a, uint32x4_t b, uint32x4_t c)
          A32: VMLS.I32 Qd, Qn, Qm
          A64: MLS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmls_n_s16 (int16x4_t a, int16x4_t b, int16_t c)
          A32: VMLS.I16 Dd, Dn, Dm[0]
          A64: MLS Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmls_n_s32 (int32x2_t a, int32x2_t b, int32_t c)
          A32: VMLS.I32 Dd, Dn, Dm[0]
          A64: MLS Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmls_n_u16 (uint16x4_t a, uint16x4_t b, uint16_t c)
          A32: VMLS.I16 Dd, Dn, Dm[0]
          A64: MLS Vd.4H, Vn.4H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmls_n_u32 (uint32x2_t a, uint32x2_t b, uint32_t c)
          A32: VMLS.I32 Dd, Dn, Dm[0]
          A64: MLS Vd.2S, Vn.2S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlsq_n_s16 (int16x8_t a, int16x8_t b, int16_t c)
          A32: VMLS.I16 Qd, Qn, Dm[0]
          A64: MLS Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsq_n_s32 (int32x4_t a, int32x4_t b, int32_t c)
          A32: VMLS.I32 Qd, Qn, Dm[0]
          A64: MLS Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlsq_n_u16 (uint16x8_t a, uint16x8_t b, uint16_t c)
          A32: VMLS.I16 Qd, Qn, Dm[0]
          A64: MLS Vd.8H, Vn.8H, Vm.H[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractByScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsq_n_u32 (uint32x4_t a, uint32x4_t b, uint32_t c)
          A32: VMLS.I32 Qd, Qn, Dm[0]
          A64: MLS Vd.4S, Vn.4S, Vm.S[0]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmls_lane_s16 (int16x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VMLS.I16 Dd, Dn, Dm[lane]
          A64: MLS Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmls_laneq_s16 (int16x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VMLS.I16 Dd, Dn, Dm[lane]
          A64: MLS Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmls_lane_s32 (int32x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VMLS.I32 Dd, Dn, Dm[lane]
          A64: MLS Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmls_laneq_s32 (int32x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VMLS.I32 Dd, Dn, Dm[lane]
          A64: MLS Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmls_lane_u16 (uint16x4_t a, uint16x4_t b, uint16x4_t v, const int lane)
          A32: VMLS.I16 Dd, Dn, Dm[lane]
          A64: MLS Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmls_laneq_u16 (uint16x4_t a, uint16x4_t b, uint16x8_t v, const int lane)
          A32: VMLS.I16 Dd, Dn, Dm[lane]
          A64: MLS Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmls_lane_u32 (uint32x2_t a, uint32x2_t b, uint32x2_t v, const int lane)
          A32: VMLS.I32 Dd, Dn, Dm[lane]
          A64: MLS Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmls_laneq_u32 (uint32x2_t a, uint32x2_t b, uint32x4_t v, const int lane)
          A32: VMLS.I32 Dd, Dn, Dm[lane]
          A64: MLS Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlsq_lane_s16 (int16x8_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VMLS.I16 Qd, Qn, Dm[lane]
          A64: MLS Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlsq_laneq_s16 (int16x8_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VMLS.I16 Qd, Qn, Dm[lane]
          A64: MLS Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsq_lane_s32 (int32x4_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VMLS.I32 Qd, Qn, Dm[lane]
          A64: MLS Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsq_laneq_s32 (int32x4_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VMLS.I32 Qd, Qn, Dm[lane]
          A64: MLS Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlsq_lane_u16 (uint16x8_t a, uint16x8_t b, uint16x4_t v, const int lane)
          A32: VMLS.I16 Qd, Qn, Dm[lane]
          A64: MLS Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlsq_laneq_u16 (uint16x8_t a, uint16x8_t b, uint16x8_t v, const int lane)
          A32: VMLS.I16 Qd, Qn, Dm[lane]
          A64: MLS Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsq_lane_u32 (uint32x4_t a, uint32x4_t b, uint32x2_t v, const int lane)
          A32: VMLS.I32 Qd, Qn, Dm[lane]
          A64: MLS Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplySubtractBySelectedScalar(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsq_laneq_u32 (uint32x4_t a, uint32x4_t b, uint32x4_t v, const int lane)
          A32: VMLS.I32 Qd, Qn, Dm[lane]
          A64: MLS Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmull_u8 (uint8x8_t a, uint8x8_t b)
          A32: VMULL.U8 Qd, Dn, Dm
          A64: UMULL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmull_s16 (int16x4_t a, int16x4_t b)
          A32: VMULL.S16 Qd, Dn, Dm
          A64: SMULL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmull_s32 (int32x2_t a, int32x2_t b)
          A32: VMULL.S32 Qd, Dn, Dm
          A64: SMULL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmull_s8 (int8x8_t a, int8x8_t b)
          A32: VMULL.S8 Qd, Dn, Dm
          A64: SMULL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmull_u16 (uint16x4_t a, uint16x4_t b)
          A32: VMULL.U16 Qd, Dn, Dm
          A64: UMULL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmull_u32 (uint32x2_t a, uint32x2_t b)
          A32: VMULL.U32 Qd, Dn, Dm
          A64: UMULL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlal_u8 (uint16x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VMLAL.U8 Qd, Dn, Dm
          A64: UMLAL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlal_s16 (int32x4_t a, int16x4_t b, int16x4_t c)
          A32: VMLAL.S16 Qd, Dn, Dm
          A64: SMLAL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlal_s32 (int64x2_t a, int32x2_t b, int32x2_t c)
          A32: VMLAL.S32 Qd, Dn, Dm
          A64: SMLAL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlal_s8 (int16x8_t a, int8x8_t b, int8x8_t c)
          A32: VMLAL.S8 Qd, Dn, Dm
          A64: SMLAL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlal_u16 (uint32x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VMLAL.U16 Qd, Dn, Dm
          A64: UMLAL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlal_u32 (uint64x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VMLAL.U32 Qd, Dn, Dm
          A64: UMLAL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlsl_u8 (uint16x8_t a, uint8x8_t b, uint8x8_t c)
          A32: VMLSL.U8 Qd, Dn, Dm
          A64: UMLSL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsl_s16 (int32x4_t a, int16x4_t b, int16x4_t c)
          A32: VMLSL.S16 Qd, Dn, Dm
          A64: SMLSL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlsl_s32 (int64x2_t a, int32x2_t b, int32x2_t c)
          A32: VMLSL.S32 Qd, Dn, Dm
          A64: SMLSL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlsl_s8 (int16x8_t a, int8x8_t b, int8x8_t c)
          A32: VMLSL.S8 Qd, Dn, Dm
          A64: SMLSL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsl_u16 (uint32x4_t a, uint16x4_t b, uint16x4_t c)
          A32: VMLSL.U16 Qd, Dn, Dm
          A64: UMLSL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningLowerAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlsl_u32 (uint64x2_t a, uint32x2_t b, uint32x2_t c)
          A32: VMLSL.U32 Qd, Dn, Dm
          A64: UMLSL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmull_high_u8 (uint8x16_t a, uint8x16_t b)
          A32: VMULL.U8 Qd, Dn+1, Dm+1
          A64: UMULL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmull_high_s16 (int16x8_t a, int16x8_t b)
          A32: VMULL.S16 Qd, Dn+1, Dm+1
          A64: SMULL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmull_high_s32 (int32x4_t a, int32x4_t b)
          A32: VMULL.S32 Qd, Dn+1, Dm+1
          A64: SMULL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmull_high_s8 (int8x16_t a, int8x16_t b)
          A32: VMULL.S8 Qd, Dn+1, Dm+1
          A64: SMULL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmull_high_u16 (uint16x8_t a, uint16x8_t b)
          A32: VMULL.U16 Qd, Dn+1, Dm+1
          A64: UMULL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmull_high_u32 (uint32x4_t a, uint32x4_t b)
          A32: VMULL.U32 Qd, Dn+1, Dm+1
          A64: UMULL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlal_high_u8 (uint16x8_t a, uint8x16_t b, uint8x16_t c)
          A32: VMLAL.U8 Qd, Dn+1, Dm+1
          A64: UMLAL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlal_high_s16 (int32x4_t a, int16x8_t b, int16x8_t c)
          A32: VMLAL.S16 Qd, Dn+1, Dm+1
          A64: SMLAL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlal_high_s32 (int64x2_t a, int32x4_t b, int32x4_t c)
          A32: VMLAL.S32 Qd, Dn+1, Dm+1
          A64: SMLAL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlal_high_s8 (int16x8_t a, int8x16_t b, int8x16_t c)
          A32: VMLAL.S8 Qd, Dn+1, Dm+1
          A64: SMLAL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlal_high_u16 (uint32x4_t a, uint16x8_t b, uint16x8_t c)
          A32: VMLAL.U16 Qd, Dn+1, Dm+1
          A64: UMLAL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndAdd(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlal_high_u32 (uint64x2_t a, uint32x4_t b, uint32x4_t c)
          A32: VMLAL.U32 Qd, Dn+1, Dm+1
          A64: UMLAL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmlsl_high_u8 (uint16x8_t a, uint8x16_t b, uint8x16_t c)
          A32: VMLSL.U8 Qd, Dn+1, Dm+1
          A64: UMLSL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmlsl_high_s16 (int32x4_t a, int16x8_t b, int16x8_t c)
          A32: VMLSL.S16 Qd, Dn+1, Dm+1
          A64: SMLSL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmlsl_high_s32 (int64x2_t a, int32x4_t b, int32x4_t c)
          A32: VMLSL.S32 Qd, Dn+1, Dm+1
          A64: SMLSL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmlsl_high_s8 (int16x8_t a, int8x16_t b, int8x16_t c)
          A32: VMLSL.S8 Qd, Dn+1, Dm+1
          A64: SMLSL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmlsl_high_u16 (uint32x4_t a, uint16x8_t b, uint16x8_t c)
          A32: VMLSL.U16 Qd, Dn+1, Dm+1
          A64: UMLSL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyWideningUpperAndSubtract(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmlsl_high_u32 (uint64x2_t a, uint32x4_t b, uint32x4_t c)
          A32: VMLSL.U32 Qd, Dn+1, Dm+1
          A64: UMLSL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vneg_s16 (int16x4_t a)
          A32: VNEG.S16 Dd, Dm
          A64: NEG Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vneg_s32 (int32x2_t a)
          A32: VNEG.S32 Dd, Dm
          A64: NEG Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vneg_s8 (int8x8_t a)
          A32: VNEG.S8 Dd, Dm
          A64: NEG Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vneg_f32 (float32x2_t a)
          A32: VNEG.F32 Dd, Dm
          A64: FNEG Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vnegq_s16 (int16x8_t a)
          A32: VNEG.S16 Qd, Qm
          A64: NEG Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vnegq_s32 (int32x4_t a)
          A32: VNEG.S32 Qd, Qm
          A64: NEG Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vnegq_s8 (int8x16_t a)
          A32: VNEG.S8 Qd, Qm
          A64: NEG Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Negate(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vnegq_f32 (float32x4_t a)
          A32: VNEG.F32 Qd, Qm
          A64: FNEG Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def NegateSaturate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqneg_s16 (int16x4_t a)
          A32: VQNEG.S16 Dd, Dm
          A64: SQNEG Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def NegateSaturate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqneg_s32 (int32x2_t a)
          A32: VQNEG.S32 Dd, Dm
          A64: SQNEG Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def NegateSaturate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqneg_s8 (int8x8_t a)
          A32: VQNEG.S8 Dd, Dm
          A64: SQNEG Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def NegateSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqnegq_s16 (int16x8_t a)
          A32: VQNEG.S16 Qd, Qm
          A64: SQNEG Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def NegateSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqnegq_s32 (int32x4_t a)
          A32: VQNEG.S32 Qd, Qm
          A64: SQNEG Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def NegateSaturate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqnegq_s8 (int8x16_t a)
          A32: VQNEG.S8 Qd, Qm
          A64: SQNEG Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def NegateScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vneg_f64 (float64x1_t a)
          A32: VNEG.F64 Dd, Dm
          A64: FNEG Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def NegateScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vnegs_f32 (float32_t a)
          A32: VNEG.F32 Sd, Sm
          A64: FNEG Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vmvn_u8 (uint8x8_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vmvn_f64 (float64x1_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vmvn_s16 (int16x4_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vmvn_s32 (int32x2_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vmvn_s64 (int64x1_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vmvn_s8 (int8x8_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vmvn_f32 (float32x2_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vmvn_u16 (uint16x4_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vmvn_u32 (uint32x2_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vmvn_u64 (uint64x1_t a)
          A32: VMVN Dd, Dm
          A64: MVN Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vmvnq_u8 (uint8x16_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vmvnq_f64 (float64x2_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmvnq_s16 (int16x8_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmvnq_s32 (int32x4_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmvnq_s64 (int64x2_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vmvnq_s8 (int8x16_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vmvnq_f32 (float32x4_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmvnq_u16 (uint16x8_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmvnq_u32 (uint32x4_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Not(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmvnq_u64 (uint64x2_t a)
          A32: VMVN Qd, Qm
          A64: MVN Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vorr_u8 (uint8x8_t a, uint8x8_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vorr_f64 (float64x1_t a, float64x1_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vorr_s16 (int16x4_t a, int16x4_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vorr_s32 (int32x2_t a, int32x2_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vorr_s64 (int64x1_t a, int64x1_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vorr_s8 (int8x8_t a, int8x8_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vorr_f32 (float32x2_t a, float32x2_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vorr_u16 (uint16x4_t a, uint16x4_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vorr_u32 (uint32x2_t a, uint32x2_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vorr_u64 (uint64x1_t a, uint64x1_t b)
          A32: VORR Dd, Dn, Dm
          A64: ORR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vorrq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vorrq_f64 (float64x2_t a, float64x2_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vorrq_s16 (int16x8_t a, int16x8_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vorrq_s32 (int32x4_t a, int32x4_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vorrq_s64 (int64x2_t a, int64x2_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vorrq_s8 (int8x16_t a, int8x16_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vorrq_f32 (float32x4_t a, float32x4_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vorrq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vorrq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Or(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vorrq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VORR Qd, Qn, Qm
          A64: ORR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vorn_u8 (uint8x8_t a, uint8x8_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vorn_f64 (float64x1_t a, float64x1_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vorn_s16 (int16x4_t a, int16x4_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vorn_s32 (int32x2_t a, int32x2_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vorn_s64 (int64x1_t a, int64x1_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vorn_s8 (int8x8_t a, int8x8_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vorn_f32 (float32x2_t a, float32x2_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vorn_u16 (uint16x4_t a, uint16x4_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vorn_u32 (uint32x2_t a, uint32x2_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vorn_u64 (uint64x1_t a, uint64x1_t b)
          A32: VORN Dd, Dn, Dm
          A64: ORN Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vornq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t vornq_f64 (float64x2_t a, float64x2_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vornq_s16 (int16x8_t a, int16x8_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vornq_s32 (int32x4_t a, int32x4_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vornq_s64 (int64x2_t a, int64x2_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vornq_s8 (int8x16_t a, int8x16_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vornq_f32 (float32x4_t a, float32x4_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vornq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vornq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def OrNot(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vornq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VORN Qd, Qn, Qm
          A64: ORN Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        poly8x8_t vmul_p8 (poly8x8_t a, poly8x8_t b)
          A32: VMUL.P8 Dd, Dn, Dm
          A64: PMUL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiply(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        poly8x8_t vmul_p8 (poly8x8_t a, poly8x8_t b)
          A32: VMUL.P8 Dd, Dn, Dm
          A64: PMUL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly8x16_t vmulq_p8 (poly8x16_t a, poly8x16_t b)
          A32: VMUL.P8 Qd, Qn, Qm
          A64: PMUL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiply(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly8x16_t vmulq_p8 (poly8x16_t a, poly8x16_t b)
          A32: VMUL.P8 Qd, Qn, Qm
          A64: PMUL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly16x8_t vmull_p8 (poly8x8_t a, poly8x8_t b)
          A32: VMULL.P8 Qd, Dn, Dm
          A64: PMULL Vd.16B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly16x8_t vmull_p8 (poly8x8_t a, poly8x8_t b)
          A32: VMULL.P8 Qd, Dn, Dm
          A64: PMULL Vd.16B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly16x8_t vmull_high_p8 (poly8x16_t a, poly8x16_t b)
          A32: VMULL.P8 Qd, Dn+1, Dm+1
          A64: PMULL2 Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly16x8_t vmull_high_p8 (poly8x16_t a, poly8x16_t b)
          A32: VMULL.P8 Qd, Dn+1, Dm+1
          A64: PMULL2 Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def PopCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vcnt_u8 (uint8x8_t a)
          A32: VCNT.I8 Dd, Dm
          A64: CNT Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def PopCount(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vcnt_s8 (int8x8_t a)
          A32: VCNT.I8 Dd, Dm
          A64: CNT Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def PopCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vcntq_u8 (uint8x16_t a)
          A32: VCNT.I8 Qd, Qm
          A64: CNT Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def PopCount(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vcntq_s8 (int8x16_t a)
          A32: VCNT.I8 Qd, Qm
          A64: CNT Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ReciprocalEstimate(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrecpe_f32 (float32x2_t a)
          A32: VRECPE.F32 Dd, Dm
          A64: FRECPE Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalEstimate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrecpe_u32 (uint32x2_t a)
          A32: VRECPE.U32 Dd, Dm
          A64: URECPE Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalEstimate(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrecpeq_f32 (float32x4_t a)
          A32: VRECPE.F32 Qd, Qm
          A64: FRECPE Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalEstimate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrecpeq_u32 (uint32x4_t a)
          A32: VRECPE.U32 Qd, Qm
          A64: URECPE Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalSquareRootEstimate(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrsqrte_f32 (float32x2_t a)
          A32: VRSQRTE.F32 Dd, Dm
          A64: FRSQRTE Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalSquareRootEstimate(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrsqrte_u32 (uint32x2_t a)
          A32: VRSQRTE.U32 Dd, Dm
          A64: URSQRTE Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalSquareRootEstimate(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrsqrteq_f32 (float32x4_t a)
          A32: VRSQRTE.F32 Qd, Qm
          A64: FRSQRTE Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalSquareRootEstimate(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrsqrteq_u32 (uint32x4_t a)
          A32: VRSQRTE.U32 Qd, Qm
          A64: URSQRTE Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalSquareRootStep(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrsqrts_f32 (float32x2_t a, float32x2_t b)
          A32: VRSQRTS.F32 Dd, Dn, Dm
          A64: FRSQRTS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalSquareRootStep(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrsqrtsq_f32 (float32x4_t a, float32x4_t b)
          A32: VRSQRTS.F32 Qd, Qn, Qm
          A64: FRSQRTS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalStep(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrecps_f32 (float32x2_t a, float32x2_t b)
          A32: VRECPS.F32 Dd, Dn, Dm
          A64: FRECPS Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ReciprocalStep(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrecpsq_f32 (float32x4_t a, float32x4_t b)
          A32: VRECPS.F32 Qd, Qn, Qm
          A64: FRECPS Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrev32_s16 (int16x4_t vec)
          A32: VREV32.16 Dd, Dm
          A64: REV32 Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrev64_s16 (int16x4_t vec)
          A32: VREV64.16 Dd, Dm
          A64: REV64 Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrev32_u16 (uint16x4_t vec)
          A32: VREV32.16 Dd, Dm
          A64: REV32 Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrev64_u16 (uint16x4_t vec)
          A32: VREV64.16 Dd, Dm
          A64: REV64 Vd.4H, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrev32q_s16 (int16x8_t vec)
          A32: VREV32.16 Qd, Qm
          A64: REV32 Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrev64q_s16 (int16x8_t vec)
          A32: VREV64.16 Qd, Qm
          A64: REV64 Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrev32q_u16 (uint16x8_t vec)
          A32: VREV32.16 Qd, Qm
          A64: REV32 Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement16(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrev64q_u16 (uint16x8_t vec)
          A32: VREV64.16 Qd, Qm
          A64: REV64 Vd.8H, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ReverseElement32(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrev64_s32 (int32x2_t vec)
          A32: VREV64.32 Dd, Dm
          A64: REV64 Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ReverseElement32(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrev64_u32 (uint32x2_t vec)
          A32: VREV64.32 Dd, Dm
          A64: REV64 Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ReverseElement32(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrev64q_s32 (int32x4_t vec)
          A32: VREV64.32 Qd, Qm
          A64: REV64 Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ReverseElement32(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrev64q_u32 (uint32x4_t vec)
          A32: VREV64.32 Qd, Qm
          A64: REV64 Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrev16_s8 (int8x8_t vec)
          A32: VREV16.8 Dd, Dm
          A64: REV16 Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrev32_s8 (int8x8_t vec)
          A32: VREV32.8 Dd, Dm
          A64: REV32 Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrev64_s8 (int8x8_t vec)
          A32: VREV64.8 Dd, Dm
          A64: REV64 Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrev16_u8 (uint8x8_t vec)
          A32: VREV16.8 Dd, Dm
          A64: REV16 Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrev32_u8 (uint8x8_t vec)
          A32: VREV32.8 Dd, Dm
          A64: REV32 Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrev64_u8 (uint8x8_t vec)
          A32: VREV64.8 Dd, Dm
          A64: REV64 Vd.8B, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrev16q_s8 (int8x16_t vec)
          A32: VREV16.8 Qd, Qm
          A64: REV16 Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrev32q_s8 (int8x16_t vec)
          A32: VREV32.8 Qd, Qm
          A64: REV32 Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrev64q_s8 (int8x16_t vec)
          A32: VREV64.8 Qd, Qm
          A64: REV64 Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrev16q_u8 (uint8x16_t vec)
          A32: VREV16.8 Qd, Qm
          A64: REV16 Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrev32q_u8 (uint8x16_t vec)
          A32: VREV32.8 Qd, Qm
          A64: REV32 Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ReverseElement8(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrev64q_u8 (uint8x16_t vec)
          A32: VREV64.8 Qd, Qm
          A64: REV64 Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrnda_f32 (float32x2_t a)
          A32: VRINTA.F32 Dd, Dm
          A64: FRINTA Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def RoundAwayFromZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndaq_f32 (float32x4_t a)
          A32: VRINTA.F32 Qd, Qm
          A64: FRINTA Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def RoundAwayFromZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrnda_f64 (float64x1_t a)
          A32: VRINTA.F64 Dd, Dm
          A64: FRINTA Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def RoundAwayFromZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrndas_f32 (float32_t a)
          A32: VRINTA.F32 Sd, Sm
          A64: FRINTA Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def RoundToNearest(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrndn_f32 (float32x2_t a)
          A32: VRINTN.F32 Dd, Dm
          A64: FRINTN Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def RoundToNearest(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndnq_f32 (float32x4_t a)
          A32: VRINTN.F32 Qd, Qm
          A64: FRINTN Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def RoundToNearestScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrndn_f64 (float64x1_t a)
          A32: VRINTN.F64 Dd, Dm
          A64: FRINTN Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def RoundToNearestScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrndns_f32 (float32_t a)
          A32: VRINTN.F32 Sd, Sm
          A64: FRINTN Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrndm_f32 (float32x2_t a)
          A32: VRINTM.F32 Dd, Dm
          A64: FRINTM Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def RoundToNegativeInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndmq_f32 (float32x4_t a)
          A32: VRINTM.F32 Qd, Qm
          A64: FRINTM Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def RoundToNegativeInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrndm_f64 (float64x1_t a)
          A32: VRINTM.F64 Dd, Dm
          A64: FRINTM Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def RoundToNegativeInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrndms_f32 (float32_t a)
          A32: VRINTM.F32 Sd, Sm
          A64: FRINTM Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrndp_f32 (float32x2_t a)
          A32: VRINTP.F32 Dd, Dm
          A64: FRINTP Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def RoundToPositiveInfinity(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndpq_f32 (float32x4_t a)
          A32: VRINTP.F32 Qd, Qm
          A64: FRINTP Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def RoundToPositiveInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrndp_f64 (float64x1_t a)
          A32: VRINTP.F64 Dd, Dm
          A64: FRINTP Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def RoundToPositiveInfinityScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrndps_f32 (float32_t a)
          A32: VRINTP.F32 Sd, Sm
          A64: FRINTP Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def RoundToZero(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vrnd_f32 (float32x2_t a)
          A32: VRINTZ.F32 Dd, Dm
          A64: FRINTZ Vd.2S, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def RoundToZero(value: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vrndq_f32 (float32x4_t a)
          A32: VRINTZ.F32 Qd, Qm
          A64: FRINTZ Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def RoundToZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vrnd_f64 (float64x1_t a)
          A32: VRINTZ.F64 Dd, Dm
          A64: FRINTZ Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def RoundToZeroScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vrnds_f32 (float32_t a)
          A32: VRINTZ.F32 Sd, Sm
          A64: FRINTZ Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vshl_s16 (int16x4_t a, int16x4_t b)
          A32: VSHL.S16 Dd, Dn, Dm
          A64: SSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vshl_s32 (int32x2_t a, int32x2_t b)
          A32: VSHL.S32 Dd, Dn, Dm
          A64: SSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vshl_s8 (int8x8_t a, int8x8_t b)
          A32: VSHL.S8 Dd, Dn, Dm
          A64: SSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vshlq_s16 (int16x8_t a, int16x8_t b)
          A32: VSHL.S16 Qd, Qn, Qm
          A64: SSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vshlq_s32 (int32x4_t a, int32x4_t b)
          A32: VSHL.S32 Qd, Qn, Qm
          A64: SSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vshlq_s64 (int64x2_t a, int64x2_t b)
          A32: VSHL.S64 Qd, Qn, Qm
          A64: SSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vshlq_s8 (int8x16_t a, int8x16_t b)
          A32: VSHL.S8 Qd, Qn, Qm
          A64: SSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrshl_s16 (int16x4_t a, int16x4_t b)
          A32: VRSHL.S16 Dd, Dn, Dm
          A64: SRSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrshl_s32 (int32x2_t a, int32x2_t b)
          A32: VRSHL.S32 Dd, Dn, Dm
          A64: SRSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrshl_s8 (int8x8_t a, int8x8_t b)
          A32: VRSHL.S8 Dd, Dn, Dm
          A64: SRSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrshlq_s16 (int16x8_t a, int16x8_t b)
          A32: VRSHL.S16 Qd, Qn, Qm
          A64: SRSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrshlq_s32 (int32x4_t a, int32x4_t b)
          A32: VRSHL.S32 Qd, Qn, Qm
          A64: SRSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vrshlq_s64 (int64x2_t a, int64x2_t b)
          A32: VRSHL.S64 Qd, Qn, Qm
          A64: SRSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrshlq_s8 (int8x16_t a, int8x16_t b)
          A32: VRSHL.S8 Qd, Qn, Qm
          A64: SRSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrshl_s16 (int16x4_t a, int16x4_t b)
          A32: VQRSHL.S16 Dd, Dn, Dm
          A64: SQRSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrshl_s32 (int32x2_t a, int32x2_t b)
          A32: VQRSHL.S32 Dd, Dn, Dm
          A64: SQRSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqrshl_s8 (int8x8_t a, int8x8_t b)
          A32: VQRSHL.S8 Dd, Dn, Dm
          A64: SQRSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrshlq_s16 (int16x8_t a, int16x8_t b)
          A32: VQRSHL.S16 Qd, Qn, Qm
          A64: SQRSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrshlq_s32 (int32x4_t a, int32x4_t b)
          A32: VQRSHL.S32 Qd, Qn, Qm
          A64: SQRSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqrshlq_s64 (int64x2_t a, int64x2_t b)
          A32: VQRSHL.S64 Qd, Qn, Qm
          A64: SQRSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqrshlq_s8 (int8x16_t a, int8x16_t b)
          A32: VQRSHL.S8 Qd, Qn, Qm
          A64: SQRSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    def ShiftArithmeticRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vqrshl_s64 (int64x1_t a, int64x1_t b)
          A32: VQRSHL.S64 Dd, Dn, Dm
          A64: SQRSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    def ShiftArithmeticRoundedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vrshl_s64 (int64x1_t a, int64x1_t b)
          A32: VRSHL.S64 Dd, Dn, Dm
          A64: SRSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqshl_s16 (int16x4_t a, int16x4_t b)
          A32: VQSHL.S16 Dd, Dn, Dm
          A64: SQSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqshl_s32 (int32x2_t a, int32x2_t b)
          A32: VQSHL.S32 Dd, Dn, Dm
          A64: SQSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqshl_s8 (int8x8_t a, int8x8_t b)
          A32: VQSHL.S8 Dd, Dn, Dm
          A64: SQSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqshlq_s16 (int16x8_t a, int16x8_t b)
          A32: VQSHL.S16 Qd, Qn, Qm
          A64: SQSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqshlq_s32 (int32x4_t a, int32x4_t b)
          A32: VQSHL.S32 Qd, Qn, Qm
          A64: SQSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqshlq_s64 (int64x2_t a, int64x2_t b)
          A32: VQSHL.S64 Qd, Qn, Qm
          A64: SQSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftArithmeticSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqshlq_s8 (int8x16_t a, int8x16_t b)
          A32: VQSHL.S8 Qd, Qn, Qm
          A64: SQSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    def ShiftArithmeticSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vqshl_s64 (int64x1_t a, int64x1_t b)
          A32: VQSHL.S64 Dd, Dn, Dm
          A64: SQSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    def ShiftArithmeticScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vshl_s64 (int64x1_t a, int64x1_t b)
          A32: VSHL.S64 Dd, Dn, Dm
          A64: SSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vsli_n_u8(uint8x8_t a, uint8x8_t b, __builtin_constant_p(n))
          A32: VSLI.8 Dd, Dm, #n
          A64: SLI Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vsli_n_s16(int16x4_t a, int16x4_t b, __builtin_constant_p(n))
          A32: VSLI.16 Dd, Dm, #n
          A64: SLI Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vsli_n_s32(int32x2_t a, int32x2_t b, __builtin_constant_p(n))
          A32: VSLI.32 Dd, Dm, #n
          A64: SLI Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vsli_n_s8(int8x8_t a, int8x8_t b, __builtin_constant_p(n))
          A32: VSLI.8 Dd, Dm, #n
          A64: SLI Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vsli_n_u16(uint16x4_t a, uint16x4_t b, __builtin_constant_p(n))
          A32: VSLI.16 Dd, Dm, #n
          A64: SLI Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vsli_n_u32(uint32x2_t a, uint32x2_t b, __builtin_constant_p(n))
          A32: VSLI.32 Dd, Dm, #n
          A64: SLI Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsliq_n_u8(uint8x16_t a, uint8x16_t b, __builtin_constant_p(n))
          A32: VSLI.8 Qd, Qm, #n
          A64: SLI Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsliq_n_s16(int16x8_t a, int16x8_t b, __builtin_constant_p(n))
          A32: VSLI.16 Qd, Qm, #n
          A64: SLI Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsliq_n_s32(int32x4_t a, int32x4_t b, __builtin_constant_p(n))
          A32: VSLI.32 Qd, Qm, #n
          A64: SLI Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsliq_n_s64(int64x2_t a, int64x2_t b, __builtin_constant_p(n))
          A32: VSLI.64 Qd, Qm, #n
          A64: SLI Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vsliq_n_s8(int8x16_t a, int8x16_t b, __builtin_constant_p(n))
          A32: VSLI.8 Qd, Qm, #n
          A64: SLI Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsliq_n_u16(uint16x8_t a, uint16x8_t b, __builtin_constant_p(n))
          A32: VSLI.16 Qd, Qm, #n
          A64: SLI Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsliq_n_u32(uint32x4_t a, uint32x4_t b, __builtin_constant_p(n))
          A32: VSLI.32 Qd, Qm, #n
          A64: SLI Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsliq_n_u64(uint64x2_t a, uint64x2_t b, __builtin_constant_p(n))
          A32: VSLI.64 Qd, Qm, #n
          A64: SLI Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsertScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64_t vslid_n_s64(int64_t a, int64_t b, __builtin_constant_p(n))
          A32: VSLI.64 Dd, Dm, #n
          A64: SLI Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftAndInsertScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64_t vslid_n_u64(uint64_t a, uint64_t b, __builtin_constant_p(n))
          A32: VSLI.64 Dd, Dm, #n
          A64: SLI Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vshl_n_u8 (uint8x8_t a, const int n)
          A32: VSHL.I8 Dd, Dm, #n
          A64: SHL Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vshl_n_s16 (int16x4_t a, const int n)
          A32: VSHL.I16 Dd, Dm, #n
          A64: SHL Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vshl_n_s32 (int32x2_t a, const int n)
          A32: VSHL.I32 Dd, Dm, #n
          A64: SHL Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vshl_n_s8 (int8x8_t a, const int n)
          A32: VSHL.I8 Dd, Dm, #n
          A64: SHL Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vshl_n_u16 (uint16x4_t a, const int n)
          A32: VSHL.I16 Dd, Dm, #n
          A64: SHL Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vshl_n_u32 (uint32x2_t a, const int n)
          A32: VSHL.I32 Dd, Dm, #n
          A64: SHL Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vshlq_n_u8 (uint8x16_t a, const int n)
          A32: VSHL.I8 Qd, Qm, #n
          A64: SHL Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vshlq_n_s16 (int16x8_t a, const int n)
          A32: VSHL.I16 Qd, Qm, #n
          A64: SHL Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vshlq_n_s64 (int64x2_t a, const int n)
          A32: VSHL.I64 Qd, Qm, #n
          A64: SHL Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vshlq_n_s8 (int8x16_t a, const int n)
          A32: VSHL.I8 Qd, Qm, #n
          A64: SHL Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshlq_n_u16 (uint16x8_t a, const int n)
          A32: VSHL.I16 Qd, Qm, #n
          A64: SHL Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshlq_n_u32 (uint32x4_t a, const int n)
          A32: VSHL.I32 Qd, Qm, #n
          A64: SHL Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshlq_n_u64 (uint64x2_t a, const int n)
          A32: VSHL.I64 Qd, Qm, #n
          A64: SHL Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshl_n_u8 (uint8x8_t a, const int n)
          A32: VQSHL.U8 Dd, Dm, #n
          A64: UQSHL Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqshl_n_s16 (int16x4_t a, const int n)
          A32: VQSHL.S16 Dd, Dm, #n
          A64: SQSHL Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqshl_n_s32 (int32x2_t a, const int n)
          A32: VQSHL.S32 Dd, Dm, #n
          A64: SQSHL Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqshl_n_s8 (int8x8_t a, const int n)
          A32: VQSHL.S8 Dd, Dm, #n
          A64: SQSHL Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshl_n_u16 (uint16x4_t a, const int n)
          A32: VQSHL.U16 Dd, Dm, #n
          A64: UQSHL Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshl_n_u32 (uint32x2_t a, const int n)
          A32: VQSHL.U32 Dd, Dm, #n
          A64: UQSHL Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshlq_n_u8 (uint8x16_t a, const int n)
          A32: VQSHL.U8 Qd, Qm, #n
          A64: UQSHL Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqshlq_n_s16 (int16x8_t a, const int n)
          A32: VQSHL.S16 Qd, Qm, #n
          A64: SQSHL Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqshlq_n_s32 (int32x4_t a, const int n)
          A32: VQSHL.S32 Qd, Qm, #n
          A64: SQSHL Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqshlq_n_s64 (int64x2_t a, const int n)
          A32: VQSHL.S64 Qd, Qm, #n
          A64: SQSHL Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqshlq_n_s8 (int8x16_t a, const int n)
          A32: VQSHL.S8 Qd, Qm, #n
          A64: SQSHL Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshlq_n_u16 (uint16x8_t a, const int n)
          A32: VQSHL.U16 Qd, Qm, #n
          A64: UQSHL Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshlq_n_u32 (uint32x4_t a, const int n)
          A32: VQSHL.U32 Qd, Qm, #n
          A64: UQSHL Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqshlq_n_u64 (uint64x2_t a, const int n)
          A32: VQSHL.U64 Qd, Qm, #n
          A64: UQSHL Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vqshl_n_s64 (int64x1_t a, const int n)
          A32: VQSHL.S64 Dd, Dm, #n
          A64: SQSHL Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqshl_n_u64 (uint64x1_t a, const int n)
          A32: VQSHL.U64 Dd, Dm, #n
          A64: UQSHL Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshlu_n_s16 (int16x4_t a, const int n)
          A32: VQSHLU.S16 Dd, Dm, #n
          A64: SQSHLU Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshlu_n_s32 (int32x2_t a, const int n)
          A32: VQSHLU.S32 Dd, Dm, #n
          A64: SQSHLU Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshlu_n_s8 (int8x8_t a, const int n)
          A32: VQSHLU.S8 Dd, Dm, #n
          A64: SQSHLU Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshluq_n_s16 (int16x8_t a, const int n)
          A32: VQSHLU.S16 Qd, Qm, #n
          A64: SQSHLU Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshluq_n_s32 (int32x4_t a, const int n)
          A32: VQSHLU.S32 Qd, Qm, #n
          A64: SQSHLU Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqshluq_n_s64 (int64x2_t a, const int n)
          A32: VQSHLU.S64 Qd, Qm, #n
          A64: SQSHLU Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalSaturateUnsigned(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshluq_n_s8 (int8x16_t a, const int n)
          A32: VQSHLU.S8 Qd, Qm, #n
          A64: SQSHLU Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    def ShiftLeftLogicalSaturateUnsignedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqshlu_n_s64 (int64x1_t a, const int n)
          A32: VQSHLU.S64 Dd, Dm, #n
          A64: SQSHLU Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vshl_n_s64 (int64x1_t a, const int n)
          A32: VSHL.I64 Dd, Dm, #n
          A64: SHL Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vshl_n_u64 (uint64x1_t a, const int n)
          A32: VSHL.I64 Dd, Dm, #n
          A64: SHL Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningLower(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshll_n_u8 (uint8x8_t a, const int n)
          A32: VSHLL.U8 Qd, Dm, #n
          A64: USHLL Vd.8H, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningLower(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vshll_n_s16 (int16x4_t a, const int n)
          A32: VSHLL.S16 Qd, Dm, #n
          A64: SSHLL Vd.4S, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningLower(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vshll_n_s32 (int32x2_t a, const int n)
          A32: VSHLL.S32 Qd, Dm, #n
          A64: SSHLL Vd.2D, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningLower(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vshll_n_s8 (int8x8_t a, const int n)
          A32: VSHLL.S8 Qd, Dm, #n
          A64: SSHLL Vd.8H, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningLower(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshll_n_u16 (uint16x4_t a, const int n)
          A32: VSHLL.U16 Qd, Dm, #n
          A64: USHLL Vd.4S, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningLower(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshll_n_u32 (uint32x2_t a, const int n)
          A32: VSHLL.U32 Qd, Dm, #n
          A64: USHLL Vd.2D, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningUpper(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshll_high_n_u8 (uint8x16_t a, const int n)
          A32: VSHLL.U8 Qd, Dm+1, #n
          A64: USHLL2 Vd.8H, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningUpper(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vshll_high_n_s16 (int16x8_t a, const int n)
          A32: VSHLL.S16 Qd, Dm+1, #n
          A64: SSHLL2 Vd.4S, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningUpper(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vshll_high_n_s32 (int32x4_t a, const int n)
          A32: VSHLL.S32 Qd, Dm+1, #n
          A64: SSHLL2 Vd.2D, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningUpper(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vshll_high_n_s8 (int8x16_t a, const int n)
          A32: VSHLL.S8 Qd, Dm+1, #n
          A64: SSHLL2 Vd.8H, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningUpper(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshll_high_n_u16 (uint16x8_t a, const int n)
          A32: VSHLL.U16 Qd, Dm+1, #n
          A64: USHLL2 Vd.4S, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLeftLogicalWideningUpper(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshll_high_n_u32 (uint32x4_t a, const int n)
          A32: VSHLL.U32 Qd, Dm+1, #n
          A64: USHLL2 Vd.2D, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VSHL.U8 Dd, Dn, Dm
          A64: USHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VSHL.U16 Dd, Dn, Dm
          A64: USHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VSHL.U32 Dd, Dn, Dm
          A64: USHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VSHL.U8 Dd, Dn, Dm
          A64: USHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VSHL.U16 Dd, Dn, Dm
          A64: USHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VSHL.U32 Dd, Dn, Dm
          A64: USHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VSHL.U8 Qd, Qn, Qm
          A64: USHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VSHL.U16 Qd, Qn, Qm
          A64: USHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VSHL.U32 Qd, Qn, Qm
          A64: USHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VSHL.U64 Qd, Qn, Qm
          A64: USHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VSHL.U8 Qd, Qn, Qm
          A64: USHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VSHL.U16 Qd, Qn, Qm
          A64: USHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VSHL.U32 Qd, Qn, Qm
          A64: USHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogical(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VSHL.U64 Qd, Qn, Qm
          A64: USHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VRSHL.U8 Dd, Dn, Dm
          A64: URSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VRSHL.U16 Dd, Dn, Dm
          A64: URSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VRSHL.U32 Dd, Dn, Dm
          A64: URSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VRSHL.U8 Dd, Dn, Dm
          A64: URSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VRSHL.U16 Dd, Dn, Dm
          A64: URSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VRSHL.U32 Dd, Dn, Dm
          A64: URSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VRSHL.U8 Qd, Qn, Qm
          A64: URSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VRSHL.U16 Qd, Qn, Qm
          A64: URSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VRSHL.U32 Qd, Qn, Qm
          A64: URSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vrshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VRSHL.U64 Qd, Qn, Qm
          A64: URSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VRSHL.U8 Qd, Qn, Qm
          A64: URSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VRSHL.U16 Qd, Qn, Qm
          A64: URSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VRSHL.U32 Qd, Qn, Qm
          A64: URSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vrshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VRSHL.U64 Qd, Qn, Qm
          A64: URSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqrshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VQRSHL.U8 Dd, Dn, Dm
          A64: UQRSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqrshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VQRSHL.U16 Dd, Dn, Dm
          A64: UQRSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqrshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VQRSHL.U32 Dd, Dn, Dm
          A64: UQRSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqrshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VQRSHL.U8 Dd, Dn, Dm
          A64: UQRSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqrshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VQRSHL.U16 Dd, Dn, Dm
          A64: UQRSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqrshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VQRSHL.U32 Dd, Dn, Dm
          A64: UQRSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqrshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VQRSHL.U8 Qd, Qn, Qm
          A64: UQRSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqrshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VQRSHL.U16 Qd, Qn, Qm
          A64: UQRSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqrshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VQRSHL.U32 Qd, Qn, Qm
          A64: UQRSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqrshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VQRSHL.U64 Qd, Qn, Qm
          A64: UQRSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqrshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VQRSHL.U8 Qd, Qn, Qm
          A64: UQRSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqrshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VQRSHL.U16 Qd, Qn, Qm
          A64: UQRSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqrshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VQRSHL.U32 Qd, Qn, Qm
          A64: UQRSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqrshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VQRSHL.U64 Qd, Qn, Qm
          A64: UQRSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqrshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VQRSHL.U64 Dd, Dn, Dm
          A64: UQRSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqrshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VQRSHL.U64 Dd, Dn, Dm
          A64: UQRSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vrshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VRSHL.U64 Dd, Dn, Dm
          A64: URSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalRoundedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vrshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VRSHL.U64 Dd, Dn, Dm
          A64: URSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VQSHL.U8 Dd, Dn, Dm
          A64: UQSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VQSHL.U16 Dd, Dn, Dm
          A64: UQSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VQSHL.U32 Dd, Dn, Dm
          A64: UQSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshl_u8 (uint8x8_t a, int8x8_t b)
          A32: VQSHL.U8 Dd, Dn, Dm
          A64: UQSHL Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshl_u16 (uint16x4_t a, int16x4_t b)
          A32: VQSHL.U16 Dd, Dn, Dm
          A64: UQSHL Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshl_u32 (uint32x2_t a, int32x2_t b)
          A32: VQSHL.U32 Dd, Dn, Dm
          A64: UQSHL Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VQSHL.U8 Qd, Qn, Qm
          A64: UQSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VQSHL.U16 Qd, Qn, Qm
          A64: UQSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VQSHL.U32 Qd, Qn, Qm
          A64: UQSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VQSHL.U64 Qd, Qn, Qm
          A64: UQSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshlq_u8 (uint8x16_t a, int8x16_t b)
          A32: VQSHL.U8 Qd, Qn, Qm
          A64: UQSHL Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshlq_u16 (uint16x8_t a, int16x8_t b)
          A32: VQSHL.U16 Qd, Qn, Qm
          A64: UQSHL Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshlq_u32 (uint32x4_t a, int32x4_t b)
          A32: VQSHL.U32 Qd, Qn, Qm
          A64: UQSHL Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturate(value: System.Runtime.Intrinsics.Vector128[int], count: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqshlq_u64 (uint64x2_t a, int64x2_t b)
          A32: VQSHL.U64 Qd, Qn, Qm
          A64: UQSHL Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VQSHL.U64 Dd, Dn, Dm
          A64: UQSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalSaturateScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VQSHL.U64 Dd, Dn, Dm
          A64: UQSHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VSHL.U64 Dd, Dn, Dm
          A64: USHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftLogicalScalar(value: System.Runtime.Intrinsics.Vector64[int], count: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vshl_u64 (uint64x1_t a, int64x1_t b)
          A32: VSHL.U64 Dd, Dn, Dm
          A64: USHL Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vsri_n_u8(uint8x8_t a, uint8x8_t b, __builtin_constant_p(n))
          A32: VSRI.8 Dd, Dm, #n
          A64: SRI Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vsri_n_s16(int16x4_t a, int16x4_t b, __builtin_constant_p(n))
          A32: VSRI.16 Dd, Dm, #n
          A64: SRI Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vsri_n_s32(int32x2_t a, int32x2_t b, __builtin_constant_p(n))
          A32: VSRI.32 Dd, Dm, #n
          A64: SRI Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vsri_n_s8(int8x8_t a, int8x8_t b, __builtin_constant_p(n))
          A32: VSRI.8 Dd, Dm, #n
          A64: SRI Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vsri_n_u16(uint16x4_t a, uint16x4_t b, __builtin_constant_p(n))
          A32: VSRI.16 Dd, Dm, #n
          A64: SRI Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vsri_n_u32(uint32x2_t a, uint32x2_t b, __builtin_constant_p(n))
          A32: VSRI.32 Dd, Dm, #n
          A64: SRI Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsriq_n_u8(uint8x16_t a, uint8x16_t b, __builtin_constant_p(n))
          A32: VSRI.8 Qd, Qm, #n
          A64: SRI Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsriq_n_s16(int16x8_t a, int16x8_t b, __builtin_constant_p(n))
          A32: VSRI.16 Qd, Qm, #n
          A64: SRI Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsriq_n_s32(int32x4_t a, int32x4_t b, __builtin_constant_p(n))
          A32: VSRI.32 Qd, Qm, #n
          A64: SRI Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsriq_n_s64(int64x2_t a, int64x2_t b, __builtin_constant_p(n))
          A32: VSRI.64 Qd, Qm, #n
          A64: SRI Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vsriq_n_s8(int8x16_t a, int8x16_t b, __builtin_constant_p(n))
          A32: VSRI.8 Qd, Qm, #n
          A64: SRI Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsriq_n_u16(uint16x8_t a, uint16x8_t b, __builtin_constant_p(n))
          A32: VSRI.16 Qd, Qm, #n
          A64: SRI Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsriq_n_u32(uint32x4_t a, uint32x4_t b, __builtin_constant_p(n))
          A32: VSRI.32 Qd, Qm, #n
          A64: SRI Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsert(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], shift: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsriq_n_u64(uint64x2_t a, uint64x2_t b, __builtin_constant_p(n))
          A32: VSRI.64 Qd, Qm, #n
          A64: SRI Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsertScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64_t vsrid_n_s64(int64_t a, int64_t b, __builtin_constant_p(n))
          A32: VSRI.64 Dd, Dm, #n
          A64: SRI Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightAndInsertScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], shift: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64_t vsrid_n_u64(uint64_t a, uint64_t b, __builtin_constant_p(n))
          A32: VSRI.64 Dd, Dm, #n
          A64: SRI Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vshr_n_s16 (int16x4_t a, const int n)
          A32: VSHR.S16 Dd, Dm, #n
          A64: SSHR Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vshr_n_s32 (int32x2_t a, const int n)
          A32: VSHR.S32 Dd, Dm, #n
          A64: SSHR Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vshr_n_s8 (int8x8_t a, const int n)
          A32: VSHR.S8 Dd, Dm, #n
          A64: SSHR Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vshrq_n_s16 (int16x8_t a, const int n)
          A32: VSHR.S16 Qd, Qm, #n
          A64: SSHR Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vshrq_n_s32 (int32x4_t a, const int n)
          A32: VSHR.S32 Qd, Qm, #n
          A64: SSHR Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vshrq_n_s64 (int64x2_t a, const int n)
          A32: VSHR.S64 Qd, Qm, #n
          A64: SSHR Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmetic(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vshrq_n_s8 (int8x16_t a, const int n)
          A32: VSHR.S8 Qd, Qm, #n
          A64: SSHR Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vsra_n_s16 (int16x4_t a, int16x4_t b, const int n)
          A32: VSRA.S16 Dd, Dm, #n
          A64: SSRA Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vsra_n_s32 (int32x2_t a, int32x2_t b, const int n)
          A32: VSRA.S32 Dd, Dm, #n
          A64: SSRA Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vsra_n_s8 (int8x8_t a, int8x8_t b, const int n)
          A32: VSRA.S8 Dd, Dm, #n
          A64: SSRA Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsraq_n_s16 (int16x8_t a, int16x8_t b, const int n)
          A32: VSRA.S16 Qd, Qm, #n
          A64: SSRA Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsraq_n_s32 (int32x4_t a, int32x4_t b, const int n)
          A32: VSRA.S32 Qd, Qm, #n
          A64: SSRA Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsraq_n_s64 (int64x2_t a, int64x2_t b, const int n)
          A32: VSRA.S64 Qd, Qm, #n
          A64: SSRA Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vsraq_n_s8 (int8x16_t a, int8x16_t b, const int n)
          A32: VSRA.S8 Qd, Qm, #n
          A64: SSRA Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    def ShiftRightArithmeticAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vsra_n_s64 (int64x1_t a, int64x1_t b, const int n)
          A32: VSRA.S64 Dd, Dm, #n
          A64: SSRA Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqshrn_n_s32 (int32x4_t a, const int n)
          A32: VQSHRN.S32 Dd, Qm, #n
          A64: SQSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqshrn_n_s64 (int64x2_t a, const int n)
          A32: VQSHRN.S64 Dd, Qm, #n
          A64: SQSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqshrn_n_s16 (int16x8_t a, const int n)
          A32: VQSHRN.S16 Dd, Qm, #n
          A64: SQSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshrun_n_s16 (int16x8_t a, const int n)
          A32: VQSHRUN.S16 Dd, Qm, #n
          A64: SQSHRUN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshrun_n_s32 (int32x4_t a, const int n)
          A32: VQSHRUN.S32 Dd, Qm, #n
          A64: SQSHRUN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshrun_n_s64 (int64x2_t a, const int n)
          A32: VQSHRUN.S64 Dd, Qm, #n
          A64: SQSHRUN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshrun_high_n_s16 (uint8x8_t r, int16x8_t a, const int n)
          A32: VQSHRUN.S16 Dd+1, Dn, #n
          A64: SQSHRUN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshrun_high_n_s32 (uint16x4_t r, int32x4_t a, const int n)
          A32: VQSHRUN.S32 Dd+1, Dn, #n
          A64: SQSHRUN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshrun_high_n_s64 (uint32x2_t r, int64x2_t a, const int n)
          A32: VQSHRUN.S64 Dd+1, Dn, #n
          A64: SQSHRUN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqshrn_high_n_s32 (int16x4_t r, int32x4_t a, const int n)
          A32: VQSHRN.S32 Dd+1, Qm, #n
          A64: SQSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqshrn_high_n_s64 (int32x2_t r, int64x2_t a, const int n)
          A32: VQSHRN.S64 Dd+1, Qm, #n
          A64: SQSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqshrn_high_n_s16 (int8x8_t r, int16x8_t a, const int n)
          A32: VQSHRN.S16 Dd+1, Qm, #n
          A64: SQSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrshr_n_s16 (int16x4_t a, const int n)
          A32: VRSHR.S16 Dd, Dm, #n
          A64: SRSHR Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrshr_n_s32 (int32x2_t a, const int n)
          A32: VRSHR.S32 Dd, Dm, #n
          A64: SRSHR Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrshr_n_s8 (int8x8_t a, const int n)
          A32: VRSHR.S8 Dd, Dm, #n
          A64: SRSHR Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrshrq_n_s16 (int16x8_t a, const int n)
          A32: VRSHR.S16 Qd, Qm, #n
          A64: SRSHR Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrshrq_n_s32 (int32x4_t a, const int n)
          A32: VRSHR.S32 Qd, Qm, #n
          A64: SRSHR Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vrshrq_n_s64 (int64x2_t a, const int n)
          A32: VRSHR.S64 Qd, Qm, #n
          A64: SRSHR Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrshrq_n_s8 (int8x16_t a, const int n)
          A32: VRSHR.S8 Qd, Qm, #n
          A64: SRSHR Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrsra_n_s16 (int16x4_t a, int16x4_t b, const int n)
          A32: VRSRA.S16 Dd, Dm, #n
          A64: SRSRA Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrsra_n_s32 (int32x2_t a, int32x2_t b, const int n)
          A32: VRSRA.S32 Dd, Dm, #n
          A64: SRSRA Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrsra_n_s8 (int8x8_t a, int8x8_t b, const int n)
          A32: VRSRA.S8 Dd, Dm, #n
          A64: SRSRA Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrsraq_n_s16 (int16x8_t a, int16x8_t b, const int n)
          A32: VRSRA.S16 Qd, Qm, #n
          A64: SRSRA Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrsraq_n_s32 (int32x4_t a, int32x4_t b, const int n)
          A32: VRSRA.S32 Qd, Qm, #n
          A64: SRSRA Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vrsraq_n_s64 (int64x2_t a, int64x2_t b, const int n)
          A32: VRSRA.S64 Qd, Qm, #n
          A64: SRSRA Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrsraq_n_s8 (int8x16_t a, int8x16_t b, const int n)
          A32: VRSRA.S8 Qd, Qm, #n
          A64: SRSRA Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    def ShiftRightArithmeticRoundedAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vrsra_n_s64 (int64x1_t a, int64x1_t b, const int n)
          A32: VRSRA.S64 Dd, Dm, #n
          A64: SRSRA Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrshrn_n_s32 (int32x4_t a, const int n)
          A32: VQRSHRN.S32 Dd, Qm, #n
          A64: SQRSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrshrn_n_s64 (int64x2_t a, const int n)
          A32: VQRSHRN.S64 Dd, Qm, #n
          A64: SQRSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqrshrn_n_s16 (int16x8_t a, const int n)
          A32: VQRSHRN.S16 Dd, Qm, #n
          A64: SQRSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqrshrun_n_s16 (int16x8_t a, const int n)
          A32: VQRSHRUN.S16 Dd, Qm, #n
          A64: SQRSHRUN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqrshrun_n_s32 (int32x4_t a, const int n)
          A32: VQRSHRUN.S32 Dd, Qm, #n
          A64: SQRSHRUN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqrshrun_n_s64 (int64x2_t a, const int n)
          A32: VQRSHRUN.S64 Dd, Qm, #n
          A64: SQRSHRUN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqrshrun_high_n_s16 (uint8x8_t r, int16x8_t a, const int n)
          A32: VQRSHRUN.S16 Dd+1, Dn, #n
          A64: SQRSHRUN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqrshrun_high_n_s32 (uint16x4_t r, int32x4_t a, const int n)
          A32: VQRSHRUN.S32 Dd+1, Dn, #n
          A64: SQRSHRUN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUnsignedUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqrshrun_high_n_s64 (uint32x2_t r, int64x2_t a, const int n)
          A32: VQRSHRUN.S64 Dd+1, Dn, #n
          A64: SQRSHRUN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrshrn_high_n_s32 (int16x4_t r, int32x4_t a, const int n)
          A32: VQRSHRN.S32 Dd+1, Dn, #n
          A64: SQRSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrshrn_high_n_s64 (int32x2_t r, int64x2_t a, const int n)
          A32: VQRSHRN.S64 Dd+1, Dn, #n
          A64: SQRSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightArithmeticRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqrshrn_high_n_s16 (int8x8_t r, int16x8_t a, const int n)
          A32: VQRSHRN.S16 Dd+1, Dn, #n
          A64: SQRSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    def ShiftRightArithmeticRoundedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vrshr_n_s64 (int64x1_t a, const int n)
          A32: VRSHR.S64 Dd, Dm, #n
          A64: SRSHR Dd, Dn, #n
        """
        ...

    @staticmethod
    def ShiftRightArithmeticScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vshr_n_s64 (int64x1_t a, const int n)
          A32: VSHR.S64 Dd, Dm, #n
          A64: SSHR Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vshr_n_u8 (uint8x8_t a, const int n)
          A32: VSHR.U8 Dd, Dm, #n
          A64: USHR Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vshr_n_u16 (uint16x4_t a, const int n)
          A32: VSHR.U16 Dd, Dm, #n
          A64: USHR Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vshr_n_u32 (uint32x2_t a, const int n)
          A32: VSHR.U32 Dd, Dm, #n
          A64: USHR Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vshr_n_u8 (uint8x8_t a, const int n)
          A32: VSHR.U8 Dd, Dm, #n
          A64: USHR Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vshr_n_u16 (uint16x4_t a, const int n)
          A32: VSHR.U16 Dd, Dm, #n
          A64: USHR Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vshr_n_u32 (uint32x2_t a, const int n)
          A32: VSHR.U32 Dd, Dm, #n
          A64: USHR Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vshrq_n_u8 (uint8x16_t a, const int n)
          A32: VSHR.U8 Qd, Qm, #n
          A64: USHR Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshrq_n_u16 (uint16x8_t a, const int n)
          A32: VSHR.U16 Qd, Qm, #n
          A64: USHR Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshrq_n_u32 (uint32x4_t a, const int n)
          A32: VSHR.U32 Qd, Qm, #n
          A64: USHR Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshrq_n_u64 (uint64x2_t a, const int n)
          A32: VSHR.U64 Qd, Qm, #n
          A64: USHR Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vshrq_n_u8 (uint8x16_t a, const int n)
          A32: VSHR.U8 Qd, Qm, #n
          A64: USHR Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshrq_n_u16 (uint16x8_t a, const int n)
          A32: VSHR.U16 Qd, Qm, #n
          A64: USHR Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshrq_n_u32 (uint32x4_t a, const int n)
          A32: VSHR.U32 Qd, Qm, #n
          A64: USHR Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogical(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vshrq_n_u64 (uint64x2_t a, const int n)
          A32: VSHR.U64 Qd, Qm, #n
          A64: USHR Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vsra_n_u8 (uint8x8_t a, uint8x8_t b, const int n)
          A32: VSRA.U8 Dd, Dm, #n
          A64: USRA Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vsra_n_u16 (uint16x4_t a, uint16x4_t b, const int n)
          A32: VSRA.U16 Dd, Dm, #n
          A64: USRA Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vsra_n_u32 (uint32x2_t a, uint32x2_t b, const int n)
          A32: VSRA.U32 Dd, Dm, #n
          A64: USRA Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vsra_n_u8 (uint8x8_t a, uint8x8_t b, const int n)
          A32: VSRA.U8 Dd, Dm, #n
          A64: USRA Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vsra_n_u16 (uint16x4_t a, uint16x4_t b, const int n)
          A32: VSRA.U16 Dd, Dm, #n
          A64: USRA Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vsra_n_u32 (uint32x2_t a, uint32x2_t b, const int n)
          A32: VSRA.U32 Dd, Dm, #n
          A64: USRA Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsraq_n_u8 (uint8x16_t a, uint8x16_t b, const int n)
          A32: VSRA.U8 Qd, Qm, #n
          A64: USRA Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsraq_n_u16 (uint16x8_t a, uint16x8_t b, const int n)
          A32: VSRA.U16 Qd, Qm, #n
          A64: USRA Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsraq_n_u32 (uint32x4_t a, uint32x4_t b, const int n)
          A32: VSRA.U32 Qd, Qm, #n
          A64: USRA Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsraq_n_u64 (uint64x2_t a, uint64x2_t b, const int n)
          A32: VSRA.U64 Qd, Qm, #n
          A64: USRA Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsraq_n_u8 (uint8x16_t a, uint8x16_t b, const int n)
          A32: VSRA.U8 Qd, Qm, #n
          A64: USRA Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsraq_n_u16 (uint16x8_t a, uint16x8_t b, const int n)
          A32: VSRA.U16 Qd, Qm, #n
          A64: USRA Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsraq_n_u32 (uint32x4_t a, uint32x4_t b, const int n)
          A32: VSRA.U32 Qd, Qm, #n
          A64: USRA Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsraq_n_u64 (uint64x2_t a, uint64x2_t b, const int n)
          A32: VSRA.U64 Qd, Qm, #n
          A64: USRA Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vsra_n_u64 (uint64x1_t a, uint64x1_t b, const int n)
          A32: VSRA.U64 Dd, Dm, #n
          A64: USRA Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vsra_n_u64 (uint64x1_t a, uint64x1_t b, const int n)
          A32: VSRA.U64 Dd, Dm, #n
          A64: USRA Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vshrn_n_u16 (uint16x8_t a, const int n)
          A32: VSHRN.I16 Dd, Qm, #n
          A64: SHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vshrn_n_s32 (int32x4_t a, const int n)
          A32: VSHRN.I32 Dd, Qm, #n
          A64: SHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vshrn_n_s64 (int64x2_t a, const int n)
          A32: VSHRN.I64 Dd, Qm, #n
          A64: SHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vshrn_n_s16 (int16x8_t a, const int n)
          A32: VSHRN.I16 Dd, Qm, #n
          A64: SHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vshrn_n_u32 (uint32x4_t a, const int n)
          A32: VSHRN.I32 Dd, Qm, #n
          A64: SHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vshrn_n_u64 (uint64x2_t a, const int n)
          A32: VSHRN.I64 Dd, Qm, #n
          A64: SHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshrn_n_u16 (uint16x8_t a, const int n)
          A32: VQSHRN.U16 Dd, Qm, #n
          A64: UQSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshrn_n_u32 (uint32x4_t a, const int n)
          A32: VQSHRN.U32 Dd, Qm, #n
          A64: UQSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshrn_n_u64 (uint64x2_t a, const int n)
          A32: VQSHRN.U64 Dd, Qm, #n
          A64: UQSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqshrn_n_u16 (uint16x8_t a, const int n)
          A32: VQSHRN.U16 Dd, Qm, #n
          A64: UQSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqshrn_n_u32 (uint32x4_t a, const int n)
          A32: VQSHRN.U32 Dd, Qm, #n
          A64: UQSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqshrn_n_u64 (uint64x2_t a, const int n)
          A32: VQSHRN.U64 Dd, Qm, #n
          A64: UQSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshrn_high_n_u16 (uint8x8_t r, uint16x8_t a, const int n)
          A32: VQSHRN.U16 Dd+1, Qm, #n
          A64: UQSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshrn_high_n_u32 (uint16x4_t r, uint32x4_t a, const int n)
          A32: VQSHRN.U32 Dd+1, Qm, #n
          A64: UQSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshrn_high_n_u64 (uint32x2_t r, uint64x2_t a, const int n)
          A32: VQSHRN.U64 Dd+1, Qm, #n
          A64: UQSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqshrn_high_n_u16 (uint8x8_t r, uint16x8_t a, const int n)
          A32: VQSHRN.U16 Dd+1, Qm, #n
          A64: UQSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqshrn_high_n_u32 (uint16x4_t r, uint32x4_t a, const int n)
          A32: VQSHRN.U32 Dd+1, Qm, #n
          A64: UQSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqshrn_high_n_u64 (uint32x2_t r, uint64x2_t a, const int n)
          A32: VQSHRN.U64 Dd+1, Qm, #n
          A64: UQSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vshrn_high_n_u16 (uint8x8_t r, uint16x8_t a, const int n)
          A32: VSHRN.I16 Dd+1, Qm, #n
          A64: SHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vshrn_high_n_s32 (int16x4_t r, int32x4_t a, const int n)
          A32: VSHRN.I32 Dd+1, Qm, #n
          A64: SHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vshrn_high_n_s64 (int32x2_t r, int64x2_t a, const int n)
          A32: VSHRN.I64 Dd+1, Qm, #n
          A64: SHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vshrn_high_n_s16 (int8x8_t r, int16x8_t a, const int n)
          A32: VSHRN.I16 Dd+1, Qm, #n
          A64: SHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vshrn_high_n_u32 (uint16x4_t r, uint32x4_t a, const int n)
          A32: VSHRN.I32 Dd+1, Qm, #n
          A64: SHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vshrn_high_n_u64 (uint32x2_t r, uint64x2_t a, const int n)
          A32: VSHRN.I64 Dd+1, Qm, #n
          A64: SHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrshr_n_u8 (uint8x8_t a, const int n)
          A32: VRSHR.U8 Dd, Dm, #n
          A64: URSHR Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrshr_n_u16 (uint16x4_t a, const int n)
          A32: VRSHR.U16 Dd, Dm, #n
          A64: URSHR Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrshr_n_u32 (uint32x2_t a, const int n)
          A32: VRSHR.U32 Dd, Dm, #n
          A64: URSHR Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrshr_n_u8 (uint8x8_t a, const int n)
          A32: VRSHR.U8 Dd, Dm, #n
          A64: URSHR Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrshr_n_u16 (uint16x4_t a, const int n)
          A32: VRSHR.U16 Dd, Dm, #n
          A64: URSHR Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrshr_n_u32 (uint32x2_t a, const int n)
          A32: VRSHR.U32 Dd, Dm, #n
          A64: URSHR Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrshrq_n_u8 (uint8x16_t a, const int n)
          A32: VRSHR.U8 Qd, Qm, #n
          A64: URSHR Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrshrq_n_u16 (uint16x8_t a, const int n)
          A32: VRSHR.U16 Qd, Qm, #n
          A64: URSHR Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrshrq_n_u32 (uint32x4_t a, const int n)
          A32: VRSHR.U32 Qd, Qm, #n
          A64: URSHR Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vrshrq_n_u64 (uint64x2_t a, const int n)
          A32: VRSHR.U64 Qd, Qm, #n
          A64: URSHR Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrshrq_n_u8 (uint8x16_t a, const int n)
          A32: VRSHR.U8 Qd, Qm, #n
          A64: URSHR Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrshrq_n_u16 (uint16x8_t a, const int n)
          A32: VRSHR.U16 Qd, Qm, #n
          A64: URSHR Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrshrq_n_u32 (uint32x4_t a, const int n)
          A32: VRSHR.U32 Qd, Qm, #n
          A64: URSHR Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRounded(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vrshrq_n_u64 (uint64x2_t a, const int n)
          A32: VRSHR.U64 Qd, Qm, #n
          A64: URSHR Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrsra_n_u8 (uint8x8_t a, uint8x8_t b, const int n)
          A32: VRSRA.U8 Dd, Dm, #n
          A64: URSRA Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrsra_n_u16 (uint16x4_t a, uint16x4_t b, const int n)
          A32: VRSRA.U16 Dd, Dm, #n
          A64: URSRA Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrsra_n_u32 (uint32x2_t a, uint32x2_t b, const int n)
          A32: VRSRA.U32 Dd, Dm, #n
          A64: URSRA Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrsra_n_u8 (uint8x8_t a, uint8x8_t b, const int n)
          A32: VRSRA.U8 Dd, Dm, #n
          A64: URSRA Vd.8B, Vn.8B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrsra_n_u16 (uint16x4_t a, uint16x4_t b, const int n)
          A32: VRSRA.U16 Dd, Dm, #n
          A64: URSRA Vd.4H, Vn.4H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrsra_n_u32 (uint32x2_t a, uint32x2_t b, const int n)
          A32: VRSRA.U32 Dd, Dm, #n
          A64: URSRA Vd.2S, Vn.2S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrsraq_n_u8 (uint8x16_t a, uint8x16_t b, const int n)
          A32: VRSRA.U8 Qd, Qm, #n
          A64: URSRA Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrsraq_n_u16 (uint16x8_t a, uint16x8_t b, const int n)
          A32: VRSRA.U16 Qd, Qm, #n
          A64: URSRA Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrsraq_n_u32 (uint32x4_t a, uint32x4_t b, const int n)
          A32: VRSRA.U32 Qd, Qm, #n
          A64: URSRA Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vrsraq_n_u64 (uint64x2_t a, uint64x2_t b, const int n)
          A32: VRSRA.U64 Qd, Qm, #n
          A64: URSRA Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrsraq_n_u8 (uint8x16_t a, uint8x16_t b, const int n)
          A32: VRSRA.U8 Qd, Qm, #n
          A64: URSRA Vd.16B, Vn.16B, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrsraq_n_u16 (uint16x8_t a, uint16x8_t b, const int n)
          A32: VRSRA.U16 Qd, Qm, #n
          A64: URSRA Vd.8H, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrsraq_n_u32 (uint32x4_t a, uint32x4_t b, const int n)
          A32: VRSRA.U32 Qd, Qm, #n
          A64: URSRA Vd.4S, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAdd(addend: System.Runtime.Intrinsics.Vector128[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vrsraq_n_u64 (uint64x2_t a, uint64x2_t b, const int n)
          A32: VRSRA.U64 Qd, Qm, #n
          A64: URSRA Vd.2D, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vrsra_n_u64 (uint64x1_t a, uint64x1_t b, const int n)
          A32: VRSRA.U64 Dd, Dm, #n
          A64: URSRA Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedAddScalar(addend: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vrsra_n_u64 (uint64x1_t a, uint64x1_t b, const int n)
          A32: VRSRA.U64 Dd, Dm, #n
          A64: URSRA Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrshrn_n_u16 (uint16x8_t a, const int n)
          A32: VRSHRN.I16 Dd, Qm, #n
          A64: RSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrshrn_n_s32 (int32x4_t a, const int n)
          A32: VRSHRN.I32 Dd, Qm, #n
          A64: RSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrshrn_n_s64 (int64x2_t a, const int n)
          A32: VRSHRN.I64 Dd, Qm, #n
          A64: RSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrshrn_n_s16 (int16x8_t a, const int n)
          A32: VRSHRN.I16 Dd, Qm, #n
          A64: RSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrshrn_n_u32 (uint32x4_t a, const int n)
          A32: VRSHRN.I32 Dd, Qm, #n
          A64: RSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrshrn_n_u64 (uint64x2_t a, const int n)
          A32: VRSHRN.I64 Dd, Qm, #n
          A64: RSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqrshrn_n_u16 (uint16x8_t a, const int n)
          A32: VQRSHRN.U16 Dd, Qm, #n
          A64: UQRSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqrshrn_n_u32 (uint32x4_t a, const int n)
          A32: VQRSHRN.U32 Dd, Qm, #n
          A64: UQRSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqrshrn_n_u64 (uint64x2_t a, const int n)
          A32: VQRSHRN.U64 Dd, Qm, #n
          A64: UQRSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqrshrn_n_u16 (uint16x8_t a, const int n)
          A32: VQRSHRN.U16 Dd, Qm, #n
          A64: UQRSHRN Vd.8B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqrshrn_n_u32 (uint32x4_t a, const int n)
          A32: VQRSHRN.U32 Dd, Qm, #n
          A64: UQRSHRN Vd.4H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateLower(value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqrshrn_n_u64 (uint64x2_t a, const int n)
          A32: VQRSHRN.U64 Dd, Qm, #n
          A64: UQRSHRN Vd.2S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqrshrn_high_n_u16 (uint8x8_t r, uint16x8_t a, const int n)
          A32: VQRSHRN.U16 Dd+1, Dn, #n
          A64: UQRSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqrshrn_high_n_u32 (uint16x4_t r, uint32x4_t a, const int n)
          A32: VQRSHRN.U32 Dd+1, Dn, #n
          A64: UQRSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqrshrn_high_n_u64 (uint32x2_t r, uint64x2_t a, const int n)
          A32: VQRSHRN.U64 Dd+1, Dn, #n
          A64: UQRSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqrshrn_high_n_u16 (uint8x8_t r, uint16x8_t a, const int n)
          A32: VQRSHRN.U16 Dd+1, Dn, #n
          A64: UQRSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqrshrn_high_n_u32 (uint16x4_t r, uint32x4_t a, const int n)
          A32: VQRSHRN.U32 Dd+1, Dn, #n
          A64: UQRSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingSaturateUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqrshrn_high_n_u64 (uint32x2_t r, uint64x2_t a, const int n)
          A32: VQRSHRN.U64 Dd+1, Dn, #n
          A64: UQRSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrshrn_high_n_u16 (uint8x8_t r, uint16x8_t a, const int n)
          A32: VRSHRN.I16 Dd+1, Qm, #n
          A64: RSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrshrn_high_n_s32 (int16x4_t r, int32x4_t a, const int n)
          A32: VRSHRN.I32 Dd+1, Qm, #n
          A64: RSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrshrn_high_n_s64 (int32x2_t r, int64x2_t a, const int n)
          A32: VRSHRN.I64 Dd+1, Qm, #n
          A64: RSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrshrn_high_n_s16 (int8x8_t r, int16x8_t a, const int n)
          A32: VRSHRN.I16 Dd+1, Qm, #n
          A64: RSHRN2 Vd.16B, Vn.8H, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrshrn_high_n_u32 (uint16x4_t r, uint32x4_t a, const int n)
          A32: VRSHRN.I32 Dd+1, Qm, #n
          A64: RSHRN2 Vd.8H, Vn.4S, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], value: System.Runtime.Intrinsics.Vector128[int], count: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrshrn_high_n_u64 (uint32x2_t r, uint64x2_t a, const int n)
          A32: VRSHRN.I64 Dd+1, Qm, #n
          A64: RSHRN2 Vd.4S, Vn.2D, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vrshr_n_u64 (uint64x1_t a, const int n)
          A32: VRSHR.U64 Dd, Dm, #n
          A64: URSHR Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalRoundedScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vrshr_n_u64 (uint64x1_t a, const int n)
          A32: VRSHR.U64 Dd, Dm, #n
          A64: URSHR Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vshr_n_u64 (uint64x1_t a, const int n)
          A32: VSHR.U64 Dd, Dm, #n
          A64: USHR Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def ShiftRightLogicalScalar(value: System.Runtime.Intrinsics.Vector64[int], count: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vshr_n_u64 (uint64x1_t a, const int n)
          A32: VSHR.U64 Dd, Dm, #n
          A64: USHR Dd, Dn, #n
        """
        ...

    @staticmethod
    @overload
    def SignExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmovl_s16 (int16x4_t a)
          A32: VMOVL.S16 Qd, Dm
          A64: SXTL Vd.4S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def SignExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmovl_s32 (int32x2_t a)
          A32: VMOVL.S32 Qd, Dm
          A64: SXTL Vd.2D, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def SignExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmovl_s8 (int8x8_t a)
          A32: VMOVL.S8 Qd, Dm
          A64: SXTL Vd.8H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def SignExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vmovl_high_s16 (int16x8_t a)
          A32: VMOVL.S16 Qd, Dm+1
          A64: SXTL2 Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def SignExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vmovl_high_s32 (int32x4_t a)
          A32: VMOVL.S32 Qd, Dm+1
          A64: SXTL2 Vd.2D, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def SignExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vmovl_high_s8 (int8x16_t a)
          A32: VMOVL.S8 Qd, Dm+1
          A64: SXTL2 Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def SqrtScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vsqrt_f64 (float64x1_t a)
          A32: VSQRT.F64 Dd, Dm
          A64: FSQRT Dd, Dn
        """
        ...

    @staticmethod
    @overload
    def SqrtScalar(value: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vsqrts_f32 (float32_t a)
          A32: VSQRT.F32 Sd, Sm
          A64: FSQRT Sd, Sn
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_u8 (uint8_t * ptr, uint8x8_t val)
          A32: VST1.8 { Dd }, [Rn]
          A64: ST1 { Vt.8B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[float]) -> None:
        """
        void vst1_f64 (float64_t * ptr, float64x1_t val)
          A32: VST1.64 { Dd }, [Rn]
          A64: ST1 { Vt.1D }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_s16 (int16_t * ptr, int16x4_t val)
          A32: VST1.16 { Dd }, [Rn]
          A64: ST1 {Vt.4H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_s32 (int32_t * ptr, int32x2_t val)
          A32: VST1.32 { Dd }, [Rn]
          A64: ST1 { Vt.2S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_s64 (int64_t * ptr, int64x1_t val)
          A32: VST1.64 { Dd }, [Rn]
          A64: ST1 { Vt.1D }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_s8 (int8_t * ptr, int8x8_t val)
          A32: VST1.8 { Dd }, [Rn]
          A64: ST1 { Vt.8B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[float]) -> None:
        """
        void vst1_f32 (float32_t * ptr, float32x2_t val)
          A32: VST1.32 { Dd }, [Rn]
          A64: ST1 { Vt.2S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_u16 (uint16_t * ptr, uint16x4_t val)
          A32: VST1.16 { Dd }, [Rn]
          A64: ST1 { Vt.4H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_u32 (uint32_t * ptr, uint32x2_t val)
          A32: VST1.32 { Dd }, [Rn]
          A64: ST1 { Vt.2S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector64[int]) -> None:
        """
        void vst1_u64 (uint64_t * ptr, uint64x1_t val)
          A32: VST1.64 { Dd }, [Rn]
          A64: ST1 { Vt.1D }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_u8 (uint8_t * ptr, uint8x16_t val)
          A32: VST1.8 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.16B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[float]) -> None:
        """
        void vst1q_f64 (float64_t * ptr, float64x2_t val)
          A32: VST1.64 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.2D }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_s16 (int16_t * ptr, int16x8_t val)
          A32: VST1.16 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.8H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_s32 (int32_t * ptr, int32x4_t val)
          A32: VST1.32 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.4S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_s64 (int64_t * ptr, int64x2_t val)
          A32: VST1.64 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.2D }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_s8 (int8_t * ptr, int8x16_t val)
          A32: VST1.8 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.16B }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[float]) -> None:
        """
        void vst1q_f32 (float32_t * ptr, float32x4_t val)
          A32: VST1.32 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.4S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_u16 (uint16_t * ptr, uint16x8_t val)
          A32: VST1.16 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.8H }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_u32 (uint32_t * ptr, uint32x4_t val)
          A32: VST1.32 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.4S }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def Store(address: typing.Any, source: System.Runtime.Intrinsics.Vector128[int]) -> None:
        """
        void vst1q_u64 (uint64_t * ptr, uint64x2_t val)
          A32: VST1.64 { Dd, Dd+1 }, [Rn]
          A64: ST1 { Vt.2D }, [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[int], index: int) -> None:
        """
        void vst1_lane_u8 (uint8_t * ptr, uint8x8_t val, const int lane)
          A32: VST1.8 { Dd[index] }, [Rn]
          A64: ST1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[int], index: int) -> None:
        """
        void vst1_lane_s16 (int16_t * ptr, int16x4_t val, const int lane)
          A32: VST1.16 { Dd[index] }, [Rn]
          A64: ST1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[int], index: int) -> None:
        """
        void vst1_lane_s32 (int32_t * ptr, int32x2_t val, const int lane)
          A32: VST1.32 { Dd[index] }, [Rn]
          A64: ST1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[int], index: int) -> None:
        """
        void vst1_lane_s8 (int8_t * ptr, int8x8_t val, const int lane)
          A32: VST1.8 { Dd[index] }, [Rn]
          A64: ST1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[float], index: int) -> None:
        """
        void vst1_lane_f32 (float32_t * ptr, float32x2_t val, const int lane)
          A32: VST1.32 { Dd[index] }, [Rn]
          A64: ST1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[int], index: int) -> None:
        """
        void vst1_lane_u16 (uint16_t * ptr, uint16x4_t val, const int lane)
          A32: VST1.16 { Dd[index] }, [Rn]
          A64: ST1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector64[int], index: int) -> None:
        """
        void vst1_lane_u32 (uint32_t * ptr, uint32x2_t val, const int lane)
          A32: VST1.32 { Dd[index] }, [Rn]
          A64: ST1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_u8 (uint8_t * ptr, uint8x16_t val, const int lane)
          A32: VST1.8 { Dd[index] }, [Rn]
          A64: ST1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[float], index: int) -> None:
        """
        void vst1q_lane_f64 (float64_t * ptr, float64x2_t val, const int lane)
          A32: VSTR.64 Dd, [Rn]
          A64: ST1 { Vt.D }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_s16 (int16_t * ptr, int16x8_t val, const int lane)
          A32: VST1.16 { Dd[index] }, [Rn]
          A64: ST1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_s32 (int32_t * ptr, int32x4_t val, const int lane)
          A32: VST1.32 { Dd[index] }, [Rn]
          A64: ST1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_s64 (int64_t * ptr, int64x2_t val, const int lane)
          A32: VSTR.64 Dd, [Rn]
          A64: ST1 { Vt.D }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_s8 (int8_t * ptr, int8x16_t val, const int lane)
          A32: VST1.8 { Dd[index] }, [Rn]
          A64: ST1 { Vt.B }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[float], index: int) -> None:
        """
        void vst1q_lane_f32 (float32_t * ptr, float32x4_t val, const int lane)
          A32: VST1.32 { Dd[index] }, [Rn]
          A64: ST1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_u16 (uint16_t * ptr, uint16x8_t val, const int lane)
          A32: VST1.16 { Dd[index] }, [Rn]
          A64: ST1 { Vt.H }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_u32 (uint32_t * ptr, uint32x4_t val, const int lane)
          A32: VST1.32 { Dd[index] }, [Rn]
          A64: ST1 { Vt.S }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def StoreSelectedScalar(address: typing.Any, value: System.Runtime.Intrinsics.Vector128[int], index: int) -> None:
        """
        void vst1q_lane_u64 (uint64_t * ptr, uint64x2_t val, const int lane)
          A32: VSTR.64 Dd, [Rn]
          A64: ST1 { Vt.D }[index], [Xn]
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vsub_u8 (uint8x8_t a, uint8x8_t b)
          A32: VSUB.I8 Dd, Dn, Dm
          A64: SUB Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vsub_s16 (int16x4_t a, int16x4_t b)
          A32: VSUB.I16 Dd, Dn, Dm
          A64: SUB Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vsub_s32 (int32x2_t a, int32x2_t b)
          A32: VSUB.I32 Dd, Dn, Dm
          A64: SUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vsub_s8 (int8x8_t a, int8x8_t b)
          A32: VSUB.I8 Dd, Dn, Dm
          A64: SUB Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t vsub_f32 (float32x2_t a, float32x2_t b)
          A32: VSUB.F32 Dd, Dn, Dm
          A64: FSUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vsub_u16 (uint16x4_t a, uint16x4_t b)
          A32: VSUB.I16 Dd, Dn, Dm
          A64: SUB Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vsub_u32 (uint32x2_t a, uint32x2_t b)
          A32: VSUB.I32 Dd, Dn, Dm
          A64: SUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsubq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VSUB.I8 Qd, Qn, Qm
          A64: SUB Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsubq_s16 (int16x8_t a, int16x8_t b)
          A32: VSUB.I16 Qd, Qn, Qm
          A64: SUB Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsubq_s32 (int32x4_t a, int32x4_t b)
          A32: VSUB.I32 Qd, Qn, Qm
          A64: SUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsubq_s64 (int64x2_t a, int64x2_t b)
          A32: VSUB.I64 Qd, Qn, Qm
          A64: SUB Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vsubq_s8 (int8x16_t a, int8x16_t b)
          A32: VSUB.I8 Qd, Qn, Qm
          A64: SUB Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t vsubq_f32 (float32x4_t a, float32x4_t b)
          A32: VSUB.F32 Qd, Qn, Qm
          A64: FSUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsubq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VSUB.I16 Qd, Qn, Qm
          A64: SUB Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsubq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VSUB.I32 Qd, Qn, Qm
          A64: SUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def Subtract(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsubq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VSUB.I64 Qd, Qn, Qm
          A64: SUB Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vsubhn_u16 (uint16x8_t a, uint16x8_t b)
          A32: VSUBHN.I16 Dd, Qn, Qm
          A64: SUBHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vsubhn_s32 (int32x4_t a, int32x4_t b)
          A32: VSUBHN.I32 Dd, Qn, Qm
          A64: SUBHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vsubhn_s64 (int64x2_t a, int64x2_t b)
          A32: VSUBHN.I64 Dd, Qn, Qm
          A64: SUBHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vsubhn_s16 (int16x8_t a, int16x8_t b)
          A32: VSUBHN.I16 Dd, Qn, Qm
          A64: SUBHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vsubhn_u32 (uint32x4_t a, uint32x4_t b)
          A32: VSUBHN.I32 Dd, Qn, Qm
          A64: SUBHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vsubhn_u64 (uint64x2_t a, uint64x2_t b)
          A32: VSUBHN.I64 Dd, Qn, Qm
          A64: SUBHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vsubhn_high_u16 (uint8x8_t r, uint16x8_t a, uint16x8_t b)
          A32: VSUBHN.I16 Dd+1, Qn, Qm
          A64: SUBHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsubhn_high_s32 (int16x4_t r, int32x4_t a, int32x4_t b)
          A32: VSUBHN.I32 Dd+1, Qn, Qm
          A64: SUBHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsubhn_high_s64 (int32x2_t r, int64x2_t a, int64x2_t b)
          A32: VSUBHN.I64 Dd+1, Qn, Qm
          A64: SUBHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vsubhn_high_s16 (int8x8_t r, int16x8_t a, int16x8_t b)
          A32: VSUBHN.I16 Dd+1, Qn, Qm
          A64: SUBHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsubhn_high_u32 (uint16x4_t r, uint32x4_t a, uint32x4_t b)
          A32: VSUBHN.I32 Dd+1, Qn, Qm
          A64: SUBHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsubhn_high_u64 (uint32x2_t r, uint64x2_t a, uint64x2_t b)
          A32: VSUBHN.I64 Dd+1, Qn, Qm
          A64: SUBHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vrsubhn_u16 (uint16x8_t a, uint16x8_t b)
          A32: VRSUBHN.I16 Dd, Qn, Qm
          A64: RSUBHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vrsubhn_s32 (int32x4_t a, int32x4_t b)
          A32: VRSUBHN.I32 Dd, Qn, Qm
          A64: RSUBHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vrsubhn_s64 (int64x2_t a, int64x2_t b)
          A32: VRSUBHN.I64 Dd, Qn, Qm
          A64: RSUBHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vrsubhn_s16 (int16x8_t a, int16x8_t b)
          A32: VRSUBHN.I16 Dd, Qn, Qm
          A64: RSUBHN Vd.8B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vrsubhn_u32 (uint32x4_t a, uint32x4_t b)
          A32: VRSUBHN.I32 Dd, Qn, Qm
          A64: RSUBHN Vd.4H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vrsubhn_u64 (uint64x2_t a, uint64x2_t b)
          A32: VRSUBHN.I64 Dd, Qn, Qm
          A64: RSUBHN Vd.2S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vrsubhn_high_u16 (uint8x8_t r, uint16x8_t a, uint16x8_t b)
          A32: VRSUBHN.I16 Dd+1, Qn, Qm
          A64: RSUBHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vrsubhn_high_s32 (int16x4_t r, int32x4_t a, int32x4_t b)
          A32: VRSUBHN.I32 Dd+1, Qn, Qm
          A64: RSUBHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vrsubhn_high_s64 (int32x2_t r, int64x2_t a, int64x2_t b)
          A32: VRSUBHN.I64 Dd+1, Qn, Qm
          A64: RSUBHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vrsubhn_high_s16 (int8x8_t r, int16x8_t a, int16x8_t b)
          A32: VRSUBHN.I16 Dd+1, Qn, Qm
          A64: RSUBHN2 Vd.16B, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vrsubhn_high_u32 (uint16x4_t r, uint32x4_t a, uint32x4_t b)
          A32: VRSUBHN.I32 Dd+1, Qn, Qm
          A64: RSUBHN2 Vd.8H, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractRoundedHighNarrowingUpper(lower: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vrsubhn_high_u64 (uint32x2_t r, uint64x2_t a, uint64x2_t b)
          A32: VRSUBHN.I64 Dd+1, Qn, Qm
          A64: RSUBHN2 Vd.4S, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqsub_u8 (uint8x8_t a, uint8x8_t b)
          A32: VQSUB.U8 Dd, Dn, Dm
          A64: UQSUB Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqsub_s16 (int16x4_t a, int16x4_t b)
          A32: VQSUB.S16 Dd, Dn, Dm
          A64: SQSUB Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqsub_s32 (int32x2_t a, int32x2_t b)
          A32: VQSUB.S32 Dd, Dn, Dm
          A64: SQSUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqsub_s8 (int8x8_t a, int8x8_t b)
          A32: VQSUB.S8 Dd, Dn, Dm
          A64: SQSUB Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t vqsub_u16 (uint16x4_t a, uint16x4_t b)
          A32: VQSUB.U16 Dd, Dn, Dm
          A64: UQSUB Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vqsub_u32 (uint32x2_t a, uint32x2_t b)
          A32: VQSUB.U32 Dd, Dn, Dm
          A64: UQSUB Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vqsubq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VQSUB.U8 Qd, Qn, Qm
          A64: UQSUB Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqsubq_s16 (int16x8_t a, int16x8_t b)
          A32: VQSUB.S16 Qd, Qn, Qm
          A64: SQSUB Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqsubq_s32 (int32x4_t a, int32x4_t b)
          A32: VQSUB.S32 Qd, Qn, Qm
          A64: SQSUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vqsubq_s64 (int64x2_t a, int64x2_t b)
          A32: VQSUB.S64 Qd, Qn, Qm
          A64: SQSUB Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t vqsubq_s8 (int8x16_t a, int8x16_t b)
          A32: VQSUB.S8 Qd, Qn, Qm
          A64: SQSUB Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vqsubq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VQSUB.U16 Qd, Qn, Qm
          A64: UQSUB Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vqsubq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VQSUB.U32 Qd, Qn, Qm
          A64: UQSUB Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturate(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vqsubq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VQSUB.U64 Qd, Qn, Qm
          A64: UQSUB Vd.2D, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vqsub_s64 (int64x1_t a, int64x1_t b)
          A32: VQSUB.S64 Dd, Dn, Dm
          A64: SQSUB Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def SubtractSaturateScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vqsub_u64 (uint64x1_t a, uint64x1_t b)
          A32: VQSUB.U64 Dd, Dn, Dm
          A64: UQSUB Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def SubtractScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t vsub_f64 (float64x1_t a, float64x1_t b)
          A32: VSUB.F64 Dd, Dn, Dm
          A64: FSUB Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def SubtractScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t vsub_s64 (int64x1_t a, int64x1_t b)
          A32: VSUB.I64 Dd, Dn, Dm
          A64: SUB Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def SubtractScalar(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32_t vsubs_f32 (float32_t a, float32_t b)
          A32: VSUB.F32 Sd, Sn, Sm
          A64: FSUB Sd, Sn, Sm
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def SubtractScalar(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t vsub_u64 (uint64x1_t a, uint64x1_t b)
          A32: VSUB.I64 Dd, Dn, Dm
          A64: SUB Dd, Dn, Dm
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsubl_u8 (uint8x8_t a, uint8x8_t b)
          A32: VSUBL.U8 Qd, Dn, Dm
          A64: USUBL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsubl_s16 (int16x4_t a, int16x4_t b)
          A32: VSUBL.S16 Qd, Dn, Dm
          A64: SSUBL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsubl_s32 (int32x2_t a, int32x2_t b)
          A32: VSUBL.S32 Qd, Dn, Dm
          A64: SSUBL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsubl_s8 (int8x8_t a, int8x8_t b)
          A32: VSUBL.S8 Qd, Dn, Dm
          A64: SSUBL Vd.8H, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsubl_u16 (uint16x4_t a, uint16x4_t b)
          A32: VSUBL.U16 Qd, Dn, Dm
          A64: USUBL Vd.4S, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsubl_u32 (uint32x2_t a, uint32x2_t b)
          A32: VSUBL.U32 Qd, Dn, Dm
          A64: USUBL Vd.2D, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsubw_s8 (int16x8_t a, int8x8_t b)
          A32: VSUBW.S8 Qd, Qn, Dm
          A64: SSUBW Vd.8H, Vn.8H, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsubw_s16 (int32x4_t a, int16x4_t b)
          A32: VSUBW.S16 Qd, Qn, Dm
          A64: SSUBW Vd.4S, Vn.4S, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsubw_s32 (int64x2_t a, int32x2_t b)
          A32: VSUBW.S32 Qd, Qn, Dm
          A64: SSUBW Vd.2D, Vn.2D, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsubw_u8 (uint16x8_t a, uint8x8_t b)
          A32: VSUBW.U8 Qd, Qn, Dm
          A64: USUBW Vd.8H, Vn.8H, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsubw_u16 (uint32x4_t a, uint16x4_t b)
          A32: VSUBW.U16 Qd, Qn, Dm
          A64: USUBW Vd.4S, Vn.4S, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningLower(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsubw_u32 (uint64x2_t a, uint32x2_t b)
          A32: VSUBW.U32 Qd, Qn, Dm
          A64: USUBW Vd.2D, Vn.2D, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsubl_high_u8 (uint8x16_t a, uint8x16_t b)
          A32: VSUBL.U8 Qd, Dn+1, Dm+1
          A64: USUBL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsubl_high_s16 (int16x8_t a, int16x8_t b)
          A32: VSUBL.S16 Qd, Dn+1, Dm+1
          A64: SSUBL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsubw_high_s8 (int16x8_t a, int8x16_t b)
          A32: VSUBW.S8 Qd, Qn, Dm+1
          A64: SSUBW2 Vd.8H, Vn.8H, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vsubw_high_s16 (int32x4_t a, int16x8_t b)
          A32: VSUBW.S16 Qd, Qn, Dm+1
          A64: SSUBW2 Vd.4S, Vn.4S, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsubl_high_s32 (int32x4_t a, int32x4_t b)
          A32: VSUBL.S32 Qd, Dn+1, Dm+1
          A64: SSUBL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t vsubw_high_s32 (int64x2_t a, int32x4_t b)
          A32: VSUBW.S32 Qd, Qn, Dm+1
          A64: SSUBW2 Vd.2D, Vn.2D, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vsubl_high_s8 (int8x16_t a, int8x16_t b)
          A32: VSUBL.S8 Qd, Dn+1, Dm+1
          A64: SSUBL2 Vd.8H, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vsubw_high_u8 (uint16x8_t a, uint8x16_t b)
          A32: VSUBW.U8 Qd, Qn, Dm+1
          A64: USUBW2 Vd.8H, Vn.8H, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsubl_high_u16 (uint16x8_t a, uint16x8_t b)
          A32: VSUBL.U16 Qd, Dn+1, Dm+1
          A64: USUBL2 Vd.4S, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsubw_high_u16 (uint32x4_t a, uint16x8_t b)
          A32: VSUBW.U16 Qd, Qn, Dm+1
          A64: USUBW2 Vd.4S, Vn.4S, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsubl_high_u32 (uint32x4_t a, uint32x4_t b)
          A32: VSUBL.U32 Qd, Dn+1, Dm+1
          A64: USUBL2 Vd.2D, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def SubtractWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vsubw_high_u32 (uint64x2_t a, uint32x4_t b)
          A32: VSUBW.U32 Qd, Qn, Dm+1
          A64: USUBW2 Vd.2D, Vn.2D, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqvtbl1_u8(uint8x16_t t, uint8x8_t idx)
          A32: VTBL Dd, {Dn, Dn+1}, Dm
          A64: TBL Vd.8B, {Vn.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqvtbl1_s8(int8x16_t t, uint8x8_t idx)
          A32: VTBL Dd, {Dn, Dn+1}, Dm
          A64: TBL Vd.8B, {Vn.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqtbl2q_u8(uint8x16x2_t t, uint8x8_t idx)
          A64: TBL Vd.8B, {Vn.16B, Vn+1.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqtbl2q_u8(int8x16x2_t t, uint8x8_t idx)
          A64: TBL Vd.8B, {Vn.16B, Vn+1.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqtbl3q_u8(uint8x16x3_t t, uint8x8_t idx)
          A64: TBL Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqtbl3q_u8(int8x16x3_t t, uint8x8_t idx)
          A64: TBL Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqtbl4q_u8(uint8x16x4_t t, uint8x8_t idx)
          A64: TBL Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookup(table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqtbl4q_u8(int8x16x4_t t, uint8x8_t idx)
          A64: TBL Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqvtbx1_u8(uint8x8_t r, uint8x16_t t, uint8x8_t idx)
          A32: VTBX Dd, {Dn, Dn+1}, Dm
          A64: TBX Vd.8B, {Vn.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.Runtime.Intrinsics.Vector128[int], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqvtbx1_s8(int8x8_t r, int8x16_t t, uint8x8_t idx)
          A32: VTBX Dd, {Dn, Dn+1}, Dm
          A64: TBX Vd.8B, {Vn.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqtbx2q_u8(uint8x16x2_t t, uint8x8_t idx)
          A64: TBX Vd.8B, {Vn.16B, Vn+1.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqtbx2q_u8(int8x16x2_t t, uint8x8_t idx)
          A64: TBX Vd.8B, {Vn.16B, Vn+1.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqtbx3q_u8(uint8x16x3_t t, uint8x8_t idx)
          A64: TBX Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqtbx3q_u8(int8x16x3_t t, uint8x8_t idx)
          A64: TBX Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t vqtbx4q_u8(uint8x16x4_t t, uint8x8_t idx)
          A64: TBX Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def VectorTableLookupExtension(defaultValues: System.Runtime.Intrinsics.Vector64[int], table: System.ValueTuple[System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int], System.Runtime.Intrinsics.Vector128[int]], byteIndexes: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t vqtbx4q_u8(int8x16x4_t t, uint8x8_t idx)
          A64: TBX Vd.8B, {Vn.16B, Vn+1.16B, Vn+2.16B, Vn+3.16B}, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint8x8_t veor_u8 (uint8x8_t a, uint8x8_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float64x1_t veor_f64 (float64x1_t a, float64x1_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t veor_s16 (int16x4_t a, int16x4_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t veor_s32 (int32x2_t a, int32x2_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int64x1_t veor_s64 (int64x1_t a, int64x1_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int8x8_t veor_s8 (int8x8_t a, int8x8_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[float], right: System.Runtime.Intrinsics.Vector64[float]) -> System.Runtime.Intrinsics.Vector64[float]:
        """
        float32x2_t veor_f32 (float32x2_t a, float32x2_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint16x4_t veor_u16 (uint16x4_t a, uint16x4_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t veor_u32 (uint32x2_t a, uint32x2_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint64x1_t veor_u64 (uint64x1_t a, uint64x1_t b)
          A32: VEOR Dd, Dn, Dm
          A64: EOR Vd.8B, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t veorq_u8 (uint8x16_t a, uint8x16_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float64x2_t veorq_f64 (float64x2_t a, float64x2_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t veorq_s16 (int16x8_t a, int16x8_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t veorq_s32 (int32x4_t a, int32x4_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int64x2_t veorq_s64 (int64x2_t a, int64x2_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int8x16_t veorq_s8 (int8x16_t a, int8x16_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[float], right: System.Runtime.Intrinsics.Vector128[float]) -> System.Runtime.Intrinsics.Vector128[float]:
        """
        float32x4_t veorq_f32 (float32x4_t a, float32x4_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        The above native signature does not exist. We provide this additional overload for consistency with the other scalar APIs.
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t veorq_u16 (uint16x8_t a, uint16x8_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t veorq_u32 (uint32x4_t a, uint32x4_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def Xor(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t veorq_u64 (uint64x2_t a, uint64x2_t b)
          A32: VEOR Qd, Qn, Qm
          A64: EOR Vd.16B, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmovl_u8 (uint8x8_t a)
          A32: VMOVL.U8 Qd, Dm
          A64: UXTL Vd.8H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmovl_u16 (uint16x4_t a)
          A32: VMOVL.U16 Qd, Dm
          A64: UXTL Vd.4S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmovl_u32 (uint32x2_t a)
          A32: VMOVL.U32 Qd, Dm
          A64: UXTL Vd.2D, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmovl_u8 (uint8x8_t a)
          A32: VMOVL.U8 Qd, Dm
          A64: UXTL Vd.8H, Vn.8B
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmovl_u16 (uint16x4_t a)
          A32: VMOVL.U16 Qd, Dm
          A64: UXTL Vd.4S, Vn.4H
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningLower(value: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmovl_u32 (uint32x2_t a)
          A32: VMOVL.U32 Qd, Dm
          A64: UXTL Vd.2D, Vn.2S
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmovl_high_u8 (uint8x16_t a)
          A32: VMOVL.U8 Qd, Dm+1
          A64: UXTL2 Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmovl_high_u16 (uint16x8_t a)
          A32: VMOVL.U16 Qd, Dm+1
          A64: UXTL2 Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmovl_high_u32 (uint32x4_t a)
          A32: VMOVL.U32 Qd, Dm+1
          A64: UXTL2 Vd.2D, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint16x8_t vmovl_high_u8 (uint8x16_t a)
          A32: VMOVL.U8 Qd, Dm+1
          A64: UXTL2 Vd.8H, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vmovl_high_u16 (uint16x8_t a)
          A32: VMOVL.U16 Qd, Dm+1
          A64: UXTL2 Vd.4S, Vn.8H
        """
        ...

    @staticmethod
    @overload
    def ZeroExtendWideningUpper(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint64x2_t vmovl_high_u32 (uint32x4_t a)
          A32: VMOVL.U32 Qd, Dm+1
          A64: UXTL2 Vd.2D, Vn.4S
        """
        ...


class Dp(System.Runtime.Intrinsics.Arm.AdvSimd, metaclass=abc.ABCMeta):
    """This class provides access to the ARMv8.2-DotProd hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.AdvSimd.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

    IsSupported: bool

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdot_s32 (int32x2_t r, int8x8_t a, int8x8_t b)
          A32: VSDOT.S8 Dd, Dn, Dm
          A64: SDOT Vd.2S, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdot_u32 (uint32x2_t r, uint8x8_t a, uint8x8_t b)
          A32: VUDOT.U8 Dd, Dn, Dm
          A64: UDOT Vd.2S, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdotq_s32 (int32x4_t r, int8x16_t a, int8x16_t b)
          A32: VSDOT.S8 Qd, Qn, Qm
          A64: SDOT Vd.4S, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdotq_u32 (uint32x4_t r, uint8x16_t a, uint8x16_t b)
          A32: VUDOT.U8 Qd, Qn, Qm
          A64: UDOT Vd.4S, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdot_s32 (int32x2_t r, int8x8_t a, int8x8_t b)
          A32: VSDOT.S8 Dd, Dn, Dm
          A64: SDOT Vd.2S, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdot_u32 (uint32x2_t r, uint8x8_t a, uint8x8_t b)
          A32: VUDOT.U8 Dd, Dn, Dm
          A64: UDOT Vd.2S, Vn.8B, Vm.8B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdotq_s32 (int32x4_t r, int8x16_t a, int8x16_t b)
          A32: VSDOT.S8 Qd, Qn, Qm
          A64: SDOT Vd.4S, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def DotProduct(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdotq_u32 (uint32x4_t r, uint8x16_t a, uint8x16_t b)
          A32: VUDOT.U8 Qd, Qn, Qm
          A64: UDOT Vd.4S, Vn.16B, Vm.16B
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdot_lane_s32 (int32x2_t r, int8x8_t a, int8x8_t b, const int lane)
          A32: VSDOT.S8 Dd, Dn, Dm[lane]
          A64: SDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdot_laneq_s32 (int32x2_t r, int8x8_t a, int8x16_t b, const int lane)
          A32: VSDOT.S8 Dd, Dn, Dm[lane]
          A64: SDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdot_lane_u32 (uint32x2_t r, uint8x8_t a, uint8x8_t b, const int lane)
          A32: VUDOT.U8 Dd, Dn, Dm[lane]
          A64: UDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdot_laneq_u32 (uint32x2_t r, uint8x8_t a, uint8x16_t b, const int lane)
          A32: VUDOT.U8 Dd, Dn, Dm[lane]
          A64: UDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdotq_laneq_s32 (int32x4_t r, int8x16_t a, int8x16_t b, const int lane)
          A32: VSDOT.S8 Qd, Qn, Dm[lane]
          A64: SDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdotq_lane_s32 (int32x4_t r, int8x16_t a, int8x8_t b, const int lane)
          A32: VSDOT.S8 Qd, Qn, Dm[lane]
          A64: SDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdotq_laneq_u32 (uint32x4_t r, uint8x16_t a, uint8x16_t b, const int lane)
          A32: VUDOT.U8 Qd, Qn, Dm[lane]
          A64: UDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdotq_lane_u32 (uint32x4_t r, uint8x16_t a, uint8x8_t b, const int lane)
          A32: VUDOT.U8 Qd, Qn, Dm[lane]
          A64: UDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdot_lane_s32 (int32x2_t r, int8x8_t a, int8x8_t b, const int lane)
          A32: VSDOT.S8 Dd, Dn, Dm[lane]
          A64: SDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vdot_laneq_s32 (int32x2_t r, int8x8_t a, int8x16_t b, const int lane)
          A32: VSDOT.S8 Dd, Dn, Dm[lane]
          A64: SDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdot_lane_u32 (uint32x2_t r, uint8x8_t a, uint8x8_t b, const int lane)
          A32: VUDOT.U8 Dd, Dn, Dm[lane]
          A64: UDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32x2_t vdot_laneq_u32 (uint32x2_t r, uint8x8_t a, uint8x16_t b, const int lane)
          A32: VUDOT.U8 Dd, Dn, Dm[lane]
          A64: UDOT Vd.2S, Vn.8B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdotq_laneq_s32 (int32x4_t r, int8x16_t a, int8x16_t b, const int lane)
          A32: VSDOT.S8 Qd, Qn, Dm[lane]
          A64: SDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vdotq_lane_s32 (int32x4_t r, int8x16_t a, int8x8_t b, const int lane)
          A32: VSDOT.S8 Qd, Qn, Dm[lane]
          A64: SDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdotq_laneq_u32 (uint32x4_t r, uint8x16_t a, uint8x16_t b, const int lane)
          A32: VUDOT.U8 Qd, Qn, Dm[lane]
          A64: UDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...

    @staticmethod
    @overload
    def DotProductBySelectedQuadruplet(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightScaledIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vdotq_lane_u32 (uint32x4_t r, uint8x16_t a, uint8x8_t b, const int lane)
          A32: VUDOT.U8 Qd, Qn, Dm[lane]
          A64: UDOT Vd.4S, Vn.16B, Vm.4B[lane]
        """
        ...


class Sha1(System.Runtime.Intrinsics.Arm.ArmBase, metaclass=abc.ABCMeta):
    """This class provides access to the ARM SHA1 hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.ArmBase.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

    IsSupported: bool

    @staticmethod
    @overload
    def FixedRotate(hash_e: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vsha1h_u32 (uint32_t hash_e)
          A32: SHA1H.32 Qd, Qm
          A64: SHA1H Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def FixedRotate(hash_e: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        uint32_t vsha1h_u32 (uint32_t hash_e)
          A32: SHA1H.32 Qd, Qm
          A64: SHA1H Sd, Sn
        """
        ...

    @staticmethod
    @overload
    def HashUpdateChoose(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_e: System.Runtime.Intrinsics.Vector64[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1cq_u32 (uint32x4_t hash_abcd, uint32_t hash_e, uint32x4_t wk)
          A32: SHA1C.32 Qd, Qn, Qm
          A64: SHA1C Qd, Sn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdateChoose(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_e: System.Runtime.Intrinsics.Vector64[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1cq_u32 (uint32x4_t hash_abcd, uint32_t hash_e, uint32x4_t wk)
          A32: SHA1C.32 Qd, Qn, Qm
          A64: SHA1C Qd, Sn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdateMajority(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_e: System.Runtime.Intrinsics.Vector64[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1mq_u32 (uint32x4_t hash_abcd, uint32_t hash_e, uint32x4_t wk)
          A32: SHA1M.32 Qd, Qn, Qm
          A64: SHA1M Qd, Sn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdateMajority(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_e: System.Runtime.Intrinsics.Vector64[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1mq_u32 (uint32x4_t hash_abcd, uint32_t hash_e, uint32x4_t wk)
          A32: SHA1M.32 Qd, Qn, Qm
          A64: SHA1M Qd, Sn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdateParity(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_e: System.Runtime.Intrinsics.Vector64[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1pq_u32 (uint32x4_t hash_abcd, uint32_t hash_e, uint32x4_t wk)
          A32: SHA1P.32 Qd, Qn, Qm
          A64: SHA1P Qd, Sn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def HashUpdateParity(hash_abcd: System.Runtime.Intrinsics.Vector128[int], hash_e: System.Runtime.Intrinsics.Vector64[int], wk: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1pq_u32 (uint32x4_t hash_abcd, uint32_t hash_e, uint32x4_t wk)
          A32: SHA1P.32 Qd, Qn, Qm
          A64: SHA1P Qd, Sn, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate0(w0_3: System.Runtime.Intrinsics.Vector128[int], w4_7: System.Runtime.Intrinsics.Vector128[int], w8_11: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1su0q_u32 (uint32x4_t w0_3, uint32x4_t w4_7, uint32x4_t w8_11)
          A32: SHA1SU0.32 Qd, Qn, Qm
          A64: SHA1SU0 Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate0(w0_3: System.Runtime.Intrinsics.Vector128[int], w4_7: System.Runtime.Intrinsics.Vector128[int], w8_11: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1su0q_u32 (uint32x4_t w0_3, uint32x4_t w4_7, uint32x4_t w8_11)
          A32: SHA1SU0.32 Qd, Qn, Qm
          A64: SHA1SU0 Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate1(tw0_3: System.Runtime.Intrinsics.Vector128[int], w12_15: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1su1q_u32 (uint32x4_t tw0_3, uint32x4_t w12_15)
          A32: SHA1SU1.32 Qd, Qm
          A64: SHA1SU1 Vd.4S, Vn.4S
        """
        ...

    @staticmethod
    @overload
    def ScheduleUpdate1(tw0_3: System.Runtime.Intrinsics.Vector128[int], w12_15: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint32x4_t vsha1su1q_u32 (uint32x4_t tw0_3, uint32x4_t w12_15)
          A32: SHA1SU1.32 Qd, Qm
          A64: SHA1SU1 Vd.4S, Vn.4S
        """
        ...


class Aes(System.Runtime.Intrinsics.Arm.ArmBase, metaclass=abc.ABCMeta):
    """This class provides access to the ARM AES hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.ArmBase.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

    IsSupported: bool

    @staticmethod
    @overload
    def Decrypt(value: System.Runtime.Intrinsics.Vector128[int], roundKey: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaesdq_u8 (uint8x16_t data, uint8x16_t key)
          A32: AESD.8 Qd, Qm
          A64: AESD Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Decrypt(value: System.Runtime.Intrinsics.Vector128[int], roundKey: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaesdq_u8 (uint8x16_t data, uint8x16_t key)
          A32: AESD.8 Qd, Qm
          A64: AESD Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Encrypt(value: System.Runtime.Intrinsics.Vector128[int], roundKey: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaeseq_u8 (uint8x16_t data, uint8x16_t key)
          A32: AESE.8 Qd, Qm
          A64: AESE Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def Encrypt(value: System.Runtime.Intrinsics.Vector128[int], roundKey: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaeseq_u8 (uint8x16_t data, uint8x16_t key)
          A32: AESE.8 Qd, Qm
          A64: AESE Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def InverseMixColumns(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaesimcq_u8 (uint8x16_t data)
          A32: AESIMC.8 Qd, Qm
          A64: AESIMC Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def InverseMixColumns(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaesimcq_u8 (uint8x16_t data)
          A32: AESIMC.8 Qd, Qm
          A64: AESIMC Vd.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def MixColumns(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaesmcq_u8 (uint8x16_t data)
          A32: AESMC.8 Qd, Qm
          A64: AESMC V>.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def MixColumns(value: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        uint8x16_t vaesmcq_u8 (uint8x16_t data)
          A32: AESMC.8 Qd, Qm
          A64: AESMC V>.16B, Vn.16B
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_p64 (poly64_t a, poly64_t b)
          A32: VMULL.P8 Qd, Dn, Dm
          A64: PMULL Vd.1Q, Vn.1D, Vm.1D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_p64 (poly64_t a, poly64_t b)
          A32: VMULL.P8 Qd, Dn, Dm
          A64: PMULL Vd.1Q, Vn.1D, Vm.1D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_p64 (poly64_t a, poly64_t b)
          A32: VMULL.P8 Qd, Dn, Dm
          A64: PMULL Vd.1Q, Vn.1D, Vm.1D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningLower(left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_p64 (poly64_t a, poly64_t b)
          A32: VMULL.P8 Qd, Dn, Dm
          A64: PMULL Vd.1Q, Vn.1D, Vm.1D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_high_p64 (poly64x2_t a, poly64x2_t b)
          A32: VMULL.P8 Qd, Dn+1, Dm+1
          A64: PMULL2 Vd.1Q, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_high_p64 (poly64x2_t a, poly64x2_t b)
          A32: VMULL.P8 Qd, Dn+1, Dm+1
          A64: PMULL2 Vd.1Q, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_high_p64 (poly64x2_t a, poly64x2_t b)
          A32: VMULL.P8 Qd, Dn+1, Dm+1
          A64: PMULL2 Vd.1Q, Vn.2D, Vm.2D
        """
        ...

    @staticmethod
    @overload
    def PolynomialMultiplyWideningUpper(left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        poly128_t vmull_high_p64 (poly64x2_t a, poly64x2_t b)
          A32: VMULL.P8 Qd, Dn+1, Dm+1
          A64: PMULL2 Vd.1Q, Vn.2D, Vm.2D
        """
        ...


class Crc32(System.Runtime.Intrinsics.Arm.ArmBase, metaclass=abc.ABCMeta):
    """This class provides access to the ARM Crc32 hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.ArmBase.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

        @staticmethod
        @overload
        def ComputeCrc32(crc: int, data: int) -> int:
            """
            uint32_t __crc32d (uint32_t a, uint64_t b)
              A64: CRC32X Wd, Wn, Xm
            """
            ...

        @staticmethod
        @overload
        def ComputeCrc32(crc: int, data: int) -> int:
            """
            uint32_t __crc32d (uint32_t a, uint64_t b)
              A64: CRC32X Wd, Wn, Xm
            """
            ...

        @staticmethod
        @overload
        def ComputeCrc32C(crc: int, data: int) -> int:
            """
            uint32_t __crc32cd (uint32_t a, uint64_t b)
              A64: CRC32CX Wd, Wn, Xm
            """
            ...

        @staticmethod
        @overload
        def ComputeCrc32C(crc: int, data: int) -> int:
            """
            uint32_t __crc32cd (uint32_t a, uint64_t b)
              A64: CRC32CX Wd, Wn, Xm
            """
            ...

    IsSupported: bool

    @staticmethod
    @overload
    def ComputeCrc32(crc: int, data: int) -> int:
        """
        uint32_t __crc32b (uint32_t a, uint8_t b)
          A32: CRC32B Rd, Rn, Rm
          A64: CRC32B Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32(crc: int, data: int) -> int:
        """
        uint32_t __crc32h (uint32_t a, uint16_t b)
          A32: CRC32H Rd, Rn, Rm
          A64: CRC32H Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32(crc: int, data: int) -> int:
        """
        uint32_t __crc32w (uint32_t a, uint32_t b)
          A32: CRC32W Rd, Rn, Rm
          A64: CRC32W Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32(crc: int, data: int) -> int:
        """
        uint32_t __crc32b (uint32_t a, uint8_t b)
          A32: CRC32B Rd, Rn, Rm
          A64: CRC32B Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32(crc: int, data: int) -> int:
        """
        uint32_t __crc32h (uint32_t a, uint16_t b)
          A32: CRC32H Rd, Rn, Rm
          A64: CRC32H Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32(crc: int, data: int) -> int:
        """
        uint32_t __crc32w (uint32_t a, uint32_t b)
          A32: CRC32W Rd, Rn, Rm
          A64: CRC32W Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32C(crc: int, data: int) -> int:
        """
        uint32_t __crc32cb (uint32_t a, uint8_t b)
          A32: CRC32CB Rd, Rn, Rm
          A64: CRC32CB Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32C(crc: int, data: int) -> int:
        """
        uint32_t __crc32ch (uint32_t a, uint16_t b)
          A32: CRC32CH Rd, Rn, Rm
          A64: CRC32CH Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32C(crc: int, data: int) -> int:
        """
        uint32_t __crc32cw (uint32_t a, uint32_t b)
          A32: CRC32CW Rd, Rn, Rm
          A64: CRC32CW Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32C(crc: int, data: int) -> int:
        """
        uint32_t __crc32cb (uint32_t a, uint8_t b)
          A32: CRC32CB Rd, Rn, Rm
          A64: CRC32CB Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32C(crc: int, data: int) -> int:
        """
        uint32_t __crc32ch (uint32_t a, uint16_t b)
          A32: CRC32CH Rd, Rn, Rm
          A64: CRC32CH Wd, Wn, Wm
        """
        ...

    @staticmethod
    @overload
    def ComputeCrc32C(crc: int, data: int) -> int:
        """
        uint32_t __crc32cw (uint32_t a, uint32_t b)
          A32: CRC32CW Rd, Rn, Rm
          A64: CRC32CW Wd, Wn, Wm
        """
        ...


class Rdm(System.Runtime.Intrinsics.Arm.AdvSimd, metaclass=abc.ABCMeta):
    """This class provides access to the ARMv8.1-RDMA hardware instructions via intrinsics"""

    class Arm64(System.Runtime.Intrinsics.Arm.AdvSimd.Arm64, metaclass=abc.ABCMeta):
        """This class has no documentation."""

        IsSupported: bool

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndAddSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlahh_s16 (int16_t a, int16_t b, int16_t c)
              A64: SQRDMLAH Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndAddSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlahs_s32 (int32_t a, int32_t b, int32_t c)
              A64: SQRDMLAH Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndAddSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlahh_s16 (int16_t a, int16_t b, int16_t c)
              A64: SQRDMLAH Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndAddSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlahs_s32 (int32_t a, int32_t b, int32_t c)
              A64: SQRDMLAH Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndSubtractSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlshh_s16 (int16_t a, int16_t b, int16_t c)
              A64: SQRDMLSH Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndSubtractSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlshs_s32 (int32_t a, int32_t b, int32_t c)
              A64: SQRDMLSH Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndSubtractSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlshh_s16 (int16_t a, int16_t b, int16_t c)
              A64: SQRDMLSH Hd, Hn, Hm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingAndSubtractSaturateHighScalar(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlshs_s32 (int32_t a, int32_t b, int32_t c)
              A64: SQRDMLSH Sd, Sn, Sm
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlahh_lane_s16 (int16_t a, int16_t b, int16x4_t v, const int lane)
              A64: SQRDMLAH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlahh_laneq_s16 (int16_t a, int16_t b, int16x8_t v, const int lane)
              A64: SQRDMLAH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlahs_lane_s32 (int32_t a, int32_t b, int32x2_t v, const int lane)
              A64: SQRDMLAH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlahs_laneq_s32 (int32_t a, int32_t b, int32x4_t v, const int lane)
              A64: SQRDMLAH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlahh_lane_s16 (int16_t a, int16_t b, int16x4_t v, const int lane)
              A64: SQRDMLAH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlahh_laneq_s16 (int16_t a, int16_t b, int16x8_t v, const int lane)
              A64: SQRDMLAH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlahs_lane_s32 (int32_t a, int32_t b, int32x2_t v, const int lane)
              A64: SQRDMLAH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlahs_laneq_s32 (int32_t a, int32_t b, int32x4_t v, const int lane)
              A64: SQRDMLAH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlshh_lane_s16 (int16_t a, int16_t b, int16x4_t v, const int lane)
              A64: SQRDMLSH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlshh_laneq_s16 (int16_t a, int16_t b, int16x8_t v, const int lane)
              A64: SQRDMLSH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlshs_lane_s32 (int32_t a, int32_t b, int32x2_t v, const int lane)
              A64: SQRDMLSH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlshs_laneq_s32 (int32_t a, int32_t b, int32x4_t v, const int lane)
              A64: SQRDMLSH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlshh_lane_s16 (int16_t a, int16_t b, int16x4_t v, const int lane)
              A64: SQRDMLSH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int16_t vqrdmlshh_laneq_s16 (int16_t a, int16_t b, int16x8_t v, const int lane)
              A64: SQRDMLSH Hd, Hn, Vm.H[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlshs_lane_s32 (int32_t a, int32_t b, int32x2_t v, const int lane)
              A64: SQRDMLSH Sd, Sn, Vm.S[lane]
            """
            ...

        @staticmethod
        @overload
        def MultiplyRoundedDoublingScalarBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
            """
            int32_t vqrdmlshs_laneq_s32 (int32_t a, int32_t b, int32x4_t v, const int lane)
              A64: SQRDMLSH Sd, Sn, Vm.S[lane]
            """
            ...

    IsSupported: bool

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlah_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VQRDMLAH.S16 Dd, Dn, Dm
          A64: SQRDMLAH Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlah_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VQRDMLAH.S32 Dd, Dn, Dm
          A64: SQRDMLAH Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlahq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VQRDMLAH.S16 Qd, Qn, Qm
          A64: SQRDMLAH Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlahq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VQRDMLAH.S32 Qd, Qn, Qm
          A64: SQRDMLAH Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlah_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VQRDMLAH.S16 Dd, Dn, Dm
          A64: SQRDMLAH Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlah_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VQRDMLAH.S32 Dd, Dn, Dm
          A64: SQRDMLAH Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlahq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VQRDMLAH.S16 Qd, Qn, Qm
          A64: SQRDMLAH Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlahq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VQRDMLAH.S32 Qd, Qn, Qm
          A64: SQRDMLAH Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlsh_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VQRDMLSH.S16 Dd, Dn, Dm
          A64: SQRDMLSH Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlsh_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VQRDMLSH.S32 Dd, Dn, Dm
          A64: SQRDMLSH Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlshq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VQRDMLSH.S16 Qd, Qn, Qm
          A64: SQRDMLSH Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlshq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VQRDMLSH.S32 Qd, Qn, Qm
          A64: SQRDMLSH Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlsh_s16 (int16x4_t a, int16x4_t b, int16x4_t c)
          A32: VQRDMLSH.S16 Dd, Dn, Dm
          A64: SQRDMLSH Vd.4H, Vn.4H, Vm.4H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int]) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlsh_s32 (int32x2_t a, int32x2_t b, int32x2_t c)
          A32: VQRDMLSH.S32 Dd, Dn, Dm
          A64: SQRDMLSH Vd.2S, Vn.2S, Vm.2S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlshq_s16 (int16x8_t a, int16x8_t b, int16x8_t c)
          A32: VQRDMLSH.S16 Qd, Qn, Qm
          A64: SQRDMLSH Vd.8H, Vn.8H, Vm.8H
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int]) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlshq_s32 (int32x4_t a, int32x4_t b, int32x4_t c)
          A32: VQRDMLSH.S32 Qd, Qn, Qm
          A64: SQRDMLSH Vd.4S, Vn.4S, Vm.4S
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlah_lane_s16 (int16x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VQRDMLAH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlah_laneq_s16 (int16x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VQRDMLAH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlah_lane_s32 (int32x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VQRDMLAH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlah_laneq_s32 (int32x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VQRDMLAH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlahq_lane_s16 (int16x8_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VQRDMLAH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlahq_laneq_s16 (int16x8_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VQRDMLAH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlahq_lane_s32 (int32x4_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VQRDMLAH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlahq_laneq_s32 (int32x4_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VQRDMLAH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlah_lane_s16 (int16x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VQRDMLAH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlah_laneq_s16 (int16x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VQRDMLAH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlah_lane_s32 (int32x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VQRDMLAH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlah_laneq_s32 (int32x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VQRDMLAH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLAH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlahq_lane_s16 (int16x8_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VQRDMLAH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlahq_laneq_s16 (int16x8_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VQRDMLAH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlahq_lane_s32 (int32x4_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VQRDMLAH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndAddSaturateHigh(addend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlahq_laneq_s32 (int32x4_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VQRDMLAH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLAH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlsh_lane_s16 (int16x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VQRDMLSH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlsh_laneq_s16 (int16x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VQRDMLSH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlsh_lane_s32 (int32x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VQRDMLSH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlsh_laneq_s32 (int32x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VQRDMLSH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlshq_lane_s16 (int16x8_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VQRDMLSH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlshq_laneq_s16 (int16x8_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VQRDMLSH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlshq_lane_s32 (int32x4_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VQRDMLSH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlshq_laneq_s32 (int32x4_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VQRDMLSH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlsh_lane_s16 (int16x4_t a, int16x4_t b, int16x4_t v, const int lane)
          A32: VQRDMLSH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int16x4_t vqrdmlsh_laneq_s16 (int16x4_t a, int16x4_t b, int16x8_t v, const int lane)
          A32: VQRDMLSH.S16 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.4H, Vn.4H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlsh_lane_s32 (int32x2_t a, int32x2_t b, int32x2_t v, const int lane)
          A32: VQRDMLSH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector64[int], left: System.Runtime.Intrinsics.Vector64[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector64[int]:
        """
        int32x2_t vqrdmlsh_laneq_s32 (int32x2_t a, int32x2_t b, int32x4_t v, const int lane)
          A32: VQRDMLSH.S32 Dd, Dn, Dm[lane]
          A64: SQRDMLSH Vd.2S, Vn.2S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlshq_lane_s16 (int16x8_t a, int16x8_t b, int16x4_t v, const int lane)
          A32: VQRDMLSH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int16x8_t vqrdmlshq_laneq_s16 (int16x8_t a, int16x8_t b, int16x8_t v, const int lane)
          A32: VQRDMLSH.S16 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.8H, Vn.8H, Vm.H[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector64[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlshq_lane_s32 (int32x4_t a, int32x4_t b, int32x2_t v, const int lane)
          A32: VQRDMLSH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...

    @staticmethod
    @overload
    def MultiplyRoundedDoublingBySelectedScalarAndSubtractSaturateHigh(minuend: System.Runtime.Intrinsics.Vector128[int], left: System.Runtime.Intrinsics.Vector128[int], right: System.Runtime.Intrinsics.Vector128[int], rightIndex: int) -> System.Runtime.Intrinsics.Vector128[int]:
        """
        int32x4_t vqrdmlshq_laneq_s32 (int32x4_t a, int32x4_t b, int32x4_t v, const int lane)
          A32: VQRDMLSH.S32 Qd, Qn, Dm[lane]
          A64: SQRDMLSH Vd.4S, Vn.4S, Vm.S[lane]
        """
        ...


