"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheckType = exports.RoutingPolicy = exports.DnsRecordType = exports.Service = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const alias_target_instance_1 = require("./alias-target-instance");
const cname_instance_1 = require("./cname-instance");
const ip_instance_1 = require("./ip-instance");
const namespace_1 = require("./namespace");
const non_ip_instance_1 = require("./non-ip-instance");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
class ServiceBase extends core_1.Resource {
}
/**
 * Define a CloudMap Service.
 *
 * @stability stable
 */
class Service extends ServiceBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_ServiceProps(props);
        const namespaceType = props.namespace.type;
        // Validations
        if (namespaceType === namespace_1.NamespaceType.HTTP && (props.routingPolicy || props.dnsRecordType)) {
            throw new Error('Cannot specify `routingPolicy` or `dnsRecord` when using an HTTP namespace.');
        }
        if (props.healthCheck && props.customHealthCheck) {
            throw new Error('Cannot specify both `healthCheckConfig` and `healthCheckCustomConfig`.');
        }
        if (namespaceType === namespace_1.NamespaceType.DNS_PRIVATE && props.healthCheck) {
            throw new Error('Cannot specify `healthCheckConfig` for a Private DNS namespace.');
        }
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.dnsRecordType === DnsRecordType.CNAME) {
            throw new Error('Cannot use `CNAME` record when routing policy is `Multivalue`.');
        }
        // Additional validation for eventual attachment of LBs
        // The same validation happens later on during the actual attachment
        // of LBs, but we need the property for the correct configuration of
        // routingPolicy anyway, so might as well do the validation as well.
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.loadBalancer) {
            throw new Error('Cannot register loadbalancers when routing policy is `Multivalue`.');
        }
        if (props.healthCheck
            && props.healthCheck.type === HealthCheckType.TCP
            && props.healthCheck.resourcePath) {
            throw new Error('Cannot specify `resourcePath` when using a `TCP` health check.');
        }
        // Set defaults where necessary
        const routingPolicy = (props.dnsRecordType === DnsRecordType.CNAME) || props.loadBalancer
            ? RoutingPolicy.WEIGHTED
            : RoutingPolicy.MULTIVALUE;
        const dnsRecordType = props.dnsRecordType || DnsRecordType.A;
        if (props.loadBalancer
            && (!(dnsRecordType === DnsRecordType.A
                || dnsRecordType === DnsRecordType.AAAA
                || dnsRecordType === DnsRecordType.A_AAAA))) {
            throw new Error('Must support `A` or `AAAA` records to register loadbalancers.');
        }
        const dnsConfig = props.namespace.type === namespace_1.NamespaceType.HTTP
            ? undefined
            : {
                dnsRecords: renderDnsRecords(dnsRecordType, props.dnsTtl),
                namespaceId: props.namespace.namespaceId,
                routingPolicy,
            };
        const healthCheckConfigDefaults = {
            type: HealthCheckType.HTTP,
            failureThreshold: 1,
            resourcePath: props.healthCheck && props.healthCheck.type !== HealthCheckType.TCP
                ? '/'
                : undefined,
        };
        const healthCheckConfig = props.healthCheck && { ...healthCheckConfigDefaults, ...props.healthCheck };
        const healthCheckCustomConfig = props.customHealthCheck;
        // Create service
        const service = new servicediscovery_generated_1.CfnService(this, 'Resource', {
            name: props.name,
            description: props.description,
            dnsConfig,
            healthCheckConfig,
            healthCheckCustomConfig,
            namespaceId: props.namespace.namespaceId,
        });
        this.serviceName = service.attrName;
        this.serviceArn = service.attrArn;
        this.serviceId = service.attrId;
        this.namespace = props.namespace;
        this.dnsRecordType = dnsRecordType;
        this.routingPolicy = routingPolicy;
    }
    /**
     * @stability stable
     */
    static fromServiceAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_ServiceAttributes(attrs);
        class Import extends ServiceBase {
            constructor() {
                super(...arguments);
                this.namespace = attrs.namespace;
                this.serviceId = attrs.serviceId;
                this.serviceArn = attrs.serviceArn;
                this.dnsRecordType = attrs.dnsRecordType;
                this.routingPolicy = attrs.routingPolicy;
                this.serviceName = attrs.serviceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Registers an ELB as a new instance with unique name instanceId in this service.
     *
     * @stability stable
     */
    registerLoadBalancer(id, loadBalancer, customAttributes) {
        return new alias_target_instance_1.AliasTargetInstance(this, id, {
            service: this,
            dnsName: loadBalancer.loadBalancerDnsName,
            customAttributes,
        });
    }
    /**
     * Registers a resource that is accessible using values other than an IP address or a domain name (CNAME).
     *
     * @stability stable
     */
    registerNonIpInstance(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_NonIpInstanceBaseProps(props);
        return new non_ip_instance_1.NonIpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using an IP address.
     *
     * @stability stable
     */
    registerIpInstance(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_IpInstanceBaseProps(props);
        return new ip_instance_1.IpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using a CNAME.
     *
     * @stability stable
     */
    registerCnameInstance(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_servicediscovery_CnameInstanceBaseProps(props);
        return new cname_instance_1.CnameInstance(this, id, {
            service: this,
            ...props,
        });
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "@aws-cdk/aws-servicediscovery.Service", version: "1.139.0" };
function renderDnsRecords(dnsRecordType, dnsTtl = core_1.Duration.minutes(1)) {
    const ttl = dnsTtl.toSeconds();
    if (dnsRecordType === DnsRecordType.A_AAAA) {
        return [{
                type: DnsRecordType.A,
                ttl,
            }, {
                type: DnsRecordType.AAAA,
                ttl,
            }];
    }
    else {
        return [{ type: dnsRecordType, ttl }];
    }
}
/**
 * @stability stable
 */
var DnsRecordType;
(function (DnsRecordType) {
    DnsRecordType["A"] = "A";
    DnsRecordType["AAAA"] = "AAAA";
    DnsRecordType["A_AAAA"] = "A, AAAA";
    DnsRecordType["SRV"] = "SRV";
    DnsRecordType["CNAME"] = "CNAME";
})(DnsRecordType = exports.DnsRecordType || (exports.DnsRecordType = {}));
/**
 * @stability stable
 */
var RoutingPolicy;
(function (RoutingPolicy) {
    RoutingPolicy["WEIGHTED"] = "WEIGHTED";
    RoutingPolicy["MULTIVALUE"] = "MULTIVALUE";
})(RoutingPolicy = exports.RoutingPolicy || (exports.RoutingPolicy = {}));
/**
 * @stability stable
 */
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["HTTP"] = "HTTP";
    HealthCheckType["HTTPS"] = "HTTPS";
    HealthCheckType["TCP"] = "TCP";
})(HealthCheckType = exports.HealthCheckType || (exports.HealthCheckType = {}));
//# sourceMappingURL=data:application/json;base64,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