//! Types for kcl project and modeling-app settings.

pub mod project;

use anyhow::Result;
use parse_display::{Display, FromStr};
use schemars::JsonSchema;
use serde::{Deserialize, Deserializer, Serialize};
use validator::{Validate, ValidateRange};

const DEFAULT_THEME_COLOR: f64 = 264.5;
const DEFAULT_PROJECT_NAME_TEMPLATE: &str = "untitled";

/// User specific settings for the app.
/// These live in `user.toml` in the app's configuration directory.
/// Updating the settings in the app will update this file automatically.
/// Do not edit this file manually, as it may be overwritten by the app.
/// Manual edits can cause corruption of the settings file.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct Configuration {
    /// The settings for the Design Studio.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub settings: Settings,
}

impl Configuration {
    pub fn parse_and_validate(toml_str: &str) -> Result<Self> {
        let settings = toml::from_str::<Self>(toml_str)?;

        settings.validate()?;

        Ok(settings)
    }
}

/// High level settings.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct Settings {
    /// The settings for the Design Studio.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub app: AppSettings,
    /// Settings that affect the behavior while modeling.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub modeling: ModelingSettings,
    /// Settings that affect the behavior of the KCL text editor.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub text_editor: TextEditorSettings,
    /// Settings that affect the behavior of project management.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub project: ProjectSettings,
    /// Settings that affect the behavior of the command bar.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub command_bar: CommandBarSettings,
}

/// Application wide settings.
#[derive(Debug, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct AppSettings {
    /// The settings for the appearance of the app.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub appearance: AppearanceSettings,
    /// The onboarding status of the app.
    #[serde(default, skip_serializing_if = "is_default")]
    pub onboarding_status: OnboardingStatus,
    /// Permanently dismiss the banner warning to download the desktop app.
    /// This setting only applies to the web app. And is temporary until we have Linux support.
    #[serde(default, skip_serializing_if = "is_default")]
    pub dismiss_web_banner: bool,
    /// When the user is idle, teardown the stream after some time.
    #[serde(
        default,
        deserialize_with = "deserialize_stream_idle_mode",
        alias = "streamIdleMode",
        skip_serializing_if = "is_default"
    )]
    stream_idle_mode: Option<u32>,
    /// Allow orbiting in sketch mode.
    #[serde(default, skip_serializing_if = "is_default")]
    pub allow_orbit_in_sketch_mode: bool,
    /// Whether to show the debug panel, which lets you see various states
    /// of the app to aid in development.
    #[serde(default, skip_serializing_if = "is_default")]
    pub show_debug_panel: bool,
    /// If true, the grid cells will be fixed-size, where the width is your default length unit.
    /// If false, the grid will get larger as you zoom out, and smaller as you zoom in.
    #[serde(default = "make_it_so", skip_serializing_if = "is_true")]
    pub fixed_size_grid: bool,
}

/// Default to true.
fn make_it_so() -> bool {
    true
}

fn is_true(b: &bool) -> bool {
    *b
}

impl Default for AppSettings {
    fn default() -> Self {
        Self {
            appearance: Default::default(),
            onboarding_status: Default::default(),
            dismiss_web_banner: Default::default(),
            stream_idle_mode: Default::default(),
            allow_orbit_in_sketch_mode: Default::default(),
            show_debug_panel: Default::default(),
            fixed_size_grid: make_it_so(),
        }
    }
}

fn deserialize_stream_idle_mode<'de, D>(deserializer: D) -> Result<Option<u32>, D::Error>
where
    D: Deserializer<'de>,
{
    #[derive(Deserialize)]
    #[serde(untagged)]
    enum StreamIdleModeValue {
        Number(u32),
        String(String),
        Boolean(bool),
    }

    const DEFAULT_TIMEOUT: u32 = 1000 * 60 * 5;

    Ok(match StreamIdleModeValue::deserialize(deserializer) {
        Ok(StreamIdleModeValue::Number(value)) => Some(value),
        Ok(StreamIdleModeValue::String(value)) => Some(value.parse::<u32>().unwrap_or(DEFAULT_TIMEOUT)),
        // The old type of this value. I'm willing to say no one used it but
        // we can never guarantee it.
        Ok(StreamIdleModeValue::Boolean(true)) => Some(DEFAULT_TIMEOUT),
        Ok(StreamIdleModeValue::Boolean(false)) => None,
        _ => None,
    })
}

#[derive(Debug, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq)]
#[ts(export)]
#[serde(untagged)]
pub enum FloatOrInt {
    String(String),
    Float(f64),
    Int(i64),
}

impl From<FloatOrInt> for f64 {
    fn from(float_or_int: FloatOrInt) -> Self {
        match float_or_int {
            FloatOrInt::String(s) => s.parse().unwrap(),
            FloatOrInt::Float(f) => f,
            FloatOrInt::Int(i) => i as f64,
        }
    }
}

impl From<FloatOrInt> for AppColor {
    fn from(float_or_int: FloatOrInt) -> Self {
        match float_or_int {
            FloatOrInt::String(s) => s.parse::<f64>().unwrap().into(),
            FloatOrInt::Float(f) => f.into(),
            FloatOrInt::Int(i) => (i as f64).into(),
        }
    }
}

/// The settings for the theme of the app.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Validate)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
pub struct AppearanceSettings {
    /// The overall theme of the app.
    #[serde(default, skip_serializing_if = "is_default")]
    pub theme: AppTheme,
    /// The hue of the primary theme color for the app.
    #[serde(default, skip_serializing_if = "is_default")]
    #[validate(nested)]
    pub color: AppColor,
}

#[derive(Debug, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq)]
#[ts(export)]
#[serde(transparent)]
pub struct AppColor(pub f64);

impl Default for AppColor {
    fn default() -> Self {
        Self(DEFAULT_THEME_COLOR)
    }
}

impl From<AppColor> for f64 {
    fn from(color: AppColor) -> Self {
        color.0
    }
}

impl From<f64> for AppColor {
    fn from(color: f64) -> Self {
        Self(color)
    }
}

impl Validate for AppColor {
    fn validate(&self) -> Result<(), validator::ValidationErrors> {
        if !self.0.validate_range(Some(0.0), None, None, Some(360.0)) {
            let mut errors = validator::ValidationErrors::new();
            let mut err = validator::ValidationError::new("color");
            err.add_param(std::borrow::Cow::from("min"), &0.0);
            err.add_param(std::borrow::Cow::from("exclusive_max"), &360.0);
            errors.add("color", err);
            return Err(errors);
        }
        Ok(())
    }
}

/// The overall appearance of the app.
#[derive(
    Debug, Default, Copy, Clone, Deserialize, Serialize, JsonSchema, Display, FromStr, ts_rs::TS, PartialEq, Eq,
)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
#[display(style = "snake_case")]
pub enum AppTheme {
    /// A light theme.
    Light,
    /// A dark theme.
    Dark,
    /// Use the system theme.
    /// This will use dark theme if the system theme is dark, and light theme if the system theme is light.
    #[default]
    System,
}

impl From<AppTheme> for kittycad::types::Color {
    fn from(theme: AppTheme) -> Self {
        match theme {
            AppTheme::Light => kittycad::types::Color {
                r: 249.0 / 255.0,
                g: 249.0 / 255.0,
                b: 249.0 / 255.0,
                a: 1.0,
            },
            AppTheme::Dark => kittycad::types::Color {
                r: 28.0 / 255.0,
                g: 28.0 / 255.0,
                b: 28.0 / 255.0,
                a: 1.0,
            },
            AppTheme::System => {
                // TODO: Check the system setting for the user.
                todo!()
            }
        }
    }
}

/// Settings that affect the behavior while modeling.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Eq, Validate)]
#[serde(rename_all = "snake_case")]
#[ts(export)]
pub struct ModelingSettings {
    /// The default unit to use in modeling dimensions.
    #[serde(default, skip_serializing_if = "is_default")]
    pub base_unit: UnitLength,
    /// The projection mode the camera should use while modeling.
    #[serde(default, skip_serializing_if = "is_default")]
    pub camera_projection: CameraProjectionType,
    /// The methodology the camera should use to orbit around the model.
    #[serde(default, skip_serializing_if = "is_default")]
    pub camera_orbit: CameraOrbitType,
    /// The controls for how to navigate the 3D view.
    #[serde(default, skip_serializing_if = "is_default")]
    pub mouse_controls: MouseControlType,
    /// Toggle touch controls for 3D view navigation
    #[serde(default, skip_serializing_if = "is_default")]
    pub enable_touch_controls: DefaultTrue,
    /// Highlight edges of 3D objects?
    #[serde(default, skip_serializing_if = "is_default")]
    pub highlight_edges: DefaultTrue,
    /// Whether or not Screen Space Ambient Occlusion (SSAO) is enabled.
    #[serde(default, skip_serializing_if = "is_default")]
    pub enable_ssao: DefaultTrue,
    /// Whether or not to show a scale grid in the 3D modeling view
    #[serde(default, skip_serializing_if = "is_default")]
    pub show_scale_grid: bool,
}

#[derive(Debug, Copy, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Eq)]
#[ts(export)]
#[serde(transparent)]
pub struct DefaultTrue(pub bool);

impl Default for DefaultTrue {
    fn default() -> Self {
        Self(true)
    }
}

impl From<DefaultTrue> for bool {
    fn from(default_true: DefaultTrue) -> Self {
        default_true.0
    }
}

impl From<bool> for DefaultTrue {
    fn from(b: bool) -> Self {
        Self(b)
    }
}

/// The valid types of length units.
#[derive(
    Debug, Default, Eq, PartialEq, Copy, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, Display, FromStr,
)]
#[cfg_attr(feature = "pyo3", pyo3::pyclass(eq, eq_int))]
#[ts(export)]
#[serde(rename_all = "lowercase")]
#[display(style = "lowercase")]
pub enum UnitLength {
    /// Centimeters <https://en.wikipedia.org/wiki/Centimeter>
    Cm,
    /// Feet <https://en.wikipedia.org/wiki/Foot_(unit)>
    Ft,
    /// Inches <https://en.wikipedia.org/wiki/Inch>
    In,
    /// Meters <https://en.wikipedia.org/wiki/Meter>
    M,
    /// Millimeters <https://en.wikipedia.org/wiki/Millimeter>
    #[default]
    Mm,
    /// Yards <https://en.wikipedia.org/wiki/Yard>
    Yd,
}

impl From<kittycad::types::UnitLength> for UnitLength {
    fn from(unit: kittycad::types::UnitLength) -> Self {
        match unit {
            kittycad::types::UnitLength::Cm => UnitLength::Cm,
            kittycad::types::UnitLength::Ft => UnitLength::Ft,
            kittycad::types::UnitLength::In => UnitLength::In,
            kittycad::types::UnitLength::M => UnitLength::M,
            kittycad::types::UnitLength::Mm => UnitLength::Mm,
            kittycad::types::UnitLength::Yd => UnitLength::Yd,
        }
    }
}

impl From<UnitLength> for kittycad::types::UnitLength {
    fn from(unit: UnitLength) -> Self {
        match unit {
            UnitLength::Cm => kittycad::types::UnitLength::Cm,
            UnitLength::Ft => kittycad::types::UnitLength::Ft,
            UnitLength::In => kittycad::types::UnitLength::In,
            UnitLength::M => kittycad::types::UnitLength::M,
            UnitLength::Mm => kittycad::types::UnitLength::Mm,
            UnitLength::Yd => kittycad::types::UnitLength::Yd,
        }
    }
}

impl From<kittycad_modeling_cmds::units::UnitLength> for UnitLength {
    fn from(unit: kittycad_modeling_cmds::units::UnitLength) -> Self {
        match unit {
            kittycad_modeling_cmds::units::UnitLength::Centimeters => UnitLength::Cm,
            kittycad_modeling_cmds::units::UnitLength::Feet => UnitLength::Ft,
            kittycad_modeling_cmds::units::UnitLength::Inches => UnitLength::In,
            kittycad_modeling_cmds::units::UnitLength::Meters => UnitLength::M,
            kittycad_modeling_cmds::units::UnitLength::Millimeters => UnitLength::Mm,
            kittycad_modeling_cmds::units::UnitLength::Yards => UnitLength::Yd,
        }
    }
}

impl From<UnitLength> for kittycad_modeling_cmds::units::UnitLength {
    fn from(unit: UnitLength) -> Self {
        match unit {
            UnitLength::Cm => kittycad_modeling_cmds::units::UnitLength::Centimeters,
            UnitLength::Ft => kittycad_modeling_cmds::units::UnitLength::Feet,
            UnitLength::In => kittycad_modeling_cmds::units::UnitLength::Inches,
            UnitLength::M => kittycad_modeling_cmds::units::UnitLength::Meters,
            UnitLength::Mm => kittycad_modeling_cmds::units::UnitLength::Millimeters,
            UnitLength::Yd => kittycad_modeling_cmds::units::UnitLength::Yards,
        }
    }
}

/// The types of controls for how to navigate the 3D view.
#[derive(Debug, Default, Eq, PartialEq, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, Display, FromStr)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
#[display(style = "snake_case")]
pub enum MouseControlType {
    #[default]
    #[display("zoo")]
    #[serde(rename = "zoo")]
    Zoo,
    #[display("onshape")]
    #[serde(rename = "onshape")]
    OnShape,
    TrackpadFriendly,
    Solidworks,
    Nx,
    Creo,
    #[display("autocad")]
    #[serde(rename = "autocad")]
    AutoCad,
}

/// The types of camera projection for the 3D view.
#[derive(Debug, Default, Eq, PartialEq, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, Display, FromStr)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
#[display(style = "snake_case")]
pub enum CameraProjectionType {
    /// Perspective projection https://en.wikipedia.org/wiki/3D_projection#Perspective_projection
    Perspective,
    /// Orthographic projection https://en.wikipedia.org/wiki/3D_projection#Orthographic_projection
    #[default]
    Orthographic,
}

/// The types of camera orbit methods.
#[derive(Debug, Default, Eq, PartialEq, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, Display, FromStr)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
#[display(style = "snake_case")]
pub enum CameraOrbitType {
    /// Orbit using a spherical camera movement.
    #[default]
    #[display("spherical")]
    Spherical,
    /// Orbit using a trackball camera movement.
    #[display("trackball")]
    Trackball,
}

/// Settings that affect the behavior of the KCL text editor.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Eq, Validate)]
#[serde(rename_all = "snake_case")]
#[ts(export)]
pub struct TextEditorSettings {
    /// Whether to wrap text in the editor or overflow with scroll.
    #[serde(default, skip_serializing_if = "is_default")]
    pub text_wrapping: DefaultTrue,
    /// Whether to make the cursor blink in the editor.
    #[serde(default, skip_serializing_if = "is_default")]
    pub blinking_cursor: DefaultTrue,
}

/// Settings that affect the behavior of project management.
#[derive(Debug, Clone, Default, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Eq, Validate)]
#[serde(rename_all = "snake_case")]
#[ts(export)]
pub struct ProjectSettings {
    /// The directory to save and load projects from.
    #[serde(default, skip_serializing_if = "is_default")]
    pub directory: std::path::PathBuf,
    /// The default project name to use when creating a new project.
    #[serde(default, skip_serializing_if = "is_default")]
    pub default_project_name: ProjectNameTemplate,
}

#[derive(Debug, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Eq)]
#[ts(export)]
#[serde(transparent)]
pub struct ProjectNameTemplate(pub String);

impl Default for ProjectNameTemplate {
    fn default() -> Self {
        Self(DEFAULT_PROJECT_NAME_TEMPLATE.to_string())
    }
}

impl From<ProjectNameTemplate> for String {
    fn from(project_name: ProjectNameTemplate) -> Self {
        project_name.0
    }
}

impl From<String> for ProjectNameTemplate {
    fn from(s: String) -> Self {
        Self(s)
    }
}

/// Settings that affect the behavior of the command bar.
#[derive(Debug, Default, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, PartialEq, Eq, Validate)]
#[serde(rename_all = "snake_case")]
#[ts(export)]
pub struct CommandBarSettings {
    /// Whether to include settings in the command bar.
    #[serde(default, skip_serializing_if = "is_default")]
    pub include_settings: DefaultTrue,
}

/// The types of onboarding status.
#[derive(Debug, Default, Eq, PartialEq, Clone, Deserialize, Serialize, JsonSchema, ts_rs::TS, Display, FromStr)]
#[ts(export)]
#[serde(rename_all = "snake_case")]
#[display(style = "snake_case")]
pub enum OnboardingStatus {
    /// The unset state.
    #[serde(rename = "")]
    #[display("")]
    Unset,
    /// The user has completed onboarding.
    Completed,
    /// The user has not completed onboarding.
    #[default]
    Incomplete,
    /// The user has dismissed onboarding.
    Dismissed,

    // Desktop Routes
    #[serde(rename = "/desktop")]
    #[display("/desktop")]
    DesktopWelcome,
    #[serde(rename = "/desktop/scene")]
    #[display("/desktop/scene")]
    DesktopScene,
    #[serde(rename = "/desktop/toolbar")]
    #[display("/desktop/toolbar")]
    DesktopToolbar,
    #[serde(rename = "/desktop/text-to-cad")]
    #[display("/desktop/text-to-cad")]
    DesktopTextToCadWelcome,
    #[serde(rename = "/desktop/text-to-cad-prompt")]
    #[display("/desktop/text-to-cad-prompt")]
    DesktopTextToCadPrompt,
    #[serde(rename = "/desktop/feature-tree-pane")]
    #[display("/desktop/feature-tree-pane")]
    DesktopFeatureTreePane,
    #[serde(rename = "/desktop/code-pane")]
    #[display("/desktop/code-pane")]
    DesktopCodePane,
    #[serde(rename = "/desktop/project-pane")]
    #[display("/desktop/project-pane")]
    DesktopProjectFilesPane,
    #[serde(rename = "/desktop/other-panes")]
    #[display("/desktop/other-panes")]
    DesktopOtherPanes,
    #[serde(rename = "/desktop/prompt-to-edit")]
    #[display("/desktop/prompt-to-edit")]
    DesktopPromptToEditWelcome,
    #[serde(rename = "/desktop/prompt-to-edit-prompt")]
    #[display("/desktop/prompt-to-edit-prompt")]
    DesktopPromptToEditPrompt,
    #[serde(rename = "/desktop/prompt-to-edit-result")]
    #[display("/desktop/prompt-to-edit-result")]
    DesktopPromptToEditResult,
    #[serde(rename = "/desktop/imports")]
    #[display("/desktop/imports")]
    DesktopImports,
    #[serde(rename = "/desktop/exports")]
    #[display("/desktop/exports")]
    DesktopExports,
    #[serde(rename = "/desktop/conclusion")]
    #[display("/desktop/conclusion")]
    DesktopConclusion,

    // Browser Routes
    #[serde(rename = "/browser")]
    #[display("/browser")]
    BrowserWelcome,
    #[serde(rename = "/browser/scene")]
    #[display("/browser/scene")]
    BrowserScene,
    #[serde(rename = "/browser/toolbar")]
    #[display("/browser/toolbar")]
    BrowserToolbar,
    #[serde(rename = "/browser/text-to-cad")]
    #[display("/browser/text-to-cad")]
    BrowserTextToCadWelcome,
    #[serde(rename = "/browser/text-to-cad-prompt")]
    #[display("/browser/text-to-cad-prompt")]
    BrowserTextToCadPrompt,
    #[serde(rename = "/browser/feature-tree-pane")]
    #[display("/browser/feature-tree-pane")]
    BrowserFeatureTreePane,
    #[serde(rename = "/browser/prompt-to-edit")]
    #[display("/browser/prompt-to-edit")]
    BrowserPromptToEditWelcome,
    #[serde(rename = "/browser/prompt-to-edit-prompt")]
    #[display("/browser/prompt-to-edit-prompt")]
    BrowserPromptToEditPrompt,
    #[serde(rename = "/browser/prompt-to-edit-result")]
    #[display("/browser/prompt-to-edit-result")]
    BrowserPromptToEditResult,
    #[serde(rename = "/browser/conclusion")]
    #[display("/browser/conclusion")]
    BrowserConclusion,
}

fn is_default<T: Default + PartialEq>(t: &T) -> bool {
    t == &T::default()
}

#[cfg(test)]
mod tests {
    use pretty_assertions::assert_eq;
    use validator::Validate;

    use super::{
        AppColor, AppSettings, AppTheme, AppearanceSettings, CameraProjectionType, CommandBarSettings, Configuration,
        ModelingSettings, MouseControlType, OnboardingStatus, ProjectNameTemplate, ProjectSettings, Settings,
        TextEditorSettings, UnitLength,
    };

    #[test]
    fn test_settings_empty_file_parses() {
        let empty_settings_file = r#""#;

        let parsed = toml::from_str::<Configuration>(empty_settings_file).unwrap();
        assert_eq!(parsed, Configuration::default());

        // Write the file back out.
        let serialized = toml::to_string(&parsed).unwrap();
        assert_eq!(serialized, r#""#);

        let parsed = Configuration::parse_and_validate(empty_settings_file).unwrap();
        assert_eq!(parsed, Configuration::default());
    }

    #[test]
    fn test_settings_parse_basic() {
        let settings_file = r#"[settings.app]
default_project_name = "untitled"
directory = ""
onboarding_status = "dismissed"

  [settings.app.appearance]
  theme = "dark"

[settings.modeling]
enable_ssao = false
base_unit = "in"
mouse_controls = "zoo"
camera_projection = "perspective"

[settings.project]
default_project_name = "untitled"
directory = ""

[settings.text_editor]
text_wrapping = true"#;

        let expected = Configuration {
            settings: Settings {
                app: AppSettings {
                    onboarding_status: OnboardingStatus::Dismissed,
                    appearance: AppearanceSettings {
                        theme: AppTheme::Dark,
                        color: AppColor(264.5),
                    },
                    ..Default::default()
                },
                modeling: ModelingSettings {
                    enable_ssao: false.into(),
                    base_unit: UnitLength::In,
                    mouse_controls: MouseControlType::Zoo,
                    camera_projection: CameraProjectionType::Perspective,
                    ..Default::default()
                },
                project: ProjectSettings {
                    default_project_name: ProjectNameTemplate("untitled".to_string()),
                    directory: "".into(),
                },
                text_editor: TextEditorSettings {
                    text_wrapping: true.into(),
                    ..Default::default()
                },
                command_bar: CommandBarSettings {
                    include_settings: true.into(),
                },
            },
        };
        let parsed = toml::from_str::<Configuration>(settings_file).unwrap();
        assert_eq!(parsed, expected);

        // Write the file back out.
        let serialized = toml::to_string(&parsed).unwrap();
        assert_eq!(
            serialized,
            r#"[settings.app]
onboarding_status = "dismissed"

[settings.app.appearance]
theme = "dark"

[settings.modeling]
base_unit = "in"
camera_projection = "perspective"
enable_ssao = false
"#
        );

        let parsed = Configuration::parse_and_validate(settings_file).unwrap();
        assert_eq!(parsed, expected);
    }

    #[test]
    fn test_color_validation() {
        let color = AppColor(360.0);

        let result = color.validate();
        if let Ok(r) = result {
            panic!("Expected an error, but got success: {r:?}");
        }
        assert!(result.is_err());
        assert!(
            result
                .unwrap_err()
                .to_string()
                .contains("color: Validation error: color")
        );

        let appearance = AppearanceSettings {
            theme: AppTheme::System,
            color: AppColor(361.5),
        };
        let result = appearance.validate();
        if let Ok(r) = result {
            panic!("Expected an error, but got success: {r:?}");
        }
        assert!(result.is_err());
        assert!(
            result
                .unwrap_err()
                .to_string()
                .contains("color: Validation error: color")
        );
    }

    #[test]
    fn test_settings_color_validation_error() {
        let settings_file = r#"[settings.app.appearance]
color = 1567.4"#;

        let result = Configuration::parse_and_validate(settings_file);
        if let Ok(r) = result {
            panic!("Expected an error, but got success: {r:?}");
        }
        assert!(result.is_err());

        assert!(
            result
                .unwrap_err()
                .to_string()
                .contains("color: Validation error: color")
        );
    }
}
