#  Copyright © 2025 Bentley Systems, Incorporated
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#      http://www.apache.org/licenses/LICENSE-2.0
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

"""
Compute Task API
=============

Normally this code would be generated from an OpenAPI specification. However, the API specification for
the Compute Task API is new and not yet finalized. Therefore, this code is written against the intended API,
with some customisation as needed to make it work.

The client implementation should still abstract these details away from the user, so that the user can interact
with the API in a more user-friendly way, and so that this implementation can be changed without affecting the user.

This code is based on the OpenAPI specification for Compute Task API.
API version: 0.0.1
"""

from evo.common.connector import APIConnector
from evo.common.data import EmptyResponse, HTTPResponse, RequestMethod

from ..models import *  # noqa: F403

__all__ = ["JobApi"]


class JobApi:
    """API client for the job endpoint.

    NOTE: This class is auto generated by OpenAPI Generator
    Ref: https://openapi-generator.tech

    Do not edit the class manually.

    :param connector: Client for communicating with the API.
    """

    def __init__(self, connector: APIConnector):
        self.connector = connector

    async def cancel_job(
        self,
        org_id: str,
        topic: str,
        task: str,
        job_id: str,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> EmptyResponse:
        """Cancels a specific job

        Cancel a specific job, identified by the organization ID, topic, task, and job ID. The job's status is updated to 'canceling' and the job is stopped.

        :param org_id: The organization identifier
            Format: `uuid`
            Example: `'org_id_example'`
        :param topic: The compute topic
            Example: `'topic_example'`
        :param task: The compute task
            Example: `'task_example'`
        :param job_id: The job identifier
            Format: `uuid`
            Example: `'job_id_example'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """  # noqa: E501
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "topic": topic,
            "task": task,
            "job_id": job_id,
        }

        # Prepare the header parameters.
        _header_params = {}
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "204": EmptyResponse,
        }

        return await self.connector.call_api(
            method=RequestMethod.DELETE,
            resource_path="/compute/orgs/{org_id}/{topic}/{task}/{job_id}",
            path_params=_path_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def get_job_results(
        self,
        org_id: str,
        topic: str,
        task: str,
        job_id: str,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> HTTPResponse:  # noqa: F405
        """Retrieves the results of a specific job

        Get the results of a specific job, identified by the organization ID, topic, task, and job ID. The job's status is returned in the response, along with either the results or an error object, depending on the status of the job. If the job is ongoing, the response includes links to the job's cancellation endpoint and detailed status endpoint.

        :param org_id: The organization identifier
            Format: `uuid`
            Example: `'org_id_example'`
        :param topic: The compute topic
            Example: `'topic_example'`
        :param task: The compute task
            Example: `'task_example'`
        :param job_id: The job identifier
            Format: `uuid`
            Example: `'job_id_example'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """  # noqa: E501
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "topic": topic,
            "task": task,
            "job_id": job_id,
        }

        # Prepare the header parameters.
        _header_params = {
            "Accept": "application/json",
        }
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            # Get the raw response so we can include headers if the job failed.
            "200": HTTPResponse,
            "202": HTTPResponse,
        }

        return await self.connector.call_api(
            method=RequestMethod.GET,
            resource_path="/compute/orgs/{org_id}/{topic}/{task}/{job_id}",
            path_params=_path_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def get_job_status(
        self,
        org_id: str,
        topic: str,
        task: str,
        job_id: str,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> JobStatusResponse:  # noqa: F405
        """Retrieves the status of a specific job

        Get the status of a specific job, identified by the organization ID, topic, task, and job ID. The job's status is returned in the response, along with a progress indicator, a message, and an error object, if applicable. If the job is ongoing, the response includes links to the job's cancellation endpoint.

        :param org_id: The organization identifier
            Format: `uuid`
            Example: `'org_id_example'`
        :param topic: The compute topic
            Example: `'topic_example'`
        :param task: The compute task
            Example: `'task_example'`
        :param job_id: The job identifier
            Format: `uuid`
            Example: `'job_id_example'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """  # noqa: E501
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "topic": topic,
            "task": task,
            "job_id": job_id,
        }

        # Prepare the header parameters.
        _header_params = {
            "Accept": "application/json",
        }
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "200": JobStatusResponse,  # noqa: F405
            "202": JobStatusResponse,  # noqa: F405
        }

        return await self.connector.call_api(
            method=RequestMethod.GET,
            resource_path="/compute/orgs/{org_id}/{topic}/{task}/{job_id}/status",
            path_params=_path_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )
