#  Copyright © 2025 Bentley Systems, Incorporated
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#      http://www.apache.org/licenses/LICENSE-2.0
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
"""
Geoscience Object API
=============


The Geoscience Object API enables technological integrations at the object level. It enables users to access and utilise their data across all products through a common and accessible data structure.

A Geoscience Object is a data structure that represents a concrete geological, geotechnical, or geophysical concept. Geoscience Objects can be referenced by their UUID or by a user-defined object path.

For more information on using the Geoscience Object API, see the [Geoscience Object API overview](/docs/guides/objects), or the API references here.


This code is generated from the OpenAPI specification for Geoscience Object API.
API version: 1.14.0
"""

from evo.common.connector import APIConnector
from evo.common.data import EmptyResponse, RequestMethod

from ..models import *  # noqa: F403

__all__ = ["ObjectsApi"]


class ObjectsApi:
    """API client for the Objects endpoint.

    NOTE: This class is auto generated by OpenAPI Generator
    Ref: https://openapi-generator.tech

    Do not edit the class manually.

    :param connector: Client for communicating with the API.
    """

    def __init__(self, connector: APIConnector):
        self.connector = connector

    async def delete_object_by_path(
        self,
        org_id: str,
        workspace_id: str,
        objects_path: str,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> EmptyResponse:
        """Soft delete a geoscience object by path

        Soft delete a geoscience object by path. This endpoint will mark the object as deleted but won't actually remove it from the database.

        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param objects_path: Objects path. This parameter was automatically generated from a wildcard path.
            Example: `'objects_path_example'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "workspace_id": workspace_id,
            "objects_path": objects_path,
        }

        # Prepare the header parameters.
        _header_params = {}
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "204": EmptyResponse,
        }

        return await self.connector.call_api(
            method=RequestMethod.DELETE,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects/path/{objects_path}",
            path_params=_path_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def delete_objects_by_id(
        self,
        object_id: str,
        org_id: str,
        workspace_id: str,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> EmptyResponse:
        """Soft delete a geoscience object

        Delete a geoscience object by UUID. Deleting an object to this endpoint will soft delete the existing object with a corresponding UUID. Note that the value of the `uuid` field in the object must match the value in the URL path.

        :param object_id:
            Format: `uuid`
            Example: `'object_id_example'`
        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "object_id": object_id,
            "org_id": org_id,
            "workspace_id": workspace_id,
        }

        # Prepare the header parameters.
        _header_params = {}
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "204": EmptyResponse,
        }

        return await self.connector.call_api(
            method=RequestMethod.DELETE,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects/{object_id}",
            path_params=_path_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def get_object(
        self,
        org_id: str,
        workspace_id: str,
        objects_path: str,
        version: str | None = None,
        include_versions: bool | None = None,
        if_none_match: str | None = None,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> GetObjectResponse:  # noqa: F405
        """Download a geoscience object

        Download a geoscience object from the provided path. If required, the object can be compressed, currently gzip compression is supported. To receive a compressed payload set header `Accept-Encoding: gzip`.

        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param objects_path: Objects path. This parameter was automatically generated from a wildcard path.
            Example: `'objects_path_example'`
        :param version: (optional) The (timestamp) version of the object to retrieve. Defaults to the latest version if not specified.
            Example: `'2022-08-24T05:43:04.8822793Z'`
        :param include_versions: (optional) Whether to return all versions of the object
            Example: `False`
        :param if_none_match: (optional) Optional header used to check if there are any more recent versions of an object than the one specified. Returns 304 (Not Modified) if there are no newer versions.
            Example: `'1706583776404684724'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "workspace_id": workspace_id,
            "objects_path": objects_path,
        }

        # Prepare the query parameters.
        _query_params = {}
        if version is not None:
            _query_params["version"] = version
        if include_versions is not None:
            _query_params["include_versions"] = include_versions

        # Prepare the header parameters.
        _header_params = {
            "Accept": "application/json",
        }
        if if_none_match is not None:
            _header_params["If-None-Match"] = if_none_match
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "200": GetObjectResponse,  # noqa: F405
            "304": EmptyResponse,
        }

        return await self.connector.call_api(
            method=RequestMethod.GET,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects/path/{objects_path}",
            path_params=_path_params,
            query_params=_query_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def get_object_by_id(
        self,
        object_id: str,
        org_id: str,
        workspace_id: str,
        version: str | None = None,
        include_versions: bool | None = None,
        deleted: bool | None = None,
        if_none_match: str | None = None,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> GetObjectResponse:  # noqa: F405
        """Download a geoscience object by UUID

        Download a geoscience object with the provided UUID. If required, the object can be compressed, currently gzip compression is supported. To receive a compressed payload set header `Accept-Encoding: gzip`.

        :param object_id: UUID of a geoscience object.
            Format: `uuid`
            Example: `'355fa5a6-f37d-11ed-93c1-00155d19a71b'`
        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param version: (optional) The (timestamp) version of the object to retrieve. Defaults to the latest version if not specified.
            Example: `'2022-08-24T05:43:04.8822793Z'`
        :param include_versions: (optional) Whether to return all versions of the object
            Example: `False`
        :param deleted: (optional) When true, only objects that have been deleted will be returned
            Example: `True`
        :param if_none_match: (optional) Optional header used to check if there are any more recent versions of an object than the one specified. Returns 304 (Not Modified) if there are no newer versions.
            Example: `'1706583776404684724'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "object_id": object_id,
            "org_id": org_id,
            "workspace_id": workspace_id,
        }

        # Prepare the query parameters.
        _query_params = {}
        if version is not None:
            _query_params["version"] = version
        if include_versions is not None:
            _query_params["include_versions"] = include_versions
        if deleted is not None:
            _query_params["deleted"] = deleted

        # Prepare the header parameters.
        _header_params = {
            "Accept": "application/json",
        }
        if if_none_match is not None:
            _header_params["If-None-Match"] = if_none_match
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "200": GetObjectResponse,  # noqa: F405
            "304": EmptyResponse,
        }

        return await self.connector.call_api(
            method=RequestMethod.GET,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects/{object_id}",
            path_params=_path_params,
            query_params=_query_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def list_object_version_ids(
        self,
        org_id: str,
        workspace_id: str,
        request_body: list[str],
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> list[LatestObjectVersionIdResponse]:  # noqa: F405
        """List the latest version IDs for geoscience objects

        Request to fetch the latest version IDs for a collection of geoscience objects (maximum of 500). The request body must only contain a list of UUIDs of the objects to check. The response will contain a list of objects with their UUIDs and the latest version ID. If the object from a given UUID does not exist or has been deleted, the request will be successful, but the version ID will be `null`.

        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param request_body:
            Example: `['request_body_example']`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "workspace_id": workspace_id,
        }

        # Prepare the header parameters.
        _header_params = {
            "Content-Type": "application/json",
            "Accept": "application/json",
        }
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {
            "request_body": "",
        }

        _response_types_map = {
            "200": list[LatestObjectVersionIdResponse],  # noqa: F405
        }

        return await self.connector.call_api(
            method=RequestMethod.PATCH,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects",
            path_params=_path_params,
            header_params=_header_params,
            body=request_body,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def list_objects(
        self,
        org_id: str,
        workspace_id: str,
        limit: int | None = None,
        offset: int | None = None,
        deleted: bool | None = None,
        order_by: str | None = None,
        author: list[str] | None = None,
        created_by: list[str] | None = None,
        created_at: list[str] | None = None,
        modified_by: list[str] | None = None,
        modified_at: list[str] | None = None,
        object_name: list[str] | None = None,
        schema_id: list[str] | None = None,
        deleted_by: list[str] | None = None,
        deleted_at: list[str] | None = None,
        geojson_bounding_box: list[str] | None = None,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> ListObjectsResponse:  # noqa: F405
        """List geoscience objects

        List geoscience objects. The response will include all geoscience objects matching the given filters.The geoscience objects will be the latest version of the object. If there are no objects matching the given filters, an empty list is returned.

        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param limit: (optional) Max number of results per page.
            Example: `5000`
        :param offset: (optional) Number of results to skip before returning `limit` number of results.
            Example: `0`
        :param deleted: (optional) When true, only objects that have been deleted will be returned
            Example: `True`
        :param order_by: (optional) A comma separated list of fields to order by, where the default sort order is ascending. To specify the sort order, prefix the field name with either `asc:` or `desc:` for ascending or descending respectively. Field names can either be arbitrary, representing a field nested inside the data, or one of the following known sort fields: `author`, `created_at`, `created_by`, `deleted_at`, `modified_at`, `modified_by`, `object_name`
            Example: `'order_by=name,desc:created_by,asc:object.a.b.c'`
        :param author: (optional) (deprecated, use `created_by`) The profile ID that you want to filter by
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param created_by: (optional) The UUID of the user that created an object
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param created_at: (optional)  A date or dates (max 2) to filter objects by. Dates may contain operator prefixes, in the form `<operator>:<datetime>`. The following operators are available (`lt`=less than, `lte`=less than or equal to, `gt`=greater than, `gte`=greater than or equal to). If you omit the operator, then it is assumed the operator is 'equal to'. In this case you may only supply one date. The dates must also be in a valid ISO 8601 format. Dates may include a UTC offset. If the offset is omitted, the timezone is assumed to be UTC.
            Example: `['gte:2023-03-10T22:56:53Z']`
        :param modified_by: (optional) The UUID of the user that modified an object
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param modified_at: (optional)  A date or dates (max 2) to filter objects by. Dates may contain operator prefixes, in the form `<operator>:<datetime>`. The following operators are available (`lt`=less than, `lte`=less than or equal to, `gt`=greater than, `gte`=greater than or equal to). If you omit the operator, then it is assumed the operator is 'equal to'. In this case you may only supply one date. The dates must also be in a valid ISO 8601 format. Dates may include a UTC offset. If the offset is omitted, the timezone is assumed to be UTC.
            Example: `['gte:2023-03-10T22:56:53Z']`
        :param object_name: (optional) The name of the object to filter on. For backwards compatibility, when no operators are supplied, this will perform a case sensitive prefix match of the object file name. The query `object_name=gold` will match an object with the name `goldcolumns.json`. Providing an operator will query over the entire path to and including the file name.
            Example: `['object.json']`
        :param schema_id: (optional) The type of schema you want to filter by.
            Example: `['/objects/variogram/0.1.0/variogram.schema.json']`
        :param deleted_by: (optional) The UUID of the user that deleted an object
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param deleted_at: (optional)  A date or dates (max 2) to filter objects by. Dates may contain operator prefixes, in the form `<operator>:<datetime>`. The following operators are available (`lt`=less than, `lte`=less than or equal to, `gt`=greater than, `gte`=greater than or equal to). If you omit the operator, then it is assumed the operator is 'equal to'. In this case you may only supply one date. The dates must also be in a valid ISO 8601 format. Dates may include a UTC offset. If the offset is omitted, the timezone is assumed to be UTC.
            Example: `['gte:2023-03-10T22:56:53Z']`
        :param geojson_bounding_box: (optional) The coordinates of the data to spatially search on. The value may optionally include one of two spatial operators, `geowithin` or `geointersects`. If an operator is not defined, then `geointersects` will be used by default. The coordinates must be 5 pairs of longitude and latitude, representing a closed polygon. The first and last coordinates must be the same to close the polygon.
            Example: `['geointersects:(171.6,-44.5),(173.7,-44.5),(173.7,-42.9),(171.6,-42.9),(171.6,-44.5)']`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "workspace_id": workspace_id,
        }

        # Prepare the query parameters.
        _query_params = {}
        if limit is not None:
            _query_params["limit"] = limit
        if offset is not None:
            _query_params["offset"] = offset
        if deleted is not None:
            _query_params["deleted"] = deleted
        if order_by is not None:
            _query_params["order_by"] = order_by
        if author is not None:
            _query_params["author"] = author
        if created_by is not None:
            _query_params["created_by"] = created_by
        if created_at is not None:
            _query_params["created_at"] = created_at
        if modified_by is not None:
            _query_params["modified_by"] = modified_by
        if modified_at is not None:
            _query_params["modified_at"] = modified_at
        if object_name is not None:
            _query_params["object_name"] = object_name
        if schema_id is not None:
            _query_params["schema_id"] = schema_id
        if deleted_by is not None:
            _query_params["deleted_by"] = deleted_by
        if deleted_at is not None:
            _query_params["deleted_at"] = deleted_at
        if geojson_bounding_box is not None:
            _query_params["geojson_bounding_box"] = geojson_bounding_box

        # Prepare the header parameters.
        _header_params = {
            "Accept": "application/json",
        }
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {
            "author": "multi",
            "created_by": "multi",
            "created_at": "multi",
            "modified_by": "multi",
            "modified_at": "multi",
            "object_name": "multi",
            "schema_id": "multi",
            "deleted_by": "multi",
            "deleted_at": "multi",
            "geojson_bounding_box": "multi",
        }

        _response_types_map = {
            "200": ListObjectsResponse,  # noqa: F405
        }

        return await self.connector.call_api(
            method=RequestMethod.GET,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects",
            path_params=_path_params,
            query_params=_query_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def list_objects_by_org(
        self,
        org_id: str,
        offset: int | None = None,
        limit: int | None = None,
        deleted: bool | None = None,
        permitted_workspaces_only: bool | None = None,
        order_by: str | None = None,
        created_by: list[str] | None = None,
        created_at: list[str] | None = None,
        modified_by: list[str] | None = None,
        modified_at: list[str] | None = None,
        object_name: list[str] | None = None,
        schema_id: list[str] | None = None,
        geojson_bounding_box: list[str] | None = None,
        deleted_by: list[str] | None = None,
        deleted_at: list[str] | None = None,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> ListOrgObjectsResponse:  # noqa: F405
        """List geoscience objects by org

        The response will list geoscience objects for the provided organisation ID.

        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param offset: (optional) Number of results to skip before returning `limit` number of results.
            Example: `0`
        :param limit: (optional) Max number of results per page.
            Example: `5000`
        :param deleted: (optional) When true, only objects that have been deleted will be returned
            Example: `True`
        :param permitted_workspaces_only: (optional) Only return objects in workspaces the user is permitted to access.
            Example: `True`
        :param order_by: (optional) A comma separated list of fields to order by, where the default sort order is ascending. To specify the sort order, prefix the field name with either `asc:` or `desc:` for ascending or descending respectively. Field names can either be arbitrary, representing a field nested inside the data, or one of the following known sort fields: `created_at`, `created_by`, `modified_at`, `modified_by`, `object_name`, `deleted_at`
            Example: `'order_by=name,desc:created_by,asc:object.a.b.c'`
        :param created_by: (optional) The UUID of the user that created an object
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param created_at: (optional)  A date or dates (max 2) to filter objects by. Dates may contain operator prefixes, in the form `<operator>:<datetime>`. The following operators are available (`lt`=less than, `lte`=less than or equal to, `gt`=greater than, `gte`=greater than or equal to). If you omit the operator, then it is assumed the operator is 'equal to'. In this case you may only supply one date. The dates must also be in a valid ISO 8601 format. Dates may include a UTC offset. If the offset is omitted, the timezone is assumed to be UTC.
            Example: `['gte:2023-03-10T22:56:53Z']`
        :param modified_by: (optional) The UUID of the user that modified an object
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param modified_at: (optional)  A date or dates (max 2) to filter objects by. Dates may contain operator prefixes, in the form `<operator>:<datetime>`. The following operators are available (`lt`=less than, `lte`=less than or equal to, `gt`=greater than, `gte`=greater than or equal to). If you omit the operator, then it is assumed the operator is 'equal to'. In this case you may only supply one date. The dates must also be in a valid ISO 8601 format. Dates may include a UTC offset. If the offset is omitted, the timezone is assumed to be UTC.
            Example: `['gte:2023-03-10T22:56:53Z']`
        :param object_name: (optional) The name of the object to filter on. For backwards compatibility, when no operators are supplied, this will perform a case sensitive prefix match of the object file name. The query `object_name=gold` will match an object with the name `goldcolumns.json`. Providing an operator will query over the entire path to and including the file name.
            Example: `['object.json']`
        :param schema_id: (optional) The type of schema you want to filter by.
            Example: `['/objects/variogram/0.1.0/variogram.schema.json']`
        :param geojson_bounding_box: (optional) The coordinates of the data to spatially search on. The value may optionally include one of two spatial operators, `geowithin` or `geointersects`. If an operator is not defined, then `geointersects` will be used by default. The coordinates must be 5 pairs of longitude and latitude, representing a closed polygon. The first and last coordinates must be the same to close the polygon.
            Example: `['geointersects:(171.6,-44.5),(173.7,-44.5),(173.7,-42.9),(171.6,-42.9),(171.6,-44.5)']`
        :param deleted_by: (optional) The UUID of the user that deleted an object
            Example: `['00000000-0000-0000-0000-000000000000']`
        :param deleted_at: (optional)  A date or dates (max 2) to filter objects by. Dates may contain operator prefixes, in the form `<operator>:<datetime>`. The following operators are available (`lt`=less than, `lte`=less than or equal to, `gt`=greater than, `gte`=greater than or equal to). If you omit the operator, then it is assumed the operator is 'equal to'. In this case you may only supply one date. The dates must also be in a valid ISO 8601 format. Dates may include a UTC offset. If the offset is omitted, the timezone is assumed to be UTC.
            Example: `['gte:2023-03-10T22:56:53Z']`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
        }

        # Prepare the query parameters.
        _query_params = {}
        if offset is not None:
            _query_params["offset"] = offset
        if limit is not None:
            _query_params["limit"] = limit
        if deleted is not None:
            _query_params["deleted"] = deleted
        if permitted_workspaces_only is not None:
            _query_params["permitted_workspaces_only"] = permitted_workspaces_only
        if order_by is not None:
            _query_params["order_by"] = order_by
        if created_by is not None:
            _query_params["created_by"] = created_by
        if created_at is not None:
            _query_params["created_at"] = created_at
        if modified_by is not None:
            _query_params["modified_by"] = modified_by
        if modified_at is not None:
            _query_params["modified_at"] = modified_at
        if object_name is not None:
            _query_params["object_name"] = object_name
        if schema_id is not None:
            _query_params["schema_id"] = schema_id
        if geojson_bounding_box is not None:
            _query_params["geojson_bounding_box"] = geojson_bounding_box
        if deleted_by is not None:
            _query_params["deleted_by"] = deleted_by
        if deleted_at is not None:
            _query_params["deleted_at"] = deleted_at

        # Prepare the header parameters.
        _header_params = {
            "Accept": "application/json",
        }
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {
            "created_by": "multi",
            "created_at": "multi",
            "modified_by": "multi",
            "modified_at": "multi",
            "object_name": "multi",
            "schema_id": "multi",
            "geojson_bounding_box": "multi",
            "deleted_by": "multi",
            "deleted_at": "multi",
        }

        _response_types_map = {
            "200": ListOrgObjectsResponse,  # noqa: F405
        }

        return await self.connector.call_api(
            method=RequestMethod.GET,
            resource_path="/geoscience-object/orgs/{org_id}/objects",
            path_params=_path_params,
            query_params=_query_params,
            header_params=_header_params,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def post_objects(
        self,
        org_id: str,
        workspace_id: str,
        objects_path: str,
        geoscience_object: GeoscienceObject,  # noqa: F405
        if_match: str | None = None,
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> PostObjectResponse:  # noqa: F405
        """Create a geoscience object

        Create or update a geoscience object.  Upload a geoscience object based on the schema element. Geoscience objects can be stored hierarchically,  by providing the path to the object, including the object name, in the request URL.  A detailed list of the available object schemas can be found in  Seequent's [evo-schemas](https://github.com/seequentevo/evo-schemas) repository.   The submitted properties will be validated against the specified schema properties.  To create a new object, upload it to a unique path with the UUID field set to `null`. The response will contain a reflection of the uploaded object with a newly assigned object ID. This can be used for future requests to update the object.  If a version of the geoscience object already exists at the specified path a new version will be created. To move or rename an existing object, simply upload a new version of it (with the same UUID) to a new, unique path or file name.  Note that every upload of an object is considered a new version, even if no changes have been made.  If required, the object should be compressed, currently gzip compression is supported. Set header `Content-Encoding: gzip` if the content is compressed and `Accept-Encoding: gzip` to receive compressed content.

        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param objects_path: Objects path. This parameter was automatically generated from a wildcard path.
            Example: `'objects_path_example'`
        :param geoscience_object:
            Example: `endpoints.GeoscienceObject()`
        :param if_match: (optional) Optional header to apply an update _only if_ the specified object version is the most recent available on the server.
            Example: `'1706583776404684724'`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "org_id": org_id,
            "workspace_id": workspace_id,
            "objects_path": objects_path,
        }

        # Prepare the header parameters.
        _header_params = {
            "Content-Type": "application/json",
            "Accept": "application/json",
        }
        if if_match is not None:
            _header_params["If-Match"] = if_match
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "201": PostObjectResponse,  # noqa: F405
        }

        return await self.connector.call_api(
            method=RequestMethod.POST,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects/path/{objects_path}",
            path_params=_path_params,
            header_params=_header_params,
            body=geoscience_object,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )

    async def update_objects_by_id(
        self,
        object_id: str,
        org_id: str,
        workspace_id: str,
        deleted: bool | None = None,
        if_match: str | None = None,
        update_geoscience_object: UpdateGeoscienceObject | None = None,  # noqa: F405
        additional_headers: dict[str, str] | None = None,
        request_timeout: int | float | tuple[int | float, int | float] | None = None,
    ) -> PostObjectResponse:  # noqa: F405
        """Update a geoscience object

        Update or restore a geoscience object by UUID. Uploading an object to this endpoint will update (replace, i.e. no partial updates) the existing object with a corresponding UUID. If the object was previously deleted, and the `deleted` parameter is set to `false`, this operation will attempt to restore the object. You cannot provide an object body when attempting to restore an object. Note that the value of the `uuid` field in the object must match the value in the URL path. New objects must be uploaded by path.

        :param object_id: UUID of a geoscience object.
            Format: `uuid`
            Example: `'355fa5a6-f37d-11ed-93c1-00155d19a71b'`
        :param org_id: The customer's organisation ID
            Format: `uuid`
            Example: `'b208a6c9-6881-4b97-b02d-acb5d81299bb'`
        :param workspace_id:
            Format: `uuid`
            Example: `'032806a8-dcd7-11ed-8d5c-00155d8f28b5'`
        :param deleted: (optional) When set to false, the operation will attempt to restore the object.
            Example: `True`
        :param if_match: (optional) Optional header to apply an update _only if_ the specified object version is the most recent available on the server.
            Example: `'1706583776404684724'`
        :param update_geoscience_object: (optional)
            Example: `endpoints.UpdateGeoscienceObject()`
        :param additional_headers: (optional) Additional headers to send with the request.
        :param request_timeout: (optional) Timeout setting for this request. If one number is provided, it will be the
            total request timeout. It can also be a pair (tuple) of (connection, read) timeouts.

        :return: Returns the result object.

        :raise evo.common.exceptions.BadRequestException: If the server responds with HTTP status 400.
        :raise evo.common.exceptions.UnauthorizedException: If the server responds with HTTP status 401.
        :raise evo.common.exceptions.ForbiddenException: If the server responds with HTTP status 403.
        :raise evo.common.exceptions.NotFoundException: If the server responds with HTTP status 404.
        :raise evo.common.exceptions.BaseTypedError: If the server responds with any other HTTP status between
            400 and 599, and the body of the response contains a descriptive `type` parameter.
        :raise evo.common.exceptions.EvoAPIException: If the server responds with any other HTTP status between 400
            and 599, and the body of the response does not contain a `type` parameter.
        :raise evo.common.exceptions.UnknownResponseError: For other HTTP status codes with no corresponding response
            type in `response_types_map`.
        """
        # Prepare the path parameters.
        _path_params = {
            "object_id": object_id,
            "org_id": org_id,
            "workspace_id": workspace_id,
        }

        # Prepare the query parameters.
        _query_params = {}
        if deleted is not None:
            _query_params["deleted"] = deleted

        # Prepare the header parameters.
        _header_params = {
            "Content-Type": "application/json",
            "Accept": "application/json",
        }
        if if_match is not None:
            _header_params["If-Match"] = if_match
        if additional_headers is not None:
            _header_params.update(additional_headers)

        # Define the collection formats.
        _collection_formats = {}

        _response_types_map = {
            "201": PostObjectResponse,  # noqa: F405
            "204": EmptyResponse,
            "303": PostObjectResponse,  # noqa: F405
        }

        return await self.connector.call_api(
            method=RequestMethod.POST,
            resource_path="/geoscience-object/orgs/{org_id}/workspaces/{workspace_id}/objects/{object_id}",
            path_params=_path_params,
            query_params=_query_params,
            header_params=_header_params,
            body=update_geoscience_object,
            collection_formats=_collection_formats,
            response_types_map=_response_types_map,
            request_timeout=request_timeout,
        )
