"""The project shell entry point—orchestrate all commands"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/05_cli.ipynb.

# %% auto 0
__all__ = ['main']

# %% ../nbs/05_cli.ipynb 3
import argparse
import sys
from .core import kill_processes
from .checks import check_prereqs
from .setup import init_nbdev
from .sync import sync
from .ship import ship

# %% ../nbs/05_cli.ipynb 5
def main():
    """Main CLI entry point"""
    parser = argparse.ArgumentParser(
        description="pj: the ProJect management toolkit",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  pj init my-project
  pj init my-project -v --desc "My awesome ML library" --private
  pj init my-project --python 3.11 --author "John Doe"
  pj sync
  pj sync -m "Added new feature"
  pj ship
  pj ship --dry-run

For more info, visit: https://nbdev.fast.ai
        """
    )
    
    subparsers = parser.add_subparsers(dest="command", help="Available commands")
    
    # init subcommand
    init_parser = subparsers.add_parser(
        "init",
        help="Initialize a new nbdev project",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description="""
Initialize a new nbdev project with all the bells and whistles:
- GitHub repository creation
- nbdev project structure with Jupyter hooks
- Virtual environment with uv
- Jupyter kernel registration
- direnv auto-activation
        """,
        epilog="""
Examples:
  pj init my-project
  pj init my-project -v --desc "ML utilities" --private
  pj init my-project --python 3.11 --license mit
        """
    )
    
    # Required arguments
    init_parser.add_argument("name", help="Project name (will be repo and package name)")
    
    # Output options
    output_group = init_parser.add_argument_group("output options")
    output_group.add_argument(
        "-v", "--verbose",
        action="store_true",
        help="Show detailed command output"
    )
    output_group.add_argument(
        "--logfile",
        help="Path to log file (default: PROJECT/init.log)"
    )
    output_group.add_argument(
        "--no-log",
        action="store_true",
        help="Disable logging to file"
    )
    
    # GitHub options
    gh_group = init_parser.add_argument_group("GitHub options")
    gh_group.add_argument(
        "--org",
        help="Create repository under this organization (default: personal account)"
    )
    gh_group.add_argument(
        "--public",
        action="store_true",
        help="Create public repository (default: private)"
    )
    gh_group.add_argument(
        "--description", "--desc",
        help="Repository description"
    )
    
    # Python/venv options
    py_group = init_parser.add_argument_group("Python options")
    py_group.add_argument(
        "--python",
        help="Python version (e.g., 3.11, 3.12)"
    )
    
    # nbdev options
    nbdev_group = init_parser.add_argument_group("nbdev options")
    nbdev_group.add_argument(
        "--author",
        help="Author name (default: from git config)"
    )
    nbdev_group.add_argument(
        "--author-email",
        dest="author_email",
        help="Author email (default: from git config)"
    )
    nbdev_group.add_argument(
        "--license",
        choices=["apache2", "mit", "gpl3", "bsd3"],
        help="License type (default: apache2)"
    )
    nbdev_group.add_argument(
        "--min-python",
        dest="min_python",
        help="Minimum Python version (default: 3.9)"
    )
    
    # Post-init options
    post_group = init_parser.add_argument_group("post-init options")
    post_group.add_argument(
        "--no-preview",
        action="store_true",
        help="Skip opening nbdev_preview"
    )
    post_group.add_argument(
        "--no-lab",
        action="store_true",
        help="Skip launching Jupyter Lab"
    )
    post_group.add_argument(
        "-c", "--code",
        action="store_true",
        help="Open VSCode"
    )
    post_group.add_argument(
        "-q", "--quiet",
        action="store_true",
        help="Quiet mode: skip preview and lab (just cd + tree)"
    )

    # sync subcommand
    sync_parser = subparsers.add_parser(
        "sync",
        help="Sync project: pull, prepare, and push to GitHub",
        description="""
Sync your nbdev project in one command:
1. git pull (aborts on merge conflicts)
2. nbdev_prepare (export, test, clean - aborts if tests fail)
3. git commit -am "message"
4. git push
        """,
        epilog="""
Examples:
  pj sync                    # Uses default message "save"
  pj sync -m "Added tests"   # Custom commit message
  pj sync -v                 # Verbose output
        """
    )
    sync_parser.add_argument(
        "--message", "-m",
        help="Commit message (default: 'save')"
    )
    sync_parser.add_argument(
        "-v", "--verbose",
        action="store_true",
        help="Show detailed command output"
    )

    # ship subcommand
    ship_parser = subparsers.add_parser(
        "ship",
        help="Ship a new release: bump version, build, upload, tag, and release",
        description="""
Ship a complete release in one command:
1. Check for uncommitted changes
2. Bump version with nbdev_bump_version
3. Commit and push version bump
4. Build and upload to PyPI with nbdev_pypi
5. Create git tag and push
6. Create GitHub release with auto-generated notes
        """,
        epilog="""
Examples:
  pj ship                    # Bump patch version (0.0.X)
  pj ship --part 1           # Bump minor version (0.X.0)
  pj ship --part 0           # Bump major version (X.0.0)
  pj ship --dry-run          # Preview without making changes
  pj ship --skip-pypi        # Skip PyPI upload (for testing)
        """
    )
    ship_parser.add_argument(
        "--part",
        type=int,
        default=2,
        choices=[0, 1, 2],
        help="Version part to bump: 0=major, 1=minor, 2=patch (default: 2)"
    )
    ship_parser.add_argument(
        "--dry-run",
        action="store_true",
        help="Show what would be done without making changes"
    )
    ship_parser.add_argument(
        "--force",
        action="store_true",
        help="Ship even with uncommitted changes (not recommended)"
    )
    ship_parser.add_argument(
        "--skip-pypi",
        action="store_true",
        help="Skip PyPI upload (for testing)"
    )
    ship_parser.add_argument(
        "--skip-gh-release",
        action="store_true",
        help="Skip GitHub release creation"
    )
    ship_parser.add_argument(
        "-v", "--verbose",
        action="store_true",
        help="Show detailed command output"
    )

    # kill subcommand
    kill_parser = subparsers.add_parser(
        "kill",
        help="Kill all running background processes (jupyter, nbdev_preview, quarto)"
    )
    
    args = parser.parse_args()
    
    if args.command == "init":
        init_nbdev(args)
    elif args.command == "sync":
        sync(args)
    elif args.command == "ship":
        ship(args)
    elif args.command == "kill":
        kill_processes(args)
    else:
        parser.print_help()
        sys.exit(1)
