"""Main parser interface for the chat parser library."""

from typing import List, Optional, Union, Dict, Tuple, Any

from .models import Chat, Message
from .store import LogStore
from .errors import ErrorCollector


class ChatParser:
    """Main interface for parsing and composing agent chats."""

    def __init__(self, agent_name: str = "claude-code", location: str = None):
        """Initialize parser for a specific agent.

        Args:
            agent_name: Name of the agent (default: "claude-code")
            location: Optional custom location for logs (implementation-specific)
        """
        self.agent_name = agent_name
        self.store = self._create_store(agent_name, location)
        self.error_collector = ErrorCollector()
        self.selected_messages: List[Message] = []
        self._chat_cache: Dict[str, Chat] = {}

    def _create_store(self, agent_name: str, location: str = None) -> LogStore:
        """Create appropriate log store for the agent.

        Args:
            agent_name: Name of the agent
            location: Optional custom location

        Returns:
            LogStore implementation for the agent
        """
        if agent_name in ("claude-code", "claude"):
            from .claude import ClaudeStore
            return ClaudeStore(location=location)
        else:
            raise ValueError(f"Unsupported agent: {agent_name}")

    def list_logs(self) -> List[Tuple[str, Dict[str, Any]]]:
        """Show available logs for the agent.

        Returns:
            List of (log_uri, metadata) tuples
        """
        return self.store.list()

    def parse(self, log_uri: str = None) -> Chat:
        """Extract chat from specific or live log.

        Args:
            log_uri: Log URI (None for live log)

        Returns:
            Parsed Chat object
        """
        if log_uri:
            content = self.store.get(log_uri)
            return self._parse_content(content, log_uri)
        else:
            live_uri = self.store.live()
            if live_uri is None:
                return None
            content = self.store.get(live_uri)
            return self._parse_content(content, live_uri)

    def compose(self, *args) -> str:
        """Create Tigs text output from selected content.

        Args:
            *args: Messages or chats to compose (uses selected items if none provided)

        Returns:
            Tigs text output
        """
        # Determine what to compose
        if args:
            items = list(args)
            if all(isinstance(item, Message) for item in items):
                chat = Chat(messages=items)
            elif all(isinstance(item, Chat) for item in items):
                chat = self._merge_chats(items)
            else:
                raise ValueError("Mixed types in composition")
        else:
            # Use selected messages
            if not self.selected_messages:
                raise ValueError("No messages selected for composition")
            chat = Chat(messages=self.selected_messages)

        return chat.export()

    def select(self, log_uri: str, indices: List[int] = None) -> None:
        """Select messages for composition.

        Args:
            log_uri: Log URI
            indices: Message indices to select (None for all messages)
        """
        # Get or cache the chat
        if log_uri not in self._chat_cache:
            self._chat_cache[log_uri] = self.parse(log_uri)

        chat = self._chat_cache[log_uri]

        if indices is None:
            # Select all messages from the chat
            self.selected_messages.extend(chat.messages)
        else:
            # Select specific messages
            for i in indices:
                if 0 <= i < len(chat.messages):
                    self.selected_messages.append(chat.messages[i])

    def unselect(self, log_uri: str, indices: List[int] = None) -> None:
        """Remove messages from selection.

        Args:
            log_uri: Log URI
            indices: Message indices to unselect (None for all messages from this log)
        """
        # Get or cache the chat
        if log_uri not in self._chat_cache:
            self._chat_cache[log_uri] = self.parse(log_uri)

        chat = self._chat_cache[log_uri]

        if indices is None:
            # Remove all messages from this chat
            self.selected_messages = [msg for msg in self.selected_messages
                                    if msg not in chat.messages]
        else:
            # Remove specific messages
            messages_to_remove = []
            for i in indices:
                if 0 <= i < len(chat.messages):
                    messages_to_remove.append(chat.messages[i])

            self.selected_messages = [msg for msg in self.selected_messages
                                    if msg not in messages_to_remove]

    def clear_selection(self) -> None:
        """Clear current selection."""
        self.selected_messages.clear()
        self._chat_cache.clear()

    def _merge_chats(self, chats: List[Chat]) -> Chat:
        """Merge multiple chats into one.

        Args:
            chats: List of chats to merge

        Returns:
            Merged chat
        """
        if not chats:
            return Chat()

        if len(chats) == 1:
            return chats[0]

        # Use the merge method from the first chat
        result = chats[0]
        for chat in chats[1:]:
            result = result.merge(chat)

        return result

    def _parse_content(self, content: str, log_uri: str) -> Chat:
        """Parse raw log content into Chat object.

        Args:
            content: Raw log content
            log_uri: Log URI (session ID or path)

        Returns:
            Parsed Chat object
        """
        if self.agent_name in ("claude-code", "claude"):
            from .claude.claude_code import Session, ClaudeStore
            from pathlib import Path

            # Determine actual file path
            if "/" in log_uri or "\\" in log_uri:
                # Full path provided
                file_path = Path(log_uri)
            else:
                # Session ID provided - get the actual path from store
                if isinstance(self.store, ClaudeStore):
                    file_path = self.store._project_dir / f"{log_uri}.jsonl"
                else:
                    # Fallback
                    file_path = Path(log_uri)

            session = Session(file_path=file_path)
            session.load()  # Actually load the content
            return session.to_chat()
        else:
            raise ValueError(f"Unsupported agent: {self.agent_name}")

    def get_errors(self) -> Optional[str]:
        """Get error report if any errors occurred.

        Returns:
            Error report string or None
        """
        if self.error_collector.has_errors():
            return self.error_collector.generate_report()
        return None


