"""CLAUDE.md Rule Enforcement.

Rules cannot be overridden under any circumstances.
"""

import importlib.util
import os
import sys
from pathlib import Path
from typing import Any

# Import optional dependencies when available
if importlib.util.find_spec("click"):
    import click
else:
    click = None

if importlib.util.find_spec("rich"):
    from rich.console import Console
    from rich.panel import Panel
    from rich.text import Text
else:
    Console = None
    Panel = None
    Text = None

# Check optional dependencies availability
click_available = click is not None
rich_available = Console is not None
todowrite_available = importlib.util.find_spec("todowrite") is not None


class ClaudeRuleViolationError(Exception):
    """Raised when CLAUDE.md rules are violated."""

    pass


class ClaudeRuleEnforcer:
    """Enforces CLAUDE.md rules - ZERO EXCEPTIONS."""

    def __init__(self, console: Console | None = None):
        self.console = console or Console()
        self.claude_path = Path.cwd() / ".claude" / "CLAUDE.md"
        self.violations: list[str] = []

    def enforce_all_rules(self) -> None:
        """Enforce all CLAUDE.md rules - cannot be bypassed."""
        self._enforce_virtual_environment()
        self._enforce_database_configuration()
        self._enforce_database_content()

        if self.violations:
            self._report_violations()
            raise ClaudeRuleViolationError(
                "CLAUDE.md rules violated - session terminated"
            )

    def _enforce_virtual_environment(self) -> None:
        """MUST have virtual environment activated."""
        # Check if .venv exists
        venv_path = Path.cwd() / ".venv"
        if not venv_path.exists():
            self.violations.append("❌ Virtual environment .venv not found")
            return

        # Check if virtual environment is active
        python_path = sys.executable
        if ".venv" not in python_path:
            error_msg = (
                "❌ Virtual environment not activated - "
                "run: source .venv/bin/activate"
            )
            self.violations.append(error_msg)
            return

        # Verify essential packages are available
        if (
            not click_available
            or not rich_available
            or not todowrite_available
        ):
            missing = []
            if not click_available:
                missing.append("click")
            if not rich_available:
                missing.append("rich")
            if not todowrite_available:
                missing.append("todowrite")
            self.violations.append(
                f"❌ Required packages not available: {', '.join(missing)}"
            )
            return

    def _enforce_database_configuration(self) -> None:
        """MUST use correct database configuration."""
        # Check database URL environment variable
        db_url = os.environ.get("TODOWRITE_DATABASE_URL", "")
        # pragma: allowlist secret
        expected_pattern = "postgresql://todowrite:todowrite_dev_password@localhost:5432/todowrite"

        if not db_url:
            error_msg = (
                "❌ TODOWRITE_DATABASE_URL not set - "
                f"must be: {expected_pattern}"
            )
            self.violations.append(error_msg)
            return

        # Check for forbidden hardcoded paths
        forbidden_patterns = [
            "/Users/",
            "/home/",
            "/opt/",
            "/var/",
            "sqlite:///",  # SQLite is forbidden, must use PostgreSQL
        ]
        # Note: /tmp/ is allowed as it's a legitimate temporary directory
        # location and doesn't pose a security risk in this context

        for pattern in forbidden_patterns:
            if pattern in db_url:
                self.violations.append(
                    f"❌ Forbidden hardcoded path or protocol in database URL: {pattern}"
                )
                return

        # Verify correct pattern
        if not db_url.startswith("postgresql://"):
            self.violations.append(
                f"❌ Incorrect database URL: must use PostgreSQL, found: {db_url[:20]}..."
            )
            return

    def _enforce_database_content(self) -> None:
        """Database must contain required planning structure."""
        try:
            from sqlalchemy import create_engine, text
            from sqlalchemy.orm import sessionmaker
            from todowrite.core.models import Goal

            # Connect to PostgreSQL database
            db_url = os.environ.get("TODOWRITE_DATABASE_URL", "")
            if not db_url:
                self.violations.append(
                    "❌ TODOWRITE_DATABASE_URL not set for database verification"
                )
                return

            engine = create_engine(db_url)
            Session = sessionmaker(bind=engine)
            session = Session()

            # Check database exists and is accessible
            try:
                session.execute(text("SELECT COUNT(*) FROM goals"))
            except Exception as e:
                self.violations.append(f"❌ Database not accessible: {e}")
                session.close()
                return

            # Check for required goal
            required_goal = "Enhance ToDoWrite Planning Capabilities"
            goal_count = (
                session.query(Goal).filter(Goal.title == required_goal).count()
            )

            if goal_count == 0:
                self.violations.append(
                    f"❌ Required goal not found: '{required_goal}'"
                )
                session.close()
                return

            # Check for complete hierarchy (143+ records)
            total_records = 0
            tables = [
                "goals",
                "concepts",
                "contexts",
                "constraints",
                "requirements",
                "acceptance_criteria",
                "interface_contracts",
                "phases",
                "steps",
                "tasks",
                "sub_tasks",
                "commands",
                "labels",
            ]

            # Safe table name mapping to prevent SQL injection
            table_queries = {
                "goals": "SELECT COUNT(*) FROM goals",
                "concepts": "SELECT COUNT(*) FROM concepts",
                "requirements": "SELECT COUNT(*) FROM requirements",
                "acceptance_criteria": (
                    "SELECT COUNT(*) FROM acceptance_criteria"
                ),
                "interface_contracts": (
                    "SELECT COUNT(*) FROM interface_contracts"
                ),
                "phases": "SELECT COUNT(*) FROM phases",
                "steps": "SELECT COUNT(*) FROM steps",
                "tasks": "SELECT COUNT(*) FROM tasks",
                "sub_tasks": "SELECT COUNT(*) FROM sub_tasks",
                "commands": "SELECT COUNT(*) FROM commands",
                "labels": "SELECT COUNT(*) FROM labels",
            }

            for table in tables:
                try:
                    if table not in table_queries:
                        continue

                    query = text(table_queries[table])
                    result = session.execute(query)
                    count = result.fetchone()[0]
                    total_records += count
                except Exception as e:
                    # Log exception but continue with other tables
                    print(f"Warning: Could not count records in {table}: {e}")
                    continue

            if total_records < 143:
                error_msg = (
                    f"❌ Incomplete database structure: {total_records} "
                    "records (expected 143+)"
                )
                self.violations.append(error_msg)
                session.close()
                return

            session.close()

        except Exception as e:
            self.violations.append(f"❌ Database verification failed: {e}")

    def _report_violations(self) -> None:
        """Report rule violations and terminate."""
        error_text = Text(
            "\n🚨 CLAUDE.md RULE VIOLATIONS DETECTED 🚨\n", style="bold red"
        )
        error_text.append(
            "These rules cannot be overridden under any circumstances.\n\n",
            style="red",
        )

        for violation in self.violations:
            error_text.append(f"{violation}\n", style="red")

        error_text.append("\nREQUIRED ACTIONS:\n", style="bold yellow")

        # Add specific fixes based on violations
        if any("virtual environment" in v.lower() for v in self.violations):
            error_text.append(
                "• Run: source $PWD/.venv/bin/activate\n", style="yellow"
            )

        if any("database" in v.lower() for v in self.violations):
            # pragma: allowlist secret
            db_export_cmd = (
                "• Run: export TODOWRITE_DATABASE_URL="
                '"postgresql://todowrite:todowrite_dev_password@localhost:5432/todowrite"\n'
            )
            error_text.append(db_export_cmd, style="yellow")

        if any(
            "Enhance ToDoWrite Planning Capabilities" in v
            for v in self.violations
        ):
            init_cmd = (
                "• Initialize database: python .claude/"
                "auto_init_todowrite_models.py\n"
            )
            error_text.append(init_cmd, style="yellow")

        error_text.append(
            "\nSession terminated. Fix violations and retry.\n",
            style="bold red",
        )

        panel = Panel(
            error_text,
            title="[bold red]ENFORCEMENT FAILURE[/bold red]",
            border_style="red",
            padding=(1, 2),
        )

        self.console.print(panel)

    def verify_startup_sequence(self) -> None:
        """Verify the complete startup sequence is followed."""
        required_env_vars = ["TODOWRITE_DATABASE_URL", "PYTHONPATH"]

        missing_vars = []
        for var in required_env_vars:
            if not os.environ.get(var):
                missing_vars.append(var)

        if missing_vars:
            vars_str = ", ".join(missing_vars)
            error_msg = (
                f"❌ Missing required environment variables: {vars_str}"
            )
            self.violations.append(error_msg)


def enforce_claude_rules(
    _ctx: click.Context, _param: click.Parameter, value: Any
) -> Any:
    """Click callback to enforce CLAUDE.md rules."""
    if value:  # Only enforce when --enforce-claude-rules is used
        enforcer = ClaudeRuleEnforcer()
        try:
            enforcer.enforce_all_rules()
            console = Console()
            console.print(
                Panel(
                    "✅ All CLAUDE.md rules verified and enforced",
                    title="[bold green]RULE ENFORCEMENT SUCCESS[/bold green]",
                    border_style="green",
                )
            )
        except ClaudeRuleViolationError:
            sys.exit(1)
    return value


def verify_database_completeness(
    _ctx: click.Context, _param: click.Parameter, value: Any
) -> Any:
    """Click callback to verify database completeness."""
    if value:
        enforcer = ClaudeRuleEnforcer()
        enforcer._enforce_database_content()
        if enforcer.violations:
            enforcer._report_violations()
            sys.exit(1)
        else:
            console = Console()
            console.print(
                Panel(
                    "✅ Database completeness verified",
                    title=(
                        "[bold green]DATABASE VERIFICATION "
                        "SUCCESS[/bold green]"
                    ),
                    border_style="green",
                )
            )
    return value


def clear_context_enforcement(
    _ctx: click.Context, _param: click.Parameter, value: Any
) -> Any:
    """Enforce rules when /clear command is used."""
    if value:
        # Re-enforce all rules before clearing context
        enforcer = ClaudeRuleEnforcer()
        try:
            enforcer.enforce_all_rules()
        except ClaudeRuleViolationError:
            sys.exit(1)
    return value
