# Copyright 2017 Internap.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
import json
import unittest

from flask import Flask

from fake_ubersmith.api.adapters.data_store import DataStore
from fake_ubersmith.api.methods.client import Client
from fake_ubersmith.api.ubersmith import FakeUbersmithError, UbersmithBase


class TestClientModule(unittest.TestCase):
    def setUp(self):
        self.data_store = DataStore()
        self.client = Client(self.data_store)

        self.app = Flask(__name__)
        self.base_uber_api = UbersmithBase(self.data_store)

        self.client.hook_to(self.base_uber_api)
        self.base_uber_api.hook_to(self.app)

    def test_client_add_creates_a_client(self):

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={
                    "method": "client.add",
                    "first": "John",
                    "last": "Smith",
                    "email": "john.smith@invalid.com",
                    "uber_login": "john",
                    "uber_pass": "smith"
                }
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "1",
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_get_returns_successfully(self):
        self.data_store.clients = [{"clientid": "1"}]

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.get", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": {"clientid": "1"},
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_get_errs_when_no_match(self):
        self.data_store.clients = [{"clientid": "100"}]

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.get", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "",
                "error_code": 1,
                "error_message": "Client ID '1' not found.",
                "status": False
            }
        )

    def test_client_contact_add_creates_a_contact(self):
        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={
                    "method": "client.contact_add",
                    "first": "John",
                    "last": "Smith",
                    "email": "john.smith@invalid.com",
                    "uber_login": "john",
                    "uber_pass": "smith"
                }
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "1",
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_cc_add_is_successful(self):

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_add", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": 1,
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_cc_add_fails_returns_error(self):
        self.client.credit_card_response = FakeUbersmithError(999, 'oh fail')

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_add", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "",
                "error_code": 999,
                "error_message": "oh fail",
                "status": False}
        )

    def test_client_cc_update_is_successful(self):
        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_update", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": True,
                "error_code": None,
                "error_message": "",
                "status": True
             }
        )

    def test_client_cc_update_fails_returns_error(self):
        self.client.credit_card_response = FakeUbersmithError(999, 'oh fail')

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_update", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "",
                "error_code": 999,
                "error_message": "oh fail",
                "status": False
             }
        )

    def test_client_cc_info_with_billing_info_id(self):
        self.data_store.credit_cards = [{"billing_info_id": "123"}]

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_info", "billing_info_id": "123"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": {"123": {"billing_info_id": "123"}},
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_cc_info_with_client_id(self):
        self.data_store.credit_cards = [
            {
                "clientid": "1",
                "billing_info_id": "123"
            }
        ]

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_info", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": {"123": {"billing_info_id": "123", "clientid": "1"}},
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_cc_info_fails(self):
        self.data_store.credit_cards = [
            {
                "clientid": "1",
                "billing_info_id": "123"
            }
        ]

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_info", "bogus": "thing"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "",
                "error_code": 1,
                "error_message":
                    "request failed: client_id parameter not supplied",
                "status": False
            }
        )

    def test_client_cc_delete_is_successful(self):
        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_delete", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": True,
                "error_code": None,
                "error_message": "",
                "status": True
            }
        )

    def test_client_cc_delete_fails(self):
        self.client.credit_card_delete_response = FakeUbersmithError(
            999, 'oh fail'
        )

        with self.app.test_client() as c:
            resp = c.post(
                'api/2.0/',
                data={"method": "client.cc_delete", "client_id": "1"}
            )

        self.assertEqual(resp.status_code, 200)
        self.assertEqual(
            json.loads(resp.data.decode('utf-8')),
            {
                "data": "",
                "error_code": 999,
                "error_message": "oh fail",
                "status": False
            }
        )
