#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
SARM SDK 漏洞 API

提供漏洞相关的API操作。
"""

from typing import List, Dict, Any, Optional, TYPE_CHECKING
from ..models.response import BatchOperationResult
from ..exceptions import SARMValidationError

if TYPE_CHECKING:
    from ..client import SARMClient


class VulnerabilityAPI:
    """
    漏洞API类
    
    注意：漏洞创建只能通过 CarrierDataImportAPI 的载体维度多数据录入接口实现，
    本API仅提供更新和查询功能。
    """
    
    def __init__(self, client: 'SARMClient'):
        self.client = client
    
    def update(self, vulnerability_data: Dict[str, Any]) -> Dict[str, Any]:
        """更新漏洞信息"""
        if 'vuln_unique_id' not in vulnerability_data:
            raise SARMValidationError("更新漏洞时必须提供 vuln_unique_id")
        
        response = self.client.post('/api/vuln/update', data=vulnerability_data)
        return response
    
    def get_list(
        self,
        page: int = 1,
        limit: int = 50,
        cve_id:str = None,
        cwe_id:str = None,
        title:str = None,
        vuln_union_id:str = None,
        severity: Optional[str] = None,
        status: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        获取漏洞列表
        
        Args:
            page: 页码
            limit: 每页条数
            cve_id: CVE ID
            cwe_id: CWE ID
            title: 漏洞标题
            vuln_union_id: 漏洞联合ID
            severity: 漏洞严重程度
            status: 漏洞状态
            
        Returns:
            漏洞列表
        """
        params = {"page": page, "limit": limit}
        
        # 处理字符串参数
        if cve_id:
            params["cve_id"] = cve_id
        if cwe_id:
            params["cwe_id"] = cwe_id
        if title:
            params["title"] = title
        if vuln_union_id:
            params["vuln_union_id"] = vuln_union_id   
        # 处理枚举参数
        if severity:
            params["severity"] = severity
        if status:
            params["status"] = status
        
        response = self.client.get('/api/vuln/', params=params)
        return response 