#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
SARM SDK 漏洞数据模型

定义漏洞相关的数据结构。
"""

from typing import List, Optional, Any, Dict
from pydantic import BaseModel, Field
from datetime import datetime


class VulnIdentifier(BaseModel):
    """漏洞标识符"""
    type: str = Field(..., description="标识符类型")
    id: str = Field(..., description="标识符值")


class VulnCvss(BaseModel):
    """CVSS评分信息"""
    version: str = Field(..., description="CVSS版本")
    score: float = Field(..., description="CVSS评分")
    vector: Optional[str] = Field(None, description="CVSS向量")


class VulnContext(BaseModel):
    """漏洞上下文信息"""
    type: str = Field(..., description="上下文类型")
    name: str = Field(..., description="上下文名称")
    context: str = Field(..., description="上下文内容")
    description: Optional[str] = Field(None, description="上下文描述")


class VulnSolution(BaseModel):
    """漏洞修复方案"""
    type: str = Field(..., description="修复方案类型")
    description: str = Field(..., description="修复方案描述")
    details :str =Field(...,description="修复详情")


class VulnReference(BaseModel):
    """漏洞参考链接"""
    type: str = Field(..., description="参考类型")
    url: str = Field(..., description="参考链接")


class VulnerabilityInsert(BaseModel):
    """漏洞创建模型"""
    vuln_unique_id: str = Field(..., description="漏洞唯一ID")
    security_capability_unique_id: str = Field(..., description="安全能力唯一ID")
    title: Optional[str] = Field(None, description="漏洞标题")
    description: Optional[str] = Field(None, description="漏洞描述")
    severity: Optional[str] = Field(None, description="漏洞等级(Critical,High,Medium,Low,Info)")
    status: str = Field(default="open", description="漏洞状态")
    vulnerability_type: Optional[str] = Field(None, description="漏洞类型")
    cwe_id: Optional[str] = Field(None, description="CWE编号")
    cve_id: Optional[str] = Field(None, description="CVE编号")
    tags: Optional[List[str]] = Field(default_factory=list, description="标签")
    vuln_identifiers: Optional[List[VulnIdentifier]] = Field(default_factory=list, description="其他漏洞编号")
    cvss: Optional[List[VulnCvss]] = Field(default_factory=list, description="CVSS评分信息")
    impact: Optional[str] = Field(None, description="漏洞影响")
    context: Optional[List[VulnContext]] = Field(default_factory=list, description="漏洞上下文信息")
    solution: Optional[List[VulnSolution]] = Field(default_factory=list, description="修复方案")
    reference: Optional[List[VulnReference]] = Field(default_factory=list, description="参考链接")
    report_url: Optional[str] = Field(None, description="报告URL")
    discovery_at: Optional[datetime] = Field(None, description="发现时间")
    owner_name: Optional[str] = Field(None, description="负责人姓名")


class VulnerabilityUpdate(BaseModel):
    """漏洞更新模型"""
    vuln_unique_id: str = Field(..., description="漏洞唯一ID")
    title: Optional[str] = Field(None, description="漏洞标题")
    description: Optional[str] = Field(None, description="漏洞描述")
    severity: Optional[str] = Field(None, description="漏洞等级")
    status: Optional[str] = Field(None, description="漏洞状态")
    vulnerability_type: Optional[str] = Field(None, description="漏洞类型")
    cwe_id: Optional[str] = Field(None, description="CWE编号")
    cve_id: Optional[str] = Field(None, description="CVE编号")
    tags: Optional[List[str]] = Field(None, description="标签")
    vuln_identifiers: Optional[List[VulnIdentifier]] = Field(None, description="其他漏洞编号")
    cvss: Optional[List[VulnCvss]] = Field(None, description="CVSS评分信息")
    impact: Optional[str] = Field(None, description="漏洞影响")
    context: Optional[List[VulnContext]] = Field(None, description="漏洞上下文信息")
    solution: Optional[List[VulnSolution]] = Field(None, description="修复方案")
    reference: Optional[List[VulnReference]] = Field(None, description="参考链接")
    report_url: Optional[str] = Field(None, description="报告URL")
    discovery_at: Optional[datetime] = Field(None, description="发现时间")
    owner_name: Optional[str] = Field(None, description="负责人姓名") 