# Context Engineering Improvements for MAP Framework

> Применение принципов контекстной инженерии для AI-агентов к MAP Framework

## 🎯 Быстрый старт

### Что нового?

**Фаза 1.1 [ГОТОВО]:** Recitation Pattern — механизм фокусировки внимания

```python
from mapify_cli.recitation_manager import RecitationManager

# После TaskDecomposer
manager = RecitationManager(project_root)
plan = manager.create_plan('feat_auth', 'Implement JWT auth', subtasks)

# Перед каждой подзадачей
manager.update_subtask_status(1, 'in_progress')
context = manager.get_current_context()  # Добавить в Actor prompt

# После завершения
manager.update_subtask_status(1, 'completed')
```

**Результат:** `.map/current_plan.md` держит цели свежими в контексте

```markdown
# Current Task: feat_auth

## Overall Goal
Implement JWT-based authentication

## Progress: 1/5 subtasks completed

## Subtasks
- [✓] 1/5: Create User model
- [→] **2/5: Implement login endpoint** (CURRENT)
  - Iterations: 2
  - Last error: Missing JWT import...
- [☐] 3/5: Add token validation
...

## Current Focus
**Subtask 2:** Implement login endpoint
**Acceptance Criteria:** POST /auth/login returns JWT
⚠️ **Retry attempt 2** - review previous errors
```

### Зачем это нужно?

**Проблема:** На длинных задачах (5+ подзадач) модель:
- Забывает исходную цель
- Теряет фокус
- Повторяет ошибки

**Решение:** Recitation — обновлять план перед каждым шагом
- Цели в недавних токенах → высокое внимание
- Видно прогресс (2/5 сделано)
- История ошибок для избежания повторов

**Эффект:**
- +20-30% success rate на задачах 10+ шагов
- -20-30% токены (меньше ретраев)
- +50% наблюдаемость прогресса

## 📚 Документация

### Основные документы

1. **[CONTEXT-ENGINEERING-IMPROVEMENTS.md](../CONTEXT-ENGINEERING-IMPROVEMENTS.md)**
   - 📖 Полный план улучшений (94 стр)
   - 🎯 6 областей, 4 фазы
   - 📊 Метрики и KPI
   - **Для:** Архитекторов и разработчиков MAP

2. **[RECITATION-PATTERN.md](../RECITATION-PATTERN.md)**
   - 📖 Документация Recitation паттерна (50 стр)
   - 💻 Примеры использования
   - ✅ Best practices
   - 📊 Сравнение с альтернативами
   - **Для:** Разработчиков, интегрирующих RecitationManager

3. **[CONTEXT-ENGINEERING-SUMMARY.md](../CONTEXT-ENGINEERING-SUMMARY.md)**
   - 📊 Резюме выполненной работы
   - 🚀 Реализованные функции
   - 📈 Ожидаемые результаты
   - 🗺️ Roadmap
   - **Для:** Менеджеров и стейкхолдеров

### Исходники

- `src/mapify_cli/recitation_manager.py` — RecitationManager implementation
- `tests/test_recitation_manager.py` — 41 unit/integration тест

## 🏗️ Архитектура

### Что было (до улучшений)

```
Оркестратор → TaskDecomposer → [Subtask 1, 2, 3, ...]
                                       ↓
                              Actor ←→ Monitor (N итераций)
                                       ↓
                              ❌ ПРОБЛЕМА: На подзадаче 10
                              модель забыла цель подзадачи 1
```

### Что стало (с Recitation)

```
Оркестратор → TaskDecomposer → RecitationManager.create_plan()
                                       ↓
                    .map/current_plan.md (обновляется каждый шаг)
                                       ↓
              Для каждой подзадачи:
                update_status('in_progress')
                       ↓
              Actor получает контекст:
                {base_prompt} + {playbook} + {CURRENT_PLAN} ← СВЕЖИЕ ЦЕЛИ
                       ↓
              Actor ←→ Monitor
                       ↓
              update_status('completed' | retry with error)
                       ↓
              ✅ РЕШЕНИЕ: Цели всегда в недавних токенах
```

### Интеграция

```python
# В оркестраторе (map-feature.md)

# 1. После TaskDecomposer
decomposition = task_decomposer.run(request)
recitation = RecitationManager(project_root)
recitation.create_plan(task_id, goal, decomposition['subtasks'])

# 2. Для каждой подзадачи
for subtask in subtasks:
    recitation.update_subtask_status(subtask.id, 'in_progress')
    plan_context = recitation.get_current_context()

    # 3. Добавить в Actor prompt
    actor_prompt = f"""
    {base_prompt}
    {{{{playbook_bullets}}}}

    ## CURRENT TASK PLAN (Review before starting)
    {plan_context}

    ## Your subtask
    {subtask.description}
    """

    # 4. Actor-Monitor loop
    for attempt in range(MAX_ITER):
        output = actor.run(actor_prompt)
        result = monitor.validate(output)

        if result.approved:
            recitation.update_subtask_status(subtask.id, 'completed')
            break
        else:
            # Записать ошибку для следующей итерации
            recitation.update_subtask_status(
                subtask.id,
                'in_progress',
                error=result.feedback
            )
            # plan автоматически обновится с info об ошибке
```

## 📊 Ожидаемые результаты

### Метрики (после Фазы 1)

| Метрика | До | После | Δ |
|---------|-----|-------|---|
| Success rate (5+ subtasks) | 80% | 90-95% | +13% |
| Avg iterations/subtask | 3.0 | 2.0 | -33% |
| Token usage (5-task feature) | 15K | 11K | -27% |
| Cost per feature | $0.35 | $0.28 | -20% |
| Time to complete | 8 min | 5 min | -37% |

### Качественные улучшения

**Для разработчиков:**
- 🔍 Лучшая отладка (`.map/current_plan.md` + логи)
- 📊 Метрики для оптимизации
- 🧪 Тестируемость (checkpoints)

**Для пользователей:**
- 👀 Прозрачность (видно прогресс в реальном времени)
- ⚡ Быстрее (меньше ретраев, параллелизм)
- 💰 Дешевле (оптимизация токенов)

**Для AI модели:**
- 🧠 Свежие цели в контексте (Recitation)
- 🎓 Лучшие паттерны (ограничено до 3-5)
- 📝 Лаконичные выводы (меньше шума)

## 🗺️ Roadmap

### ✅ Фаза 1.1 [ГОТОВО]
- [x] Recitation Pattern
- [x] Документация
- [x] Тесты

### ⏳ Фаза 1 (осталось 3 задачи)
- [ ] 1.2: Подробное логирование
- [ ] 1.3: Ограничение паттернов (3-5)
- [ ] 1.4: Оптимизация verbose выводов

### 📅 Фаза 2 (2-3 недели)
- Checkpoints (снимки состояния)
- Кэширование MCP-инструментов
- Варьирование формулировок playbook
- Комбинированный keyword+semantic поиск

### 📅 Фаза 3 (1-2 месяца)
- Параллелизм для Reflector/Curator
- Автотесты с mock-агентами
- Настройка температуры по агентам
- Мониторинг и профилирование

### 📅 Фаза 4 (опционально)
- LangChain адаптеры
- Document Loaders для chunking

## 🧪 Тестирование

### Запуск тестов Recitation Manager

```bash
# Все тесты
pytest tests/test_recitation_manager.py -v

# Конкретный класс
pytest tests/test_recitation_manager.py::TestRecitationManagerCreation -v

# С coverage
pytest tests/test_recitation_manager.py --cov=mapify_cli.recitation_manager

# Example usage
python src/mapify_cli/recitation_manager.py
```

### Пример output

```
Created plan: /tmp/test_project/.map/current_plan.md

Plan markdown:

# Current Task: feat_auth

## Overall Goal
Implement JWT-based authentication with email/password

## Progress: 0/5 subtasks completed

## Subtasks
- [☐] 1/5: Create User model with password hashing
- [☐] 2/5: Implement login endpoint
- [☐] 3/5: Add JWT token generation
- [☐] 4/5: Implement token validation middleware
- [☐] 5/5: Add refresh token mechanism

## Current Focus
**Subtask 1:** Create User model with password hashing
**Acceptance Criteria:** Model validates email, hashes password with bcrypt
**Complexity:** low

---
_Updated: 2025-10-18 14:30:22_
```

## 🔗 Источники

1. **Статья:** ["Context Engineering for AI Agents: Lessons from Building Manus"](https://manus.im/blog/Context-Engineering-for-AI-Agents-Lessons-from-Building-Manus)
   - Recitation pattern
   - KV-кэш оптимизация
   - Внешняя память
   - Few-shot bias

2. **PDF:** `../research/context-engenering.pdf`
   - Адаптация для MAP Framework
   - Детальные рекомендации
   - Примеры реализации

3. **LangChain/LangGraph**
   - Multi-action agents
   - State machines
   - Memory patterns

## 📞 Контакты и поддержка

- **Issues:** [GitHub Issues](https://github.com/azalio/map-framework/issues)
- **Discussions:** [GitHub Discussions](https://github.com/azalio/map-framework/discussions)
- **Docs:** `docs/CONTEXT-ENGINEERING-*.md`

## 📝 Changelog

### v1.0.0 (2025-10-18)
- ✨ Добавлен RecitationManager
- 📖 Создана документация (144 стр)
- 🧪 Написаны тесты (41 тест)
- 📋 План улучшений на 4 фазы

---

**Статус:** Active Development — Фаза 1 в процессе
**Maintainer:** MAP Framework Team
**License:** MIT
