import logging
import os
from typing import Any, Dict, List, Optional

from langchain.prompts import ChatPromptTemplate
from langchain_core.language_models.chat_models import BaseChatModel

from adaptiq.core.entities import (
    AgentTool,
    FormattedAnalysis,
    HypotheticalStateRepresentation,
    StatusSummary,
    TaskIntent,
)
from adaptiq.core.q_table.q_table_manager import QTableManager


class PromptEstimator:
    """
    PromptEstimator generates optimized system prompts for agents based on pre-run pipeline analysis results.

    This class takes the results from various pre-run components (parsing, hypothetical states, Q-table, analysis)
    and uses an LLM to synthesize an improved system prompt that incorporates best practices and recommendations.
    """

    def __init__(
        self,
        status: StatusSummary,
        old_prompt: str,
        agent_id: str,
        llm: BaseChatModel,
        parsed_steps: Optional[List[TaskIntent]] = None,
        hypothetical_states: Optional[List[HypotheticalStateRepresentation]] = None,
        offline_learner: Optional[QTableManager] = None,
        prompt_analysis: Optional[FormattedAnalysis] = None,
        agent_tools: Optional[List[AgentTool]] = None,
        output_path: Optional[str] = None,
    ):
        """
        Initialize the PromptPreEstimator.

        Args:
            status: StatusSummary object containing the status of pre-run components.
            old_prompt: The original prompt to be optimized.
            agent_id: Unique identifier for the agent.
            llm: BaseChatModel instance for LLM interactions.
            parsed_steps: List of TaskIntent objects representing parsed steps.
            hypothetical_states: List of HypotheticalStateRepresentation objects.
            offline_learner: QTableManager instance for Q-table management.
            prompt_analysis: FormattedAnalysis object containing prompt analysis results.
            agent_tools: List of AgentTool objects available to the agent.
            output_path: Directory path to save the generated prompt report.
        """
        # Configure logging
        self.logger = logging.getLogger("PromptPreEstimator")

        # Store core parameters
        self.status = status
        self.old_prompt = old_prompt
        self.agent_id = agent_id
        self.llm = llm
        # Store analysis results
        self.parsed_steps = parsed_steps or []
        self.hypothetical_states = hypothetical_states or []
        self.offline_learner = offline_learner
        self.prompt_analysis = prompt_analysis or {}
        self.agent_tools = agent_tools or []
        self.output_path = output_path

        # Process agent tools into string format
        self.tool_strings = [
            f"{tool_dict.name}: {tool_dict.description}"
            for tool_dict in self.agent_tools
        ]
        self.tools_string = "\n".join(self.tool_strings)

    def generate_estimated_prompt(self) -> str:
        """
        Generate an optimized system prompt for the agent based on the pre-run pipeline results.

        This method:
        - Checks if all pre-run pipeline components have completed
        - Summarizes key findings from each phase
        - Uses an LLM to synthesize a new, improved system prompt
        - Saves the generated prompt to a report file if output_path is provided

        Returns:
            str: The optimized system prompt generated by the LLM.
        """
        self.logger.info("Generating comprehensive pre-run analysis report...")

        # Check for missing components
        missing_components = []
        if not self.status.prompt_parsing.completed:
            missing_components.append("prompt_parsing")
        if not self.status.hypothetical_representation.completed:
            missing_components.append("hypothetical_representation")
        if not self.status.scenario_simulation.completed:
            missing_components.append("scenario_simulation")
        if not self.status.qtable_initialization.completed:
            missing_components.append("qtable_initialization")
        if not self.status.prompt_analysis.completed:
            missing_components.append("prompt_analysis")

        if missing_components:
            self.logger.warning(
                "Missing results from: %s. Some analysis may be incomplete.",
                ", ".join(missing_components),
            )

        try:
            # Process parsed steps
            num_parsed_steps = len(self.parsed_steps)
            first_few_subtasks = []
            for step in self.parsed_steps[:3]:  # Get first three steps
                subtask_name = step.intended_subtask
                first_few_subtasks.append(subtask_name)

            # Process hypothetical states
            num_hypothetical_states = len(self.hypothetical_states)

            # Process Q-table information
            q_table_size = (
                len(self.offline_learner.Q_table) if self.offline_learner else 0
            )

            # Get heuristics summary
            heuristic_counts = self._analyze_heuristics()

            # Format heuristics summary
            key_heuristics = []
            for heuristic, count in heuristic_counts.items():
                key_heuristics.append(f"{heuristic} (applied {count} times)")

            # Process prompt analysis
            prompt_analysis_summary = self._create_prompt_analysis_summary()

            # Generate the optimized prompt using LLM
            optimized_prompt = self._generate_prompt_with_llm(
                num_parsed_steps=num_parsed_steps,
                first_few_subtasks=first_few_subtasks,
                num_hypothetical_states=num_hypothetical_states,
                q_table_size=q_table_size,
                key_heuristics=key_heuristics,
                prompt_analysis_summary=prompt_analysis_summary,
            )

            # Save the report if output path is provided
            if self.output_path:
                self._save_report(optimized_prompt)

            return optimized_prompt

        except ImportError as e:
            self.logger.error(f"Required package not installed: {str(e)}")
            return f"Failed to generate report: {str(e)}. Please ensure langchain and openai packages are installed."
        except Exception as e:
            self.logger.error(f"Report generation failed: {str(e)}")
            return f"Failed to generate report: {str(e)}"

    def _analyze_heuristics(self) -> Dict[str, int]:
        """Analyze Q-table heuristics and count their applications."""
        heuristic_counts = {}

        if not self.offline_learner or not hasattr(self.offline_learner, "Q_table"):
            return heuristic_counts

        for i, hypothetical_step in enumerate(
            self.hypothetical_states[:20]
        ):  # Sample from first 20 states
            state_repr = hypothetical_step.state
            action = hypothetical_step.action

            # Convert state representation to a tuple if it's in string form
            state_key = self._parse_state_key(state_repr)

            if (state_key, action) in self.offline_learner.Q_table:
                q_value = self.offline_learner.Q_table[(state_key, action)]

                # Infer which heuristics might have been applied based on Q-value
                if q_value < 0:
                    if "undeclared_tool" in action.lower() or not any(
                        tool.name in action for tool in self.agent_tools
                    ):
                        heuristic_counts["undeclared_tool_penalty"] = (
                            heuristic_counts.get("undeclared_tool_penalty", 0) + 1
                        )
                    if "unknown" in action.lower():
                        heuristic_counts["ambiguous_action_penalty"] = (
                            heuristic_counts.get("ambiguous_action_penalty", 0) + 1
                        )
                elif q_value > 0:
                    if isinstance(state_key, tuple) and len(state_key) >= 2:
                        if (state_key[1] == "None" or not state_key[1]) and any(
                            x in str(state_key[0])
                            for x in ["Information", "Initial", "Query"]
                        ):
                            heuristic_counts["good_first_step_reward"] = (
                                heuristic_counts.get("good_first_step_reward", 0) + 1
                            )

        return heuristic_counts

    def _parse_state_key(self, state_repr: Any) -> Any:
        """Parse state representation to appropriate format."""
        state_key = None
        try:
            import ast

            if (
                isinstance(state_repr, str)
                and state_repr.strip().startswith("(")
                and state_repr.strip().endswith(")")
            ):
                state_key = ast.literal_eval(state_repr)
            else:
                state_key = state_repr
        except ValueError:
            state_key = state_repr
        return state_key

    def _create_prompt_analysis_summary(self) -> str:
        """Create a formatted summary of prompt analysis results."""
        weaknesses = self.prompt_analysis.weaknesses
        suggestions = self.prompt_analysis.suggested_modifications
        strengths = self.prompt_analysis.strengths

        prompt_analysis_summary = ""
        if strengths:
            prompt_analysis_summary += "Strengths:\n"
            for strength in strengths[:3]:  # Limit to first 3 for brevity
                prompt_analysis_summary += f"- {strength}\n"

        if weaknesses:
            prompt_analysis_summary += "\nWeaknesses:\n"
            for weakness in weaknesses[:3]:  # Limit to first 3 for brevity
                prompt_analysis_summary += f"- {weakness}\n"

        if suggestions:
            prompt_analysis_summary += "\nSuggested Modifications:\n"
            for suggestion in suggestions[:3]:  # Limit to first 3 for brevity
                prompt_analysis_summary += f"- {suggestion}\n"

        return prompt_analysis_summary

    def _generate_prompt_with_llm(
        self,
        num_parsed_steps: int,
        first_few_subtasks: List[str],
        num_hypothetical_states: int,
        q_table_size: int,
        key_heuristics: List[str],
        prompt_analysis_summary: str,
    ) -> str:
        """Generate the optimized prompt using LLM."""
        # Create LLM report generation prompt
        template = """
        You are Adaptiq, an AI assistant specializing in optimizing AI agent prompts through reinforcement learning analysis. You have completed a comprehensive 'Pre-Run Analysis Phase' for a developer's agent and must now generate an improved, optimized version of their agent's prompt.

        ## Analysis Summary:

        ### Agent Configuration Context:
        - **Agent ID**: {agent_id}
        - **Current Prompt**: {old_prompt}

        ### Available Agent Tools:
        {agent_tools}

        ### Analysis Results:
        - **Prompt Structure Analysis**: {num_parsed_steps} intended steps identified
        - **Key Sub-Tasks Extracted**: {first_few_subtasks}
        - **Hypothetical State Generation**: {num_hypothetical_states} state-action pairs created
        - **Representative States**: {hypothetical_states_sample}
        - **Q-Table Initialization**: {q_table_size} entries with heuristics: {key_heuristics}
        - **Automated Prompt Analysis**: {prompt_analysis_summary}

        ## Optimization Guidelines:

        Your optimized prompt must:

        1. **Preserve All Placeholders**: Any placeholders (variables in curly braces, template syntax, or dynamic content markers) found in the original prompt are agent inputs that process runtime data. These MUST be preserved exactly as they appear - they represent essential dynamic content that the agent needs to function.

        2. **Structural Optimization**: 
        - Reflect the intended workflow from the parsed steps and sub-tasks
        - Create clear, logical instruction sequences
        - Eliminate redundancy and ambiguity

        3. **Analysis-Driven Improvements**:
        - Incorporate specific recommendations from the prompt analysis phase
        - Address weaknesses identified in the hypothetical state generation
        - Leverage insights from Q-table heuristics

        4. **Tool Integration**:
        - Ensure seamless integration with available tools
        - Provide clear guidance on when and how to use each tool
        - Align tool usage with the hypothetical states and decision patterns

        5. **Clarity and Efficiency**:
        - Use precise, actionable language
        - Remove unnecessary verbosity
        - Structure information hierarchically for better comprehension
        - Focus on decision-making support

        6. **Quality Assurance**:
        - Ensure the prompt supports robust error handling
        - Include guidance for edge cases identified in analysis
        - Maintain consistency with the agent's intended behavior patterns

        ## Output Requirements:

        Return ONLY the optimized prompt. Do not include:
        - Explanations of changes made
        - Reasoning behind modifications  
        - Additional commentary or analysis
        - Formatting beyond the prompt itself
        """

        # Create the prompt
        prompt = ChatPromptTemplate.from_template(template)

        # Format the message with our data
        formatted_prompt = prompt.format(
            agent_id=self.agent_id,
            old_prompt=self.old_prompt,
            agent_tools=self.tools_string,
            num_parsed_steps=num_parsed_steps,
            first_few_subtasks=", ".join(first_few_subtasks),
            num_hypothetical_states=num_hypothetical_states,
            hypothetical_states_sample=[
                item.state for item in self.hypothetical_states
            ],
            q_table_size=q_table_size,
            key_heuristics=(
                ", ".join(key_heuristics) if key_heuristics else "None identified"
            ),
            prompt_analysis_summary=prompt_analysis_summary,
        )

        # Generate the report
        response = self.llm.invoke(formatted_prompt)
        return response.content

    def _save_report(self, report: str) -> None:
        """Save the generated report to file."""
        if not self.output_path:
            return

        try:
            os.makedirs(self.output_path, exist_ok=True)
            report_path = os.path.join(
                self.output_path, "adaptiq_analysis_pre_run_report.txt"
            )

            with open(report_path, "w", encoding="utf-8") as f:
                f.write(report)
            self.logger.info("Report saved to %s", report_path)
        except (OSError, IOError) as e:
            self.logger.warning("Failed to save report: %s", str(e))
