from typing import Union, Literal, List
from base64 import b64decode
from .bound_methods import (
    MessageBoundMethods,
    FileBoundMethods,
    CallbackQueryBoundMethods,
    MessageSenderBoundMethods,
)
import pytdbot


class TlObject:
    """Base class for TL Objects"""

    def __getitem__(self, item):
        if item == "@type":
            return self.getType()

        return self.__dict__[item]

    def __setitem__(self, item, value):
        self.__dict__[item] = value

    def __bool__(self):
        return not isinstance(self, Error)

    @property
    def is_error(self):  # for backward compatibility
        return isinstance(self, Error)

    @property
    def limited_seconds(self):
        if self.is_error and self.code == 429:
            return pytdbot.utils.get_retry_after_time(self.message)
        else:
            return 0

    @classmethod
    def getType(cls):
        raise NotImplementedError

    @classmethod
    def getClass(cls):
        raise NotImplementedError

    def to_dict(self) -> dict:
        raise NotImplementedError

    @classmethod
    def from_dict(cls, data: dict):
        raise NotImplementedError


class AuthenticationCodeType:
    r"""Provides information about the method by which an authentication code is delivered to the user"""

    pass


class EmailAddressAuthentication:
    r"""Contains authentication data for an email address"""

    pass


class EmailAddressResetState:
    r"""Describes reset state of an email address"""

    pass


class AuthorizationState:
    r"""Represents the current authorization state of the TDLib client"""

    pass


class FirebaseDeviceVerificationParameters:
    r"""Describes parameters to be used for device verification"""

    pass


class InputFile:
    r"""Points to a file"""

    pass


class ThumbnailFormat:
    r"""Describes format of a thumbnail"""

    pass


class MaskPoint:
    r"""Part of the face, relative to which a mask is placed"""

    pass


class StickerFormat:
    r"""Describes format of a sticker"""

    pass


class StickerType:
    r"""Describes type of sticker"""

    pass


class StickerFullType:
    r"""Contains full information about sticker type"""

    pass


class PollType:
    r"""Describes the type of poll"""

    pass


class ProfileTab:
    r"""Describes a tab shown in a user or a chat profile"""

    pass


class UserType:
    r"""Represents the type of user\. The following types are possible: regular users, deleted users and bots"""

    pass


class BusinessAwayMessageSchedule:
    r"""Describes conditions for sending of away messages by a Telegram Business account"""

    pass


class ChatPhotoStickerType:
    r"""Describes type of sticker, which was used to create a chat photo"""

    pass


class InputChatPhoto:
    r"""Describes a photo to be set as a user profile or chat photo"""

    pass


class GiftResalePrice:
    r"""Describes price of a resold gift"""

    pass


class SuggestedPostPrice:
    r"""Describes price of a suggested post"""

    pass


class SuggestedPostState:
    r"""Describes state of a suggested post"""

    pass


class SuggestedPostRefundReason:
    r"""Describes reason for refund of the payment for a suggested post"""

    pass


class StarSubscriptionType:
    r"""Describes type of subscription paid in Telegram Stars"""

    pass


class AffiliateType:
    r"""Describes type of affiliate for an affiliate program"""

    pass


class AffiliateProgramSortOrder:
    r"""Describes the order of the found affiliate programs"""

    pass


class CanSendGiftResult:
    r"""Describes whether a gift can be sent now by the current user"""

    pass


class UpgradedGiftOrigin:
    r"""Describes origin from which the upgraded gift was obtained"""

    pass


class UpgradedGiftAttributeId:
    r"""Contains identifier of an upgraded gift attribute to search for"""

    pass


class GiftForResaleOrder:
    r"""Describes order in which upgraded gifts for resale will be sorted"""

    pass


class GiftResaleResult:
    r"""Describes result of sending a resold gift"""

    pass


class SentGift:
    r"""Represents content of a gift received by a user or a channel chat"""

    pass


class TransactionDirection:
    r"""Describes direction of transactions in a transaction list"""

    pass


class StarTransactionType:
    r"""Describes type of transaction with Telegram Stars"""

    pass


class TonTransactionType:
    r"""Describes type of transaction with Toncoins"""

    pass


class GiveawayParticipantStatus:
    r"""Contains information about status of a user in a giveaway"""

    pass


class GiveawayInfo:
    r"""Contains information about a giveaway"""

    pass


class GiveawayPrize:
    r"""Contains information about a giveaway prize"""

    pass


class EmojiStatusType:
    r"""Describes type of emoji status"""

    pass


class ChatMemberStatus:
    r"""Provides information about the status of a member in a chat"""

    pass


class ChatMembersFilter:
    r"""Specifies the kind of chat members to return in searchChatMembers"""

    pass


class SupergroupMembersFilter:
    r"""Specifies the kind of chat members to return in getSupergroupMembers"""

    pass


class InviteLinkChatType:
    r"""Describes the type of chat to which points an invite link"""

    pass


class SecretChatState:
    r"""Describes the current secret chat state"""

    pass


class MessageSender(MessageSenderBoundMethods):
    r"""Contains information about the sender of a message"""

    pass


class MessageReadDate:
    r"""Describes read date of a recent outgoing message in a private chat"""

    pass


class MessageOrigin:
    r"""Contains information about the origin of a message"""

    pass


class ReactionType:
    r"""Describes type of message reaction"""

    pass


class PaidReactionType:
    r"""Describes type of paid message reaction"""

    pass


class MessageTopic:
    r"""Describes a topic of messages in a chat"""

    pass


class MessageEffectType:
    r"""Describes type of emoji effect"""

    pass


class MessageSendingState:
    r"""Contains information about the sending state of the message"""

    pass


class MessageReplyTo:
    r"""Contains information about the message or the story a message is replying to"""

    pass


class InputMessageReplyTo:
    r"""Contains information about the message or the story to be replied"""

    pass


class MessageSource:
    r"""Describes source of a message"""

    pass


class ReportSponsoredResult:
    r"""Describes result of sponsored message or chat report"""

    pass


class NotificationSettingsScope:
    r"""Describes the types of chats to which notification settings are relevant"""

    pass


class ReactionNotificationSource:
    r"""Describes sources of reactions for which notifications will be shown"""

    pass


class ChatType:
    r"""Describes the type of chat"""

    pass


class ChatList:
    r"""Describes a list of chats"""

    pass


class ChatSource:
    r"""Describes a reason why an external chat is shown in a chat list"""

    pass


class ChatAvailableReactions:
    r"""Describes reactions available in the chat"""

    pass


class PublicChatType:
    r"""Describes type of public chat"""

    pass


class ChatActionBar:
    r"""Describes actions which must be possible to do through a chat action bar"""

    pass


class KeyboardButtonType:
    r"""Describes a keyboard button type"""

    pass


class InlineKeyboardButtonType:
    r"""Describes the type of inline keyboard button"""

    pass


class ReplyMarkup:
    r"""Contains a description of a custom keyboard and actions that can be done with it to quickly reply to bots"""

    pass


class LoginUrlInfo:
    r"""Contains information about an inline button of type inlineKeyboardButtonTypeLoginUrl"""

    pass


class WebAppOpenMode:
    r"""Describes mode in which a Web App is opened"""

    pass


class SavedMessagesTopicType:
    r"""Describes type of Saved Messages topic"""

    pass


class BuiltInTheme:
    r"""Describes a built\-in theme of an official app"""

    pass


class RichText:
    r"""Describes a formatted text object"""

    pass


class PageBlockHorizontalAlignment:
    r"""Describes a horizontal alignment of a table cell content"""

    pass


class PageBlockVerticalAlignment:
    r"""Describes a Vertical alignment of a table cell content"""

    pass


class PageBlock:
    r"""Describes a block of an instant view for a web page"""

    pass


class LinkPreviewAlbumMedia:
    r"""Describes a media from a link preview album"""

    pass


class LinkPreviewType:
    r"""Describes type of link preview"""

    pass


class CollectibleItemType:
    r"""Describes a collectible item that can be purchased at https://fragment\.com"""

    pass


class InputCredentials:
    r"""Contains information about the payment method chosen by the user"""

    pass


class PaymentProvider:
    r"""Contains information about a payment provider"""

    pass


class PaymentFormType:
    r"""Describes type of payment form"""

    pass


class PaymentReceiptType:
    r"""Describes type of successful payment"""

    pass


class InputInvoice:
    r"""Describes an invoice to process"""

    pass


class PaidMedia:
    r"""Describes a paid media"""

    pass


class PassportElementType:
    r"""Contains the type of Telegram Passport element"""

    pass


class PassportElement:
    r"""Contains information about a Telegram Passport element"""

    pass


class InputPassportElement:
    r"""Contains information about a Telegram Passport element to be saved"""

    pass


class PassportElementErrorSource:
    r"""Contains the description of an error in a Telegram Passport element"""

    pass


class InputPassportElementErrorSource:
    r"""Contains the description of an error in a Telegram Passport element; for bots only"""

    pass


class MessageContent:
    r"""Contains the content of a message"""

    pass


class TextEntityType:
    r"""Represents a part of the text which must be formatted differently"""

    pass


class InputPaidMediaType:
    r"""Describes type of paid media to sent"""

    pass


class MessageSchedulingState:
    r"""Contains information about the time when a scheduled message will be sent"""

    pass


class MessageSelfDestructType:
    r"""Describes when a message will be self\-destructed"""

    pass


class InputMessageContent:
    r"""The content of a message to send"""

    pass


class SearchMessagesFilter:
    r"""Represents a filter for message search results"""

    pass


class SearchMessagesChatTypeFilter:
    r"""Represents a filter for type of the chats in which to search messages"""

    pass


class ChatAction:
    r"""Describes the different types of activity in a chat"""

    pass


class UserStatus:
    r"""Describes the last time the user was online"""

    pass


class EmojiCategorySource:
    r"""Describes source of stickers for an emoji category"""

    pass


class EmojiCategoryType:
    r"""Describes type of emoji category"""

    pass


class StoryAreaType:
    r"""Describes type of clickable area on a story media"""

    pass


class InputStoryAreaType:
    r"""Describes type of clickable area on a story media to be added"""

    pass


class StoryContent:
    r"""Contains the content of a story"""

    pass


class InputStoryContent:
    r"""The content of a story to post"""

    pass


class StoryList:
    r"""Describes a list of stories"""

    pass


class StoryOrigin:
    r"""Contains information about the origin of a story that was reposted"""

    pass


class StoryInteractionType:
    r"""Describes type of interaction with a story"""

    pass


class PublicForward:
    r"""Describes a public forward or repost of a story"""

    pass


class ChatBoostSource:
    r"""Describes source of a chat boost"""

    pass


class ResendCodeReason:
    r"""Describes the reason why a code needs to be re\-sent"""

    pass


class CallDiscardReason:
    r"""Describes the reason why a call was discarded"""

    pass


class CallServerType:
    r"""Describes the type of call server"""

    pass


class CallState:
    r"""Describes the current call state"""

    pass


class GroupCallVideoQuality:
    r"""Describes the quality of a group call video"""

    pass


class InviteGroupCallParticipantResult:
    r"""Describes result of group call participant invitation"""

    pass


class GroupCallDataChannel:
    r"""Describes data channel for a group call"""

    pass


class InputGroupCall:
    r"""Describes a non\-joined group call that isn't bound to a chat"""

    pass


class CallProblem:
    r"""Describes the exact type of problem with a call"""

    pass


class FirebaseAuthenticationSettings:
    r"""Contains settings for Firebase Authentication in the official applications"""

    pass


class ReactionUnavailabilityReason:
    r"""Describes why the current user can't add reactions to the message, despite some other users can"""

    pass


class DiceStickers:
    r"""Contains animated stickers which must be used for dice animation rendering"""

    pass


class SpeechRecognitionResult:
    r"""Describes result of speech recognition in a voice note"""

    pass


class BotWriteAccessAllowReason:
    r"""Describes a reason why a bot was allowed to write messages to the current user"""

    pass


class TargetChat:
    r"""Describes the target chat to be opened"""

    pass


class InputInlineQueryResult:
    r"""Represents a single result of an inline query; for bots only"""

    pass


class InlineQueryResult:
    r"""Represents a single result of an inline query"""

    pass


class InlineQueryResultsButtonType:
    r"""Represents type of button in results of inline query"""

    pass


class CallbackQueryPayload:
    r"""Represents a payload of a callback query"""

    pass


class ChatEventAction:
    r"""Represents a chat event"""

    pass


class LanguagePackStringValue:
    r"""Represents the value of a string in a language pack"""

    pass


class PremiumLimitType:
    r"""Describes type of limit, increased for Premium users"""

    pass


class PremiumFeature:
    r"""Describes a feature available to Premium users"""

    pass


class BusinessFeature:
    r"""Describes a feature available to Business user accounts"""

    pass


class PremiumStoryFeature:
    r"""Describes a story feature available to Premium users"""

    pass


class PremiumSource:
    r"""Describes a source from which the Premium features screen is opened"""

    pass


class StorePaymentPurpose:
    r"""Describes a purpose of an in\-store payment"""

    pass


class StoreTransaction:
    r"""Describes an in\-store transaction"""

    pass


class TelegramPaymentPurpose:
    r"""Describes a purpose of a payment toward Telegram"""

    pass


class DeviceToken:
    r"""Represents a data needed to subscribe for push notifications through registerDevice method\. To use specific push notification service, the correct application platform must be specified and a valid server authentication data must be uploaded at https://my\.telegram\.org"""

    pass


class BackgroundFill:
    r"""Describes a fill of a background"""

    pass


class BackgroundType:
    r"""Describes the type of background"""

    pass


class InputBackground:
    r"""Contains information about background to set"""

    pass


class ChatTheme:
    r"""Describes a chat theme"""

    pass


class InputChatTheme:
    r"""Describes a chat theme to set"""

    pass


class CanPostStoryResult:
    r"""Represents result of checking whether the current user can post a story on behalf of the specific chat"""

    pass


class CanTransferOwnershipResult:
    r"""Represents result of checking whether the current session can be used to transfer a chat ownership to another user"""

    pass


class CheckChatUsernameResult:
    r"""Represents result of checking whether a username can be set for a chat"""

    pass


class CheckStickerSetNameResult:
    r"""Represents result of checking whether a name can be used for a new sticker set"""

    pass


class ResetPasswordResult:
    r"""Represents result of 2\-step verification password reset"""

    pass


class MessageFileType:
    r"""Contains information about a file with messages exported from another app"""

    pass


class PushMessageContent:
    r"""Contains content of a push message notification"""

    pass


class NotificationType:
    r"""Contains detailed information about a notification"""

    pass


class NotificationGroupType:
    r"""Describes the type of notifications in a notification group"""

    pass


class OptionValue:
    r"""Represents the value of an option"""

    pass


class JsonValue:
    r"""Represents a JSON value"""

    pass


class StoryPrivacySettings:
    r"""Describes privacy settings of a story"""

    pass


class UserPrivacySettingRule:
    r"""Represents a single rule for managing user privacy settings"""

    pass


class UserPrivacySetting:
    r"""Describes available user privacy settings"""

    pass


class CanSendMessageToUserResult:
    r"""Describes result of canSendMessageToUser"""

    pass


class SessionType:
    r"""Represents the type of session"""

    pass


class ReportReason:
    r"""Describes the reason why a chat is reported"""

    pass


class ReportChatResult:
    r"""Describes result of chat report"""

    pass


class ReportStoryResult:
    r"""Describes result of story report"""

    pass


class InternalLinkType:
    r"""Describes an internal https://t\.me or tg: link, which must be processed by the application in a special way"""

    pass


class BlockList:
    r"""Describes type of block list"""

    pass


class FileType:
    r"""Represents the type of file"""

    pass


class NetworkType:
    r"""Represents the type of network"""

    pass


class NetworkStatisticsEntry:
    r"""Contains statistics about network usage"""

    pass


class AutosaveSettingsScope:
    r"""Describes scope of autosave settings"""

    pass


class ConnectionState:
    r"""Describes the current state of the connection to Telegram servers"""

    pass


class TopChatCategory:
    r"""Represents the categories of chats for which a list of frequently used chats can be retrieved"""

    pass


class TMeUrlType:
    r"""Describes the type of URL linking to an internal Telegram entity"""

    pass


class SuggestedAction:
    r"""Describes an action suggested to the current user"""

    pass


class TextParseMode:
    r"""Describes the way the text needs to be parsed for text entities"""

    pass


class ProxyType:
    r"""Describes the type of proxy server"""

    pass


class StatisticalGraph:
    r"""Describes a statistical graph"""

    pass


class ChatStatisticsObjectType:
    r"""Describes type of object, for which statistics are provided"""

    pass


class ChatStatistics:
    r"""Contains a detailed statistics about a chat"""

    pass


class RevenueWithdrawalState:
    r"""Describes state of a revenue withdrawal"""

    pass


class ChatRevenueTransactionType:
    r"""Describes type of transaction for revenue earned from sponsored messages in a chat"""

    pass


class VectorPathCommand:
    r"""Represents a vector path command"""

    pass


class BotCommandScope:
    r"""Represents the scope to which bot commands are relevant"""

    pass


class PhoneNumberCodeType:
    r"""Describes type of the request for which a code is sent to a phone number"""

    pass


class Update:
    r"""Contains notifications about data changes"""

    pass


class LogStream:
    r"""Describes a stream to which TDLib internal log is written"""

    pass


class Error(TlObject):
    r"""An object of this type can be returned on every function call, in case of an error

    Parameters:
        code (:class:`int`):
            Error code; subject to future changes\. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user

        message (:class:`str`):
            Error message; subject to future changes

    """

    def __init__(self, code: int = 0, message: str = "") -> None:
        self.code: int = int(code)
        r"""Error code; subject to future changes\. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user"""
        self.message: Union[str, None] = message
        r"""Error message; subject to future changes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["error"]:
        return "error"

    @classmethod
    def getClass(self) -> Literal["Error"]:
        return "Error"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "code": self.code, "message": self.message}

    @classmethod
    def from_dict(cls, data: dict) -> Union["Error", None]:
        if data:
            data_class = cls()
            data_class.code = int(data.get("code", 0))
            data_class.message = data.get("message", "")

        return data_class


class Ok(TlObject):
    r"""An object of this type is returned on a successful function call for certain functions"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["ok"]:
        return "ok"

    @classmethod
    def getClass(self) -> Literal["Ok"]:
        return "Ok"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["Ok", None]:
        if data:
            data_class = cls()

        return data_class


class AuthenticationCodeTypeTelegramMessage(TlObject, AuthenticationCodeType):
    r"""A digit\-only authentication code is delivered via a private Telegram message, which can be viewed from another active session

    Parameters:
        length (:class:`int`):
            Length of the code

    """

    def __init__(self, length: int = 0) -> None:
        self.length: int = int(length)
        r"""Length of the code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeTelegramMessage"]:
        return "authenticationCodeTypeTelegramMessage"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "length": self.length}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AuthenticationCodeTypeTelegramMessage", None]:
        if data:
            data_class = cls()
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeTypeSms(TlObject, AuthenticationCodeType):
    r"""A digit\-only authentication code is delivered via an SMS message to the specified phone number; non\-official applications may not receive this type of code

    Parameters:
        length (:class:`int`):
            Length of the code

    """

    def __init__(self, length: int = 0) -> None:
        self.length: int = int(length)
        r"""Length of the code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeSms"]:
        return "authenticationCodeTypeSms"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "length": self.length}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeSms", None]:
        if data:
            data_class = cls()
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeTypeSmsWord(TlObject, AuthenticationCodeType):
    r"""An authentication code is a word delivered via an SMS message to the specified phone number; non\-official applications may not receive this type of code

    Parameters:
        first_letter (:class:`str`):
            The first letters of the word if known

    """

    def __init__(self, first_letter: str = "") -> None:
        self.first_letter: Union[str, None] = first_letter
        r"""The first letters of the word if known"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeSmsWord"]:
        return "authenticationCodeTypeSmsWord"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "first_letter": self.first_letter}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeSmsWord", None]:
        if data:
            data_class = cls()
            data_class.first_letter = data.get("first_letter", "")

        return data_class


class AuthenticationCodeTypeSmsPhrase(TlObject, AuthenticationCodeType):
    r"""An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non\-official applications may not receive this type of code

    Parameters:
        first_word (:class:`str`):
            The first word of the phrase if known

    """

    def __init__(self, first_word: str = "") -> None:
        self.first_word: Union[str, None] = first_word
        r"""The first word of the phrase if known"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeSmsPhrase"]:
        return "authenticationCodeTypeSmsPhrase"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "first_word": self.first_word}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeSmsPhrase", None]:
        if data:
            data_class = cls()
            data_class.first_word = data.get("first_word", "")

        return data_class


class AuthenticationCodeTypeCall(TlObject, AuthenticationCodeType):
    r"""A digit\-only authentication code is delivered via a phone call to the specified phone number

    Parameters:
        length (:class:`int`):
            Length of the code

    """

    def __init__(self, length: int = 0) -> None:
        self.length: int = int(length)
        r"""Length of the code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeCall"]:
        return "authenticationCodeTypeCall"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "length": self.length}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeCall", None]:
        if data:
            data_class = cls()
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeTypeFlashCall(TlObject, AuthenticationCodeType):
    r"""An authentication code is delivered by an immediately canceled call to the specified phone number\. The phone number that calls is the code that must be entered automatically

    Parameters:
        pattern (:class:`str`):
            Pattern of the phone number from which the call will be made

    """

    def __init__(self, pattern: str = "") -> None:
        self.pattern: Union[str, None] = pattern
        r"""Pattern of the phone number from which the call will be made"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeFlashCall"]:
        return "authenticationCodeTypeFlashCall"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "pattern": self.pattern}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeFlashCall", None]:
        if data:
            data_class = cls()
            data_class.pattern = data.get("pattern", "")

        return data_class


class AuthenticationCodeTypeMissedCall(TlObject, AuthenticationCodeType):
    r"""An authentication code is delivered by an immediately canceled call to the specified phone number\. The last digits of the phone number that calls are the code that must be entered manually by the user

    Parameters:
        phone_number_prefix (:class:`str`):
            Prefix of the phone number from which the call will be made

        length (:class:`int`):
            Number of digits in the code, excluding the prefix

    """

    def __init__(self, phone_number_prefix: str = "", length: int = 0) -> None:
        self.phone_number_prefix: Union[str, None] = phone_number_prefix
        r"""Prefix of the phone number from which the call will be made"""
        self.length: int = int(length)
        r"""Number of digits in the code, excluding the prefix"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeMissedCall"]:
        return "authenticationCodeTypeMissedCall"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "phone_number_prefix": self.phone_number_prefix,
            "length": self.length,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeMissedCall", None]:
        if data:
            data_class = cls()
            data_class.phone_number_prefix = data.get("phone_number_prefix", "")
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeTypeFragment(TlObject, AuthenticationCodeType):
    r"""A digit\-only authentication code is delivered to https://fragment\.com\. The user must be logged in there via a wallet owning the phone number's NFT

    Parameters:
        url (:class:`str`):
            URL to open to receive the code

        length (:class:`int`):
            Length of the code

    """

    def __init__(self, url: str = "", length: int = 0) -> None:
        self.url: Union[str, None] = url
        r"""URL to open to receive the code"""
        self.length: int = int(length)
        r"""Length of the code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeFragment"]:
        return "authenticationCodeTypeFragment"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "url": self.url, "length": self.length}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeFragment", None]:
        if data:
            data_class = cls()
            data_class.url = data.get("url", "")
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeTypeFirebaseAndroid(TlObject, AuthenticationCodeType):
    r"""A digit\-only authentication code is delivered via Firebase Authentication to the official Android application

    Parameters:
        device_verification_parameters (:class:`~pytdbot.types.FirebaseDeviceVerificationParameters`):
            Parameters to be used for device verification

        length (:class:`int`):
            Length of the code

    """

    def __init__(
        self,
        device_verification_parameters: FirebaseDeviceVerificationParameters = None,
        length: int = 0,
    ) -> None:
        self.device_verification_parameters: Union[
            FirebaseDeviceVerificationParametersSafetyNet,
            FirebaseDeviceVerificationParametersPlayIntegrity,
            None,
        ] = device_verification_parameters
        r"""Parameters to be used for device verification"""
        self.length: int = int(length)
        r"""Length of the code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeFirebaseAndroid"]:
        return "authenticationCodeTypeFirebaseAndroid"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "device_verification_parameters": self.device_verification_parameters,
            "length": self.length,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AuthenticationCodeTypeFirebaseAndroid", None]:
        if data:
            data_class = cls()
            data_class.device_verification_parameters = data.get(
                "device_verification_parameters", None
            )
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeTypeFirebaseIos(TlObject, AuthenticationCodeType):
    r"""A digit\-only authentication code is delivered via Firebase Authentication to the official iOS application

    Parameters:
        receipt (:class:`str`):
            Receipt of successful application token validation to compare with receipt from push notification

        push_timeout (:class:`int`):
            Time after the next authentication method is expected to be used if verification push notification isn't received, in seconds

        length (:class:`int`):
            Length of the code

    """

    def __init__(
        self, receipt: str = "", push_timeout: int = 0, length: int = 0
    ) -> None:
        self.receipt: Union[str, None] = receipt
        r"""Receipt of successful application token validation to compare with receipt from push notification"""
        self.push_timeout: int = int(push_timeout)
        r"""Time after the next authentication method is expected to be used if verification push notification isn't received, in seconds"""
        self.length: int = int(length)
        r"""Length of the code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeTypeFirebaseIos"]:
        return "authenticationCodeTypeFirebaseIos"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeType"]:
        return "AuthenticationCodeType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "receipt": self.receipt,
            "push_timeout": self.push_timeout,
            "length": self.length,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeTypeFirebaseIos", None]:
        if data:
            data_class = cls()
            data_class.receipt = data.get("receipt", "")
            data_class.push_timeout = int(data.get("push_timeout", 0))
            data_class.length = int(data.get("length", 0))

        return data_class


class AuthenticationCodeInfo(TlObject):
    r"""Information about the authentication code that was sent

    Parameters:
        phone_number (:class:`str`):
            A phone number that is being authenticated

        type (:class:`~pytdbot.types.AuthenticationCodeType`):
            The way the code was sent to the user

        next_type (:class:`~pytdbot.types.AuthenticationCodeType`):
            The way the next code will be sent to the user; may be null

        timeout (:class:`int`):
            Timeout before the code can be re\-sent, in seconds

    """

    def __init__(
        self,
        phone_number: str = "",
        type: AuthenticationCodeType = None,
        next_type: AuthenticationCodeType = None,
        timeout: int = 0,
    ) -> None:
        self.phone_number: Union[str, None] = phone_number
        r"""A phone number that is being authenticated"""
        self.type: Union[
            AuthenticationCodeTypeTelegramMessage,
            AuthenticationCodeTypeSms,
            AuthenticationCodeTypeSmsWord,
            AuthenticationCodeTypeSmsPhrase,
            AuthenticationCodeTypeCall,
            AuthenticationCodeTypeFlashCall,
            AuthenticationCodeTypeMissedCall,
            AuthenticationCodeTypeFragment,
            AuthenticationCodeTypeFirebaseAndroid,
            AuthenticationCodeTypeFirebaseIos,
            None,
        ] = type
        r"""The way the code was sent to the user"""
        self.next_type: Union[
            AuthenticationCodeTypeTelegramMessage,
            AuthenticationCodeTypeSms,
            AuthenticationCodeTypeSmsWord,
            AuthenticationCodeTypeSmsPhrase,
            AuthenticationCodeTypeCall,
            AuthenticationCodeTypeFlashCall,
            AuthenticationCodeTypeMissedCall,
            AuthenticationCodeTypeFragment,
            AuthenticationCodeTypeFirebaseAndroid,
            AuthenticationCodeTypeFirebaseIos,
            None,
        ] = next_type
        r"""The way the next code will be sent to the user; may be null"""
        self.timeout: int = int(timeout)
        r"""Timeout before the code can be re\-sent, in seconds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authenticationCodeInfo"]:
        return "authenticationCodeInfo"

    @classmethod
    def getClass(self) -> Literal["AuthenticationCodeInfo"]:
        return "AuthenticationCodeInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "phone_number": self.phone_number,
            "type": self.type,
            "next_type": self.next_type,
            "timeout": self.timeout,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthenticationCodeInfo", None]:
        if data:
            data_class = cls()
            data_class.phone_number = data.get("phone_number", "")
            data_class.type = data.get("type", None)
            data_class.next_type = data.get("next_type", None)
            data_class.timeout = int(data.get("timeout", 0))

        return data_class


class EmailAddressAuthenticationCodeInfo(TlObject):
    r"""Information about the email address authentication code that was sent

    Parameters:
        email_address_pattern (:class:`str`):
            Pattern of the email address to which an authentication code was sent

        length (:class:`int`):
            Length of the code; 0 if unknown

    """

    def __init__(self, email_address_pattern: str = "", length: int = 0) -> None:
        self.email_address_pattern: Union[str, None] = email_address_pattern
        r"""Pattern of the email address to which an authentication code was sent"""
        self.length: int = int(length)
        r"""Length of the code; 0 if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emailAddressAuthenticationCodeInfo"]:
        return "emailAddressAuthenticationCodeInfo"

    @classmethod
    def getClass(self) -> Literal["EmailAddressAuthenticationCodeInfo"]:
        return "EmailAddressAuthenticationCodeInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "email_address_pattern": self.email_address_pattern,
            "length": self.length,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmailAddressAuthenticationCodeInfo", None]:
        if data:
            data_class = cls()
            data_class.email_address_pattern = data.get("email_address_pattern", "")
            data_class.length = int(data.get("length", 0))

        return data_class


class EmailAddressAuthenticationCode(TlObject, EmailAddressAuthentication):
    r"""An authentication code delivered to a user's email address

    Parameters:
        code (:class:`str`):
            The code

    """

    def __init__(self, code: str = "") -> None:
        self.code: Union[str, None] = code
        r"""The code"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emailAddressAuthenticationCode"]:
        return "emailAddressAuthenticationCode"

    @classmethod
    def getClass(self) -> Literal["EmailAddressAuthentication"]:
        return "EmailAddressAuthentication"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "code": self.code}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmailAddressAuthenticationCode", None]:
        if data:
            data_class = cls()
            data_class.code = data.get("code", "")

        return data_class


class EmailAddressAuthenticationAppleId(TlObject, EmailAddressAuthentication):
    r"""An authentication token received through Apple ID

    Parameters:
        token (:class:`str`):
            The token

    """

    def __init__(self, token: str = "") -> None:
        self.token: Union[str, None] = token
        r"""The token"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emailAddressAuthenticationAppleId"]:
        return "emailAddressAuthenticationAppleId"

    @classmethod
    def getClass(self) -> Literal["EmailAddressAuthentication"]:
        return "EmailAddressAuthentication"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "token": self.token}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmailAddressAuthenticationAppleId", None]:
        if data:
            data_class = cls()
            data_class.token = data.get("token", "")

        return data_class


class EmailAddressAuthenticationGoogleId(TlObject, EmailAddressAuthentication):
    r"""An authentication token received through Google ID

    Parameters:
        token (:class:`str`):
            The token

    """

    def __init__(self, token: str = "") -> None:
        self.token: Union[str, None] = token
        r"""The token"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emailAddressAuthenticationGoogleId"]:
        return "emailAddressAuthenticationGoogleId"

    @classmethod
    def getClass(self) -> Literal["EmailAddressAuthentication"]:
        return "EmailAddressAuthentication"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "token": self.token}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmailAddressAuthenticationGoogleId", None]:
        if data:
            data_class = cls()
            data_class.token = data.get("token", "")

        return data_class


class EmailAddressResetStateAvailable(TlObject, EmailAddressResetState):
    r"""Email address can be reset after the given period\. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number

    Parameters:
        wait_period (:class:`int`):
            Time required to wait before the email address can be reset; 0 if the user is subscribed to Telegram Premium

    """

    def __init__(self, wait_period: int = 0) -> None:
        self.wait_period: int = int(wait_period)
        r"""Time required to wait before the email address can be reset; 0 if the user is subscribed to Telegram Premium"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emailAddressResetStateAvailable"]:
        return "emailAddressResetStateAvailable"

    @classmethod
    def getClass(self) -> Literal["EmailAddressResetState"]:
        return "EmailAddressResetState"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "wait_period": self.wait_period}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmailAddressResetStateAvailable", None]:
        if data:
            data_class = cls()
            data_class.wait_period = int(data.get("wait_period", 0))

        return data_class


class EmailAddressResetStatePending(TlObject, EmailAddressResetState):
    r"""Email address reset has already been requested\. Call resetAuthenticationEmailAddress to check whether immediate reset is possible

    Parameters:
        reset_in (:class:`int`):
            Left time before the email address will be reset, in seconds\. updateAuthorizationState is not sent when this field changes

    """

    def __init__(self, reset_in: int = 0) -> None:
        self.reset_in: int = int(reset_in)
        r"""Left time before the email address will be reset, in seconds\. updateAuthorizationState is not sent when this field changes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emailAddressResetStatePending"]:
        return "emailAddressResetStatePending"

    @classmethod
    def getClass(self) -> Literal["EmailAddressResetState"]:
        return "EmailAddressResetState"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "reset_in": self.reset_in}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmailAddressResetStatePending", None]:
        if data:
            data_class = cls()
            data_class.reset_in = int(data.get("reset_in", 0))

        return data_class


class TextEntity(TlObject):
    r"""Represents a part of the text that needs to be formatted in some unusual way

    Parameters:
        offset (:class:`int`):
            Offset of the entity, in UTF\-16 code units

        length (:class:`int`):
            Length of the entity, in UTF\-16 code units

        type (:class:`~pytdbot.types.TextEntityType`):
            Type of the entity

    """

    def __init__(
        self, offset: int = 0, length: int = 0, type: TextEntityType = None
    ) -> None:
        self.offset: int = int(offset)
        r"""Offset of the entity, in UTF\-16 code units"""
        self.length: int = int(length)
        r"""Length of the entity, in UTF\-16 code units"""
        self.type: Union[
            TextEntityTypeMention,
            TextEntityTypeHashtag,
            TextEntityTypeCashtag,
            TextEntityTypeBotCommand,
            TextEntityTypeUrl,
            TextEntityTypeEmailAddress,
            TextEntityTypePhoneNumber,
            TextEntityTypeBankCardNumber,
            TextEntityTypeBold,
            TextEntityTypeItalic,
            TextEntityTypeUnderline,
            TextEntityTypeStrikethrough,
            TextEntityTypeSpoiler,
            TextEntityTypeCode,
            TextEntityTypePre,
            TextEntityTypePreCode,
            TextEntityTypeBlockQuote,
            TextEntityTypeExpandableBlockQuote,
            TextEntityTypeTextUrl,
            TextEntityTypeMentionName,
            TextEntityTypeCustomEmoji,
            TextEntityTypeMediaTimestamp,
            None,
        ] = type
        r"""Type of the entity"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["textEntity"]:
        return "textEntity"

    @classmethod
    def getClass(self) -> Literal["TextEntity"]:
        return "TextEntity"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "offset": self.offset,
            "length": self.length,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TextEntity", None]:
        if data:
            data_class = cls()
            data_class.offset = int(data.get("offset", 0))
            data_class.length = int(data.get("length", 0))
            data_class.type = data.get("type", None)

        return data_class


class TextEntities(TlObject):
    r"""Contains a list of text entities

    Parameters:
        entities (List[:class:`~pytdbot.types.TextEntity`]):
            List of text entities

    """

    def __init__(self, entities: List[TextEntity] = None) -> None:
        self.entities: List[TextEntity] = entities or []
        r"""List of text entities"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["textEntities"]:
        return "textEntities"

    @classmethod
    def getClass(self) -> Literal["TextEntities"]:
        return "TextEntities"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "entities": self.entities}

    @classmethod
    def from_dict(cls, data: dict) -> Union["TextEntities", None]:
        if data:
            data_class = cls()
            data_class.entities = data.get("entities", None)

        return data_class


class FormattedText(TlObject):
    r"""A text with some entities

    Parameters:
        text (:class:`str`):
            The text

        entities (List[:class:`~pytdbot.types.TextEntity`]):
            Entities contained in the text\. Entities can be nested, but must not mutually intersect with each other\. Pre, Code and PreCode entities can't contain other entities\. BlockQuote entities can't contain other BlockQuote entities\. Bold, Italic, Underline, Strikethrough, and Spoiler entities can contain and can be part of any other entities\. All other entities can't contain each other

    """

    def __init__(self, text: str = "", entities: List[TextEntity] = None) -> None:
        self.text: Union[str, None] = text
        r"""The text"""
        self.entities: List[TextEntity] = entities or []
        r"""Entities contained in the text\. Entities can be nested, but must not mutually intersect with each other\. Pre, Code and PreCode entities can't contain other entities\. BlockQuote entities can't contain other BlockQuote entities\. Bold, Italic, Underline, Strikethrough, and Spoiler entities can contain and can be part of any other entities\. All other entities can't contain each other"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["formattedText"]:
        return "formattedText"

    @classmethod
    def getClass(self) -> Literal["FormattedText"]:
        return "FormattedText"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "text": self.text, "entities": self.entities}

    @classmethod
    def from_dict(cls, data: dict) -> Union["FormattedText", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", "")
            data_class.entities = data.get("entities", None)

        return data_class


class TermsOfService(TlObject):
    r"""Contains Telegram terms of service

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            Text of the terms of service

        min_user_age (:class:`int`):
            The minimum age of a user to be able to accept the terms; 0 if age isn't restricted

        show_popup (:class:`bool`):
            True, if a blocking popup with terms of service must be shown to the user

    """

    def __init__(
        self,
        text: FormattedText = None,
        min_user_age: int = 0,
        show_popup: bool = False,
    ) -> None:
        self.text: Union[FormattedText, None] = text
        r"""Text of the terms of service"""
        self.min_user_age: int = int(min_user_age)
        r"""The minimum age of a user to be able to accept the terms; 0 if age isn't restricted"""
        self.show_popup: bool = bool(show_popup)
        r"""True, if a blocking popup with terms of service must be shown to the user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["termsOfService"]:
        return "termsOfService"

    @classmethod
    def getClass(self) -> Literal["TermsOfService"]:
        return "TermsOfService"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "text": self.text,
            "min_user_age": self.min_user_age,
            "show_popup": self.show_popup,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TermsOfService", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.min_user_age = int(data.get("min_user_age", 0))
            data_class.show_popup = data.get("show_popup", False)

        return data_class


class AuthorizationStateWaitTdlibParameters(TlObject, AuthorizationState):
    r"""Initialization parameters are needed\. Call setTdlibParameters to provide them"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitTdlibParameters"]:
        return "authorizationStateWaitTdlibParameters"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AuthorizationStateWaitTdlibParameters", None]:
        if data:
            data_class = cls()

        return data_class


class AuthorizationStateWaitPhoneNumber(TlObject, AuthorizationState):
    r"""TDLib needs the user's phone number to authorize\. Call setAuthenticationPhoneNumber to provide the phone number, or use requestQrCodeAuthentication or checkAuthenticationBotToken for other authentication options"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitPhoneNumber"]:
        return "authorizationStateWaitPhoneNumber"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateWaitPhoneNumber", None]:
        if data:
            data_class = cls()

        return data_class


class AuthorizationStateWaitPremiumPurchase(TlObject, AuthorizationState):
    r"""The user must buy Telegram Premium as an in\-store purchase to log in\. Call checkAuthenticationPremiumPurchase and then setAuthenticationPremiumPurchaseTransaction

    Parameters:
        store_product_id (:class:`str`):
            Identifier of the store product that must be bought

        support_email_address (:class:`str`):
            Email address to use for support if the user has issues with Telegram Premium purchase

        support_email_subject (:class:`str`):
            Subject for the email sent to the support email address

    """

    def __init__(
        self,
        store_product_id: str = "",
        support_email_address: str = "",
        support_email_subject: str = "",
    ) -> None:
        self.store_product_id: Union[str, None] = store_product_id
        r"""Identifier of the store product that must be bought"""
        self.support_email_address: Union[str, None] = support_email_address
        r"""Email address to use for support if the user has issues with Telegram Premium purchase"""
        self.support_email_subject: Union[str, None] = support_email_subject
        r"""Subject for the email sent to the support email address"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitPremiumPurchase"]:
        return "authorizationStateWaitPremiumPurchase"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "store_product_id": self.store_product_id,
            "support_email_address": self.support_email_address,
            "support_email_subject": self.support_email_subject,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AuthorizationStateWaitPremiumPurchase", None]:
        if data:
            data_class = cls()
            data_class.store_product_id = data.get("store_product_id", "")
            data_class.support_email_address = data.get("support_email_address", "")
            data_class.support_email_subject = data.get("support_email_subject", "")

        return data_class


class AuthorizationStateWaitEmailAddress(TlObject, AuthorizationState):
    r"""TDLib needs the user's email address to authorize\. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed

    Parameters:
        allow_apple_id (:class:`bool`):
            True, if authorization through Apple ID is allowed

        allow_google_id (:class:`bool`):
            True, if authorization through Google ID is allowed

    """

    def __init__(
        self, allow_apple_id: bool = False, allow_google_id: bool = False
    ) -> None:
        self.allow_apple_id: bool = bool(allow_apple_id)
        r"""True, if authorization through Apple ID is allowed"""
        self.allow_google_id: bool = bool(allow_google_id)
        r"""True, if authorization through Google ID is allowed"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitEmailAddress"]:
        return "authorizationStateWaitEmailAddress"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "allow_apple_id": self.allow_apple_id,
            "allow_google_id": self.allow_google_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateWaitEmailAddress", None]:
        if data:
            data_class = cls()
            data_class.allow_apple_id = data.get("allow_apple_id", False)
            data_class.allow_google_id = data.get("allow_google_id", False)

        return data_class


class AuthorizationStateWaitEmailCode(TlObject, AuthorizationState):
    r"""TDLib needs the user's authentication code sent to an email address to authorize\. Call checkAuthenticationEmailCode to provide the code

    Parameters:
        allow_apple_id (:class:`bool`):
            True, if authorization through Apple ID is allowed

        allow_google_id (:class:`bool`):
            True, if authorization through Google ID is allowed

        code_info (:class:`~pytdbot.types.EmailAddressAuthenticationCodeInfo`):
            Information about the sent authentication code

        email_address_reset_state (:class:`~pytdbot.types.EmailAddressResetState`):
            Reset state of the email address; may be null if the email address can't be reset

    """

    def __init__(
        self,
        allow_apple_id: bool = False,
        allow_google_id: bool = False,
        code_info: EmailAddressAuthenticationCodeInfo = None,
        email_address_reset_state: EmailAddressResetState = None,
    ) -> None:
        self.allow_apple_id: bool = bool(allow_apple_id)
        r"""True, if authorization through Apple ID is allowed"""
        self.allow_google_id: bool = bool(allow_google_id)
        r"""True, if authorization through Google ID is allowed"""
        self.code_info: Union[EmailAddressAuthenticationCodeInfo, None] = code_info
        r"""Information about the sent authentication code"""
        self.email_address_reset_state: Union[
            EmailAddressResetStateAvailable, EmailAddressResetStatePending, None
        ] = email_address_reset_state
        r"""Reset state of the email address; may be null if the email address can't be reset"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitEmailCode"]:
        return "authorizationStateWaitEmailCode"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "allow_apple_id": self.allow_apple_id,
            "allow_google_id": self.allow_google_id,
            "code_info": self.code_info,
            "email_address_reset_state": self.email_address_reset_state,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateWaitEmailCode", None]:
        if data:
            data_class = cls()
            data_class.allow_apple_id = data.get("allow_apple_id", False)
            data_class.allow_google_id = data.get("allow_google_id", False)
            data_class.code_info = data.get("code_info", None)
            data_class.email_address_reset_state = data.get(
                "email_address_reset_state", None
            )

        return data_class


class AuthorizationStateWaitCode(TlObject, AuthorizationState):
    r"""TDLib needs the user's authentication code to authorize\. Call checkAuthenticationCode to check the code

    Parameters:
        code_info (:class:`~pytdbot.types.AuthenticationCodeInfo`):
            Information about the authorization code that was sent

    """

    def __init__(self, code_info: AuthenticationCodeInfo = None) -> None:
        self.code_info: Union[AuthenticationCodeInfo, None] = code_info
        r"""Information about the authorization code that was sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitCode"]:
        return "authorizationStateWaitCode"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "code_info": self.code_info}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateWaitCode", None]:
        if data:
            data_class = cls()
            data_class.code_info = data.get("code_info", None)

        return data_class


class AuthorizationStateWaitOtherDeviceConfirmation(TlObject, AuthorizationState):
    r"""The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link

    Parameters:
        link (:class:`str`):
            A tg:// URL for the QR code\. The link will be updated frequently

    """

    def __init__(self, link: str = "") -> None:
        self.link: Union[str, None] = link
        r"""A tg:// URL for the QR code\. The link will be updated frequently"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitOtherDeviceConfirmation"]:
        return "authorizationStateWaitOtherDeviceConfirmation"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "link": self.link}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AuthorizationStateWaitOtherDeviceConfirmation", None]:
        if data:
            data_class = cls()
            data_class.link = data.get("link", "")

        return data_class


class AuthorizationStateWaitRegistration(TlObject, AuthorizationState):
    r"""The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration\. Call registerUser to accept the terms of service and provide the data

    Parameters:
        terms_of_service (:class:`~pytdbot.types.TermsOfService`):
            Telegram terms of service

    """

    def __init__(self, terms_of_service: TermsOfService = None) -> None:
        self.terms_of_service: Union[TermsOfService, None] = terms_of_service
        r"""Telegram terms of service"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitRegistration"]:
        return "authorizationStateWaitRegistration"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "terms_of_service": self.terms_of_service}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateWaitRegistration", None]:
        if data:
            data_class = cls()
            data_class.terms_of_service = data.get("terms_of_service", None)

        return data_class


class AuthorizationStateWaitPassword(TlObject, AuthorizationState):
    r"""The user has been authorized, but needs to enter a 2\-step verification password to start using the application\. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week

    Parameters:
        password_hint (:class:`str`):
            Hint for the password; may be empty

        has_recovery_email_address (:class:`bool`):
            True, if a recovery email address has been set up

        has_passport_data (:class:`bool`):
            True, if some Telegram Passport elements were saved

        recovery_email_address_pattern (:class:`str`):
            Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent

    """

    def __init__(
        self,
        password_hint: str = "",
        has_recovery_email_address: bool = False,
        has_passport_data: bool = False,
        recovery_email_address_pattern: str = "",
    ) -> None:
        self.password_hint: Union[str, None] = password_hint
        r"""Hint for the password; may be empty"""
        self.has_recovery_email_address: bool = bool(has_recovery_email_address)
        r"""True, if a recovery email address has been set up"""
        self.has_passport_data: bool = bool(has_passport_data)
        r"""True, if some Telegram Passport elements were saved"""
        self.recovery_email_address_pattern: Union[str, None] = (
            recovery_email_address_pattern
        )
        r"""Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateWaitPassword"]:
        return "authorizationStateWaitPassword"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "password_hint": self.password_hint,
            "has_recovery_email_address": self.has_recovery_email_address,
            "has_passport_data": self.has_passport_data,
            "recovery_email_address_pattern": self.recovery_email_address_pattern,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateWaitPassword", None]:
        if data:
            data_class = cls()
            data_class.password_hint = data.get("password_hint", "")
            data_class.has_recovery_email_address = data.get(
                "has_recovery_email_address", False
            )
            data_class.has_passport_data = data.get("has_passport_data", False)
            data_class.recovery_email_address_pattern = data.get(
                "recovery_email_address_pattern", ""
            )

        return data_class


class AuthorizationStateReady(TlObject, AuthorizationState):
    r"""The user has been successfully authorized\. TDLib is now ready to answer general requests"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateReady"]:
        return "authorizationStateReady"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateReady", None]:
        if data:
            data_class = cls()

        return data_class


class AuthorizationStateLoggingOut(TlObject, AuthorizationState):
    r"""The user is currently logging out"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateLoggingOut"]:
        return "authorizationStateLoggingOut"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateLoggingOut", None]:
        if data:
            data_class = cls()

        return data_class


class AuthorizationStateClosing(TlObject, AuthorizationState):
    r"""TDLib is closing, all subsequent queries will be answered with the error 500\. Note that closing TDLib can take a while\. All resources will be freed only after authorizationStateClosed has been received"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateClosing"]:
        return "authorizationStateClosing"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateClosing", None]:
        if data:
            data_class = cls()

        return data_class


class AuthorizationStateClosed(TlObject, AuthorizationState):
    r"""TDLib client is in its final state\. All databases are closed and all resources are released\. No other updates will be received after this\. All queries will be responded to with error code 500\. To continue working, one must create a new instance of the TDLib client"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["authorizationStateClosed"]:
        return "authorizationStateClosed"

    @classmethod
    def getClass(self) -> Literal["AuthorizationState"]:
        return "AuthorizationState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AuthorizationStateClosed", None]:
        if data:
            data_class = cls()

        return data_class


class FirebaseDeviceVerificationParametersSafetyNet(
    TlObject, FirebaseDeviceVerificationParameters
):
    r"""Device verification must be performed with the SafetyNet Attestation API

    Parameters:
        nonce (:class:`bytes`):
            Nonce to pass to the SafetyNet Attestation API

    """

    def __init__(self, nonce: bytes = b"") -> None:
        self.nonce: Union[bytes, None] = nonce
        r"""Nonce to pass to the SafetyNet Attestation API"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["firebaseDeviceVerificationParametersSafetyNet"]:
        return "firebaseDeviceVerificationParametersSafetyNet"

    @classmethod
    def getClass(self) -> Literal["FirebaseDeviceVerificationParameters"]:
        return "FirebaseDeviceVerificationParameters"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "nonce": self.nonce}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["FirebaseDeviceVerificationParametersSafetyNet", None]:
        if data:
            data_class = cls()
            data_class.nonce = b64decode(data.get("nonce", b""))

        return data_class


class FirebaseDeviceVerificationParametersPlayIntegrity(
    TlObject, FirebaseDeviceVerificationParameters
):
    r"""Device verification must be performed with the classic Play Integrity verification \(https://developer\.android\.com/google/play/integrity/classic\)

    Parameters:
        nonce (:class:`str`):
            Base64url\-encoded nonce to pass to the Play Integrity API

        cloud_project_number (:class:`int`):
            Cloud project number to pass to the Play Integrity API

    """

    def __init__(self, nonce: str = "", cloud_project_number: int = 0) -> None:
        self.nonce: Union[str, None] = nonce
        r"""Base64url\-encoded nonce to pass to the Play Integrity API"""
        self.cloud_project_number: int = int(cloud_project_number)
        r"""Cloud project number to pass to the Play Integrity API"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["firebaseDeviceVerificationParametersPlayIntegrity"]:
        return "firebaseDeviceVerificationParametersPlayIntegrity"

    @classmethod
    def getClass(self) -> Literal["FirebaseDeviceVerificationParameters"]:
        return "FirebaseDeviceVerificationParameters"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "nonce": self.nonce,
            "cloud_project_number": self.cloud_project_number,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["FirebaseDeviceVerificationParametersPlayIntegrity", None]:
        if data:
            data_class = cls()
            data_class.nonce = data.get("nonce", "")
            data_class.cloud_project_number = int(data.get("cloud_project_number", 0))

        return data_class


class PasswordState(TlObject):
    r"""Represents the current state of 2\-step verification

    Parameters:
        has_password (:class:`bool`):
            True, if a 2\-step verification password is set

        password_hint (:class:`str`):
            Hint for the password; may be empty

        has_recovery_email_address (:class:`bool`):
            True, if a recovery email is set

        has_passport_data (:class:`bool`):
            True, if some Telegram Passport elements were saved

        recovery_email_address_code_info (:class:`~pytdbot.types.EmailAddressAuthenticationCodeInfo`):
            Information about the recovery email address to which the confirmation email was sent; may be null

        login_email_address_pattern (:class:`str`):
            Pattern of the email address set up for logging in

        pending_reset_date (:class:`int`):
            If not 0, point in time \(Unix timestamp\) after which the 2\-step verification password can be reset immediately using resetPassword

    """

    def __init__(
        self,
        has_password: bool = False,
        password_hint: str = "",
        has_recovery_email_address: bool = False,
        has_passport_data: bool = False,
        recovery_email_address_code_info: EmailAddressAuthenticationCodeInfo = None,
        login_email_address_pattern: str = "",
        pending_reset_date: int = 0,
    ) -> None:
        self.has_password: bool = bool(has_password)
        r"""True, if a 2\-step verification password is set"""
        self.password_hint: Union[str, None] = password_hint
        r"""Hint for the password; may be empty"""
        self.has_recovery_email_address: bool = bool(has_recovery_email_address)
        r"""True, if a recovery email is set"""
        self.has_passport_data: bool = bool(has_passport_data)
        r"""True, if some Telegram Passport elements were saved"""
        self.recovery_email_address_code_info: Union[
            EmailAddressAuthenticationCodeInfo, None
        ] = recovery_email_address_code_info
        r"""Information about the recovery email address to which the confirmation email was sent; may be null"""
        self.login_email_address_pattern: Union[str, None] = login_email_address_pattern
        r"""Pattern of the email address set up for logging in"""
        self.pending_reset_date: int = int(pending_reset_date)
        r"""If not 0, point in time \(Unix timestamp\) after which the 2\-step verification password can be reset immediately using resetPassword"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["passwordState"]:
        return "passwordState"

    @classmethod
    def getClass(self) -> Literal["PasswordState"]:
        return "PasswordState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "has_password": self.has_password,
            "password_hint": self.password_hint,
            "has_recovery_email_address": self.has_recovery_email_address,
            "has_passport_data": self.has_passport_data,
            "recovery_email_address_code_info": self.recovery_email_address_code_info,
            "login_email_address_pattern": self.login_email_address_pattern,
            "pending_reset_date": self.pending_reset_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PasswordState", None]:
        if data:
            data_class = cls()
            data_class.has_password = data.get("has_password", False)
            data_class.password_hint = data.get("password_hint", "")
            data_class.has_recovery_email_address = data.get(
                "has_recovery_email_address", False
            )
            data_class.has_passport_data = data.get("has_passport_data", False)
            data_class.recovery_email_address_code_info = data.get(
                "recovery_email_address_code_info", None
            )
            data_class.login_email_address_pattern = data.get(
                "login_email_address_pattern", ""
            )
            data_class.pending_reset_date = int(data.get("pending_reset_date", 0))

        return data_class


class RecoveryEmailAddress(TlObject):
    r"""Contains information about the current recovery email address

    Parameters:
        recovery_email_address (:class:`str`):
            Recovery email address

    """

    def __init__(self, recovery_email_address: str = "") -> None:
        self.recovery_email_address: Union[str, None] = recovery_email_address
        r"""Recovery email address"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["recoveryEmailAddress"]:
        return "recoveryEmailAddress"

    @classmethod
    def getClass(self) -> Literal["RecoveryEmailAddress"]:
        return "RecoveryEmailAddress"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "recovery_email_address": self.recovery_email_address,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["RecoveryEmailAddress", None]:
        if data:
            data_class = cls()
            data_class.recovery_email_address = data.get("recovery_email_address", "")

        return data_class


class TemporaryPasswordState(TlObject):
    r"""Returns information about the availability of a temporary password, which can be used for payments

    Parameters:
        has_password (:class:`bool`):
            True, if a temporary password is available

        valid_for (:class:`int`):
            Time left before the temporary password expires, in seconds

    """

    def __init__(self, has_password: bool = False, valid_for: int = 0) -> None:
        self.has_password: bool = bool(has_password)
        r"""True, if a temporary password is available"""
        self.valid_for: int = int(valid_for)
        r"""Time left before the temporary password expires, in seconds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["temporaryPasswordState"]:
        return "temporaryPasswordState"

    @classmethod
    def getClass(self) -> Literal["TemporaryPasswordState"]:
        return "TemporaryPasswordState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "has_password": self.has_password,
            "valid_for": self.valid_for,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TemporaryPasswordState", None]:
        if data:
            data_class = cls()
            data_class.has_password = data.get("has_password", False)
            data_class.valid_for = int(data.get("valid_for", 0))

        return data_class


class LocalFile(TlObject):
    r"""Represents a local file

    Parameters:
        path (:class:`str`):
            Local path to the locally available file part; may be empty

        can_be_downloaded (:class:`bool`):
            True, if it is possible to download or generate the file

        can_be_deleted (:class:`bool`):
            True, if the file can be deleted

        is_downloading_active (:class:`bool`):
            True, if the file is currently being downloaded \(or a local copy is being generated by some other means\)

        is_downloading_completed (:class:`bool`):
            True, if the local copy is fully available

        download_offset (:class:`int`):
            Download will be started from this offset\. downloaded\_prefix\_size is calculated from this offset

        downloaded_prefix_size (:class:`int`):
            If is\_downloading\_completed is false, then only some prefix of the file starting from download\_offset is ready to be read\. downloaded\_prefix\_size is the size of that prefix in bytes

        downloaded_size (:class:`int`):
            Total downloaded file size, in bytes\. Can be used only for calculating download progress\. The actual file size may be bigger, and some parts of it may contain garbage

    """

    def __init__(
        self,
        path: str = "",
        can_be_downloaded: bool = False,
        can_be_deleted: bool = False,
        is_downloading_active: bool = False,
        is_downloading_completed: bool = False,
        download_offset: int = 0,
        downloaded_prefix_size: int = 0,
        downloaded_size: int = 0,
    ) -> None:
        self.path: Union[str, None] = path
        r"""Local path to the locally available file part; may be empty"""
        self.can_be_downloaded: bool = bool(can_be_downloaded)
        r"""True, if it is possible to download or generate the file"""
        self.can_be_deleted: bool = bool(can_be_deleted)
        r"""True, if the file can be deleted"""
        self.is_downloading_active: bool = bool(is_downloading_active)
        r"""True, if the file is currently being downloaded \(or a local copy is being generated by some other means\)"""
        self.is_downloading_completed: bool = bool(is_downloading_completed)
        r"""True, if the local copy is fully available"""
        self.download_offset: int = int(download_offset)
        r"""Download will be started from this offset\. downloaded\_prefix\_size is calculated from this offset"""
        self.downloaded_prefix_size: int = int(downloaded_prefix_size)
        r"""If is\_downloading\_completed is false, then only some prefix of the file starting from download\_offset is ready to be read\. downloaded\_prefix\_size is the size of that prefix in bytes"""
        self.downloaded_size: int = int(downloaded_size)
        r"""Total downloaded file size, in bytes\. Can be used only for calculating download progress\. The actual file size may be bigger, and some parts of it may contain garbage"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["localFile"]:
        return "localFile"

    @classmethod
    def getClass(self) -> Literal["LocalFile"]:
        return "LocalFile"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "path": self.path,
            "can_be_downloaded": self.can_be_downloaded,
            "can_be_deleted": self.can_be_deleted,
            "is_downloading_active": self.is_downloading_active,
            "is_downloading_completed": self.is_downloading_completed,
            "download_offset": self.download_offset,
            "downloaded_prefix_size": self.downloaded_prefix_size,
            "downloaded_size": self.downloaded_size,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["LocalFile", None]:
        if data:
            data_class = cls()
            data_class.path = data.get("path", "")
            data_class.can_be_downloaded = data.get("can_be_downloaded", False)
            data_class.can_be_deleted = data.get("can_be_deleted", False)
            data_class.is_downloading_active = data.get("is_downloading_active", False)
            data_class.is_downloading_completed = data.get(
                "is_downloading_completed", False
            )
            data_class.download_offset = int(data.get("download_offset", 0))
            data_class.downloaded_prefix_size = int(
                data.get("downloaded_prefix_size", 0)
            )
            data_class.downloaded_size = int(data.get("downloaded_size", 0))

        return data_class


class RemoteFile(TlObject, FileBoundMethods):
    r"""Represents a remote file

    Parameters:
        id (:class:`str`):
            Remote file identifier; may be empty\. Can be used by the current user across application restarts or even from other devices\. Uniquely identifies a file, but a file can have a lot of different valid identifiers\. If the identifier starts with \"http://\" or \"https://\", it represents the HTTP URL of the file\. TDLib is currently unable to download files if only their URL is known\. If downloadFile/addFileToDownloads is called on such a file or if it is sent to a secret chat, TDLib starts a file generation process by sending updateFileGenerationStart to the application with the HTTP URL in the original\_path and \"\#url\#\" as the conversion string\. Application must generate the file by downloading it to the specified location

        unique_id (:class:`str`):
            Unique file identifier; may be empty if unknown\. The unique file identifier which is the same for the same file even for different users and is persistent over time

        is_uploading_active (:class:`bool`):
            True, if the file is currently being uploaded \(or a remote copy is being generated by some other means\)

        is_uploading_completed (:class:`bool`):
            True, if a remote copy is fully available

        uploaded_size (:class:`int`):
            Size of the remote available part of the file, in bytes; 0 if unknown

    """

    def __init__(
        self,
        id: str = "",
        unique_id: str = "",
        is_uploading_active: bool = False,
        is_uploading_completed: bool = False,
        uploaded_size: int = 0,
    ) -> None:
        self.id: Union[str, None] = id
        r"""Remote file identifier; may be empty\. Can be used by the current user across application restarts or even from other devices\. Uniquely identifies a file, but a file can have a lot of different valid identifiers\. If the identifier starts with \"http://\" or \"https://\", it represents the HTTP URL of the file\. TDLib is currently unable to download files if only their URL is known\. If downloadFile/addFileToDownloads is called on such a file or if it is sent to a secret chat, TDLib starts a file generation process by sending updateFileGenerationStart to the application with the HTTP URL in the original\_path and \"\#url\#\" as the conversion string\. Application must generate the file by downloading it to the specified location"""
        self.unique_id: Union[str, None] = unique_id
        r"""Unique file identifier; may be empty if unknown\. The unique file identifier which is the same for the same file even for different users and is persistent over time"""
        self.is_uploading_active: bool = bool(is_uploading_active)
        r"""True, if the file is currently being uploaded \(or a remote copy is being generated by some other means\)"""
        self.is_uploading_completed: bool = bool(is_uploading_completed)
        r"""True, if a remote copy is fully available"""
        self.uploaded_size: int = int(uploaded_size)
        r"""Size of the remote available part of the file, in bytes; 0 if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["remoteFile"]:
        return "remoteFile"

    @classmethod
    def getClass(self) -> Literal["RemoteFile"]:
        return "RemoteFile"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "unique_id": self.unique_id,
            "is_uploading_active": self.is_uploading_active,
            "is_uploading_completed": self.is_uploading_completed,
            "uploaded_size": self.uploaded_size,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["RemoteFile", None]:
        if data:
            data_class = cls()
            data_class.id = data.get("id", "")
            data_class.unique_id = data.get("unique_id", "")
            data_class.is_uploading_active = data.get("is_uploading_active", False)
            data_class.is_uploading_completed = data.get(
                "is_uploading_completed", False
            )
            data_class.uploaded_size = int(data.get("uploaded_size", 0))

        return data_class


class File(TlObject, FileBoundMethods):
    r"""Represents a file

    Parameters:
        id (:class:`int`):
            Unique file identifier

        size (:class:`int`):
            File size, in bytes; 0 if unknown

        expected_size (:class:`int`):
            Approximate file size in bytes in case the exact file size is unknown\. Can be used to show download/upload progress

        local (:class:`~pytdbot.types.LocalFile`):
            Information about the local copy of the file

        remote (:class:`~pytdbot.types.RemoteFile`):
            Information about the remote copy of the file

    """

    def __init__(
        self,
        id: int = 0,
        size: int = 0,
        expected_size: int = 0,
        local: LocalFile = None,
        remote: RemoteFile = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique file identifier"""
        self.size: int = int(size)
        r"""File size, in bytes; 0 if unknown"""
        self.expected_size: int = int(expected_size)
        r"""Approximate file size in bytes in case the exact file size is unknown\. Can be used to show download/upload progress"""
        self.local: Union[LocalFile, None] = local
        r"""Information about the local copy of the file"""
        self.remote: Union[RemoteFile, None] = remote
        r"""Information about the remote copy of the file"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["file"]:
        return "file"

    @classmethod
    def getClass(self) -> Literal["File"]:
        return "File"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "size": self.size,
            "expected_size": self.expected_size,
            "local": self.local,
            "remote": self.remote,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["File", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.size = int(data.get("size", 0))
            data_class.expected_size = int(data.get("expected_size", 0))
            data_class.local = data.get("local", None)
            data_class.remote = data.get("remote", None)

        return data_class


class InputFileId(TlObject, InputFile):
    r"""A file defined by its unique identifier

    Parameters:
        id (:class:`int`):
            Unique file identifier

    """

    def __init__(self, id: int = 0) -> None:
        self.id: int = int(id)
        r"""Unique file identifier"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputFileId"]:
        return "inputFileId"

    @classmethod
    def getClass(self) -> Literal["InputFile"]:
        return "InputFile"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "id": self.id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputFileId", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))

        return data_class


class InputFileRemote(TlObject, InputFile):
    r"""A file defined by its remote identifier\. The remote identifier is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib\. For example, if the file is from a message, then the message must be not deleted and accessible to the user\. If the file database is disabled, then the corresponding object with the file must be preloaded by the application

    Parameters:
        id (:class:`str`):
            Remote file identifier

    """

    def __init__(self, id: str = "") -> None:
        self.id: Union[str, None] = id
        r"""Remote file identifier"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputFileRemote"]:
        return "inputFileRemote"

    @classmethod
    def getClass(self) -> Literal["InputFile"]:
        return "InputFile"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "id": self.id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputFileRemote", None]:
        if data:
            data_class = cls()
            data_class.id = data.get("id", "")

        return data_class


class InputFileLocal(TlObject, InputFile):
    r"""A file defined by a local path

    Parameters:
        path (:class:`str`):
            Local path to the file

    """

    def __init__(self, path: str = "") -> None:
        self.path: Union[str, None] = path
        r"""Local path to the file"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputFileLocal"]:
        return "inputFileLocal"

    @classmethod
    def getClass(self) -> Literal["InputFile"]:
        return "InputFile"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "path": self.path}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputFileLocal", None]:
        if data:
            data_class = cls()
            data_class.path = data.get("path", "")

        return data_class


class InputFileGenerated(TlObject, InputFile):
    r"""A file generated by the application\. The application must handle updates updateFileGenerationStart and updateFileGenerationStop to generate the file when asked by TDLib

    Parameters:
        original_path (:class:`str`):
            Local path to a file from which the file is generated\. The path doesn't have to be a valid path and is used by TDLib only to detect name and MIME type of the generated file

        conversion (:class:`str`):
            String specifying the conversion applied to the original file; must be persistent across application restarts\. Conversions beginning with '\#' are reserved for internal TDLib usage

        expected_size (:class:`int`):
            Expected size of the generated file, in bytes; pass 0 if unknown

    """

    def __init__(
        self, original_path: str = "", conversion: str = "", expected_size: int = 0
    ) -> None:
        self.original_path: Union[str, None] = original_path
        r"""Local path to a file from which the file is generated\. The path doesn't have to be a valid path and is used by TDLib only to detect name and MIME type of the generated file"""
        self.conversion: Union[str, None] = conversion
        r"""String specifying the conversion applied to the original file; must be persistent across application restarts\. Conversions beginning with '\#' are reserved for internal TDLib usage"""
        self.expected_size: int = int(expected_size)
        r"""Expected size of the generated file, in bytes; pass 0 if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputFileGenerated"]:
        return "inputFileGenerated"

    @classmethod
    def getClass(self) -> Literal["InputFile"]:
        return "InputFile"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "original_path": self.original_path,
            "conversion": self.conversion,
            "expected_size": self.expected_size,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputFileGenerated", None]:
        if data:
            data_class = cls()
            data_class.original_path = data.get("original_path", "")
            data_class.conversion = data.get("conversion", "")
            data_class.expected_size = int(data.get("expected_size", 0))

        return data_class


class PhotoSize(TlObject):
    r"""Describes an image in JPEG format

    Parameters:
        type (:class:`str`):
            Image type \(see https://core\.telegram\.org/constructor/photoSize\)

        photo (:class:`~pytdbot.types.File`):
            Information about the image file

        width (:class:`int`):
            Image width

        height (:class:`int`):
            Image height

        progressive_sizes (List[:class:`int`]):
            Sizes of progressive JPEG file prefixes, which can be used to preliminarily show the image; in bytes

    """

    def __init__(
        self,
        type: str = "",
        photo: File = None,
        width: int = 0,
        height: int = 0,
        progressive_sizes: List[int] = None,
    ) -> None:
        self.type: Union[str, None] = type
        r"""Image type \(see https://core\.telegram\.org/constructor/photoSize\)"""
        self.photo: Union[File, None] = photo
        r"""Information about the image file"""
        self.width: int = int(width)
        r"""Image width"""
        self.height: int = int(height)
        r"""Image height"""
        self.progressive_sizes: List[int] = progressive_sizes or []
        r"""Sizes of progressive JPEG file prefixes, which can be used to preliminarily show the image; in bytes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["photoSize"]:
        return "photoSize"

    @classmethod
    def getClass(self) -> Literal["PhotoSize"]:
        return "PhotoSize"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "type": self.type,
            "photo": self.photo,
            "width": self.width,
            "height": self.height,
            "progressive_sizes": self.progressive_sizes,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PhotoSize", None]:
        if data:
            data_class = cls()
            data_class.type = data.get("type", "")
            data_class.photo = data.get("photo", None)
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.progressive_sizes = data.get("progressive_sizes", None)

        return data_class


class Minithumbnail(TlObject):
    r"""Thumbnail image of a very poor quality and low resolution

    Parameters:
        width (:class:`int`):
            Thumbnail width, usually doesn't exceed 40

        height (:class:`int`):
            Thumbnail height, usually doesn't exceed 40

        data (:class:`bytes`):
            The thumbnail in JPEG format

    """

    def __init__(self, width: int = 0, height: int = 0, data: bytes = b"") -> None:
        self.width: int = int(width)
        r"""Thumbnail width, usually doesn't exceed 40"""
        self.height: int = int(height)
        r"""Thumbnail height, usually doesn't exceed 40"""
        self.data: Union[bytes, None] = data
        r"""The thumbnail in JPEG format"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["minithumbnail"]:
        return "minithumbnail"

    @classmethod
    def getClass(self) -> Literal["Minithumbnail"]:
        return "Minithumbnail"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "width": self.width,
            "height": self.height,
            "data": self.data,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Minithumbnail", None]:
        if data:
            data_class = cls()
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.data = b64decode(data.get("data", b""))

        return data_class


class ThumbnailFormatJpeg(TlObject, ThumbnailFormat):
    r"""The thumbnail is in JPEG format"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatJpeg"]:
        return "thumbnailFormatJpeg"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatJpeg", None]:
        if data:
            data_class = cls()

        return data_class


class ThumbnailFormatGif(TlObject, ThumbnailFormat):
    r"""The thumbnail is in static GIF format\. It will be used only for some bot inline query results"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatGif"]:
        return "thumbnailFormatGif"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatGif", None]:
        if data:
            data_class = cls()

        return data_class


class ThumbnailFormatMpeg4(TlObject, ThumbnailFormat):
    r"""The thumbnail is in MPEG4 format\. It will be used only for some animations and videos"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatMpeg4"]:
        return "thumbnailFormatMpeg4"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatMpeg4", None]:
        if data:
            data_class = cls()

        return data_class


class ThumbnailFormatPng(TlObject, ThumbnailFormat):
    r"""The thumbnail is in PNG format\. It will be used only for background patterns"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatPng"]:
        return "thumbnailFormatPng"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatPng", None]:
        if data:
            data_class = cls()

        return data_class


class ThumbnailFormatTgs(TlObject, ThumbnailFormat):
    r"""The thumbnail is in TGS format\. It will be used only for sticker sets"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatTgs"]:
        return "thumbnailFormatTgs"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatTgs", None]:
        if data:
            data_class = cls()

        return data_class


class ThumbnailFormatWebm(TlObject, ThumbnailFormat):
    r"""The thumbnail is in WEBM format\. It will be used only for sticker sets"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatWebm"]:
        return "thumbnailFormatWebm"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatWebm", None]:
        if data:
            data_class = cls()

        return data_class


class ThumbnailFormatWebp(TlObject, ThumbnailFormat):
    r"""The thumbnail is in WEBP format\. It will be used only for some stickers and sticker sets"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnailFormatWebp"]:
        return "thumbnailFormatWebp"

    @classmethod
    def getClass(self) -> Literal["ThumbnailFormat"]:
        return "ThumbnailFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ThumbnailFormatWebp", None]:
        if data:
            data_class = cls()

        return data_class


class Thumbnail(TlObject):
    r"""Represents a thumbnail

    Parameters:
        format (:class:`~pytdbot.types.ThumbnailFormat`):
            Thumbnail format

        width (:class:`int`):
            Thumbnail width

        height (:class:`int`):
            Thumbnail height

        file (:class:`~pytdbot.types.File`):
            The thumbnail

    """

    def __init__(
        self,
        format: ThumbnailFormat = None,
        width: int = 0,
        height: int = 0,
        file: File = None,
    ) -> None:
        self.format: Union[
            ThumbnailFormatJpeg,
            ThumbnailFormatGif,
            ThumbnailFormatMpeg4,
            ThumbnailFormatPng,
            ThumbnailFormatTgs,
            ThumbnailFormatWebm,
            ThumbnailFormatWebp,
            None,
        ] = format
        r"""Thumbnail format"""
        self.width: int = int(width)
        r"""Thumbnail width"""
        self.height: int = int(height)
        r"""Thumbnail height"""
        self.file: Union[File, None] = file
        r"""The thumbnail"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["thumbnail"]:
        return "thumbnail"

    @classmethod
    def getClass(self) -> Literal["Thumbnail"]:
        return "Thumbnail"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "format": self.format,
            "width": self.width,
            "height": self.height,
            "file": self.file,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Thumbnail", None]:
        if data:
            data_class = cls()
            data_class.format = data.get("format", None)
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.file = data.get("file", None)

        return data_class


class MaskPointForehead(TlObject, MaskPoint):
    r"""The mask is placed relatively to the forehead"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["maskPointForehead"]:
        return "maskPointForehead"

    @classmethod
    def getClass(self) -> Literal["MaskPoint"]:
        return "MaskPoint"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MaskPointForehead", None]:
        if data:
            data_class = cls()

        return data_class


class MaskPointEyes(TlObject, MaskPoint):
    r"""The mask is placed relatively to the eyes"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["maskPointEyes"]:
        return "maskPointEyes"

    @classmethod
    def getClass(self) -> Literal["MaskPoint"]:
        return "MaskPoint"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MaskPointEyes", None]:
        if data:
            data_class = cls()

        return data_class


class MaskPointMouth(TlObject, MaskPoint):
    r"""The mask is placed relatively to the mouth"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["maskPointMouth"]:
        return "maskPointMouth"

    @classmethod
    def getClass(self) -> Literal["MaskPoint"]:
        return "MaskPoint"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MaskPointMouth", None]:
        if data:
            data_class = cls()

        return data_class


class MaskPointChin(TlObject, MaskPoint):
    r"""The mask is placed relatively to the chin"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["maskPointChin"]:
        return "maskPointChin"

    @classmethod
    def getClass(self) -> Literal["MaskPoint"]:
        return "MaskPoint"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MaskPointChin", None]:
        if data:
            data_class = cls()

        return data_class


class MaskPosition(TlObject):
    r"""Position on a photo where a mask is placed

    Parameters:
        point (:class:`~pytdbot.types.MaskPoint`):
            Part of the face, relative to which the mask is placed

        x_shift (:class:`float`):
            Shift by X\-axis measured in widths of the mask scaled to the face size, from left to right\. \(For example, \-1\.0 will place the mask just to the left of the default mask position\)

        y_shift (:class:`float`):
            Shift by Y\-axis measured in heights of the mask scaled to the face size, from top to bottom\. \(For example, 1\.0 will place the mask just below the default mask position\)

        scale (:class:`float`):
            Mask scaling coefficient\. \(For example, 2\.0 means a doubled size\)

    """

    def __init__(
        self,
        point: MaskPoint = None,
        x_shift: float = 0.0,
        y_shift: float = 0.0,
        scale: float = 0.0,
    ) -> None:
        self.point: Union[
            MaskPointForehead, MaskPointEyes, MaskPointMouth, MaskPointChin, None
        ] = point
        r"""Part of the face, relative to which the mask is placed"""
        self.x_shift: float = float(x_shift)
        r"""Shift by X\-axis measured in widths of the mask scaled to the face size, from left to right\. \(For example, \-1\.0 will place the mask just to the left of the default mask position\)"""
        self.y_shift: float = float(y_shift)
        r"""Shift by Y\-axis measured in heights of the mask scaled to the face size, from top to bottom\. \(For example, 1\.0 will place the mask just below the default mask position\)"""
        self.scale: float = float(scale)
        r"""Mask scaling coefficient\. \(For example, 2\.0 means a doubled size\)"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["maskPosition"]:
        return "maskPosition"

    @classmethod
    def getClass(self) -> Literal["MaskPosition"]:
        return "MaskPosition"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "point": self.point,
            "x_shift": self.x_shift,
            "y_shift": self.y_shift,
            "scale": self.scale,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MaskPosition", None]:
        if data:
            data_class = cls()
            data_class.point = data.get("point", None)
            data_class.x_shift = data.get("x_shift", 0.0)
            data_class.y_shift = data.get("y_shift", 0.0)
            data_class.scale = data.get("scale", 0.0)

        return data_class


class StickerFormatWebp(TlObject, StickerFormat):
    r"""The sticker is an image in WEBP format"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerFormatWebp"]:
        return "stickerFormatWebp"

    @classmethod
    def getClass(self) -> Literal["StickerFormat"]:
        return "StickerFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerFormatWebp", None]:
        if data:
            data_class = cls()

        return data_class


class StickerFormatTgs(TlObject, StickerFormat):
    r"""The sticker is an animation in TGS format"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerFormatTgs"]:
        return "stickerFormatTgs"

    @classmethod
    def getClass(self) -> Literal["StickerFormat"]:
        return "StickerFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerFormatTgs", None]:
        if data:
            data_class = cls()

        return data_class


class StickerFormatWebm(TlObject, StickerFormat):
    r"""The sticker is a video in WEBM format"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerFormatWebm"]:
        return "stickerFormatWebm"

    @classmethod
    def getClass(self) -> Literal["StickerFormat"]:
        return "StickerFormat"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerFormatWebm", None]:
        if data:
            data_class = cls()

        return data_class


class StickerTypeRegular(TlObject, StickerType):
    r"""The sticker is a regular sticker"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerTypeRegular"]:
        return "stickerTypeRegular"

    @classmethod
    def getClass(self) -> Literal["StickerType"]:
        return "StickerType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerTypeRegular", None]:
        if data:
            data_class = cls()

        return data_class


class StickerTypeMask(TlObject, StickerType):
    r"""The sticker is a mask in WEBP format to be placed on photos or videos"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerTypeMask"]:
        return "stickerTypeMask"

    @classmethod
    def getClass(self) -> Literal["StickerType"]:
        return "StickerType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerTypeMask", None]:
        if data:
            data_class = cls()

        return data_class


class StickerTypeCustomEmoji(TlObject, StickerType):
    r"""The sticker is a custom emoji to be used inside message text and caption"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerTypeCustomEmoji"]:
        return "stickerTypeCustomEmoji"

    @classmethod
    def getClass(self) -> Literal["StickerType"]:
        return "StickerType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerTypeCustomEmoji", None]:
        if data:
            data_class = cls()

        return data_class


class StickerFullTypeRegular(TlObject, StickerFullType):
    r"""The sticker is a regular sticker

    Parameters:
        premium_animation (:class:`~pytdbot.types.File`):
            Premium animation of the sticker; may be null\. If present, only Telegram Premium users can use the sticker

    """

    def __init__(self, premium_animation: File = None) -> None:
        self.premium_animation: Union[File, None] = premium_animation
        r"""Premium animation of the sticker; may be null\. If present, only Telegram Premium users can use the sticker"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerFullTypeRegular"]:
        return "stickerFullTypeRegular"

    @classmethod
    def getClass(self) -> Literal["StickerFullType"]:
        return "StickerFullType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "premium_animation": self.premium_animation}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerFullTypeRegular", None]:
        if data:
            data_class = cls()
            data_class.premium_animation = data.get("premium_animation", None)

        return data_class


class StickerFullTypeMask(TlObject, StickerFullType):
    r"""The sticker is a mask in WEBP format to be placed on photos or videos

    Parameters:
        mask_position (:class:`~pytdbot.types.MaskPosition`):
            Position where the mask is placed; may be null

    """

    def __init__(self, mask_position: MaskPosition = None) -> None:
        self.mask_position: Union[MaskPosition, None] = mask_position
        r"""Position where the mask is placed; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerFullTypeMask"]:
        return "stickerFullTypeMask"

    @classmethod
    def getClass(self) -> Literal["StickerFullType"]:
        return "StickerFullType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "mask_position": self.mask_position}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerFullTypeMask", None]:
        if data:
            data_class = cls()
            data_class.mask_position = data.get("mask_position", None)

        return data_class


class StickerFullTypeCustomEmoji(TlObject, StickerFullType):
    r"""The sticker is a custom emoji to be used inside message text and caption\. Currently, only Telegram Premium users can use custom emoji

    Parameters:
        custom_emoji_id (:class:`int`):
            Identifier of the custom emoji

        needs_repainting (:class:`bool`):
            True, if the sticker must be repainted to a text color in messages, the color of the Telegram Premium badge in emoji status, white color on chat photos, or another appropriate color in other places

    """

    def __init__(
        self, custom_emoji_id: int = 0, needs_repainting: bool = False
    ) -> None:
        self.custom_emoji_id: int = int(custom_emoji_id)
        r"""Identifier of the custom emoji"""
        self.needs_repainting: bool = bool(needs_repainting)
        r"""True, if the sticker must be repainted to a text color in messages, the color of the Telegram Premium badge in emoji status, white color on chat photos, or another appropriate color in other places"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["stickerFullTypeCustomEmoji"]:
        return "stickerFullTypeCustomEmoji"

    @classmethod
    def getClass(self) -> Literal["StickerFullType"]:
        return "StickerFullType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "custom_emoji_id": self.custom_emoji_id,
            "needs_repainting": self.needs_repainting,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StickerFullTypeCustomEmoji", None]:
        if data:
            data_class = cls()
            data_class.custom_emoji_id = int(data.get("custom_emoji_id", 0))
            data_class.needs_repainting = data.get("needs_repainting", False)

        return data_class


class ClosedVectorPath(TlObject):
    r"""Represents a closed vector path\. The path begins at the end point of the last command\. The coordinate system origin is in the upper\-left corner

    Parameters:
        commands (List[:class:`~pytdbot.types.VectorPathCommand`]):
            List of vector path commands

    """

    def __init__(self, commands: List[VectorPathCommand] = None) -> None:
        self.commands: List[VectorPathCommand] = commands or []
        r"""List of vector path commands"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["closedVectorPath"]:
        return "closedVectorPath"

    @classmethod
    def getClass(self) -> Literal["ClosedVectorPath"]:
        return "ClosedVectorPath"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "commands": self.commands}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ClosedVectorPath", None]:
        if data:
            data_class = cls()
            data_class.commands = data.get("commands", None)

        return data_class


class Outline(TlObject):
    r"""Represents outline of an image

    Parameters:
        paths (List[:class:`~pytdbot.types.ClosedVectorPath`]):
            The list of closed vector paths

    """

    def __init__(self, paths: List[ClosedVectorPath] = None) -> None:
        self.paths: List[ClosedVectorPath] = paths or []
        r"""The list of closed vector paths"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["outline"]:
        return "outline"

    @classmethod
    def getClass(self) -> Literal["Outline"]:
        return "Outline"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "paths": self.paths}

    @classmethod
    def from_dict(cls, data: dict) -> Union["Outline", None]:
        if data:
            data_class = cls()
            data_class.paths = data.get("paths", None)

        return data_class


class PollOption(TlObject):
    r"""Describes one answer option of a poll

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            Option text; 1\-100 characters\. Only custom emoji entities are allowed

        voter_count (:class:`int`):
            Number of voters for this option, available only for closed or voted polls

        vote_percentage (:class:`int`):
            The percentage of votes for this option; 0\-100

        is_chosen (:class:`bool`):
            True, if the option was chosen by the user

        is_being_chosen (:class:`bool`):
            True, if the option is being chosen by a pending setPollAnswer request

    """

    def __init__(
        self,
        text: FormattedText = None,
        voter_count: int = 0,
        vote_percentage: int = 0,
        is_chosen: bool = False,
        is_being_chosen: bool = False,
    ) -> None:
        self.text: Union[FormattedText, None] = text
        r"""Option text; 1\-100 characters\. Only custom emoji entities are allowed"""
        self.voter_count: int = int(voter_count)
        r"""Number of voters for this option, available only for closed or voted polls"""
        self.vote_percentage: int = int(vote_percentage)
        r"""The percentage of votes for this option; 0\-100"""
        self.is_chosen: bool = bool(is_chosen)
        r"""True, if the option was chosen by the user"""
        self.is_being_chosen: bool = bool(is_being_chosen)
        r"""True, if the option is being chosen by a pending setPollAnswer request"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["pollOption"]:
        return "pollOption"

    @classmethod
    def getClass(self) -> Literal["PollOption"]:
        return "PollOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "text": self.text,
            "voter_count": self.voter_count,
            "vote_percentage": self.vote_percentage,
            "is_chosen": self.is_chosen,
            "is_being_chosen": self.is_being_chosen,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PollOption", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.voter_count = int(data.get("voter_count", 0))
            data_class.vote_percentage = int(data.get("vote_percentage", 0))
            data_class.is_chosen = data.get("is_chosen", False)
            data_class.is_being_chosen = data.get("is_being_chosen", False)

        return data_class


class PollTypeRegular(TlObject, PollType):
    r"""A regular poll

    Parameters:
        allow_multiple_answers (:class:`bool`):
            True, if multiple answer options can be chosen simultaneously

    """

    def __init__(self, allow_multiple_answers: bool = False) -> None:
        self.allow_multiple_answers: bool = bool(allow_multiple_answers)
        r"""True, if multiple answer options can be chosen simultaneously"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["pollTypeRegular"]:
        return "pollTypeRegular"

    @classmethod
    def getClass(self) -> Literal["PollType"]:
        return "PollType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "allow_multiple_answers": self.allow_multiple_answers,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PollTypeRegular", None]:
        if data:
            data_class = cls()
            data_class.allow_multiple_answers = data.get(
                "allow_multiple_answers", False
            )

        return data_class


class PollTypeQuiz(TlObject, PollType):
    r"""A poll in quiz mode, which has exactly one correct answer option and can be answered only once

    Parameters:
        correct_option_id (:class:`int`):
            0\-based identifier of the correct answer option; \-1 for a yet unanswered poll

        explanation (:class:`~pytdbot.types.FormattedText`):
            Text that is shown when the user chooses an incorrect answer or taps on the lamp icon; 0\-200 characters with at most 2 line feeds; empty for a yet unanswered poll

    """

    def __init__(
        self, correct_option_id: int = 0, explanation: FormattedText = None
    ) -> None:
        self.correct_option_id: int = int(correct_option_id)
        r"""0\-based identifier of the correct answer option; \-1 for a yet unanswered poll"""
        self.explanation: Union[FormattedText, None] = explanation
        r"""Text that is shown when the user chooses an incorrect answer or taps on the lamp icon; 0\-200 characters with at most 2 line feeds; empty for a yet unanswered poll"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["pollTypeQuiz"]:
        return "pollTypeQuiz"

    @classmethod
    def getClass(self) -> Literal["PollType"]:
        return "PollType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "correct_option_id": self.correct_option_id,
            "explanation": self.explanation,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PollTypeQuiz", None]:
        if data:
            data_class = cls()
            data_class.correct_option_id = int(data.get("correct_option_id", 0))
            data_class.explanation = data.get("explanation", None)

        return data_class


class ChecklistTask(TlObject):
    r"""Describes a task in a checklist

    Parameters:
        id (:class:`int`):
            Unique identifier of the task

        text (:class:`~pytdbot.types.FormattedText`):
            Text of the task; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, CustomEmoji, Url, EmailAddress, Mention, Hashtag, Cashtag and PhoneNumber entities

        completed_by_user_id (:class:`int`):
            Identifier of the user that completed the task; 0 if the task isn't completed

        completion_date (:class:`int`):
            Point in time \(Unix timestamp\) when the task was completed; 0 if the task isn't completed

    """

    def __init__(
        self,
        id: int = 0,
        text: FormattedText = None,
        completed_by_user_id: int = 0,
        completion_date: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the task"""
        self.text: Union[FormattedText, None] = text
        r"""Text of the task; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, CustomEmoji, Url, EmailAddress, Mention, Hashtag, Cashtag and PhoneNumber entities"""
        self.completed_by_user_id: int = int(completed_by_user_id)
        r"""Identifier of the user that completed the task; 0 if the task isn't completed"""
        self.completion_date: int = int(completion_date)
        r"""Point in time \(Unix timestamp\) when the task was completed; 0 if the task isn't completed"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["checklistTask"]:
        return "checklistTask"

    @classmethod
    def getClass(self) -> Literal["ChecklistTask"]:
        return "ChecklistTask"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "text": self.text,
            "completed_by_user_id": self.completed_by_user_id,
            "completion_date": self.completion_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChecklistTask", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.text = data.get("text", None)
            data_class.completed_by_user_id = int(data.get("completed_by_user_id", 0))
            data_class.completion_date = int(data.get("completion_date", 0))

        return data_class


class InputChecklistTask(TlObject):
    r"""Describes a task in a checklist to be sent

    Parameters:
        id (:class:`int`):
            Unique identifier of the task; must be positive

        text (:class:`~pytdbot.types.FormattedText`):
            Text of the task; 1\-getOption\(\"checklist\_task\_text\_length\_max\"\) characters without line feeds\. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities

    """

    def __init__(self, id: int = 0, text: FormattedText = None) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the task; must be positive"""
        self.text: Union[FormattedText, None] = text
        r"""Text of the task; 1\-getOption\(\"checklist\_task\_text\_length\_max\"\) characters without line feeds\. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputChecklistTask"]:
        return "inputChecklistTask"

    @classmethod
    def getClass(self) -> Literal["InputChecklistTask"]:
        return "InputChecklistTask"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "id": self.id, "text": self.text}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputChecklistTask", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.text = data.get("text", None)

        return data_class


class Checklist(TlObject):
    r"""Describes a checklist

    Parameters:
        title (:class:`~pytdbot.types.FormattedText`):
            Title of the checklist; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities

        tasks (List[:class:`~pytdbot.types.ChecklistTask`]):
            List of tasks in the checklist

        others_can_add_tasks (:class:`bool`):
            True, if users other than creator of the list can add tasks to the list

        can_add_tasks (:class:`bool`):
            True, if the current user can add tasks to the list if they have Telegram Premium subscription

        others_can_mark_tasks_as_done (:class:`bool`):
            True, if users other than creator of the list can mark tasks as done or not done\. If true, then the checklist is called \"group checklist\"

        can_mark_tasks_as_done (:class:`bool`):
            True, if the current user can mark tasks as done or not done if they have Telegram Premium subscription

    """

    def __init__(
        self,
        title: FormattedText = None,
        tasks: List[ChecklistTask] = None,
        others_can_add_tasks: bool = False,
        can_add_tasks: bool = False,
        others_can_mark_tasks_as_done: bool = False,
        can_mark_tasks_as_done: bool = False,
    ) -> None:
        self.title: Union[FormattedText, None] = title
        r"""Title of the checklist; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities"""
        self.tasks: List[ChecklistTask] = tasks or []
        r"""List of tasks in the checklist"""
        self.others_can_add_tasks: bool = bool(others_can_add_tasks)
        r"""True, if users other than creator of the list can add tasks to the list"""
        self.can_add_tasks: bool = bool(can_add_tasks)
        r"""True, if the current user can add tasks to the list if they have Telegram Premium subscription"""
        self.others_can_mark_tasks_as_done: bool = bool(others_can_mark_tasks_as_done)
        r"""True, if users other than creator of the list can mark tasks as done or not done\. If true, then the checklist is called \"group checklist\""""
        self.can_mark_tasks_as_done: bool = bool(can_mark_tasks_as_done)
        r"""True, if the current user can mark tasks as done or not done if they have Telegram Premium subscription"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["checklist"]:
        return "checklist"

    @classmethod
    def getClass(self) -> Literal["Checklist"]:
        return "Checklist"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "title": self.title,
            "tasks": self.tasks,
            "others_can_add_tasks": self.others_can_add_tasks,
            "can_add_tasks": self.can_add_tasks,
            "others_can_mark_tasks_as_done": self.others_can_mark_tasks_as_done,
            "can_mark_tasks_as_done": self.can_mark_tasks_as_done,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Checklist", None]:
        if data:
            data_class = cls()
            data_class.title = data.get("title", None)
            data_class.tasks = data.get("tasks", None)
            data_class.others_can_add_tasks = data.get("others_can_add_tasks", False)
            data_class.can_add_tasks = data.get("can_add_tasks", False)
            data_class.others_can_mark_tasks_as_done = data.get(
                "others_can_mark_tasks_as_done", False
            )
            data_class.can_mark_tasks_as_done = data.get(
                "can_mark_tasks_as_done", False
            )

        return data_class


class InputChecklist(TlObject):
    r"""Describes a checklist to be sent

    Parameters:
        title (:class:`~pytdbot.types.FormattedText`):
            Title of the checklist; 1\-getOption\(\"checklist\_title\_length\_max\"\) characters\. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities

        tasks (List[:class:`~pytdbot.types.InputChecklistTask`]):
            List of tasks in the checklist; 1\-getOption\(\"checklist\_task\_count\_max\"\) tasks

        others_can_add_tasks (:class:`bool`):
            True, if other users can add tasks to the list

        others_can_mark_tasks_as_done (:class:`bool`):
            True, if other users can mark tasks as done or not done

    """

    def __init__(
        self,
        title: FormattedText = None,
        tasks: List[InputChecklistTask] = None,
        others_can_add_tasks: bool = False,
        others_can_mark_tasks_as_done: bool = False,
    ) -> None:
        self.title: Union[FormattedText, None] = title
        r"""Title of the checklist; 1\-getOption\(\"checklist\_title\_length\_max\"\) characters\. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities"""
        self.tasks: List[InputChecklistTask] = tasks or []
        r"""List of tasks in the checklist; 1\-getOption\(\"checklist\_task\_count\_max\"\) tasks"""
        self.others_can_add_tasks: bool = bool(others_can_add_tasks)
        r"""True, if other users can add tasks to the list"""
        self.others_can_mark_tasks_as_done: bool = bool(others_can_mark_tasks_as_done)
        r"""True, if other users can mark tasks as done or not done"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputChecklist"]:
        return "inputChecklist"

    @classmethod
    def getClass(self) -> Literal["InputChecklist"]:
        return "InputChecklist"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "title": self.title,
            "tasks": self.tasks,
            "others_can_add_tasks": self.others_can_add_tasks,
            "others_can_mark_tasks_as_done": self.others_can_mark_tasks_as_done,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputChecklist", None]:
        if data:
            data_class = cls()
            data_class.title = data.get("title", None)
            data_class.tasks = data.get("tasks", None)
            data_class.others_can_add_tasks = data.get("others_can_add_tasks", False)
            data_class.others_can_mark_tasks_as_done = data.get(
                "others_can_mark_tasks_as_done", False
            )

        return data_class


class Animation(TlObject):
    r"""Describes an animation file\. The animation must be encoded in GIF or MPEG4 format

    Parameters:
        duration (:class:`int`):
            Duration of the animation, in seconds; as defined by the sender

        width (:class:`int`):
            Width of the animation

        height (:class:`int`):
            Height of the animation

        file_name (:class:`str`):
            Original name of the file; as defined by the sender

        mime_type (:class:`str`):
            MIME type of the file, usually \"image/gif\" or \"video/mp4\"

        has_stickers (:class:`bool`):
            True, if stickers were added to the animation\. The list of corresponding sticker set can be received using getAttachedStickerSets

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Animation minithumbnail; may be null

        thumbnail (:class:`~pytdbot.types.Thumbnail`):
            Animation thumbnail in JPEG or MPEG4 format; may be null

        animation (:class:`~pytdbot.types.File`):
            File containing the animation

    """

    def __init__(
        self,
        duration: int = 0,
        width: int = 0,
        height: int = 0,
        file_name: str = "",
        mime_type: str = "",
        has_stickers: bool = False,
        minithumbnail: Minithumbnail = None,
        thumbnail: Thumbnail = None,
        animation: File = None,
    ) -> None:
        self.duration: int = int(duration)
        r"""Duration of the animation, in seconds; as defined by the sender"""
        self.width: int = int(width)
        r"""Width of the animation"""
        self.height: int = int(height)
        r"""Height of the animation"""
        self.file_name: Union[str, None] = file_name
        r"""Original name of the file; as defined by the sender"""
        self.mime_type: Union[str, None] = mime_type
        r"""MIME type of the file, usually \"image/gif\" or \"video/mp4\""""
        self.has_stickers: bool = bool(has_stickers)
        r"""True, if stickers were added to the animation\. The list of corresponding sticker set can be received using getAttachedStickerSets"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Animation minithumbnail; may be null"""
        self.thumbnail: Union[Thumbnail, None] = thumbnail
        r"""Animation thumbnail in JPEG or MPEG4 format; may be null"""
        self.animation: Union[File, None] = animation
        r"""File containing the animation"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["animation"]:
        return "animation"

    @classmethod
    def getClass(self) -> Literal["Animation"]:
        return "Animation"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "duration": self.duration,
            "width": self.width,
            "height": self.height,
            "file_name": self.file_name,
            "mime_type": self.mime_type,
            "has_stickers": self.has_stickers,
            "minithumbnail": self.minithumbnail,
            "thumbnail": self.thumbnail,
            "animation": self.animation,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Animation", None]:
        if data:
            data_class = cls()
            data_class.duration = int(data.get("duration", 0))
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.file_name = data.get("file_name", "")
            data_class.mime_type = data.get("mime_type", "")
            data_class.has_stickers = data.get("has_stickers", False)
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.thumbnail = data.get("thumbnail", None)
            data_class.animation = data.get("animation", None)

        return data_class


class Audio(TlObject):
    r"""Describes an audio file\. Audio is usually in MP3 or M4A format

    Parameters:
        duration (:class:`int`):
            Duration of the audio, in seconds; as defined by the sender

        title (:class:`str`):
            Title of the audio; as defined by the sender

        performer (:class:`str`):
            Performer of the audio; as defined by the sender

        file_name (:class:`str`):
            Original name of the file; as defined by the sender

        mime_type (:class:`str`):
            The MIME type of the file; as defined by the sender

        album_cover_minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            The minithumbnail of the album cover; may be null

        album_cover_thumbnail (:class:`~pytdbot.types.Thumbnail`):
            The thumbnail of the album cover in JPEG format; as defined by the sender\. The full size thumbnail is expected to be extracted from the downloaded audio file; may be null

        external_album_covers (List[:class:`~pytdbot.types.Thumbnail`]):
            Album cover variants to use if the downloaded audio file contains no album cover\. Provided thumbnail dimensions are approximate

        audio (:class:`~pytdbot.types.File`):
            File containing the audio

    """

    def __init__(
        self,
        duration: int = 0,
        title: str = "",
        performer: str = "",
        file_name: str = "",
        mime_type: str = "",
        album_cover_minithumbnail: Minithumbnail = None,
        album_cover_thumbnail: Thumbnail = None,
        external_album_covers: List[Thumbnail] = None,
        audio: File = None,
    ) -> None:
        self.duration: int = int(duration)
        r"""Duration of the audio, in seconds; as defined by the sender"""
        self.title: Union[str, None] = title
        r"""Title of the audio; as defined by the sender"""
        self.performer: Union[str, None] = performer
        r"""Performer of the audio; as defined by the sender"""
        self.file_name: Union[str, None] = file_name
        r"""Original name of the file; as defined by the sender"""
        self.mime_type: Union[str, None] = mime_type
        r"""The MIME type of the file; as defined by the sender"""
        self.album_cover_minithumbnail: Union[Minithumbnail, None] = (
            album_cover_minithumbnail
        )
        r"""The minithumbnail of the album cover; may be null"""
        self.album_cover_thumbnail: Union[Thumbnail, None] = album_cover_thumbnail
        r"""The thumbnail of the album cover in JPEG format; as defined by the sender\. The full size thumbnail is expected to be extracted from the downloaded audio file; may be null"""
        self.external_album_covers: List[Thumbnail] = external_album_covers or []
        r"""Album cover variants to use if the downloaded audio file contains no album cover\. Provided thumbnail dimensions are approximate"""
        self.audio: Union[File, None] = audio
        r"""File containing the audio"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["audio"]:
        return "audio"

    @classmethod
    def getClass(self) -> Literal["Audio"]:
        return "Audio"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "duration": self.duration,
            "title": self.title,
            "performer": self.performer,
            "file_name": self.file_name,
            "mime_type": self.mime_type,
            "album_cover_minithumbnail": self.album_cover_minithumbnail,
            "album_cover_thumbnail": self.album_cover_thumbnail,
            "external_album_covers": self.external_album_covers,
            "audio": self.audio,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Audio", None]:
        if data:
            data_class = cls()
            data_class.duration = int(data.get("duration", 0))
            data_class.title = data.get("title", "")
            data_class.performer = data.get("performer", "")
            data_class.file_name = data.get("file_name", "")
            data_class.mime_type = data.get("mime_type", "")
            data_class.album_cover_minithumbnail = data.get(
                "album_cover_minithumbnail", None
            )
            data_class.album_cover_thumbnail = data.get("album_cover_thumbnail", None)
            data_class.external_album_covers = data.get("external_album_covers", None)
            data_class.audio = data.get("audio", None)

        return data_class


class Audios(TlObject):
    r"""Contains a list of audio files

    Parameters:
        total_count (:class:`int`):
            Approximate total number of audio files found

        audios (List[:class:`~pytdbot.types.Audio`]):
            List of audio files

    """

    def __init__(self, total_count: int = 0, audios: List[Audio] = None) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of audio files found"""
        self.audios: List[Audio] = audios or []
        r"""List of audio files"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["audios"]:
        return "audios"

    @classmethod
    def getClass(self) -> Literal["Audios"]:
        return "Audios"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "audios": self.audios,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Audios", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.audios = data.get("audios", None)

        return data_class


class Document(TlObject):
    r"""Describes a document of any type

    Parameters:
        file_name (:class:`str`):
            Original name of the file; as defined by the sender

        mime_type (:class:`str`):
            MIME type of the file; as defined by the sender

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Document minithumbnail; may be null

        thumbnail (:class:`~pytdbot.types.Thumbnail`):
            Document thumbnail in JPEG or PNG format \(PNG will be used only for background patterns\); as defined by the sender; may be null

        document (:class:`~pytdbot.types.File`):
            File containing the document

    """

    def __init__(
        self,
        file_name: str = "",
        mime_type: str = "",
        minithumbnail: Minithumbnail = None,
        thumbnail: Thumbnail = None,
        document: File = None,
    ) -> None:
        self.file_name: Union[str, None] = file_name
        r"""Original name of the file; as defined by the sender"""
        self.mime_type: Union[str, None] = mime_type
        r"""MIME type of the file; as defined by the sender"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Document minithumbnail; may be null"""
        self.thumbnail: Union[Thumbnail, None] = thumbnail
        r"""Document thumbnail in JPEG or PNG format \(PNG will be used only for background patterns\); as defined by the sender; may be null"""
        self.document: Union[File, None] = document
        r"""File containing the document"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["document"]:
        return "document"

    @classmethod
    def getClass(self) -> Literal["Document"]:
        return "Document"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "file_name": self.file_name,
            "mime_type": self.mime_type,
            "minithumbnail": self.minithumbnail,
            "thumbnail": self.thumbnail,
            "document": self.document,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Document", None]:
        if data:
            data_class = cls()
            data_class.file_name = data.get("file_name", "")
            data_class.mime_type = data.get("mime_type", "")
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.thumbnail = data.get("thumbnail", None)
            data_class.document = data.get("document", None)

        return data_class


class Photo(TlObject):
    r"""Describes a photo

    Parameters:
        has_stickers (:class:`bool`):
            True, if stickers were added to the photo\. The list of corresponding sticker sets can be received using getAttachedStickerSets

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Photo minithumbnail; may be null

        sizes (List[:class:`~pytdbot.types.PhotoSize`]):
            Available variants of the photo, in different sizes

    """

    def __init__(
        self,
        has_stickers: bool = False,
        minithumbnail: Minithumbnail = None,
        sizes: List[PhotoSize] = None,
    ) -> None:
        self.has_stickers: bool = bool(has_stickers)
        r"""True, if stickers were added to the photo\. The list of corresponding sticker sets can be received using getAttachedStickerSets"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Photo minithumbnail; may be null"""
        self.sizes: List[PhotoSize] = sizes or []
        r"""Available variants of the photo, in different sizes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["photo"]:
        return "photo"

    @classmethod
    def getClass(self) -> Literal["Photo"]:
        return "Photo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "has_stickers": self.has_stickers,
            "minithumbnail": self.minithumbnail,
            "sizes": self.sizes,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Photo", None]:
        if data:
            data_class = cls()
            data_class.has_stickers = data.get("has_stickers", False)
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.sizes = data.get("sizes", None)

        return data_class


class Sticker(TlObject):
    r"""Describes a sticker

    Parameters:
        id (:class:`int`):
            Unique sticker identifier within the set; 0 if none

        set_id (:class:`int`):
            Identifier of the sticker set to which the sticker belongs; 0 if none

        width (:class:`int`):
            Sticker width; as defined by the sender

        height (:class:`int`):
            Sticker height; as defined by the sender

        emoji (:class:`str`):
            Emoji corresponding to the sticker

        format (:class:`~pytdbot.types.StickerFormat`):
            Sticker format

        full_type (:class:`~pytdbot.types.StickerFullType`):
            Sticker's full type

        thumbnail (:class:`~pytdbot.types.Thumbnail`):
            Sticker thumbnail in WEBP or JPEG format; may be null

        sticker (:class:`~pytdbot.types.File`):
            File containing the sticker

    """

    def __init__(
        self,
        id: int = 0,
        set_id: int = 0,
        width: int = 0,
        height: int = 0,
        emoji: str = "",
        format: StickerFormat = None,
        full_type: StickerFullType = None,
        thumbnail: Thumbnail = None,
        sticker: File = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique sticker identifier within the set; 0 if none"""
        self.set_id: int = int(set_id)
        r"""Identifier of the sticker set to which the sticker belongs; 0 if none"""
        self.width: int = int(width)
        r"""Sticker width; as defined by the sender"""
        self.height: int = int(height)
        r"""Sticker height; as defined by the sender"""
        self.emoji: Union[str, None] = emoji
        r"""Emoji corresponding to the sticker"""
        self.format: Union[
            StickerFormatWebp, StickerFormatTgs, StickerFormatWebm, None
        ] = format
        r"""Sticker format"""
        self.full_type: Union[
            StickerFullTypeRegular,
            StickerFullTypeMask,
            StickerFullTypeCustomEmoji,
            None,
        ] = full_type
        r"""Sticker's full type"""
        self.thumbnail: Union[Thumbnail, None] = thumbnail
        r"""Sticker thumbnail in WEBP or JPEG format; may be null"""
        self.sticker: Union[File, None] = sticker
        r"""File containing the sticker"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sticker"]:
        return "sticker"

    @classmethod
    def getClass(self) -> Literal["Sticker"]:
        return "Sticker"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "set_id": self.set_id,
            "width": self.width,
            "height": self.height,
            "emoji": self.emoji,
            "format": self.format,
            "full_type": self.full_type,
            "thumbnail": self.thumbnail,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Sticker", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.set_id = int(data.get("set_id", 0))
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.emoji = data.get("emoji", "")
            data_class.format = data.get("format", None)
            data_class.full_type = data.get("full_type", None)
            data_class.thumbnail = data.get("thumbnail", None)
            data_class.sticker = data.get("sticker", None)

        return data_class


class Video(TlObject):
    r"""Describes a video file

    Parameters:
        duration (:class:`int`):
            Duration of the video, in seconds; as defined by the sender

        width (:class:`int`):
            Video width; as defined by the sender

        height (:class:`int`):
            Video height; as defined by the sender

        file_name (:class:`str`):
            Original name of the file; as defined by the sender

        mime_type (:class:`str`):
            MIME type of the file; as defined by the sender

        has_stickers (:class:`bool`):
            True, if stickers were added to the video\. The list of corresponding sticker sets can be received using getAttachedStickerSets

        supports_streaming (:class:`bool`):
            True, if the video is expected to be streamed

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Video minithumbnail; may be null

        thumbnail (:class:`~pytdbot.types.Thumbnail`):
            Video thumbnail in JPEG or MPEG4 format; as defined by the sender; may be null

        video (:class:`~pytdbot.types.File`):
            File containing the video

    """

    def __init__(
        self,
        duration: int = 0,
        width: int = 0,
        height: int = 0,
        file_name: str = "",
        mime_type: str = "",
        has_stickers: bool = False,
        supports_streaming: bool = False,
        minithumbnail: Minithumbnail = None,
        thumbnail: Thumbnail = None,
        video: File = None,
    ) -> None:
        self.duration: int = int(duration)
        r"""Duration of the video, in seconds; as defined by the sender"""
        self.width: int = int(width)
        r"""Video width; as defined by the sender"""
        self.height: int = int(height)
        r"""Video height; as defined by the sender"""
        self.file_name: Union[str, None] = file_name
        r"""Original name of the file; as defined by the sender"""
        self.mime_type: Union[str, None] = mime_type
        r"""MIME type of the file; as defined by the sender"""
        self.has_stickers: bool = bool(has_stickers)
        r"""True, if stickers were added to the video\. The list of corresponding sticker sets can be received using getAttachedStickerSets"""
        self.supports_streaming: bool = bool(supports_streaming)
        r"""True, if the video is expected to be streamed"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Video minithumbnail; may be null"""
        self.thumbnail: Union[Thumbnail, None] = thumbnail
        r"""Video thumbnail in JPEG or MPEG4 format; as defined by the sender; may be null"""
        self.video: Union[File, None] = video
        r"""File containing the video"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["video"]:
        return "video"

    @classmethod
    def getClass(self) -> Literal["Video"]:
        return "Video"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "duration": self.duration,
            "width": self.width,
            "height": self.height,
            "file_name": self.file_name,
            "mime_type": self.mime_type,
            "has_stickers": self.has_stickers,
            "supports_streaming": self.supports_streaming,
            "minithumbnail": self.minithumbnail,
            "thumbnail": self.thumbnail,
            "video": self.video,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Video", None]:
        if data:
            data_class = cls()
            data_class.duration = int(data.get("duration", 0))
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.file_name = data.get("file_name", "")
            data_class.mime_type = data.get("mime_type", "")
            data_class.has_stickers = data.get("has_stickers", False)
            data_class.supports_streaming = data.get("supports_streaming", False)
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.thumbnail = data.get("thumbnail", None)
            data_class.video = data.get("video", None)

        return data_class


class VideoNote(TlObject):
    r"""Describes a video note\. The video must be equal in width and height, cropped to a circle, and stored in MPEG4 format

    Parameters:
        duration (:class:`int`):
            Duration of the video, in seconds; as defined by the sender

        waveform (:class:`bytes`):
            A waveform representation of the video note's audio in 5\-bit format; may be empty if unknown

        length (:class:`int`):
            Video width and height; as defined by the sender

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Video minithumbnail; may be null

        thumbnail (:class:`~pytdbot.types.Thumbnail`):
            Video thumbnail in JPEG format; as defined by the sender; may be null

        speech_recognition_result (:class:`~pytdbot.types.SpeechRecognitionResult`):
            Result of speech recognition in the video note; may be null

        video (:class:`~pytdbot.types.File`):
            File containing the video

    """

    def __init__(
        self,
        duration: int = 0,
        waveform: bytes = b"",
        length: int = 0,
        minithumbnail: Minithumbnail = None,
        thumbnail: Thumbnail = None,
        speech_recognition_result: SpeechRecognitionResult = None,
        video: File = None,
    ) -> None:
        self.duration: int = int(duration)
        r"""Duration of the video, in seconds; as defined by the sender"""
        self.waveform: Union[bytes, None] = waveform
        r"""A waveform representation of the video note's audio in 5\-bit format; may be empty if unknown"""
        self.length: int = int(length)
        r"""Video width and height; as defined by the sender"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Video minithumbnail; may be null"""
        self.thumbnail: Union[Thumbnail, None] = thumbnail
        r"""Video thumbnail in JPEG format; as defined by the sender; may be null"""
        self.speech_recognition_result: Union[
            SpeechRecognitionResultPending,
            SpeechRecognitionResultText,
            SpeechRecognitionResultError,
            None,
        ] = speech_recognition_result
        r"""Result of speech recognition in the video note; may be null"""
        self.video: Union[File, None] = video
        r"""File containing the video"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["videoNote"]:
        return "videoNote"

    @classmethod
    def getClass(self) -> Literal["VideoNote"]:
        return "VideoNote"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "duration": self.duration,
            "waveform": self.waveform,
            "length": self.length,
            "minithumbnail": self.minithumbnail,
            "thumbnail": self.thumbnail,
            "speech_recognition_result": self.speech_recognition_result,
            "video": self.video,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["VideoNote", None]:
        if data:
            data_class = cls()
            data_class.duration = int(data.get("duration", 0))
            data_class.waveform = b64decode(data.get("waveform", b""))
            data_class.length = int(data.get("length", 0))
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.thumbnail = data.get("thumbnail", None)
            data_class.speech_recognition_result = data.get(
                "speech_recognition_result", None
            )
            data_class.video = data.get("video", None)

        return data_class


class VoiceNote(TlObject):
    r"""Describes a voice note

    Parameters:
        duration (:class:`int`):
            Duration of the voice note, in seconds; as defined by the sender

        waveform (:class:`bytes`):
            A waveform representation of the voice note in 5\-bit format

        mime_type (:class:`str`):
            MIME type of the file; as defined by the sender\. Usually, one of \"audio/ogg\" for Opus in an OGG container, \"audio/mpeg\" for an MP3 audio, or \"audio/mp4\" for an M4A audio

        speech_recognition_result (:class:`~pytdbot.types.SpeechRecognitionResult`):
            Result of speech recognition in the voice note; may be null

        voice (:class:`~pytdbot.types.File`):
            File containing the voice note

    """

    def __init__(
        self,
        duration: int = 0,
        waveform: bytes = b"",
        mime_type: str = "",
        speech_recognition_result: SpeechRecognitionResult = None,
        voice: File = None,
    ) -> None:
        self.duration: int = int(duration)
        r"""Duration of the voice note, in seconds; as defined by the sender"""
        self.waveform: Union[bytes, None] = waveform
        r"""A waveform representation of the voice note in 5\-bit format"""
        self.mime_type: Union[str, None] = mime_type
        r"""MIME type of the file; as defined by the sender\. Usually, one of \"audio/ogg\" for Opus in an OGG container, \"audio/mpeg\" for an MP3 audio, or \"audio/mp4\" for an M4A audio"""
        self.speech_recognition_result: Union[
            SpeechRecognitionResultPending,
            SpeechRecognitionResultText,
            SpeechRecognitionResultError,
            None,
        ] = speech_recognition_result
        r"""Result of speech recognition in the voice note; may be null"""
        self.voice: Union[File, None] = voice
        r"""File containing the voice note"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["voiceNote"]:
        return "voiceNote"

    @classmethod
    def getClass(self) -> Literal["VoiceNote"]:
        return "VoiceNote"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "duration": self.duration,
            "waveform": self.waveform,
            "mime_type": self.mime_type,
            "speech_recognition_result": self.speech_recognition_result,
            "voice": self.voice,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["VoiceNote", None]:
        if data:
            data_class = cls()
            data_class.duration = int(data.get("duration", 0))
            data_class.waveform = b64decode(data.get("waveform", b""))
            data_class.mime_type = data.get("mime_type", "")
            data_class.speech_recognition_result = data.get(
                "speech_recognition_result", None
            )
            data_class.voice = data.get("voice", None)

        return data_class


class AnimatedEmoji(TlObject):
    r"""Describes an animated or custom representation of an emoji

    Parameters:
        sticker (:class:`~pytdbot.types.Sticker`):
            Sticker for the emoji; may be null if yet unknown for a custom emoji\. If the sticker is a custom emoji, then it can have arbitrary format

        sticker_width (:class:`int`):
            Expected width of the sticker, which can be used if the sticker is null

        sticker_height (:class:`int`):
            Expected height of the sticker, which can be used if the sticker is null

        fitzpatrick_type (:class:`int`):
            Emoji modifier fitzpatrick type; 0\-6; 0 if none

        sound (:class:`~pytdbot.types.File`):
            File containing the sound to be played when the sticker is clicked; may be null\. The sound is encoded with the Opus codec, and stored inside an OGG container

    """

    def __init__(
        self,
        sticker: Sticker = None,
        sticker_width: int = 0,
        sticker_height: int = 0,
        fitzpatrick_type: int = 0,
        sound: File = None,
    ) -> None:
        self.sticker: Union[Sticker, None] = sticker
        r"""Sticker for the emoji; may be null if yet unknown for a custom emoji\. If the sticker is a custom emoji, then it can have arbitrary format"""
        self.sticker_width: int = int(sticker_width)
        r"""Expected width of the sticker, which can be used if the sticker is null"""
        self.sticker_height: int = int(sticker_height)
        r"""Expected height of the sticker, which can be used if the sticker is null"""
        self.fitzpatrick_type: int = int(fitzpatrick_type)
        r"""Emoji modifier fitzpatrick type; 0\-6; 0 if none"""
        self.sound: Union[File, None] = sound
        r"""File containing the sound to be played when the sticker is clicked; may be null\. The sound is encoded with the Opus codec, and stored inside an OGG container"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["animatedEmoji"]:
        return "animatedEmoji"

    @classmethod
    def getClass(self) -> Literal["AnimatedEmoji"]:
        return "AnimatedEmoji"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sticker": self.sticker,
            "sticker_width": self.sticker_width,
            "sticker_height": self.sticker_height,
            "fitzpatrick_type": self.fitzpatrick_type,
            "sound": self.sound,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AnimatedEmoji", None]:
        if data:
            data_class = cls()
            data_class.sticker = data.get("sticker", None)
            data_class.sticker_width = int(data.get("sticker_width", 0))
            data_class.sticker_height = int(data.get("sticker_height", 0))
            data_class.fitzpatrick_type = int(data.get("fitzpatrick_type", 0))
            data_class.sound = data.get("sound", None)

        return data_class


class Contact(TlObject):
    r"""Describes a contact of a user

    Parameters:
        phone_number (:class:`str`):
            Phone number of the user

        first_name (:class:`str`):
            First name of the user; 1\-64 characters

        last_name (:class:`str`):
            Last name of the user; 0\-64 characters

        vcard (:class:`str`):
            Additional data about the user in a form of vCard; 0\-2048 bytes in length

        user_id (:class:`int`):
            Identifier of the user, if known; 0 otherwise

    """

    def __init__(
        self,
        phone_number: str = "",
        first_name: str = "",
        last_name: str = "",
        vcard: str = "",
        user_id: int = 0,
    ) -> None:
        self.phone_number: Union[str, None] = phone_number
        r"""Phone number of the user"""
        self.first_name: Union[str, None] = first_name
        r"""First name of the user; 1\-64 characters"""
        self.last_name: Union[str, None] = last_name
        r"""Last name of the user; 0\-64 characters"""
        self.vcard: Union[str, None] = vcard
        r"""Additional data about the user in a form of vCard; 0\-2048 bytes in length"""
        self.user_id: int = int(user_id)
        r"""Identifier of the user, if known; 0 otherwise"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["contact"]:
        return "contact"

    @classmethod
    def getClass(self) -> Literal["Contact"]:
        return "Contact"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "phone_number": self.phone_number,
            "first_name": self.first_name,
            "last_name": self.last_name,
            "vcard": self.vcard,
            "user_id": self.user_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Contact", None]:
        if data:
            data_class = cls()
            data_class.phone_number = data.get("phone_number", "")
            data_class.first_name = data.get("first_name", "")
            data_class.last_name = data.get("last_name", "")
            data_class.vcard = data.get("vcard", "")
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class Location(TlObject):
    r"""Describes a location on planet Earth

    Parameters:
        latitude (:class:`float`):
            Latitude of the location in degrees; as defined by the sender

        longitude (:class:`float`):
            Longitude of the location, in degrees; as defined by the sender

        horizontal_accuracy (:class:`float`):
            The estimated horizontal accuracy of the location, in meters; as defined by the sender\. 0 if unknown

    """

    def __init__(
        self,
        latitude: float = 0.0,
        longitude: float = 0.0,
        horizontal_accuracy: float = 0.0,
    ) -> None:
        self.latitude: float = float(latitude)
        r"""Latitude of the location in degrees; as defined by the sender"""
        self.longitude: float = float(longitude)
        r"""Longitude of the location, in degrees; as defined by the sender"""
        self.horizontal_accuracy: float = float(horizontal_accuracy)
        r"""The estimated horizontal accuracy of the location, in meters; as defined by the sender\. 0 if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["location"]:
        return "location"

    @classmethod
    def getClass(self) -> Literal["Location"]:
        return "Location"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "latitude": self.latitude,
            "longitude": self.longitude,
            "horizontal_accuracy": self.horizontal_accuracy,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Location", None]:
        if data:
            data_class = cls()
            data_class.latitude = data.get("latitude", 0.0)
            data_class.longitude = data.get("longitude", 0.0)
            data_class.horizontal_accuracy = data.get("horizontal_accuracy", 0.0)

        return data_class


class Venue(TlObject):
    r"""Describes a venue

    Parameters:
        location (:class:`~pytdbot.types.Location`):
            Venue location; as defined by the sender

        title (:class:`str`):
            Venue name; as defined by the sender

        address (:class:`str`):
            Venue address; as defined by the sender

        provider (:class:`str`):
            Provider of the venue database; as defined by the sender\. Currently, only \"foursquare\" and \"gplaces\" \(Google Places\) need to be supported

        id (:class:`str`):
            Identifier of the venue in the provider database; as defined by the sender

        type (:class:`str`):
            Type of the venue in the provider database; as defined by the sender

    """

    def __init__(
        self,
        location: Location = None,
        title: str = "",
        address: str = "",
        provider: str = "",
        id: str = "",
        type: str = "",
    ) -> None:
        self.location: Union[Location, None] = location
        r"""Venue location; as defined by the sender"""
        self.title: Union[str, None] = title
        r"""Venue name; as defined by the sender"""
        self.address: Union[str, None] = address
        r"""Venue address; as defined by the sender"""
        self.provider: Union[str, None] = provider
        r"""Provider of the venue database; as defined by the sender\. Currently, only \"foursquare\" and \"gplaces\" \(Google Places\) need to be supported"""
        self.id: Union[str, None] = id
        r"""Identifier of the venue in the provider database; as defined by the sender"""
        self.type: Union[str, None] = type
        r"""Type of the venue in the provider database; as defined by the sender"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["venue"]:
        return "venue"

    @classmethod
    def getClass(self) -> Literal["Venue"]:
        return "Venue"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "location": self.location,
            "title": self.title,
            "address": self.address,
            "provider": self.provider,
            "id": self.id,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Venue", None]:
        if data:
            data_class = cls()
            data_class.location = data.get("location", None)
            data_class.title = data.get("title", "")
            data_class.address = data.get("address", "")
            data_class.provider = data.get("provider", "")
            data_class.id = data.get("id", "")
            data_class.type = data.get("type", "")

        return data_class


class Game(TlObject):
    r"""Describes a game\. Use getInternalLink with internalLinkTypeGame to share the game

    Parameters:
        id (:class:`int`):
            Unique game identifier

        short_name (:class:`str`):
            Game short name

        title (:class:`str`):
            Game title

        text (:class:`~pytdbot.types.FormattedText`):
            Game text, usually containing scoreboards for a game

        description (:class:`str`):
            Game description

        photo (:class:`~pytdbot.types.Photo`):
            Game photo

        animation (:class:`~pytdbot.types.Animation`):
            Game animation; may be null

    """

    def __init__(
        self,
        id: int = 0,
        short_name: str = "",
        title: str = "",
        text: FormattedText = None,
        description: str = "",
        photo: Photo = None,
        animation: Animation = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique game identifier"""
        self.short_name: Union[str, None] = short_name
        r"""Game short name"""
        self.title: Union[str, None] = title
        r"""Game title"""
        self.text: Union[FormattedText, None] = text
        r"""Game text, usually containing scoreboards for a game"""
        self.description: Union[str, None] = description
        r"""Game description"""
        self.photo: Union[Photo, None] = photo
        r"""Game photo"""
        self.animation: Union[Animation, None] = animation
        r"""Game animation; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["game"]:
        return "game"

    @classmethod
    def getClass(self) -> Literal["Game"]:
        return "Game"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "short_name": self.short_name,
            "title": self.title,
            "text": self.text,
            "description": self.description,
            "photo": self.photo,
            "animation": self.animation,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Game", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.short_name = data.get("short_name", "")
            data_class.title = data.get("title", "")
            data_class.text = data.get("text", None)
            data_class.description = data.get("description", "")
            data_class.photo = data.get("photo", None)
            data_class.animation = data.get("animation", None)

        return data_class


class WebApp(TlObject):
    r"""Describes a Web App\. Use getInternalLink with internalLinkTypeWebApp to share the Web App

    Parameters:
        short_name (:class:`str`):
            Web App short name

        title (:class:`str`):
            Web App title

        description (:class:`str`):
            Web App description

        photo (:class:`~pytdbot.types.Photo`):
            Web App photo

        animation (:class:`~pytdbot.types.Animation`):
            Web App animation; may be null

    """

    def __init__(
        self,
        short_name: str = "",
        title: str = "",
        description: str = "",
        photo: Photo = None,
        animation: Animation = None,
    ) -> None:
        self.short_name: Union[str, None] = short_name
        r"""Web App short name"""
        self.title: Union[str, None] = title
        r"""Web App title"""
        self.description: Union[str, None] = description
        r"""Web App description"""
        self.photo: Union[Photo, None] = photo
        r"""Web App photo"""
        self.animation: Union[Animation, None] = animation
        r"""Web App animation; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["webApp"]:
        return "webApp"

    @classmethod
    def getClass(self) -> Literal["WebApp"]:
        return "WebApp"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "short_name": self.short_name,
            "title": self.title,
            "description": self.description,
            "photo": self.photo,
            "animation": self.animation,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["WebApp", None]:
        if data:
            data_class = cls()
            data_class.short_name = data.get("short_name", "")
            data_class.title = data.get("title", "")
            data_class.description = data.get("description", "")
            data_class.photo = data.get("photo", None)
            data_class.animation = data.get("animation", None)

        return data_class


class Poll(TlObject):
    r"""Describes a poll

    Parameters:
        id (:class:`int`):
            Unique poll identifier

        question (:class:`~pytdbot.types.FormattedText`):
            Poll question; 1\-300 characters\. Only custom emoji entities are allowed

        options (List[:class:`~pytdbot.types.PollOption`]):
            List of poll answer options

        total_voter_count (:class:`int`):
            Total number of voters, participating in the poll

        recent_voter_ids (List[:class:`~pytdbot.types.MessageSender`]):
            Identifiers of recent voters, if the poll is non\-anonymous

        is_anonymous (:class:`bool`):
            True, if the poll is anonymous

        type (:class:`~pytdbot.types.PollType`):
            Type of the poll

        open_period (:class:`int`):
            Amount of time the poll will be active after creation, in seconds

        close_date (:class:`int`):
            Point in time \(Unix timestamp\) when the poll will automatically be closed

        is_closed (:class:`bool`):
            True, if the poll is closed

    """

    def __init__(
        self,
        id: int = 0,
        question: FormattedText = None,
        options: List[PollOption] = None,
        total_voter_count: int = 0,
        recent_voter_ids: List[MessageSender] = None,
        is_anonymous: bool = False,
        type: PollType = None,
        open_period: int = 0,
        close_date: int = 0,
        is_closed: bool = False,
    ) -> None:
        self.id: int = int(id)
        r"""Unique poll identifier"""
        self.question: Union[FormattedText, None] = question
        r"""Poll question; 1\-300 characters\. Only custom emoji entities are allowed"""
        self.options: List[PollOption] = options or []
        r"""List of poll answer options"""
        self.total_voter_count: int = int(total_voter_count)
        r"""Total number of voters, participating in the poll"""
        self.recent_voter_ids: List[MessageSender] = recent_voter_ids or []
        r"""Identifiers of recent voters, if the poll is non\-anonymous"""
        self.is_anonymous: bool = bool(is_anonymous)
        r"""True, if the poll is anonymous"""
        self.type: Union[PollTypeRegular, PollTypeQuiz, None] = type
        r"""Type of the poll"""
        self.open_period: int = int(open_period)
        r"""Amount of time the poll will be active after creation, in seconds"""
        self.close_date: int = int(close_date)
        r"""Point in time \(Unix timestamp\) when the poll will automatically be closed"""
        self.is_closed: bool = bool(is_closed)
        r"""True, if the poll is closed"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["poll"]:
        return "poll"

    @classmethod
    def getClass(self) -> Literal["Poll"]:
        return "Poll"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "question": self.question,
            "options": self.options,
            "total_voter_count": self.total_voter_count,
            "recent_voter_ids": self.recent_voter_ids,
            "is_anonymous": self.is_anonymous,
            "type": self.type,
            "open_period": self.open_period,
            "close_date": self.close_date,
            "is_closed": self.is_closed,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Poll", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.question = data.get("question", None)
            data_class.options = data.get("options", None)
            data_class.total_voter_count = int(data.get("total_voter_count", 0))
            data_class.recent_voter_ids = data.get("recent_voter_ids", None)
            data_class.is_anonymous = data.get("is_anonymous", False)
            data_class.type = data.get("type", None)
            data_class.open_period = int(data.get("open_period", 0))
            data_class.close_date = int(data.get("close_date", 0))
            data_class.is_closed = data.get("is_closed", False)

        return data_class


class AlternativeVideo(TlObject):
    r"""Describes an alternative re\-encoded quality of a video file

    Parameters:
        id (:class:`int`):
            Unique identifier of the alternative video, which is used in the HLS file

        width (:class:`int`):
            Video width

        height (:class:`int`):
            Video height

        codec (:class:`str`):
            Codec used for video file encoding, for example, \"h264\", \"h265\", or \"av1\"

        hls_file (:class:`~pytdbot.types.File`):
            HLS file describing the video

        video (:class:`~pytdbot.types.File`):
            File containing the video

    """

    def __init__(
        self,
        id: int = 0,
        width: int = 0,
        height: int = 0,
        codec: str = "",
        hls_file: File = None,
        video: File = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the alternative video, which is used in the HLS file"""
        self.width: int = int(width)
        r"""Video width"""
        self.height: int = int(height)
        r"""Video height"""
        self.codec: Union[str, None] = codec
        r"""Codec used for video file encoding, for example, \"h264\", \"h265\", or \"av1\""""
        self.hls_file: Union[File, None] = hls_file
        r"""HLS file describing the video"""
        self.video: Union[File, None] = video
        r"""File containing the video"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["alternativeVideo"]:
        return "alternativeVideo"

    @classmethod
    def getClass(self) -> Literal["AlternativeVideo"]:
        return "AlternativeVideo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "width": self.width,
            "height": self.height,
            "codec": self.codec,
            "hls_file": self.hls_file,
            "video": self.video,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AlternativeVideo", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.codec = data.get("codec", "")
            data_class.hls_file = data.get("hls_file", None)
            data_class.video = data.get("video", None)

        return data_class


class VideoStoryboard(TlObject):
    r"""Describes a storyboard for a video

    Parameters:
        storyboard_file (:class:`~pytdbot.types.File`):
            A JPEG file that contains tiled previews of video

        width (:class:`int`):
            Width of a tile

        height (:class:`int`):
            Height of a tile

        map_file (:class:`~pytdbot.types.File`):
            File that describes mapping of position in the video to a tile in the JPEG file

    """

    def __init__(
        self,
        storyboard_file: File = None,
        width: int = 0,
        height: int = 0,
        map_file: File = None,
    ) -> None:
        self.storyboard_file: Union[File, None] = storyboard_file
        r"""A JPEG file that contains tiled previews of video"""
        self.width: int = int(width)
        r"""Width of a tile"""
        self.height: int = int(height)
        r"""Height of a tile"""
        self.map_file: Union[File, None] = map_file
        r"""File that describes mapping of position in the video to a tile in the JPEG file"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["videoStoryboard"]:
        return "videoStoryboard"

    @classmethod
    def getClass(self) -> Literal["VideoStoryboard"]:
        return "VideoStoryboard"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "storyboard_file": self.storyboard_file,
            "width": self.width,
            "height": self.height,
            "map_file": self.map_file,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["VideoStoryboard", None]:
        if data:
            data_class = cls()
            data_class.storyboard_file = data.get("storyboard_file", None)
            data_class.width = int(data.get("width", 0))
            data_class.height = int(data.get("height", 0))
            data_class.map_file = data.get("map_file", None)

        return data_class


class Background(TlObject):
    r"""Describes a chat background

    Parameters:
        id (:class:`int`):
            Unique background identifier

        is_default (:class:`bool`):
            True, if this is one of default backgrounds

        is_dark (:class:`bool`):
            True, if the background is dark and is recommended to be used with dark theme

        name (:class:`str`):
            Unique background name

        document (:class:`~pytdbot.types.Document`):
            Document with the background; may be null\. Null only for filled and chat theme backgrounds

        type (:class:`~pytdbot.types.BackgroundType`):
            Type of the background

    """

    def __init__(
        self,
        id: int = 0,
        is_default: bool = False,
        is_dark: bool = False,
        name: str = "",
        document: Document = None,
        type: BackgroundType = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique background identifier"""
        self.is_default: bool = bool(is_default)
        r"""True, if this is one of default backgrounds"""
        self.is_dark: bool = bool(is_dark)
        r"""True, if the background is dark and is recommended to be used with dark theme"""
        self.name: Union[str, None] = name
        r"""Unique background name"""
        self.document: Union[Document, None] = document
        r"""Document with the background; may be null\. Null only for filled and chat theme backgrounds"""
        self.type: Union[
            BackgroundTypeWallpaper,
            BackgroundTypePattern,
            BackgroundTypeFill,
            BackgroundTypeChatTheme,
            None,
        ] = type
        r"""Type of the background"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["background"]:
        return "background"

    @classmethod
    def getClass(self) -> Literal["Background"]:
        return "Background"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "is_default": self.is_default,
            "is_dark": self.is_dark,
            "name": self.name,
            "document": self.document,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Background", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.is_default = data.get("is_default", False)
            data_class.is_dark = data.get("is_dark", False)
            data_class.name = data.get("name", "")
            data_class.document = data.get("document", None)
            data_class.type = data.get("type", None)

        return data_class


class Backgrounds(TlObject):
    r"""Contains a list of backgrounds

    Parameters:
        backgrounds (List[:class:`~pytdbot.types.Background`]):
            A list of backgrounds

    """

    def __init__(self, backgrounds: List[Background] = None) -> None:
        self.backgrounds: List[Background] = backgrounds or []
        r"""A list of backgrounds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["backgrounds"]:
        return "backgrounds"

    @classmethod
    def getClass(self) -> Literal["Backgrounds"]:
        return "Backgrounds"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "backgrounds": self.backgrounds}

    @classmethod
    def from_dict(cls, data: dict) -> Union["Backgrounds", None]:
        if data:
            data_class = cls()
            data_class.backgrounds = data.get("backgrounds", None)

        return data_class


class ChatBackground(TlObject):
    r"""Describes a background set for a specific chat

    Parameters:
        background (:class:`~pytdbot.types.Background`):
            The background

        dark_theme_dimming (:class:`int`):
            Dimming of the background in dark themes, as a percentage; 0\-100\. Applied only to Wallpaper and Fill types of background

    """

    def __init__(
        self, background: Background = None, dark_theme_dimming: int = 0
    ) -> None:
        self.background: Union[Background, None] = background
        r"""The background"""
        self.dark_theme_dimming: int = int(dark_theme_dimming)
        r"""Dimming of the background in dark themes, as a percentage; 0\-100\. Applied only to Wallpaper and Fill types of background"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatBackground"]:
        return "chatBackground"

    @classmethod
    def getClass(self) -> Literal["ChatBackground"]:
        return "ChatBackground"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "background": self.background,
            "dark_theme_dimming": self.dark_theme_dimming,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatBackground", None]:
        if data:
            data_class = cls()
            data_class.background = data.get("background", None)
            data_class.dark_theme_dimming = int(data.get("dark_theme_dimming", 0))

        return data_class


class ProfilePhoto(TlObject):
    r"""Describes a user profile photo

    Parameters:
        id (:class:`int`):
            Photo identifier; 0 for an empty photo\. Can be used to find a photo in a list of user profile photos

        small (:class:`~pytdbot.types.File`):
            A small \(160x160\) user profile photo\. The file can be downloaded only before the photo is changed

        big (:class:`~pytdbot.types.File`):
            A big \(640x640\) user profile photo\. The file can be downloaded only before the photo is changed

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            User profile photo minithumbnail; may be null

        has_animation (:class:`bool`):
            True, if the photo has animated variant

        is_personal (:class:`bool`):
            True, if the photo is visible only for the current user

    """

    def __init__(
        self,
        id: int = 0,
        small: File = None,
        big: File = None,
        minithumbnail: Minithumbnail = None,
        has_animation: bool = False,
        is_personal: bool = False,
    ) -> None:
        self.id: int = int(id)
        r"""Photo identifier; 0 for an empty photo\. Can be used to find a photo in a list of user profile photos"""
        self.small: Union[File, None] = small
        r"""A small \(160x160\) user profile photo\. The file can be downloaded only before the photo is changed"""
        self.big: Union[File, None] = big
        r"""A big \(640x640\) user profile photo\. The file can be downloaded only before the photo is changed"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""User profile photo minithumbnail; may be null"""
        self.has_animation: bool = bool(has_animation)
        r"""True, if the photo has animated variant"""
        self.is_personal: bool = bool(is_personal)
        r"""True, if the photo is visible only for the current user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profilePhoto"]:
        return "profilePhoto"

    @classmethod
    def getClass(self) -> Literal["ProfilePhoto"]:
        return "ProfilePhoto"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "small": self.small,
            "big": self.big,
            "minithumbnail": self.minithumbnail,
            "has_animation": self.has_animation,
            "is_personal": self.is_personal,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfilePhoto", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.small = data.get("small", None)
            data_class.big = data.get("big", None)
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.has_animation = data.get("has_animation", False)
            data_class.is_personal = data.get("is_personal", False)

        return data_class


class ChatPhotoInfo(TlObject):
    r"""Contains basic information about the photo of a chat

    Parameters:
        small (:class:`~pytdbot.types.File`):
            A small \(160x160\) chat photo variant in JPEG format\. The file can be downloaded only before the photo is changed

        big (:class:`~pytdbot.types.File`):
            A big \(640x640\) chat photo variant in JPEG format\. The file can be downloaded only before the photo is changed

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Chat photo minithumbnail; may be null

        has_animation (:class:`bool`):
            True, if the photo has animated variant

        is_personal (:class:`bool`):
            True, if the photo is visible only for the current user

    """

    def __init__(
        self,
        small: File = None,
        big: File = None,
        minithumbnail: Minithumbnail = None,
        has_animation: bool = False,
        is_personal: bool = False,
    ) -> None:
        self.small: Union[File, None] = small
        r"""A small \(160x160\) chat photo variant in JPEG format\. The file can be downloaded only before the photo is changed"""
        self.big: Union[File, None] = big
        r"""A big \(640x640\) chat photo variant in JPEG format\. The file can be downloaded only before the photo is changed"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Chat photo minithumbnail; may be null"""
        self.has_animation: bool = bool(has_animation)
        r"""True, if the photo has animated variant"""
        self.is_personal: bool = bool(is_personal)
        r"""True, if the photo is visible only for the current user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPhotoInfo"]:
        return "chatPhotoInfo"

    @classmethod
    def getClass(self) -> Literal["ChatPhotoInfo"]:
        return "ChatPhotoInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "small": self.small,
            "big": self.big,
            "minithumbnail": self.minithumbnail,
            "has_animation": self.has_animation,
            "is_personal": self.is_personal,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPhotoInfo", None]:
        if data:
            data_class = cls()
            data_class.small = data.get("small", None)
            data_class.big = data.get("big", None)
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.has_animation = data.get("has_animation", False)
            data_class.is_personal = data.get("is_personal", False)

        return data_class


class ProfileTabPosts(TlObject, ProfileTab):
    r"""A tab with stories posted by the user or the channel chat and saved to profile"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabPosts"]:
        return "profileTabPosts"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabPosts", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabGifts(TlObject, ProfileTab):
    r"""A tab with gifts received by the user or the channel chat"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabGifts"]:
        return "profileTabGifts"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabGifts", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabMedia(TlObject, ProfileTab):
    r"""A tab with photos and videos posted by the channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabMedia"]:
        return "profileTabMedia"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabMedia", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabFiles(TlObject, ProfileTab):
    r"""A tab with documents posted by the channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabFiles"]:
        return "profileTabFiles"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabFiles", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabLinks(TlObject, ProfileTab):
    r"""A tab with messages posted by the channel and containing links"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabLinks"]:
        return "profileTabLinks"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabLinks", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabMusic(TlObject, ProfileTab):
    r"""A tab with audio messages posted by the channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabMusic"]:
        return "profileTabMusic"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabMusic", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabVoice(TlObject, ProfileTab):
    r"""A tab with voice notes posted by the channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabVoice"]:
        return "profileTabVoice"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabVoice", None]:
        if data:
            data_class = cls()

        return data_class


class ProfileTabGifs(TlObject, ProfileTab):
    r"""A tab with animations posted by the channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileTabGifs"]:
        return "profileTabGifs"

    @classmethod
    def getClass(self) -> Literal["ProfileTab"]:
        return "ProfileTab"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileTabGifs", None]:
        if data:
            data_class = cls()

        return data_class


class UserTypeRegular(TlObject, UserType):
    r"""A regular user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["userTypeRegular"]:
        return "userTypeRegular"

    @classmethod
    def getClass(self) -> Literal["UserType"]:
        return "UserType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UserTypeRegular", None]:
        if data:
            data_class = cls()

        return data_class


class UserTypeDeleted(TlObject, UserType):
    r"""A deleted user or deleted bot\. No information on the user besides the user identifier is available\. It is not possible to perform any active actions on this type of user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["userTypeDeleted"]:
        return "userTypeDeleted"

    @classmethod
    def getClass(self) -> Literal["UserType"]:
        return "UserType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UserTypeDeleted", None]:
        if data:
            data_class = cls()

        return data_class


class UserTypeBot(TlObject, UserType):
    r"""A bot \(see https://core\.telegram\.org/bots\)

    Parameters:
        can_be_edited (:class:`bool`):
            True, if the bot is owned by the current user and can be edited using the methods toggleBotUsernameIsActive, reorderBotActiveUsernames, setBotProfilePhoto, setBotName, setBotInfoDescription, and setBotInfoShortDescription

        can_join_groups (:class:`bool`):
            True, if the bot can be invited to basic group and supergroup chats

        can_read_all_group_messages (:class:`bool`):
            True, if the bot can read all messages in basic group or supergroup chats and not just those addressed to the bot\. In private and channel chats a bot can always read all messages

        has_main_web_app (:class:`bool`):
            True, if the bot has the main Web App

        has_topics (:class:`bool`):
            True, if the bot has topics

        is_inline (:class:`bool`):
            True, if the bot supports inline queries

        inline_query_placeholder (:class:`str`):
            Placeholder for inline queries \(displayed on the application input field\)

        need_location (:class:`bool`):
            True, if the location of the user is expected to be sent with every inline query to this bot

        can_connect_to_business (:class:`bool`):
            True, if the bot supports connection to Telegram Business accounts

        can_be_added_to_attachment_menu (:class:`bool`):
            True, if the bot can be added to attachment or side menu

        active_user_count (:class:`int`):
            The number of recently active users of the bot

    """

    def __init__(
        self,
        can_be_edited: bool = False,
        can_join_groups: bool = False,
        can_read_all_group_messages: bool = False,
        has_main_web_app: bool = False,
        has_topics: bool = False,
        is_inline: bool = False,
        inline_query_placeholder: str = "",
        need_location: bool = False,
        can_connect_to_business: bool = False,
        can_be_added_to_attachment_menu: bool = False,
        active_user_count: int = 0,
    ) -> None:
        self.can_be_edited: bool = bool(can_be_edited)
        r"""True, if the bot is owned by the current user and can be edited using the methods toggleBotUsernameIsActive, reorderBotActiveUsernames, setBotProfilePhoto, setBotName, setBotInfoDescription, and setBotInfoShortDescription"""
        self.can_join_groups: bool = bool(can_join_groups)
        r"""True, if the bot can be invited to basic group and supergroup chats"""
        self.can_read_all_group_messages: bool = bool(can_read_all_group_messages)
        r"""True, if the bot can read all messages in basic group or supergroup chats and not just those addressed to the bot\. In private and channel chats a bot can always read all messages"""
        self.has_main_web_app: bool = bool(has_main_web_app)
        r"""True, if the bot has the main Web App"""
        self.has_topics: bool = bool(has_topics)
        r"""True, if the bot has topics"""
        self.is_inline: bool = bool(is_inline)
        r"""True, if the bot supports inline queries"""
        self.inline_query_placeholder: Union[str, None] = inline_query_placeholder
        r"""Placeholder for inline queries \(displayed on the application input field\)"""
        self.need_location: bool = bool(need_location)
        r"""True, if the location of the user is expected to be sent with every inline query to this bot"""
        self.can_connect_to_business: bool = bool(can_connect_to_business)
        r"""True, if the bot supports connection to Telegram Business accounts"""
        self.can_be_added_to_attachment_menu: bool = bool(
            can_be_added_to_attachment_menu
        )
        r"""True, if the bot can be added to attachment or side menu"""
        self.active_user_count: int = int(active_user_count)
        r"""The number of recently active users of the bot"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["userTypeBot"]:
        return "userTypeBot"

    @classmethod
    def getClass(self) -> Literal["UserType"]:
        return "UserType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "can_be_edited": self.can_be_edited,
            "can_join_groups": self.can_join_groups,
            "can_read_all_group_messages": self.can_read_all_group_messages,
            "has_main_web_app": self.has_main_web_app,
            "has_topics": self.has_topics,
            "is_inline": self.is_inline,
            "inline_query_placeholder": self.inline_query_placeholder,
            "need_location": self.need_location,
            "can_connect_to_business": self.can_connect_to_business,
            "can_be_added_to_attachment_menu": self.can_be_added_to_attachment_menu,
            "active_user_count": self.active_user_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UserTypeBot", None]:
        if data:
            data_class = cls()
            data_class.can_be_edited = data.get("can_be_edited", False)
            data_class.can_join_groups = data.get("can_join_groups", False)
            data_class.can_read_all_group_messages = data.get(
                "can_read_all_group_messages", False
            )
            data_class.has_main_web_app = data.get("has_main_web_app", False)
            data_class.has_topics = data.get("has_topics", False)
            data_class.is_inline = data.get("is_inline", False)
            data_class.inline_query_placeholder = data.get(
                "inline_query_placeholder", ""
            )
            data_class.need_location = data.get("need_location", False)
            data_class.can_connect_to_business = data.get(
                "can_connect_to_business", False
            )
            data_class.can_be_added_to_attachment_menu = data.get(
                "can_be_added_to_attachment_menu", False
            )
            data_class.active_user_count = int(data.get("active_user_count", 0))

        return data_class


class UserTypeUnknown(TlObject, UserType):
    r"""No information on the user besides the user identifier is available, yet this user has not been deleted\. This object is extremely rare and must be handled like a deleted user\. It is not possible to perform any actions on users of this type"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["userTypeUnknown"]:
        return "userTypeUnknown"

    @classmethod
    def getClass(self) -> Literal["UserType"]:
        return "UserType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UserTypeUnknown", None]:
        if data:
            data_class = cls()

        return data_class


class BotCommand(TlObject):
    r"""Represents a command supported by a bot

    Parameters:
        command (:class:`str`):
            Text of the bot command

        description (:class:`str`):
            Description of the bot command

    """

    def __init__(self, command: str = "", description: str = "") -> None:
        self.command: Union[str, None] = command
        r"""Text of the bot command"""
        self.description: Union[str, None] = description
        r"""Description of the bot command"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["botCommand"]:
        return "botCommand"

    @classmethod
    def getClass(self) -> Literal["BotCommand"]:
        return "BotCommand"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "command": self.command,
            "description": self.description,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BotCommand", None]:
        if data:
            data_class = cls()
            data_class.command = data.get("command", "")
            data_class.description = data.get("description", "")

        return data_class


class BotCommands(TlObject):
    r"""Contains a list of bot commands

    Parameters:
        bot_user_id (:class:`int`):
            Bot's user identifier

        commands (List[:class:`~pytdbot.types.BotCommand`]):
            List of bot commands

    """

    def __init__(self, bot_user_id: int = 0, commands: List[BotCommand] = None) -> None:
        self.bot_user_id: int = int(bot_user_id)
        r"""Bot's user identifier"""
        self.commands: List[BotCommand] = commands or []
        r"""List of bot commands"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["botCommands"]:
        return "botCommands"

    @classmethod
    def getClass(self) -> Literal["BotCommands"]:
        return "BotCommands"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "bot_user_id": self.bot_user_id,
            "commands": self.commands,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BotCommands", None]:
        if data:
            data_class = cls()
            data_class.bot_user_id = int(data.get("bot_user_id", 0))
            data_class.commands = data.get("commands", None)

        return data_class


class BotMenuButton(TlObject):
    r"""Describes a button to be shown instead of bot commands menu button

    Parameters:
        text (:class:`str`):
            Text of the button

        url (:class:`str`):
            URL of a Web App to open when the button is pressed\. If the link is of the type internalLinkTypeWebApp, then it must be processed accordingly\. Otherwise, the link must be passed to openWebApp

    """

    def __init__(self, text: str = "", url: str = "") -> None:
        self.text: Union[str, None] = text
        r"""Text of the button"""
        self.url: Union[str, None] = url
        r"""URL of a Web App to open when the button is pressed\. If the link is of the type internalLinkTypeWebApp, then it must be processed accordingly\. Otherwise, the link must be passed to openWebApp"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["botMenuButton"]:
        return "botMenuButton"

    @classmethod
    def getClass(self) -> Literal["BotMenuButton"]:
        return "BotMenuButton"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "text": self.text, "url": self.url}

    @classmethod
    def from_dict(cls, data: dict) -> Union["BotMenuButton", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", "")
            data_class.url = data.get("url", "")

        return data_class


class BotVerificationParameters(TlObject):
    r"""Describes parameters of verification that is provided by a bot

    Parameters:
        icon_custom_emoji_id (:class:`int`):
            Identifier of the custom emoji that is used as the verification sign

        organization_name (:class:`str`):
            Name of the organization that provides verification

        default_custom_description (:class:`~pytdbot.types.FormattedText`):
            Default custom description of verification reason to be used as placeholder in setMessageSenderBotVerification; may be null if none

        can_set_custom_description (:class:`bool`):
            True, if the bot is allowed to provide custom description for verified entities

    """

    def __init__(
        self,
        icon_custom_emoji_id: int = 0,
        organization_name: str = "",
        default_custom_description: FormattedText = None,
        can_set_custom_description: bool = False,
    ) -> None:
        self.icon_custom_emoji_id: int = int(icon_custom_emoji_id)
        r"""Identifier of the custom emoji that is used as the verification sign"""
        self.organization_name: Union[str, None] = organization_name
        r"""Name of the organization that provides verification"""
        self.default_custom_description: Union[FormattedText, None] = (
            default_custom_description
        )
        r"""Default custom description of verification reason to be used as placeholder in setMessageSenderBotVerification; may be null if none"""
        self.can_set_custom_description: bool = bool(can_set_custom_description)
        r"""True, if the bot is allowed to provide custom description for verified entities"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["botVerificationParameters"]:
        return "botVerificationParameters"

    @classmethod
    def getClass(self) -> Literal["BotVerificationParameters"]:
        return "BotVerificationParameters"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "icon_custom_emoji_id": self.icon_custom_emoji_id,
            "organization_name": self.organization_name,
            "default_custom_description": self.default_custom_description,
            "can_set_custom_description": self.can_set_custom_description,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BotVerificationParameters", None]:
        if data:
            data_class = cls()
            data_class.icon_custom_emoji_id = int(data.get("icon_custom_emoji_id", 0))
            data_class.organization_name = data.get("organization_name", "")
            data_class.default_custom_description = data.get(
                "default_custom_description", None
            )
            data_class.can_set_custom_description = data.get(
                "can_set_custom_description", False
            )

        return data_class


class BotVerification(TlObject):
    r"""Describes verification status provided by a bot

    Parameters:
        bot_user_id (:class:`int`):
            Identifier of the bot that provided the verification

        icon_custom_emoji_id (:class:`int`):
            Identifier of the custom emoji that is used as the verification sign

        custom_description (:class:`~pytdbot.types.FormattedText`):
            Custom description of verification reason set by the bot\. Can contain only Mention, Hashtag, Cashtag, PhoneNumber, BankCardNumber, Url, and EmailAddress entities

    """

    def __init__(
        self,
        bot_user_id: int = 0,
        icon_custom_emoji_id: int = 0,
        custom_description: FormattedText = None,
    ) -> None:
        self.bot_user_id: int = int(bot_user_id)
        r"""Identifier of the bot that provided the verification"""
        self.icon_custom_emoji_id: int = int(icon_custom_emoji_id)
        r"""Identifier of the custom emoji that is used as the verification sign"""
        self.custom_description: Union[FormattedText, None] = custom_description
        r"""Custom description of verification reason set by the bot\. Can contain only Mention, Hashtag, Cashtag, PhoneNumber, BankCardNumber, Url, and EmailAddress entities"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["botVerification"]:
        return "botVerification"

    @classmethod
    def getClass(self) -> Literal["BotVerification"]:
        return "BotVerification"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "bot_user_id": self.bot_user_id,
            "icon_custom_emoji_id": self.icon_custom_emoji_id,
            "custom_description": self.custom_description,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BotVerification", None]:
        if data:
            data_class = cls()
            data_class.bot_user_id = int(data.get("bot_user_id", 0))
            data_class.icon_custom_emoji_id = int(data.get("icon_custom_emoji_id", 0))
            data_class.custom_description = data.get("custom_description", None)

        return data_class


class VerificationStatus(TlObject):
    r"""Contains information about verification status of a chat or a user

    Parameters:
        is_verified (:class:`bool`):
            True, if the chat or the user is verified by Telegram

        is_scam (:class:`bool`):
            True, if the chat or the user is marked as scam by Telegram

        is_fake (:class:`bool`):
            True, if the chat or the user is marked as fake by Telegram

        bot_verification_icon_custom_emoji_id (:class:`int`):
            Identifier of the custom emoji to be shown as verification sign provided by a bot for the user; 0 if none

    """

    def __init__(
        self,
        is_verified: bool = False,
        is_scam: bool = False,
        is_fake: bool = False,
        bot_verification_icon_custom_emoji_id: int = 0,
    ) -> None:
        self.is_verified: bool = bool(is_verified)
        r"""True, if the chat or the user is verified by Telegram"""
        self.is_scam: bool = bool(is_scam)
        r"""True, if the chat or the user is marked as scam by Telegram"""
        self.is_fake: bool = bool(is_fake)
        r"""True, if the chat or the user is marked as fake by Telegram"""
        self.bot_verification_icon_custom_emoji_id: int = int(
            bot_verification_icon_custom_emoji_id
        )
        r"""Identifier of the custom emoji to be shown as verification sign provided by a bot for the user; 0 if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["verificationStatus"]:
        return "verificationStatus"

    @classmethod
    def getClass(self) -> Literal["VerificationStatus"]:
        return "VerificationStatus"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "is_verified": self.is_verified,
            "is_scam": self.is_scam,
            "is_fake": self.is_fake,
            "bot_verification_icon_custom_emoji_id": self.bot_verification_icon_custom_emoji_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["VerificationStatus", None]:
        if data:
            data_class = cls()
            data_class.is_verified = data.get("is_verified", False)
            data_class.is_scam = data.get("is_scam", False)
            data_class.is_fake = data.get("is_fake", False)
            data_class.bot_verification_icon_custom_emoji_id = int(
                data.get("bot_verification_icon_custom_emoji_id", 0)
            )

        return data_class


class ChatLocation(TlObject):
    r"""Represents a location to which a chat is connected

    Parameters:
        location (:class:`~pytdbot.types.Location`):
            The location

        address (:class:`str`):
            Location address; 1\-64 characters, as defined by the chat owner

    """

    def __init__(self, location: Location = None, address: str = "") -> None:
        self.location: Union[Location, None] = location
        r"""The location"""
        self.address: Union[str, None] = address
        r"""Location address; 1\-64 characters, as defined by the chat owner"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatLocation"]:
        return "chatLocation"

    @classmethod
    def getClass(self) -> Literal["ChatLocation"]:
        return "ChatLocation"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "location": self.location,
            "address": self.address,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatLocation", None]:
        if data:
            data_class = cls()
            data_class.location = data.get("location", None)
            data_class.address = data.get("address", "")

        return data_class


class Birthdate(TlObject):
    r"""Represents a birthdate of a user

    Parameters:
        day (:class:`int`):
            Day of the month; 1\-31

        month (:class:`int`):
            Month of the year; 1\-12

        year (:class:`int`):
            Birth year; 0 if unknown

    """

    def __init__(self, day: int = 0, month: int = 0, year: int = 0) -> None:
        self.day: int = int(day)
        r"""Day of the month; 1\-31"""
        self.month: int = int(month)
        r"""Month of the year; 1\-12"""
        self.year: int = int(year)
        r"""Birth year; 0 if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["birthdate"]:
        return "birthdate"

    @classmethod
    def getClass(self) -> Literal["Birthdate"]:
        return "Birthdate"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "day": self.day,
            "month": self.month,
            "year": self.year,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Birthdate", None]:
        if data:
            data_class = cls()
            data_class.day = int(data.get("day", 0))
            data_class.month = int(data.get("month", 0))
            data_class.year = int(data.get("year", 0))

        return data_class


class CloseBirthdayUser(TlObject):
    r"""Describes a user that had or will have a birthday soon

    Parameters:
        user_id (:class:`int`):
            User identifier

        birthdate (:class:`~pytdbot.types.Birthdate`):
            Birthdate of the user

    """

    def __init__(self, user_id: int = 0, birthdate: Birthdate = None) -> None:
        self.user_id: int = int(user_id)
        r"""User identifier"""
        self.birthdate: Union[Birthdate, None] = birthdate
        r"""Birthdate of the user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["closeBirthdayUser"]:
        return "closeBirthdayUser"

    @classmethod
    def getClass(self) -> Literal["CloseBirthdayUser"]:
        return "CloseBirthdayUser"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "birthdate": self.birthdate,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["CloseBirthdayUser", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.birthdate = data.get("birthdate", None)

        return data_class


class BusinessAwayMessageScheduleAlways(TlObject, BusinessAwayMessageSchedule):
    r"""Send away messages always"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessAwayMessageScheduleAlways"]:
        return "businessAwayMessageScheduleAlways"

    @classmethod
    def getClass(self) -> Literal["BusinessAwayMessageSchedule"]:
        return "BusinessAwayMessageSchedule"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessAwayMessageScheduleAlways", None]:
        if data:
            data_class = cls()

        return data_class


class BusinessAwayMessageScheduleOutsideOfOpeningHours(
    TlObject, BusinessAwayMessageSchedule
):
    r"""Send away messages outside of the business opening hours"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessAwayMessageScheduleOutsideOfOpeningHours"]:
        return "businessAwayMessageScheduleOutsideOfOpeningHours"

    @classmethod
    def getClass(self) -> Literal["BusinessAwayMessageSchedule"]:
        return "BusinessAwayMessageSchedule"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["BusinessAwayMessageScheduleOutsideOfOpeningHours", None]:
        if data:
            data_class = cls()

        return data_class


class BusinessAwayMessageScheduleCustom(TlObject, BusinessAwayMessageSchedule):
    r"""Send away messages only in the specified time span

    Parameters:
        start_date (:class:`int`):
            Point in time \(Unix timestamp\) when the away messages will start to be sent

        end_date (:class:`int`):
            Point in time \(Unix timestamp\) when the away messages will stop to be sent

    """

    def __init__(self, start_date: int = 0, end_date: int = 0) -> None:
        self.start_date: int = int(start_date)
        r"""Point in time \(Unix timestamp\) when the away messages will start to be sent"""
        self.end_date: int = int(end_date)
        r"""Point in time \(Unix timestamp\) when the away messages will stop to be sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessAwayMessageScheduleCustom"]:
        return "businessAwayMessageScheduleCustom"

    @classmethod
    def getClass(self) -> Literal["BusinessAwayMessageSchedule"]:
        return "BusinessAwayMessageSchedule"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "start_date": self.start_date,
            "end_date": self.end_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessAwayMessageScheduleCustom", None]:
        if data:
            data_class = cls()
            data_class.start_date = int(data.get("start_date", 0))
            data_class.end_date = int(data.get("end_date", 0))

        return data_class


class BusinessLocation(TlObject):
    r"""Represents a location of a business

    Parameters:
        location (:class:`~pytdbot.types.Location`):
            The location; may be null if not specified

        address (:class:`str`):
            Location address; 1\-96 characters

    """

    def __init__(self, location: Location = None, address: str = "") -> None:
        self.location: Union[Location, None] = location
        r"""The location; may be null if not specified"""
        self.address: Union[str, None] = address
        r"""Location address; 1\-96 characters"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessLocation"]:
        return "businessLocation"

    @classmethod
    def getClass(self) -> Literal["BusinessLocation"]:
        return "BusinessLocation"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "location": self.location,
            "address": self.address,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessLocation", None]:
        if data:
            data_class = cls()
            data_class.location = data.get("location", None)
            data_class.address = data.get("address", "")

        return data_class


class BusinessRecipients(TlObject):
    r"""Describes private chats chosen for automatic interaction with a business

    Parameters:
        chat_ids (List[:class:`int`]):
            Identifiers of selected private chats

        excluded_chat_ids (List[:class:`int`]):
            Identifiers of private chats that are always excluded; for businessConnectedBot only

        select_existing_chats (:class:`bool`):
            True, if all existing private chats are selected

        select_new_chats (:class:`bool`):
            True, if all new private chats are selected

        select_contacts (:class:`bool`):
            True, if all private chats with contacts are selected

        select_non_contacts (:class:`bool`):
            True, if all private chats with non\-contacts are selected

        exclude_selected (:class:`bool`):
            If true, then all private chats except the selected are chosen\. Otherwise, only the selected chats are chosen

    """

    def __init__(
        self,
        chat_ids: List[int] = None,
        excluded_chat_ids: List[int] = None,
        select_existing_chats: bool = False,
        select_new_chats: bool = False,
        select_contacts: bool = False,
        select_non_contacts: bool = False,
        exclude_selected: bool = False,
    ) -> None:
        self.chat_ids: List[int] = chat_ids or []
        r"""Identifiers of selected private chats"""
        self.excluded_chat_ids: List[int] = excluded_chat_ids or []
        r"""Identifiers of private chats that are always excluded; for businessConnectedBot only"""
        self.select_existing_chats: bool = bool(select_existing_chats)
        r"""True, if all existing private chats are selected"""
        self.select_new_chats: bool = bool(select_new_chats)
        r"""True, if all new private chats are selected"""
        self.select_contacts: bool = bool(select_contacts)
        r"""True, if all private chats with contacts are selected"""
        self.select_non_contacts: bool = bool(select_non_contacts)
        r"""True, if all private chats with non\-contacts are selected"""
        self.exclude_selected: bool = bool(exclude_selected)
        r"""If true, then all private chats except the selected are chosen\. Otherwise, only the selected chats are chosen"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessRecipients"]:
        return "businessRecipients"

    @classmethod
    def getClass(self) -> Literal["BusinessRecipients"]:
        return "BusinessRecipients"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_ids": self.chat_ids,
            "excluded_chat_ids": self.excluded_chat_ids,
            "select_existing_chats": self.select_existing_chats,
            "select_new_chats": self.select_new_chats,
            "select_contacts": self.select_contacts,
            "select_non_contacts": self.select_non_contacts,
            "exclude_selected": self.exclude_selected,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessRecipients", None]:
        if data:
            data_class = cls()
            data_class.chat_ids = data.get("chat_ids", None)
            data_class.excluded_chat_ids = data.get("excluded_chat_ids", None)
            data_class.select_existing_chats = data.get("select_existing_chats", False)
            data_class.select_new_chats = data.get("select_new_chats", False)
            data_class.select_contacts = data.get("select_contacts", False)
            data_class.select_non_contacts = data.get("select_non_contacts", False)
            data_class.exclude_selected = data.get("exclude_selected", False)

        return data_class


class BusinessAwayMessageSettings(TlObject):
    r"""Describes settings for messages that are automatically sent by a Telegram Business account when it is away

    Parameters:
        shortcut_id (:class:`int`):
            Unique quick reply shortcut identifier for the away messages

        recipients (:class:`~pytdbot.types.BusinessRecipients`):
            Chosen recipients of the away messages

        schedule (:class:`~pytdbot.types.BusinessAwayMessageSchedule`):
            Settings used to check whether the current user is away

        offline_only (:class:`bool`):
            True, if the messages must not be sent if the account was online in the last 10 minutes

    """

    def __init__(
        self,
        shortcut_id: int = 0,
        recipients: BusinessRecipients = None,
        schedule: BusinessAwayMessageSchedule = None,
        offline_only: bool = False,
    ) -> None:
        self.shortcut_id: int = int(shortcut_id)
        r"""Unique quick reply shortcut identifier for the away messages"""
        self.recipients: Union[BusinessRecipients, None] = recipients
        r"""Chosen recipients of the away messages"""
        self.schedule: Union[
            BusinessAwayMessageScheduleAlways,
            BusinessAwayMessageScheduleOutsideOfOpeningHours,
            BusinessAwayMessageScheduleCustom,
            None,
        ] = schedule
        r"""Settings used to check whether the current user is away"""
        self.offline_only: bool = bool(offline_only)
        r"""True, if the messages must not be sent if the account was online in the last 10 minutes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessAwayMessageSettings"]:
        return "businessAwayMessageSettings"

    @classmethod
    def getClass(self) -> Literal["BusinessAwayMessageSettings"]:
        return "BusinessAwayMessageSettings"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "shortcut_id": self.shortcut_id,
            "recipients": self.recipients,
            "schedule": self.schedule,
            "offline_only": self.offline_only,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessAwayMessageSettings", None]:
        if data:
            data_class = cls()
            data_class.shortcut_id = int(data.get("shortcut_id", 0))
            data_class.recipients = data.get("recipients", None)
            data_class.schedule = data.get("schedule", None)
            data_class.offline_only = data.get("offline_only", False)

        return data_class


class BusinessGreetingMessageSettings(TlObject):
    r"""Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat

    Parameters:
        shortcut_id (:class:`int`):
            Unique quick reply shortcut identifier for the greeting messages

        recipients (:class:`~pytdbot.types.BusinessRecipients`):
            Chosen recipients of the greeting messages

        inactivity_days (:class:`int`):
            The number of days after which a chat will be considered as inactive; currently, must be on of 7, 14, 21, or 28

    """

    def __init__(
        self,
        shortcut_id: int = 0,
        recipients: BusinessRecipients = None,
        inactivity_days: int = 0,
    ) -> None:
        self.shortcut_id: int = int(shortcut_id)
        r"""Unique quick reply shortcut identifier for the greeting messages"""
        self.recipients: Union[BusinessRecipients, None] = recipients
        r"""Chosen recipients of the greeting messages"""
        self.inactivity_days: int = int(inactivity_days)
        r"""The number of days after which a chat will be considered as inactive; currently, must be on of 7, 14, 21, or 28"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessGreetingMessageSettings"]:
        return "businessGreetingMessageSettings"

    @classmethod
    def getClass(self) -> Literal["BusinessGreetingMessageSettings"]:
        return "BusinessGreetingMessageSettings"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "shortcut_id": self.shortcut_id,
            "recipients": self.recipients,
            "inactivity_days": self.inactivity_days,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessGreetingMessageSettings", None]:
        if data:
            data_class = cls()
            data_class.shortcut_id = int(data.get("shortcut_id", 0))
            data_class.recipients = data.get("recipients", None)
            data_class.inactivity_days = int(data.get("inactivity_days", 0))

        return data_class


class BusinessBotRights(TlObject):
    r"""Describes rights of a business bot

    Parameters:
        can_reply (:class:`bool`):
            True, if the bot can send and edit messages in the private chats that had incoming messages in the last 24 hours

        can_read_messages (:class:`bool`):
            True, if the bot can mark incoming private messages as read

        can_delete_sent_messages (:class:`bool`):
            True, if the bot can delete sent messages

        can_delete_all_messages (:class:`bool`):
            True, if the bot can delete any message

        can_edit_name (:class:`bool`):
            True, if the bot can edit name of the business account

        can_edit_bio (:class:`bool`):
            True, if the bot can edit bio of the business account

        can_edit_profile_photo (:class:`bool`):
            True, if the bot can edit profile photo of the business account

        can_edit_username (:class:`bool`):
            True, if the bot can edit username of the business account

        can_view_gifts_and_stars (:class:`bool`):
            True, if the bot can view gifts and amount of Telegram Stars owned by the business account

        can_sell_gifts (:class:`bool`):
            True, if the bot can sell regular gifts received by the business account

        can_change_gift_settings (:class:`bool`):
            True, if the bot can change gift receiving settings of the business account

        can_transfer_and_upgrade_gifts (:class:`bool`):
            True, if the bot can transfer and upgrade gifts received by the business account

        can_transfer_stars (:class:`bool`):
            True, if the bot can transfer Telegram Stars received by the business account to account of the bot, or use them to upgrade and transfer gifts

        can_manage_stories (:class:`bool`):
            True, if the bot can post, edit and delete stories

    """

    def __init__(
        self,
        can_reply: bool = False,
        can_read_messages: bool = False,
        can_delete_sent_messages: bool = False,
        can_delete_all_messages: bool = False,
        can_edit_name: bool = False,
        can_edit_bio: bool = False,
        can_edit_profile_photo: bool = False,
        can_edit_username: bool = False,
        can_view_gifts_and_stars: bool = False,
        can_sell_gifts: bool = False,
        can_change_gift_settings: bool = False,
        can_transfer_and_upgrade_gifts: bool = False,
        can_transfer_stars: bool = False,
        can_manage_stories: bool = False,
    ) -> None:
        self.can_reply: bool = bool(can_reply)
        r"""True, if the bot can send and edit messages in the private chats that had incoming messages in the last 24 hours"""
        self.can_read_messages: bool = bool(can_read_messages)
        r"""True, if the bot can mark incoming private messages as read"""
        self.can_delete_sent_messages: bool = bool(can_delete_sent_messages)
        r"""True, if the bot can delete sent messages"""
        self.can_delete_all_messages: bool = bool(can_delete_all_messages)
        r"""True, if the bot can delete any message"""
        self.can_edit_name: bool = bool(can_edit_name)
        r"""True, if the bot can edit name of the business account"""
        self.can_edit_bio: bool = bool(can_edit_bio)
        r"""True, if the bot can edit bio of the business account"""
        self.can_edit_profile_photo: bool = bool(can_edit_profile_photo)
        r"""True, if the bot can edit profile photo of the business account"""
        self.can_edit_username: bool = bool(can_edit_username)
        r"""True, if the bot can edit username of the business account"""
        self.can_view_gifts_and_stars: bool = bool(can_view_gifts_and_stars)
        r"""True, if the bot can view gifts and amount of Telegram Stars owned by the business account"""
        self.can_sell_gifts: bool = bool(can_sell_gifts)
        r"""True, if the bot can sell regular gifts received by the business account"""
        self.can_change_gift_settings: bool = bool(can_change_gift_settings)
        r"""True, if the bot can change gift receiving settings of the business account"""
        self.can_transfer_and_upgrade_gifts: bool = bool(can_transfer_and_upgrade_gifts)
        r"""True, if the bot can transfer and upgrade gifts received by the business account"""
        self.can_transfer_stars: bool = bool(can_transfer_stars)
        r"""True, if the bot can transfer Telegram Stars received by the business account to account of the bot, or use them to upgrade and transfer gifts"""
        self.can_manage_stories: bool = bool(can_manage_stories)
        r"""True, if the bot can post, edit and delete stories"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessBotRights"]:
        return "businessBotRights"

    @classmethod
    def getClass(self) -> Literal["BusinessBotRights"]:
        return "BusinessBotRights"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "can_reply": self.can_reply,
            "can_read_messages": self.can_read_messages,
            "can_delete_sent_messages": self.can_delete_sent_messages,
            "can_delete_all_messages": self.can_delete_all_messages,
            "can_edit_name": self.can_edit_name,
            "can_edit_bio": self.can_edit_bio,
            "can_edit_profile_photo": self.can_edit_profile_photo,
            "can_edit_username": self.can_edit_username,
            "can_view_gifts_and_stars": self.can_view_gifts_and_stars,
            "can_sell_gifts": self.can_sell_gifts,
            "can_change_gift_settings": self.can_change_gift_settings,
            "can_transfer_and_upgrade_gifts": self.can_transfer_and_upgrade_gifts,
            "can_transfer_stars": self.can_transfer_stars,
            "can_manage_stories": self.can_manage_stories,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessBotRights", None]:
        if data:
            data_class = cls()
            data_class.can_reply = data.get("can_reply", False)
            data_class.can_read_messages = data.get("can_read_messages", False)
            data_class.can_delete_sent_messages = data.get(
                "can_delete_sent_messages", False
            )
            data_class.can_delete_all_messages = data.get(
                "can_delete_all_messages", False
            )
            data_class.can_edit_name = data.get("can_edit_name", False)
            data_class.can_edit_bio = data.get("can_edit_bio", False)
            data_class.can_edit_profile_photo = data.get(
                "can_edit_profile_photo", False
            )
            data_class.can_edit_username = data.get("can_edit_username", False)
            data_class.can_view_gifts_and_stars = data.get(
                "can_view_gifts_and_stars", False
            )
            data_class.can_sell_gifts = data.get("can_sell_gifts", False)
            data_class.can_change_gift_settings = data.get(
                "can_change_gift_settings", False
            )
            data_class.can_transfer_and_upgrade_gifts = data.get(
                "can_transfer_and_upgrade_gifts", False
            )
            data_class.can_transfer_stars = data.get("can_transfer_stars", False)
            data_class.can_manage_stories = data.get("can_manage_stories", False)

        return data_class


class BusinessConnectedBot(TlObject):
    r"""Describes a bot connected to a business account

    Parameters:
        bot_user_id (:class:`int`):
            User identifier of the bot

        recipients (:class:`~pytdbot.types.BusinessRecipients`):
            Private chats that will be accessible to the bot

        rights (:class:`~pytdbot.types.BusinessBotRights`):
            Rights of the bot

    """

    def __init__(
        self,
        bot_user_id: int = 0,
        recipients: BusinessRecipients = None,
        rights: BusinessBotRights = None,
    ) -> None:
        self.bot_user_id: int = int(bot_user_id)
        r"""User identifier of the bot"""
        self.recipients: Union[BusinessRecipients, None] = recipients
        r"""Private chats that will be accessible to the bot"""
        self.rights: Union[BusinessBotRights, None] = rights
        r"""Rights of the bot"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessConnectedBot"]:
        return "businessConnectedBot"

    @classmethod
    def getClass(self) -> Literal["BusinessConnectedBot"]:
        return "BusinessConnectedBot"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "bot_user_id": self.bot_user_id,
            "recipients": self.recipients,
            "rights": self.rights,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessConnectedBot", None]:
        if data:
            data_class = cls()
            data_class.bot_user_id = int(data.get("bot_user_id", 0))
            data_class.recipients = data.get("recipients", None)
            data_class.rights = data.get("rights", None)

        return data_class


class BusinessStartPage(TlObject):
    r"""Describes settings for a business account start page

    Parameters:
        title (:class:`str`):
            Title text of the start page

        message (:class:`str`):
            Message text of the start page

        sticker (:class:`~pytdbot.types.Sticker`):
            Greeting sticker of the start page; may be null if none

    """

    def __init__(
        self, title: str = "", message: str = "", sticker: Sticker = None
    ) -> None:
        self.title: Union[str, None] = title
        r"""Title text of the start page"""
        self.message: Union[str, None] = message
        r"""Message text of the start page"""
        self.sticker: Union[Sticker, None] = sticker
        r"""Greeting sticker of the start page; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessStartPage"]:
        return "businessStartPage"

    @classmethod
    def getClass(self) -> Literal["BusinessStartPage"]:
        return "BusinessStartPage"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "title": self.title,
            "message": self.message,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessStartPage", None]:
        if data:
            data_class = cls()
            data_class.title = data.get("title", "")
            data_class.message = data.get("message", "")
            data_class.sticker = data.get("sticker", None)

        return data_class


class InputBusinessStartPage(TlObject):
    r"""Describes settings for a business account start page to set

    Parameters:
        title (:class:`str`):
            Title text of the start page; 0\-getOption\(\"business\_start\_page\_title\_length\_max\"\) characters

        message (:class:`str`):
            Message text of the start page; 0\-getOption\(\"business\_start\_page\_message\_length\_max\"\) characters

        sticker (:class:`~pytdbot.types.InputFile`):
            Greeting sticker of the start page; pass null if none\. The sticker must belong to a sticker set and must not be a custom emoji

    """

    def __init__(
        self, title: str = "", message: str = "", sticker: InputFile = None
    ) -> None:
        self.title: Union[str, None] = title
        r"""Title text of the start page; 0\-getOption\(\"business\_start\_page\_title\_length\_max\"\) characters"""
        self.message: Union[str, None] = message
        r"""Message text of the start page; 0\-getOption\(\"business\_start\_page\_message\_length\_max\"\) characters"""
        self.sticker: Union[
            InputFileId, InputFileRemote, InputFileLocal, InputFileGenerated, None
        ] = sticker
        r"""Greeting sticker of the start page; pass null if none\. The sticker must belong to a sticker set and must not be a custom emoji"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputBusinessStartPage"]:
        return "inputBusinessStartPage"

    @classmethod
    def getClass(self) -> Literal["InputBusinessStartPage"]:
        return "InputBusinessStartPage"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "title": self.title,
            "message": self.message,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputBusinessStartPage", None]:
        if data:
            data_class = cls()
            data_class.title = data.get("title", "")
            data_class.message = data.get("message", "")
            data_class.sticker = data.get("sticker", None)

        return data_class


class BusinessOpeningHoursInterval(TlObject):
    r"""Describes an interval of time when the business is open

    Parameters:
        start_minute (:class:`int`):
            The minute's sequence number in a week, starting on Monday, marking the start of the time interval during which the business is open; 0\-7\*24\*60

        end_minute (:class:`int`):
            The minute's sequence number in a week, starting on Monday, marking the end of the time interval during which the business is open; 1\-8\*24\*60

    """

    def __init__(self, start_minute: int = 0, end_minute: int = 0) -> None:
        self.start_minute: int = int(start_minute)
        r"""The minute's sequence number in a week, starting on Monday, marking the start of the time interval during which the business is open; 0\-7\*24\*60"""
        self.end_minute: int = int(end_minute)
        r"""The minute's sequence number in a week, starting on Monday, marking the end of the time interval during which the business is open; 1\-8\*24\*60"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessOpeningHoursInterval"]:
        return "businessOpeningHoursInterval"

    @classmethod
    def getClass(self) -> Literal["BusinessOpeningHoursInterval"]:
        return "BusinessOpeningHoursInterval"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "start_minute": self.start_minute,
            "end_minute": self.end_minute,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessOpeningHoursInterval", None]:
        if data:
            data_class = cls()
            data_class.start_minute = int(data.get("start_minute", 0))
            data_class.end_minute = int(data.get("end_minute", 0))

        return data_class


class BusinessOpeningHours(TlObject):
    r"""Describes opening hours of a business

    Parameters:
        time_zone_id (:class:`str`):
            Unique time zone identifier

        opening_hours (List[:class:`~pytdbot.types.BusinessOpeningHoursInterval`]):
            Intervals of the time when the business is open

    """

    def __init__(
        self,
        time_zone_id: str = "",
        opening_hours: List[BusinessOpeningHoursInterval] = None,
    ) -> None:
        self.time_zone_id: Union[str, None] = time_zone_id
        r"""Unique time zone identifier"""
        self.opening_hours: List[BusinessOpeningHoursInterval] = opening_hours or []
        r"""Intervals of the time when the business is open"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessOpeningHours"]:
        return "businessOpeningHours"

    @classmethod
    def getClass(self) -> Literal["BusinessOpeningHours"]:
        return "BusinessOpeningHours"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "time_zone_id": self.time_zone_id,
            "opening_hours": self.opening_hours,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessOpeningHours", None]:
        if data:
            data_class = cls()
            data_class.time_zone_id = data.get("time_zone_id", "")
            data_class.opening_hours = data.get("opening_hours", None)

        return data_class


class BusinessInfo(TlObject):
    r"""Contains information about a Telegram Business account

    Parameters:
        location (:class:`~pytdbot.types.BusinessLocation`):
            Location of the business; may be null if none

        opening_hours (:class:`~pytdbot.types.BusinessOpeningHours`):
            Opening hours of the business; may be null if none\. The hours are guaranteed to be valid and has already been split by week days

        local_opening_hours (:class:`~pytdbot.types.BusinessOpeningHours`):
            Opening hours of the business in the local time; may be null if none\. The hours are guaranteed to be valid and has already been split by week days\. Local time zone identifier will be empty\. An updateUserFullInfo update is not triggered when value of this field changes

        next_open_in (:class:`int`):
            Time left before the business will open the next time, in seconds; 0 if unknown\. An updateUserFullInfo update is not triggered when value of this field changes

        next_close_in (:class:`int`):
            Time left before the business will close the next time, in seconds; 0 if unknown\. An updateUserFullInfo update is not triggered when value of this field changes

        greeting_message_settings (:class:`~pytdbot.types.BusinessGreetingMessageSettings`):
            The greeting message; may be null if none or the Business account is not of the current user

        away_message_settings (:class:`~pytdbot.types.BusinessAwayMessageSettings`):
            The away message; may be null if none or the Business account is not of the current user

        start_page (:class:`~pytdbot.types.BusinessStartPage`):
            Information about start page of the account; may be null if none

    """

    def __init__(
        self,
        location: BusinessLocation = None,
        opening_hours: BusinessOpeningHours = None,
        local_opening_hours: BusinessOpeningHours = None,
        next_open_in: int = 0,
        next_close_in: int = 0,
        greeting_message_settings: BusinessGreetingMessageSettings = None,
        away_message_settings: BusinessAwayMessageSettings = None,
        start_page: BusinessStartPage = None,
    ) -> None:
        self.location: Union[BusinessLocation, None] = location
        r"""Location of the business; may be null if none"""
        self.opening_hours: Union[BusinessOpeningHours, None] = opening_hours
        r"""Opening hours of the business; may be null if none\. The hours are guaranteed to be valid and has already been split by week days"""
        self.local_opening_hours: Union[BusinessOpeningHours, None] = (
            local_opening_hours
        )
        r"""Opening hours of the business in the local time; may be null if none\. The hours are guaranteed to be valid and has already been split by week days\. Local time zone identifier will be empty\. An updateUserFullInfo update is not triggered when value of this field changes"""
        self.next_open_in: int = int(next_open_in)
        r"""Time left before the business will open the next time, in seconds; 0 if unknown\. An updateUserFullInfo update is not triggered when value of this field changes"""
        self.next_close_in: int = int(next_close_in)
        r"""Time left before the business will close the next time, in seconds; 0 if unknown\. An updateUserFullInfo update is not triggered when value of this field changes"""
        self.greeting_message_settings: Union[BusinessGreetingMessageSettings, None] = (
            greeting_message_settings
        )
        r"""The greeting message; may be null if none or the Business account is not of the current user"""
        self.away_message_settings: Union[BusinessAwayMessageSettings, None] = (
            away_message_settings
        )
        r"""The away message; may be null if none or the Business account is not of the current user"""
        self.start_page: Union[BusinessStartPage, None] = start_page
        r"""Information about start page of the account; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessInfo"]:
        return "businessInfo"

    @classmethod
    def getClass(self) -> Literal["BusinessInfo"]:
        return "BusinessInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "location": self.location,
            "opening_hours": self.opening_hours,
            "local_opening_hours": self.local_opening_hours,
            "next_open_in": self.next_open_in,
            "next_close_in": self.next_close_in,
            "greeting_message_settings": self.greeting_message_settings,
            "away_message_settings": self.away_message_settings,
            "start_page": self.start_page,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessInfo", None]:
        if data:
            data_class = cls()
            data_class.location = data.get("location", None)
            data_class.opening_hours = data.get("opening_hours", None)
            data_class.local_opening_hours = data.get("local_opening_hours", None)
            data_class.next_open_in = int(data.get("next_open_in", 0))
            data_class.next_close_in = int(data.get("next_close_in", 0))
            data_class.greeting_message_settings = data.get(
                "greeting_message_settings", None
            )
            data_class.away_message_settings = data.get("away_message_settings", None)
            data_class.start_page = data.get("start_page", None)

        return data_class


class BusinessChatLink(TlObject):
    r"""Contains information about a business chat link

    Parameters:
        link (:class:`str`):
            The HTTPS link

        text (:class:`~pytdbot.types.FormattedText`):
            Message draft text that will be added to the input field

        title (:class:`str`):
            Link title

        view_count (:class:`int`):
            Number of times the link was used

    """

    def __init__(
        self,
        link: str = "",
        text: FormattedText = None,
        title: str = "",
        view_count: int = 0,
    ) -> None:
        self.link: Union[str, None] = link
        r"""The HTTPS link"""
        self.text: Union[FormattedText, None] = text
        r"""Message draft text that will be added to the input field"""
        self.title: Union[str, None] = title
        r"""Link title"""
        self.view_count: int = int(view_count)
        r"""Number of times the link was used"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessChatLink"]:
        return "businessChatLink"

    @classmethod
    def getClass(self) -> Literal["BusinessChatLink"]:
        return "BusinessChatLink"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "link": self.link,
            "text": self.text,
            "title": self.title,
            "view_count": self.view_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessChatLink", None]:
        if data:
            data_class = cls()
            data_class.link = data.get("link", "")
            data_class.text = data.get("text", None)
            data_class.title = data.get("title", "")
            data_class.view_count = int(data.get("view_count", 0))

        return data_class


class BusinessChatLinks(TlObject):
    r"""Contains a list of business chat links created by the user

    Parameters:
        links (List[:class:`~pytdbot.types.BusinessChatLink`]):
            List of links

    """

    def __init__(self, links: List[BusinessChatLink] = None) -> None:
        self.links: List[BusinessChatLink] = links or []
        r"""List of links"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessChatLinks"]:
        return "businessChatLinks"

    @classmethod
    def getClass(self) -> Literal["BusinessChatLinks"]:
        return "BusinessChatLinks"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "links": self.links}

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessChatLinks", None]:
        if data:
            data_class = cls()
            data_class.links = data.get("links", None)

        return data_class


class InputBusinessChatLink(TlObject):
    r"""Describes a business chat link to create or edit

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            Message draft text that will be added to the input field

        title (:class:`str`):
            Link title

    """

    def __init__(self, text: FormattedText = None, title: str = "") -> None:
        self.text: Union[FormattedText, None] = text
        r"""Message draft text that will be added to the input field"""
        self.title: Union[str, None] = title
        r"""Link title"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputBusinessChatLink"]:
        return "inputBusinessChatLink"

    @classmethod
    def getClass(self) -> Literal["InputBusinessChatLink"]:
        return "InputBusinessChatLink"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "text": self.text, "title": self.title}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputBusinessChatLink", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.title = data.get("title", "")

        return data_class


class BusinessChatLinkInfo(TlObject):
    r"""Contains information about a business chat link

    Parameters:
        chat_id (:class:`int`):
            Identifier of the private chat that created the link

        text (:class:`~pytdbot.types.FormattedText`):
            Message draft text that must be added to the input field

    """

    def __init__(self, chat_id: int = 0, text: FormattedText = None) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the private chat that created the link"""
        self.text: Union[FormattedText, None] = text
        r"""Message draft text that must be added to the input field"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessChatLinkInfo"]:
        return "businessChatLinkInfo"

    @classmethod
    def getClass(self) -> Literal["BusinessChatLinkInfo"]:
        return "BusinessChatLinkInfo"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id, "text": self.text}

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessChatLinkInfo", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.text = data.get("text", None)

        return data_class


class ChatPhotoStickerTypeRegularOrMask(TlObject, ChatPhotoStickerType):
    r"""Information about the sticker, which was used to create the chat photo

    Parameters:
        sticker_set_id (:class:`int`):
            Sticker set identifier

        sticker_id (:class:`int`):
            Identifier of the sticker in the set

    """

    def __init__(self, sticker_set_id: int = 0, sticker_id: int = 0) -> None:
        self.sticker_set_id: int = int(sticker_set_id)
        r"""Sticker set identifier"""
        self.sticker_id: int = int(sticker_id)
        r"""Identifier of the sticker in the set"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPhotoStickerTypeRegularOrMask"]:
        return "chatPhotoStickerTypeRegularOrMask"

    @classmethod
    def getClass(self) -> Literal["ChatPhotoStickerType"]:
        return "ChatPhotoStickerType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sticker_set_id": self.sticker_set_id,
            "sticker_id": self.sticker_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPhotoStickerTypeRegularOrMask", None]:
        if data:
            data_class = cls()
            data_class.sticker_set_id = int(data.get("sticker_set_id", 0))
            data_class.sticker_id = int(data.get("sticker_id", 0))

        return data_class


class ChatPhotoStickerTypeCustomEmoji(TlObject, ChatPhotoStickerType):
    r"""Information about the custom emoji, which was used to create the chat photo

    Parameters:
        custom_emoji_id (:class:`int`):
            Identifier of the custom emoji

    """

    def __init__(self, custom_emoji_id: int = 0) -> None:
        self.custom_emoji_id: int = int(custom_emoji_id)
        r"""Identifier of the custom emoji"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPhotoStickerTypeCustomEmoji"]:
        return "chatPhotoStickerTypeCustomEmoji"

    @classmethod
    def getClass(self) -> Literal["ChatPhotoStickerType"]:
        return "ChatPhotoStickerType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "custom_emoji_id": self.custom_emoji_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPhotoStickerTypeCustomEmoji", None]:
        if data:
            data_class = cls()
            data_class.custom_emoji_id = int(data.get("custom_emoji_id", 0))

        return data_class


class ChatPhotoSticker(TlObject):
    r"""Information about the sticker, which was used to create the chat photo\. The sticker is shown at the center of the photo and occupies at most 67% of it

    Parameters:
        type (:class:`~pytdbot.types.ChatPhotoStickerType`):
            Type of the sticker

        background_fill (:class:`~pytdbot.types.BackgroundFill`):
            The fill to be used as background for the sticker; rotation angle in backgroundFillGradient isn't supported

    """

    def __init__(
        self, type: ChatPhotoStickerType = None, background_fill: BackgroundFill = None
    ) -> None:
        self.type: Union[
            ChatPhotoStickerTypeRegularOrMask, ChatPhotoStickerTypeCustomEmoji, None
        ] = type
        r"""Type of the sticker"""
        self.background_fill: Union[
            BackgroundFillSolid,
            BackgroundFillGradient,
            BackgroundFillFreeformGradient,
            None,
        ] = background_fill
        r"""The fill to be used as background for the sticker; rotation angle in backgroundFillGradient isn't supported"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPhotoSticker"]:
        return "chatPhotoSticker"

    @classmethod
    def getClass(self) -> Literal["ChatPhotoSticker"]:
        return "ChatPhotoSticker"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "type": self.type,
            "background_fill": self.background_fill,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPhotoSticker", None]:
        if data:
            data_class = cls()
            data_class.type = data.get("type", None)
            data_class.background_fill = data.get("background_fill", None)

        return data_class


class AnimatedChatPhoto(TlObject):
    r"""Animated variant of a chat photo in MPEG4 format

    Parameters:
        length (:class:`int`):
            Animation width and height

        file (:class:`~pytdbot.types.File`):
            Information about the animation file

        main_frame_timestamp (:class:`float`):
            Timestamp of the frame, used as a static chat photo

    """

    def __init__(
        self, length: int = 0, file: File = None, main_frame_timestamp: float = 0.0
    ) -> None:
        self.length: int = int(length)
        r"""Animation width and height"""
        self.file: Union[File, None] = file
        r"""Information about the animation file"""
        self.main_frame_timestamp: float = float(main_frame_timestamp)
        r"""Timestamp of the frame, used as a static chat photo"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["animatedChatPhoto"]:
        return "animatedChatPhoto"

    @classmethod
    def getClass(self) -> Literal["AnimatedChatPhoto"]:
        return "AnimatedChatPhoto"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "length": self.length,
            "file": self.file,
            "main_frame_timestamp": self.main_frame_timestamp,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AnimatedChatPhoto", None]:
        if data:
            data_class = cls()
            data_class.length = int(data.get("length", 0))
            data_class.file = data.get("file", None)
            data_class.main_frame_timestamp = data.get("main_frame_timestamp", 0.0)

        return data_class


class ChatPhoto(TlObject):
    r"""Describes a chat or user profile photo

    Parameters:
        id (:class:`int`):
            Unique photo identifier

        added_date (:class:`int`):
            Point in time \(Unix timestamp\) when the photo has been added

        minithumbnail (:class:`~pytdbot.types.Minithumbnail`):
            Photo minithumbnail; may be null

        sizes (List[:class:`~pytdbot.types.PhotoSize`]):
            Available variants of the photo in JPEG format, in different size

        animation (:class:`~pytdbot.types.AnimatedChatPhoto`):
            A big \(up to 1280x1280\) animated variant of the photo in MPEG4 format; may be null

        small_animation (:class:`~pytdbot.types.AnimatedChatPhoto`):
            A small \(160x160\) animated variant of the photo in MPEG4 format; may be null even if the big animation is available

        sticker (:class:`~pytdbot.types.ChatPhotoSticker`):
            Sticker\-based version of the chat photo; may be null

    """

    def __init__(
        self,
        id: int = 0,
        added_date: int = 0,
        minithumbnail: Minithumbnail = None,
        sizes: List[PhotoSize] = None,
        animation: AnimatedChatPhoto = None,
        small_animation: AnimatedChatPhoto = None,
        sticker: ChatPhotoSticker = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique photo identifier"""
        self.added_date: int = int(added_date)
        r"""Point in time \(Unix timestamp\) when the photo has been added"""
        self.minithumbnail: Union[Minithumbnail, None] = minithumbnail
        r"""Photo minithumbnail; may be null"""
        self.sizes: List[PhotoSize] = sizes or []
        r"""Available variants of the photo in JPEG format, in different size"""
        self.animation: Union[AnimatedChatPhoto, None] = animation
        r"""A big \(up to 1280x1280\) animated variant of the photo in MPEG4 format; may be null"""
        self.small_animation: Union[AnimatedChatPhoto, None] = small_animation
        r"""A small \(160x160\) animated variant of the photo in MPEG4 format; may be null even if the big animation is available"""
        self.sticker: Union[ChatPhotoSticker, None] = sticker
        r"""Sticker\-based version of the chat photo; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPhoto"]:
        return "chatPhoto"

    @classmethod
    def getClass(self) -> Literal["ChatPhoto"]:
        return "ChatPhoto"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "added_date": self.added_date,
            "minithumbnail": self.minithumbnail,
            "sizes": self.sizes,
            "animation": self.animation,
            "small_animation": self.small_animation,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPhoto", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.added_date = int(data.get("added_date", 0))
            data_class.minithumbnail = data.get("minithumbnail", None)
            data_class.sizes = data.get("sizes", None)
            data_class.animation = data.get("animation", None)
            data_class.small_animation = data.get("small_animation", None)
            data_class.sticker = data.get("sticker", None)

        return data_class


class ChatPhotos(TlObject):
    r"""Contains a list of chat or user profile photos

    Parameters:
        total_count (:class:`int`):
            Total number of photos

        photos (List[:class:`~pytdbot.types.ChatPhoto`]):
            List of photos

    """

    def __init__(self, total_count: int = 0, photos: List[ChatPhoto] = None) -> None:
        self.total_count: int = int(total_count)
        r"""Total number of photos"""
        self.photos: List[ChatPhoto] = photos or []
        r"""List of photos"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPhotos"]:
        return "chatPhotos"

    @classmethod
    def getClass(self) -> Literal["ChatPhotos"]:
        return "ChatPhotos"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "photos": self.photos,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPhotos", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.photos = data.get("photos", None)

        return data_class


class InputChatPhotoPrevious(TlObject, InputChatPhoto):
    r"""A previously used profile photo of the current user

    Parameters:
        chat_photo_id (:class:`int`):
            Identifier of the current user's profile photo to reuse

    """

    def __init__(self, chat_photo_id: int = 0) -> None:
        self.chat_photo_id: int = int(chat_photo_id)
        r"""Identifier of the current user's profile photo to reuse"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputChatPhotoPrevious"]:
        return "inputChatPhotoPrevious"

    @classmethod
    def getClass(self) -> Literal["InputChatPhoto"]:
        return "InputChatPhoto"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_photo_id": self.chat_photo_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputChatPhotoPrevious", None]:
        if data:
            data_class = cls()
            data_class.chat_photo_id = int(data.get("chat_photo_id", 0))

        return data_class


class InputChatPhotoStatic(TlObject, InputChatPhoto):
    r"""A static photo in JPEG format

    Parameters:
        photo (:class:`~pytdbot.types.InputFile`):
            Photo to be set as profile photo\. Only inputFileLocal and inputFileGenerated are allowed

    """

    def __init__(self, photo: InputFile = None) -> None:
        self.photo: Union[
            InputFileId, InputFileRemote, InputFileLocal, InputFileGenerated, None
        ] = photo
        r"""Photo to be set as profile photo\. Only inputFileLocal and inputFileGenerated are allowed"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputChatPhotoStatic"]:
        return "inputChatPhotoStatic"

    @classmethod
    def getClass(self) -> Literal["InputChatPhoto"]:
        return "InputChatPhoto"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "photo": self.photo}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputChatPhotoStatic", None]:
        if data:
            data_class = cls()
            data_class.photo = data.get("photo", None)

        return data_class


class InputChatPhotoAnimation(TlObject, InputChatPhoto):
    r"""An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size

    Parameters:
        animation (:class:`~pytdbot.types.InputFile`):
            Animation to be set as profile photo\. Only inputFileLocal and inputFileGenerated are allowed

        main_frame_timestamp (:class:`float`):
            Timestamp of the frame, which will be used as static chat photo

    """

    def __init__(
        self, animation: InputFile = None, main_frame_timestamp: float = 0.0
    ) -> None:
        self.animation: Union[
            InputFileId, InputFileRemote, InputFileLocal, InputFileGenerated, None
        ] = animation
        r"""Animation to be set as profile photo\. Only inputFileLocal and inputFileGenerated are allowed"""
        self.main_frame_timestamp: float = float(main_frame_timestamp)
        r"""Timestamp of the frame, which will be used as static chat photo"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputChatPhotoAnimation"]:
        return "inputChatPhotoAnimation"

    @classmethod
    def getClass(self) -> Literal["InputChatPhoto"]:
        return "InputChatPhoto"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "animation": self.animation,
            "main_frame_timestamp": self.main_frame_timestamp,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputChatPhotoAnimation", None]:
        if data:
            data_class = cls()
            data_class.animation = data.get("animation", None)
            data_class.main_frame_timestamp = data.get("main_frame_timestamp", 0.0)

        return data_class


class InputChatPhotoSticker(TlObject, InputChatPhoto):
    r"""A sticker on a custom background

    Parameters:
        sticker (:class:`~pytdbot.types.ChatPhotoSticker`):
            Information about the sticker

    """

    def __init__(self, sticker: ChatPhotoSticker = None) -> None:
        self.sticker: Union[ChatPhotoSticker, None] = sticker
        r"""Information about the sticker"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputChatPhotoSticker"]:
        return "inputChatPhotoSticker"

    @classmethod
    def getClass(self) -> Literal["InputChatPhoto"]:
        return "InputChatPhoto"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sticker": self.sticker}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputChatPhotoSticker", None]:
        if data:
            data_class = cls()
            data_class.sticker = data.get("sticker", None)

        return data_class


class ChatPermissions(TlObject):
    r"""Describes actions that a user is allowed to take in a chat

    Parameters:
        can_send_basic_messages (:class:`bool`):
            True, if the user can send text messages, contacts, giveaways, giveaway winners, invoices, locations, and venues

        can_send_audios (:class:`bool`):
            True, if the user can send music files

        can_send_documents (:class:`bool`):
            True, if the user can send documents

        can_send_photos (:class:`bool`):
            True, if the user can send photos

        can_send_videos (:class:`bool`):
            True, if the user can send videos

        can_send_video_notes (:class:`bool`):
            True, if the user can send video notes

        can_send_voice_notes (:class:`bool`):
            True, if the user can send voice notes

        can_send_polls (:class:`bool`):
            True, if the user can send polls and checklists

        can_send_other_messages (:class:`bool`):
            True, if the user can send animations, games, stickers, and dice and use inline bots

        can_add_link_previews (:class:`bool`):
            True, if the user may add a link preview to their messages

        can_change_info (:class:`bool`):
            True, if the user can change the chat title, photo, and other settings

        can_invite_users (:class:`bool`):
            True, if the user can invite new users to the chat

        can_pin_messages (:class:`bool`):
            True, if the user can pin messages

        can_create_topics (:class:`bool`):
            True, if the user can create topics

    """

    def __init__(
        self,
        can_send_basic_messages: bool = False,
        can_send_audios: bool = False,
        can_send_documents: bool = False,
        can_send_photos: bool = False,
        can_send_videos: bool = False,
        can_send_video_notes: bool = False,
        can_send_voice_notes: bool = False,
        can_send_polls: bool = False,
        can_send_other_messages: bool = False,
        can_add_link_previews: bool = False,
        can_change_info: bool = False,
        can_invite_users: bool = False,
        can_pin_messages: bool = False,
        can_create_topics: bool = False,
    ) -> None:
        self.can_send_basic_messages: bool = bool(can_send_basic_messages)
        r"""True, if the user can send text messages, contacts, giveaways, giveaway winners, invoices, locations, and venues"""
        self.can_send_audios: bool = bool(can_send_audios)
        r"""True, if the user can send music files"""
        self.can_send_documents: bool = bool(can_send_documents)
        r"""True, if the user can send documents"""
        self.can_send_photos: bool = bool(can_send_photos)
        r"""True, if the user can send photos"""
        self.can_send_videos: bool = bool(can_send_videos)
        r"""True, if the user can send videos"""
        self.can_send_video_notes: bool = bool(can_send_video_notes)
        r"""True, if the user can send video notes"""
        self.can_send_voice_notes: bool = bool(can_send_voice_notes)
        r"""True, if the user can send voice notes"""
        self.can_send_polls: bool = bool(can_send_polls)
        r"""True, if the user can send polls and checklists"""
        self.can_send_other_messages: bool = bool(can_send_other_messages)
        r"""True, if the user can send animations, games, stickers, and dice and use inline bots"""
        self.can_add_link_previews: bool = bool(can_add_link_previews)
        r"""True, if the user may add a link preview to their messages"""
        self.can_change_info: bool = bool(can_change_info)
        r"""True, if the user can change the chat title, photo, and other settings"""
        self.can_invite_users: bool = bool(can_invite_users)
        r"""True, if the user can invite new users to the chat"""
        self.can_pin_messages: bool = bool(can_pin_messages)
        r"""True, if the user can pin messages"""
        self.can_create_topics: bool = bool(can_create_topics)
        r"""True, if the user can create topics"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatPermissions"]:
        return "chatPermissions"

    @classmethod
    def getClass(self) -> Literal["ChatPermissions"]:
        return "ChatPermissions"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "can_send_basic_messages": self.can_send_basic_messages,
            "can_send_audios": self.can_send_audios,
            "can_send_documents": self.can_send_documents,
            "can_send_photos": self.can_send_photos,
            "can_send_videos": self.can_send_videos,
            "can_send_video_notes": self.can_send_video_notes,
            "can_send_voice_notes": self.can_send_voice_notes,
            "can_send_polls": self.can_send_polls,
            "can_send_other_messages": self.can_send_other_messages,
            "can_add_link_previews": self.can_add_link_previews,
            "can_change_info": self.can_change_info,
            "can_invite_users": self.can_invite_users,
            "can_pin_messages": self.can_pin_messages,
            "can_create_topics": self.can_create_topics,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatPermissions", None]:
        if data:
            data_class = cls()
            data_class.can_send_basic_messages = data.get(
                "can_send_basic_messages", False
            )
            data_class.can_send_audios = data.get("can_send_audios", False)
            data_class.can_send_documents = data.get("can_send_documents", False)
            data_class.can_send_photos = data.get("can_send_photos", False)
            data_class.can_send_videos = data.get("can_send_videos", False)
            data_class.can_send_video_notes = data.get("can_send_video_notes", False)
            data_class.can_send_voice_notes = data.get("can_send_voice_notes", False)
            data_class.can_send_polls = data.get("can_send_polls", False)
            data_class.can_send_other_messages = data.get(
                "can_send_other_messages", False
            )
            data_class.can_add_link_previews = data.get("can_add_link_previews", False)
            data_class.can_change_info = data.get("can_change_info", False)
            data_class.can_invite_users = data.get("can_invite_users", False)
            data_class.can_pin_messages = data.get("can_pin_messages", False)
            data_class.can_create_topics = data.get("can_create_topics", False)

        return data_class


class ChatAdministratorRights(TlObject):
    r"""Describes rights of the administrator

    Parameters:
        can_manage_chat (:class:`bool`):
            True, if the administrator can access the chat event log, get boost list, see hidden supergroup and channel members, report supergroup spam messages, ignore slow mode, and send messages to the chat without paying Telegram Stars\. Implied by any other privilege; applicable to supergroups and channels only

        can_change_info (:class:`bool`):
            True, if the administrator can change the chat title, photo, and other settings

        can_post_messages (:class:`bool`):
            True, if the administrator can create channel posts, approve suggested channel posts, or view channel statistics; applicable to channels only

        can_edit_messages (:class:`bool`):
            True, if the administrator can edit messages of other users and pin messages; applicable to channels only

        can_delete_messages (:class:`bool`):
            True, if the administrator can delete messages of other users

        can_invite_users (:class:`bool`):
            True, if the administrator can invite new users to the chat

        can_restrict_members (:class:`bool`):
            True, if the administrator can restrict, ban, or unban chat members or view supergroup statistics; always true for channels

        can_pin_messages (:class:`bool`):
            True, if the administrator can pin messages; applicable to basic groups and supergroups only

        can_manage_topics (:class:`bool`):
            True, if the administrator can create, rename, close, reopen, hide, and unhide forum topics; applicable to forum supergroups only

        can_promote_members (:class:`bool`):
            True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them

        can_manage_video_chats (:class:`bool`):
            True, if the administrator can manage video chats

        can_post_stories (:class:`bool`):
            True, if the administrator can create new chat stories, or edit and delete posted stories; applicable to supergroups and channels only

        can_edit_stories (:class:`bool`):
            True, if the administrator can edit stories posted by other users, post stories to the chat page, pin chat stories, and access story archive; applicable to supergroups and channels only

        can_delete_stories (:class:`bool`):
            True, if the administrator can delete stories posted by other users; applicable to supergroups and channels only

        can_manage_direct_messages (:class:`bool`):
            True, if the administrator can answer to channel direct messages; applicable to channels only

        is_anonymous (:class:`bool`):
            True, if the administrator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only

    """

    def __init__(
        self,
        can_manage_chat: bool = False,
        can_change_info: bool = False,
        can_post_messages: bool = False,
        can_edit_messages: bool = False,
        can_delete_messages: bool = False,
        can_invite_users: bool = False,
        can_restrict_members: bool = False,
        can_pin_messages: bool = False,
        can_manage_topics: bool = False,
        can_promote_members: bool = False,
        can_manage_video_chats: bool = False,
        can_post_stories: bool = False,
        can_edit_stories: bool = False,
        can_delete_stories: bool = False,
        can_manage_direct_messages: bool = False,
        is_anonymous: bool = False,
    ) -> None:
        self.can_manage_chat: bool = bool(can_manage_chat)
        r"""True, if the administrator can access the chat event log, get boost list, see hidden supergroup and channel members, report supergroup spam messages, ignore slow mode, and send messages to the chat without paying Telegram Stars\. Implied by any other privilege; applicable to supergroups and channels only"""
        self.can_change_info: bool = bool(can_change_info)
        r"""True, if the administrator can change the chat title, photo, and other settings"""
        self.can_post_messages: bool = bool(can_post_messages)
        r"""True, if the administrator can create channel posts, approve suggested channel posts, or view channel statistics; applicable to channels only"""
        self.can_edit_messages: bool = bool(can_edit_messages)
        r"""True, if the administrator can edit messages of other users and pin messages; applicable to channels only"""
        self.can_delete_messages: bool = bool(can_delete_messages)
        r"""True, if the administrator can delete messages of other users"""
        self.can_invite_users: bool = bool(can_invite_users)
        r"""True, if the administrator can invite new users to the chat"""
        self.can_restrict_members: bool = bool(can_restrict_members)
        r"""True, if the administrator can restrict, ban, or unban chat members or view supergroup statistics; always true for channels"""
        self.can_pin_messages: bool = bool(can_pin_messages)
        r"""True, if the administrator can pin messages; applicable to basic groups and supergroups only"""
        self.can_manage_topics: bool = bool(can_manage_topics)
        r"""True, if the administrator can create, rename, close, reopen, hide, and unhide forum topics; applicable to forum supergroups only"""
        self.can_promote_members: bool = bool(can_promote_members)
        r"""True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them"""
        self.can_manage_video_chats: bool = bool(can_manage_video_chats)
        r"""True, if the administrator can manage video chats"""
        self.can_post_stories: bool = bool(can_post_stories)
        r"""True, if the administrator can create new chat stories, or edit and delete posted stories; applicable to supergroups and channels only"""
        self.can_edit_stories: bool = bool(can_edit_stories)
        r"""True, if the administrator can edit stories posted by other users, post stories to the chat page, pin chat stories, and access story archive; applicable to supergroups and channels only"""
        self.can_delete_stories: bool = bool(can_delete_stories)
        r"""True, if the administrator can delete stories posted by other users; applicable to supergroups and channels only"""
        self.can_manage_direct_messages: bool = bool(can_manage_direct_messages)
        r"""True, if the administrator can answer to channel direct messages; applicable to channels only"""
        self.is_anonymous: bool = bool(is_anonymous)
        r"""True, if the administrator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatAdministratorRights"]:
        return "chatAdministratorRights"

    @classmethod
    def getClass(self) -> Literal["ChatAdministratorRights"]:
        return "ChatAdministratorRights"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "can_manage_chat": self.can_manage_chat,
            "can_change_info": self.can_change_info,
            "can_post_messages": self.can_post_messages,
            "can_edit_messages": self.can_edit_messages,
            "can_delete_messages": self.can_delete_messages,
            "can_invite_users": self.can_invite_users,
            "can_restrict_members": self.can_restrict_members,
            "can_pin_messages": self.can_pin_messages,
            "can_manage_topics": self.can_manage_topics,
            "can_promote_members": self.can_promote_members,
            "can_manage_video_chats": self.can_manage_video_chats,
            "can_post_stories": self.can_post_stories,
            "can_edit_stories": self.can_edit_stories,
            "can_delete_stories": self.can_delete_stories,
            "can_manage_direct_messages": self.can_manage_direct_messages,
            "is_anonymous": self.is_anonymous,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatAdministratorRights", None]:
        if data:
            data_class = cls()
            data_class.can_manage_chat = data.get("can_manage_chat", False)
            data_class.can_change_info = data.get("can_change_info", False)
            data_class.can_post_messages = data.get("can_post_messages", False)
            data_class.can_edit_messages = data.get("can_edit_messages", False)
            data_class.can_delete_messages = data.get("can_delete_messages", False)
            data_class.can_invite_users = data.get("can_invite_users", False)
            data_class.can_restrict_members = data.get("can_restrict_members", False)
            data_class.can_pin_messages = data.get("can_pin_messages", False)
            data_class.can_manage_topics = data.get("can_manage_topics", False)
            data_class.can_promote_members = data.get("can_promote_members", False)
            data_class.can_manage_video_chats = data.get(
                "can_manage_video_chats", False
            )
            data_class.can_post_stories = data.get("can_post_stories", False)
            data_class.can_edit_stories = data.get("can_edit_stories", False)
            data_class.can_delete_stories = data.get("can_delete_stories", False)
            data_class.can_manage_direct_messages = data.get(
                "can_manage_direct_messages", False
            )
            data_class.is_anonymous = data.get("is_anonymous", False)

        return data_class


class GiftResalePriceStar(TlObject, GiftResalePrice):
    r"""Describes price of a resold gift in Telegram Stars

    Parameters:
        star_count (:class:`int`):
            The amount of Telegram Stars expected to be paid for the gift\. Must be in range getOption\(\"gift\_resale\_star\_count\_min\"\)\-getOption\(\"gift\_resale\_star\_count\_max\"\) for gifts put for resale

    """

    def __init__(self, star_count: int = 0) -> None:
        self.star_count: int = int(star_count)
        r"""The amount of Telegram Stars expected to be paid for the gift\. Must be in range getOption\(\"gift\_resale\_star\_count\_min\"\)\-getOption\(\"gift\_resale\_star\_count\_max\"\) for gifts put for resale"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftResalePriceStar"]:
        return "giftResalePriceStar"

    @classmethod
    def getClass(self) -> Literal["GiftResalePrice"]:
        return "GiftResalePrice"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "star_count": self.star_count}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftResalePriceStar", None]:
        if data:
            data_class = cls()
            data_class.star_count = int(data.get("star_count", 0))

        return data_class


class GiftResalePriceTon(TlObject, GiftResalePrice):
    r"""Describes price of a resold gift in Toncoins

    Parameters:
        toncoin_cent_count (:class:`int`):
            The amount of 1/100 of Toncoin expected to be paid for the gift\. Must be in range getOption\(\"gift\_resale\_toncoin\_cent\_count\_min\"\)\-getOption\(\"gift\_resale\_toncoin\_cent\_count\_max\"\)

    """

    def __init__(self, toncoin_cent_count: int = 0) -> None:
        self.toncoin_cent_count: int = int(toncoin_cent_count)
        r"""The amount of 1/100 of Toncoin expected to be paid for the gift\. Must be in range getOption\(\"gift\_resale\_toncoin\_cent\_count\_min\"\)\-getOption\(\"gift\_resale\_toncoin\_cent\_count\_max\"\)"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftResalePriceTon"]:
        return "giftResalePriceTon"

    @classmethod
    def getClass(self) -> Literal["GiftResalePrice"]:
        return "GiftResalePrice"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "toncoin_cent_count": self.toncoin_cent_count}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftResalePriceTon", None]:
        if data:
            data_class = cls()
            data_class.toncoin_cent_count = int(data.get("toncoin_cent_count", 0))

        return data_class


class SuggestedPostPriceStar(TlObject, SuggestedPostPrice):
    r"""Describes price of a suggested post in Telegram Stars

    Parameters:
        star_count (:class:`int`):
            The amount of Telegram Stars expected to be paid for the post; getOption\(\"suggested\_post\_star\_count\_min\"\)\-getOption\(\"suggested\_post\_star\_count\_max\"\)

    """

    def __init__(self, star_count: int = 0) -> None:
        self.star_count: int = int(star_count)
        r"""The amount of Telegram Stars expected to be paid for the post; getOption\(\"suggested\_post\_star\_count\_min\"\)\-getOption\(\"suggested\_post\_star\_count\_max\"\)"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostPriceStar"]:
        return "suggestedPostPriceStar"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostPrice"]:
        return "SuggestedPostPrice"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "star_count": self.star_count}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SuggestedPostPriceStar", None]:
        if data:
            data_class = cls()
            data_class.star_count = int(data.get("star_count", 0))

        return data_class


class SuggestedPostPriceTon(TlObject, SuggestedPostPrice):
    r"""Describes price of a suggested post in Toncoins

    Parameters:
        toncoin_cent_count (:class:`int`):
            The amount of 1/100 of Toncoin expected to be paid for the post; getOption\(\"suggested\_post\_toncoin\_cent\_count\_min\"\)\-getOption\(\"suggested\_post\_toncoin\_cent\_count\_max\"\)

    """

    def __init__(self, toncoin_cent_count: int = 0) -> None:
        self.toncoin_cent_count: int = int(toncoin_cent_count)
        r"""The amount of 1/100 of Toncoin expected to be paid for the post; getOption\(\"suggested\_post\_toncoin\_cent\_count\_min\"\)\-getOption\(\"suggested\_post\_toncoin\_cent\_count\_max\"\)"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostPriceTon"]:
        return "suggestedPostPriceTon"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostPrice"]:
        return "SuggestedPostPrice"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "toncoin_cent_count": self.toncoin_cent_count}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SuggestedPostPriceTon", None]:
        if data:
            data_class = cls()
            data_class.toncoin_cent_count = int(data.get("toncoin_cent_count", 0))

        return data_class


class SuggestedPostStatePending(TlObject, SuggestedPostState):
    r"""The post must be approved or declined"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostStatePending"]:
        return "suggestedPostStatePending"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostState"]:
        return "SuggestedPostState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SuggestedPostStatePending", None]:
        if data:
            data_class = cls()

        return data_class


class SuggestedPostStateApproved(TlObject, SuggestedPostState):
    r"""The post was approved"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostStateApproved"]:
        return "suggestedPostStateApproved"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostState"]:
        return "SuggestedPostState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SuggestedPostStateApproved", None]:
        if data:
            data_class = cls()

        return data_class


class SuggestedPostStateDeclined(TlObject, SuggestedPostState):
    r"""The post was declined"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostStateDeclined"]:
        return "suggestedPostStateDeclined"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostState"]:
        return "SuggestedPostState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SuggestedPostStateDeclined", None]:
        if data:
            data_class = cls()

        return data_class


class SuggestedPostInfo(TlObject):
    r"""Contains information about a suggested post\. If the post can be approved or declined, then changes to the post can be also suggested\. Use sendMessage with reply to the message and suggested post information to suggest message changes\. Use addOffer to suggest price or time changes

    Parameters:
        price (:class:`~pytdbot.types.SuggestedPostPrice`):
            Price of the suggested post; may be null if the post is non\-paid

        send_date (:class:`int`):
            Point in time \(Unix timestamp\) when the post is expected to be published; 0 if the specific date isn't set yet

        state (:class:`~pytdbot.types.SuggestedPostState`):
            State of the post

        can_be_approved (:class:`bool`):
            True, if the suggested post can be approved by the current user using approveSuggestedPost; updates aren't sent when value of this field changes

        can_be_declined (:class:`bool`):
            True, if the suggested post can be declined by the current user using declineSuggestedPost; updates aren't sent when value of this field changes

    """

    def __init__(
        self,
        price: SuggestedPostPrice = None,
        send_date: int = 0,
        state: SuggestedPostState = None,
        can_be_approved: bool = False,
        can_be_declined: bool = False,
    ) -> None:
        self.price: Union[SuggestedPostPriceStar, SuggestedPostPriceTon, None] = price
        r"""Price of the suggested post; may be null if the post is non\-paid"""
        self.send_date: int = int(send_date)
        r"""Point in time \(Unix timestamp\) when the post is expected to be published; 0 if the specific date isn't set yet"""
        self.state: Union[
            SuggestedPostStatePending,
            SuggestedPostStateApproved,
            SuggestedPostStateDeclined,
            None,
        ] = state
        r"""State of the post"""
        self.can_be_approved: bool = bool(can_be_approved)
        r"""True, if the suggested post can be approved by the current user using approveSuggestedPost; updates aren't sent when value of this field changes"""
        self.can_be_declined: bool = bool(can_be_declined)
        r"""True, if the suggested post can be declined by the current user using declineSuggestedPost; updates aren't sent when value of this field changes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostInfo"]:
        return "suggestedPostInfo"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostInfo"]:
        return "SuggestedPostInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "price": self.price,
            "send_date": self.send_date,
            "state": self.state,
            "can_be_approved": self.can_be_approved,
            "can_be_declined": self.can_be_declined,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["SuggestedPostInfo", None]:
        if data:
            data_class = cls()
            data_class.price = data.get("price", None)
            data_class.send_date = int(data.get("send_date", 0))
            data_class.state = data.get("state", None)
            data_class.can_be_approved = data.get("can_be_approved", False)
            data_class.can_be_declined = data.get("can_be_declined", False)

        return data_class


class InputSuggestedPostInfo(TlObject):
    r"""Contains information about a post to suggest

    Parameters:
        price (:class:`~pytdbot.types.SuggestedPostPrice`):
            Price of the suggested post; pass null to suggest a post without payment\. If the current user isn't an administrator of the channel direct messages chat and has no enough funds to pay for the post, then the error \"BALANCE\_TOO\_LOW\" will be returned immediately

        send_date (:class:`int`):
            Point in time \(Unix timestamp\) when the post is expected to be published; pass 0 if the date isn't restricted\. If specified, then the date must be getOption\(\"suggested\_post\_send\_delay\_min\"\)\-getOption\(\"suggested\_post\_send\_delay\_max\"\) seconds in the future

    """

    def __init__(self, price: SuggestedPostPrice = None, send_date: int = 0) -> None:
        self.price: Union[SuggestedPostPriceStar, SuggestedPostPriceTon, None] = price
        r"""Price of the suggested post; pass null to suggest a post without payment\. If the current user isn't an administrator of the channel direct messages chat and has no enough funds to pay for the post, then the error \"BALANCE\_TOO\_LOW\" will be returned immediately"""
        self.send_date: int = int(send_date)
        r"""Point in time \(Unix timestamp\) when the post is expected to be published; pass 0 if the date isn't restricted\. If specified, then the date must be getOption\(\"suggested\_post\_send\_delay\_min\"\)\-getOption\(\"suggested\_post\_send\_delay\_max\"\) seconds in the future"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputSuggestedPostInfo"]:
        return "inputSuggestedPostInfo"

    @classmethod
    def getClass(self) -> Literal["InputSuggestedPostInfo"]:
        return "InputSuggestedPostInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "price": self.price,
            "send_date": self.send_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputSuggestedPostInfo", None]:
        if data:
            data_class = cls()
            data_class.price = data.get("price", None)
            data_class.send_date = int(data.get("send_date", 0))

        return data_class


class SuggestedPostRefundReasonPostDeleted(TlObject, SuggestedPostRefundReason):
    r"""The post was refunded, because it was deleted by channel administrators in less than getOption\(\"suggested\_post\_lifetime\_min\"\) seconds"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostRefundReasonPostDeleted"]:
        return "suggestedPostRefundReasonPostDeleted"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostRefundReason"]:
        return "SuggestedPostRefundReason"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["SuggestedPostRefundReasonPostDeleted", None]:
        if data:
            data_class = cls()

        return data_class


class SuggestedPostRefundReasonPaymentRefunded(TlObject, SuggestedPostRefundReason):
    r"""The post was refunded, because the payment for the post was refunded"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["suggestedPostRefundReasonPaymentRefunded"]:
        return "suggestedPostRefundReasonPaymentRefunded"

    @classmethod
    def getClass(self) -> Literal["SuggestedPostRefundReason"]:
        return "SuggestedPostRefundReason"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["SuggestedPostRefundReasonPaymentRefunded", None]:
        if data:
            data_class = cls()

        return data_class


class StarAmount(TlObject):
    r"""Describes a possibly non\-integer amount of Telegram Stars

    Parameters:
        star_count (:class:`int`):
            The integer amount of Telegram Stars rounded to 0

        nanostar_count (:class:`int`):
            The number of 1/1000000000 shares of Telegram Stars; from \-999999999 to 999999999

    """

    def __init__(self, star_count: int = 0, nanostar_count: int = 0) -> None:
        self.star_count: int = int(star_count)
        r"""The integer amount of Telegram Stars rounded to 0"""
        self.nanostar_count: int = int(nanostar_count)
        r"""The number of 1/1000000000 shares of Telegram Stars; from \-999999999 to 999999999"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starAmount"]:
        return "starAmount"

    @classmethod
    def getClass(self) -> Literal["StarAmount"]:
        return "StarAmount"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "star_count": self.star_count,
            "nanostar_count": self.nanostar_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarAmount", None]:
        if data:
            data_class = cls()
            data_class.star_count = int(data.get("star_count", 0))
            data_class.nanostar_count = int(data.get("nanostar_count", 0))

        return data_class


class StarSubscriptionTypeChannel(TlObject, StarSubscriptionType):
    r"""Describes a subscription to a channel chat

    Parameters:
        can_reuse (:class:`bool`):
            True, if the subscription is active and the user can use the method reuseStarSubscription to join the subscribed chat again

        invite_link (:class:`str`):
            The invite link that can be used to renew the subscription if it has been expired; may be empty, if the link isn't available anymore

    """

    def __init__(self, can_reuse: bool = False, invite_link: str = "") -> None:
        self.can_reuse: bool = bool(can_reuse)
        r"""True, if the subscription is active and the user can use the method reuseStarSubscription to join the subscribed chat again"""
        self.invite_link: Union[str, None] = invite_link
        r"""The invite link that can be used to renew the subscription if it has been expired; may be empty, if the link isn't available anymore"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starSubscriptionTypeChannel"]:
        return "starSubscriptionTypeChannel"

    @classmethod
    def getClass(self) -> Literal["StarSubscriptionType"]:
        return "StarSubscriptionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "can_reuse": self.can_reuse,
            "invite_link": self.invite_link,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarSubscriptionTypeChannel", None]:
        if data:
            data_class = cls()
            data_class.can_reuse = data.get("can_reuse", False)
            data_class.invite_link = data.get("invite_link", "")

        return data_class


class StarSubscriptionTypeBot(TlObject, StarSubscriptionType):
    r"""Describes a subscription in a bot or a business account

    Parameters:
        is_canceled_by_bot (:class:`bool`):
            True, if the subscription was canceled by the bot and can't be extended

        title (:class:`str`):
            Subscription invoice title

        photo (:class:`~pytdbot.types.Photo`):
            Subscription invoice photo

        invoice_link (:class:`str`):
            The link to the subscription invoice

    """

    def __init__(
        self,
        is_canceled_by_bot: bool = False,
        title: str = "",
        photo: Photo = None,
        invoice_link: str = "",
    ) -> None:
        self.is_canceled_by_bot: bool = bool(is_canceled_by_bot)
        r"""True, if the subscription was canceled by the bot and can't be extended"""
        self.title: Union[str, None] = title
        r"""Subscription invoice title"""
        self.photo: Union[Photo, None] = photo
        r"""Subscription invoice photo"""
        self.invoice_link: Union[str, None] = invoice_link
        r"""The link to the subscription invoice"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starSubscriptionTypeBot"]:
        return "starSubscriptionTypeBot"

    @classmethod
    def getClass(self) -> Literal["StarSubscriptionType"]:
        return "StarSubscriptionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "is_canceled_by_bot": self.is_canceled_by_bot,
            "title": self.title,
            "photo": self.photo,
            "invoice_link": self.invoice_link,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarSubscriptionTypeBot", None]:
        if data:
            data_class = cls()
            data_class.is_canceled_by_bot = data.get("is_canceled_by_bot", False)
            data_class.title = data.get("title", "")
            data_class.photo = data.get("photo", None)
            data_class.invoice_link = data.get("invoice_link", "")

        return data_class


class StarSubscriptionPricing(TlObject):
    r"""Describes subscription plan paid in Telegram Stars

    Parameters:
        period (:class:`int`):
            The number of seconds between consecutive Telegram Star debiting

        star_count (:class:`int`):
            The amount of Telegram Stars that must be paid for each period

    """

    def __init__(self, period: int = 0, star_count: int = 0) -> None:
        self.period: int = int(period)
        r"""The number of seconds between consecutive Telegram Star debiting"""
        self.star_count: int = int(star_count)
        r"""The amount of Telegram Stars that must be paid for each period"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starSubscriptionPricing"]:
        return "starSubscriptionPricing"

    @classmethod
    def getClass(self) -> Literal["StarSubscriptionPricing"]:
        return "StarSubscriptionPricing"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "period": self.period,
            "star_count": self.star_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarSubscriptionPricing", None]:
        if data:
            data_class = cls()
            data_class.period = int(data.get("period", 0))
            data_class.star_count = int(data.get("star_count", 0))

        return data_class


class StarSubscription(TlObject):
    r"""Contains information about subscription to a channel chat, a bot, or a business account that was paid in Telegram Stars

    Parameters:
        id (:class:`str`):
            Unique identifier of the subscription

        chat_id (:class:`int`):
            Identifier of the chat that is subscribed

        expiration_date (:class:`int`):
            Point in time \(Unix timestamp\) when the subscription will expire or expired

        is_canceled (:class:`bool`):
            True, if the subscription was canceled

        is_expiring (:class:`bool`):
            True, if the subscription expires soon and there are no enough Telegram Stars on the user's balance to extend it

        pricing (:class:`~pytdbot.types.StarSubscriptionPricing`):
            The subscription plan

        type (:class:`~pytdbot.types.StarSubscriptionType`):
            Type of the subscription

    """

    def __init__(
        self,
        id: str = "",
        chat_id: int = 0,
        expiration_date: int = 0,
        is_canceled: bool = False,
        is_expiring: bool = False,
        pricing: StarSubscriptionPricing = None,
        type: StarSubscriptionType = None,
    ) -> None:
        self.id: Union[str, None] = id
        r"""Unique identifier of the subscription"""
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat that is subscribed"""
        self.expiration_date: int = int(expiration_date)
        r"""Point in time \(Unix timestamp\) when the subscription will expire or expired"""
        self.is_canceled: bool = bool(is_canceled)
        r"""True, if the subscription was canceled"""
        self.is_expiring: bool = bool(is_expiring)
        r"""True, if the subscription expires soon and there are no enough Telegram Stars on the user's balance to extend it"""
        self.pricing: Union[StarSubscriptionPricing, None] = pricing
        r"""The subscription plan"""
        self.type: Union[StarSubscriptionTypeChannel, StarSubscriptionTypeBot, None] = (
            type
        )
        r"""Type of the subscription"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starSubscription"]:
        return "starSubscription"

    @classmethod
    def getClass(self) -> Literal["StarSubscription"]:
        return "StarSubscription"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "chat_id": self.chat_id,
            "expiration_date": self.expiration_date,
            "is_canceled": self.is_canceled,
            "is_expiring": self.is_expiring,
            "pricing": self.pricing,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarSubscription", None]:
        if data:
            data_class = cls()
            data_class.id = data.get("id", "")
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.expiration_date = int(data.get("expiration_date", 0))
            data_class.is_canceled = data.get("is_canceled", False)
            data_class.is_expiring = data.get("is_expiring", False)
            data_class.pricing = data.get("pricing", None)
            data_class.type = data.get("type", None)

        return data_class


class StarSubscriptions(TlObject):
    r"""Represents a list of Telegram Star subscriptions

    Parameters:
        star_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of owned Telegram Stars

        subscriptions (List[:class:`~pytdbot.types.StarSubscription`]):
            List of subscriptions for Telegram Stars

        required_star_count (:class:`int`):
            The number of Telegram Stars required to buy to extend subscriptions expiring soon

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        star_amount: StarAmount = None,
        subscriptions: List[StarSubscription] = None,
        required_star_count: int = 0,
        next_offset: str = "",
    ) -> None:
        self.star_amount: Union[StarAmount, None] = star_amount
        r"""The amount of owned Telegram Stars"""
        self.subscriptions: List[StarSubscription] = subscriptions or []
        r"""List of subscriptions for Telegram Stars"""
        self.required_star_count: int = int(required_star_count)
        r"""The number of Telegram Stars required to buy to extend subscriptions expiring soon"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starSubscriptions"]:
        return "starSubscriptions"

    @classmethod
    def getClass(self) -> Literal["StarSubscriptions"]:
        return "StarSubscriptions"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "star_amount": self.star_amount,
            "subscriptions": self.subscriptions,
            "required_star_count": self.required_star_count,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarSubscriptions", None]:
        if data:
            data_class = cls()
            data_class.star_amount = data.get("star_amount", None)
            data_class.subscriptions = data.get("subscriptions", None)
            data_class.required_star_count = int(data.get("required_star_count", 0))
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class AffiliateTypeCurrentUser(TlObject, AffiliateType):
    r"""The affiliate is the current user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateTypeCurrentUser"]:
        return "affiliateTypeCurrentUser"

    @classmethod
    def getClass(self) -> Literal["AffiliateType"]:
        return "AffiliateType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateTypeCurrentUser", None]:
        if data:
            data_class = cls()

        return data_class


class AffiliateTypeBot(TlObject, AffiliateType):
    r"""The affiliate is a bot owned by the current user

    Parameters:
        user_id (:class:`int`):
            User identifier of the bot

    """

    def __init__(self, user_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""User identifier of the bot"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateTypeBot"]:
        return "affiliateTypeBot"

    @classmethod
    def getClass(self) -> Literal["AffiliateType"]:
        return "AffiliateType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateTypeBot", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class AffiliateTypeChannel(TlObject, AffiliateType):
    r"""The affiliate is a channel chat where the current user has can\_post\_messages administrator right

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat

    """

    def __init__(self, chat_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateTypeChannel"]:
        return "affiliateTypeChannel"

    @classmethod
    def getClass(self) -> Literal["AffiliateType"]:
        return "AffiliateType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateTypeChannel", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))

        return data_class


class AffiliateProgramSortOrderProfitability(TlObject, AffiliateProgramSortOrder):
    r"""The affiliate programs must be sorted by the profitability"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateProgramSortOrderProfitability"]:
        return "affiliateProgramSortOrderProfitability"

    @classmethod
    def getClass(self) -> Literal["AffiliateProgramSortOrder"]:
        return "AffiliateProgramSortOrder"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AffiliateProgramSortOrderProfitability", None]:
        if data:
            data_class = cls()

        return data_class


class AffiliateProgramSortOrderCreationDate(TlObject, AffiliateProgramSortOrder):
    r"""The affiliate programs must be sorted by creation date"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateProgramSortOrderCreationDate"]:
        return "affiliateProgramSortOrderCreationDate"

    @classmethod
    def getClass(self) -> Literal["AffiliateProgramSortOrder"]:
        return "AffiliateProgramSortOrder"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["AffiliateProgramSortOrderCreationDate", None]:
        if data:
            data_class = cls()

        return data_class


class AffiliateProgramSortOrderRevenue(TlObject, AffiliateProgramSortOrder):
    r"""The affiliate programs must be sorted by the expected revenue"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateProgramSortOrderRevenue"]:
        return "affiliateProgramSortOrderRevenue"

    @classmethod
    def getClass(self) -> Literal["AffiliateProgramSortOrder"]:
        return "AffiliateProgramSortOrder"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateProgramSortOrderRevenue", None]:
        if data:
            data_class = cls()

        return data_class


class AffiliateProgramParameters(TlObject):
    r"""Describes parameters of an affiliate program

    Parameters:
        commission_per_mille (:class:`int`):
            The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner; getOption\(\"affiliate\_program\_commission\_per\_mille\_min\"\)\-getOption\(\"affiliate\_program\_commission\_per\_mille\_max\"\)

        month_count (:class:`int`):
            Number of months the program will be active; 0\-36\. If 0, then the program is eternal

    """

    def __init__(self, commission_per_mille: int = 0, month_count: int = 0) -> None:
        self.commission_per_mille: int = int(commission_per_mille)
        r"""The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner; getOption\(\"affiliate\_program\_commission\_per\_mille\_min\"\)\-getOption\(\"affiliate\_program\_commission\_per\_mille\_max\"\)"""
        self.month_count: int = int(month_count)
        r"""Number of months the program will be active; 0\-36\. If 0, then the program is eternal"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateProgramParameters"]:
        return "affiliateProgramParameters"

    @classmethod
    def getClass(self) -> Literal["AffiliateProgramParameters"]:
        return "AffiliateProgramParameters"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "commission_per_mille": self.commission_per_mille,
            "month_count": self.month_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateProgramParameters", None]:
        if data:
            data_class = cls()
            data_class.commission_per_mille = int(data.get("commission_per_mille", 0))
            data_class.month_count = int(data.get("month_count", 0))

        return data_class


class AffiliateProgramInfo(TlObject):
    r"""Contains information about an active affiliate program

    Parameters:
        parameters (:class:`~pytdbot.types.AffiliateProgramParameters`):
            Parameters of the affiliate program

        end_date (:class:`int`):
            Point in time \(Unix timestamp\) when the affiliate program will be closed; 0 if the affiliate program isn't scheduled to be closed\. If positive, then the program can't be connected using connectAffiliateProgram, but active connections will work until the date

        daily_revenue_per_user_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of daily revenue per user in Telegram Stars of the bot that created the affiliate program

    """

    def __init__(
        self,
        parameters: AffiliateProgramParameters = None,
        end_date: int = 0,
        daily_revenue_per_user_amount: StarAmount = None,
    ) -> None:
        self.parameters: Union[AffiliateProgramParameters, None] = parameters
        r"""Parameters of the affiliate program"""
        self.end_date: int = int(end_date)
        r"""Point in time \(Unix timestamp\) when the affiliate program will be closed; 0 if the affiliate program isn't scheduled to be closed\. If positive, then the program can't be connected using connectAffiliateProgram, but active connections will work until the date"""
        self.daily_revenue_per_user_amount: Union[StarAmount, None] = (
            daily_revenue_per_user_amount
        )
        r"""The amount of daily revenue per user in Telegram Stars of the bot that created the affiliate program"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateProgramInfo"]:
        return "affiliateProgramInfo"

    @classmethod
    def getClass(self) -> Literal["AffiliateProgramInfo"]:
        return "AffiliateProgramInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "parameters": self.parameters,
            "end_date": self.end_date,
            "daily_revenue_per_user_amount": self.daily_revenue_per_user_amount,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateProgramInfo", None]:
        if data:
            data_class = cls()
            data_class.parameters = data.get("parameters", None)
            data_class.end_date = int(data.get("end_date", 0))
            data_class.daily_revenue_per_user_amount = data.get(
                "daily_revenue_per_user_amount", None
            )

        return data_class


class AffiliateInfo(TlObject):
    r"""Contains information about an affiliate that received commission from a Telegram Star transaction

    Parameters:
        commission_per_mille (:class:`int`):
            The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner

        affiliate_chat_id (:class:`int`):
            Identifier of the chat which received the commission

        star_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of Telegram Stars that were received by the affiliate; can be negative for refunds

    """

    def __init__(
        self,
        commission_per_mille: int = 0,
        affiliate_chat_id: int = 0,
        star_amount: StarAmount = None,
    ) -> None:
        self.commission_per_mille: int = int(commission_per_mille)
        r"""The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner"""
        self.affiliate_chat_id: int = int(affiliate_chat_id)
        r"""Identifier of the chat which received the commission"""
        self.star_amount: Union[StarAmount, None] = star_amount
        r"""The amount of Telegram Stars that were received by the affiliate; can be negative for refunds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["affiliateInfo"]:
        return "affiliateInfo"

    @classmethod
    def getClass(self) -> Literal["AffiliateInfo"]:
        return "AffiliateInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "commission_per_mille": self.commission_per_mille,
            "affiliate_chat_id": self.affiliate_chat_id,
            "star_amount": self.star_amount,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AffiliateInfo", None]:
        if data:
            data_class = cls()
            data_class.commission_per_mille = int(data.get("commission_per_mille", 0))
            data_class.affiliate_chat_id = int(data.get("affiliate_chat_id", 0))
            data_class.star_amount = data.get("star_amount", None)

        return data_class


class FoundAffiliateProgram(TlObject):
    r"""Describes a found affiliate program

    Parameters:
        bot_user_id (:class:`int`):
            User identifier of the bot created the program

        info (:class:`~pytdbot.types.AffiliateProgramInfo`):
            Information about the affiliate program

    """

    def __init__(self, bot_user_id: int = 0, info: AffiliateProgramInfo = None) -> None:
        self.bot_user_id: int = int(bot_user_id)
        r"""User identifier of the bot created the program"""
        self.info: Union[AffiliateProgramInfo, None] = info
        r"""Information about the affiliate program"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundAffiliateProgram"]:
        return "foundAffiliateProgram"

    @classmethod
    def getClass(self) -> Literal["FoundAffiliateProgram"]:
        return "FoundAffiliateProgram"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "bot_user_id": self.bot_user_id,
            "info": self.info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundAffiliateProgram", None]:
        if data:
            data_class = cls()
            data_class.bot_user_id = int(data.get("bot_user_id", 0))
            data_class.info = data.get("info", None)

        return data_class


class FoundAffiliatePrograms(TlObject):
    r"""Represents a list of found affiliate programs

    Parameters:
        total_count (:class:`int`):
            The total number of found affiliate programs

        programs (List[:class:`~pytdbot.types.FoundAffiliateProgram`]):
            The list of affiliate programs

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        total_count: int = 0,
        programs: List[FoundAffiliateProgram] = None,
        next_offset: str = "",
    ) -> None:
        self.total_count: int = int(total_count)
        r"""The total number of found affiliate programs"""
        self.programs: List[FoundAffiliateProgram] = programs or []
        r"""The list of affiliate programs"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundAffiliatePrograms"]:
        return "foundAffiliatePrograms"

    @classmethod
    def getClass(self) -> Literal["FoundAffiliatePrograms"]:
        return "FoundAffiliatePrograms"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "programs": self.programs,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundAffiliatePrograms", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.programs = data.get("programs", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class ConnectedAffiliateProgram(TlObject):
    r"""Describes an affiliate program that was connected to an affiliate

    Parameters:
        url (:class:`str`):
            The link that can be used to refer users if the program is still active

        bot_user_id (:class:`int`):
            User identifier of the bot created the program

        parameters (:class:`~pytdbot.types.AffiliateProgramParameters`):
            The parameters of the affiliate program

        connection_date (:class:`int`):
            Point in time \(Unix timestamp\) when the affiliate program was connected

        is_disconnected (:class:`bool`):
            True, if the program was canceled by the bot, or disconnected by the chat owner and isn't available anymore

        user_count (:class:`int`):
            The number of users that used the affiliate program

        revenue_star_count (:class:`int`):
            The number of Telegram Stars that were earned by the affiliate program

    """

    def __init__(
        self,
        url: str = "",
        bot_user_id: int = 0,
        parameters: AffiliateProgramParameters = None,
        connection_date: int = 0,
        is_disconnected: bool = False,
        user_count: int = 0,
        revenue_star_count: int = 0,
    ) -> None:
        self.url: Union[str, None] = url
        r"""The link that can be used to refer users if the program is still active"""
        self.bot_user_id: int = int(bot_user_id)
        r"""User identifier of the bot created the program"""
        self.parameters: Union[AffiliateProgramParameters, None] = parameters
        r"""The parameters of the affiliate program"""
        self.connection_date: int = int(connection_date)
        r"""Point in time \(Unix timestamp\) when the affiliate program was connected"""
        self.is_disconnected: bool = bool(is_disconnected)
        r"""True, if the program was canceled by the bot, or disconnected by the chat owner and isn't available anymore"""
        self.user_count: int = int(user_count)
        r"""The number of users that used the affiliate program"""
        self.revenue_star_count: int = int(revenue_star_count)
        r"""The number of Telegram Stars that were earned by the affiliate program"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["connectedAffiliateProgram"]:
        return "connectedAffiliateProgram"

    @classmethod
    def getClass(self) -> Literal["ConnectedAffiliateProgram"]:
        return "ConnectedAffiliateProgram"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "url": self.url,
            "bot_user_id": self.bot_user_id,
            "parameters": self.parameters,
            "connection_date": self.connection_date,
            "is_disconnected": self.is_disconnected,
            "user_count": self.user_count,
            "revenue_star_count": self.revenue_star_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ConnectedAffiliateProgram", None]:
        if data:
            data_class = cls()
            data_class.url = data.get("url", "")
            data_class.bot_user_id = int(data.get("bot_user_id", 0))
            data_class.parameters = data.get("parameters", None)
            data_class.connection_date = int(data.get("connection_date", 0))
            data_class.is_disconnected = data.get("is_disconnected", False)
            data_class.user_count = int(data.get("user_count", 0))
            data_class.revenue_star_count = int(data.get("revenue_star_count", 0))

        return data_class


class ConnectedAffiliatePrograms(TlObject):
    r"""Represents a list of affiliate programs that were connected to an affiliate

    Parameters:
        total_count (:class:`int`):
            The total number of affiliate programs that were connected to the affiliate

        programs (List[:class:`~pytdbot.types.ConnectedAffiliateProgram`]):
            The list of connected affiliate programs

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        total_count: int = 0,
        programs: List[ConnectedAffiliateProgram] = None,
        next_offset: str = "",
    ) -> None:
        self.total_count: int = int(total_count)
        r"""The total number of affiliate programs that were connected to the affiliate"""
        self.programs: List[ConnectedAffiliateProgram] = programs or []
        r"""The list of connected affiliate programs"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["connectedAffiliatePrograms"]:
        return "connectedAffiliatePrograms"

    @classmethod
    def getClass(self) -> Literal["ConnectedAffiliatePrograms"]:
        return "ConnectedAffiliatePrograms"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "programs": self.programs,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ConnectedAffiliatePrograms", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.programs = data.get("programs", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class ProductInfo(TlObject):
    r"""Contains information about a product that can be paid with invoice

    Parameters:
        title (:class:`str`):
            Product title

        description (:class:`~pytdbot.types.FormattedText`):
            Product description

        photo (:class:`~pytdbot.types.Photo`):
            Product photo; may be null

    """

    def __init__(
        self, title: str = "", description: FormattedText = None, photo: Photo = None
    ) -> None:
        self.title: Union[str, None] = title
        r"""Product title"""
        self.description: Union[FormattedText, None] = description
        r"""Product description"""
        self.photo: Union[Photo, None] = photo
        r"""Product photo; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["productInfo"]:
        return "productInfo"

    @classmethod
    def getClass(self) -> Literal["ProductInfo"]:
        return "ProductInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "title": self.title,
            "description": self.description,
            "photo": self.photo,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProductInfo", None]:
        if data:
            data_class = cls()
            data_class.title = data.get("title", "")
            data_class.description = data.get("description", None)
            data_class.photo = data.get("photo", None)

        return data_class


class PremiumPaymentOption(TlObject):
    r"""Describes an option for buying Telegram Premium to a user

    Parameters:
        currency (:class:`str`):
            ISO 4217 currency code for Telegram Premium subscription payment

        amount (:class:`int`):
            The amount to pay, in the smallest units of the currency

        discount_percentage (:class:`int`):
            The discount associated with this option, as a percentage

        month_count (:class:`int`):
            Number of months the Telegram Premium subscription will be active\. Use getPremiumInfoSticker to get the sticker to be used as representation of the Telegram Premium subscription

        store_product_id (:class:`str`):
            Identifier of the store product associated with the option

        payment_link (:class:`~pytdbot.types.InternalLinkType`):
            An internal link to be opened for buying Telegram Premium to the user if store payment isn't possible; may be null if direct payment isn't available

    """

    def __init__(
        self,
        currency: str = "",
        amount: int = 0,
        discount_percentage: int = 0,
        month_count: int = 0,
        store_product_id: str = "",
        payment_link: InternalLinkType = None,
    ) -> None:
        self.currency: Union[str, None] = currency
        r"""ISO 4217 currency code for Telegram Premium subscription payment"""
        self.amount: int = int(amount)
        r"""The amount to pay, in the smallest units of the currency"""
        self.discount_percentage: int = int(discount_percentage)
        r"""The discount associated with this option, as a percentage"""
        self.month_count: int = int(month_count)
        r"""Number of months the Telegram Premium subscription will be active\. Use getPremiumInfoSticker to get the sticker to be used as representation of the Telegram Premium subscription"""
        self.store_product_id: Union[str, None] = store_product_id
        r"""Identifier of the store product associated with the option"""
        self.payment_link: Union[
            InternalLinkTypeActiveSessions,
            InternalLinkTypeAttachmentMenuBot,
            InternalLinkTypeAuthenticationCode,
            InternalLinkTypeBackground,
            InternalLinkTypeBotAddToChannel,
            InternalLinkTypeBotStart,
            InternalLinkTypeBotStartInGroup,
            InternalLinkTypeBusinessChat,
            InternalLinkTypeBuyStars,
            InternalLinkTypeChangePhoneNumber,
            InternalLinkTypeChatAffiliateProgram,
            InternalLinkTypeChatBoost,
            InternalLinkTypeChatFolderInvite,
            InternalLinkTypeChatFolderSettings,
            InternalLinkTypeChatInvite,
            InternalLinkTypeDefaultMessageAutoDeleteTimerSettings,
            InternalLinkTypeDirectMessagesChat,
            InternalLinkTypeEditProfileSettings,
            InternalLinkTypeGame,
            InternalLinkTypeGiftCollection,
            InternalLinkTypeGroupCall,
            InternalLinkTypeInstantView,
            InternalLinkTypeInvoice,
            InternalLinkTypeLanguagePack,
            InternalLinkTypeLanguageSettings,
            InternalLinkTypeMainWebApp,
            InternalLinkTypeMessage,
            InternalLinkTypeMessageDraft,
            InternalLinkTypeMyStars,
            InternalLinkTypeMyToncoins,
            InternalLinkTypePassportDataRequest,
            InternalLinkTypePhoneNumberConfirmation,
            InternalLinkTypePremiumFeatures,
            InternalLinkTypePremiumGift,
            InternalLinkTypePremiumGiftCode,
            InternalLinkTypePrivacyAndSecuritySettings,
            InternalLinkTypeProxy,
            InternalLinkTypePublicChat,
            InternalLinkTypeQrCodeAuthentication,
            InternalLinkTypeRestorePurchases,
            InternalLinkTypeSettings,
            InternalLinkTypeStickerSet,
            InternalLinkTypeStory,
            InternalLinkTypeStoryAlbum,
            InternalLinkTypeTheme,
            InternalLinkTypeThemeSettings,
            InternalLinkTypeUnknownDeepLink,
            InternalLinkTypeUnsupportedProxy,
            InternalLinkTypeUpgradedGift,
            InternalLinkTypeUserPhoneNumber,
            InternalLinkTypeUserToken,
            InternalLinkTypeVideoChat,
            InternalLinkTypeWebApp,
            None,
        ] = payment_link
        r"""An internal link to be opened for buying Telegram Premium to the user if store payment isn't possible; may be null if direct payment isn't available"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumPaymentOption"]:
        return "premiumPaymentOption"

    @classmethod
    def getClass(self) -> Literal["PremiumPaymentOption"]:
        return "PremiumPaymentOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "currency": self.currency,
            "amount": self.amount,
            "discount_percentage": self.discount_percentage,
            "month_count": self.month_count,
            "store_product_id": self.store_product_id,
            "payment_link": self.payment_link,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumPaymentOption", None]:
        if data:
            data_class = cls()
            data_class.currency = data.get("currency", "")
            data_class.amount = int(data.get("amount", 0))
            data_class.discount_percentage = int(data.get("discount_percentage", 0))
            data_class.month_count = int(data.get("month_count", 0))
            data_class.store_product_id = data.get("store_product_id", "")
            data_class.payment_link = data.get("payment_link", None)

        return data_class


class PremiumStatePaymentOption(TlObject):
    r"""Describes an option for buying or upgrading Telegram Premium for self

    Parameters:
        payment_option (:class:`~pytdbot.types.PremiumPaymentOption`):
            Information about the payment option

        is_current (:class:`bool`):
            True, if this is the currently used Telegram Premium subscription option

        is_upgrade (:class:`bool`):
            True, if the payment option can be used to upgrade the existing Telegram Premium subscription

        last_transaction_id (:class:`str`):
            Identifier of the last in\-store transaction for the currently used option

    """

    def __init__(
        self,
        payment_option: PremiumPaymentOption = None,
        is_current: bool = False,
        is_upgrade: bool = False,
        last_transaction_id: str = "",
    ) -> None:
        self.payment_option: Union[PremiumPaymentOption, None] = payment_option
        r"""Information about the payment option"""
        self.is_current: bool = bool(is_current)
        r"""True, if this is the currently used Telegram Premium subscription option"""
        self.is_upgrade: bool = bool(is_upgrade)
        r"""True, if the payment option can be used to upgrade the existing Telegram Premium subscription"""
        self.last_transaction_id: Union[str, None] = last_transaction_id
        r"""Identifier of the last in\-store transaction for the currently used option"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumStatePaymentOption"]:
        return "premiumStatePaymentOption"

    @classmethod
    def getClass(self) -> Literal["PremiumStatePaymentOption"]:
        return "PremiumStatePaymentOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "payment_option": self.payment_option,
            "is_current": self.is_current,
            "is_upgrade": self.is_upgrade,
            "last_transaction_id": self.last_transaction_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumStatePaymentOption", None]:
        if data:
            data_class = cls()
            data_class.payment_option = data.get("payment_option", None)
            data_class.is_current = data.get("is_current", False)
            data_class.is_upgrade = data.get("is_upgrade", False)
            data_class.last_transaction_id = data.get("last_transaction_id", "")

        return data_class


class PremiumGiftPaymentOption(TlObject):
    r"""Describes an option for gifting Telegram Premium to a user\. Use telegramPaymentPurposePremiumGift for out\-of\-store payments or payments in Telegram Stars

    Parameters:
        currency (:class:`str`):
            ISO 4217 currency code for the payment

        amount (:class:`int`):
            The amount to pay, in the smallest units of the currency

        star_count (:class:`int`):
            The alternative amount of Telegram Stars to pay; 0 if payment in Telegram Stars is not possible

        discount_percentage (:class:`int`):
            The discount associated with this option, as a percentage

        month_count (:class:`int`):
            Number of months the Telegram Premium subscription will be active

        store_product_id (:class:`str`):
            Identifier of the store product associated with the option

        sticker (:class:`~pytdbot.types.Sticker`):
            A sticker to be shown along with the option; may be null if unknown

    """

    def __init__(
        self,
        currency: str = "",
        amount: int = 0,
        star_count: int = 0,
        discount_percentage: int = 0,
        month_count: int = 0,
        store_product_id: str = "",
        sticker: Sticker = None,
    ) -> None:
        self.currency: Union[str, None] = currency
        r"""ISO 4217 currency code for the payment"""
        self.amount: int = int(amount)
        r"""The amount to pay, in the smallest units of the currency"""
        self.star_count: int = int(star_count)
        r"""The alternative amount of Telegram Stars to pay; 0 if payment in Telegram Stars is not possible"""
        self.discount_percentage: int = int(discount_percentage)
        r"""The discount associated with this option, as a percentage"""
        self.month_count: int = int(month_count)
        r"""Number of months the Telegram Premium subscription will be active"""
        self.store_product_id: Union[str, None] = store_product_id
        r"""Identifier of the store product associated with the option"""
        self.sticker: Union[Sticker, None] = sticker
        r"""A sticker to be shown along with the option; may be null if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumGiftPaymentOption"]:
        return "premiumGiftPaymentOption"

    @classmethod
    def getClass(self) -> Literal["PremiumGiftPaymentOption"]:
        return "PremiumGiftPaymentOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "currency": self.currency,
            "amount": self.amount,
            "star_count": self.star_count,
            "discount_percentage": self.discount_percentage,
            "month_count": self.month_count,
            "store_product_id": self.store_product_id,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumGiftPaymentOption", None]:
        if data:
            data_class = cls()
            data_class.currency = data.get("currency", "")
            data_class.amount = int(data.get("amount", 0))
            data_class.star_count = int(data.get("star_count", 0))
            data_class.discount_percentage = int(data.get("discount_percentage", 0))
            data_class.month_count = int(data.get("month_count", 0))
            data_class.store_product_id = data.get("store_product_id", "")
            data_class.sticker = data.get("sticker", None)

        return data_class


class PremiumGiftPaymentOptions(TlObject):
    r"""Contains a list of options for gifting Telegram Premium to a user

    Parameters:
        options (List[:class:`~pytdbot.types.PremiumGiftPaymentOption`]):
            The list of options sorted by Telegram Premium subscription duration

    """

    def __init__(self, options: List[PremiumGiftPaymentOption] = None) -> None:
        self.options: List[PremiumGiftPaymentOption] = options or []
        r"""The list of options sorted by Telegram Premium subscription duration"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumGiftPaymentOptions"]:
        return "premiumGiftPaymentOptions"

    @classmethod
    def getClass(self) -> Literal["PremiumGiftPaymentOptions"]:
        return "PremiumGiftPaymentOptions"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "options": self.options}

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumGiftPaymentOptions", None]:
        if data:
            data_class = cls()
            data_class.options = data.get("options", None)

        return data_class


class PremiumGiveawayPaymentOption(TlObject):
    r"""Describes an option for creating of Telegram Premium giveaway or manual distribution of Telegram Premium among chat members\. Use telegramPaymentPurposePremiumGiftCodes or telegramPaymentPurposePremiumGiveaway for out\-of\-store payments

    Parameters:
        currency (:class:`str`):
            ISO 4217 currency code for Telegram Premium gift code payment

        amount (:class:`int`):
            The amount to pay, in the smallest units of the currency

        winner_count (:class:`int`):
            Number of users which will be able to activate the gift codes

        month_count (:class:`int`):
            Number of months the Telegram Premium subscription will be active

        store_product_id (:class:`str`):
            Identifier of the store product associated with the option; may be empty if none

        store_product_quantity (:class:`int`):
            Number of times the store product must be paid

    """

    def __init__(
        self,
        currency: str = "",
        amount: int = 0,
        winner_count: int = 0,
        month_count: int = 0,
        store_product_id: str = "",
        store_product_quantity: int = 0,
    ) -> None:
        self.currency: Union[str, None] = currency
        r"""ISO 4217 currency code for Telegram Premium gift code payment"""
        self.amount: int = int(amount)
        r"""The amount to pay, in the smallest units of the currency"""
        self.winner_count: int = int(winner_count)
        r"""Number of users which will be able to activate the gift codes"""
        self.month_count: int = int(month_count)
        r"""Number of months the Telegram Premium subscription will be active"""
        self.store_product_id: Union[str, None] = store_product_id
        r"""Identifier of the store product associated with the option; may be empty if none"""
        self.store_product_quantity: int = int(store_product_quantity)
        r"""Number of times the store product must be paid"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumGiveawayPaymentOption"]:
        return "premiumGiveawayPaymentOption"

    @classmethod
    def getClass(self) -> Literal["PremiumGiveawayPaymentOption"]:
        return "PremiumGiveawayPaymentOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "currency": self.currency,
            "amount": self.amount,
            "winner_count": self.winner_count,
            "month_count": self.month_count,
            "store_product_id": self.store_product_id,
            "store_product_quantity": self.store_product_quantity,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumGiveawayPaymentOption", None]:
        if data:
            data_class = cls()
            data_class.currency = data.get("currency", "")
            data_class.amount = int(data.get("amount", 0))
            data_class.winner_count = int(data.get("winner_count", 0))
            data_class.month_count = int(data.get("month_count", 0))
            data_class.store_product_id = data.get("store_product_id", "")
            data_class.store_product_quantity = int(
                data.get("store_product_quantity", 0)
            )

        return data_class


class PremiumGiveawayPaymentOptions(TlObject):
    r"""Contains a list of options for creating of Telegram Premium giveaway or manual distribution of Telegram Premium among chat members

    Parameters:
        options (List[:class:`~pytdbot.types.PremiumGiveawayPaymentOption`]):
            The list of options

    """

    def __init__(self, options: List[PremiumGiveawayPaymentOption] = None) -> None:
        self.options: List[PremiumGiveawayPaymentOption] = options or []
        r"""The list of options"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumGiveawayPaymentOptions"]:
        return "premiumGiveawayPaymentOptions"

    @classmethod
    def getClass(self) -> Literal["PremiumGiveawayPaymentOptions"]:
        return "PremiumGiveawayPaymentOptions"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "options": self.options}

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumGiveawayPaymentOptions", None]:
        if data:
            data_class = cls()
            data_class.options = data.get("options", None)

        return data_class


class PremiumGiftCodeInfo(TlObject):
    r"""Contains information about a Telegram Premium gift code

    Parameters:
        creator_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of a chat or a user that created the gift code; may be null if unknown\. If null and the code is from messagePremiumGiftCode message, then creator\_id from the message can be used

        creation_date (:class:`int`):
            Point in time \(Unix timestamp\) when the code was created

        is_from_giveaway (:class:`bool`):
            True, if the gift code was created for a giveaway

        giveaway_message_id (:class:`int`):
            Identifier of the corresponding giveaway message in the creator\_id chat; can be 0 or an identifier of a deleted message

        month_count (:class:`int`):
            Number of months the Telegram Premium subscription will be active after code activation

        user_id (:class:`int`):
            Identifier of a user for which the code was created; 0 if none

        use_date (:class:`int`):
            Point in time \(Unix timestamp\) when the code was activated; 0 if none

    """

    def __init__(
        self,
        creator_id: MessageSender = None,
        creation_date: int = 0,
        is_from_giveaway: bool = False,
        giveaway_message_id: int = 0,
        month_count: int = 0,
        user_id: int = 0,
        use_date: int = 0,
    ) -> None:
        self.creator_id: Union[MessageSenderUser, MessageSenderChat, None] = creator_id
        r"""Identifier of a chat or a user that created the gift code; may be null if unknown\. If null and the code is from messagePremiumGiftCode message, then creator\_id from the message can be used"""
        self.creation_date: int = int(creation_date)
        r"""Point in time \(Unix timestamp\) when the code was created"""
        self.is_from_giveaway: bool = bool(is_from_giveaway)
        r"""True, if the gift code was created for a giveaway"""
        self.giveaway_message_id: int = int(giveaway_message_id)
        r"""Identifier of the corresponding giveaway message in the creator\_id chat; can be 0 or an identifier of a deleted message"""
        self.month_count: int = int(month_count)
        r"""Number of months the Telegram Premium subscription will be active after code activation"""
        self.user_id: int = int(user_id)
        r"""Identifier of a user for which the code was created; 0 if none"""
        self.use_date: int = int(use_date)
        r"""Point in time \(Unix timestamp\) when the code was activated; 0 if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["premiumGiftCodeInfo"]:
        return "premiumGiftCodeInfo"

    @classmethod
    def getClass(self) -> Literal["PremiumGiftCodeInfo"]:
        return "PremiumGiftCodeInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "creator_id": self.creator_id,
            "creation_date": self.creation_date,
            "is_from_giveaway": self.is_from_giveaway,
            "giveaway_message_id": self.giveaway_message_id,
            "month_count": self.month_count,
            "user_id": self.user_id,
            "use_date": self.use_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PremiumGiftCodeInfo", None]:
        if data:
            data_class = cls()
            data_class.creator_id = data.get("creator_id", None)
            data_class.creation_date = int(data.get("creation_date", 0))
            data_class.is_from_giveaway = data.get("is_from_giveaway", False)
            data_class.giveaway_message_id = int(data.get("giveaway_message_id", 0))
            data_class.month_count = int(data.get("month_count", 0))
            data_class.user_id = int(data.get("user_id", 0))
            data_class.use_date = int(data.get("use_date", 0))

        return data_class


class StarPaymentOption(TlObject):
    r"""Describes an option for buying Telegram Stars\. Use telegramPaymentPurposeStars for out\-of\-store payments

    Parameters:
        currency (:class:`str`):
            ISO 4217 currency code for the payment

        amount (:class:`int`):
            The amount to pay, in the smallest units of the currency

        star_count (:class:`int`):
            Number of Telegram Stars that will be purchased

        store_product_id (:class:`str`):
            Identifier of the store product associated with the option; may be empty if none

        is_additional (:class:`bool`):
            True, if the option must be shown only in the full list of payment options

    """

    def __init__(
        self,
        currency: str = "",
        amount: int = 0,
        star_count: int = 0,
        store_product_id: str = "",
        is_additional: bool = False,
    ) -> None:
        self.currency: Union[str, None] = currency
        r"""ISO 4217 currency code for the payment"""
        self.amount: int = int(amount)
        r"""The amount to pay, in the smallest units of the currency"""
        self.star_count: int = int(star_count)
        r"""Number of Telegram Stars that will be purchased"""
        self.store_product_id: Union[str, None] = store_product_id
        r"""Identifier of the store product associated with the option; may be empty if none"""
        self.is_additional: bool = bool(is_additional)
        r"""True, if the option must be shown only in the full list of payment options"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starPaymentOption"]:
        return "starPaymentOption"

    @classmethod
    def getClass(self) -> Literal["StarPaymentOption"]:
        return "StarPaymentOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "currency": self.currency,
            "amount": self.amount,
            "star_count": self.star_count,
            "store_product_id": self.store_product_id,
            "is_additional": self.is_additional,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarPaymentOption", None]:
        if data:
            data_class = cls()
            data_class.currency = data.get("currency", "")
            data_class.amount = int(data.get("amount", 0))
            data_class.star_count = int(data.get("star_count", 0))
            data_class.store_product_id = data.get("store_product_id", "")
            data_class.is_additional = data.get("is_additional", False)

        return data_class


class StarPaymentOptions(TlObject):
    r"""Contains a list of options for buying Telegram Stars

    Parameters:
        options (List[:class:`~pytdbot.types.StarPaymentOption`]):
            The list of options

    """

    def __init__(self, options: List[StarPaymentOption] = None) -> None:
        self.options: List[StarPaymentOption] = options or []
        r"""The list of options"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starPaymentOptions"]:
        return "starPaymentOptions"

    @classmethod
    def getClass(self) -> Literal["StarPaymentOptions"]:
        return "StarPaymentOptions"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "options": self.options}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarPaymentOptions", None]:
        if data:
            data_class = cls()
            data_class.options = data.get("options", None)

        return data_class


class StarGiveawayWinnerOption(TlObject):
    r"""Describes an option for the number of winners of a Telegram Star giveaway

    Parameters:
        winner_count (:class:`int`):
            The number of users that will be chosen as winners

        won_star_count (:class:`int`):
            The number of Telegram Stars that will be won by the winners of the giveaway

        is_default (:class:`bool`):
            True, if the option must be chosen by default

    """

    def __init__(
        self, winner_count: int = 0, won_star_count: int = 0, is_default: bool = False
    ) -> None:
        self.winner_count: int = int(winner_count)
        r"""The number of users that will be chosen as winners"""
        self.won_star_count: int = int(won_star_count)
        r"""The number of Telegram Stars that will be won by the winners of the giveaway"""
        self.is_default: bool = bool(is_default)
        r"""True, if the option must be chosen by default"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starGiveawayWinnerOption"]:
        return "starGiveawayWinnerOption"

    @classmethod
    def getClass(self) -> Literal["StarGiveawayWinnerOption"]:
        return "StarGiveawayWinnerOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "winner_count": self.winner_count,
            "won_star_count": self.won_star_count,
            "is_default": self.is_default,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarGiveawayWinnerOption", None]:
        if data:
            data_class = cls()
            data_class.winner_count = int(data.get("winner_count", 0))
            data_class.won_star_count = int(data.get("won_star_count", 0))
            data_class.is_default = data.get("is_default", False)

        return data_class


class StarGiveawayPaymentOption(TlObject):
    r"""Describes an option for creating of Telegram Star giveaway\. Use telegramPaymentPurposeStarGiveaway for out\-of\-store payments

    Parameters:
        currency (:class:`str`):
            ISO 4217 currency code for the payment

        amount (:class:`int`):
            The amount to pay, in the smallest units of the currency

        star_count (:class:`int`):
            Number of Telegram Stars that will be distributed among winners

        store_product_id (:class:`str`):
            Identifier of the store product associated with the option; may be empty if none

        yearly_boost_count (:class:`int`):
            Number of times the chat will be boosted for one year if the option is chosen

        winner_options (List[:class:`~pytdbot.types.StarGiveawayWinnerOption`]):
            Allowed options for the number of giveaway winners

        is_default (:class:`bool`):
            True, if the option must be chosen by default

        is_additional (:class:`bool`):
            True, if the option must be shown only in the full list of payment options

    """

    def __init__(
        self,
        currency: str = "",
        amount: int = 0,
        star_count: int = 0,
        store_product_id: str = "",
        yearly_boost_count: int = 0,
        winner_options: List[StarGiveawayWinnerOption] = None,
        is_default: bool = False,
        is_additional: bool = False,
    ) -> None:
        self.currency: Union[str, None] = currency
        r"""ISO 4217 currency code for the payment"""
        self.amount: int = int(amount)
        r"""The amount to pay, in the smallest units of the currency"""
        self.star_count: int = int(star_count)
        r"""Number of Telegram Stars that will be distributed among winners"""
        self.store_product_id: Union[str, None] = store_product_id
        r"""Identifier of the store product associated with the option; may be empty if none"""
        self.yearly_boost_count: int = int(yearly_boost_count)
        r"""Number of times the chat will be boosted for one year if the option is chosen"""
        self.winner_options: List[StarGiveawayWinnerOption] = winner_options or []
        r"""Allowed options for the number of giveaway winners"""
        self.is_default: bool = bool(is_default)
        r"""True, if the option must be chosen by default"""
        self.is_additional: bool = bool(is_additional)
        r"""True, if the option must be shown only in the full list of payment options"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starGiveawayPaymentOption"]:
        return "starGiveawayPaymentOption"

    @classmethod
    def getClass(self) -> Literal["StarGiveawayPaymentOption"]:
        return "StarGiveawayPaymentOption"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "currency": self.currency,
            "amount": self.amount,
            "star_count": self.star_count,
            "store_product_id": self.store_product_id,
            "yearly_boost_count": self.yearly_boost_count,
            "winner_options": self.winner_options,
            "is_default": self.is_default,
            "is_additional": self.is_additional,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarGiveawayPaymentOption", None]:
        if data:
            data_class = cls()
            data_class.currency = data.get("currency", "")
            data_class.amount = int(data.get("amount", 0))
            data_class.star_count = int(data.get("star_count", 0))
            data_class.store_product_id = data.get("store_product_id", "")
            data_class.yearly_boost_count = int(data.get("yearly_boost_count", 0))
            data_class.winner_options = data.get("winner_options", None)
            data_class.is_default = data.get("is_default", False)
            data_class.is_additional = data.get("is_additional", False)

        return data_class


class StarGiveawayPaymentOptions(TlObject):
    r"""Contains a list of options for creating of Telegram Star giveaway

    Parameters:
        options (List[:class:`~pytdbot.types.StarGiveawayPaymentOption`]):
            The list of options

    """

    def __init__(self, options: List[StarGiveawayPaymentOption] = None) -> None:
        self.options: List[StarGiveawayPaymentOption] = options or []
        r"""The list of options"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starGiveawayPaymentOptions"]:
        return "starGiveawayPaymentOptions"

    @classmethod
    def getClass(self) -> Literal["StarGiveawayPaymentOptions"]:
        return "StarGiveawayPaymentOptions"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "options": self.options}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarGiveawayPaymentOptions", None]:
        if data:
            data_class = cls()
            data_class.options = data.get("options", None)

        return data_class


class AcceptedGiftTypes(TlObject):
    r"""Describes gift types that are accepted by a user

    Parameters:
        unlimited_gifts (:class:`bool`):
            True, if unlimited regular gifts are accepted

        limited_gifts (:class:`bool`):
            True, if limited regular gifts are accepted

        upgraded_gifts (:class:`bool`):
            True, if upgraded gifts and regular gifts that can be upgraded for free are accepted

        premium_subscription (:class:`bool`):
            True, if Telegram Premium subscription is accepted

    """

    def __init__(
        self,
        unlimited_gifts: bool = False,
        limited_gifts: bool = False,
        upgraded_gifts: bool = False,
        premium_subscription: bool = False,
    ) -> None:
        self.unlimited_gifts: bool = bool(unlimited_gifts)
        r"""True, if unlimited regular gifts are accepted"""
        self.limited_gifts: bool = bool(limited_gifts)
        r"""True, if limited regular gifts are accepted"""
        self.upgraded_gifts: bool = bool(upgraded_gifts)
        r"""True, if upgraded gifts and regular gifts that can be upgraded for free are accepted"""
        self.premium_subscription: bool = bool(premium_subscription)
        r"""True, if Telegram Premium subscription is accepted"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["acceptedGiftTypes"]:
        return "acceptedGiftTypes"

    @classmethod
    def getClass(self) -> Literal["AcceptedGiftTypes"]:
        return "AcceptedGiftTypes"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "unlimited_gifts": self.unlimited_gifts,
            "limited_gifts": self.limited_gifts,
            "upgraded_gifts": self.upgraded_gifts,
            "premium_subscription": self.premium_subscription,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AcceptedGiftTypes", None]:
        if data:
            data_class = cls()
            data_class.unlimited_gifts = data.get("unlimited_gifts", False)
            data_class.limited_gifts = data.get("limited_gifts", False)
            data_class.upgraded_gifts = data.get("upgraded_gifts", False)
            data_class.premium_subscription = data.get("premium_subscription", False)

        return data_class


class GiftSettings(TlObject):
    r"""Contains settings for gift receiving for a user

    Parameters:
        show_gift_button (:class:`bool`):
            True, if a button for sending a gift to the user or by the user must always be shown in the input field

        accepted_gift_types (:class:`~pytdbot.types.AcceptedGiftTypes`):
            Types of gifts accepted by the user; for Telegram Premium users only

    """

    def __init__(
        self,
        show_gift_button: bool = False,
        accepted_gift_types: AcceptedGiftTypes = None,
    ) -> None:
        self.show_gift_button: bool = bool(show_gift_button)
        r"""True, if a button for sending a gift to the user or by the user must always be shown in the input field"""
        self.accepted_gift_types: Union[AcceptedGiftTypes, None] = accepted_gift_types
        r"""Types of gifts accepted by the user; for Telegram Premium users only"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftSettings"]:
        return "giftSettings"

    @classmethod
    def getClass(self) -> Literal["GiftSettings"]:
        return "GiftSettings"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "show_gift_button": self.show_gift_button,
            "accepted_gift_types": self.accepted_gift_types,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftSettings", None]:
        if data:
            data_class = cls()
            data_class.show_gift_button = data.get("show_gift_button", False)
            data_class.accepted_gift_types = data.get("accepted_gift_types", None)

        return data_class


class GiftPurchaseLimits(TlObject):
    r"""Describes the maximum number of times that a specific gift can be purchased

    Parameters:
        total_count (:class:`int`):
            The maximum number of times the gifts can be purchased

        remaining_count (:class:`int`):
            Number of remaining times the gift can be purchased

    """

    def __init__(self, total_count: int = 0, remaining_count: int = 0) -> None:
        self.total_count: int = int(total_count)
        r"""The maximum number of times the gifts can be purchased"""
        self.remaining_count: int = int(remaining_count)
        r"""Number of remaining times the gift can be purchased"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftPurchaseLimits"]:
        return "giftPurchaseLimits"

    @classmethod
    def getClass(self) -> Literal["GiftPurchaseLimits"]:
        return "GiftPurchaseLimits"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "remaining_count": self.remaining_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftPurchaseLimits", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.remaining_count = int(data.get("remaining_count", 0))

        return data_class


class GiftResaleParameters(TlObject):
    r"""Describes parameters of a unique gift available for resale

    Parameters:
        star_count (:class:`int`):
            Resale price of the gift in Telegram Stars

        toncoin_cent_count (:class:`int`):
            Resale price of the gift in 1/100 of Toncoin

        toncoin_only (:class:`bool`):
            True, if the gift can be bought only using Toncoins

    """

    def __init__(
        self,
        star_count: int = 0,
        toncoin_cent_count: int = 0,
        toncoin_only: bool = False,
    ) -> None:
        self.star_count: int = int(star_count)
        r"""Resale price of the gift in Telegram Stars"""
        self.toncoin_cent_count: int = int(toncoin_cent_count)
        r"""Resale price of the gift in 1/100 of Toncoin"""
        self.toncoin_only: bool = bool(toncoin_only)
        r"""True, if the gift can be bought only using Toncoins"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftResaleParameters"]:
        return "giftResaleParameters"

    @classmethod
    def getClass(self) -> Literal["GiftResaleParameters"]:
        return "GiftResaleParameters"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "star_count": self.star_count,
            "toncoin_cent_count": self.toncoin_cent_count,
            "toncoin_only": self.toncoin_only,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftResaleParameters", None]:
        if data:
            data_class = cls()
            data_class.star_count = int(data.get("star_count", 0))
            data_class.toncoin_cent_count = int(data.get("toncoin_cent_count", 0))
            data_class.toncoin_only = data.get("toncoin_only", False)

        return data_class


class GiftCollection(TlObject):
    r"""Describes collection of gifts

    Parameters:
        id (:class:`int`):
            Unique identifier of the collection

        name (:class:`str`):
            Name of the collection

        icon (:class:`~pytdbot.types.Sticker`):
            Icon of the collection; may be null if none

        gift_count (:class:`int`):
            Total number of gifts in the collection

    """

    def __init__(
        self, id: int = 0, name: str = "", icon: Sticker = None, gift_count: int = 0
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the collection"""
        self.name: Union[str, None] = name
        r"""Name of the collection"""
        self.icon: Union[Sticker, None] = icon
        r"""Icon of the collection; may be null if none"""
        self.gift_count: int = int(gift_count)
        r"""Total number of gifts in the collection"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftCollection"]:
        return "giftCollection"

    @classmethod
    def getClass(self) -> Literal["GiftCollection"]:
        return "GiftCollection"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "name": self.name,
            "icon": self.icon,
            "gift_count": self.gift_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftCollection", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.name = data.get("name", "")
            data_class.icon = data.get("icon", None)
            data_class.gift_count = int(data.get("gift_count", 0))

        return data_class


class GiftCollections(TlObject):
    r"""Contains a list of gift collections

    Parameters:
        collections (List[:class:`~pytdbot.types.GiftCollection`]):
            List of gift collections

    """

    def __init__(self, collections: List[GiftCollection] = None) -> None:
        self.collections: List[GiftCollection] = collections or []
        r"""List of gift collections"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftCollections"]:
        return "giftCollections"

    @classmethod
    def getClass(self) -> Literal["GiftCollections"]:
        return "GiftCollections"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "collections": self.collections}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftCollections", None]:
        if data:
            data_class = cls()
            data_class.collections = data.get("collections", None)

        return data_class


class CanSendGiftResultOk(TlObject, CanSendGiftResult):
    r"""The gift can be sent now by the current user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["canSendGiftResultOk"]:
        return "canSendGiftResultOk"

    @classmethod
    def getClass(self) -> Literal["CanSendGiftResult"]:
        return "CanSendGiftResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["CanSendGiftResultOk", None]:
        if data:
            data_class = cls()

        return data_class


class CanSendGiftResultFail(TlObject, CanSendGiftResult):
    r"""The gift can't be sent now by the current user

    Parameters:
        reason (:class:`~pytdbot.types.FormattedText`):
            Reason to be shown to the user

    """

    def __init__(self, reason: FormattedText = None) -> None:
        self.reason: Union[FormattedText, None] = reason
        r"""Reason to be shown to the user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["canSendGiftResultFail"]:
        return "canSendGiftResultFail"

    @classmethod
    def getClass(self) -> Literal["CanSendGiftResult"]:
        return "CanSendGiftResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "reason": self.reason}

    @classmethod
    def from_dict(cls, data: dict) -> Union["CanSendGiftResultFail", None]:
        if data:
            data_class = cls()
            data_class.reason = data.get("reason", None)

        return data_class


class UpgradedGiftOriginUpgrade(TlObject, UpgradedGiftOrigin):
    r"""The gift was obtained by upgrading of a previously received gift

    Parameters:
        gift_message_id (:class:`int`):
            Identifier of the message with the regular gift that was upgraded; can be 0 or an identifier of a deleted message

    """

    def __init__(self, gift_message_id: int = 0) -> None:
        self.gift_message_id: int = int(gift_message_id)
        r"""Identifier of the message with the regular gift that was upgraded; can be 0 or an identifier of a deleted message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftOriginUpgrade"]:
        return "upgradedGiftOriginUpgrade"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftOrigin"]:
        return "UpgradedGiftOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "gift_message_id": self.gift_message_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftOriginUpgrade", None]:
        if data:
            data_class = cls()
            data_class.gift_message_id = int(data.get("gift_message_id", 0))

        return data_class


class UpgradedGiftOriginTransfer(TlObject, UpgradedGiftOrigin):
    r"""The gift was transferred from another owner"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftOriginTransfer"]:
        return "upgradedGiftOriginTransfer"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftOrigin"]:
        return "UpgradedGiftOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftOriginTransfer", None]:
        if data:
            data_class = cls()

        return data_class


class UpgradedGiftOriginResale(TlObject, UpgradedGiftOrigin):
    r"""The gift was bought from another user

    Parameters:
        price (:class:`~pytdbot.types.GiftResalePrice`):
            Price paid by the sender for the gift

    """

    def __init__(self, price: GiftResalePrice = None) -> None:
        self.price: Union[GiftResalePriceStar, GiftResalePriceTon, None] = price
        r"""Price paid by the sender for the gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftOriginResale"]:
        return "upgradedGiftOriginResale"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftOrigin"]:
        return "UpgradedGiftOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "price": self.price}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftOriginResale", None]:
        if data:
            data_class = cls()
            data_class.price = data.get("price", None)

        return data_class


class UpgradedGiftOriginBlockchain(TlObject, UpgradedGiftOrigin):
    r"""The gift was assigned from blockchain and isn't owned by the current user\. The gift can't be transferred, resold or withdrawn to blockchain"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftOriginBlockchain"]:
        return "upgradedGiftOriginBlockchain"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftOrigin"]:
        return "UpgradedGiftOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftOriginBlockchain", None]:
        if data:
            data_class = cls()

        return data_class


class UpgradedGiftOriginPrepaidUpgrade(TlObject, UpgradedGiftOrigin):
    r"""The sender or receiver of the message has paid for upgraid of the gift, which has been completed"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftOriginPrepaidUpgrade"]:
        return "upgradedGiftOriginPrepaidUpgrade"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftOrigin"]:
        return "UpgradedGiftOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftOriginPrepaidUpgrade", None]:
        if data:
            data_class = cls()

        return data_class


class UpgradedGiftModel(TlObject):
    r"""Describes a model of an upgraded gift

    Parameters:
        name (:class:`str`):
            Name of the model

        sticker (:class:`~pytdbot.types.Sticker`):
            The sticker representing the upgraded gift

        rarity_per_mille (:class:`int`):
            The number of upgraded gifts that receive this model for each 1000 gifts upgraded

    """

    def __init__(
        self, name: str = "", sticker: Sticker = None, rarity_per_mille: int = 0
    ) -> None:
        self.name: Union[str, None] = name
        r"""Name of the model"""
        self.sticker: Union[Sticker, None] = sticker
        r"""The sticker representing the upgraded gift"""
        self.rarity_per_mille: int = int(rarity_per_mille)
        r"""The number of upgraded gifts that receive this model for each 1000 gifts upgraded"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftModel"]:
        return "upgradedGiftModel"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftModel"]:
        return "UpgradedGiftModel"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "name": self.name,
            "sticker": self.sticker,
            "rarity_per_mille": self.rarity_per_mille,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftModel", None]:
        if data:
            data_class = cls()
            data_class.name = data.get("name", "")
            data_class.sticker = data.get("sticker", None)
            data_class.rarity_per_mille = int(data.get("rarity_per_mille", 0))

        return data_class


class UpgradedGiftSymbol(TlObject):
    r"""Describes a symbol shown on the pattern of an upgraded gift

    Parameters:
        name (:class:`str`):
            Name of the symbol

        sticker (:class:`~pytdbot.types.Sticker`):
            The sticker representing the symbol

        rarity_per_mille (:class:`int`):
            The number of upgraded gifts that receive this symbol for each 1000 gifts upgraded

    """

    def __init__(
        self, name: str = "", sticker: Sticker = None, rarity_per_mille: int = 0
    ) -> None:
        self.name: Union[str, None] = name
        r"""Name of the symbol"""
        self.sticker: Union[Sticker, None] = sticker
        r"""The sticker representing the symbol"""
        self.rarity_per_mille: int = int(rarity_per_mille)
        r"""The number of upgraded gifts that receive this symbol for each 1000 gifts upgraded"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftSymbol"]:
        return "upgradedGiftSymbol"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftSymbol"]:
        return "UpgradedGiftSymbol"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "name": self.name,
            "sticker": self.sticker,
            "rarity_per_mille": self.rarity_per_mille,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftSymbol", None]:
        if data:
            data_class = cls()
            data_class.name = data.get("name", "")
            data_class.sticker = data.get("sticker", None)
            data_class.rarity_per_mille = int(data.get("rarity_per_mille", 0))

        return data_class


class UpgradedGiftBackdropColors(TlObject):
    r"""Describes colors of a backdrop of an upgraded gift

    Parameters:
        center_color (:class:`int`):
            A color in the center of the backdrop in the RGB format

        edge_color (:class:`int`):
            A color on the edges of the backdrop in the RGB format

        symbol_color (:class:`int`):
            A color to be applied for the symbol in the RGB format

        text_color (:class:`int`):
            A color for the text on the backdrop in the RGB format

    """

    def __init__(
        self,
        center_color: int = 0,
        edge_color: int = 0,
        symbol_color: int = 0,
        text_color: int = 0,
    ) -> None:
        self.center_color: int = int(center_color)
        r"""A color in the center of the backdrop in the RGB format"""
        self.edge_color: int = int(edge_color)
        r"""A color on the edges of the backdrop in the RGB format"""
        self.symbol_color: int = int(symbol_color)
        r"""A color to be applied for the symbol in the RGB format"""
        self.text_color: int = int(text_color)
        r"""A color for the text on the backdrop in the RGB format"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftBackdropColors"]:
        return "upgradedGiftBackdropColors"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftBackdropColors"]:
        return "UpgradedGiftBackdropColors"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "center_color": self.center_color,
            "edge_color": self.edge_color,
            "symbol_color": self.symbol_color,
            "text_color": self.text_color,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftBackdropColors", None]:
        if data:
            data_class = cls()
            data_class.center_color = int(data.get("center_color", 0))
            data_class.edge_color = int(data.get("edge_color", 0))
            data_class.symbol_color = int(data.get("symbol_color", 0))
            data_class.text_color = int(data.get("text_color", 0))

        return data_class


class UpgradedGiftBackdrop(TlObject):
    r"""Describes a backdrop of an upgraded gift

    Parameters:
        id (:class:`int`):
            Unique identifier of the backdrop

        name (:class:`str`):
            Name of the backdrop

        colors (:class:`~pytdbot.types.UpgradedGiftBackdropColors`):
            Colors of the backdrop

        rarity_per_mille (:class:`int`):
            The number of upgraded gifts that receive this backdrop for each 1000 gifts upgraded

    """

    def __init__(
        self,
        id: int = 0,
        name: str = "",
        colors: UpgradedGiftBackdropColors = None,
        rarity_per_mille: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the backdrop"""
        self.name: Union[str, None] = name
        r"""Name of the backdrop"""
        self.colors: Union[UpgradedGiftBackdropColors, None] = colors
        r"""Colors of the backdrop"""
        self.rarity_per_mille: int = int(rarity_per_mille)
        r"""The number of upgraded gifts that receive this backdrop for each 1000 gifts upgraded"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftBackdrop"]:
        return "upgradedGiftBackdrop"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftBackdrop"]:
        return "UpgradedGiftBackdrop"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "name": self.name,
            "colors": self.colors,
            "rarity_per_mille": self.rarity_per_mille,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftBackdrop", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.name = data.get("name", "")
            data_class.colors = data.get("colors", None)
            data_class.rarity_per_mille = int(data.get("rarity_per_mille", 0))

        return data_class


class UpgradedGiftOriginalDetails(TlObject):
    r"""Describes the original details about the gift

    Parameters:
        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the chat that sent the gift; may be null if the gift was private

        receiver_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the chat that received the gift

        text (:class:`~pytdbot.types.FormattedText`):
            Message added to the gift

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift was sent

    """

    def __init__(
        self,
        sender_id: MessageSender = None,
        receiver_id: MessageSender = None,
        text: FormattedText = None,
        date: int = 0,
    ) -> None:
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of the user or the chat that sent the gift; may be null if the gift was private"""
        self.receiver_id: Union[MessageSenderUser, MessageSenderChat, None] = (
            receiver_id
        )
        r"""Identifier of the user or the chat that received the gift"""
        self.text: Union[FormattedText, None] = text
        r"""Message added to the gift"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the gift was sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftOriginalDetails"]:
        return "upgradedGiftOriginalDetails"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftOriginalDetails"]:
        return "UpgradedGiftOriginalDetails"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sender_id": self.sender_id,
            "receiver_id": self.receiver_id,
            "text": self.text,
            "date": self.date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftOriginalDetails", None]:
        if data:
            data_class = cls()
            data_class.sender_id = data.get("sender_id", None)
            data_class.receiver_id = data.get("receiver_id", None)
            data_class.text = data.get("text", None)
            data_class.date = int(data.get("date", 0))

        return data_class


class UpgradedGiftColors(TlObject):
    r"""Contains information about color scheme for user's name, background of empty chat photo, replies to messages and link previews

    Parameters:
        id (:class:`int`):
            Unique identifier of the upgraded gift colors

        model_custom_emoji_id (:class:`int`):
            Custom emoji identifier of the model of the upgraded gift

        symbol_custom_emoji_id (:class:`int`):
            Custom emoji identifier of the symbol of the upgraded gift

        light_theme_accent_color (:class:`int`):
            Accent color to use in light themes in RGB format

        light_theme_colors (List[:class:`int`]):
            The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in light themes

        dark_theme_accent_color (:class:`int`):
            Accent color to use in dark themes in RGB format

        dark_theme_colors (List[:class:`int`]):
            The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes

    """

    def __init__(
        self,
        id: int = 0,
        model_custom_emoji_id: int = 0,
        symbol_custom_emoji_id: int = 0,
        light_theme_accent_color: int = 0,
        light_theme_colors: List[int] = None,
        dark_theme_accent_color: int = 0,
        dark_theme_colors: List[int] = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the upgraded gift colors"""
        self.model_custom_emoji_id: int = int(model_custom_emoji_id)
        r"""Custom emoji identifier of the model of the upgraded gift"""
        self.symbol_custom_emoji_id: int = int(symbol_custom_emoji_id)
        r"""Custom emoji identifier of the symbol of the upgraded gift"""
        self.light_theme_accent_color: int = int(light_theme_accent_color)
        r"""Accent color to use in light themes in RGB format"""
        self.light_theme_colors: List[int] = light_theme_colors or []
        r"""The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in light themes"""
        self.dark_theme_accent_color: int = int(dark_theme_accent_color)
        r"""Accent color to use in dark themes in RGB format"""
        self.dark_theme_colors: List[int] = dark_theme_colors or []
        r"""The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftColors"]:
        return "upgradedGiftColors"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftColors"]:
        return "UpgradedGiftColors"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "model_custom_emoji_id": self.model_custom_emoji_id,
            "symbol_custom_emoji_id": self.symbol_custom_emoji_id,
            "light_theme_accent_color": self.light_theme_accent_color,
            "light_theme_colors": self.light_theme_colors,
            "dark_theme_accent_color": self.dark_theme_accent_color,
            "dark_theme_colors": self.dark_theme_colors,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftColors", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.model_custom_emoji_id = int(data.get("model_custom_emoji_id", 0))
            data_class.symbol_custom_emoji_id = int(
                data.get("symbol_custom_emoji_id", 0)
            )
            data_class.light_theme_accent_color = int(
                data.get("light_theme_accent_color", 0)
            )
            data_class.light_theme_colors = data.get("light_theme_colors", None)
            data_class.dark_theme_accent_color = int(
                data.get("dark_theme_accent_color", 0)
            )
            data_class.dark_theme_colors = data.get("dark_theme_colors", None)

        return data_class


class Gift(TlObject):
    r"""Describes a gift that can be sent to another user or channel chat

    Parameters:
        id (:class:`int`):
            Unique identifier of the gift

        publisher_chat_id (:class:`int`):
            Identifier of the chat that published the gift; 0 if none

        sticker (:class:`~pytdbot.types.Sticker`):
            The sticker representing the gift

        star_count (:class:`int`):
            Number of Telegram Stars that must be paid for the gift

        default_sell_star_count (:class:`int`):
            Number of Telegram Stars that can be claimed by the receiver instead of the regular gift by default\. If the gift was paid with just bought Telegram Stars, then full value can be claimed

        upgrade_star_count (:class:`int`):
            Number of Telegram Stars that must be paid to upgrade the gift; 0 if upgrade isn't possible

        has_colors (:class:`bool`):
            True, if the gift can be used to customize the user's name, and backgrounds of profile photo, reply header, and link preview

        is_for_birthday (:class:`bool`):
            True, if the gift is a birthday gift

        is_premium (:class:`bool`):
            True, if the gift can be bought only by Telegram Premium subscribers

        next_send_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift can be sent next time by the current user; can be 0 or a date in the past\. If the date is in the future, then call canSendGift to get the reason, why the gift can't be sent now

        user_limits (:class:`~pytdbot.types.GiftPurchaseLimits`):
            Number of times the gift can be purchased by the current user; may be null if not limited

        overall_limits (:class:`~pytdbot.types.GiftPurchaseLimits`):
            Number of times the gift can be purchased all users; may be null if not limited

        first_send_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift was send for the first time; for sold out gifts only

        last_send_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift was send for the last time; for sold out gifts only

    """

    def __init__(
        self,
        id: int = 0,
        publisher_chat_id: int = 0,
        sticker: Sticker = None,
        star_count: int = 0,
        default_sell_star_count: int = 0,
        upgrade_star_count: int = 0,
        has_colors: bool = False,
        is_for_birthday: bool = False,
        is_premium: bool = False,
        next_send_date: int = 0,
        user_limits: GiftPurchaseLimits = None,
        overall_limits: GiftPurchaseLimits = None,
        first_send_date: int = 0,
        last_send_date: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the gift"""
        self.publisher_chat_id: int = int(publisher_chat_id)
        r"""Identifier of the chat that published the gift; 0 if none"""
        self.sticker: Union[Sticker, None] = sticker
        r"""The sticker representing the gift"""
        self.star_count: int = int(star_count)
        r"""Number of Telegram Stars that must be paid for the gift"""
        self.default_sell_star_count: int = int(default_sell_star_count)
        r"""Number of Telegram Stars that can be claimed by the receiver instead of the regular gift by default\. If the gift was paid with just bought Telegram Stars, then full value can be claimed"""
        self.upgrade_star_count: int = int(upgrade_star_count)
        r"""Number of Telegram Stars that must be paid to upgrade the gift; 0 if upgrade isn't possible"""
        self.has_colors: bool = bool(has_colors)
        r"""True, if the gift can be used to customize the user's name, and backgrounds of profile photo, reply header, and link preview"""
        self.is_for_birthday: bool = bool(is_for_birthday)
        r"""True, if the gift is a birthday gift"""
        self.is_premium: bool = bool(is_premium)
        r"""True, if the gift can be bought only by Telegram Premium subscribers"""
        self.next_send_date: int = int(next_send_date)
        r"""Point in time \(Unix timestamp\) when the gift can be sent next time by the current user; can be 0 or a date in the past\. If the date is in the future, then call canSendGift to get the reason, why the gift can't be sent now"""
        self.user_limits: Union[GiftPurchaseLimits, None] = user_limits
        r"""Number of times the gift can be purchased by the current user; may be null if not limited"""
        self.overall_limits: Union[GiftPurchaseLimits, None] = overall_limits
        r"""Number of times the gift can be purchased all users; may be null if not limited"""
        self.first_send_date: int = int(first_send_date)
        r"""Point in time \(Unix timestamp\) when the gift was send for the first time; for sold out gifts only"""
        self.last_send_date: int = int(last_send_date)
        r"""Point in time \(Unix timestamp\) when the gift was send for the last time; for sold out gifts only"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["gift"]:
        return "gift"

    @classmethod
    def getClass(self) -> Literal["Gift"]:
        return "Gift"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "publisher_chat_id": self.publisher_chat_id,
            "sticker": self.sticker,
            "star_count": self.star_count,
            "default_sell_star_count": self.default_sell_star_count,
            "upgrade_star_count": self.upgrade_star_count,
            "has_colors": self.has_colors,
            "is_for_birthday": self.is_for_birthday,
            "is_premium": self.is_premium,
            "next_send_date": self.next_send_date,
            "user_limits": self.user_limits,
            "overall_limits": self.overall_limits,
            "first_send_date": self.first_send_date,
            "last_send_date": self.last_send_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Gift", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.publisher_chat_id = int(data.get("publisher_chat_id", 0))
            data_class.sticker = data.get("sticker", None)
            data_class.star_count = int(data.get("star_count", 0))
            data_class.default_sell_star_count = int(
                data.get("default_sell_star_count", 0)
            )
            data_class.upgrade_star_count = int(data.get("upgrade_star_count", 0))
            data_class.has_colors = data.get("has_colors", False)
            data_class.is_for_birthday = data.get("is_for_birthday", False)
            data_class.is_premium = data.get("is_premium", False)
            data_class.next_send_date = int(data.get("next_send_date", 0))
            data_class.user_limits = data.get("user_limits", None)
            data_class.overall_limits = data.get("overall_limits", None)
            data_class.first_send_date = int(data.get("first_send_date", 0))
            data_class.last_send_date = int(data.get("last_send_date", 0))

        return data_class


class UpgradedGift(TlObject):
    r"""Describes an upgraded gift that can be transferred to another owner or transferred to the TON blockchain as an NFT

    Parameters:
        id (:class:`int`):
            Unique identifier of the gift

        regular_gift_id (:class:`int`):
            Unique identifier of the regular gift from which the gift was upgraded; may be 0 for short period of time for old gifts from database

        publisher_chat_id (:class:`int`):
            Identifier of the chat that published the gift; 0 if none

        title (:class:`str`):
            The title of the upgraded gift

        name (:class:`str`):
            Unique name of the upgraded gift that can be used with internalLinkTypeUpgradedGift or sendResoldGift

        number (:class:`int`):
            Unique number of the upgraded gift among gifts upgraded from the same gift

        total_upgraded_count (:class:`int`):
            Total number of gifts that were upgraded from the same gift

        max_upgraded_count (:class:`int`):
            The maximum number of gifts that can be upgraded from the same gift

        is_premium (:class:`bool`):
            True, if the original gift could have been bought only by Telegram Premium subscribers

        is_theme_available (:class:`bool`):
            True, if the gift can be used to set a theme in a chat

        used_theme_chat_id (:class:`int`):
            Identifier of the chat for which the gift is used to set a theme; 0 if none or the gift isn't owned by the current user

        host_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the chat to which the upgraded gift was assigned from blockchain; may be null if none or unknown

        owner_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the chat that owns the upgraded gift; may be null if none or unknown

        owner_address (:class:`str`):
            Address of the gift NFT owner in TON blockchain; may be empty if none\. Append the address to getOption\(\"ton\_blockchain\_explorer\_url\"\) to get a link with information about the address

        owner_name (:class:`str`):
            Name of the owner for the case when owner identifier and address aren't known

        gift_address (:class:`str`):
            Address of the gift NFT in TON blockchain; may be empty if none\. Append the address to getOption\(\"ton\_blockchain\_explorer\_url\"\) to get a link with information about the address

        model (:class:`~pytdbot.types.UpgradedGiftModel`):
            Model of the upgraded gift

        symbol (:class:`~pytdbot.types.UpgradedGiftSymbol`):
            Symbol of the upgraded gift

        backdrop (:class:`~pytdbot.types.UpgradedGiftBackdrop`):
            Backdrop of the upgraded gift

        original_details (:class:`~pytdbot.types.UpgradedGiftOriginalDetails`):
            Information about the originally sent gift; may be null if unknown

        colors (:class:`~pytdbot.types.UpgradedGiftColors`):
            Colors that can be set for user's name, background of empty chat photo, replies to messages and link previews; may be null if none

        resale_parameters (:class:`~pytdbot.types.GiftResaleParameters`):
            Resale parameters of the gift; may be null if resale isn't possible

        value_currency (:class:`str`):
            ISO 4217 currency code of the currency in which value of the gift is represented; may be empty if unavailable

        value_amount (:class:`int`):
            Estimated value of the gift; in the smallest units of the currency; 0 if unavailable

    """

    def __init__(
        self,
        id: int = 0,
        regular_gift_id: int = 0,
        publisher_chat_id: int = 0,
        title: str = "",
        name: str = "",
        number: int = 0,
        total_upgraded_count: int = 0,
        max_upgraded_count: int = 0,
        is_premium: bool = False,
        is_theme_available: bool = False,
        used_theme_chat_id: int = 0,
        host_id: MessageSender = None,
        owner_id: MessageSender = None,
        owner_address: str = "",
        owner_name: str = "",
        gift_address: str = "",
        model: UpgradedGiftModel = None,
        symbol: UpgradedGiftSymbol = None,
        backdrop: UpgradedGiftBackdrop = None,
        original_details: UpgradedGiftOriginalDetails = None,
        colors: UpgradedGiftColors = None,
        resale_parameters: GiftResaleParameters = None,
        value_currency: str = "",
        value_amount: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the gift"""
        self.regular_gift_id: int = int(regular_gift_id)
        r"""Unique identifier of the regular gift from which the gift was upgraded; may be 0 for short period of time for old gifts from database"""
        self.publisher_chat_id: int = int(publisher_chat_id)
        r"""Identifier of the chat that published the gift; 0 if none"""
        self.title: Union[str, None] = title
        r"""The title of the upgraded gift"""
        self.name: Union[str, None] = name
        r"""Unique name of the upgraded gift that can be used with internalLinkTypeUpgradedGift or sendResoldGift"""
        self.number: int = int(number)
        r"""Unique number of the upgraded gift among gifts upgraded from the same gift"""
        self.total_upgraded_count: int = int(total_upgraded_count)
        r"""Total number of gifts that were upgraded from the same gift"""
        self.max_upgraded_count: int = int(max_upgraded_count)
        r"""The maximum number of gifts that can be upgraded from the same gift"""
        self.is_premium: bool = bool(is_premium)
        r"""True, if the original gift could have been bought only by Telegram Premium subscribers"""
        self.is_theme_available: bool = bool(is_theme_available)
        r"""True, if the gift can be used to set a theme in a chat"""
        self.used_theme_chat_id: int = int(used_theme_chat_id)
        r"""Identifier of the chat for which the gift is used to set a theme; 0 if none or the gift isn't owned by the current user"""
        self.host_id: Union[MessageSenderUser, MessageSenderChat, None] = host_id
        r"""Identifier of the user or the chat to which the upgraded gift was assigned from blockchain; may be null if none or unknown"""
        self.owner_id: Union[MessageSenderUser, MessageSenderChat, None] = owner_id
        r"""Identifier of the user or the chat that owns the upgraded gift; may be null if none or unknown"""
        self.owner_address: Union[str, None] = owner_address
        r"""Address of the gift NFT owner in TON blockchain; may be empty if none\. Append the address to getOption\(\"ton\_blockchain\_explorer\_url\"\) to get a link with information about the address"""
        self.owner_name: Union[str, None] = owner_name
        r"""Name of the owner for the case when owner identifier and address aren't known"""
        self.gift_address: Union[str, None] = gift_address
        r"""Address of the gift NFT in TON blockchain; may be empty if none\. Append the address to getOption\(\"ton\_blockchain\_explorer\_url\"\) to get a link with information about the address"""
        self.model: Union[UpgradedGiftModel, None] = model
        r"""Model of the upgraded gift"""
        self.symbol: Union[UpgradedGiftSymbol, None] = symbol
        r"""Symbol of the upgraded gift"""
        self.backdrop: Union[UpgradedGiftBackdrop, None] = backdrop
        r"""Backdrop of the upgraded gift"""
        self.original_details: Union[UpgradedGiftOriginalDetails, None] = (
            original_details
        )
        r"""Information about the originally sent gift; may be null if unknown"""
        self.colors: Union[UpgradedGiftColors, None] = colors
        r"""Colors that can be set for user's name, background of empty chat photo, replies to messages and link previews; may be null if none"""
        self.resale_parameters: Union[GiftResaleParameters, None] = resale_parameters
        r"""Resale parameters of the gift; may be null if resale isn't possible"""
        self.value_currency: Union[str, None] = value_currency
        r"""ISO 4217 currency code of the currency in which value of the gift is represented; may be empty if unavailable"""
        self.value_amount: int = int(value_amount)
        r"""Estimated value of the gift; in the smallest units of the currency; 0 if unavailable"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGift"]:
        return "upgradedGift"

    @classmethod
    def getClass(self) -> Literal["UpgradedGift"]:
        return "UpgradedGift"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "regular_gift_id": self.regular_gift_id,
            "publisher_chat_id": self.publisher_chat_id,
            "title": self.title,
            "name": self.name,
            "number": self.number,
            "total_upgraded_count": self.total_upgraded_count,
            "max_upgraded_count": self.max_upgraded_count,
            "is_premium": self.is_premium,
            "is_theme_available": self.is_theme_available,
            "used_theme_chat_id": self.used_theme_chat_id,
            "host_id": self.host_id,
            "owner_id": self.owner_id,
            "owner_address": self.owner_address,
            "owner_name": self.owner_name,
            "gift_address": self.gift_address,
            "model": self.model,
            "symbol": self.symbol,
            "backdrop": self.backdrop,
            "original_details": self.original_details,
            "colors": self.colors,
            "resale_parameters": self.resale_parameters,
            "value_currency": self.value_currency,
            "value_amount": self.value_amount,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGift", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.regular_gift_id = int(data.get("regular_gift_id", 0))
            data_class.publisher_chat_id = int(data.get("publisher_chat_id", 0))
            data_class.title = data.get("title", "")
            data_class.name = data.get("name", "")
            data_class.number = int(data.get("number", 0))
            data_class.total_upgraded_count = int(data.get("total_upgraded_count", 0))
            data_class.max_upgraded_count = int(data.get("max_upgraded_count", 0))
            data_class.is_premium = data.get("is_premium", False)
            data_class.is_theme_available = data.get("is_theme_available", False)
            data_class.used_theme_chat_id = int(data.get("used_theme_chat_id", 0))
            data_class.host_id = data.get("host_id", None)
            data_class.owner_id = data.get("owner_id", None)
            data_class.owner_address = data.get("owner_address", "")
            data_class.owner_name = data.get("owner_name", "")
            data_class.gift_address = data.get("gift_address", "")
            data_class.model = data.get("model", None)
            data_class.symbol = data.get("symbol", None)
            data_class.backdrop = data.get("backdrop", None)
            data_class.original_details = data.get("original_details", None)
            data_class.colors = data.get("colors", None)
            data_class.resale_parameters = data.get("resale_parameters", None)
            data_class.value_currency = data.get("value_currency", "")
            data_class.value_amount = int(data.get("value_amount", 0))

        return data_class


class UpgradedGiftValueInfo(TlObject):
    r"""Contains information about value of an upgraded gift

    Parameters:
        currency (:class:`str`):
            ISO 4217 currency code of the currency in which the prices are represented

        value (:class:`int`):
            Estimated value of the gift; in the smallest units of the currency

        is_value_average (:class:`bool`):
            True, if the value is calculated as average value of similar sold gifts\. Otherwise, it is based on the sale price of the gift

        initial_sale_date (:class:`int`):
            Point in time \(Unix timestamp\) when the corresponding regular gift was originally purchased

        initial_sale_star_count (:class:`int`):
            Amount of Telegram Stars that were paid for the gift

        initial_sale_price (:class:`int`):
            Initial price of the gift; in the smallest units of the currency

        last_sale_date (:class:`int`):
            Point in time \(Unix timestamp\) when the upgraded gift was purchased last time; 0 if never

        last_sale_price (:class:`int`):
            Last purchase price of the gift; in the smallest units of the currency; 0 if the gift has never been resold

        is_last_sale_on_fragment (:class:`bool`):
            True, if the last sale was completed on Fragment

        minimum_price (:class:`int`):
            The current minimum price of gifts upgraded from the same gift; in the smallest units of the currency; 0 if there are no such gifts

        average_sale_price (:class:`int`):
            The average sale price in the last month of gifts upgraded from the same gift; in the smallest units of the currency; 0 if there were no such sales

        telegram_listed_gift_count (:class:`int`):
            Number of gifts upgraded from the same gift being resold on Telegram

        fragment_listed_gift_count (:class:`int`):
            Number of gifts upgraded from the same gift being resold on Fragment

        fragment_url (:class:`str`):
            The HTTPS link to the Fragment for the gift; may be empty if there are no such gifts being sold on Fragment

    """

    def __init__(
        self,
        currency: str = "",
        value: int = 0,
        is_value_average: bool = False,
        initial_sale_date: int = 0,
        initial_sale_star_count: int = 0,
        initial_sale_price: int = 0,
        last_sale_date: int = 0,
        last_sale_price: int = 0,
        is_last_sale_on_fragment: bool = False,
        minimum_price: int = 0,
        average_sale_price: int = 0,
        telegram_listed_gift_count: int = 0,
        fragment_listed_gift_count: int = 0,
        fragment_url: str = "",
    ) -> None:
        self.currency: Union[str, None] = currency
        r"""ISO 4217 currency code of the currency in which the prices are represented"""
        self.value: int = int(value)
        r"""Estimated value of the gift; in the smallest units of the currency"""
        self.is_value_average: bool = bool(is_value_average)
        r"""True, if the value is calculated as average value of similar sold gifts\. Otherwise, it is based on the sale price of the gift"""
        self.initial_sale_date: int = int(initial_sale_date)
        r"""Point in time \(Unix timestamp\) when the corresponding regular gift was originally purchased"""
        self.initial_sale_star_count: int = int(initial_sale_star_count)
        r"""Amount of Telegram Stars that were paid for the gift"""
        self.initial_sale_price: int = int(initial_sale_price)
        r"""Initial price of the gift; in the smallest units of the currency"""
        self.last_sale_date: int = int(last_sale_date)
        r"""Point in time \(Unix timestamp\) when the upgraded gift was purchased last time; 0 if never"""
        self.last_sale_price: int = int(last_sale_price)
        r"""Last purchase price of the gift; in the smallest units of the currency; 0 if the gift has never been resold"""
        self.is_last_sale_on_fragment: bool = bool(is_last_sale_on_fragment)
        r"""True, if the last sale was completed on Fragment"""
        self.minimum_price: int = int(minimum_price)
        r"""The current minimum price of gifts upgraded from the same gift; in the smallest units of the currency; 0 if there are no such gifts"""
        self.average_sale_price: int = int(average_sale_price)
        r"""The average sale price in the last month of gifts upgraded from the same gift; in the smallest units of the currency; 0 if there were no such sales"""
        self.telegram_listed_gift_count: int = int(telegram_listed_gift_count)
        r"""Number of gifts upgraded from the same gift being resold on Telegram"""
        self.fragment_listed_gift_count: int = int(fragment_listed_gift_count)
        r"""Number of gifts upgraded from the same gift being resold on Fragment"""
        self.fragment_url: Union[str, None] = fragment_url
        r"""The HTTPS link to the Fragment for the gift; may be empty if there are no such gifts being sold on Fragment"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftValueInfo"]:
        return "upgradedGiftValueInfo"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftValueInfo"]:
        return "UpgradedGiftValueInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "currency": self.currency,
            "value": self.value,
            "is_value_average": self.is_value_average,
            "initial_sale_date": self.initial_sale_date,
            "initial_sale_star_count": self.initial_sale_star_count,
            "initial_sale_price": self.initial_sale_price,
            "last_sale_date": self.last_sale_date,
            "last_sale_price": self.last_sale_price,
            "is_last_sale_on_fragment": self.is_last_sale_on_fragment,
            "minimum_price": self.minimum_price,
            "average_sale_price": self.average_sale_price,
            "telegram_listed_gift_count": self.telegram_listed_gift_count,
            "fragment_listed_gift_count": self.fragment_listed_gift_count,
            "fragment_url": self.fragment_url,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftValueInfo", None]:
        if data:
            data_class = cls()
            data_class.currency = data.get("currency", "")
            data_class.value = int(data.get("value", 0))
            data_class.is_value_average = data.get("is_value_average", False)
            data_class.initial_sale_date = int(data.get("initial_sale_date", 0))
            data_class.initial_sale_star_count = int(
                data.get("initial_sale_star_count", 0)
            )
            data_class.initial_sale_price = int(data.get("initial_sale_price", 0))
            data_class.last_sale_date = int(data.get("last_sale_date", 0))
            data_class.last_sale_price = int(data.get("last_sale_price", 0))
            data_class.is_last_sale_on_fragment = data.get(
                "is_last_sale_on_fragment", False
            )
            data_class.minimum_price = int(data.get("minimum_price", 0))
            data_class.average_sale_price = int(data.get("average_sale_price", 0))
            data_class.telegram_listed_gift_count = int(
                data.get("telegram_listed_gift_count", 0)
            )
            data_class.fragment_listed_gift_count = int(
                data.get("fragment_listed_gift_count", 0)
            )
            data_class.fragment_url = data.get("fragment_url", "")

        return data_class


class UpgradeGiftResult(TlObject):
    r"""Contains result of gift upgrading

    Parameters:
        gift (:class:`~pytdbot.types.UpgradedGift`):
            The upgraded gift

        received_gift_id (:class:`str`):
            Unique identifier of the received gift for the current user

        is_saved (:class:`bool`):
            True, if the gift is displayed on the user's or the channel's profile page

        can_be_transferred (:class:`bool`):
            True, if the gift can be transferred to another owner

        transfer_star_count (:class:`int`):
            Number of Telegram Stars that must be paid to transfer the upgraded gift

        drop_original_details_star_count (:class:`int`):
            Number of Telegram Stars that must be paid to drop original details of the upgraded gift; 0 if not available

        next_transfer_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift can be transferred to another owner; can be in the past; 0 if the gift can be transferred immediately or transfer isn't possible

        next_resale_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift can be resold to another user; can be in the past; 0 if the gift can't be resold; only for the receiver of the gift

        export_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift can be transferred to the TON blockchain as an NFT; can be in the past

    """

    def __init__(
        self,
        gift: UpgradedGift = None,
        received_gift_id: str = "",
        is_saved: bool = False,
        can_be_transferred: bool = False,
        transfer_star_count: int = 0,
        drop_original_details_star_count: int = 0,
        next_transfer_date: int = 0,
        next_resale_date: int = 0,
        export_date: int = 0,
    ) -> None:
        self.gift: Union[UpgradedGift, None] = gift
        r"""The upgraded gift"""
        self.received_gift_id: Union[str, None] = received_gift_id
        r"""Unique identifier of the received gift for the current user"""
        self.is_saved: bool = bool(is_saved)
        r"""True, if the gift is displayed on the user's or the channel's profile page"""
        self.can_be_transferred: bool = bool(can_be_transferred)
        r"""True, if the gift can be transferred to another owner"""
        self.transfer_star_count: int = int(transfer_star_count)
        r"""Number of Telegram Stars that must be paid to transfer the upgraded gift"""
        self.drop_original_details_star_count: int = int(
            drop_original_details_star_count
        )
        r"""Number of Telegram Stars that must be paid to drop original details of the upgraded gift; 0 if not available"""
        self.next_transfer_date: int = int(next_transfer_date)
        r"""Point in time \(Unix timestamp\) when the gift can be transferred to another owner; can be in the past; 0 if the gift can be transferred immediately or transfer isn't possible"""
        self.next_resale_date: int = int(next_resale_date)
        r"""Point in time \(Unix timestamp\) when the gift can be resold to another user; can be in the past; 0 if the gift can't be resold; only for the receiver of the gift"""
        self.export_date: int = int(export_date)
        r"""Point in time \(Unix timestamp\) when the gift can be transferred to the TON blockchain as an NFT; can be in the past"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradeGiftResult"]:
        return "upgradeGiftResult"

    @classmethod
    def getClass(self) -> Literal["UpgradeGiftResult"]:
        return "UpgradeGiftResult"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "gift": self.gift,
            "received_gift_id": self.received_gift_id,
            "is_saved": self.is_saved,
            "can_be_transferred": self.can_be_transferred,
            "transfer_star_count": self.transfer_star_count,
            "drop_original_details_star_count": self.drop_original_details_star_count,
            "next_transfer_date": self.next_transfer_date,
            "next_resale_date": self.next_resale_date,
            "export_date": self.export_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradeGiftResult", None]:
        if data:
            data_class = cls()
            data_class.gift = data.get("gift", None)
            data_class.received_gift_id = data.get("received_gift_id", "")
            data_class.is_saved = data.get("is_saved", False)
            data_class.can_be_transferred = data.get("can_be_transferred", False)
            data_class.transfer_star_count = int(data.get("transfer_star_count", 0))
            data_class.drop_original_details_star_count = int(
                data.get("drop_original_details_star_count", 0)
            )
            data_class.next_transfer_date = int(data.get("next_transfer_date", 0))
            data_class.next_resale_date = int(data.get("next_resale_date", 0))
            data_class.export_date = int(data.get("export_date", 0))

        return data_class


class AvailableGift(TlObject):
    r"""Describes a gift that is available for purchase

    Parameters:
        gift (:class:`~pytdbot.types.Gift`):
            The gift

        resale_count (:class:`int`):
            Number of gifts that are available for resale

        min_resale_star_count (:class:`int`):
            The minimum price for the gifts available for resale in Telegram Star equivalent; 0 if there are no such gifts

        title (:class:`str`):
            The title of the upgraded gift; empty if the gift isn't available for resale

    """

    def __init__(
        self,
        gift: Gift = None,
        resale_count: int = 0,
        min_resale_star_count: int = 0,
        title: str = "",
    ) -> None:
        self.gift: Union[Gift, None] = gift
        r"""The gift"""
        self.resale_count: int = int(resale_count)
        r"""Number of gifts that are available for resale"""
        self.min_resale_star_count: int = int(min_resale_star_count)
        r"""The minimum price for the gifts available for resale in Telegram Star equivalent; 0 if there are no such gifts"""
        self.title: Union[str, None] = title
        r"""The title of the upgraded gift; empty if the gift isn't available for resale"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["availableGift"]:
        return "availableGift"

    @classmethod
    def getClass(self) -> Literal["AvailableGift"]:
        return "AvailableGift"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "gift": self.gift,
            "resale_count": self.resale_count,
            "min_resale_star_count": self.min_resale_star_count,
            "title": self.title,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AvailableGift", None]:
        if data:
            data_class = cls()
            data_class.gift = data.get("gift", None)
            data_class.resale_count = int(data.get("resale_count", 0))
            data_class.min_resale_star_count = int(data.get("min_resale_star_count", 0))
            data_class.title = data.get("title", "")

        return data_class


class AvailableGifts(TlObject):
    r"""Contains a list of gifts that can be sent to another user or channel chat

    Parameters:
        gifts (List[:class:`~pytdbot.types.AvailableGift`]):
            The list of gifts

    """

    def __init__(self, gifts: List[AvailableGift] = None) -> None:
        self.gifts: List[AvailableGift] = gifts or []
        r"""The list of gifts"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["availableGifts"]:
        return "availableGifts"

    @classmethod
    def getClass(self) -> Literal["AvailableGifts"]:
        return "AvailableGifts"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "gifts": self.gifts}

    @classmethod
    def from_dict(cls, data: dict) -> Union["AvailableGifts", None]:
        if data:
            data_class = cls()
            data_class.gifts = data.get("gifts", None)

        return data_class


class GiftUpgradePrice(TlObject):
    r"""Describes a price required to pay to upgrade a gift

    Parameters:
        date (:class:`int`):
            Point in time \(Unix timestamp\) when the price will be in effect

        star_count (:class:`int`):
            The amount of Telegram Stars required to pay to upgrade the gift

    """

    def __init__(self, date: int = 0, star_count: int = 0) -> None:
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the price will be in effect"""
        self.star_count: int = int(star_count)
        r"""The amount of Telegram Stars required to pay to upgrade the gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftUpgradePrice"]:
        return "giftUpgradePrice"

    @classmethod
    def getClass(self) -> Literal["GiftUpgradePrice"]:
        return "GiftUpgradePrice"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "date": self.date,
            "star_count": self.star_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftUpgradePrice", None]:
        if data:
            data_class = cls()
            data_class.date = int(data.get("date", 0))
            data_class.star_count = int(data.get("star_count", 0))

        return data_class


class UpgradedGiftAttributeIdModel(TlObject, UpgradedGiftAttributeId):
    r"""Identifier of a gift model

    Parameters:
        sticker_id (:class:`int`):
            Identifier of the sticker representing the model

    """

    def __init__(self, sticker_id: int = 0) -> None:
        self.sticker_id: int = int(sticker_id)
        r"""Identifier of the sticker representing the model"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftAttributeIdModel"]:
        return "upgradedGiftAttributeIdModel"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftAttributeId"]:
        return "UpgradedGiftAttributeId"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sticker_id": self.sticker_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftAttributeIdModel", None]:
        if data:
            data_class = cls()
            data_class.sticker_id = int(data.get("sticker_id", 0))

        return data_class


class UpgradedGiftAttributeIdSymbol(TlObject, UpgradedGiftAttributeId):
    r"""Identifier of a gift symbol

    Parameters:
        sticker_id (:class:`int`):
            Identifier of the sticker representing the symbol

    """

    def __init__(self, sticker_id: int = 0) -> None:
        self.sticker_id: int = int(sticker_id)
        r"""Identifier of the sticker representing the symbol"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftAttributeIdSymbol"]:
        return "upgradedGiftAttributeIdSymbol"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftAttributeId"]:
        return "UpgradedGiftAttributeId"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sticker_id": self.sticker_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftAttributeIdSymbol", None]:
        if data:
            data_class = cls()
            data_class.sticker_id = int(data.get("sticker_id", 0))

        return data_class


class UpgradedGiftAttributeIdBackdrop(TlObject, UpgradedGiftAttributeId):
    r"""Identifier of a gift backdrop

    Parameters:
        backdrop_id (:class:`int`):
            Identifier of the backdrop

    """

    def __init__(self, backdrop_id: int = 0) -> None:
        self.backdrop_id: int = int(backdrop_id)
        r"""Identifier of the backdrop"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftAttributeIdBackdrop"]:
        return "upgradedGiftAttributeIdBackdrop"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftAttributeId"]:
        return "UpgradedGiftAttributeId"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "backdrop_id": self.backdrop_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftAttributeIdBackdrop", None]:
        if data:
            data_class = cls()
            data_class.backdrop_id = int(data.get("backdrop_id", 0))

        return data_class


class UpgradedGiftModelCount(TlObject):
    r"""Describes a model of an upgraded gift with the number of gifts found

    Parameters:
        model (:class:`~pytdbot.types.UpgradedGiftModel`):
            The model

        total_count (:class:`int`):
            Total number of gifts with the model

    """

    def __init__(self, model: UpgradedGiftModel = None, total_count: int = 0) -> None:
        self.model: Union[UpgradedGiftModel, None] = model
        r"""The model"""
        self.total_count: int = int(total_count)
        r"""Total number of gifts with the model"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftModelCount"]:
        return "upgradedGiftModelCount"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftModelCount"]:
        return "UpgradedGiftModelCount"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "model": self.model,
            "total_count": self.total_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftModelCount", None]:
        if data:
            data_class = cls()
            data_class.model = data.get("model", None)
            data_class.total_count = int(data.get("total_count", 0))

        return data_class


class UpgradedGiftSymbolCount(TlObject):
    r"""Describes a symbol shown on the pattern of an upgraded gift

    Parameters:
        symbol (:class:`~pytdbot.types.UpgradedGiftSymbol`):
            The symbol

        total_count (:class:`int`):
            Total number of gifts with the symbol

    """

    def __init__(self, symbol: UpgradedGiftSymbol = None, total_count: int = 0) -> None:
        self.symbol: Union[UpgradedGiftSymbol, None] = symbol
        r"""The symbol"""
        self.total_count: int = int(total_count)
        r"""Total number of gifts with the symbol"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftSymbolCount"]:
        return "upgradedGiftSymbolCount"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftSymbolCount"]:
        return "UpgradedGiftSymbolCount"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "symbol": self.symbol,
            "total_count": self.total_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftSymbolCount", None]:
        if data:
            data_class = cls()
            data_class.symbol = data.get("symbol", None)
            data_class.total_count = int(data.get("total_count", 0))

        return data_class


class UpgradedGiftBackdropCount(TlObject):
    r"""Describes a backdrop of an upgraded gift

    Parameters:
        backdrop (:class:`~pytdbot.types.UpgradedGiftBackdrop`):
            The backdrop

        total_count (:class:`int`):
            Total number of gifts with the symbol

    """

    def __init__(
        self, backdrop: UpgradedGiftBackdrop = None, total_count: int = 0
    ) -> None:
        self.backdrop: Union[UpgradedGiftBackdrop, None] = backdrop
        r"""The backdrop"""
        self.total_count: int = int(total_count)
        r"""Total number of gifts with the symbol"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["upgradedGiftBackdropCount"]:
        return "upgradedGiftBackdropCount"

    @classmethod
    def getClass(self) -> Literal["UpgradedGiftBackdropCount"]:
        return "UpgradedGiftBackdropCount"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "backdrop": self.backdrop,
            "total_count": self.total_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UpgradedGiftBackdropCount", None]:
        if data:
            data_class = cls()
            data_class.backdrop = data.get("backdrop", None)
            data_class.total_count = int(data.get("total_count", 0))

        return data_class


class GiftForResaleOrderPrice(TlObject, GiftForResaleOrder):
    r"""The gifts will be sorted by their price from the lowest to the highest"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftForResaleOrderPrice"]:
        return "giftForResaleOrderPrice"

    @classmethod
    def getClass(self) -> Literal["GiftForResaleOrder"]:
        return "GiftForResaleOrder"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftForResaleOrderPrice", None]:
        if data:
            data_class = cls()

        return data_class


class GiftForResaleOrderPriceChangeDate(TlObject, GiftForResaleOrder):
    r"""The gifts will be sorted by the last date when their price was changed from the newest to the oldest"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftForResaleOrderPriceChangeDate"]:
        return "giftForResaleOrderPriceChangeDate"

    @classmethod
    def getClass(self) -> Literal["GiftForResaleOrder"]:
        return "GiftForResaleOrder"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftForResaleOrderPriceChangeDate", None]:
        if data:
            data_class = cls()

        return data_class


class GiftForResaleOrderNumber(TlObject, GiftForResaleOrder):
    r"""The gifts will be sorted by their number from the smallest to the largest"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftForResaleOrderNumber"]:
        return "giftForResaleOrderNumber"

    @classmethod
    def getClass(self) -> Literal["GiftForResaleOrder"]:
        return "GiftForResaleOrder"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftForResaleOrderNumber", None]:
        if data:
            data_class = cls()

        return data_class


class GiftForResale(TlObject):
    r"""Describes a gift available for resale

    Parameters:
        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

        received_gift_id (:class:`str`):
            Unique identifier of the received gift for the current user; only for the gifts owned by the current user

    """

    def __init__(self, gift: UpgradedGift = None, received_gift_id: str = "") -> None:
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""
        self.received_gift_id: Union[str, None] = received_gift_id
        r"""Unique identifier of the received gift for the current user; only for the gifts owned by the current user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftForResale"]:
        return "giftForResale"

    @classmethod
    def getClass(self) -> Literal["GiftForResale"]:
        return "GiftForResale"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "gift": self.gift,
            "received_gift_id": self.received_gift_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftForResale", None]:
        if data:
            data_class = cls()
            data_class.gift = data.get("gift", None)
            data_class.received_gift_id = data.get("received_gift_id", "")

        return data_class


class GiftsForResale(TlObject):
    r"""Describes gifts available for resale

    Parameters:
        total_count (:class:`int`):
            Total number of gifts found

        gifts (List[:class:`~pytdbot.types.GiftForResale`]):
            The gifts

        models (List[:class:`~pytdbot.types.UpgradedGiftModelCount`]):
            Available models; for searchGiftsForResale requests without offset and attributes only

        symbols (List[:class:`~pytdbot.types.UpgradedGiftSymbolCount`]):
            Available symbols; for searchGiftsForResale requests without offset and attributes only

        backdrops (List[:class:`~pytdbot.types.UpgradedGiftBackdropCount`]):
            Available backdrops; for searchGiftsForResale requests without offset and attributes only

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        total_count: int = 0,
        gifts: List[GiftForResale] = None,
        models: List[UpgradedGiftModelCount] = None,
        symbols: List[UpgradedGiftSymbolCount] = None,
        backdrops: List[UpgradedGiftBackdropCount] = None,
        next_offset: str = "",
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Total number of gifts found"""
        self.gifts: List[GiftForResale] = gifts or []
        r"""The gifts"""
        self.models: List[UpgradedGiftModelCount] = models or []
        r"""Available models; for searchGiftsForResale requests without offset and attributes only"""
        self.symbols: List[UpgradedGiftSymbolCount] = symbols or []
        r"""Available symbols; for searchGiftsForResale requests without offset and attributes only"""
        self.backdrops: List[UpgradedGiftBackdropCount] = backdrops or []
        r"""Available backdrops; for searchGiftsForResale requests without offset and attributes only"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftsForResale"]:
        return "giftsForResale"

    @classmethod
    def getClass(self) -> Literal["GiftsForResale"]:
        return "GiftsForResale"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "gifts": self.gifts,
            "models": self.models,
            "symbols": self.symbols,
            "backdrops": self.backdrops,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftsForResale", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.gifts = data.get("gifts", None)
            data_class.models = data.get("models", None)
            data_class.symbols = data.get("symbols", None)
            data_class.backdrops = data.get("backdrops", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class GiftResaleResultOk(TlObject, GiftResaleResult):
    r"""Operation was successfully completed"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftResaleResultOk"]:
        return "giftResaleResultOk"

    @classmethod
    def getClass(self) -> Literal["GiftResaleResult"]:
        return "GiftResaleResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftResaleResultOk", None]:
        if data:
            data_class = cls()

        return data_class


class GiftResaleResultPriceIncreased(TlObject, GiftResaleResult):
    r"""Operation has failed, because price has increased\. If the price has decreased, then the buying will succeed anyway

    Parameters:
        price (:class:`~pytdbot.types.GiftResalePrice`):
            New price for the gift

    """

    def __init__(self, price: GiftResalePrice = None) -> None:
        self.price: Union[GiftResalePriceStar, GiftResalePriceTon, None] = price
        r"""New price for the gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftResaleResultPriceIncreased"]:
        return "giftResaleResultPriceIncreased"

    @classmethod
    def getClass(self) -> Literal["GiftResaleResult"]:
        return "GiftResaleResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "price": self.price}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftResaleResultPriceIncreased", None]:
        if data:
            data_class = cls()
            data_class.price = data.get("price", None)

        return data_class


class SentGiftRegular(TlObject, SentGift):
    r"""Regular gift

    Parameters:
        gift (:class:`~pytdbot.types.Gift`):
            The gift

    """

    def __init__(self, gift: Gift = None) -> None:
        self.gift: Union[Gift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sentGiftRegular"]:
        return "sentGiftRegular"

    @classmethod
    def getClass(self) -> Literal["SentGift"]:
        return "SentGift"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "gift": self.gift}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SentGiftRegular", None]:
        if data:
            data_class = cls()
            data_class.gift = data.get("gift", None)

        return data_class


class SentGiftUpgraded(TlObject, SentGift):
    r"""Upgraded gift

    Parameters:
        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

    """

    def __init__(self, gift: UpgradedGift = None) -> None:
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sentGiftUpgraded"]:
        return "sentGiftUpgraded"

    @classmethod
    def getClass(self) -> Literal["SentGift"]:
        return "SentGift"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "gift": self.gift}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SentGiftUpgraded", None]:
        if data:
            data_class = cls()
            data_class.gift = data.get("gift", None)

        return data_class


class ReceivedGift(TlObject):
    r"""Represents a gift received by a user or a chat

    Parameters:
        received_gift_id (:class:`str`):
            Unique identifier of the received gift for the current user; only for the receiver of the gift

        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of a user or a chat that sent the gift; may be null if unknown

        text (:class:`~pytdbot.types.FormattedText`):
            Message added to the gift

        is_private (:class:`bool`):
            True, if the sender and gift text are shown only to the gift receiver; otherwise, everyone are able to see them

        is_saved (:class:`bool`):
            True, if the gift is displayed on the chat's profile page; only for the receiver of the gift

        is_pinned (:class:`bool`):
            True, if the gift is pinned to the top of the chat's profile page

        can_be_upgraded (:class:`bool`):
            True, if the gift is a regular gift that can be upgraded to a unique gift; only for the receiver of the gift

        can_be_transferred (:class:`bool`):
            True, if the gift is an upgraded gift that can be transferred to another owner; only for the receiver of the gift

        was_refunded (:class:`bool`):
            True, if the gift was refunded and isn't available anymore

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift was sent

        gift (:class:`~pytdbot.types.SentGift`):
            The gift

        collection_ids (List[:class:`int`]):
            Identifiers of collections to which the gift is added; only for the receiver of the gift

        sell_star_count (:class:`int`):
            Number of Telegram Stars that can be claimed by the receiver instead of the regular gift; 0 if the gift can't be sold by the current user

        prepaid_upgrade_star_count (:class:`int`):
            Number of Telegram Stars that were paid by the sender for the ability to upgrade the gift

        is_upgrade_separate (:class:`bool`):
            True, if the upgrade was bought after the gift was sent\. In this case, prepaid upgrade cost must not be added to the gift cost

        transfer_star_count (:class:`int`):
            Number of Telegram Stars that must be paid to transfer the upgraded gift; only for the receiver of the gift

        drop_original_details_star_count (:class:`int`):
            Number of Telegram Stars that must be paid to drop original details of the upgraded gift; 0 if not available; only for the receiver of the gift

        next_transfer_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift can be transferred to another owner; can be in the past; 0 if the gift can be transferred immediately or transfer isn't possible; only for the receiver of the gift

        next_resale_date (:class:`int`):
            Point in time \(Unix timestamp\) when the gift can be resold to another user; can be in the past; 0 if the gift can't be resold; only for the receiver of the gift

        export_date (:class:`int`):
            Point in time \(Unix timestamp\) when the upgraded gift can be transferred to the TON blockchain as an NFT; can be in the past; 0 if NFT export isn't possible; only for the receiver of the gift

        prepaid_upgrade_hash (:class:`str`):
            If non\-empty, then the user can pay for an upgrade of the gift using buyGiftUpgrade

    """

    def __init__(
        self,
        received_gift_id: str = "",
        sender_id: MessageSender = None,
        text: FormattedText = None,
        is_private: bool = False,
        is_saved: bool = False,
        is_pinned: bool = False,
        can_be_upgraded: bool = False,
        can_be_transferred: bool = False,
        was_refunded: bool = False,
        date: int = 0,
        gift: SentGift = None,
        collection_ids: List[int] = None,
        sell_star_count: int = 0,
        prepaid_upgrade_star_count: int = 0,
        is_upgrade_separate: bool = False,
        transfer_star_count: int = 0,
        drop_original_details_star_count: int = 0,
        next_transfer_date: int = 0,
        next_resale_date: int = 0,
        export_date: int = 0,
        prepaid_upgrade_hash: str = "",
    ) -> None:
        self.received_gift_id: Union[str, None] = received_gift_id
        r"""Unique identifier of the received gift for the current user; only for the receiver of the gift"""
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of a user or a chat that sent the gift; may be null if unknown"""
        self.text: Union[FormattedText, None] = text
        r"""Message added to the gift"""
        self.is_private: bool = bool(is_private)
        r"""True, if the sender and gift text are shown only to the gift receiver; otherwise, everyone are able to see them"""
        self.is_saved: bool = bool(is_saved)
        r"""True, if the gift is displayed on the chat's profile page; only for the receiver of the gift"""
        self.is_pinned: bool = bool(is_pinned)
        r"""True, if the gift is pinned to the top of the chat's profile page"""
        self.can_be_upgraded: bool = bool(can_be_upgraded)
        r"""True, if the gift is a regular gift that can be upgraded to a unique gift; only for the receiver of the gift"""
        self.can_be_transferred: bool = bool(can_be_transferred)
        r"""True, if the gift is an upgraded gift that can be transferred to another owner; only for the receiver of the gift"""
        self.was_refunded: bool = bool(was_refunded)
        r"""True, if the gift was refunded and isn't available anymore"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the gift was sent"""
        self.gift: Union[SentGiftRegular, SentGiftUpgraded, None] = gift
        r"""The gift"""
        self.collection_ids: List[int] = collection_ids or []
        r"""Identifiers of collections to which the gift is added; only for the receiver of the gift"""
        self.sell_star_count: int = int(sell_star_count)
        r"""Number of Telegram Stars that can be claimed by the receiver instead of the regular gift; 0 if the gift can't be sold by the current user"""
        self.prepaid_upgrade_star_count: int = int(prepaid_upgrade_star_count)
        r"""Number of Telegram Stars that were paid by the sender for the ability to upgrade the gift"""
        self.is_upgrade_separate: bool = bool(is_upgrade_separate)
        r"""True, if the upgrade was bought after the gift was sent\. In this case, prepaid upgrade cost must not be added to the gift cost"""
        self.transfer_star_count: int = int(transfer_star_count)
        r"""Number of Telegram Stars that must be paid to transfer the upgraded gift; only for the receiver of the gift"""
        self.drop_original_details_star_count: int = int(
            drop_original_details_star_count
        )
        r"""Number of Telegram Stars that must be paid to drop original details of the upgraded gift; 0 if not available; only for the receiver of the gift"""
        self.next_transfer_date: int = int(next_transfer_date)
        r"""Point in time \(Unix timestamp\) when the gift can be transferred to another owner; can be in the past; 0 if the gift can be transferred immediately or transfer isn't possible; only for the receiver of the gift"""
        self.next_resale_date: int = int(next_resale_date)
        r"""Point in time \(Unix timestamp\) when the gift can be resold to another user; can be in the past; 0 if the gift can't be resold; only for the receiver of the gift"""
        self.export_date: int = int(export_date)
        r"""Point in time \(Unix timestamp\) when the upgraded gift can be transferred to the TON blockchain as an NFT; can be in the past; 0 if NFT export isn't possible; only for the receiver of the gift"""
        self.prepaid_upgrade_hash: Union[str, None] = prepaid_upgrade_hash
        r"""If non\-empty, then the user can pay for an upgrade of the gift using buyGiftUpgrade"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["receivedGift"]:
        return "receivedGift"

    @classmethod
    def getClass(self) -> Literal["ReceivedGift"]:
        return "ReceivedGift"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "received_gift_id": self.received_gift_id,
            "sender_id": self.sender_id,
            "text": self.text,
            "is_private": self.is_private,
            "is_saved": self.is_saved,
            "is_pinned": self.is_pinned,
            "can_be_upgraded": self.can_be_upgraded,
            "can_be_transferred": self.can_be_transferred,
            "was_refunded": self.was_refunded,
            "date": self.date,
            "gift": self.gift,
            "collection_ids": self.collection_ids,
            "sell_star_count": self.sell_star_count,
            "prepaid_upgrade_star_count": self.prepaid_upgrade_star_count,
            "is_upgrade_separate": self.is_upgrade_separate,
            "transfer_star_count": self.transfer_star_count,
            "drop_original_details_star_count": self.drop_original_details_star_count,
            "next_transfer_date": self.next_transfer_date,
            "next_resale_date": self.next_resale_date,
            "export_date": self.export_date,
            "prepaid_upgrade_hash": self.prepaid_upgrade_hash,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReceivedGift", None]:
        if data:
            data_class = cls()
            data_class.received_gift_id = data.get("received_gift_id", "")
            data_class.sender_id = data.get("sender_id", None)
            data_class.text = data.get("text", None)
            data_class.is_private = data.get("is_private", False)
            data_class.is_saved = data.get("is_saved", False)
            data_class.is_pinned = data.get("is_pinned", False)
            data_class.can_be_upgraded = data.get("can_be_upgraded", False)
            data_class.can_be_transferred = data.get("can_be_transferred", False)
            data_class.was_refunded = data.get("was_refunded", False)
            data_class.date = int(data.get("date", 0))
            data_class.gift = data.get("gift", None)
            data_class.collection_ids = data.get("collection_ids", None)
            data_class.sell_star_count = int(data.get("sell_star_count", 0))
            data_class.prepaid_upgrade_star_count = int(
                data.get("prepaid_upgrade_star_count", 0)
            )
            data_class.is_upgrade_separate = data.get("is_upgrade_separate", False)
            data_class.transfer_star_count = int(data.get("transfer_star_count", 0))
            data_class.drop_original_details_star_count = int(
                data.get("drop_original_details_star_count", 0)
            )
            data_class.next_transfer_date = int(data.get("next_transfer_date", 0))
            data_class.next_resale_date = int(data.get("next_resale_date", 0))
            data_class.export_date = int(data.get("export_date", 0))
            data_class.prepaid_upgrade_hash = data.get("prepaid_upgrade_hash", "")

        return data_class


class ReceivedGifts(TlObject):
    r"""Represents a list of gifts received by a user or a chat

    Parameters:
        total_count (:class:`int`):
            The total number of received gifts

        gifts (List[:class:`~pytdbot.types.ReceivedGift`]):
            The list of gifts

        are_notifications_enabled (:class:`bool`):
            True, if notifications about new gifts of the owner are enabled

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        total_count: int = 0,
        gifts: List[ReceivedGift] = None,
        are_notifications_enabled: bool = False,
        next_offset: str = "",
    ) -> None:
        self.total_count: int = int(total_count)
        r"""The total number of received gifts"""
        self.gifts: List[ReceivedGift] = gifts or []
        r"""The list of gifts"""
        self.are_notifications_enabled: bool = bool(are_notifications_enabled)
        r"""True, if notifications about new gifts of the owner are enabled"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["receivedGifts"]:
        return "receivedGifts"

    @classmethod
    def getClass(self) -> Literal["ReceivedGifts"]:
        return "ReceivedGifts"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "gifts": self.gifts,
            "are_notifications_enabled": self.are_notifications_enabled,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReceivedGifts", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.gifts = data.get("gifts", None)
            data_class.are_notifications_enabled = data.get(
                "are_notifications_enabled", False
            )
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class GiftUpgradePreview(TlObject):
    r"""Contains examples of possible upgraded gifts for the given regular gift

    Parameters:
        models (List[:class:`~pytdbot.types.UpgradedGiftModel`]):
            Examples of possible models that can be chosen for the gift after upgrade

        symbols (List[:class:`~pytdbot.types.UpgradedGiftSymbol`]):
            Examples of possible symbols that can be chosen for the gift after upgrade

        backdrops (List[:class:`~pytdbot.types.UpgradedGiftBackdrop`]):
            Examples of possible backdrops that can be chosen for the gift after upgrade

        prices (List[:class:`~pytdbot.types.GiftUpgradePrice`]):
            Examples of price for gift upgrade from the maximum price to the minimum price

        next_prices (List[:class:`~pytdbot.types.GiftUpgradePrice`]):
            Next changes for the price for gift upgrade with more granularity than in prices

    """

    def __init__(
        self,
        models: List[UpgradedGiftModel] = None,
        symbols: List[UpgradedGiftSymbol] = None,
        backdrops: List[UpgradedGiftBackdrop] = None,
        prices: List[GiftUpgradePrice] = None,
        next_prices: List[GiftUpgradePrice] = None,
    ) -> None:
        self.models: List[UpgradedGiftModel] = models or []
        r"""Examples of possible models that can be chosen for the gift after upgrade"""
        self.symbols: List[UpgradedGiftSymbol] = symbols or []
        r"""Examples of possible symbols that can be chosen for the gift after upgrade"""
        self.backdrops: List[UpgradedGiftBackdrop] = backdrops or []
        r"""Examples of possible backdrops that can be chosen for the gift after upgrade"""
        self.prices: List[GiftUpgradePrice] = prices or []
        r"""Examples of price for gift upgrade from the maximum price to the minimum price"""
        self.next_prices: List[GiftUpgradePrice] = next_prices or []
        r"""Next changes for the price for gift upgrade with more granularity than in prices"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giftUpgradePreview"]:
        return "giftUpgradePreview"

    @classmethod
    def getClass(self) -> Literal["GiftUpgradePreview"]:
        return "GiftUpgradePreview"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "models": self.models,
            "symbols": self.symbols,
            "backdrops": self.backdrops,
            "prices": self.prices,
            "next_prices": self.next_prices,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiftUpgradePreview", None]:
        if data:
            data_class = cls()
            data_class.models = data.get("models", None)
            data_class.symbols = data.get("symbols", None)
            data_class.backdrops = data.get("backdrops", None)
            data_class.prices = data.get("prices", None)
            data_class.next_prices = data.get("next_prices", None)

        return data_class


class TransactionDirectionIncoming(TlObject, TransactionDirection):
    r"""The transaction is incoming and increases the amount of owned currency"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["transactionDirectionIncoming"]:
        return "transactionDirectionIncoming"

    @classmethod
    def getClass(self) -> Literal["TransactionDirection"]:
        return "TransactionDirection"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["TransactionDirectionIncoming", None]:
        if data:
            data_class = cls()

        return data_class


class TransactionDirectionOutgoing(TlObject, TransactionDirection):
    r"""The transaction is outgoing and decreases the amount of owned currency"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["transactionDirectionOutgoing"]:
        return "transactionDirectionOutgoing"

    @classmethod
    def getClass(self) -> Literal["TransactionDirection"]:
        return "TransactionDirection"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["TransactionDirectionOutgoing", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypePremiumBotDeposit(TlObject, StarTransactionType):
    r"""The transaction is a deposit of Telegram Stars from the Premium bot; for regular users only"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypePremiumBotDeposit"]:
        return "starTransactionTypePremiumBotDeposit"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypePremiumBotDeposit", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypeAppStoreDeposit(TlObject, StarTransactionType):
    r"""The transaction is a deposit of Telegram Stars from App Store; for regular users only"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeAppStoreDeposit"]:
        return "starTransactionTypeAppStoreDeposit"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeAppStoreDeposit", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypeGooglePlayDeposit(TlObject, StarTransactionType):
    r"""The transaction is a deposit of Telegram Stars from Google Play; for regular users only"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGooglePlayDeposit"]:
        return "starTransactionTypeGooglePlayDeposit"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeGooglePlayDeposit", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypeFragmentDeposit(TlObject, StarTransactionType):
    r"""The transaction is a deposit of Telegram Stars from Fragment; for regular users and bots only"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeFragmentDeposit"]:
        return "starTransactionTypeFragmentDeposit"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeFragmentDeposit", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypeUserDeposit(TlObject, StarTransactionType):
    r"""The transaction is a deposit of Telegram Stars by another user; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that gifted Telegram Stars; 0 if the user was anonymous

        sticker (:class:`~pytdbot.types.Sticker`):
            The sticker to be shown in the transaction information; may be null if unknown

    """

    def __init__(self, user_id: int = 0, sticker: Sticker = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that gifted Telegram Stars; 0 if the user was anonymous"""
        self.sticker: Union[Sticker, None] = sticker
        r"""The sticker to be shown in the transaction information; may be null if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeUserDeposit"]:
        return "starTransactionTypeUserDeposit"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeUserDeposit", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.sticker = data.get("sticker", None)

        return data_class


class StarTransactionTypeGiveawayDeposit(TlObject, StarTransactionType):
    r"""The transaction is a deposit of Telegram Stars from a giveaway; for regular users only

    Parameters:
        chat_id (:class:`int`):
            Identifier of a supergroup or a channel chat that created the giveaway

        giveaway_message_id (:class:`int`):
            Identifier of the message with the giveaway; can be 0 or an identifier of a deleted message

    """

    def __init__(self, chat_id: int = 0, giveaway_message_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of a supergroup or a channel chat that created the giveaway"""
        self.giveaway_message_id: int = int(giveaway_message_id)
        r"""Identifier of the message with the giveaway; can be 0 or an identifier of a deleted message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiveawayDeposit"]:
        return "starTransactionTypeGiveawayDeposit"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "giveaway_message_id": self.giveaway_message_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeGiveawayDeposit", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.giveaway_message_id = int(data.get("giveaway_message_id", 0))

        return data_class


class StarTransactionTypeFragmentWithdrawal(TlObject, StarTransactionType):
    r"""The transaction is a withdrawal of earned Telegram Stars to Fragment; for regular users, bots, supergroup and channel chats only

    Parameters:
        withdrawal_state (:class:`~pytdbot.types.RevenueWithdrawalState`):
            State of the withdrawal; may be null for refunds from Fragment

    """

    def __init__(self, withdrawal_state: RevenueWithdrawalState = None) -> None:
        self.withdrawal_state: Union[
            RevenueWithdrawalStatePending,
            RevenueWithdrawalStateSucceeded,
            RevenueWithdrawalStateFailed,
            None,
        ] = withdrawal_state
        r"""State of the withdrawal; may be null for refunds from Fragment"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeFragmentWithdrawal"]:
        return "starTransactionTypeFragmentWithdrawal"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "withdrawal_state": self.withdrawal_state}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeFragmentWithdrawal", None]:
        if data:
            data_class = cls()
            data_class.withdrawal_state = data.get("withdrawal_state", None)

        return data_class


class StarTransactionTypeTelegramAdsWithdrawal(TlObject, StarTransactionType):
    r"""The transaction is a withdrawal of earned Telegram Stars to Telegram Ad platform; for bots and channel chats only"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeTelegramAdsWithdrawal"]:
        return "starTransactionTypeTelegramAdsWithdrawal"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeTelegramAdsWithdrawal", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypeTelegramApiUsage(TlObject, StarTransactionType):
    r"""The transaction is a payment for Telegram API usage; for bots only

    Parameters:
        request_count (:class:`int`):
            The number of billed requests

    """

    def __init__(self, request_count: int = 0) -> None:
        self.request_count: int = int(request_count)
        r"""The number of billed requests"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeTelegramApiUsage"]:
        return "starTransactionTypeTelegramApiUsage"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "request_count": self.request_count}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeTelegramApiUsage", None]:
        if data:
            data_class = cls()
            data_class.request_count = int(data.get("request_count", 0))

        return data_class


class StarTransactionTypeBotPaidMediaPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of paid media from a bot or a business account by the current user; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the bot or the business account user that sent the paid media

        media (List[:class:`~pytdbot.types.PaidMedia`]):
            The bought media if the transaction wasn't refunded

    """

    def __init__(self, user_id: int = 0, media: List[PaidMedia] = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the bot or the business account user that sent the paid media"""
        self.media: List[PaidMedia] = media or []
        r"""The bought media if the transaction wasn't refunded"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBotPaidMediaPurchase"]:
        return "starTransactionTypeBotPaidMediaPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id, "media": self.media}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBotPaidMediaPurchase", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.media = data.get("media", None)

        return data_class


class StarTransactionTypeBotPaidMediaSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of paid media by the bot or a business account managed by the bot; for bots only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the media

        media (List[:class:`~pytdbot.types.PaidMedia`]):
            The bought media

        payload (:class:`str`):
            Bot\-provided payload

        affiliate (:class:`~pytdbot.types.AffiliateInfo`):
            Information about the affiliate which received commission from the transaction; may be null if none

    """

    def __init__(
        self,
        user_id: int = 0,
        media: List[PaidMedia] = None,
        payload: str = "",
        affiliate: AffiliateInfo = None,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the media"""
        self.media: List[PaidMedia] = media or []
        r"""The bought media"""
        self.payload: Union[str, None] = payload
        r"""Bot\-provided payload"""
        self.affiliate: Union[AffiliateInfo, None] = affiliate
        r"""Information about the affiliate which received commission from the transaction; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBotPaidMediaSale"]:
        return "starTransactionTypeBotPaidMediaSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "media": self.media,
            "payload": self.payload,
            "affiliate": self.affiliate,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBotPaidMediaSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.media = data.get("media", None)
            data_class.payload = data.get("payload", "")
            data_class.affiliate = data.get("affiliate", None)

        return data_class


class StarTransactionTypeChannelPaidMediaPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of paid media from a channel by the current user; for regular users only

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat that sent the paid media

        message_id (:class:`int`):
            Identifier of the corresponding message with paid media; can be 0 or an identifier of a deleted message

        media (List[:class:`~pytdbot.types.PaidMedia`]):
            The bought media if the transaction wasn't refunded

    """

    def __init__(
        self, chat_id: int = 0, message_id: int = 0, media: List[PaidMedia] = None
    ) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat that sent the paid media"""
        self.message_id: int = int(message_id)
        r"""Identifier of the corresponding message with paid media; can be 0 or an identifier of a deleted message"""
        self.media: List[PaidMedia] = media or []
        r"""The bought media if the transaction wasn't refunded"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeChannelPaidMediaPurchase"]:
        return "starTransactionTypeChannelPaidMediaPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_id": self.message_id,
            "media": self.media,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeChannelPaidMediaPurchase", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.media = data.get("media", None)

        return data_class


class StarTransactionTypeChannelPaidMediaSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of paid media by the channel chat; for channel chats only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the media

        message_id (:class:`int`):
            Identifier of the corresponding message with paid media; can be 0 or an identifier of a deleted message

        media (List[:class:`~pytdbot.types.PaidMedia`]):
            The bought media

    """

    def __init__(
        self, user_id: int = 0, message_id: int = 0, media: List[PaidMedia] = None
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the media"""
        self.message_id: int = int(message_id)
        r"""Identifier of the corresponding message with paid media; can be 0 or an identifier of a deleted message"""
        self.media: List[PaidMedia] = media or []
        r"""The bought media"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeChannelPaidMediaSale"]:
        return "starTransactionTypeChannelPaidMediaSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "message_id": self.message_id,
            "media": self.media,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeChannelPaidMediaSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.media = data.get("media", None)

        return data_class


class StarTransactionTypeBotInvoicePurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of a product from a bot or a business account by the current user; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the bot or the business account user that created the invoice

        product_info (:class:`~pytdbot.types.ProductInfo`):
            Information about the bought product

    """

    def __init__(self, user_id: int = 0, product_info: ProductInfo = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the bot or the business account user that created the invoice"""
        self.product_info: Union[ProductInfo, None] = product_info
        r"""Information about the bought product"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBotInvoicePurchase"]:
        return "starTransactionTypeBotInvoicePurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "product_info": self.product_info,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBotInvoicePurchase", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.product_info = data.get("product_info", None)

        return data_class


class StarTransactionTypeBotInvoiceSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of a product by the bot; for bots only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the product

        product_info (:class:`~pytdbot.types.ProductInfo`):
            Information about the bought product

        invoice_payload (:class:`bytes`):
            Invoice payload

        affiliate (:class:`~pytdbot.types.AffiliateInfo`):
            Information about the affiliate which received commission from the transaction; may be null if none

    """

    def __init__(
        self,
        user_id: int = 0,
        product_info: ProductInfo = None,
        invoice_payload: bytes = b"",
        affiliate: AffiliateInfo = None,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the product"""
        self.product_info: Union[ProductInfo, None] = product_info
        r"""Information about the bought product"""
        self.invoice_payload: Union[bytes, None] = invoice_payload
        r"""Invoice payload"""
        self.affiliate: Union[AffiliateInfo, None] = affiliate
        r"""Information about the affiliate which received commission from the transaction; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBotInvoiceSale"]:
        return "starTransactionTypeBotInvoiceSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "product_info": self.product_info,
            "invoice_payload": self.invoice_payload,
            "affiliate": self.affiliate,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeBotInvoiceSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.product_info = data.get("product_info", None)
            data_class.invoice_payload = b64decode(data.get("invoice_payload", b""))
            data_class.affiliate = data.get("affiliate", None)

        return data_class


class StarTransactionTypeBotSubscriptionPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of a subscription from a bot or a business account by the current user; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the bot or the business account user that created the subscription link

        subscription_period (:class:`int`):
            The number of seconds between consecutive Telegram Star debitings

        product_info (:class:`~pytdbot.types.ProductInfo`):
            Information about the bought subscription

    """

    def __init__(
        self,
        user_id: int = 0,
        subscription_period: int = 0,
        product_info: ProductInfo = None,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the bot or the business account user that created the subscription link"""
        self.subscription_period: int = int(subscription_period)
        r"""The number of seconds between consecutive Telegram Star debitings"""
        self.product_info: Union[ProductInfo, None] = product_info
        r"""Information about the bought subscription"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBotSubscriptionPurchase"]:
        return "starTransactionTypeBotSubscriptionPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "subscription_period": self.subscription_period,
            "product_info": self.product_info,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBotSubscriptionPurchase", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.subscription_period = int(data.get("subscription_period", 0))
            data_class.product_info = data.get("product_info", None)

        return data_class


class StarTransactionTypeBotSubscriptionSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of a subscription by the bot; for bots only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the subscription

        subscription_period (:class:`int`):
            The number of seconds between consecutive Telegram Star debitings

        product_info (:class:`~pytdbot.types.ProductInfo`):
            Information about the bought subscription

        invoice_payload (:class:`bytes`):
            Invoice payload

        affiliate (:class:`~pytdbot.types.AffiliateInfo`):
            Information about the affiliate which received commission from the transaction; may be null if none

    """

    def __init__(
        self,
        user_id: int = 0,
        subscription_period: int = 0,
        product_info: ProductInfo = None,
        invoice_payload: bytes = b"",
        affiliate: AffiliateInfo = None,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the subscription"""
        self.subscription_period: int = int(subscription_period)
        r"""The number of seconds between consecutive Telegram Star debitings"""
        self.product_info: Union[ProductInfo, None] = product_info
        r"""Information about the bought subscription"""
        self.invoice_payload: Union[bytes, None] = invoice_payload
        r"""Invoice payload"""
        self.affiliate: Union[AffiliateInfo, None] = affiliate
        r"""Information about the affiliate which received commission from the transaction; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBotSubscriptionSale"]:
        return "starTransactionTypeBotSubscriptionSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "subscription_period": self.subscription_period,
            "product_info": self.product_info,
            "invoice_payload": self.invoice_payload,
            "affiliate": self.affiliate,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBotSubscriptionSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.subscription_period = int(data.get("subscription_period", 0))
            data_class.product_info = data.get("product_info", None)
            data_class.invoice_payload = b64decode(data.get("invoice_payload", b""))
            data_class.affiliate = data.get("affiliate", None)

        return data_class


class StarTransactionTypeChannelSubscriptionPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of a subscription to a channel chat by the current user; for regular users only

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat that created the subscription

        subscription_period (:class:`int`):
            The number of seconds between consecutive Telegram Star debitings

    """

    def __init__(self, chat_id: int = 0, subscription_period: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat that created the subscription"""
        self.subscription_period: int = int(subscription_period)
        r"""The number of seconds between consecutive Telegram Star debitings"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeChannelSubscriptionPurchase"]:
        return "starTransactionTypeChannelSubscriptionPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "subscription_period": self.subscription_period,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeChannelSubscriptionPurchase", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.subscription_period = int(data.get("subscription_period", 0))

        return data_class


class StarTransactionTypeChannelSubscriptionSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of a subscription by the channel chat; for channel chats only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the subscription

        subscription_period (:class:`int`):
            The number of seconds between consecutive Telegram Star debitings

    """

    def __init__(self, user_id: int = 0, subscription_period: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the subscription"""
        self.subscription_period: int = int(subscription_period)
        r"""The number of seconds between consecutive Telegram Star debitings"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeChannelSubscriptionSale"]:
        return "starTransactionTypeChannelSubscriptionSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "subscription_period": self.subscription_period,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeChannelSubscriptionSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.subscription_period = int(data.get("subscription_period", 0))

        return data_class


class StarTransactionTypeGiftPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of a regular gift; for regular users and bots only

    Parameters:
        owner_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the channel that received the gift

        gift (:class:`~pytdbot.types.Gift`):
            The gift

    """

    def __init__(self, owner_id: MessageSender = None, gift: Gift = None) -> None:
        self.owner_id: Union[MessageSenderUser, MessageSenderChat, None] = owner_id
        r"""Identifier of the user or the channel that received the gift"""
        self.gift: Union[Gift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiftPurchase"]:
        return "starTransactionTypeGiftPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "owner_id": self.owner_id, "gift": self.gift}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeGiftPurchase", None]:
        if data:
            data_class = cls()
            data_class.owner_id = data.get("owner_id", None)
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeGiftTransfer(TlObject, StarTransactionType):
    r"""The transaction is a transfer of an upgraded gift; for regular users only

    Parameters:
        owner_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the channel that received the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

    """

    def __init__(
        self, owner_id: MessageSender = None, gift: UpgradedGift = None
    ) -> None:
        self.owner_id: Union[MessageSenderUser, MessageSenderChat, None] = owner_id
        r"""Identifier of the user or the channel that received the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiftTransfer"]:
        return "starTransactionTypeGiftTransfer"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "owner_id": self.owner_id, "gift": self.gift}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeGiftTransfer", None]:
        if data:
            data_class = cls()
            data_class.owner_id = data.get("owner_id", None)
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeGiftOriginalDetailsDrop(TlObject, StarTransactionType):
    r"""The transaction is a drop of original details of an upgraded gift; for regular users only

    Parameters:
        owner_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or the channel that owns the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

    """

    def __init__(
        self, owner_id: MessageSender = None, gift: UpgradedGift = None
    ) -> None:
        self.owner_id: Union[MessageSenderUser, MessageSenderChat, None] = owner_id
        r"""Identifier of the user or the channel that owns the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiftOriginalDetailsDrop"]:
        return "starTransactionTypeGiftOriginalDetailsDrop"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "owner_id": self.owner_id, "gift": self.gift}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeGiftOriginalDetailsDrop", None]:
        if data:
            data_class = cls()
            data_class.owner_id = data.get("owner_id", None)
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeGiftSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of a received gift; for regular users and channel chats only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that sent the gift

        gift (:class:`~pytdbot.types.Gift`):
            The gift

    """

    def __init__(self, user_id: int = 0, gift: Gift = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that sent the gift"""
        self.gift: Union[Gift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiftSale"]:
        return "starTransactionTypeGiftSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id, "gift": self.gift}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeGiftSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeGiftUpgrade(TlObject, StarTransactionType):
    r"""The transaction is an upgrade of a gift; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that initially sent the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The upgraded gift

    """

    def __init__(self, user_id: int = 0, gift: UpgradedGift = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that initially sent the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The upgraded gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiftUpgrade"]:
        return "starTransactionTypeGiftUpgrade"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id, "gift": self.gift}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeGiftUpgrade", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeGiftUpgradePurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of an upgrade of a gift owned by another user or channel; for regular users only

    Parameters:
        owner_id (:class:`~pytdbot.types.MessageSender`):
            Owner of the upgraded gift

        gift (:class:`~pytdbot.types.Gift`):
            The gift

    """

    def __init__(self, owner_id: MessageSender = None, gift: Gift = None) -> None:
        self.owner_id: Union[MessageSenderUser, MessageSenderChat, None] = owner_id
        r"""Owner of the upgraded gift"""
        self.gift: Union[Gift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeGiftUpgradePurchase"]:
        return "starTransactionTypeGiftUpgradePurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "owner_id": self.owner_id, "gift": self.gift}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeGiftUpgradePurchase", None]:
        if data:
            data_class = cls()
            data_class.owner_id = data.get("owner_id", None)
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeUpgradedGiftPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of an upgraded gift for some user or channel; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that sold the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

    """

    def __init__(self, user_id: int = 0, gift: UpgradedGift = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that sold the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeUpgradedGiftPurchase"]:
        return "starTransactionTypeUpgradedGiftPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id, "gift": self.gift}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeUpgradedGiftPurchase", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.gift = data.get("gift", None)

        return data_class


class StarTransactionTypeUpgradedGiftSale(TlObject, StarTransactionType):
    r"""The transaction is a sale of an upgraded gift; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

        commission_per_mille (:class:`int`):
            The number of Telegram Stars received by the Telegram for each 1000 Telegram Stars received by the seller of the gift

        commission_star_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of Telegram Stars that were received by Telegram; can be negative for refunds

    """

    def __init__(
        self,
        user_id: int = 0,
        gift: UpgradedGift = None,
        commission_per_mille: int = 0,
        commission_star_amount: StarAmount = None,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""
        self.commission_per_mille: int = int(commission_per_mille)
        r"""The number of Telegram Stars received by the Telegram for each 1000 Telegram Stars received by the seller of the gift"""
        self.commission_star_amount: Union[StarAmount, None] = commission_star_amount
        r"""The amount of Telegram Stars that were received by Telegram; can be negative for refunds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeUpgradedGiftSale"]:
        return "starTransactionTypeUpgradedGiftSale"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "gift": self.gift,
            "commission_per_mille": self.commission_per_mille,
            "commission_star_amount": self.commission_star_amount,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeUpgradedGiftSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.gift = data.get("gift", None)
            data_class.commission_per_mille = int(data.get("commission_per_mille", 0))
            data_class.commission_star_amount = data.get("commission_star_amount", None)

        return data_class


class StarTransactionTypeChannelPaidReactionSend(TlObject, StarTransactionType):
    r"""The transaction is a sending of a paid reaction to a message in a channel chat by the current user; for regular users only

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat

        message_id (:class:`int`):
            Identifier of the reacted message; can be 0 or an identifier of a deleted message

    """

    def __init__(self, chat_id: int = 0, message_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat"""
        self.message_id: int = int(message_id)
        r"""Identifier of the reacted message; can be 0 or an identifier of a deleted message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeChannelPaidReactionSend"]:
        return "starTransactionTypeChannelPaidReactionSend"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_id": self.message_id,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeChannelPaidReactionSend", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_id = int(data.get("message_id", 0))

        return data_class


class StarTransactionTypeChannelPaidReactionReceive(TlObject, StarTransactionType):
    r"""The transaction is a receiving of a paid reaction to a message by the channel chat; for channel chats only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that added the paid reaction

        message_id (:class:`int`):
            Identifier of the reacted message; can be 0 or an identifier of a deleted message

    """

    def __init__(self, user_id: int = 0, message_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that added the paid reaction"""
        self.message_id: int = int(message_id)
        r"""Identifier of the reacted message; can be 0 or an identifier of a deleted message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeChannelPaidReactionReceive"]:
        return "starTransactionTypeChannelPaidReactionReceive"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "message_id": self.message_id,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeChannelPaidReactionReceive", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.message_id = int(data.get("message_id", 0))

        return data_class


class StarTransactionTypeAffiliateProgramCommission(TlObject, StarTransactionType):
    r"""The transaction is a receiving of a commission from an affiliate program; for regular users, bots and channel chats only

    Parameters:
        chat_id (:class:`int`):
            Identifier of the chat that created the affiliate program

        commission_per_mille (:class:`int`):
            The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner

    """

    def __init__(self, chat_id: int = 0, commission_per_mille: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat that created the affiliate program"""
        self.commission_per_mille: int = int(commission_per_mille)
        r"""The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeAffiliateProgramCommission"]:
        return "starTransactionTypeAffiliateProgramCommission"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "commission_per_mille": self.commission_per_mille,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeAffiliateProgramCommission", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.commission_per_mille = int(data.get("commission_per_mille", 0))

        return data_class


class StarTransactionTypePaidMessageSend(TlObject, StarTransactionType):
    r"""The transaction is a sending of a paid message; for regular users only

    Parameters:
        chat_id (:class:`int`):
            Identifier of the chat that received the payment

        message_count (:class:`int`):
            Number of sent paid messages

    """

    def __init__(self, chat_id: int = 0, message_count: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat that received the payment"""
        self.message_count: int = int(message_count)
        r"""Number of sent paid messages"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypePaidMessageSend"]:
        return "starTransactionTypePaidMessageSend"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_count": self.message_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypePaidMessageSend", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_count = int(data.get("message_count", 0))

        return data_class


class StarTransactionTypePaidMessageReceive(TlObject, StarTransactionType):
    r"""The transaction is a receiving of a paid message; for regular users, supergroup and channel chats only

    Parameters:
        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the sender of the message

        message_count (:class:`int`):
            Number of received paid messages

        commission_per_mille (:class:`int`):
            The number of Telegram Stars received by the Telegram for each 1000 Telegram Stars paid for message sending

        commission_star_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of Telegram Stars that were received by Telegram; can be negative for refunds

    """

    def __init__(
        self,
        sender_id: MessageSender = None,
        message_count: int = 0,
        commission_per_mille: int = 0,
        commission_star_amount: StarAmount = None,
    ) -> None:
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of the sender of the message"""
        self.message_count: int = int(message_count)
        r"""Number of received paid messages"""
        self.commission_per_mille: int = int(commission_per_mille)
        r"""The number of Telegram Stars received by the Telegram for each 1000 Telegram Stars paid for message sending"""
        self.commission_star_amount: Union[StarAmount, None] = commission_star_amount
        r"""The amount of Telegram Stars that were received by Telegram; can be negative for refunds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypePaidMessageReceive"]:
        return "starTransactionTypePaidMessageReceive"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sender_id": self.sender_id,
            "message_count": self.message_count,
            "commission_per_mille": self.commission_per_mille,
            "commission_star_amount": self.commission_star_amount,
        }

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypePaidMessageReceive", None]:
        if data:
            data_class = cls()
            data_class.sender_id = data.get("sender_id", None)
            data_class.message_count = int(data.get("message_count", 0))
            data_class.commission_per_mille = int(data.get("commission_per_mille", 0))
            data_class.commission_star_amount = data.get("commission_star_amount", None)

        return data_class


class StarTransactionTypeSuggestedPostPaymentSend(TlObject, StarTransactionType):
    r"""The transaction is a payment for a suggested post; for regular users only

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat that posted the post

    """

    def __init__(self, chat_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat that posted the post"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeSuggestedPostPaymentSend"]:
        return "starTransactionTypeSuggestedPostPaymentSend"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeSuggestedPostPaymentSend", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))

        return data_class


class StarTransactionTypeSuggestedPostPaymentReceive(TlObject, StarTransactionType):
    r"""The transaction is a receiving of a payment for a suggested post by the channel chat; for channel chats only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that paid for the suggested post

    """

    def __init__(self, user_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that paid for the suggested post"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeSuggestedPostPaymentReceive"]:
        return "starTransactionTypeSuggestedPostPaymentReceive"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeSuggestedPostPaymentReceive", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class StarTransactionTypePremiumPurchase(TlObject, StarTransactionType):
    r"""The transaction is a purchase of Telegram Premium subscription; for regular users and bots only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that received the Telegram Premium subscription

        month_count (:class:`int`):
            Number of months the Telegram Premium subscription will be active

        sticker (:class:`~pytdbot.types.Sticker`):
            A sticker to be shown in the transaction information; may be null if unknown

    """

    def __init__(
        self, user_id: int = 0, month_count: int = 0, sticker: Sticker = None
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that received the Telegram Premium subscription"""
        self.month_count: int = int(month_count)
        r"""Number of months the Telegram Premium subscription will be active"""
        self.sticker: Union[Sticker, None] = sticker
        r"""A sticker to be shown in the transaction information; may be null if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypePremiumPurchase"]:
        return "starTransactionTypePremiumPurchase"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "month_count": self.month_count,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypePremiumPurchase", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.month_count = int(data.get("month_count", 0))
            data_class.sticker = data.get("sticker", None)

        return data_class


class StarTransactionTypeBusinessBotTransferSend(TlObject, StarTransactionType):
    r"""The transaction is a transfer of Telegram Stars to a business bot; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the bot that received Telegram Stars

    """

    def __init__(self, user_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the bot that received Telegram Stars"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBusinessBotTransferSend"]:
        return "starTransactionTypeBusinessBotTransferSend"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBusinessBotTransferSend", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class StarTransactionTypeBusinessBotTransferReceive(TlObject, StarTransactionType):
    r"""The transaction is a transfer of Telegram Stars from a business account; for bots only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that sent Telegram Stars

    """

    def __init__(self, user_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that sent Telegram Stars"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeBusinessBotTransferReceive"]:
        return "starTransactionTypeBusinessBotTransferReceive"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypeBusinessBotTransferReceive", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class StarTransactionTypePublicPostSearch(TlObject, StarTransactionType):
    r"""The transaction is a payment for search of posts in public Telegram channels; for regular users only"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypePublicPostSearch"]:
        return "starTransactionTypePublicPostSearch"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["StarTransactionTypePublicPostSearch", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransactionTypeUnsupported(TlObject, StarTransactionType):
    r"""The transaction is a transaction of an unsupported type"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactionTypeUnsupported"]:
        return "starTransactionTypeUnsupported"

    @classmethod
    def getClass(self) -> Literal["StarTransactionType"]:
        return "StarTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactionTypeUnsupported", None]:
        if data:
            data_class = cls()

        return data_class


class StarTransaction(TlObject):
    r"""Represents a transaction changing the amount of owned Telegram Stars

    Parameters:
        id (:class:`str`):
            Unique identifier of the transaction

        star_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of added owned Telegram Stars; negative for outgoing transactions

        is_refund (:class:`bool`):
            True, if the transaction is a refund of a previous transaction

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the transaction was completed

        type (:class:`~pytdbot.types.StarTransactionType`):
            Type of the transaction

    """

    def __init__(
        self,
        id: str = "",
        star_amount: StarAmount = None,
        is_refund: bool = False,
        date: int = 0,
        type: StarTransactionType = None,
    ) -> None:
        self.id: Union[str, None] = id
        r"""Unique identifier of the transaction"""
        self.star_amount: Union[StarAmount, None] = star_amount
        r"""The amount of added owned Telegram Stars; negative for outgoing transactions"""
        self.is_refund: bool = bool(is_refund)
        r"""True, if the transaction is a refund of a previous transaction"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the transaction was completed"""
        self.type: Union[
            StarTransactionTypePremiumBotDeposit,
            StarTransactionTypeAppStoreDeposit,
            StarTransactionTypeGooglePlayDeposit,
            StarTransactionTypeFragmentDeposit,
            StarTransactionTypeUserDeposit,
            StarTransactionTypeGiveawayDeposit,
            StarTransactionTypeFragmentWithdrawal,
            StarTransactionTypeTelegramAdsWithdrawal,
            StarTransactionTypeTelegramApiUsage,
            StarTransactionTypeBotPaidMediaPurchase,
            StarTransactionTypeBotPaidMediaSale,
            StarTransactionTypeChannelPaidMediaPurchase,
            StarTransactionTypeChannelPaidMediaSale,
            StarTransactionTypeBotInvoicePurchase,
            StarTransactionTypeBotInvoiceSale,
            StarTransactionTypeBotSubscriptionPurchase,
            StarTransactionTypeBotSubscriptionSale,
            StarTransactionTypeChannelSubscriptionPurchase,
            StarTransactionTypeChannelSubscriptionSale,
            StarTransactionTypeGiftPurchase,
            StarTransactionTypeGiftTransfer,
            StarTransactionTypeGiftOriginalDetailsDrop,
            StarTransactionTypeGiftSale,
            StarTransactionTypeGiftUpgrade,
            StarTransactionTypeGiftUpgradePurchase,
            StarTransactionTypeUpgradedGiftPurchase,
            StarTransactionTypeUpgradedGiftSale,
            StarTransactionTypeChannelPaidReactionSend,
            StarTransactionTypeChannelPaidReactionReceive,
            StarTransactionTypeAffiliateProgramCommission,
            StarTransactionTypePaidMessageSend,
            StarTransactionTypePaidMessageReceive,
            StarTransactionTypeSuggestedPostPaymentSend,
            StarTransactionTypeSuggestedPostPaymentReceive,
            StarTransactionTypePremiumPurchase,
            StarTransactionTypeBusinessBotTransferSend,
            StarTransactionTypeBusinessBotTransferReceive,
            StarTransactionTypePublicPostSearch,
            StarTransactionTypeUnsupported,
            None,
        ] = type
        r"""Type of the transaction"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransaction"]:
        return "starTransaction"

    @classmethod
    def getClass(self) -> Literal["StarTransaction"]:
        return "StarTransaction"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "star_amount": self.star_amount,
            "is_refund": self.is_refund,
            "date": self.date,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransaction", None]:
        if data:
            data_class = cls()
            data_class.id = data.get("id", "")
            data_class.star_amount = data.get("star_amount", None)
            data_class.is_refund = data.get("is_refund", False)
            data_class.date = int(data.get("date", 0))
            data_class.type = data.get("type", None)

        return data_class


class StarTransactions(TlObject):
    r"""Represents a list of Telegram Star transactions

    Parameters:
        star_amount (:class:`~pytdbot.types.StarAmount`):
            The amount of owned Telegram Stars

        transactions (List[:class:`~pytdbot.types.StarTransaction`]):
            List of transactions with Telegram Stars

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        star_amount: StarAmount = None,
        transactions: List[StarTransaction] = None,
        next_offset: str = "",
    ) -> None:
        self.star_amount: Union[StarAmount, None] = star_amount
        r"""The amount of owned Telegram Stars"""
        self.transactions: List[StarTransaction] = transactions or []
        r"""List of transactions with Telegram Stars"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["starTransactions"]:
        return "starTransactions"

    @classmethod
    def getClass(self) -> Literal["StarTransactions"]:
        return "StarTransactions"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "star_amount": self.star_amount,
            "transactions": self.transactions,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["StarTransactions", None]:
        if data:
            data_class = cls()
            data_class.star_amount = data.get("star_amount", None)
            data_class.transactions = data.get("transactions", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class TonTransactionTypeFragmentDeposit(TlObject, TonTransactionType):
    r"""The transaction is a deposit of Toncoins from Fragment

    Parameters:
        is_gift (:class:`bool`):
            True, if the transaction is a gift from another user

        sticker (:class:`~pytdbot.types.Sticker`):
            The sticker to be shown in the transaction information; may be null if unknown

    """

    def __init__(self, is_gift: bool = False, sticker: Sticker = None) -> None:
        self.is_gift: bool = bool(is_gift)
        r"""True, if the transaction is a gift from another user"""
        self.sticker: Union[Sticker, None] = sticker
        r"""The sticker to be shown in the transaction information; may be null if unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransactionTypeFragmentDeposit"]:
        return "tonTransactionTypeFragmentDeposit"

    @classmethod
    def getClass(self) -> Literal["TonTransactionType"]:
        return "TonTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "is_gift": self.is_gift,
            "sticker": self.sticker,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TonTransactionTypeFragmentDeposit", None]:
        if data:
            data_class = cls()
            data_class.is_gift = data.get("is_gift", False)
            data_class.sticker = data.get("sticker", None)

        return data_class


class TonTransactionTypeSuggestedPostPayment(TlObject, TonTransactionType):
    r"""The transaction is a payment for a suggested post

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat that posted the post

    """

    def __init__(self, chat_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat that posted the post"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransactionTypeSuggestedPostPayment"]:
        return "tonTransactionTypeSuggestedPostPayment"

    @classmethod
    def getClass(self) -> Literal["TonTransactionType"]:
        return "TonTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["TonTransactionTypeSuggestedPostPayment", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))

        return data_class


class TonTransactionTypeUpgradedGiftPurchase(TlObject, TonTransactionType):
    r"""The transaction is a purchase of an upgraded gift for some user or channel; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that sold the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

    """

    def __init__(self, user_id: int = 0, gift: UpgradedGift = None) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that sold the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransactionTypeUpgradedGiftPurchase"]:
        return "tonTransactionTypeUpgradedGiftPurchase"

    @classmethod
    def getClass(self) -> Literal["TonTransactionType"]:
        return "TonTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id, "gift": self.gift}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["TonTransactionTypeUpgradedGiftPurchase", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.gift = data.get("gift", None)

        return data_class


class TonTransactionTypeUpgradedGiftSale(TlObject, TonTransactionType):
    r"""The transaction is a sale of an upgraded gift; for regular users only

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that bought the gift

        gift (:class:`~pytdbot.types.UpgradedGift`):
            The gift

        commission_per_mille (:class:`int`):
            The number of Toncoins received by the Telegram for each 1000 Toncoins received by the seller of the gift

        commission_toncoin_amount (:class:`int`):
            The amount of Toncoins that were received by the Telegram; in the smallest units of the currency

    """

    def __init__(
        self,
        user_id: int = 0,
        gift: UpgradedGift = None,
        commission_per_mille: int = 0,
        commission_toncoin_amount: int = 0,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that bought the gift"""
        self.gift: Union[UpgradedGift, None] = gift
        r"""The gift"""
        self.commission_per_mille: int = int(commission_per_mille)
        r"""The number of Toncoins received by the Telegram for each 1000 Toncoins received by the seller of the gift"""
        self.commission_toncoin_amount: int = int(commission_toncoin_amount)
        r"""The amount of Toncoins that were received by the Telegram; in the smallest units of the currency"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransactionTypeUpgradedGiftSale"]:
        return "tonTransactionTypeUpgradedGiftSale"

    @classmethod
    def getClass(self) -> Literal["TonTransactionType"]:
        return "TonTransactionType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "gift": self.gift,
            "commission_per_mille": self.commission_per_mille,
            "commission_toncoin_amount": self.commission_toncoin_amount,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TonTransactionTypeUpgradedGiftSale", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.gift = data.get("gift", None)
            data_class.commission_per_mille = int(data.get("commission_per_mille", 0))
            data_class.commission_toncoin_amount = int(
                data.get("commission_toncoin_amount", 0)
            )

        return data_class


class TonTransactionTypeUnsupported(TlObject, TonTransactionType):
    r"""The transaction is a transaction of an unsupported type"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransactionTypeUnsupported"]:
        return "tonTransactionTypeUnsupported"

    @classmethod
    def getClass(self) -> Literal["TonTransactionType"]:
        return "TonTransactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["TonTransactionTypeUnsupported", None]:
        if data:
            data_class = cls()

        return data_class


class TonTransaction(TlObject):
    r"""Represents a transaction changing the amount of owned Toncoins

    Parameters:
        id (:class:`str`):
            Unique identifier of the transaction

        ton_amount (:class:`int`):
            The amount of added owned Toncoins; negative for outgoing transactions

        is_refund (:class:`bool`):
            True, if the transaction is a refund of a previous transaction

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the transaction was completed

        type (:class:`~pytdbot.types.TonTransactionType`):
            Type of the transaction

    """

    def __init__(
        self,
        id: str = "",
        ton_amount: int = 0,
        is_refund: bool = False,
        date: int = 0,
        type: TonTransactionType = None,
    ) -> None:
        self.id: Union[str, None] = id
        r"""Unique identifier of the transaction"""
        self.ton_amount: int = int(ton_amount)
        r"""The amount of added owned Toncoins; negative for outgoing transactions"""
        self.is_refund: bool = bool(is_refund)
        r"""True, if the transaction is a refund of a previous transaction"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the transaction was completed"""
        self.type: Union[
            TonTransactionTypeFragmentDeposit,
            TonTransactionTypeSuggestedPostPayment,
            TonTransactionTypeUpgradedGiftPurchase,
            TonTransactionTypeUpgradedGiftSale,
            TonTransactionTypeUnsupported,
            None,
        ] = type
        r"""Type of the transaction"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransaction"]:
        return "tonTransaction"

    @classmethod
    def getClass(self) -> Literal["TonTransaction"]:
        return "TonTransaction"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "ton_amount": self.ton_amount,
            "is_refund": self.is_refund,
            "date": self.date,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TonTransaction", None]:
        if data:
            data_class = cls()
            data_class.id = data.get("id", "")
            data_class.ton_amount = int(data.get("ton_amount", 0))
            data_class.is_refund = data.get("is_refund", False)
            data_class.date = int(data.get("date", 0))
            data_class.type = data.get("type", None)

        return data_class


class TonTransactions(TlObject):
    r"""Represents a list of Toncoin transactions

    Parameters:
        ton_amount (:class:`int`):
            The total amount of owned Toncoins

        transactions (List[:class:`~pytdbot.types.TonTransaction`]):
            List of Toncoin transactions

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        ton_amount: int = 0,
        transactions: List[TonTransaction] = None,
        next_offset: str = "",
    ) -> None:
        self.ton_amount: int = int(ton_amount)
        r"""The total amount of owned Toncoins"""
        self.transactions: List[TonTransaction] = transactions or []
        r"""List of Toncoin transactions"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["tonTransactions"]:
        return "tonTransactions"

    @classmethod
    def getClass(self) -> Literal["TonTransactions"]:
        return "TonTransactions"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "ton_amount": self.ton_amount,
            "transactions": self.transactions,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TonTransactions", None]:
        if data:
            data_class = cls()
            data_class.ton_amount = int(data.get("ton_amount", 0))
            data_class.transactions = data.get("transactions", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class GiveawayParticipantStatusEligible(TlObject, GiveawayParticipantStatus):
    r"""The user is eligible for the giveaway"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayParticipantStatusEligible"]:
        return "giveawayParticipantStatusEligible"

    @classmethod
    def getClass(self) -> Literal["GiveawayParticipantStatus"]:
        return "GiveawayParticipantStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiveawayParticipantStatusEligible", None]:
        if data:
            data_class = cls()

        return data_class


class GiveawayParticipantStatusParticipating(TlObject, GiveawayParticipantStatus):
    r"""The user participates in the giveaway"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayParticipantStatusParticipating"]:
        return "giveawayParticipantStatusParticipating"

    @classmethod
    def getClass(self) -> Literal["GiveawayParticipantStatus"]:
        return "GiveawayParticipantStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["GiveawayParticipantStatusParticipating", None]:
        if data:
            data_class = cls()

        return data_class


class GiveawayParticipantStatusAlreadyWasMember(TlObject, GiveawayParticipantStatus):
    r"""The user can't participate in the giveaway, because they have already been member of the chat

    Parameters:
        joined_chat_date (:class:`int`):
            Point in time \(Unix timestamp\) when the user joined the chat

    """

    def __init__(self, joined_chat_date: int = 0) -> None:
        self.joined_chat_date: int = int(joined_chat_date)
        r"""Point in time \(Unix timestamp\) when the user joined the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayParticipantStatusAlreadyWasMember"]:
        return "giveawayParticipantStatusAlreadyWasMember"

    @classmethod
    def getClass(self) -> Literal["GiveawayParticipantStatus"]:
        return "GiveawayParticipantStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "joined_chat_date": self.joined_chat_date}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["GiveawayParticipantStatusAlreadyWasMember", None]:
        if data:
            data_class = cls()
            data_class.joined_chat_date = int(data.get("joined_chat_date", 0))

        return data_class


class GiveawayParticipantStatusAdministrator(TlObject, GiveawayParticipantStatus):
    r"""The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway

    Parameters:
        chat_id (:class:`int`):
            Identifier of the chat administered by the user

    """

    def __init__(self, chat_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat administered by the user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayParticipantStatusAdministrator"]:
        return "giveawayParticipantStatusAdministrator"

    @classmethod
    def getClass(self) -> Literal["GiveawayParticipantStatus"]:
        return "GiveawayParticipantStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["GiveawayParticipantStatusAdministrator", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))

        return data_class


class GiveawayParticipantStatusDisallowedCountry(TlObject, GiveawayParticipantStatus):
    r"""The user can't participate in the giveaway, because they phone number is from a disallowed country

    Parameters:
        user_country_code (:class:`str`):
            A two\-letter ISO 3166\-1 alpha\-2 country code of the user's country

    """

    def __init__(self, user_country_code: str = "") -> None:
        self.user_country_code: Union[str, None] = user_country_code
        r"""A two\-letter ISO 3166\-1 alpha\-2 country code of the user's country"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayParticipantStatusDisallowedCountry"]:
        return "giveawayParticipantStatusDisallowedCountry"

    @classmethod
    def getClass(self) -> Literal["GiveawayParticipantStatus"]:
        return "GiveawayParticipantStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_country_code": self.user_country_code}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["GiveawayParticipantStatusDisallowedCountry", None]:
        if data:
            data_class = cls()
            data_class.user_country_code = data.get("user_country_code", "")

        return data_class


class GiveawayInfoOngoing(TlObject, GiveawayInfo):
    r"""Describes an ongoing giveaway

    Parameters:
        creation_date (:class:`int`):
            Point in time \(Unix timestamp\) when the giveaway was created

        status (:class:`~pytdbot.types.GiveawayParticipantStatus`):
            Status of the current user in the giveaway

        is_ended (:class:`bool`):
            True, if the giveaway has ended and results are being prepared

    """

    def __init__(
        self,
        creation_date: int = 0,
        status: GiveawayParticipantStatus = None,
        is_ended: bool = False,
    ) -> None:
        self.creation_date: int = int(creation_date)
        r"""Point in time \(Unix timestamp\) when the giveaway was created"""
        self.status: Union[
            GiveawayParticipantStatusEligible,
            GiveawayParticipantStatusParticipating,
            GiveawayParticipantStatusAlreadyWasMember,
            GiveawayParticipantStatusAdministrator,
            GiveawayParticipantStatusDisallowedCountry,
            None,
        ] = status
        r"""Status of the current user in the giveaway"""
        self.is_ended: bool = bool(is_ended)
        r"""True, if the giveaway has ended and results are being prepared"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayInfoOngoing"]:
        return "giveawayInfoOngoing"

    @classmethod
    def getClass(self) -> Literal["GiveawayInfo"]:
        return "GiveawayInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "creation_date": self.creation_date,
            "status": self.status,
            "is_ended": self.is_ended,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiveawayInfoOngoing", None]:
        if data:
            data_class = cls()
            data_class.creation_date = int(data.get("creation_date", 0))
            data_class.status = data.get("status", None)
            data_class.is_ended = data.get("is_ended", False)

        return data_class


class GiveawayInfoCompleted(TlObject, GiveawayInfo):
    r"""Describes a completed giveaway

    Parameters:
        creation_date (:class:`int`):
            Point in time \(Unix timestamp\) when the giveaway was created

        actual_winners_selection_date (:class:`int`):
            Point in time \(Unix timestamp\) when the winners were selected\. May be bigger than winners selection date specified in parameters of the giveaway

        was_refunded (:class:`bool`):
            True, if the giveaway was canceled and was fully refunded

        is_winner (:class:`bool`):
            True, if the current user is a winner of the giveaway

        winner_count (:class:`int`):
            Number of winners in the giveaway

        activation_count (:class:`int`):
            Number of winners, which activated their gift codes; for Telegram Premium giveaways only

        gift_code (:class:`str`):
            Telegram Premium gift code that was received by the current user; empty if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Premium giveaway

        won_star_count (:class:`int`):
            The amount of Telegram Stars won by the current user; 0 if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Star giveaway

    """

    def __init__(
        self,
        creation_date: int = 0,
        actual_winners_selection_date: int = 0,
        was_refunded: bool = False,
        is_winner: bool = False,
        winner_count: int = 0,
        activation_count: int = 0,
        gift_code: str = "",
        won_star_count: int = 0,
    ) -> None:
        self.creation_date: int = int(creation_date)
        r"""Point in time \(Unix timestamp\) when the giveaway was created"""
        self.actual_winners_selection_date: int = int(actual_winners_selection_date)
        r"""Point in time \(Unix timestamp\) when the winners were selected\. May be bigger than winners selection date specified in parameters of the giveaway"""
        self.was_refunded: bool = bool(was_refunded)
        r"""True, if the giveaway was canceled and was fully refunded"""
        self.is_winner: bool = bool(is_winner)
        r"""True, if the current user is a winner of the giveaway"""
        self.winner_count: int = int(winner_count)
        r"""Number of winners in the giveaway"""
        self.activation_count: int = int(activation_count)
        r"""Number of winners, which activated their gift codes; for Telegram Premium giveaways only"""
        self.gift_code: Union[str, None] = gift_code
        r"""Telegram Premium gift code that was received by the current user; empty if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Premium giveaway"""
        self.won_star_count: int = int(won_star_count)
        r"""The amount of Telegram Stars won by the current user; 0 if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Star giveaway"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayInfoCompleted"]:
        return "giveawayInfoCompleted"

    @classmethod
    def getClass(self) -> Literal["GiveawayInfo"]:
        return "GiveawayInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "creation_date": self.creation_date,
            "actual_winners_selection_date": self.actual_winners_selection_date,
            "was_refunded": self.was_refunded,
            "is_winner": self.is_winner,
            "winner_count": self.winner_count,
            "activation_count": self.activation_count,
            "gift_code": self.gift_code,
            "won_star_count": self.won_star_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiveawayInfoCompleted", None]:
        if data:
            data_class = cls()
            data_class.creation_date = int(data.get("creation_date", 0))
            data_class.actual_winners_selection_date = int(
                data.get("actual_winners_selection_date", 0)
            )
            data_class.was_refunded = data.get("was_refunded", False)
            data_class.is_winner = data.get("is_winner", False)
            data_class.winner_count = int(data.get("winner_count", 0))
            data_class.activation_count = int(data.get("activation_count", 0))
            data_class.gift_code = data.get("gift_code", "")
            data_class.won_star_count = int(data.get("won_star_count", 0))

        return data_class


class GiveawayPrizePremium(TlObject, GiveawayPrize):
    r"""The giveaway sends Telegram Premium subscriptions to the winners

    Parameters:
        month_count (:class:`int`):
            Number of months the Telegram Premium subscription will be active after code activation

    """

    def __init__(self, month_count: int = 0) -> None:
        self.month_count: int = int(month_count)
        r"""Number of months the Telegram Premium subscription will be active after code activation"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayPrizePremium"]:
        return "giveawayPrizePremium"

    @classmethod
    def getClass(self) -> Literal["GiveawayPrize"]:
        return "GiveawayPrize"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "month_count": self.month_count}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiveawayPrizePremium", None]:
        if data:
            data_class = cls()
            data_class.month_count = int(data.get("month_count", 0))

        return data_class


class GiveawayPrizeStars(TlObject, GiveawayPrize):
    r"""The giveaway sends Telegram Stars to the winners

    Parameters:
        star_count (:class:`int`):
            Number of Telegram Stars that will be shared by all winners

    """

    def __init__(self, star_count: int = 0) -> None:
        self.star_count: int = int(star_count)
        r"""Number of Telegram Stars that will be shared by all winners"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["giveawayPrizeStars"]:
        return "giveawayPrizeStars"

    @classmethod
    def getClass(self) -> Literal["GiveawayPrize"]:
        return "GiveawayPrize"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "star_count": self.star_count}

    @classmethod
    def from_dict(cls, data: dict) -> Union["GiveawayPrizeStars", None]:
        if data:
            data_class = cls()
            data_class.star_count = int(data.get("star_count", 0))

        return data_class


class AccentColor(TlObject):
    r"""Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews

    Parameters:
        id (:class:`int`):
            Accent color identifier

        built_in_accent_color_id (:class:`int`):
            Identifier of a built\-in color to use in places, where only one color is needed; 0\-6

        light_theme_colors (List[:class:`int`]):
            The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in light themes

        dark_theme_colors (List[:class:`int`]):
            The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes

        min_channel_chat_boost_level (:class:`int`):
            The minimum chat boost level required to use the color in a channel chat

    """

    def __init__(
        self,
        id: int = 0,
        built_in_accent_color_id: int = 0,
        light_theme_colors: List[int] = None,
        dark_theme_colors: List[int] = None,
        min_channel_chat_boost_level: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Accent color identifier"""
        self.built_in_accent_color_id: int = int(built_in_accent_color_id)
        r"""Identifier of a built\-in color to use in places, where only one color is needed; 0\-6"""
        self.light_theme_colors: List[int] = light_theme_colors or []
        r"""The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in light themes"""
        self.dark_theme_colors: List[int] = dark_theme_colors or []
        r"""The list of 1\-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes"""
        self.min_channel_chat_boost_level: int = int(min_channel_chat_boost_level)
        r"""The minimum chat boost level required to use the color in a channel chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["accentColor"]:
        return "accentColor"

    @classmethod
    def getClass(self) -> Literal["AccentColor"]:
        return "AccentColor"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "built_in_accent_color_id": self.built_in_accent_color_id,
            "light_theme_colors": self.light_theme_colors,
            "dark_theme_colors": self.dark_theme_colors,
            "min_channel_chat_boost_level": self.min_channel_chat_boost_level,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AccentColor", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.built_in_accent_color_id = int(
                data.get("built_in_accent_color_id", 0)
            )
            data_class.light_theme_colors = data.get("light_theme_colors", None)
            data_class.dark_theme_colors = data.get("dark_theme_colors", None)
            data_class.min_channel_chat_boost_level = int(
                data.get("min_channel_chat_boost_level", 0)
            )

        return data_class


class ProfileAccentColors(TlObject):
    r"""Contains information about supported accent colors for user profile photo background in RGB format

    Parameters:
        palette_colors (List[:class:`int`]):
            The list of 1\-2 colors in RGB format, describing the colors, as expected to be shown in the color palette settings

        background_colors (List[:class:`int`]):
            The list of 1\-2 colors in RGB format, describing the colors, as expected to be used for the profile photo background

        story_colors (List[:class:`int`]):
            The list of 2 colors in RGB format, describing the colors of the gradient to be used for the unread active story indicator around profile photo

    """

    def __init__(
        self,
        palette_colors: List[int] = None,
        background_colors: List[int] = None,
        story_colors: List[int] = None,
    ) -> None:
        self.palette_colors: List[int] = palette_colors or []
        r"""The list of 1\-2 colors in RGB format, describing the colors, as expected to be shown in the color palette settings"""
        self.background_colors: List[int] = background_colors or []
        r"""The list of 1\-2 colors in RGB format, describing the colors, as expected to be used for the profile photo background"""
        self.story_colors: List[int] = story_colors or []
        r"""The list of 2 colors in RGB format, describing the colors of the gradient to be used for the unread active story indicator around profile photo"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileAccentColors"]:
        return "profileAccentColors"

    @classmethod
    def getClass(self) -> Literal["ProfileAccentColors"]:
        return "ProfileAccentColors"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "palette_colors": self.palette_colors,
            "background_colors": self.background_colors,
            "story_colors": self.story_colors,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileAccentColors", None]:
        if data:
            data_class = cls()
            data_class.palette_colors = data.get("palette_colors", None)
            data_class.background_colors = data.get("background_colors", None)
            data_class.story_colors = data.get("story_colors", None)

        return data_class


class ProfileAccentColor(TlObject):
    r"""Contains information about supported accent color for user profile photo background

    Parameters:
        id (:class:`int`):
            Profile accent color identifier

        light_theme_colors (:class:`~pytdbot.types.ProfileAccentColors`):
            Accent colors expected to be used in light themes

        dark_theme_colors (:class:`~pytdbot.types.ProfileAccentColors`):
            Accent colors expected to be used in dark themes

        min_supergroup_chat_boost_level (:class:`int`):
            The minimum chat boost level required to use the color in a supergroup chat

        min_channel_chat_boost_level (:class:`int`):
            The minimum chat boost level required to use the color in a channel chat

    """

    def __init__(
        self,
        id: int = 0,
        light_theme_colors: ProfileAccentColors = None,
        dark_theme_colors: ProfileAccentColors = None,
        min_supergroup_chat_boost_level: int = 0,
        min_channel_chat_boost_level: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Profile accent color identifier"""
        self.light_theme_colors: Union[ProfileAccentColors, None] = light_theme_colors
        r"""Accent colors expected to be used in light themes"""
        self.dark_theme_colors: Union[ProfileAccentColors, None] = dark_theme_colors
        r"""Accent colors expected to be used in dark themes"""
        self.min_supergroup_chat_boost_level: int = int(min_supergroup_chat_boost_level)
        r"""The minimum chat boost level required to use the color in a supergroup chat"""
        self.min_channel_chat_boost_level: int = int(min_channel_chat_boost_level)
        r"""The minimum chat boost level required to use the color in a channel chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["profileAccentColor"]:
        return "profileAccentColor"

    @classmethod
    def getClass(self) -> Literal["ProfileAccentColor"]:
        return "ProfileAccentColor"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "light_theme_colors": self.light_theme_colors,
            "dark_theme_colors": self.dark_theme_colors,
            "min_supergroup_chat_boost_level": self.min_supergroup_chat_boost_level,
            "min_channel_chat_boost_level": self.min_channel_chat_boost_level,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ProfileAccentColor", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.light_theme_colors = data.get("light_theme_colors", None)
            data_class.dark_theme_colors = data.get("dark_theme_colors", None)
            data_class.min_supergroup_chat_boost_level = int(
                data.get("min_supergroup_chat_boost_level", 0)
            )
            data_class.min_channel_chat_boost_level = int(
                data.get("min_channel_chat_boost_level", 0)
            )

        return data_class


class UserRating(TlObject):
    r"""Contains description of user rating

    Parameters:
        level (:class:`int`):
            The level of the user; may be negative

        is_maximum_level_reached (:class:`bool`):
            True, if the maximum level is reached

        rating (:class:`int`):
            Numerical value of the rating

        current_level_rating (:class:`int`):
            The rating required for the current level

        next_level_rating (:class:`int`):
            The rating required for the next level; 0 if the maximum level is reached

    """

    def __init__(
        self,
        level: int = 0,
        is_maximum_level_reached: bool = False,
        rating: int = 0,
        current_level_rating: int = 0,
        next_level_rating: int = 0,
    ) -> None:
        self.level: int = int(level)
        r"""The level of the user; may be negative"""
        self.is_maximum_level_reached: bool = bool(is_maximum_level_reached)
        r"""True, if the maximum level is reached"""
        self.rating: int = int(rating)
        r"""Numerical value of the rating"""
        self.current_level_rating: int = int(current_level_rating)
        r"""The rating required for the current level"""
        self.next_level_rating: int = int(next_level_rating)
        r"""The rating required for the next level; 0 if the maximum level is reached"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["userRating"]:
        return "userRating"

    @classmethod
    def getClass(self) -> Literal["UserRating"]:
        return "UserRating"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "level": self.level,
            "is_maximum_level_reached": self.is_maximum_level_reached,
            "rating": self.rating,
            "current_level_rating": self.current_level_rating,
            "next_level_rating": self.next_level_rating,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UserRating", None]:
        if data:
            data_class = cls()
            data_class.level = int(data.get("level", 0))
            data_class.is_maximum_level_reached = data.get(
                "is_maximum_level_reached", False
            )
            data_class.rating = int(data.get("rating", 0))
            data_class.current_level_rating = int(data.get("current_level_rating", 0))
            data_class.next_level_rating = int(data.get("next_level_rating", 0))

        return data_class


class RestrictionInfo(TlObject):
    r"""Contains information about restrictions that must be applied to a chat or a message

    Parameters:
        restriction_reason (:class:`str`):
            A human\-readable description of the reason why access to the content must be restricted\. If empty, then the content can be accessed, but may be covered by hidden with 18\+ spoiler anyway

        has_sensitive_content (:class:`bool`):
            True, if media content of the messages must be hidden with 18\+ spoiler\. Use value of the option \"can\_ignore\_sensitive\_content\_restrictions\" to check whether the current user can ignore the restriction\. If age verification parameters were received in updateAgeVerificationParameters, then the user must complete age verification to ignore the restriction\. Set the option \"ignore\_sensitive\_content\_restrictions\" to true if the user passes age verification

    """

    def __init__(
        self, restriction_reason: str = "", has_sensitive_content: bool = False
    ) -> None:
        self.restriction_reason: Union[str, None] = restriction_reason
        r"""A human\-readable description of the reason why access to the content must be restricted\. If empty, then the content can be accessed, but may be covered by hidden with 18\+ spoiler anyway"""
        self.has_sensitive_content: bool = bool(has_sensitive_content)
        r"""True, if media content of the messages must be hidden with 18\+ spoiler\. Use value of the option \"can\_ignore\_sensitive\_content\_restrictions\" to check whether the current user can ignore the restriction\. If age verification parameters were received in updateAgeVerificationParameters, then the user must complete age verification to ignore the restriction\. Set the option \"ignore\_sensitive\_content\_restrictions\" to true if the user passes age verification"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["restrictionInfo"]:
        return "restrictionInfo"

    @classmethod
    def getClass(self) -> Literal["RestrictionInfo"]:
        return "RestrictionInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "restriction_reason": self.restriction_reason,
            "has_sensitive_content": self.has_sensitive_content,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["RestrictionInfo", None]:
        if data:
            data_class = cls()
            data_class.restriction_reason = data.get("restriction_reason", "")
            data_class.has_sensitive_content = data.get("has_sensitive_content", False)

        return data_class


class EmojiStatusTypeCustomEmoji(TlObject, EmojiStatusType):
    r"""A custom emoji set as emoji status

    Parameters:
        custom_emoji_id (:class:`int`):
            Identifier of the custom emoji in stickerFormatTgs format

    """

    def __init__(self, custom_emoji_id: int = 0) -> None:
        self.custom_emoji_id: int = int(custom_emoji_id)
        r"""Identifier of the custom emoji in stickerFormatTgs format"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emojiStatusTypeCustomEmoji"]:
        return "emojiStatusTypeCustomEmoji"

    @classmethod
    def getClass(self) -> Literal["EmojiStatusType"]:
        return "EmojiStatusType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "custom_emoji_id": self.custom_emoji_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmojiStatusTypeCustomEmoji", None]:
        if data:
            data_class = cls()
            data_class.custom_emoji_id = int(data.get("custom_emoji_id", 0))

        return data_class


class EmojiStatusTypeUpgradedGift(TlObject, EmojiStatusType):
    r"""An upgraded gift set as emoji status

    Parameters:
        upgraded_gift_id (:class:`int`):
            Identifier of the upgraded gift

        gift_title (:class:`str`):
            The title of the upgraded gift

        gift_name (:class:`str`):
            Unique name of the upgraded gift that can be used with internalLinkTypeUpgradedGift

        model_custom_emoji_id (:class:`int`):
            Custom emoji identifier of the model of the upgraded gift

        symbol_custom_emoji_id (:class:`int`):
            Custom emoji identifier of the symbol of the upgraded gift

        backdrop_colors (:class:`~pytdbot.types.UpgradedGiftBackdropColors`):
            Colors of the backdrop of the upgraded gift

    """

    def __init__(
        self,
        upgraded_gift_id: int = 0,
        gift_title: str = "",
        gift_name: str = "",
        model_custom_emoji_id: int = 0,
        symbol_custom_emoji_id: int = 0,
        backdrop_colors: UpgradedGiftBackdropColors = None,
    ) -> None:
        self.upgraded_gift_id: int = int(upgraded_gift_id)
        r"""Identifier of the upgraded gift"""
        self.gift_title: Union[str, None] = gift_title
        r"""The title of the upgraded gift"""
        self.gift_name: Union[str, None] = gift_name
        r"""Unique name of the upgraded gift that can be used with internalLinkTypeUpgradedGift"""
        self.model_custom_emoji_id: int = int(model_custom_emoji_id)
        r"""Custom emoji identifier of the model of the upgraded gift"""
        self.symbol_custom_emoji_id: int = int(symbol_custom_emoji_id)
        r"""Custom emoji identifier of the symbol of the upgraded gift"""
        self.backdrop_colors: Union[UpgradedGiftBackdropColors, None] = backdrop_colors
        r"""Colors of the backdrop of the upgraded gift"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emojiStatusTypeUpgradedGift"]:
        return "emojiStatusTypeUpgradedGift"

    @classmethod
    def getClass(self) -> Literal["EmojiStatusType"]:
        return "EmojiStatusType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "upgraded_gift_id": self.upgraded_gift_id,
            "gift_title": self.gift_title,
            "gift_name": self.gift_name,
            "model_custom_emoji_id": self.model_custom_emoji_id,
            "symbol_custom_emoji_id": self.symbol_custom_emoji_id,
            "backdrop_colors": self.backdrop_colors,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmojiStatusTypeUpgradedGift", None]:
        if data:
            data_class = cls()
            data_class.upgraded_gift_id = int(data.get("upgraded_gift_id", 0))
            data_class.gift_title = data.get("gift_title", "")
            data_class.gift_name = data.get("gift_name", "")
            data_class.model_custom_emoji_id = int(data.get("model_custom_emoji_id", 0))
            data_class.symbol_custom_emoji_id = int(
                data.get("symbol_custom_emoji_id", 0)
            )
            data_class.backdrop_colors = data.get("backdrop_colors", None)

        return data_class


class EmojiStatus(TlObject):
    r"""Describes an emoji to be shown instead of the Telegram Premium badge

    Parameters:
        type (:class:`~pytdbot.types.EmojiStatusType`):
            Type of the emoji status

        expiration_date (:class:`int`):
            Point in time \(Unix timestamp\) when the status will expire; 0 if never

    """

    def __init__(self, type: EmojiStatusType = None, expiration_date: int = 0) -> None:
        self.type: Union[
            EmojiStatusTypeCustomEmoji, EmojiStatusTypeUpgradedGift, None
        ] = type
        r"""Type of the emoji status"""
        self.expiration_date: int = int(expiration_date)
        r"""Point in time \(Unix timestamp\) when the status will expire; 0 if never"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emojiStatus"]:
        return "emojiStatus"

    @classmethod
    def getClass(self) -> Literal["EmojiStatus"]:
        return "EmojiStatus"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "type": self.type,
            "expiration_date": self.expiration_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmojiStatus", None]:
        if data:
            data_class = cls()
            data_class.type = data.get("type", None)
            data_class.expiration_date = int(data.get("expiration_date", 0))

        return data_class


class EmojiStatuses(TlObject):
    r"""Contains a list of emoji statuses

    Parameters:
        emoji_statuses (List[:class:`~pytdbot.types.EmojiStatus`]):
            The list of emoji statuses identifiers

    """

    def __init__(self, emoji_statuses: List[EmojiStatus] = None) -> None:
        self.emoji_statuses: List[EmojiStatus] = emoji_statuses or []
        r"""The list of emoji statuses identifiers"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emojiStatuses"]:
        return "emojiStatuses"

    @classmethod
    def getClass(self) -> Literal["EmojiStatuses"]:
        return "EmojiStatuses"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "emoji_statuses": self.emoji_statuses}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmojiStatuses", None]:
        if data:
            data_class = cls()
            data_class.emoji_statuses = data.get("emoji_statuses", None)

        return data_class


class EmojiStatusCustomEmojis(TlObject):
    r"""Contains a list of custom emoji identifiers for emoji statuses

    Parameters:
        custom_emoji_ids (List[:class:`int`]):
            The list of custom emoji identifiers

    """

    def __init__(self, custom_emoji_ids: List[int] = None) -> None:
        self.custom_emoji_ids: List[int] = custom_emoji_ids or []
        r"""The list of custom emoji identifiers"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["emojiStatusCustomEmojis"]:
        return "emojiStatusCustomEmojis"

    @classmethod
    def getClass(self) -> Literal["EmojiStatusCustomEmojis"]:
        return "EmojiStatusCustomEmojis"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "custom_emoji_ids": self.custom_emoji_ids}

    @classmethod
    def from_dict(cls, data: dict) -> Union["EmojiStatusCustomEmojis", None]:
        if data:
            data_class = cls()
            data_class.custom_emoji_ids = data.get("custom_emoji_ids", None)

        return data_class


class Usernames(TlObject):
    r"""Describes usernames assigned to a user, a supergroup, or a channel

    Parameters:
        active_usernames (List[:class:`str`]):
            List of active usernames; the first one must be shown as the primary username\. The order of active usernames can be changed with reorderActiveUsernames, reorderBotActiveUsernames or reorderSupergroupActiveUsernames

        disabled_usernames (List[:class:`str`]):
            List of currently disabled usernames; the username can be activated with toggleUsernameIsActive, toggleBotUsernameIsActive, or toggleSupergroupUsernameIsActive

        editable_username (:class:`str`):
            Active or disabled username, which may be changed with setUsername or setSupergroupUsername\. Information about other active usernames can be received using getCollectibleItemInfo

    """

    def __init__(
        self,
        active_usernames: List[str] = None,
        disabled_usernames: List[str] = None,
        editable_username: str = "",
    ) -> None:
        self.active_usernames: List[str] = active_usernames or []
        r"""List of active usernames; the first one must be shown as the primary username\. The order of active usernames can be changed with reorderActiveUsernames, reorderBotActiveUsernames or reorderSupergroupActiveUsernames"""
        self.disabled_usernames: List[str] = disabled_usernames or []
        r"""List of currently disabled usernames; the username can be activated with toggleUsernameIsActive, toggleBotUsernameIsActive, or toggleSupergroupUsernameIsActive"""
        self.editable_username: Union[str, None] = editable_username
        r"""Active or disabled username, which may be changed with setUsername or setSupergroupUsername\. Information about other active usernames can be received using getCollectibleItemInfo"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["usernames"]:
        return "usernames"

    @classmethod
    def getClass(self) -> Literal["Usernames"]:
        return "Usernames"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "active_usernames": self.active_usernames,
            "disabled_usernames": self.disabled_usernames,
            "editable_username": self.editable_username,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Usernames", None]:
        if data:
            data_class = cls()
            data_class.active_usernames = data.get("active_usernames", None)
            data_class.disabled_usernames = data.get("disabled_usernames", None)
            data_class.editable_username = data.get("editable_username", "")

        return data_class


class User(TlObject):
    r"""Represents a user

    Parameters:
        id (:class:`int`):
            User identifier

        first_name (:class:`str`):
            First name of the user

        last_name (:class:`str`):
            Last name of the user

        usernames (:class:`~pytdbot.types.Usernames`):
            Usernames of the user; may be null

        phone_number (:class:`str`):
            Phone number of the user

        status (:class:`~pytdbot.types.UserStatus`):
            Current online status of the user

        profile_photo (:class:`~pytdbot.types.ProfilePhoto`):
            Profile photo of the user; may be null

        accent_color_id (:class:`int`):
            Identifier of the accent color for name, and backgrounds of profile photo, reply header, and link preview

        background_custom_emoji_id (:class:`int`):
            Identifier of a custom emoji to be shown on the reply header and link preview background; 0 if none

        upgraded_gift_colors (:class:`~pytdbot.types.UpgradedGiftColors`):
            Color scheme based on an upgraded gift to be used for the user instead of accent\_color\_id and background\_custom\_emoji\_id; may be null if none

        profile_accent_color_id (:class:`int`):
            Identifier of the accent color for the user's profile; \-1 if none

        profile_background_custom_emoji_id (:class:`int`):
            Identifier of a custom emoji to be shown on the background of the user's profile; 0 if none

        emoji_status (:class:`~pytdbot.types.EmojiStatus`):
            Emoji status to be shown instead of the default Telegram Premium badge; may be null

        is_contact (:class:`bool`):
            The user is a contact of the current user

        is_mutual_contact (:class:`bool`):
            The user is a contact of the current user and the current user is a contact of the user

        is_close_friend (:class:`bool`):
            The user is a close friend of the current user; implies that the user is a contact

        verification_status (:class:`~pytdbot.types.VerificationStatus`):
            Information about verification status of the user; may be null if none

        is_premium (:class:`bool`):
            True, if the user is a Telegram Premium user

        is_support (:class:`bool`):
            True, if the user is Telegram support account

        restriction_info (:class:`~pytdbot.types.RestrictionInfo`):
            Information about restrictions that must be applied to the corresponding private chat; may be null if none

        has_active_stories (:class:`bool`):
            True, if the user has non\-expired stories available to the current user

        has_unread_active_stories (:class:`bool`):
            True, if the user has unread non\-expired stories available to the current user

        restricts_new_chats (:class:`bool`):
            True, if the user may restrict new chats with non\-contacts\. Use canSendMessageToUser to check whether the current user can message the user or try to create a chat with them

        paid_message_star_count (:class:`int`):
            Number of Telegram Stars that must be paid by general user for each sent message to the user\. If positive and userFullInfo is unknown, use canSendMessageToUser to check whether the current user must pay

        have_access (:class:`bool`):
            If false, the user is inaccessible, and the only information known about the user is inside this class\. Identifier of the user can't be passed to any method

        type (:class:`~pytdbot.types.UserType`):
            Type of the user

        language_code (:class:`str`):
            IETF language tag of the user's language; only available to bots

        added_to_attachment_menu (:class:`bool`):
            True, if the user added the current bot to attachment menu; only available to bots

    """

    def __init__(
        self,
        id: int = 0,
        first_name: str = "",
        last_name: str = "",
        usernames: Usernames = None,
        phone_number: str = "",
        status: UserStatus = None,
        profile_photo: ProfilePhoto = None,
        accent_color_id: int = 0,
        background_custom_emoji_id: int = 0,
        upgraded_gift_colors: UpgradedGiftColors = None,
        profile_accent_color_id: int = 0,
        profile_background_custom_emoji_id: int = 0,
        emoji_status: EmojiStatus = None,
        is_contact: bool = False,
        is_mutual_contact: bool = False,
        is_close_friend: bool = False,
        verification_status: VerificationStatus = None,
        is_premium: bool = False,
        is_support: bool = False,
        restriction_info: RestrictionInfo = None,
        has_active_stories: bool = False,
        has_unread_active_stories: bool = False,
        restricts_new_chats: bool = False,
        paid_message_star_count: int = 0,
        have_access: bool = False,
        type: UserType = None,
        language_code: str = "",
        added_to_attachment_menu: bool = False,
    ) -> None:
        self.id: int = int(id)
        r"""User identifier"""
        self.first_name: Union[str, None] = first_name
        r"""First name of the user"""
        self.last_name: Union[str, None] = last_name
        r"""Last name of the user"""
        self.usernames: Union[Usernames, None] = usernames
        r"""Usernames of the user; may be null"""
        self.phone_number: Union[str, None] = phone_number
        r"""Phone number of the user"""
        self.status: Union[
            UserStatusEmpty,
            UserStatusOnline,
            UserStatusOffline,
            UserStatusRecently,
            UserStatusLastWeek,
            UserStatusLastMonth,
            None,
        ] = status
        r"""Current online status of the user"""
        self.profile_photo: Union[ProfilePhoto, None] = profile_photo
        r"""Profile photo of the user; may be null"""
        self.accent_color_id: int = int(accent_color_id)
        r"""Identifier of the accent color for name, and backgrounds of profile photo, reply header, and link preview"""
        self.background_custom_emoji_id: int = int(background_custom_emoji_id)
        r"""Identifier of a custom emoji to be shown on the reply header and link preview background; 0 if none"""
        self.upgraded_gift_colors: Union[UpgradedGiftColors, None] = (
            upgraded_gift_colors
        )
        r"""Color scheme based on an upgraded gift to be used for the user instead of accent\_color\_id and background\_custom\_emoji\_id; may be null if none"""
        self.profile_accent_color_id: int = int(profile_accent_color_id)
        r"""Identifier of the accent color for the user's profile; \-1 if none"""
        self.profile_background_custom_emoji_id: int = int(
            profile_background_custom_emoji_id
        )
        r"""Identifier of a custom emoji to be shown on the background of the user's profile; 0 if none"""
        self.emoji_status: Union[EmojiStatus, None] = emoji_status
        r"""Emoji status to be shown instead of the default Telegram Premium badge; may be null"""
        self.is_contact: bool = bool(is_contact)
        r"""The user is a contact of the current user"""
        self.is_mutual_contact: bool = bool(is_mutual_contact)
        r"""The user is a contact of the current user and the current user is a contact of the user"""
        self.is_close_friend: bool = bool(is_close_friend)
        r"""The user is a close friend of the current user; implies that the user is a contact"""
        self.verification_status: Union[VerificationStatus, None] = verification_status
        r"""Information about verification status of the user; may be null if none"""
        self.is_premium: bool = bool(is_premium)
        r"""True, if the user is a Telegram Premium user"""
        self.is_support: bool = bool(is_support)
        r"""True, if the user is Telegram support account"""
        self.restriction_info: Union[RestrictionInfo, None] = restriction_info
        r"""Information about restrictions that must be applied to the corresponding private chat; may be null if none"""
        self.has_active_stories: bool = bool(has_active_stories)
        r"""True, if the user has non\-expired stories available to the current user"""
        self.has_unread_active_stories: bool = bool(has_unread_active_stories)
        r"""True, if the user has unread non\-expired stories available to the current user"""
        self.restricts_new_chats: bool = bool(restricts_new_chats)
        r"""True, if the user may restrict new chats with non\-contacts\. Use canSendMessageToUser to check whether the current user can message the user or try to create a chat with them"""
        self.paid_message_star_count: int = int(paid_message_star_count)
        r"""Number of Telegram Stars that must be paid by general user for each sent message to the user\. If positive and userFullInfo is unknown, use canSendMessageToUser to check whether the current user must pay"""
        self.have_access: bool = bool(have_access)
        r"""If false, the user is inaccessible, and the only information known about the user is inside this class\. Identifier of the user can't be passed to any method"""
        self.type: Union[
            UserTypeRegular, UserTypeDeleted, UserTypeBot, UserTypeUnknown, None
        ] = type
        r"""Type of the user"""
        self.language_code: Union[str, None] = language_code
        r"""IETF language tag of the user's language; only available to bots"""
        self.added_to_attachment_menu: bool = bool(added_to_attachment_menu)
        r"""True, if the user added the current bot to attachment menu; only available to bots"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["user"]:
        return "user"

    @classmethod
    def getClass(self) -> Literal["User"]:
        return "User"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "first_name": self.first_name,
            "last_name": self.last_name,
            "usernames": self.usernames,
            "phone_number": self.phone_number,
            "status": self.status,
            "profile_photo": self.profile_photo,
            "accent_color_id": self.accent_color_id,
            "background_custom_emoji_id": self.background_custom_emoji_id,
            "upgraded_gift_colors": self.upgraded_gift_colors,
            "profile_accent_color_id": self.profile_accent_color_id,
            "profile_background_custom_emoji_id": self.profile_background_custom_emoji_id,
            "emoji_status": self.emoji_status,
            "is_contact": self.is_contact,
            "is_mutual_contact": self.is_mutual_contact,
            "is_close_friend": self.is_close_friend,
            "verification_status": self.verification_status,
            "is_premium": self.is_premium,
            "is_support": self.is_support,
            "restriction_info": self.restriction_info,
            "has_active_stories": self.has_active_stories,
            "has_unread_active_stories": self.has_unread_active_stories,
            "restricts_new_chats": self.restricts_new_chats,
            "paid_message_star_count": self.paid_message_star_count,
            "have_access": self.have_access,
            "type": self.type,
            "language_code": self.language_code,
            "added_to_attachment_menu": self.added_to_attachment_menu,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["User", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.first_name = data.get("first_name", "")
            data_class.last_name = data.get("last_name", "")
            data_class.usernames = data.get("usernames", None)
            data_class.phone_number = data.get("phone_number", "")
            data_class.status = data.get("status", None)
            data_class.profile_photo = data.get("profile_photo", None)
            data_class.accent_color_id = int(data.get("accent_color_id", 0))
            data_class.background_custom_emoji_id = int(
                data.get("background_custom_emoji_id", 0)
            )
            data_class.upgraded_gift_colors = data.get("upgraded_gift_colors", None)
            data_class.profile_accent_color_id = int(
                data.get("profile_accent_color_id", 0)
            )
            data_class.profile_background_custom_emoji_id = int(
                data.get("profile_background_custom_emoji_id", 0)
            )
            data_class.emoji_status = data.get("emoji_status", None)
            data_class.is_contact = data.get("is_contact", False)
            data_class.is_mutual_contact = data.get("is_mutual_contact", False)
            data_class.is_close_friend = data.get("is_close_friend", False)
            data_class.verification_status = data.get("verification_status", None)
            data_class.is_premium = data.get("is_premium", False)
            data_class.is_support = data.get("is_support", False)
            data_class.restriction_info = data.get("restriction_info", None)
            data_class.has_active_stories = data.get("has_active_stories", False)
            data_class.has_unread_active_stories = data.get(
                "has_unread_active_stories", False
            )
            data_class.restricts_new_chats = data.get("restricts_new_chats", False)
            data_class.paid_message_star_count = int(
                data.get("paid_message_star_count", 0)
            )
            data_class.have_access = data.get("have_access", False)
            data_class.type = data.get("type", None)
            data_class.language_code = data.get("language_code", "")
            data_class.added_to_attachment_menu = data.get(
                "added_to_attachment_menu", False
            )

        return data_class


class BotInfo(TlObject):
    r"""Contains information about a bot

    Parameters:
        short_description (:class:`str`):
            The text that is shown on the bot's profile page and is sent together with the link when users share the bot

        description (:class:`str`):
            The text shown in the chat with the bot if the chat is empty

        photo (:class:`~pytdbot.types.Photo`):
            Photo shown in the chat with the bot if the chat is empty; may be null

        animation (:class:`~pytdbot.types.Animation`):
            Animation shown in the chat with the bot if the chat is empty; may be null

        menu_button (:class:`~pytdbot.types.BotMenuButton`):
            Information about a button to show instead of the bot commands menu button; may be null if ordinary bot commands menu must be shown

        commands (List[:class:`~pytdbot.types.BotCommand`]):
            List of the bot commands

        privacy_policy_url (:class:`str`):
            The HTTP link to the privacy policy of the bot\. If empty, then /privacy command must be used if supported by the bot\. If the command isn't supported, then https://telegram\.org/privacy\-tpa must be opened

        default_group_administrator_rights (:class:`~pytdbot.types.ChatAdministratorRights`):
            Default administrator rights for adding the bot to basic group and supergroup chats; may be null

        default_channel_administrator_rights (:class:`~pytdbot.types.ChatAdministratorRights`):
            Default administrator rights for adding the bot to channels; may be null

        affiliate_program (:class:`~pytdbot.types.AffiliateProgramInfo`):
            Information about the affiliate program of the bot; may be null if none

        web_app_background_light_color (:class:`int`):
            Default light background color for bot Web Apps; \-1 if not specified

        web_app_background_dark_color (:class:`int`):
            Default dark background color for bot Web Apps; \-1 if not specified

        web_app_header_light_color (:class:`int`):
            Default light header color for bot Web Apps; \-1 if not specified

        web_app_header_dark_color (:class:`int`):
            Default dark header color for bot Web Apps; \-1 if not specified

        verification_parameters (:class:`~pytdbot.types.BotVerificationParameters`):
            Parameters of the verification that can be provided by the bot; may be null if none or the current user isn't the owner of the bot

        can_get_revenue_statistics (:class:`bool`):
            True, if the bot's revenue statistics are available to the current user

        can_manage_emoji_status (:class:`bool`):
            True, if the bot can manage emoji status of the current user

        has_media_previews (:class:`bool`):
            True, if the bot has media previews

        edit_commands_link (:class:`~pytdbot.types.InternalLinkType`):
            The internal link, which can be used to edit bot commands; may be null

        edit_description_link (:class:`~pytdbot.types.InternalLinkType`):
            The internal link, which can be used to edit bot description; may be null

        edit_description_media_link (:class:`~pytdbot.types.InternalLinkType`):
            The internal link, which can be used to edit the photo or animation shown in the chat with the bot if the chat is empty; may be null

        edit_settings_link (:class:`~pytdbot.types.InternalLinkType`):
            The internal link, which can be used to edit bot settings; may be null

    """

    def __init__(
        self,
        short_description: str = "",
        description: str = "",
        photo: Photo = None,
        animation: Animation = None,
        menu_button: BotMenuButton = None,
        commands: List[BotCommand] = None,
        privacy_policy_url: str = "",
        default_group_administrator_rights: ChatAdministratorRights = None,
        default_channel_administrator_rights: ChatAdministratorRights = None,
        affiliate_program: AffiliateProgramInfo = None,
        web_app_background_light_color: int = 0,
        web_app_background_dark_color: int = 0,
        web_app_header_light_color: int = 0,
        web_app_header_dark_color: int = 0,
        verification_parameters: BotVerificationParameters = None,
        can_get_revenue_statistics: bool = False,
        can_manage_emoji_status: bool = False,
        has_media_previews: bool = False,
        edit_commands_link: InternalLinkType = None,
        edit_description_link: InternalLinkType = None,
        edit_description_media_link: InternalLinkType = None,
        edit_settings_link: InternalLinkType = None,
    ) -> None:
        self.short_description: Union[str, None] = short_description
        r"""The text that is shown on the bot's profile page and is sent together with the link when users share the bot"""
        self.description: Union[str, None] = description
        r"""The text shown in the chat with the bot if the chat is empty"""
        self.photo: Union[Photo, None] = photo
        r"""Photo shown in the chat with the bot if the chat is empty; may be null"""
        self.animation: Union[Animation, None] = animation
        r"""Animation shown in the chat with the bot if the chat is empty; may be null"""
        self.menu_button: Union[BotMenuButton, None] = menu_button
        r"""Information about a button to show instead of the bot commands menu button; may be null if ordinary bot commands menu must be shown"""
        self.commands: List[BotCommand] = commands or []
        r"""List of the bot commands"""
        self.privacy_policy_url: Union[str, None] = privacy_policy_url
        r"""The HTTP link to the privacy policy of the bot\. If empty, then /privacy command must be used if supported by the bot\. If the command isn't supported, then https://telegram\.org/privacy\-tpa must be opened"""
        self.default_group_administrator_rights: Union[
            ChatAdministratorRights, None
        ] = default_group_administrator_rights
        r"""Default administrator rights for adding the bot to basic group and supergroup chats; may be null"""
        self.default_channel_administrator_rights: Union[
            ChatAdministratorRights, None
        ] = default_channel_administrator_rights
        r"""Default administrator rights for adding the bot to channels; may be null"""
        self.affiliate_program: Union[AffiliateProgramInfo, None] = affiliate_program
        r"""Information about the affiliate program of the bot; may be null if none"""
        self.web_app_background_light_color: int = int(web_app_background_light_color)
        r"""Default light background color for bot Web Apps; \-1 if not specified"""
        self.web_app_background_dark_color: int = int(web_app_background_dark_color)
        r"""Default dark background color for bot Web Apps; \-1 if not specified"""
        self.web_app_header_light_color: int = int(web_app_header_light_color)
        r"""Default light header color for bot Web Apps; \-1 if not specified"""
        self.web_app_header_dark_color: int = int(web_app_header_dark_color)
        r"""Default dark header color for bot Web Apps; \-1 if not specified"""
        self.verification_parameters: Union[BotVerificationParameters, None] = (
            verification_parameters
        )
        r"""Parameters of the verification that can be provided by the bot; may be null if none or the current user isn't the owner of the bot"""
        self.can_get_revenue_statistics: bool = bool(can_get_revenue_statistics)
        r"""True, if the bot's revenue statistics are available to the current user"""
        self.can_manage_emoji_status: bool = bool(can_manage_emoji_status)
        r"""True, if the bot can manage emoji status of the current user"""
        self.has_media_previews: bool = bool(has_media_previews)
        r"""True, if the bot has media previews"""
        self.edit_commands_link: Union[
            InternalLinkTypeActiveSessions,
            InternalLinkTypeAttachmentMenuBot,
            InternalLinkTypeAuthenticationCode,
            InternalLinkTypeBackground,
            InternalLinkTypeBotAddToChannel,
            InternalLinkTypeBotStart,
            InternalLinkTypeBotStartInGroup,
            InternalLinkTypeBusinessChat,
            InternalLinkTypeBuyStars,
            InternalLinkTypeChangePhoneNumber,
            InternalLinkTypeChatAffiliateProgram,
            InternalLinkTypeChatBoost,
            InternalLinkTypeChatFolderInvite,
            InternalLinkTypeChatFolderSettings,
            InternalLinkTypeChatInvite,
            InternalLinkTypeDefaultMessageAutoDeleteTimerSettings,
            InternalLinkTypeDirectMessagesChat,
            InternalLinkTypeEditProfileSettings,
            InternalLinkTypeGame,
            InternalLinkTypeGiftCollection,
            InternalLinkTypeGroupCall,
            InternalLinkTypeInstantView,
            InternalLinkTypeInvoice,
            InternalLinkTypeLanguagePack,
            InternalLinkTypeLanguageSettings,
            InternalLinkTypeMainWebApp,
            InternalLinkTypeMessage,
            InternalLinkTypeMessageDraft,
            InternalLinkTypeMyStars,
            InternalLinkTypeMyToncoins,
            InternalLinkTypePassportDataRequest,
            InternalLinkTypePhoneNumberConfirmation,
            InternalLinkTypePremiumFeatures,
            InternalLinkTypePremiumGift,
            InternalLinkTypePremiumGiftCode,
            InternalLinkTypePrivacyAndSecuritySettings,
            InternalLinkTypeProxy,
            InternalLinkTypePublicChat,
            InternalLinkTypeQrCodeAuthentication,
            InternalLinkTypeRestorePurchases,
            InternalLinkTypeSettings,
            InternalLinkTypeStickerSet,
            InternalLinkTypeStory,
            InternalLinkTypeStoryAlbum,
            InternalLinkTypeTheme,
            InternalLinkTypeThemeSettings,
            InternalLinkTypeUnknownDeepLink,
            InternalLinkTypeUnsupportedProxy,
            InternalLinkTypeUpgradedGift,
            InternalLinkTypeUserPhoneNumber,
            InternalLinkTypeUserToken,
            InternalLinkTypeVideoChat,
            InternalLinkTypeWebApp,
            None,
        ] = edit_commands_link
        r"""The internal link, which can be used to edit bot commands; may be null"""
        self.edit_description_link: Union[
            InternalLinkTypeActiveSessions,
            InternalLinkTypeAttachmentMenuBot,
            InternalLinkTypeAuthenticationCode,
            InternalLinkTypeBackground,
            InternalLinkTypeBotAddToChannel,
            InternalLinkTypeBotStart,
            InternalLinkTypeBotStartInGroup,
            InternalLinkTypeBusinessChat,
            InternalLinkTypeBuyStars,
            InternalLinkTypeChangePhoneNumber,
            InternalLinkTypeChatAffiliateProgram,
            InternalLinkTypeChatBoost,
            InternalLinkTypeChatFolderInvite,
            InternalLinkTypeChatFolderSettings,
            InternalLinkTypeChatInvite,
            InternalLinkTypeDefaultMessageAutoDeleteTimerSettings,
            InternalLinkTypeDirectMessagesChat,
            InternalLinkTypeEditProfileSettings,
            InternalLinkTypeGame,
            InternalLinkTypeGiftCollection,
            InternalLinkTypeGroupCall,
            InternalLinkTypeInstantView,
            InternalLinkTypeInvoice,
            InternalLinkTypeLanguagePack,
            InternalLinkTypeLanguageSettings,
            InternalLinkTypeMainWebApp,
            InternalLinkTypeMessage,
            InternalLinkTypeMessageDraft,
            InternalLinkTypeMyStars,
            InternalLinkTypeMyToncoins,
            InternalLinkTypePassportDataRequest,
            InternalLinkTypePhoneNumberConfirmation,
            InternalLinkTypePremiumFeatures,
            InternalLinkTypePremiumGift,
            InternalLinkTypePremiumGiftCode,
            InternalLinkTypePrivacyAndSecuritySettings,
            InternalLinkTypeProxy,
            InternalLinkTypePublicChat,
            InternalLinkTypeQrCodeAuthentication,
            InternalLinkTypeRestorePurchases,
            InternalLinkTypeSettings,
            InternalLinkTypeStickerSet,
            InternalLinkTypeStory,
            InternalLinkTypeStoryAlbum,
            InternalLinkTypeTheme,
            InternalLinkTypeThemeSettings,
            InternalLinkTypeUnknownDeepLink,
            InternalLinkTypeUnsupportedProxy,
            InternalLinkTypeUpgradedGift,
            InternalLinkTypeUserPhoneNumber,
            InternalLinkTypeUserToken,
            InternalLinkTypeVideoChat,
            InternalLinkTypeWebApp,
            None,
        ] = edit_description_link
        r"""The internal link, which can be used to edit bot description; may be null"""
        self.edit_description_media_link: Union[
            InternalLinkTypeActiveSessions,
            InternalLinkTypeAttachmentMenuBot,
            InternalLinkTypeAuthenticationCode,
            InternalLinkTypeBackground,
            InternalLinkTypeBotAddToChannel,
            InternalLinkTypeBotStart,
            InternalLinkTypeBotStartInGroup,
            InternalLinkTypeBusinessChat,
            InternalLinkTypeBuyStars,
            InternalLinkTypeChangePhoneNumber,
            InternalLinkTypeChatAffiliateProgram,
            InternalLinkTypeChatBoost,
            InternalLinkTypeChatFolderInvite,
            InternalLinkTypeChatFolderSettings,
            InternalLinkTypeChatInvite,
            InternalLinkTypeDefaultMessageAutoDeleteTimerSettings,
            InternalLinkTypeDirectMessagesChat,
            InternalLinkTypeEditProfileSettings,
            InternalLinkTypeGame,
            InternalLinkTypeGiftCollection,
            InternalLinkTypeGroupCall,
            InternalLinkTypeInstantView,
            InternalLinkTypeInvoice,
            InternalLinkTypeLanguagePack,
            InternalLinkTypeLanguageSettings,
            InternalLinkTypeMainWebApp,
            InternalLinkTypeMessage,
            InternalLinkTypeMessageDraft,
            InternalLinkTypeMyStars,
            InternalLinkTypeMyToncoins,
            InternalLinkTypePassportDataRequest,
            InternalLinkTypePhoneNumberConfirmation,
            InternalLinkTypePremiumFeatures,
            InternalLinkTypePremiumGift,
            InternalLinkTypePremiumGiftCode,
            InternalLinkTypePrivacyAndSecuritySettings,
            InternalLinkTypeProxy,
            InternalLinkTypePublicChat,
            InternalLinkTypeQrCodeAuthentication,
            InternalLinkTypeRestorePurchases,
            InternalLinkTypeSettings,
            InternalLinkTypeStickerSet,
            InternalLinkTypeStory,
            InternalLinkTypeStoryAlbum,
            InternalLinkTypeTheme,
            InternalLinkTypeThemeSettings,
            InternalLinkTypeUnknownDeepLink,
            InternalLinkTypeUnsupportedProxy,
            InternalLinkTypeUpgradedGift,
            InternalLinkTypeUserPhoneNumber,
            InternalLinkTypeUserToken,
            InternalLinkTypeVideoChat,
            InternalLinkTypeWebApp,
            None,
        ] = edit_description_media_link
        r"""The internal link, which can be used to edit the photo or animation shown in the chat with the bot if the chat is empty; may be null"""
        self.edit_settings_link: Union[
            InternalLinkTypeActiveSessions,
            InternalLinkTypeAttachmentMenuBot,
            InternalLinkTypeAuthenticationCode,
            InternalLinkTypeBackground,
            InternalLinkTypeBotAddToChannel,
            InternalLinkTypeBotStart,
            InternalLinkTypeBotStartInGroup,
            InternalLinkTypeBusinessChat,
            InternalLinkTypeBuyStars,
            InternalLinkTypeChangePhoneNumber,
            InternalLinkTypeChatAffiliateProgram,
            InternalLinkTypeChatBoost,
            InternalLinkTypeChatFolderInvite,
            InternalLinkTypeChatFolderSettings,
            InternalLinkTypeChatInvite,
            InternalLinkTypeDefaultMessageAutoDeleteTimerSettings,
            InternalLinkTypeDirectMessagesChat,
            InternalLinkTypeEditProfileSettings,
            InternalLinkTypeGame,
            InternalLinkTypeGiftCollection,
            InternalLinkTypeGroupCall,
            InternalLinkTypeInstantView,
            InternalLinkTypeInvoice,
            InternalLinkTypeLanguagePack,
            InternalLinkTypeLanguageSettings,
            InternalLinkTypeMainWebApp,
            InternalLinkTypeMessage,
            InternalLinkTypeMessageDraft,
            InternalLinkTypeMyStars,
            InternalLinkTypeMyToncoins,
            InternalLinkTypePassportDataRequest,
            InternalLinkTypePhoneNumberConfirmation,
            InternalLinkTypePremiumFeatures,
            InternalLinkTypePremiumGift,
            InternalLinkTypePremiumGiftCode,
            InternalLinkTypePrivacyAndSecuritySettings,
            InternalLinkTypeProxy,
            InternalLinkTypePublicChat,
            InternalLinkTypeQrCodeAuthentication,
            InternalLinkTypeRestorePurchases,
            InternalLinkTypeSettings,
            InternalLinkTypeStickerSet,
            InternalLinkTypeStory,
            InternalLinkTypeStoryAlbum,
            InternalLinkTypeTheme,
            InternalLinkTypeThemeSettings,
            InternalLinkTypeUnknownDeepLink,
            InternalLinkTypeUnsupportedProxy,
            InternalLinkTypeUpgradedGift,
            InternalLinkTypeUserPhoneNumber,
            InternalLinkTypeUserToken,
            InternalLinkTypeVideoChat,
            InternalLinkTypeWebApp,
            None,
        ] = edit_settings_link
        r"""The internal link, which can be used to edit bot settings; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["botInfo"]:
        return "botInfo"

    @classmethod
    def getClass(self) -> Literal["BotInfo"]:
        return "BotInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "short_description": self.short_description,
            "description": self.description,
            "photo": self.photo,
            "animation": self.animation,
            "menu_button": self.menu_button,
            "commands": self.commands,
            "privacy_policy_url": self.privacy_policy_url,
            "default_group_administrator_rights": self.default_group_administrator_rights,
            "default_channel_administrator_rights": self.default_channel_administrator_rights,
            "affiliate_program": self.affiliate_program,
            "web_app_background_light_color": self.web_app_background_light_color,
            "web_app_background_dark_color": self.web_app_background_dark_color,
            "web_app_header_light_color": self.web_app_header_light_color,
            "web_app_header_dark_color": self.web_app_header_dark_color,
            "verification_parameters": self.verification_parameters,
            "can_get_revenue_statistics": self.can_get_revenue_statistics,
            "can_manage_emoji_status": self.can_manage_emoji_status,
            "has_media_previews": self.has_media_previews,
            "edit_commands_link": self.edit_commands_link,
            "edit_description_link": self.edit_description_link,
            "edit_description_media_link": self.edit_description_media_link,
            "edit_settings_link": self.edit_settings_link,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BotInfo", None]:
        if data:
            data_class = cls()
            data_class.short_description = data.get("short_description", "")
            data_class.description = data.get("description", "")
            data_class.photo = data.get("photo", None)
            data_class.animation = data.get("animation", None)
            data_class.menu_button = data.get("menu_button", None)
            data_class.commands = data.get("commands", None)
            data_class.privacy_policy_url = data.get("privacy_policy_url", "")
            data_class.default_group_administrator_rights = data.get(
                "default_group_administrator_rights", None
            )
            data_class.default_channel_administrator_rights = data.get(
                "default_channel_administrator_rights", None
            )
            data_class.affiliate_program = data.get("affiliate_program", None)
            data_class.web_app_background_light_color = int(
                data.get("web_app_background_light_color", 0)
            )
            data_class.web_app_background_dark_color = int(
                data.get("web_app_background_dark_color", 0)
            )
            data_class.web_app_header_light_color = int(
                data.get("web_app_header_light_color", 0)
            )
            data_class.web_app_header_dark_color = int(
                data.get("web_app_header_dark_color", 0)
            )
            data_class.verification_parameters = data.get(
                "verification_parameters", None
            )
            data_class.can_get_revenue_statistics = data.get(
                "can_get_revenue_statistics", False
            )
            data_class.can_manage_emoji_status = data.get(
                "can_manage_emoji_status", False
            )
            data_class.has_media_previews = data.get("has_media_previews", False)
            data_class.edit_commands_link = data.get("edit_commands_link", None)
            data_class.edit_description_link = data.get("edit_description_link", None)
            data_class.edit_description_media_link = data.get(
                "edit_description_media_link", None
            )
            data_class.edit_settings_link = data.get("edit_settings_link", None)

        return data_class


class UserFullInfo(TlObject):
    r"""Contains full information about a user

    Parameters:
        personal_photo (:class:`~pytdbot.types.ChatPhoto`):
            User profile photo set by the current user for the contact; may be null\. If null and user\.profile\_photo is null, then the photo is empty; otherwise, it is unknown\. If non\-null, then it is the same photo as in user\.profile\_photo and chat\.photo\. This photo isn't returned in the list of user photos

        photo (:class:`~pytdbot.types.ChatPhoto`):
            User profile photo; may be null\. If null and user\.profile\_photo is null, then the photo is empty; otherwise, it is unknown\. If non\-null and personal\_photo is null, then it is the same photo as in user\.profile\_photo and chat\.photo

        public_photo (:class:`~pytdbot.types.ChatPhoto`):
            User profile photo visible if the main photo is hidden by privacy settings; may be null\. If null and user\.profile\_photo is null, then the photo is empty; otherwise, it is unknown\. If non\-null and both photo and personal\_photo are null, then it is the same photo as in user\.profile\_photo and chat\.photo\. This photo isn't returned in the list of user photos

        block_list (:class:`~pytdbot.types.BlockList`):
            Block list to which the user is added; may be null if none

        can_be_called (:class:`bool`):
            True, if the user can be called

        supports_video_calls (:class:`bool`):
            True, if a video call can be created with the user

        has_private_calls (:class:`bool`):
            True, if the user can't be called due to their privacy settings

        has_private_forwards (:class:`bool`):
            True, if the user can't be linked in forwarded messages due to their privacy settings

        has_restricted_voice_and_video_note_messages (:class:`bool`):
            True, if voice and video notes can't be sent or forwarded to the user

        has_posted_to_profile_stories (:class:`bool`):
            True, if the user has posted to profile stories

        has_sponsored_messages_enabled (:class:`bool`):
            True, if the user always enabled sponsored messages; known only for the current user

        need_phone_number_privacy_exception (:class:`bool`):
            True, if the current user needs to explicitly allow to share their phone number with the user when the method addContact is used

        set_chat_background (:class:`bool`):
            True, if the user set chat background for both chat users and it wasn't reverted yet

        bio (:class:`~pytdbot.types.FormattedText`):
            A short user bio; may be null for bots

        birthdate (:class:`~pytdbot.types.Birthdate`):
            Birthdate of the user; may be null if unknown

        personal_chat_id (:class:`int`):
            Identifier of the personal chat of the user; 0 if none

        gift_count (:class:`int`):
            Number of saved to profile gifts for other users or the total number of received gifts for the current user

        group_in_common_count (:class:`int`):
            Number of group chats where both the other user and the current user are a member; 0 for the current user

        incoming_paid_message_star_count (:class:`int`):
            Number of Telegram Stars that must be paid by the user for each sent message to the current user

        outgoing_paid_message_star_count (:class:`int`):
            Number of Telegram Stars that must be paid by the current user for each sent message to the user

        gift_settings (:class:`~pytdbot.types.GiftSettings`):
            Settings for gift receiving for the user

        bot_verification (:class:`~pytdbot.types.BotVerification`):
            Information about verification status of the user provided by a bot; may be null if none or unknown

        main_profile_tab (:class:`~pytdbot.types.ProfileTab`):
            The main tab chosen by the user; may be null if not chosen manually

        first_profile_audio (:class:`~pytdbot.types.Audio`):
            The first audio file added to the user's profile; may be null if none

        rating (:class:`~pytdbot.types.UserRating`):
            The current rating of the user; may be null if none

        pending_rating (:class:`~pytdbot.types.UserRating`):
            The rating of the user after the next change; may be null if the user isn't the current user or there are no pending rating changes

        pending_rating_date (:class:`int`):
            Unix timestamp when rating of the user will change to pending\_rating; 0 if the user isn't the current user or there are no pending rating changes

        note (:class:`~pytdbot.types.FormattedText`):
            Note added to the user's contact; may be null if none

        business_info (:class:`~pytdbot.types.BusinessInfo`):
            Information about business settings for Telegram Business accounts; may be null if none

        bot_info (:class:`~pytdbot.types.BotInfo`):
            For bots, information about the bot; may be null if the user isn't a bot

    """

    def __init__(
        self,
        personal_photo: ChatPhoto = None,
        photo: ChatPhoto = None,
        public_photo: ChatPhoto = None,
        block_list: BlockList = None,
        can_be_called: bool = False,
        supports_video_calls: bool = False,
        has_private_calls: bool = False,
        has_private_forwards: bool = False,
        has_restricted_voice_and_video_note_messages: bool = False,
        has_posted_to_profile_stories: bool = False,
        has_sponsored_messages_enabled: bool = False,
        need_phone_number_privacy_exception: bool = False,
        set_chat_background: bool = False,
        bio: FormattedText = None,
        birthdate: Birthdate = None,
        personal_chat_id: int = 0,
        gift_count: int = 0,
        group_in_common_count: int = 0,
        incoming_paid_message_star_count: int = 0,
        outgoing_paid_message_star_count: int = 0,
        gift_settings: GiftSettings = None,
        bot_verification: BotVerification = None,
        main_profile_tab: ProfileTab = None,
        first_profile_audio: Audio = None,
        rating: UserRating = None,
        pending_rating: UserRating = None,
        pending_rating_date: int = 0,
        note: FormattedText = None,
        business_info: BusinessInfo = None,
        bot_info: BotInfo = None,
    ) -> None:
        self.personal_photo: Union[ChatPhoto, None] = personal_photo
        r"""User profile photo set by the current user for the contact; may be null\. If null and user\.profile\_photo is null, then the photo is empty; otherwise, it is unknown\. If non\-null, then it is the same photo as in user\.profile\_photo and chat\.photo\. This photo isn't returned in the list of user photos"""
        self.photo: Union[ChatPhoto, None] = photo
        r"""User profile photo; may be null\. If null and user\.profile\_photo is null, then the photo is empty; otherwise, it is unknown\. If non\-null and personal\_photo is null, then it is the same photo as in user\.profile\_photo and chat\.photo"""
        self.public_photo: Union[ChatPhoto, None] = public_photo
        r"""User profile photo visible if the main photo is hidden by privacy settings; may be null\. If null and user\.profile\_photo is null, then the photo is empty; otherwise, it is unknown\. If non\-null and both photo and personal\_photo are null, then it is the same photo as in user\.profile\_photo and chat\.photo\. This photo isn't returned in the list of user photos"""
        self.block_list: Union[BlockListMain, BlockListStories, None] = block_list
        r"""Block list to which the user is added; may be null if none"""
        self.can_be_called: bool = bool(can_be_called)
        r"""True, if the user can be called"""
        self.supports_video_calls: bool = bool(supports_video_calls)
        r"""True, if a video call can be created with the user"""
        self.has_private_calls: bool = bool(has_private_calls)
        r"""True, if the user can't be called due to their privacy settings"""
        self.has_private_forwards: bool = bool(has_private_forwards)
        r"""True, if the user can't be linked in forwarded messages due to their privacy settings"""
        self.has_restricted_voice_and_video_note_messages: bool = bool(
            has_restricted_voice_and_video_note_messages
        )
        r"""True, if voice and video notes can't be sent or forwarded to the user"""
        self.has_posted_to_profile_stories: bool = bool(has_posted_to_profile_stories)
        r"""True, if the user has posted to profile stories"""
        self.has_sponsored_messages_enabled: bool = bool(has_sponsored_messages_enabled)
        r"""True, if the user always enabled sponsored messages; known only for the current user"""
        self.need_phone_number_privacy_exception: bool = bool(
            need_phone_number_privacy_exception
        )
        r"""True, if the current user needs to explicitly allow to share their phone number with the user when the method addContact is used"""
        self.set_chat_background: bool = bool(set_chat_background)
        r"""True, if the user set chat background for both chat users and it wasn't reverted yet"""
        self.bio: Union[FormattedText, None] = bio
        r"""A short user bio; may be null for bots"""
        self.birthdate: Union[Birthdate, None] = birthdate
        r"""Birthdate of the user; may be null if unknown"""
        self.personal_chat_id: int = int(personal_chat_id)
        r"""Identifier of the personal chat of the user; 0 if none"""
        self.gift_count: int = int(gift_count)
        r"""Number of saved to profile gifts for other users or the total number of received gifts for the current user"""
        self.group_in_common_count: int = int(group_in_common_count)
        r"""Number of group chats where both the other user and the current user are a member; 0 for the current user"""
        self.incoming_paid_message_star_count: int = int(
            incoming_paid_message_star_count
        )
        r"""Number of Telegram Stars that must be paid by the user for each sent message to the current user"""
        self.outgoing_paid_message_star_count: int = int(
            outgoing_paid_message_star_count
        )
        r"""Number of Telegram Stars that must be paid by the current user for each sent message to the user"""
        self.gift_settings: Union[GiftSettings, None] = gift_settings
        r"""Settings for gift receiving for the user"""
        self.bot_verification: Union[BotVerification, None] = bot_verification
        r"""Information about verification status of the user provided by a bot; may be null if none or unknown"""
        self.main_profile_tab: Union[
            ProfileTabPosts,
            ProfileTabGifts,
            ProfileTabMedia,
            ProfileTabFiles,
            ProfileTabLinks,
            ProfileTabMusic,
            ProfileTabVoice,
            ProfileTabGifs,
            None,
        ] = main_profile_tab
        r"""The main tab chosen by the user; may be null if not chosen manually"""
        self.first_profile_audio: Union[Audio, None] = first_profile_audio
        r"""The first audio file added to the user's profile; may be null if none"""
        self.rating: Union[UserRating, None] = rating
        r"""The current rating of the user; may be null if none"""
        self.pending_rating: Union[UserRating, None] = pending_rating
        r"""The rating of the user after the next change; may be null if the user isn't the current user or there are no pending rating changes"""
        self.pending_rating_date: int = int(pending_rating_date)
        r"""Unix timestamp when rating of the user will change to pending\_rating; 0 if the user isn't the current user or there are no pending rating changes"""
        self.note: Union[FormattedText, None] = note
        r"""Note added to the user's contact; may be null if none"""
        self.business_info: Union[BusinessInfo, None] = business_info
        r"""Information about business settings for Telegram Business accounts; may be null if none"""
        self.bot_info: Union[BotInfo, None] = bot_info
        r"""For bots, information about the bot; may be null if the user isn't a bot"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["userFullInfo"]:
        return "userFullInfo"

    @classmethod
    def getClass(self) -> Literal["UserFullInfo"]:
        return "UserFullInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "personal_photo": self.personal_photo,
            "photo": self.photo,
            "public_photo": self.public_photo,
            "block_list": self.block_list,
            "can_be_called": self.can_be_called,
            "supports_video_calls": self.supports_video_calls,
            "has_private_calls": self.has_private_calls,
            "has_private_forwards": self.has_private_forwards,
            "has_restricted_voice_and_video_note_messages": self.has_restricted_voice_and_video_note_messages,
            "has_posted_to_profile_stories": self.has_posted_to_profile_stories,
            "has_sponsored_messages_enabled": self.has_sponsored_messages_enabled,
            "need_phone_number_privacy_exception": self.need_phone_number_privacy_exception,
            "set_chat_background": self.set_chat_background,
            "bio": self.bio,
            "birthdate": self.birthdate,
            "personal_chat_id": self.personal_chat_id,
            "gift_count": self.gift_count,
            "group_in_common_count": self.group_in_common_count,
            "incoming_paid_message_star_count": self.incoming_paid_message_star_count,
            "outgoing_paid_message_star_count": self.outgoing_paid_message_star_count,
            "gift_settings": self.gift_settings,
            "bot_verification": self.bot_verification,
            "main_profile_tab": self.main_profile_tab,
            "first_profile_audio": self.first_profile_audio,
            "rating": self.rating,
            "pending_rating": self.pending_rating,
            "pending_rating_date": self.pending_rating_date,
            "note": self.note,
            "business_info": self.business_info,
            "bot_info": self.bot_info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UserFullInfo", None]:
        if data:
            data_class = cls()
            data_class.personal_photo = data.get("personal_photo", None)
            data_class.photo = data.get("photo", None)
            data_class.public_photo = data.get("public_photo", None)
            data_class.block_list = data.get("block_list", None)
            data_class.can_be_called = data.get("can_be_called", False)
            data_class.supports_video_calls = data.get("supports_video_calls", False)
            data_class.has_private_calls = data.get("has_private_calls", False)
            data_class.has_private_forwards = data.get("has_private_forwards", False)
            data_class.has_restricted_voice_and_video_note_messages = data.get(
                "has_restricted_voice_and_video_note_messages", False
            )
            data_class.has_posted_to_profile_stories = data.get(
                "has_posted_to_profile_stories", False
            )
            data_class.has_sponsored_messages_enabled = data.get(
                "has_sponsored_messages_enabled", False
            )
            data_class.need_phone_number_privacy_exception = data.get(
                "need_phone_number_privacy_exception", False
            )
            data_class.set_chat_background = data.get("set_chat_background", False)
            data_class.bio = data.get("bio", None)
            data_class.birthdate = data.get("birthdate", None)
            data_class.personal_chat_id = int(data.get("personal_chat_id", 0))
            data_class.gift_count = int(data.get("gift_count", 0))
            data_class.group_in_common_count = int(data.get("group_in_common_count", 0))
            data_class.incoming_paid_message_star_count = int(
                data.get("incoming_paid_message_star_count", 0)
            )
            data_class.outgoing_paid_message_star_count = int(
                data.get("outgoing_paid_message_star_count", 0)
            )
            data_class.gift_settings = data.get("gift_settings", None)
            data_class.bot_verification = data.get("bot_verification", None)
            data_class.main_profile_tab = data.get("main_profile_tab", None)
            data_class.first_profile_audio = data.get("first_profile_audio", None)
            data_class.rating = data.get("rating", None)
            data_class.pending_rating = data.get("pending_rating", None)
            data_class.pending_rating_date = int(data.get("pending_rating_date", 0))
            data_class.note = data.get("note", None)
            data_class.business_info = data.get("business_info", None)
            data_class.bot_info = data.get("bot_info", None)

        return data_class


class Users(TlObject):
    r"""Represents a list of users

    Parameters:
        total_count (:class:`int`):
            Approximate total number of users found

        user_ids (List[:class:`int`]):
            A list of user identifiers

    """

    def __init__(self, total_count: int = 0, user_ids: List[int] = None) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of users found"""
        self.user_ids: List[int] = user_ids or []
        r"""A list of user identifiers"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["users"]:
        return "users"

    @classmethod
    def getClass(self) -> Literal["Users"]:
        return "Users"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "user_ids": self.user_ids,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Users", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.user_ids = data.get("user_ids", None)

        return data_class


class FoundUsers(TlObject):
    r"""Represents a list of found users

    Parameters:
        user_ids (List[:class:`int`]):
            Identifiers of the found users

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(self, user_ids: List[int] = None, next_offset: str = "") -> None:
        self.user_ids: List[int] = user_ids or []
        r"""Identifiers of the found users"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundUsers"]:
        return "foundUsers"

    @classmethod
    def getClass(self) -> Literal["FoundUsers"]:
        return "FoundUsers"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_ids": self.user_ids,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundUsers", None]:
        if data:
            data_class = cls()
            data_class.user_ids = data.get("user_ids", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class ChatAdministrator(TlObject):
    r"""Contains information about a chat administrator

    Parameters:
        user_id (:class:`int`):
            User identifier of the administrator

        custom_title (:class:`str`):
            Custom title of the administrator

        is_owner (:class:`bool`):
            True, if the user is the owner of the chat

    """

    def __init__(
        self, user_id: int = 0, custom_title: str = "", is_owner: bool = False
    ) -> None:
        self.user_id: int = int(user_id)
        r"""User identifier of the administrator"""
        self.custom_title: Union[str, None] = custom_title
        r"""Custom title of the administrator"""
        self.is_owner: bool = bool(is_owner)
        r"""True, if the user is the owner of the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatAdministrator"]:
        return "chatAdministrator"

    @classmethod
    def getClass(self) -> Literal["ChatAdministrator"]:
        return "ChatAdministrator"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "custom_title": self.custom_title,
            "is_owner": self.is_owner,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatAdministrator", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.custom_title = data.get("custom_title", "")
            data_class.is_owner = data.get("is_owner", False)

        return data_class


class ChatAdministrators(TlObject):
    r"""Represents a list of chat administrators

    Parameters:
        administrators (List[:class:`~pytdbot.types.ChatAdministrator`]):
            A list of chat administrators

    """

    def __init__(self, administrators: List[ChatAdministrator] = None) -> None:
        self.administrators: List[ChatAdministrator] = administrators or []
        r"""A list of chat administrators"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatAdministrators"]:
        return "chatAdministrators"

    @classmethod
    def getClass(self) -> Literal["ChatAdministrators"]:
        return "ChatAdministrators"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "administrators": self.administrators}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatAdministrators", None]:
        if data:
            data_class = cls()
            data_class.administrators = data.get("administrators", None)

        return data_class


class ChatMemberStatusCreator(TlObject, ChatMemberStatus):
    r"""The user is the owner of the chat and has all the administrator privileges

    Parameters:
        custom_title (:class:`str`):
            A custom title of the owner; 0\-16 characters without emoji; applicable to supergroups only

        is_anonymous (:class:`bool`):
            True, if the creator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only

        is_member (:class:`bool`):
            True, if the user is a member of the chat

    """

    def __init__(
        self,
        custom_title: str = "",
        is_anonymous: bool = False,
        is_member: bool = False,
    ) -> None:
        self.custom_title: Union[str, None] = custom_title
        r"""A custom title of the owner; 0\-16 characters without emoji; applicable to supergroups only"""
        self.is_anonymous: bool = bool(is_anonymous)
        r"""True, if the creator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only"""
        self.is_member: bool = bool(is_member)
        r"""True, if the user is a member of the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMemberStatusCreator"]:
        return "chatMemberStatusCreator"

    @classmethod
    def getClass(self) -> Literal["ChatMemberStatus"]:
        return "ChatMemberStatus"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "custom_title": self.custom_title,
            "is_anonymous": self.is_anonymous,
            "is_member": self.is_member,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMemberStatusCreator", None]:
        if data:
            data_class = cls()
            data_class.custom_title = data.get("custom_title", "")
            data_class.is_anonymous = data.get("is_anonymous", False)
            data_class.is_member = data.get("is_member", False)

        return data_class


class ChatMemberStatusAdministrator(TlObject, ChatMemberStatus):
    r"""The user is a member of the chat and has some additional privileges\. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats\. In supergroups and channels, there are more detailed options for administrator privileges

    Parameters:
        custom_title (:class:`str`):
            A custom title of the administrator; 0\-16 characters without emoji; applicable to supergroups only

        can_be_edited (:class:`bool`):
            True, if the current user can edit the administrator privileges for the called user

        rights (:class:`~pytdbot.types.ChatAdministratorRights`):
            Rights of the administrator

    """

    def __init__(
        self,
        custom_title: str = "",
        can_be_edited: bool = False,
        rights: ChatAdministratorRights = None,
    ) -> None:
        self.custom_title: Union[str, None] = custom_title
        r"""A custom title of the administrator; 0\-16 characters without emoji; applicable to supergroups only"""
        self.can_be_edited: bool = bool(can_be_edited)
        r"""True, if the current user can edit the administrator privileges for the called user"""
        self.rights: Union[ChatAdministratorRights, None] = rights
        r"""Rights of the administrator"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMemberStatusAdministrator"]:
        return "chatMemberStatusAdministrator"

    @classmethod
    def getClass(self) -> Literal["ChatMemberStatus"]:
        return "ChatMemberStatus"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "custom_title": self.custom_title,
            "can_be_edited": self.can_be_edited,
            "rights": self.rights,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMemberStatusAdministrator", None]:
        if data:
            data_class = cls()
            data_class.custom_title = data.get("custom_title", "")
            data_class.can_be_edited = data.get("can_be_edited", False)
            data_class.rights = data.get("rights", None)

        return data_class


class ChatMemberStatusMember(TlObject, ChatMemberStatus):
    r"""The user is a member of the chat, without any additional privileges or restrictions

    Parameters:
        member_until_date (:class:`int`):
            Point in time \(Unix timestamp\) when the user will be removed from the chat because of the expired subscription; 0 if never\. Ignored in setChatMemberStatus

    """

    def __init__(self, member_until_date: int = 0) -> None:
        self.member_until_date: int = int(member_until_date)
        r"""Point in time \(Unix timestamp\) when the user will be removed from the chat because of the expired subscription; 0 if never\. Ignored in setChatMemberStatus"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMemberStatusMember"]:
        return "chatMemberStatusMember"

    @classmethod
    def getClass(self) -> Literal["ChatMemberStatus"]:
        return "ChatMemberStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "member_until_date": self.member_until_date}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMemberStatusMember", None]:
        if data:
            data_class = cls()
            data_class.member_until_date = int(data.get("member_until_date", 0))

        return data_class


class ChatMemberStatusRestricted(TlObject, ChatMemberStatus):
    r"""The user is under certain restrictions in the chat\. Not supported in basic groups and channels

    Parameters:
        is_member (:class:`bool`):
            True, if the user is a member of the chat

        restricted_until_date (:class:`int`):
            Point in time \(Unix timestamp\) when restrictions will be lifted from the user; 0 if never\. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever

        permissions (:class:`~pytdbot.types.ChatPermissions`):
            User permissions in the chat

    """

    def __init__(
        self,
        is_member: bool = False,
        restricted_until_date: int = 0,
        permissions: ChatPermissions = None,
    ) -> None:
        self.is_member: bool = bool(is_member)
        r"""True, if the user is a member of the chat"""
        self.restricted_until_date: int = int(restricted_until_date)
        r"""Point in time \(Unix timestamp\) when restrictions will be lifted from the user; 0 if never\. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever"""
        self.permissions: Union[ChatPermissions, None] = permissions
        r"""User permissions in the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMemberStatusRestricted"]:
        return "chatMemberStatusRestricted"

    @classmethod
    def getClass(self) -> Literal["ChatMemberStatus"]:
        return "ChatMemberStatus"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "is_member": self.is_member,
            "restricted_until_date": self.restricted_until_date,
            "permissions": self.permissions,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMemberStatusRestricted", None]:
        if data:
            data_class = cls()
            data_class.is_member = data.get("is_member", False)
            data_class.restricted_until_date = int(data.get("restricted_until_date", 0))
            data_class.permissions = data.get("permissions", None)

        return data_class


class ChatMemberStatusLeft(TlObject, ChatMemberStatus):
    r"""The user or the chat is not a chat member"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMemberStatusLeft"]:
        return "chatMemberStatusLeft"

    @classmethod
    def getClass(self) -> Literal["ChatMemberStatus"]:
        return "ChatMemberStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMemberStatusLeft", None]:
        if data:
            data_class = cls()

        return data_class


class ChatMemberStatusBanned(TlObject, ChatMemberStatus):
    r"""The user or the chat was banned \(and hence is not a member of the chat\)\. Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat

    Parameters:
        banned_until_date (:class:`int`):
            Point in time \(Unix timestamp\) when the user will be unbanned; 0 if never\. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever\. Always 0 in basic groups

    """

    def __init__(self, banned_until_date: int = 0) -> None:
        self.banned_until_date: int = int(banned_until_date)
        r"""Point in time \(Unix timestamp\) when the user will be unbanned; 0 if never\. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever\. Always 0 in basic groups"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMemberStatusBanned"]:
        return "chatMemberStatusBanned"

    @classmethod
    def getClass(self) -> Literal["ChatMemberStatus"]:
        return "ChatMemberStatus"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "banned_until_date": self.banned_until_date}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMemberStatusBanned", None]:
        if data:
            data_class = cls()
            data_class.banned_until_date = int(data.get("banned_until_date", 0))

        return data_class


class ChatMember(TlObject):
    r"""Describes a user or a chat as a member of another chat

    Parameters:
        member_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the chat member\. Currently, other chats can be only Left or Banned\. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels

        inviter_user_id (:class:`int`):
            Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown

        joined_chat_date (:class:`int`):
            Point in time \(Unix timestamp\) when the user joined/was promoted/was banned in the chat

        status (:class:`~pytdbot.types.ChatMemberStatus`):
            Status of the member in the chat

    """

    def __init__(
        self,
        member_id: MessageSender = None,
        inviter_user_id: int = 0,
        joined_chat_date: int = 0,
        status: ChatMemberStatus = None,
    ) -> None:
        self.member_id: Union[MessageSenderUser, MessageSenderChat, None] = member_id
        r"""Identifier of the chat member\. Currently, other chats can be only Left or Banned\. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels"""
        self.inviter_user_id: int = int(inviter_user_id)
        r"""Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown"""
        self.joined_chat_date: int = int(joined_chat_date)
        r"""Point in time \(Unix timestamp\) when the user joined/was promoted/was banned in the chat"""
        self.status: Union[
            ChatMemberStatusCreator,
            ChatMemberStatusAdministrator,
            ChatMemberStatusMember,
            ChatMemberStatusRestricted,
            ChatMemberStatusLeft,
            ChatMemberStatusBanned,
            None,
        ] = status
        r"""Status of the member in the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMember"]:
        return "chatMember"

    @classmethod
    def getClass(self) -> Literal["ChatMember"]:
        return "ChatMember"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "member_id": self.member_id,
            "inviter_user_id": self.inviter_user_id,
            "joined_chat_date": self.joined_chat_date,
            "status": self.status,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMember", None]:
        if data:
            data_class = cls()
            data_class.member_id = data.get("member_id", None)
            data_class.inviter_user_id = int(data.get("inviter_user_id", 0))
            data_class.joined_chat_date = int(data.get("joined_chat_date", 0))
            data_class.status = data.get("status", None)

        return data_class


class ChatMembers(TlObject):
    r"""Contains a list of chat members

    Parameters:
        total_count (:class:`int`):
            Approximate total number of chat members found

        members (List[:class:`~pytdbot.types.ChatMember`]):
            A list of chat members

    """

    def __init__(self, total_count: int = 0, members: List[ChatMember] = None) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of chat members found"""
        self.members: List[ChatMember] = members or []
        r"""A list of chat members"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembers"]:
        return "chatMembers"

    @classmethod
    def getClass(self) -> Literal["ChatMembers"]:
        return "ChatMembers"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "members": self.members,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembers", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.members = data.get("members", None)

        return data_class


class ChatMembersFilterContacts(TlObject, ChatMembersFilter):
    r"""Returns contacts of the user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterContacts"]:
        return "chatMembersFilterContacts"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterContacts", None]:
        if data:
            data_class = cls()

        return data_class


class ChatMembersFilterAdministrators(TlObject, ChatMembersFilter):
    r"""Returns the owner and administrators"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterAdministrators"]:
        return "chatMembersFilterAdministrators"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterAdministrators", None]:
        if data:
            data_class = cls()

        return data_class


class ChatMembersFilterMembers(TlObject, ChatMembersFilter):
    r"""Returns all chat members, including restricted chat members"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterMembers"]:
        return "chatMembersFilterMembers"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterMembers", None]:
        if data:
            data_class = cls()

        return data_class


class ChatMembersFilterMention(TlObject, ChatMembersFilter):
    r"""Returns users which can be mentioned in the chat

    Parameters:
        topic_id (:class:`~pytdbot.types.MessageTopic`):
            Identifier of the topic in which the users will be mentioned; pass null if none

    """

    def __init__(self, topic_id: MessageTopic = None) -> None:
        self.topic_id: Union[
            MessageTopicThread,
            MessageTopicForum,
            MessageTopicDirectMessages,
            MessageTopicSavedMessages,
            None,
        ] = topic_id
        r"""Identifier of the topic in which the users will be mentioned; pass null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterMention"]:
        return "chatMembersFilterMention"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "topic_id": self.topic_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterMention", None]:
        if data:
            data_class = cls()
            data_class.topic_id = data.get("topic_id", None)

        return data_class


class ChatMembersFilterRestricted(TlObject, ChatMembersFilter):
    r"""Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterRestricted"]:
        return "chatMembersFilterRestricted"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterRestricted", None]:
        if data:
            data_class = cls()

        return data_class


class ChatMembersFilterBanned(TlObject, ChatMembersFilter):
    r"""Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterBanned"]:
        return "chatMembersFilterBanned"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterBanned", None]:
        if data:
            data_class = cls()

        return data_class


class ChatMembersFilterBots(TlObject, ChatMembersFilter):
    r"""Returns bot members of the chat"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMembersFilterBots"]:
        return "chatMembersFilterBots"

    @classmethod
    def getClass(self) -> Literal["ChatMembersFilter"]:
        return "ChatMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMembersFilterBots", None]:
        if data:
            data_class = cls()

        return data_class


class SupergroupMembersFilterRecent(TlObject, SupergroupMembersFilter):
    r"""Returns recently active users in reverse chronological order"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterRecent"]:
        return "supergroupMembersFilterRecent"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterRecent", None]:
        if data:
            data_class = cls()

        return data_class


class SupergroupMembersFilterContacts(TlObject, SupergroupMembersFilter):
    r"""Returns contacts of the user, which are members of the supergroup or channel

    Parameters:
        query (:class:`str`):
            Query to search for

    """

    def __init__(self, query: str = "") -> None:
        self.query: Union[str, None] = query
        r"""Query to search for"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterContacts"]:
        return "supergroupMembersFilterContacts"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "query": self.query}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterContacts", None]:
        if data:
            data_class = cls()
            data_class.query = data.get("query", "")

        return data_class


class SupergroupMembersFilterAdministrators(TlObject, SupergroupMembersFilter):
    r"""Returns the owner and administrators"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterAdministrators"]:
        return "supergroupMembersFilterAdministrators"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["SupergroupMembersFilterAdministrators", None]:
        if data:
            data_class = cls()

        return data_class


class SupergroupMembersFilterSearch(TlObject, SupergroupMembersFilter):
    r"""Used to search for supergroup or channel members via a \(string\) query

    Parameters:
        query (:class:`str`):
            Query to search for

    """

    def __init__(self, query: str = "") -> None:
        self.query: Union[str, None] = query
        r"""Query to search for"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterSearch"]:
        return "supergroupMembersFilterSearch"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "query": self.query}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterSearch", None]:
        if data:
            data_class = cls()
            data_class.query = data.get("query", "")

        return data_class


class SupergroupMembersFilterRestricted(TlObject, SupergroupMembersFilter):
    r"""Returns restricted supergroup members; can be used only by administrators

    Parameters:
        query (:class:`str`):
            Query to search for

    """

    def __init__(self, query: str = "") -> None:
        self.query: Union[str, None] = query
        r"""Query to search for"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterRestricted"]:
        return "supergroupMembersFilterRestricted"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "query": self.query}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterRestricted", None]:
        if data:
            data_class = cls()
            data_class.query = data.get("query", "")

        return data_class


class SupergroupMembersFilterBanned(TlObject, SupergroupMembersFilter):
    r"""Returns users banned from the supergroup or channel; can be used only by administrators

    Parameters:
        query (:class:`str`):
            Query to search for

    """

    def __init__(self, query: str = "") -> None:
        self.query: Union[str, None] = query
        r"""Query to search for"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterBanned"]:
        return "supergroupMembersFilterBanned"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "query": self.query}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterBanned", None]:
        if data:
            data_class = cls()
            data_class.query = data.get("query", "")

        return data_class


class SupergroupMembersFilterMention(TlObject, SupergroupMembersFilter):
    r"""Returns users which can be mentioned in the supergroup

    Parameters:
        query (:class:`str`):
            Query to search for

        topic_id (:class:`~pytdbot.types.MessageTopic`):
            Identifier of the topic in which the users will be mentioned; pass null if none

    """

    def __init__(self, query: str = "", topic_id: MessageTopic = None) -> None:
        self.query: Union[str, None] = query
        r"""Query to search for"""
        self.topic_id: Union[
            MessageTopicThread,
            MessageTopicForum,
            MessageTopicDirectMessages,
            MessageTopicSavedMessages,
            None,
        ] = topic_id
        r"""Identifier of the topic in which the users will be mentioned; pass null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterMention"]:
        return "supergroupMembersFilterMention"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "query": self.query, "topic_id": self.topic_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterMention", None]:
        if data:
            data_class = cls()
            data_class.query = data.get("query", "")
            data_class.topic_id = data.get("topic_id", None)

        return data_class


class SupergroupMembersFilterBots(TlObject, SupergroupMembersFilter):
    r"""Returns bot members of the supergroup or channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupMembersFilterBots"]:
        return "supergroupMembersFilterBots"

    @classmethod
    def getClass(self) -> Literal["SupergroupMembersFilter"]:
        return "SupergroupMembersFilter"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupMembersFilterBots", None]:
        if data:
            data_class = cls()

        return data_class


class ChatInviteLink(TlObject):
    r"""Contains a chat invite link

    Parameters:
        invite_link (:class:`str`):
            Chat invite link

        name (:class:`str`):
            Name of the link

        creator_user_id (:class:`int`):
            User identifier of an administrator created the link

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the link was created

        edit_date (:class:`int`):
            Point in time \(Unix timestamp\) when the link was last edited; 0 if never or unknown

        expiration_date (:class:`int`):
            Point in time \(Unix timestamp\) when the link will expire; 0 if never

        subscription_pricing (:class:`~pytdbot.types.StarSubscriptionPricing`):
            Information about subscription plan that is applied to the users joining the chat by the link; may be null if the link doesn't require subscription

        member_limit (:class:`int`):
            The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited\. Always 0 if the link requires approval

        member_count (:class:`int`):
            Number of chat members, which joined the chat using the link

        expired_member_count (:class:`int`):
            Number of chat members, which joined the chat using the link, but have already left because of expired subscription; for subscription links only

        pending_join_request_count (:class:`int`):
            Number of pending join requests created using this link

        creates_join_request (:class:`bool`):
            True, if the link only creates join request\. If true, total number of joining members will be unlimited

        is_primary (:class:`bool`):
            True, if the link is primary\. Primary invite link can't have name, expiration date, or usage limit\. There is exactly one primary invite link for each administrator with can\_invite\_users right at a given time

        is_revoked (:class:`bool`):
            True, if the link was revoked

    """

    def __init__(
        self,
        invite_link: str = "",
        name: str = "",
        creator_user_id: int = 0,
        date: int = 0,
        edit_date: int = 0,
        expiration_date: int = 0,
        subscription_pricing: StarSubscriptionPricing = None,
        member_limit: int = 0,
        member_count: int = 0,
        expired_member_count: int = 0,
        pending_join_request_count: int = 0,
        creates_join_request: bool = False,
        is_primary: bool = False,
        is_revoked: bool = False,
    ) -> None:
        self.invite_link: Union[str, None] = invite_link
        r"""Chat invite link"""
        self.name: Union[str, None] = name
        r"""Name of the link"""
        self.creator_user_id: int = int(creator_user_id)
        r"""User identifier of an administrator created the link"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the link was created"""
        self.edit_date: int = int(edit_date)
        r"""Point in time \(Unix timestamp\) when the link was last edited; 0 if never or unknown"""
        self.expiration_date: int = int(expiration_date)
        r"""Point in time \(Unix timestamp\) when the link will expire; 0 if never"""
        self.subscription_pricing: Union[StarSubscriptionPricing, None] = (
            subscription_pricing
        )
        r"""Information about subscription plan that is applied to the users joining the chat by the link; may be null if the link doesn't require subscription"""
        self.member_limit: int = int(member_limit)
        r"""The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited\. Always 0 if the link requires approval"""
        self.member_count: int = int(member_count)
        r"""Number of chat members, which joined the chat using the link"""
        self.expired_member_count: int = int(expired_member_count)
        r"""Number of chat members, which joined the chat using the link, but have already left because of expired subscription; for subscription links only"""
        self.pending_join_request_count: int = int(pending_join_request_count)
        r"""Number of pending join requests created using this link"""
        self.creates_join_request: bool = bool(creates_join_request)
        r"""True, if the link only creates join request\. If true, total number of joining members will be unlimited"""
        self.is_primary: bool = bool(is_primary)
        r"""True, if the link is primary\. Primary invite link can't have name, expiration date, or usage limit\. There is exactly one primary invite link for each administrator with can\_invite\_users right at a given time"""
        self.is_revoked: bool = bool(is_revoked)
        r"""True, if the link was revoked"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLink"]:
        return "chatInviteLink"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLink"]:
        return "ChatInviteLink"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "invite_link": self.invite_link,
            "name": self.name,
            "creator_user_id": self.creator_user_id,
            "date": self.date,
            "edit_date": self.edit_date,
            "expiration_date": self.expiration_date,
            "subscription_pricing": self.subscription_pricing,
            "member_limit": self.member_limit,
            "member_count": self.member_count,
            "expired_member_count": self.expired_member_count,
            "pending_join_request_count": self.pending_join_request_count,
            "creates_join_request": self.creates_join_request,
            "is_primary": self.is_primary,
            "is_revoked": self.is_revoked,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLink", None]:
        if data:
            data_class = cls()
            data_class.invite_link = data.get("invite_link", "")
            data_class.name = data.get("name", "")
            data_class.creator_user_id = int(data.get("creator_user_id", 0))
            data_class.date = int(data.get("date", 0))
            data_class.edit_date = int(data.get("edit_date", 0))
            data_class.expiration_date = int(data.get("expiration_date", 0))
            data_class.subscription_pricing = data.get("subscription_pricing", None)
            data_class.member_limit = int(data.get("member_limit", 0))
            data_class.member_count = int(data.get("member_count", 0))
            data_class.expired_member_count = int(data.get("expired_member_count", 0))
            data_class.pending_join_request_count = int(
                data.get("pending_join_request_count", 0)
            )
            data_class.creates_join_request = data.get("creates_join_request", False)
            data_class.is_primary = data.get("is_primary", False)
            data_class.is_revoked = data.get("is_revoked", False)

        return data_class


class ChatInviteLinks(TlObject):
    r"""Contains a list of chat invite links

    Parameters:
        total_count (:class:`int`):
            Approximate total number of chat invite links found

        invite_links (List[:class:`~pytdbot.types.ChatInviteLink`]):
            List of invite links

    """

    def __init__(
        self, total_count: int = 0, invite_links: List[ChatInviteLink] = None
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of chat invite links found"""
        self.invite_links: List[ChatInviteLink] = invite_links or []
        r"""List of invite links"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinks"]:
        return "chatInviteLinks"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinks"]:
        return "ChatInviteLinks"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "invite_links": self.invite_links,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinks", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.invite_links = data.get("invite_links", None)

        return data_class


class ChatInviteLinkCount(TlObject):
    r"""Describes a chat administrator with a number of active and revoked chat invite links

    Parameters:
        user_id (:class:`int`):
            Administrator's user identifier

        invite_link_count (:class:`int`):
            Number of active invite links

        revoked_invite_link_count (:class:`int`):
            Number of revoked invite links

    """

    def __init__(
        self,
        user_id: int = 0,
        invite_link_count: int = 0,
        revoked_invite_link_count: int = 0,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""Administrator's user identifier"""
        self.invite_link_count: int = int(invite_link_count)
        r"""Number of active invite links"""
        self.revoked_invite_link_count: int = int(revoked_invite_link_count)
        r"""Number of revoked invite links"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinkCount"]:
        return "chatInviteLinkCount"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinkCount"]:
        return "ChatInviteLinkCount"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "invite_link_count": self.invite_link_count,
            "revoked_invite_link_count": self.revoked_invite_link_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinkCount", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.invite_link_count = int(data.get("invite_link_count", 0))
            data_class.revoked_invite_link_count = int(
                data.get("revoked_invite_link_count", 0)
            )

        return data_class


class ChatInviteLinkCounts(TlObject):
    r"""Contains a list of chat invite link counts

    Parameters:
        invite_link_counts (List[:class:`~pytdbot.types.ChatInviteLinkCount`]):
            List of invite link counts

    """

    def __init__(self, invite_link_counts: List[ChatInviteLinkCount] = None) -> None:
        self.invite_link_counts: List[ChatInviteLinkCount] = invite_link_counts or []
        r"""List of invite link counts"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinkCounts"]:
        return "chatInviteLinkCounts"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinkCounts"]:
        return "ChatInviteLinkCounts"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "invite_link_counts": self.invite_link_counts}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinkCounts", None]:
        if data:
            data_class = cls()
            data_class.invite_link_counts = data.get("invite_link_counts", None)

        return data_class


class ChatInviteLinkMember(TlObject):
    r"""Describes a chat member joined a chat via an invite link

    Parameters:
        user_id (:class:`int`):
            User identifier

        joined_chat_date (:class:`int`):
            Point in time \(Unix timestamp\) when the user joined the chat

        via_chat_folder_invite_link (:class:`bool`):
            True, if the user has joined the chat using an invite link for a chat folder

        approver_user_id (:class:`int`):
            User identifier of the chat administrator, approved user join request

    """

    def __init__(
        self,
        user_id: int = 0,
        joined_chat_date: int = 0,
        via_chat_folder_invite_link: bool = False,
        approver_user_id: int = 0,
    ) -> None:
        self.user_id: int = int(user_id)
        r"""User identifier"""
        self.joined_chat_date: int = int(joined_chat_date)
        r"""Point in time \(Unix timestamp\) when the user joined the chat"""
        self.via_chat_folder_invite_link: bool = bool(via_chat_folder_invite_link)
        r"""True, if the user has joined the chat using an invite link for a chat folder"""
        self.approver_user_id: int = int(approver_user_id)
        r"""User identifier of the chat administrator, approved user join request"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinkMember"]:
        return "chatInviteLinkMember"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinkMember"]:
        return "ChatInviteLinkMember"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "joined_chat_date": self.joined_chat_date,
            "via_chat_folder_invite_link": self.via_chat_folder_invite_link,
            "approver_user_id": self.approver_user_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinkMember", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.joined_chat_date = int(data.get("joined_chat_date", 0))
            data_class.via_chat_folder_invite_link = data.get(
                "via_chat_folder_invite_link", False
            )
            data_class.approver_user_id = int(data.get("approver_user_id", 0))

        return data_class


class ChatInviteLinkMembers(TlObject):
    r"""Contains a list of chat members joined a chat via an invite link

    Parameters:
        total_count (:class:`int`):
            Approximate total number of chat members found

        members (List[:class:`~pytdbot.types.ChatInviteLinkMember`]):
            List of chat members, joined a chat via an invite link

    """

    def __init__(
        self, total_count: int = 0, members: List[ChatInviteLinkMember] = None
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of chat members found"""
        self.members: List[ChatInviteLinkMember] = members or []
        r"""List of chat members, joined a chat via an invite link"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinkMembers"]:
        return "chatInviteLinkMembers"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinkMembers"]:
        return "ChatInviteLinkMembers"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "members": self.members,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinkMembers", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.members = data.get("members", None)

        return data_class


class InviteLinkChatTypeBasicGroup(TlObject, InviteLinkChatType):
    r"""The link is an invite link for a basic group"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inviteLinkChatTypeBasicGroup"]:
        return "inviteLinkChatTypeBasicGroup"

    @classmethod
    def getClass(self) -> Literal["InviteLinkChatType"]:
        return "InviteLinkChatType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InviteLinkChatTypeBasicGroup", None]:
        if data:
            data_class = cls()

        return data_class


class InviteLinkChatTypeSupergroup(TlObject, InviteLinkChatType):
    r"""The link is an invite link for a supergroup"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inviteLinkChatTypeSupergroup"]:
        return "inviteLinkChatTypeSupergroup"

    @classmethod
    def getClass(self) -> Literal["InviteLinkChatType"]:
        return "InviteLinkChatType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InviteLinkChatTypeSupergroup", None]:
        if data:
            data_class = cls()

        return data_class


class InviteLinkChatTypeChannel(TlObject, InviteLinkChatType):
    r"""The link is an invite link for a channel"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inviteLinkChatTypeChannel"]:
        return "inviteLinkChatTypeChannel"

    @classmethod
    def getClass(self) -> Literal["InviteLinkChatType"]:
        return "InviteLinkChatType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InviteLinkChatTypeChannel", None]:
        if data:
            data_class = cls()

        return data_class


class ChatInviteLinkSubscriptionInfo(TlObject):
    r"""Contains information about subscription plan that must be paid by the user to use a chat invite link

    Parameters:
        pricing (:class:`~pytdbot.types.StarSubscriptionPricing`):
            Information about subscription plan that must be paid by the user to use the link

        can_reuse (:class:`bool`):
            True, if the user has already paid for the subscription and can use joinChatByInviteLink to join the subscribed chat again

        form_id (:class:`int`):
            Identifier of the payment form to use for subscription payment; 0 if the subscription can't be paid

    """

    def __init__(
        self,
        pricing: StarSubscriptionPricing = None,
        can_reuse: bool = False,
        form_id: int = 0,
    ) -> None:
        self.pricing: Union[StarSubscriptionPricing, None] = pricing
        r"""Information about subscription plan that must be paid by the user to use the link"""
        self.can_reuse: bool = bool(can_reuse)
        r"""True, if the user has already paid for the subscription and can use joinChatByInviteLink to join the subscribed chat again"""
        self.form_id: int = int(form_id)
        r"""Identifier of the payment form to use for subscription payment; 0 if the subscription can't be paid"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinkSubscriptionInfo"]:
        return "chatInviteLinkSubscriptionInfo"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinkSubscriptionInfo"]:
        return "ChatInviteLinkSubscriptionInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "pricing": self.pricing,
            "can_reuse": self.can_reuse,
            "form_id": self.form_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinkSubscriptionInfo", None]:
        if data:
            data_class = cls()
            data_class.pricing = data.get("pricing", None)
            data_class.can_reuse = data.get("can_reuse", False)
            data_class.form_id = int(data.get("form_id", 0))

        return data_class


class ChatInviteLinkInfo(TlObject):
    r"""Contains information about a chat invite link

    Parameters:
        chat_id (:class:`int`):
            Chat identifier of the invite link; 0 if the user has no access to the chat before joining

        accessible_for (:class:`int`):
            If non\-zero, the amount of time for which read access to the chat will remain available, in seconds

        type (:class:`~pytdbot.types.InviteLinkChatType`):
            Type of the chat

        title (:class:`str`):
            Title of the chat

        photo (:class:`~pytdbot.types.ChatPhotoInfo`):
            Chat photo; may be null

        accent_color_id (:class:`int`):
            Identifier of the accent color for chat title and background of chat photo

        description (:class:`str`):
            Chat description

        member_count (:class:`int`):
            Number of members in the chat

        member_user_ids (List[:class:`int`]):
            User identifiers of some chat members that may be known to the current user

        subscription_info (:class:`~pytdbot.types.ChatInviteLinkSubscriptionInfo`):
            Information about subscription plan that must be paid by the user to use the link; may be null if the link doesn't require subscription

        creates_join_request (:class:`bool`):
            True, if the link only creates join request

        is_public (:class:`bool`):
            True, if the chat is a public supergroup or channel, i\.e\. it has a username or it is a location\-based supergroup

        verification_status (:class:`~pytdbot.types.VerificationStatus`):
            Information about verification status of the chat; may be null if none

    """

    def __init__(
        self,
        chat_id: int = 0,
        accessible_for: int = 0,
        type: InviteLinkChatType = None,
        title: str = "",
        photo: ChatPhotoInfo = None,
        accent_color_id: int = 0,
        description: str = "",
        member_count: int = 0,
        member_user_ids: List[int] = None,
        subscription_info: ChatInviteLinkSubscriptionInfo = None,
        creates_join_request: bool = False,
        is_public: bool = False,
        verification_status: VerificationStatus = None,
    ) -> None:
        self.chat_id: int = int(chat_id)
        r"""Chat identifier of the invite link; 0 if the user has no access to the chat before joining"""
        self.accessible_for: int = int(accessible_for)
        r"""If non\-zero, the amount of time for which read access to the chat will remain available, in seconds"""
        self.type: Union[
            InviteLinkChatTypeBasicGroup,
            InviteLinkChatTypeSupergroup,
            InviteLinkChatTypeChannel,
            None,
        ] = type
        r"""Type of the chat"""
        self.title: Union[str, None] = title
        r"""Title of the chat"""
        self.photo: Union[ChatPhotoInfo, None] = photo
        r"""Chat photo; may be null"""
        self.accent_color_id: int = int(accent_color_id)
        r"""Identifier of the accent color for chat title and background of chat photo"""
        self.description: Union[str, None] = description
        r"""Chat description"""
        self.member_count: int = int(member_count)
        r"""Number of members in the chat"""
        self.member_user_ids: List[int] = member_user_ids or []
        r"""User identifiers of some chat members that may be known to the current user"""
        self.subscription_info: Union[ChatInviteLinkSubscriptionInfo, None] = (
            subscription_info
        )
        r"""Information about subscription plan that must be paid by the user to use the link; may be null if the link doesn't require subscription"""
        self.creates_join_request: bool = bool(creates_join_request)
        r"""True, if the link only creates join request"""
        self.is_public: bool = bool(is_public)
        r"""True, if the chat is a public supergroup or channel, i\.e\. it has a username or it is a location\-based supergroup"""
        self.verification_status: Union[VerificationStatus, None] = verification_status
        r"""Information about verification status of the chat; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatInviteLinkInfo"]:
        return "chatInviteLinkInfo"

    @classmethod
    def getClass(self) -> Literal["ChatInviteLinkInfo"]:
        return "ChatInviteLinkInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "accessible_for": self.accessible_for,
            "type": self.type,
            "title": self.title,
            "photo": self.photo,
            "accent_color_id": self.accent_color_id,
            "description": self.description,
            "member_count": self.member_count,
            "member_user_ids": self.member_user_ids,
            "subscription_info": self.subscription_info,
            "creates_join_request": self.creates_join_request,
            "is_public": self.is_public,
            "verification_status": self.verification_status,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatInviteLinkInfo", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.accessible_for = int(data.get("accessible_for", 0))
            data_class.type = data.get("type", None)
            data_class.title = data.get("title", "")
            data_class.photo = data.get("photo", None)
            data_class.accent_color_id = int(data.get("accent_color_id", 0))
            data_class.description = data.get("description", "")
            data_class.member_count = int(data.get("member_count", 0))
            data_class.member_user_ids = data.get("member_user_ids", None)
            data_class.subscription_info = data.get("subscription_info", None)
            data_class.creates_join_request = data.get("creates_join_request", False)
            data_class.is_public = data.get("is_public", False)
            data_class.verification_status = data.get("verification_status", None)

        return data_class


class ChatJoinRequest(TlObject):
    r"""Describes a user that sent a join request and waits for administrator approval

    Parameters:
        user_id (:class:`int`):
            User identifier

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the user sent the join request

        bio (:class:`str`):
            A short bio of the user

    """

    def __init__(self, user_id: int = 0, date: int = 0, bio: str = "") -> None:
        self.user_id: int = int(user_id)
        r"""User identifier"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the user sent the join request"""
        self.bio: Union[str, None] = bio
        r"""A short bio of the user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatJoinRequest"]:
        return "chatJoinRequest"

    @classmethod
    def getClass(self) -> Literal["ChatJoinRequest"]:
        return "ChatJoinRequest"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "date": self.date,
            "bio": self.bio,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatJoinRequest", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.date = int(data.get("date", 0))
            data_class.bio = data.get("bio", "")

        return data_class


class ChatJoinRequests(TlObject):
    r"""Contains a list of requests to join a chat

    Parameters:
        total_count (:class:`int`):
            Approximate total number of requests found

        requests (List[:class:`~pytdbot.types.ChatJoinRequest`]):
            List of the requests

    """

    def __init__(
        self, total_count: int = 0, requests: List[ChatJoinRequest] = None
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of requests found"""
        self.requests: List[ChatJoinRequest] = requests or []
        r"""List of the requests"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatJoinRequests"]:
        return "chatJoinRequests"

    @classmethod
    def getClass(self) -> Literal["ChatJoinRequests"]:
        return "ChatJoinRequests"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "requests": self.requests,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatJoinRequests", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.requests = data.get("requests", None)

        return data_class


class ChatJoinRequestsInfo(TlObject):
    r"""Contains information about pending join requests for a chat

    Parameters:
        total_count (:class:`int`):
            Total number of pending join requests

        user_ids (List[:class:`int`]):
            Identifiers of at most 3 users sent the newest pending join requests

    """

    def __init__(self, total_count: int = 0, user_ids: List[int] = None) -> None:
        self.total_count: int = int(total_count)
        r"""Total number of pending join requests"""
        self.user_ids: List[int] = user_ids or []
        r"""Identifiers of at most 3 users sent the newest pending join requests"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatJoinRequestsInfo"]:
        return "chatJoinRequestsInfo"

    @classmethod
    def getClass(self) -> Literal["ChatJoinRequestsInfo"]:
        return "ChatJoinRequestsInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "user_ids": self.user_ids,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatJoinRequestsInfo", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.user_ids = data.get("user_ids", None)

        return data_class


class BasicGroup(TlObject):
    r"""Represents a basic group of 0\-200 users \(must be upgraded to a supergroup to accommodate more than 200 users\)

    Parameters:
        id (:class:`int`):
            Group identifier

        member_count (:class:`int`):
            Number of members in the group

        status (:class:`~pytdbot.types.ChatMemberStatus`):
            Status of the current user in the group

        is_active (:class:`bool`):
            True, if the group is active

        upgraded_to_supergroup_id (:class:`int`):
            Identifier of the supergroup to which this group was upgraded; 0 if none

    """

    def __init__(
        self,
        id: int = 0,
        member_count: int = 0,
        status: ChatMemberStatus = None,
        is_active: bool = False,
        upgraded_to_supergroup_id: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Group identifier"""
        self.member_count: int = int(member_count)
        r"""Number of members in the group"""
        self.status: Union[
            ChatMemberStatusCreator,
            ChatMemberStatusAdministrator,
            ChatMemberStatusMember,
            ChatMemberStatusRestricted,
            ChatMemberStatusLeft,
            ChatMemberStatusBanned,
            None,
        ] = status
        r"""Status of the current user in the group"""
        self.is_active: bool = bool(is_active)
        r"""True, if the group is active"""
        self.upgraded_to_supergroup_id: int = int(upgraded_to_supergroup_id)
        r"""Identifier of the supergroup to which this group was upgraded; 0 if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["basicGroup"]:
        return "basicGroup"

    @classmethod
    def getClass(self) -> Literal["BasicGroup"]:
        return "BasicGroup"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "member_count": self.member_count,
            "status": self.status,
            "is_active": self.is_active,
            "upgraded_to_supergroup_id": self.upgraded_to_supergroup_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BasicGroup", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.member_count = int(data.get("member_count", 0))
            data_class.status = data.get("status", None)
            data_class.is_active = data.get("is_active", False)
            data_class.upgraded_to_supergroup_id = int(
                data.get("upgraded_to_supergroup_id", 0)
            )

        return data_class


class BasicGroupFullInfo(TlObject):
    r"""Contains full information about a basic group

    Parameters:
        photo (:class:`~pytdbot.types.ChatPhoto`):
            Chat photo; may be null if empty or unknown\. If non\-null, then it is the same photo as in chat\.photo

        description (:class:`str`):
            Group description\. Updated only after the basic group is opened

        creator_user_id (:class:`int`):
            User identifier of the creator of the group; 0 if unknown

        members (List[:class:`~pytdbot.types.ChatMember`]):
            Group members

        can_hide_members (:class:`bool`):
            True, if non\-administrators and non\-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non\-administrators after upgrading the basic group to a supergroup

        can_toggle_aggressive_anti_spam (:class:`bool`):
            True, if aggressive anti\-spam checks can be enabled or disabled in the supergroup after upgrading the basic group to a supergroup

        invite_link (:class:`~pytdbot.types.ChatInviteLink`):
            Primary invite link for this group; may be null\. For chat administrators with can\_invite\_users right only\. Updated only after the basic group is opened

        bot_commands (List[:class:`~pytdbot.types.BotCommands`]):
            List of commands of bots in the group

    """

    def __init__(
        self,
        photo: ChatPhoto = None,
        description: str = "",
        creator_user_id: int = 0,
        members: List[ChatMember] = None,
        can_hide_members: bool = False,
        can_toggle_aggressive_anti_spam: bool = False,
        invite_link: ChatInviteLink = None,
        bot_commands: List[BotCommands] = None,
    ) -> None:
        self.photo: Union[ChatPhoto, None] = photo
        r"""Chat photo; may be null if empty or unknown\. If non\-null, then it is the same photo as in chat\.photo"""
        self.description: Union[str, None] = description
        r"""Group description\. Updated only after the basic group is opened"""
        self.creator_user_id: int = int(creator_user_id)
        r"""User identifier of the creator of the group; 0 if unknown"""
        self.members: List[ChatMember] = members or []
        r"""Group members"""
        self.can_hide_members: bool = bool(can_hide_members)
        r"""True, if non\-administrators and non\-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non\-administrators after upgrading the basic group to a supergroup"""
        self.can_toggle_aggressive_anti_spam: bool = bool(
            can_toggle_aggressive_anti_spam
        )
        r"""True, if aggressive anti\-spam checks can be enabled or disabled in the supergroup after upgrading the basic group to a supergroup"""
        self.invite_link: Union[ChatInviteLink, None] = invite_link
        r"""Primary invite link for this group; may be null\. For chat administrators with can\_invite\_users right only\. Updated only after the basic group is opened"""
        self.bot_commands: List[BotCommands] = bot_commands or []
        r"""List of commands of bots in the group"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["basicGroupFullInfo"]:
        return "basicGroupFullInfo"

    @classmethod
    def getClass(self) -> Literal["BasicGroupFullInfo"]:
        return "BasicGroupFullInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "photo": self.photo,
            "description": self.description,
            "creator_user_id": self.creator_user_id,
            "members": self.members,
            "can_hide_members": self.can_hide_members,
            "can_toggle_aggressive_anti_spam": self.can_toggle_aggressive_anti_spam,
            "invite_link": self.invite_link,
            "bot_commands": self.bot_commands,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BasicGroupFullInfo", None]:
        if data:
            data_class = cls()
            data_class.photo = data.get("photo", None)
            data_class.description = data.get("description", "")
            data_class.creator_user_id = int(data.get("creator_user_id", 0))
            data_class.members = data.get("members", None)
            data_class.can_hide_members = data.get("can_hide_members", False)
            data_class.can_toggle_aggressive_anti_spam = data.get(
                "can_toggle_aggressive_anti_spam", False
            )
            data_class.invite_link = data.get("invite_link", None)
            data_class.bot_commands = data.get("bot_commands", None)

        return data_class


class Supergroup(TlObject):
    r"""Represents a supergroup or channel with zero or more members \(subscribers in the case of channels\)\. From the point of view of the system, a channel is a special kind of a supergroup: only administrators can post and see the list of members, and posts from all administrators use the name and photo of the channel instead of individual names and profile photos\. Unlike supergroups, channels can have an unlimited number of subscribers

    Parameters:
        id (:class:`int`):
            Supergroup or channel identifier

        usernames (:class:`~pytdbot.types.Usernames`):
            Usernames of the supergroup or channel; may be null

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the current user joined, or the point in time when the supergroup or channel was created, in case the user is not a member

        status (:class:`~pytdbot.types.ChatMemberStatus`):
            Status of the current user in the supergroup or channel; custom title will always be empty

        member_count (:class:`int`):
            Number of members in the supergroup or channel; 0 if unknown\. Currently, it is guaranteed to be known only if the supergroup or channel was received through getChatSimilarChats, getChatsToPostStories, getCreatedPublicChats, getGroupsInCommon, getInactiveSupergroupChats, getRecommendedChats, getSuitableDiscussionChats, getUserPrivacySettingRules, getVideoChatAvailableParticipants, searchPublicChats, or in chatFolderInviteLinkInfo\.missing\_chat\_ids, or in userFullInfo\.personal\_chat\_id, or for chats with messages or stories from publicForwards and foundStories

        boost_level (:class:`int`):
            Approximate boost level for the chat

        has_automatic_translation (:class:`bool`):
            True, if automatic translation of messages is enabled in the channel

        has_linked_chat (:class:`bool`):
            True, if the channel has a discussion group, or the supergroup is the designated discussion group for a channel

        has_location (:class:`bool`):
            True, if the supergroup is connected to a location, i\.e\. the supergroup is a location\-based supergroup

        sign_messages (:class:`bool`):
            True, if messages sent to the channel contains name of the sender\. This field is only applicable to channels

        show_message_sender (:class:`bool`):
            True, if messages sent to the channel have information about the sender user\. This field is only applicable to channels

        join_to_send_messages (:class:`bool`):
            True, if users need to join the supergroup before they can send messages\. May be false only for discussion supergroups and channel direct messages groups

        join_by_request (:class:`bool`):
            True, if all users directly joining the supergroup need to be approved by supergroup administrators\. Can be true only for non\-broadcast supergroups with username, location, or a linked chat

        is_slow_mode_enabled (:class:`bool`):
            True, if the slow mode is enabled in the supergroup

        is_channel (:class:`bool`):
            True, if the supergroup is a channel

        is_broadcast_group (:class:`bool`):
            True, if the supergroup is a broadcast group, i\.e\. only administrators can send messages and there is no limit on the number of members

        is_forum (:class:`bool`):
            True, if the supergroup is a forum with topics

        is_direct_messages_group (:class:`bool`):
            True, if the supergroup is a direct message group for a channel chat

        is_administered_direct_messages_group (:class:`bool`):
            True, if the supergroup is a direct messages group for a channel chat that is administered by the current user

        verification_status (:class:`~pytdbot.types.VerificationStatus`):
            Information about verification status of the supergroup or channel; may be null if none

        has_direct_messages_group (:class:`bool`):
            True, if the channel has direct messages group

        has_forum_tabs (:class:`bool`):
            True, if the supergroup is a forum, which topics are shown in the same way as in channel direct messages groups

        restriction_info (:class:`~pytdbot.types.RestrictionInfo`):
            Information about the restrictions that must be applied to the corresponding supergroup or channel chat; may be null if none

        paid_message_star_count (:class:`int`):
            Number of Telegram Stars that must be paid by non\-administrator users of the supergroup chat for each sent message

        has_active_stories (:class:`bool`):
            True, if the supergroup or channel has non\-expired stories available to the current user

        has_unread_active_stories (:class:`bool`):
            True, if the supergroup or channel has unread non\-expired stories available to the current user

    """

    def __init__(
        self,
        id: int = 0,
        usernames: Usernames = None,
        date: int = 0,
        status: ChatMemberStatus = None,
        member_count: int = 0,
        boost_level: int = 0,
        has_automatic_translation: bool = False,
        has_linked_chat: bool = False,
        has_location: bool = False,
        sign_messages: bool = False,
        show_message_sender: bool = False,
        join_to_send_messages: bool = False,
        join_by_request: bool = False,
        is_slow_mode_enabled: bool = False,
        is_channel: bool = False,
        is_broadcast_group: bool = False,
        is_forum: bool = False,
        is_direct_messages_group: bool = False,
        is_administered_direct_messages_group: bool = False,
        verification_status: VerificationStatus = None,
        has_direct_messages_group: bool = False,
        has_forum_tabs: bool = False,
        restriction_info: RestrictionInfo = None,
        paid_message_star_count: int = 0,
        has_active_stories: bool = False,
        has_unread_active_stories: bool = False,
    ) -> None:
        self.id: int = int(id)
        r"""Supergroup or channel identifier"""
        self.usernames: Union[Usernames, None] = usernames
        r"""Usernames of the supergroup or channel; may be null"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the current user joined, or the point in time when the supergroup or channel was created, in case the user is not a member"""
        self.status: Union[
            ChatMemberStatusCreator,
            ChatMemberStatusAdministrator,
            ChatMemberStatusMember,
            ChatMemberStatusRestricted,
            ChatMemberStatusLeft,
            ChatMemberStatusBanned,
            None,
        ] = status
        r"""Status of the current user in the supergroup or channel; custom title will always be empty"""
        self.member_count: int = int(member_count)
        r"""Number of members in the supergroup or channel; 0 if unknown\. Currently, it is guaranteed to be known only if the supergroup or channel was received through getChatSimilarChats, getChatsToPostStories, getCreatedPublicChats, getGroupsInCommon, getInactiveSupergroupChats, getRecommendedChats, getSuitableDiscussionChats, getUserPrivacySettingRules, getVideoChatAvailableParticipants, searchPublicChats, or in chatFolderInviteLinkInfo\.missing\_chat\_ids, or in userFullInfo\.personal\_chat\_id, or for chats with messages or stories from publicForwards and foundStories"""
        self.boost_level: int = int(boost_level)
        r"""Approximate boost level for the chat"""
        self.has_automatic_translation: bool = bool(has_automatic_translation)
        r"""True, if automatic translation of messages is enabled in the channel"""
        self.has_linked_chat: bool = bool(has_linked_chat)
        r"""True, if the channel has a discussion group, or the supergroup is the designated discussion group for a channel"""
        self.has_location: bool = bool(has_location)
        r"""True, if the supergroup is connected to a location, i\.e\. the supergroup is a location\-based supergroup"""
        self.sign_messages: bool = bool(sign_messages)
        r"""True, if messages sent to the channel contains name of the sender\. This field is only applicable to channels"""
        self.show_message_sender: bool = bool(show_message_sender)
        r"""True, if messages sent to the channel have information about the sender user\. This field is only applicable to channels"""
        self.join_to_send_messages: bool = bool(join_to_send_messages)
        r"""True, if users need to join the supergroup before they can send messages\. May be false only for discussion supergroups and channel direct messages groups"""
        self.join_by_request: bool = bool(join_by_request)
        r"""True, if all users directly joining the supergroup need to be approved by supergroup administrators\. Can be true only for non\-broadcast supergroups with username, location, or a linked chat"""
        self.is_slow_mode_enabled: bool = bool(is_slow_mode_enabled)
        r"""True, if the slow mode is enabled in the supergroup"""
        self.is_channel: bool = bool(is_channel)
        r"""True, if the supergroup is a channel"""
        self.is_broadcast_group: bool = bool(is_broadcast_group)
        r"""True, if the supergroup is a broadcast group, i\.e\. only administrators can send messages and there is no limit on the number of members"""
        self.is_forum: bool = bool(is_forum)
        r"""True, if the supergroup is a forum with topics"""
        self.is_direct_messages_group: bool = bool(is_direct_messages_group)
        r"""True, if the supergroup is a direct message group for a channel chat"""
        self.is_administered_direct_messages_group: bool = bool(
            is_administered_direct_messages_group
        )
        r"""True, if the supergroup is a direct messages group for a channel chat that is administered by the current user"""
        self.verification_status: Union[VerificationStatus, None] = verification_status
        r"""Information about verification status of the supergroup or channel; may be null if none"""
        self.has_direct_messages_group: bool = bool(has_direct_messages_group)
        r"""True, if the channel has direct messages group"""
        self.has_forum_tabs: bool = bool(has_forum_tabs)
        r"""True, if the supergroup is a forum, which topics are shown in the same way as in channel direct messages groups"""
        self.restriction_info: Union[RestrictionInfo, None] = restriction_info
        r"""Information about the restrictions that must be applied to the corresponding supergroup or channel chat; may be null if none"""
        self.paid_message_star_count: int = int(paid_message_star_count)
        r"""Number of Telegram Stars that must be paid by non\-administrator users of the supergroup chat for each sent message"""
        self.has_active_stories: bool = bool(has_active_stories)
        r"""True, if the supergroup or channel has non\-expired stories available to the current user"""
        self.has_unread_active_stories: bool = bool(has_unread_active_stories)
        r"""True, if the supergroup or channel has unread non\-expired stories available to the current user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroup"]:
        return "supergroup"

    @classmethod
    def getClass(self) -> Literal["Supergroup"]:
        return "Supergroup"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "usernames": self.usernames,
            "date": self.date,
            "status": self.status,
            "member_count": self.member_count,
            "boost_level": self.boost_level,
            "has_automatic_translation": self.has_automatic_translation,
            "has_linked_chat": self.has_linked_chat,
            "has_location": self.has_location,
            "sign_messages": self.sign_messages,
            "show_message_sender": self.show_message_sender,
            "join_to_send_messages": self.join_to_send_messages,
            "join_by_request": self.join_by_request,
            "is_slow_mode_enabled": self.is_slow_mode_enabled,
            "is_channel": self.is_channel,
            "is_broadcast_group": self.is_broadcast_group,
            "is_forum": self.is_forum,
            "is_direct_messages_group": self.is_direct_messages_group,
            "is_administered_direct_messages_group": self.is_administered_direct_messages_group,
            "verification_status": self.verification_status,
            "has_direct_messages_group": self.has_direct_messages_group,
            "has_forum_tabs": self.has_forum_tabs,
            "restriction_info": self.restriction_info,
            "paid_message_star_count": self.paid_message_star_count,
            "has_active_stories": self.has_active_stories,
            "has_unread_active_stories": self.has_unread_active_stories,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Supergroup", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.usernames = data.get("usernames", None)
            data_class.date = int(data.get("date", 0))
            data_class.status = data.get("status", None)
            data_class.member_count = int(data.get("member_count", 0))
            data_class.boost_level = int(data.get("boost_level", 0))
            data_class.has_automatic_translation = data.get(
                "has_automatic_translation", False
            )
            data_class.has_linked_chat = data.get("has_linked_chat", False)
            data_class.has_location = data.get("has_location", False)
            data_class.sign_messages = data.get("sign_messages", False)
            data_class.show_message_sender = data.get("show_message_sender", False)
            data_class.join_to_send_messages = data.get("join_to_send_messages", False)
            data_class.join_by_request = data.get("join_by_request", False)
            data_class.is_slow_mode_enabled = data.get("is_slow_mode_enabled", False)
            data_class.is_channel = data.get("is_channel", False)
            data_class.is_broadcast_group = data.get("is_broadcast_group", False)
            data_class.is_forum = data.get("is_forum", False)
            data_class.is_direct_messages_group = data.get(
                "is_direct_messages_group", False
            )
            data_class.is_administered_direct_messages_group = data.get(
                "is_administered_direct_messages_group", False
            )
            data_class.verification_status = data.get("verification_status", None)
            data_class.has_direct_messages_group = data.get(
                "has_direct_messages_group", False
            )
            data_class.has_forum_tabs = data.get("has_forum_tabs", False)
            data_class.restriction_info = data.get("restriction_info", None)
            data_class.paid_message_star_count = int(
                data.get("paid_message_star_count", 0)
            )
            data_class.has_active_stories = data.get("has_active_stories", False)
            data_class.has_unread_active_stories = data.get(
                "has_unread_active_stories", False
            )

        return data_class


class SupergroupFullInfo(TlObject):
    r"""Contains full information about a supergroup or channel

    Parameters:
        photo (:class:`~pytdbot.types.ChatPhoto`):
            Chat photo; may be null if empty or unknown\. If non\-null, then it is the same photo as in chat\.photo

        description (:class:`str`):
            Supergroup or channel description

        member_count (:class:`int`):
            Number of members in the supergroup or channel; 0 if unknown

        administrator_count (:class:`int`):
            Number of privileged users in the supergroup or channel; 0 if unknown

        restricted_count (:class:`int`):
            Number of restricted users in the supergroup; 0 if unknown

        banned_count (:class:`int`):
            Number of users banned from chat; 0 if unknown

        linked_chat_id (:class:`int`):
            Chat identifier of a discussion group for the channel, or a channel, for which the supergroup is the designated discussion group; 0 if none or unknown

        direct_messages_chat_id (:class:`int`):
            Chat identifier of a direct messages group for the channel, or a channel, for which the supergroup is the designated direct messages group; 0 if none

        slow_mode_delay (:class:`int`):
            Delay between consecutive sent messages for non\-administrator supergroup members, in seconds

        slow_mode_delay_expires_in (:class:`float`):
            Time left before next message can be sent in the supergroup, in seconds\. An updateSupergroupFullInfo update is not triggered when value of this field changes, but both new and old values are non\-zero

        can_enable_paid_messages (:class:`bool`):
            True, if paid messages can be enabled in the supergroup chat; for supergroup only

        can_enable_paid_reaction (:class:`bool`):
            True, if paid reaction can be enabled in the channel chat; for channels only

        can_get_members (:class:`bool`):
            True, if members of the chat can be retrieved via getSupergroupMembers or searchChatMembers

        has_hidden_members (:class:`bool`):
            True, if non\-administrators can receive only administrators and bots using getSupergroupMembers or searchChatMembers

        can_hide_members (:class:`bool`):
            True, if non\-administrators and non\-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non\-administrators

        can_set_sticker_set (:class:`bool`):
            True, if the supergroup sticker set can be changed

        can_set_location (:class:`bool`):
            True, if the supergroup location can be changed

        can_get_statistics (:class:`bool`):
            True, if the supergroup or channel statistics are available

        can_get_revenue_statistics (:class:`bool`):
            True, if the supergroup or channel revenue statistics are available

        can_get_star_revenue_statistics (:class:`bool`):
            True, if the supergroup or channel Telegram Star revenue statistics are available

        can_send_gift (:class:`bool`):
            True, if the user can send a gift to the supergroup or channel using sendGift or transferGift

        can_toggle_aggressive_anti_spam (:class:`bool`):
            True, if aggressive anti\-spam checks can be enabled or disabled in the supergroup

        is_all_history_available (:class:`bool`):
            True, if new chat members will have access to old messages\. In public, discussion, of forum groups and all channels, old messages are always available, so this option affects only private non\-forum supergroups without a linked chat\. The value of this field is only available to chat administrators

        can_have_sponsored_messages (:class:`bool`):
            True, if the chat can have sponsored messages\. The value of this field is only available to the owner of the chat

        has_aggressive_anti_spam_enabled (:class:`bool`):
            True, if aggressive anti\-spam checks are enabled in the supergroup\. The value of this field is only available to chat administrators

        has_paid_media_allowed (:class:`bool`):
            True, if paid media can be sent and forwarded to the channel chat; for channels only

        has_pinned_stories (:class:`bool`):
            True, if the supergroup or channel has pinned stories

        gift_count (:class:`int`):
            Number of saved to profile gifts for channels without can\_post\_messages administrator right, otherwise, the total number of received gifts

        my_boost_count (:class:`int`):
            Number of times the current user boosted the supergroup or channel

        unrestrict_boost_count (:class:`int`):
            Number of times the supergroup must be boosted by a user to ignore slow mode and chat permission restrictions; 0 if unspecified

        outgoing_paid_message_star_count (:class:`int`):
            Number of Telegram Stars that must be paid by the current user for each sent message to the supergroup

        sticker_set_id (:class:`int`):
            Identifier of the supergroup sticker set that must be shown before user sticker sets; 0 if none

        custom_emoji_sticker_set_id (:class:`int`):
            Identifier of the custom emoji sticker set that can be used in the supergroup without Telegram Premium subscription; 0 if none

        location (:class:`~pytdbot.types.ChatLocation`):
            Location to which the supergroup is connected; may be null if none

        invite_link (:class:`~pytdbot.types.ChatInviteLink`):
            Primary invite link for the chat; may be null\. For chat administrators with can\_invite\_users right only

        bot_commands (List[:class:`~pytdbot.types.BotCommands`]):
            List of commands of bots in the group

        bot_verification (:class:`~pytdbot.types.BotVerification`):
            Information about verification status of the supergroup or the channel provided by a bot; may be null if none or unknown

        main_profile_tab (:class:`~pytdbot.types.ProfileTab`):
            The main tab chosen by the administrators of the channel; may be null if not chosen manually

        upgraded_from_basic_group_id (:class:`int`):
            Identifier of the basic group from which supergroup was upgraded; 0 if none

        upgraded_from_max_message_id (:class:`int`):
            Identifier of the last message in the basic group from which supergroup was upgraded; 0 if none

    """

    def __init__(
        self,
        photo: ChatPhoto = None,
        description: str = "",
        member_count: int = 0,
        administrator_count: int = 0,
        restricted_count: int = 0,
        banned_count: int = 0,
        linked_chat_id: int = 0,
        direct_messages_chat_id: int = 0,
        slow_mode_delay: int = 0,
        slow_mode_delay_expires_in: float = 0.0,
        can_enable_paid_messages: bool = False,
        can_enable_paid_reaction: bool = False,
        can_get_members: bool = False,
        has_hidden_members: bool = False,
        can_hide_members: bool = False,
        can_set_sticker_set: bool = False,
        can_set_location: bool = False,
        can_get_statistics: bool = False,
        can_get_revenue_statistics: bool = False,
        can_get_star_revenue_statistics: bool = False,
        can_send_gift: bool = False,
        can_toggle_aggressive_anti_spam: bool = False,
        is_all_history_available: bool = False,
        can_have_sponsored_messages: bool = False,
        has_aggressive_anti_spam_enabled: bool = False,
        has_paid_media_allowed: bool = False,
        has_pinned_stories: bool = False,
        gift_count: int = 0,
        my_boost_count: int = 0,
        unrestrict_boost_count: int = 0,
        outgoing_paid_message_star_count: int = 0,
        sticker_set_id: int = 0,
        custom_emoji_sticker_set_id: int = 0,
        location: ChatLocation = None,
        invite_link: ChatInviteLink = None,
        bot_commands: List[BotCommands] = None,
        bot_verification: BotVerification = None,
        main_profile_tab: ProfileTab = None,
        upgraded_from_basic_group_id: int = 0,
        upgraded_from_max_message_id: int = 0,
    ) -> None:
        self.photo: Union[ChatPhoto, None] = photo
        r"""Chat photo; may be null if empty or unknown\. If non\-null, then it is the same photo as in chat\.photo"""
        self.description: Union[str, None] = description
        r"""Supergroup or channel description"""
        self.member_count: int = int(member_count)
        r"""Number of members in the supergroup or channel; 0 if unknown"""
        self.administrator_count: int = int(administrator_count)
        r"""Number of privileged users in the supergroup or channel; 0 if unknown"""
        self.restricted_count: int = int(restricted_count)
        r"""Number of restricted users in the supergroup; 0 if unknown"""
        self.banned_count: int = int(banned_count)
        r"""Number of users banned from chat; 0 if unknown"""
        self.linked_chat_id: int = int(linked_chat_id)
        r"""Chat identifier of a discussion group for the channel, or a channel, for which the supergroup is the designated discussion group; 0 if none or unknown"""
        self.direct_messages_chat_id: int = int(direct_messages_chat_id)
        r"""Chat identifier of a direct messages group for the channel, or a channel, for which the supergroup is the designated direct messages group; 0 if none"""
        self.slow_mode_delay: int = int(slow_mode_delay)
        r"""Delay between consecutive sent messages for non\-administrator supergroup members, in seconds"""
        self.slow_mode_delay_expires_in: float = float(slow_mode_delay_expires_in)
        r"""Time left before next message can be sent in the supergroup, in seconds\. An updateSupergroupFullInfo update is not triggered when value of this field changes, but both new and old values are non\-zero"""
        self.can_enable_paid_messages: bool = bool(can_enable_paid_messages)
        r"""True, if paid messages can be enabled in the supergroup chat; for supergroup only"""
        self.can_enable_paid_reaction: bool = bool(can_enable_paid_reaction)
        r"""True, if paid reaction can be enabled in the channel chat; for channels only"""
        self.can_get_members: bool = bool(can_get_members)
        r"""True, if members of the chat can be retrieved via getSupergroupMembers or searchChatMembers"""
        self.has_hidden_members: bool = bool(has_hidden_members)
        r"""True, if non\-administrators can receive only administrators and bots using getSupergroupMembers or searchChatMembers"""
        self.can_hide_members: bool = bool(can_hide_members)
        r"""True, if non\-administrators and non\-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non\-administrators"""
        self.can_set_sticker_set: bool = bool(can_set_sticker_set)
        r"""True, if the supergroup sticker set can be changed"""
        self.can_set_location: bool = bool(can_set_location)
        r"""True, if the supergroup location can be changed"""
        self.can_get_statistics: bool = bool(can_get_statistics)
        r"""True, if the supergroup or channel statistics are available"""
        self.can_get_revenue_statistics: bool = bool(can_get_revenue_statistics)
        r"""True, if the supergroup or channel revenue statistics are available"""
        self.can_get_star_revenue_statistics: bool = bool(
            can_get_star_revenue_statistics
        )
        r"""True, if the supergroup or channel Telegram Star revenue statistics are available"""
        self.can_send_gift: bool = bool(can_send_gift)
        r"""True, if the user can send a gift to the supergroup or channel using sendGift or transferGift"""
        self.can_toggle_aggressive_anti_spam: bool = bool(
            can_toggle_aggressive_anti_spam
        )
        r"""True, if aggressive anti\-spam checks can be enabled or disabled in the supergroup"""
        self.is_all_history_available: bool = bool(is_all_history_available)
        r"""True, if new chat members will have access to old messages\. In public, discussion, of forum groups and all channels, old messages are always available, so this option affects only private non\-forum supergroups without a linked chat\. The value of this field is only available to chat administrators"""
        self.can_have_sponsored_messages: bool = bool(can_have_sponsored_messages)
        r"""True, if the chat can have sponsored messages\. The value of this field is only available to the owner of the chat"""
        self.has_aggressive_anti_spam_enabled: bool = bool(
            has_aggressive_anti_spam_enabled
        )
        r"""True, if aggressive anti\-spam checks are enabled in the supergroup\. The value of this field is only available to chat administrators"""
        self.has_paid_media_allowed: bool = bool(has_paid_media_allowed)
        r"""True, if paid media can be sent and forwarded to the channel chat; for channels only"""
        self.has_pinned_stories: bool = bool(has_pinned_stories)
        r"""True, if the supergroup or channel has pinned stories"""
        self.gift_count: int = int(gift_count)
        r"""Number of saved to profile gifts for channels without can\_post\_messages administrator right, otherwise, the total number of received gifts"""
        self.my_boost_count: int = int(my_boost_count)
        r"""Number of times the current user boosted the supergroup or channel"""
        self.unrestrict_boost_count: int = int(unrestrict_boost_count)
        r"""Number of times the supergroup must be boosted by a user to ignore slow mode and chat permission restrictions; 0 if unspecified"""
        self.outgoing_paid_message_star_count: int = int(
            outgoing_paid_message_star_count
        )
        r"""Number of Telegram Stars that must be paid by the current user for each sent message to the supergroup"""
        self.sticker_set_id: int = int(sticker_set_id)
        r"""Identifier of the supergroup sticker set that must be shown before user sticker sets; 0 if none"""
        self.custom_emoji_sticker_set_id: int = int(custom_emoji_sticker_set_id)
        r"""Identifier of the custom emoji sticker set that can be used in the supergroup without Telegram Premium subscription; 0 if none"""
        self.location: Union[ChatLocation, None] = location
        r"""Location to which the supergroup is connected; may be null if none"""
        self.invite_link: Union[ChatInviteLink, None] = invite_link
        r"""Primary invite link for the chat; may be null\. For chat administrators with can\_invite\_users right only"""
        self.bot_commands: List[BotCommands] = bot_commands or []
        r"""List of commands of bots in the group"""
        self.bot_verification: Union[BotVerification, None] = bot_verification
        r"""Information about verification status of the supergroup or the channel provided by a bot; may be null if none or unknown"""
        self.main_profile_tab: Union[
            ProfileTabPosts,
            ProfileTabGifts,
            ProfileTabMedia,
            ProfileTabFiles,
            ProfileTabLinks,
            ProfileTabMusic,
            ProfileTabVoice,
            ProfileTabGifs,
            None,
        ] = main_profile_tab
        r"""The main tab chosen by the administrators of the channel; may be null if not chosen manually"""
        self.upgraded_from_basic_group_id: int = int(upgraded_from_basic_group_id)
        r"""Identifier of the basic group from which supergroup was upgraded; 0 if none"""
        self.upgraded_from_max_message_id: int = int(upgraded_from_max_message_id)
        r"""Identifier of the last message in the basic group from which supergroup was upgraded; 0 if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["supergroupFullInfo"]:
        return "supergroupFullInfo"

    @classmethod
    def getClass(self) -> Literal["SupergroupFullInfo"]:
        return "SupergroupFullInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "photo": self.photo,
            "description": self.description,
            "member_count": self.member_count,
            "administrator_count": self.administrator_count,
            "restricted_count": self.restricted_count,
            "banned_count": self.banned_count,
            "linked_chat_id": self.linked_chat_id,
            "direct_messages_chat_id": self.direct_messages_chat_id,
            "slow_mode_delay": self.slow_mode_delay,
            "slow_mode_delay_expires_in": self.slow_mode_delay_expires_in,
            "can_enable_paid_messages": self.can_enable_paid_messages,
            "can_enable_paid_reaction": self.can_enable_paid_reaction,
            "can_get_members": self.can_get_members,
            "has_hidden_members": self.has_hidden_members,
            "can_hide_members": self.can_hide_members,
            "can_set_sticker_set": self.can_set_sticker_set,
            "can_set_location": self.can_set_location,
            "can_get_statistics": self.can_get_statistics,
            "can_get_revenue_statistics": self.can_get_revenue_statistics,
            "can_get_star_revenue_statistics": self.can_get_star_revenue_statistics,
            "can_send_gift": self.can_send_gift,
            "can_toggle_aggressive_anti_spam": self.can_toggle_aggressive_anti_spam,
            "is_all_history_available": self.is_all_history_available,
            "can_have_sponsored_messages": self.can_have_sponsored_messages,
            "has_aggressive_anti_spam_enabled": self.has_aggressive_anti_spam_enabled,
            "has_paid_media_allowed": self.has_paid_media_allowed,
            "has_pinned_stories": self.has_pinned_stories,
            "gift_count": self.gift_count,
            "my_boost_count": self.my_boost_count,
            "unrestrict_boost_count": self.unrestrict_boost_count,
            "outgoing_paid_message_star_count": self.outgoing_paid_message_star_count,
            "sticker_set_id": self.sticker_set_id,
            "custom_emoji_sticker_set_id": self.custom_emoji_sticker_set_id,
            "location": self.location,
            "invite_link": self.invite_link,
            "bot_commands": self.bot_commands,
            "bot_verification": self.bot_verification,
            "main_profile_tab": self.main_profile_tab,
            "upgraded_from_basic_group_id": self.upgraded_from_basic_group_id,
            "upgraded_from_max_message_id": self.upgraded_from_max_message_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["SupergroupFullInfo", None]:
        if data:
            data_class = cls()
            data_class.photo = data.get("photo", None)
            data_class.description = data.get("description", "")
            data_class.member_count = int(data.get("member_count", 0))
            data_class.administrator_count = int(data.get("administrator_count", 0))
            data_class.restricted_count = int(data.get("restricted_count", 0))
            data_class.banned_count = int(data.get("banned_count", 0))
            data_class.linked_chat_id = int(data.get("linked_chat_id", 0))
            data_class.direct_messages_chat_id = int(
                data.get("direct_messages_chat_id", 0)
            )
            data_class.slow_mode_delay = int(data.get("slow_mode_delay", 0))
            data_class.slow_mode_delay_expires_in = data.get(
                "slow_mode_delay_expires_in", 0.0
            )
            data_class.can_enable_paid_messages = data.get(
                "can_enable_paid_messages", False
            )
            data_class.can_enable_paid_reaction = data.get(
                "can_enable_paid_reaction", False
            )
            data_class.can_get_members = data.get("can_get_members", False)
            data_class.has_hidden_members = data.get("has_hidden_members", False)
            data_class.can_hide_members = data.get("can_hide_members", False)
            data_class.can_set_sticker_set = data.get("can_set_sticker_set", False)
            data_class.can_set_location = data.get("can_set_location", False)
            data_class.can_get_statistics = data.get("can_get_statistics", False)
            data_class.can_get_revenue_statistics = data.get(
                "can_get_revenue_statistics", False
            )
            data_class.can_get_star_revenue_statistics = data.get(
                "can_get_star_revenue_statistics", False
            )
            data_class.can_send_gift = data.get("can_send_gift", False)
            data_class.can_toggle_aggressive_anti_spam = data.get(
                "can_toggle_aggressive_anti_spam", False
            )
            data_class.is_all_history_available = data.get(
                "is_all_history_available", False
            )
            data_class.can_have_sponsored_messages = data.get(
                "can_have_sponsored_messages", False
            )
            data_class.has_aggressive_anti_spam_enabled = data.get(
                "has_aggressive_anti_spam_enabled", False
            )
            data_class.has_paid_media_allowed = data.get(
                "has_paid_media_allowed", False
            )
            data_class.has_pinned_stories = data.get("has_pinned_stories", False)
            data_class.gift_count = int(data.get("gift_count", 0))
            data_class.my_boost_count = int(data.get("my_boost_count", 0))
            data_class.unrestrict_boost_count = int(
                data.get("unrestrict_boost_count", 0)
            )
            data_class.outgoing_paid_message_star_count = int(
                data.get("outgoing_paid_message_star_count", 0)
            )
            data_class.sticker_set_id = int(data.get("sticker_set_id", 0))
            data_class.custom_emoji_sticker_set_id = int(
                data.get("custom_emoji_sticker_set_id", 0)
            )
            data_class.location = data.get("location", None)
            data_class.invite_link = data.get("invite_link", None)
            data_class.bot_commands = data.get("bot_commands", None)
            data_class.bot_verification = data.get("bot_verification", None)
            data_class.main_profile_tab = data.get("main_profile_tab", None)
            data_class.upgraded_from_basic_group_id = int(
                data.get("upgraded_from_basic_group_id", 0)
            )
            data_class.upgraded_from_max_message_id = int(
                data.get("upgraded_from_max_message_id", 0)
            )

        return data_class


class SecretChatStatePending(TlObject, SecretChatState):
    r"""The secret chat is not yet created; waiting for the other user to get online"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["secretChatStatePending"]:
        return "secretChatStatePending"

    @classmethod
    def getClass(self) -> Literal["SecretChatState"]:
        return "SecretChatState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SecretChatStatePending", None]:
        if data:
            data_class = cls()

        return data_class


class SecretChatStateReady(TlObject, SecretChatState):
    r"""The secret chat is ready to use"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["secretChatStateReady"]:
        return "secretChatStateReady"

    @classmethod
    def getClass(self) -> Literal["SecretChatState"]:
        return "SecretChatState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SecretChatStateReady", None]:
        if data:
            data_class = cls()

        return data_class


class SecretChatStateClosed(TlObject, SecretChatState):
    r"""The secret chat is closed"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["secretChatStateClosed"]:
        return "secretChatStateClosed"

    @classmethod
    def getClass(self) -> Literal["SecretChatState"]:
        return "SecretChatState"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SecretChatStateClosed", None]:
        if data:
            data_class = cls()

        return data_class


class SecretChat(TlObject):
    r"""Represents a secret chat

    Parameters:
        id (:class:`int`):
            Secret chat identifier

        user_id (:class:`int`):
            Identifier of the chat partner

        state (:class:`~pytdbot.types.SecretChatState`):
            State of the secret chat

        is_outbound (:class:`bool`):
            True, if the chat was created by the current user; false otherwise

        key_hash (:class:`bytes`):
            Hash of the currently used key for comparison with the hash of the chat partner's key\. This is a string of 36 little\-endian bytes, which must be split into groups of 2 bits, each denoting a pixel of one of 4 colors FFFFFF, D5E6F3, 2D5775, and 2F99C9\. The pixels must be used to make a 12x12 square image filled from left to right, top to bottom\. Alternatively, the first 32 bytes of the hash can be converted to the hexadecimal format and printed as 32 2\-digit hex numbers

        layer (:class:`int`):
            Secret chat layer; determines features supported by the chat partner's application\. Nested text entities and underline and strikethrough entities are supported if the layer \>\= 101, files bigger than 2000MB are supported if the layer \>\= 143, spoiler and custom emoji text entities are supported if the layer \>\= 144

    """

    def __init__(
        self,
        id: int = 0,
        user_id: int = 0,
        state: SecretChatState = None,
        is_outbound: bool = False,
        key_hash: bytes = b"",
        layer: int = 0,
    ) -> None:
        self.id: int = int(id)
        r"""Secret chat identifier"""
        self.user_id: int = int(user_id)
        r"""Identifier of the chat partner"""
        self.state: Union[
            SecretChatStatePending, SecretChatStateReady, SecretChatStateClosed, None
        ] = state
        r"""State of the secret chat"""
        self.is_outbound: bool = bool(is_outbound)
        r"""True, if the chat was created by the current user; false otherwise"""
        self.key_hash: Union[bytes, None] = key_hash
        r"""Hash of the currently used key for comparison with the hash of the chat partner's key\. This is a string of 36 little\-endian bytes, which must be split into groups of 2 bits, each denoting a pixel of one of 4 colors FFFFFF, D5E6F3, 2D5775, and 2F99C9\. The pixels must be used to make a 12x12 square image filled from left to right, top to bottom\. Alternatively, the first 32 bytes of the hash can be converted to the hexadecimal format and printed as 32 2\-digit hex numbers"""
        self.layer: int = int(layer)
        r"""Secret chat layer; determines features supported by the chat partner's application\. Nested text entities and underline and strikethrough entities are supported if the layer \>\= 101, files bigger than 2000MB are supported if the layer \>\= 143, spoiler and custom emoji text entities are supported if the layer \>\= 144"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["secretChat"]:
        return "secretChat"

    @classmethod
    def getClass(self) -> Literal["SecretChat"]:
        return "SecretChat"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "user_id": self.user_id,
            "state": self.state,
            "is_outbound": self.is_outbound,
            "key_hash": self.key_hash,
            "layer": self.layer,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["SecretChat", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.user_id = int(data.get("user_id", 0))
            data_class.state = data.get("state", None)
            data_class.is_outbound = data.get("is_outbound", False)
            data_class.key_hash = b64decode(data.get("key_hash", b""))
            data_class.layer = int(data.get("layer", 0))

        return data_class


class PublicPostSearchLimits(TlObject):
    r"""Contains information about public post search limits

    Parameters:
        daily_free_query_count (:class:`int`):
            Number of queries that can be sent daily for free

        remaining_free_query_count (:class:`int`):
            Number of remaining free queries today

        next_free_query_in (:class:`int`):
            Amount of time till the next free query can be sent; 0 if it can be sent now

        star_count (:class:`int`):
            Number of Telegram Stars that must be paid for each non\-free query

        is_current_query_free (:class:`bool`):
            True, if the search for the specified query isn't charged

    """

    def __init__(
        self,
        daily_free_query_count: int = 0,
        remaining_free_query_count: int = 0,
        next_free_query_in: int = 0,
        star_count: int = 0,
        is_current_query_free: bool = False,
    ) -> None:
        self.daily_free_query_count: int = int(daily_free_query_count)
        r"""Number of queries that can be sent daily for free"""
        self.remaining_free_query_count: int = int(remaining_free_query_count)
        r"""Number of remaining free queries today"""
        self.next_free_query_in: int = int(next_free_query_in)
        r"""Amount of time till the next free query can be sent; 0 if it can be sent now"""
        self.star_count: int = int(star_count)
        r"""Number of Telegram Stars that must be paid for each non\-free query"""
        self.is_current_query_free: bool = bool(is_current_query_free)
        r"""True, if the search for the specified query isn't charged"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["publicPostSearchLimits"]:
        return "publicPostSearchLimits"

    @classmethod
    def getClass(self) -> Literal["PublicPostSearchLimits"]:
        return "PublicPostSearchLimits"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "daily_free_query_count": self.daily_free_query_count,
            "remaining_free_query_count": self.remaining_free_query_count,
            "next_free_query_in": self.next_free_query_in,
            "star_count": self.star_count,
            "is_current_query_free": self.is_current_query_free,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PublicPostSearchLimits", None]:
        if data:
            data_class = cls()
            data_class.daily_free_query_count = int(
                data.get("daily_free_query_count", 0)
            )
            data_class.remaining_free_query_count = int(
                data.get("remaining_free_query_count", 0)
            )
            data_class.next_free_query_in = int(data.get("next_free_query_in", 0))
            data_class.star_count = int(data.get("star_count", 0))
            data_class.is_current_query_free = data.get("is_current_query_free", False)

        return data_class


class MessageSenderUser(TlObject, MessageSender):
    r"""The message was sent by a known user

    Parameters:
        user_id (:class:`int`):
            Identifier of the user that sent the message

    """

    def __init__(self, user_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""Identifier of the user that sent the message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSenderUser"]:
        return "messageSenderUser"

    @classmethod
    def getClass(self) -> Literal["MessageSender"]:
        return "MessageSender"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSenderUser", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class MessageSenderChat(TlObject, MessageSender):
    r"""The message was sent on behalf of a chat

    Parameters:
        chat_id (:class:`int`):
            Identifier of the chat that sent the message

    """

    def __init__(self, chat_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat that sent the message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSenderChat"]:
        return "messageSenderChat"

    @classmethod
    def getClass(self) -> Literal["MessageSender"]:
        return "MessageSender"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSenderChat", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))

        return data_class


class MessageSenders(TlObject):
    r"""Represents a list of message senders

    Parameters:
        total_count (:class:`int`):
            Approximate total number of messages senders found

        senders (List[:class:`~pytdbot.types.MessageSender`]):
            List of message senders

    """

    def __init__(
        self, total_count: int = 0, senders: List[MessageSender] = None
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of messages senders found"""
        self.senders: List[MessageSender] = senders or []
        r"""List of message senders"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSenders"]:
        return "messageSenders"

    @classmethod
    def getClass(self) -> Literal["MessageSenders"]:
        return "MessageSenders"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "senders": self.senders,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSenders", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.senders = data.get("senders", None)

        return data_class


class ChatMessageSender(TlObject):
    r"""Represents a message sender, which can be used to send messages in a chat

    Parameters:
        sender (:class:`~pytdbot.types.MessageSender`):
            The message sender

        needs_premium (:class:`bool`):
            True, if Telegram Premium is needed to use the message sender

    """

    def __init__(
        self, sender: MessageSender = None, needs_premium: bool = False
    ) -> None:
        self.sender: Union[MessageSenderUser, MessageSenderChat, None] = sender
        r"""The message sender"""
        self.needs_premium: bool = bool(needs_premium)
        r"""True, if Telegram Premium is needed to use the message sender"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMessageSender"]:
        return "chatMessageSender"

    @classmethod
    def getClass(self) -> Literal["ChatMessageSender"]:
        return "ChatMessageSender"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sender": self.sender,
            "needs_premium": self.needs_premium,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMessageSender", None]:
        if data:
            data_class = cls()
            data_class.sender = data.get("sender", None)
            data_class.needs_premium = data.get("needs_premium", False)

        return data_class


class ChatMessageSenders(TlObject):
    r"""Represents a list of message senders, which can be used to send messages in a chat

    Parameters:
        senders (List[:class:`~pytdbot.types.ChatMessageSender`]):
            List of available message senders

    """

    def __init__(self, senders: List[ChatMessageSender] = None) -> None:
        self.senders: List[ChatMessageSender] = senders or []
        r"""List of available message senders"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatMessageSenders"]:
        return "chatMessageSenders"

    @classmethod
    def getClass(self) -> Literal["ChatMessageSenders"]:
        return "ChatMessageSenders"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "senders": self.senders}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatMessageSenders", None]:
        if data:
            data_class = cls()
            data_class.senders = data.get("senders", None)

        return data_class


class MessageReadDateRead(TlObject, MessageReadDate):
    r"""Contains read date of the message

    Parameters:
        read_date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was read by the other user

    """

    def __init__(self, read_date: int = 0) -> None:
        self.read_date: int = int(read_date)
        r"""Point in time \(Unix timestamp\) when the message was read by the other user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReadDateRead"]:
        return "messageReadDateRead"

    @classmethod
    def getClass(self) -> Literal["MessageReadDate"]:
        return "MessageReadDate"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "read_date": self.read_date}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReadDateRead", None]:
        if data:
            data_class = cls()
            data_class.read_date = int(data.get("read_date", 0))

        return data_class


class MessageReadDateUnread(TlObject, MessageReadDate):
    r"""The message is unread yet"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReadDateUnread"]:
        return "messageReadDateUnread"

    @classmethod
    def getClass(self) -> Literal["MessageReadDate"]:
        return "MessageReadDate"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReadDateUnread", None]:
        if data:
            data_class = cls()

        return data_class


class MessageReadDateTooOld(TlObject, MessageReadDate):
    r"""The message is too old to get read date"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReadDateTooOld"]:
        return "messageReadDateTooOld"

    @classmethod
    def getClass(self) -> Literal["MessageReadDate"]:
        return "MessageReadDate"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReadDateTooOld", None]:
        if data:
            data_class = cls()

        return data_class


class MessageReadDateUserPrivacyRestricted(TlObject, MessageReadDate):
    r"""The read date is unknown due to privacy settings of the other user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReadDateUserPrivacyRestricted"]:
        return "messageReadDateUserPrivacyRestricted"

    @classmethod
    def getClass(self) -> Literal["MessageReadDate"]:
        return "MessageReadDate"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["MessageReadDateUserPrivacyRestricted", None]:
        if data:
            data_class = cls()

        return data_class


class MessageReadDateMyPrivacyRestricted(TlObject, MessageReadDate):
    r"""The read date is unknown due to privacy settings of the current user, but will be known if the user subscribes to Telegram Premium"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReadDateMyPrivacyRestricted"]:
        return "messageReadDateMyPrivacyRestricted"

    @classmethod
    def getClass(self) -> Literal["MessageReadDate"]:
        return "MessageReadDate"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReadDateMyPrivacyRestricted", None]:
        if data:
            data_class = cls()

        return data_class


class MessageViewer(TlObject):
    r"""Represents a viewer of a message

    Parameters:
        user_id (:class:`int`):
            User identifier of the viewer

        view_date (:class:`int`):
            Approximate point in time \(Unix timestamp\) when the message was viewed

    """

    def __init__(self, user_id: int = 0, view_date: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""User identifier of the viewer"""
        self.view_date: int = int(view_date)
        r"""Approximate point in time \(Unix timestamp\) when the message was viewed"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageViewer"]:
        return "messageViewer"

    @classmethod
    def getClass(self) -> Literal["MessageViewer"]:
        return "MessageViewer"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "user_id": self.user_id,
            "view_date": self.view_date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageViewer", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))
            data_class.view_date = int(data.get("view_date", 0))

        return data_class


class MessageViewers(TlObject):
    r"""Represents a list of message viewers

    Parameters:
        viewers (List[:class:`~pytdbot.types.MessageViewer`]):
            List of message viewers

    """

    def __init__(self, viewers: List[MessageViewer] = None) -> None:
        self.viewers: List[MessageViewer] = viewers or []
        r"""List of message viewers"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageViewers"]:
        return "messageViewers"

    @classmethod
    def getClass(self) -> Literal["MessageViewers"]:
        return "MessageViewers"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "viewers": self.viewers}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageViewers", None]:
        if data:
            data_class = cls()
            data_class.viewers = data.get("viewers", None)

        return data_class


class MessageOriginUser(TlObject, MessageOrigin):
    r"""The message was originally sent by a known user

    Parameters:
        sender_user_id (:class:`int`):
            Identifier of the user that originally sent the message

    """

    def __init__(self, sender_user_id: int = 0) -> None:
        self.sender_user_id: int = int(sender_user_id)
        r"""Identifier of the user that originally sent the message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageOriginUser"]:
        return "messageOriginUser"

    @classmethod
    def getClass(self) -> Literal["MessageOrigin"]:
        return "MessageOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sender_user_id": self.sender_user_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageOriginUser", None]:
        if data:
            data_class = cls()
            data_class.sender_user_id = int(data.get("sender_user_id", 0))

        return data_class


class MessageOriginHiddenUser(TlObject, MessageOrigin):
    r"""The message was originally sent by a user, which is hidden by their privacy settings

    Parameters:
        sender_name (:class:`str`):
            Name of the sender

    """

    def __init__(self, sender_name: str = "") -> None:
        self.sender_name: Union[str, None] = sender_name
        r"""Name of the sender"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageOriginHiddenUser"]:
        return "messageOriginHiddenUser"

    @classmethod
    def getClass(self) -> Literal["MessageOrigin"]:
        return "MessageOrigin"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sender_name": self.sender_name}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageOriginHiddenUser", None]:
        if data:
            data_class = cls()
            data_class.sender_name = data.get("sender_name", "")

        return data_class


class MessageOriginChat(TlObject, MessageOrigin):
    r"""The message was originally sent on behalf of a chat

    Parameters:
        sender_chat_id (:class:`int`):
            Identifier of the chat that originally sent the message

        author_signature (:class:`str`):
            For messages originally sent by an anonymous chat administrator, original message author signature

    """

    def __init__(self, sender_chat_id: int = 0, author_signature: str = "") -> None:
        self.sender_chat_id: int = int(sender_chat_id)
        r"""Identifier of the chat that originally sent the message"""
        self.author_signature: Union[str, None] = author_signature
        r"""For messages originally sent by an anonymous chat administrator, original message author signature"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageOriginChat"]:
        return "messageOriginChat"

    @classmethod
    def getClass(self) -> Literal["MessageOrigin"]:
        return "MessageOrigin"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sender_chat_id": self.sender_chat_id,
            "author_signature": self.author_signature,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageOriginChat", None]:
        if data:
            data_class = cls()
            data_class.sender_chat_id = int(data.get("sender_chat_id", 0))
            data_class.author_signature = data.get("author_signature", "")

        return data_class


class MessageOriginChannel(TlObject, MessageOrigin):
    r"""The message was originally a post in a channel

    Parameters:
        chat_id (:class:`int`):
            Identifier of the channel chat to which the message was originally sent

        message_id (:class:`int`):
            Message identifier of the original message

        author_signature (:class:`str`):
            Original post author signature

    """

    def __init__(
        self, chat_id: int = 0, message_id: int = 0, author_signature: str = ""
    ) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the channel chat to which the message was originally sent"""
        self.message_id: int = int(message_id)
        r"""Message identifier of the original message"""
        self.author_signature: Union[str, None] = author_signature
        r"""Original post author signature"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageOriginChannel"]:
        return "messageOriginChannel"

    @classmethod
    def getClass(self) -> Literal["MessageOrigin"]:
        return "MessageOrigin"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_id": self.message_id,
            "author_signature": self.author_signature,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageOriginChannel", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.author_signature = data.get("author_signature", "")

        return data_class


class ForwardSource(TlObject):
    r"""Contains information about the last message from which a new message was forwarded last time

    Parameters:
        chat_id (:class:`int`):
            Identifier of the chat to which the message that was forwarded belonged; may be 0 if unknown

        message_id (:class:`int`):
            Identifier of the message; may be 0 if unknown

        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the sender of the message; may be null if unknown or the new message was forwarded not to Saved Messages

        sender_name (:class:`str`):
            Name of the sender of the message if the sender is hidden by their privacy settings

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the message is sent; 0 if unknown

        is_outgoing (:class:`bool`):
            True, if the message that was forwarded is outgoing; always false if sender is unknown

    """

    def __init__(
        self,
        chat_id: int = 0,
        message_id: int = 0,
        sender_id: MessageSender = None,
        sender_name: str = "",
        date: int = 0,
        is_outgoing: bool = False,
    ) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat to which the message that was forwarded belonged; may be 0 if unknown"""
        self.message_id: int = int(message_id)
        r"""Identifier of the message; may be 0 if unknown"""
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of the sender of the message; may be null if unknown or the new message was forwarded not to Saved Messages"""
        self.sender_name: Union[str, None] = sender_name
        r"""Name of the sender of the message if the sender is hidden by their privacy settings"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the message is sent; 0 if unknown"""
        self.is_outgoing: bool = bool(is_outgoing)
        r"""True, if the message that was forwarded is outgoing; always false if sender is unknown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["forwardSource"]:
        return "forwardSource"

    @classmethod
    def getClass(self) -> Literal["ForwardSource"]:
        return "ForwardSource"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_id": self.message_id,
            "sender_id": self.sender_id,
            "sender_name": self.sender_name,
            "date": self.date,
            "is_outgoing": self.is_outgoing,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ForwardSource", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.sender_id = data.get("sender_id", None)
            data_class.sender_name = data.get("sender_name", "")
            data_class.date = int(data.get("date", 0))
            data_class.is_outgoing = data.get("is_outgoing", False)

        return data_class


class ReactionTypeEmoji(TlObject, ReactionType):
    r"""A reaction with an emoji

    Parameters:
        emoji (:class:`str`):
            Text representation of the reaction

    """

    def __init__(self, emoji: str = "") -> None:
        self.emoji: Union[str, None] = emoji
        r"""Text representation of the reaction"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionTypeEmoji"]:
        return "reactionTypeEmoji"

    @classmethod
    def getClass(self) -> Literal["ReactionType"]:
        return "ReactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "emoji": self.emoji}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionTypeEmoji", None]:
        if data:
            data_class = cls()
            data_class.emoji = data.get("emoji", "")

        return data_class


class ReactionTypeCustomEmoji(TlObject, ReactionType):
    r"""A reaction with a custom emoji

    Parameters:
        custom_emoji_id (:class:`int`):
            Unique identifier of the custom emoji

    """

    def __init__(self, custom_emoji_id: int = 0) -> None:
        self.custom_emoji_id: int = int(custom_emoji_id)
        r"""Unique identifier of the custom emoji"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionTypeCustomEmoji"]:
        return "reactionTypeCustomEmoji"

    @classmethod
    def getClass(self) -> Literal["ReactionType"]:
        return "ReactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "custom_emoji_id": self.custom_emoji_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionTypeCustomEmoji", None]:
        if data:
            data_class = cls()
            data_class.custom_emoji_id = int(data.get("custom_emoji_id", 0))

        return data_class


class ReactionTypePaid(TlObject, ReactionType):
    r"""The paid reaction in a channel chat"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionTypePaid"]:
        return "reactionTypePaid"

    @classmethod
    def getClass(self) -> Literal["ReactionType"]:
        return "ReactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionTypePaid", None]:
        if data:
            data_class = cls()

        return data_class


class PaidReactionTypeRegular(TlObject, PaidReactionType):
    r"""A paid reaction on behalf of the current user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["paidReactionTypeRegular"]:
        return "paidReactionTypeRegular"

    @classmethod
    def getClass(self) -> Literal["PaidReactionType"]:
        return "PaidReactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["PaidReactionTypeRegular", None]:
        if data:
            data_class = cls()

        return data_class


class PaidReactionTypeAnonymous(TlObject, PaidReactionType):
    r"""An anonymous paid reaction"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["paidReactionTypeAnonymous"]:
        return "paidReactionTypeAnonymous"

    @classmethod
    def getClass(self) -> Literal["PaidReactionType"]:
        return "PaidReactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["PaidReactionTypeAnonymous", None]:
        if data:
            data_class = cls()

        return data_class


class PaidReactionTypeChat(TlObject, PaidReactionType):
    r"""A paid reaction on behalf of an owned chat

    Parameters:
        chat_id (:class:`int`):
            Identifier of the chat

    """

    def __init__(self, chat_id: int = 0) -> None:
        self.chat_id: int = int(chat_id)
        r"""Identifier of the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["paidReactionTypeChat"]:
        return "paidReactionTypeChat"

    @classmethod
    def getClass(self) -> Literal["PaidReactionType"]:
        return "PaidReactionType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chat_id": self.chat_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["PaidReactionTypeChat", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))

        return data_class


class PaidReactor(TlObject):
    r"""Contains information about a user that added paid reactions

    Parameters:
        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the user or chat that added the reactions; may be null for anonymous reactors that aren't the current user

        star_count (:class:`int`):
            Number of Telegram Stars added

        is_top (:class:`bool`):
            True, if the reactor is one of the most active reactors; may be false if the reactor is the current user

        is_me (:class:`bool`):
            True, if the paid reaction was added by the current user

        is_anonymous (:class:`bool`):
            True, if the reactor is anonymous

    """

    def __init__(
        self,
        sender_id: MessageSender = None,
        star_count: int = 0,
        is_top: bool = False,
        is_me: bool = False,
        is_anonymous: bool = False,
    ) -> None:
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of the user or chat that added the reactions; may be null for anonymous reactors that aren't the current user"""
        self.star_count: int = int(star_count)
        r"""Number of Telegram Stars added"""
        self.is_top: bool = bool(is_top)
        r"""True, if the reactor is one of the most active reactors; may be false if the reactor is the current user"""
        self.is_me: bool = bool(is_me)
        r"""True, if the paid reaction was added by the current user"""
        self.is_anonymous: bool = bool(is_anonymous)
        r"""True, if the reactor is anonymous"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["paidReactor"]:
        return "paidReactor"

    @classmethod
    def getClass(self) -> Literal["PaidReactor"]:
        return "PaidReactor"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sender_id": self.sender_id,
            "star_count": self.star_count,
            "is_top": self.is_top,
            "is_me": self.is_me,
            "is_anonymous": self.is_anonymous,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["PaidReactor", None]:
        if data:
            data_class = cls()
            data_class.sender_id = data.get("sender_id", None)
            data_class.star_count = int(data.get("star_count", 0))
            data_class.is_top = data.get("is_top", False)
            data_class.is_me = data.get("is_me", False)
            data_class.is_anonymous = data.get("is_anonymous", False)

        return data_class


class MessageForwardInfo(TlObject):
    r"""Contains information about a forwarded message

    Parameters:
        origin (:class:`~pytdbot.types.MessageOrigin`):
            Origin of the forwarded message

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was originally sent

        source (:class:`~pytdbot.types.ForwardSource`):
            For messages forwarded to the chat with the current user \(Saved Messages\), to the Replies bot chat, or to the channel's discussion group, information about the source message from which the message was forwarded last time; may be null for other forwards or if unknown

        public_service_announcement_type (:class:`str`):
            The type of public service announcement for the forwarded message

    """

    def __init__(
        self,
        origin: MessageOrigin = None,
        date: int = 0,
        source: ForwardSource = None,
        public_service_announcement_type: str = "",
    ) -> None:
        self.origin: Union[
            MessageOriginUser,
            MessageOriginHiddenUser,
            MessageOriginChat,
            MessageOriginChannel,
            None,
        ] = origin
        r"""Origin of the forwarded message"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the message was originally sent"""
        self.source: Union[ForwardSource, None] = source
        r"""For messages forwarded to the chat with the current user \(Saved Messages\), to the Replies bot chat, or to the channel's discussion group, information about the source message from which the message was forwarded last time; may be null for other forwards or if unknown"""
        self.public_service_announcement_type: Union[str, None] = (
            public_service_announcement_type
        )
        r"""The type of public service announcement for the forwarded message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageForwardInfo"]:
        return "messageForwardInfo"

    @classmethod
    def getClass(self) -> Literal["MessageForwardInfo"]:
        return "MessageForwardInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "origin": self.origin,
            "date": self.date,
            "source": self.source,
            "public_service_announcement_type": self.public_service_announcement_type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageForwardInfo", None]:
        if data:
            data_class = cls()
            data_class.origin = data.get("origin", None)
            data_class.date = int(data.get("date", 0))
            data_class.source = data.get("source", None)
            data_class.public_service_announcement_type = data.get(
                "public_service_announcement_type", ""
            )

        return data_class


class MessageImportInfo(TlObject):
    r"""Contains information about a message created with importMessages

    Parameters:
        sender_name (:class:`str`):
            Name of the original sender

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was originally sent

    """

    def __init__(self, sender_name: str = "", date: int = 0) -> None:
        self.sender_name: Union[str, None] = sender_name
        r"""Name of the original sender"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the message was originally sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageImportInfo"]:
        return "messageImportInfo"

    @classmethod
    def getClass(self) -> Literal["MessageImportInfo"]:
        return "MessageImportInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "sender_name": self.sender_name,
            "date": self.date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageImportInfo", None]:
        if data:
            data_class = cls()
            data_class.sender_name = data.get("sender_name", "")
            data_class.date = int(data.get("date", 0))

        return data_class


class MessageReplyInfo(TlObject):
    r"""Contains information about replies to a message

    Parameters:
        reply_count (:class:`int`):
            Number of times the message was directly or indirectly replied

        recent_replier_ids (List[:class:`~pytdbot.types.MessageSender`]):
            Identifiers of at most 3 recent repliers to the message; available in channels with a discussion supergroup\. The users and chats are expected to be inaccessible: only their photo and name will be available

        last_read_inbox_message_id (:class:`int`):
            Identifier of the last read incoming reply to the message

        last_read_outbox_message_id (:class:`int`):
            Identifier of the last read outgoing reply to the message

        last_message_id (:class:`int`):
            Identifier of the last reply to the message

    """

    def __init__(
        self,
        reply_count: int = 0,
        recent_replier_ids: List[MessageSender] = None,
        last_read_inbox_message_id: int = 0,
        last_read_outbox_message_id: int = 0,
        last_message_id: int = 0,
    ) -> None:
        self.reply_count: int = int(reply_count)
        r"""Number of times the message was directly or indirectly replied"""
        self.recent_replier_ids: List[MessageSender] = recent_replier_ids or []
        r"""Identifiers of at most 3 recent repliers to the message; available in channels with a discussion supergroup\. The users and chats are expected to be inaccessible: only their photo and name will be available"""
        self.last_read_inbox_message_id: int = int(last_read_inbox_message_id)
        r"""Identifier of the last read incoming reply to the message"""
        self.last_read_outbox_message_id: int = int(last_read_outbox_message_id)
        r"""Identifier of the last read outgoing reply to the message"""
        self.last_message_id: int = int(last_message_id)
        r"""Identifier of the last reply to the message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReplyInfo"]:
        return "messageReplyInfo"

    @classmethod
    def getClass(self) -> Literal["MessageReplyInfo"]:
        return "MessageReplyInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "reply_count": self.reply_count,
            "recent_replier_ids": self.recent_replier_ids,
            "last_read_inbox_message_id": self.last_read_inbox_message_id,
            "last_read_outbox_message_id": self.last_read_outbox_message_id,
            "last_message_id": self.last_message_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReplyInfo", None]:
        if data:
            data_class = cls()
            data_class.reply_count = int(data.get("reply_count", 0))
            data_class.recent_replier_ids = data.get("recent_replier_ids", None)
            data_class.last_read_inbox_message_id = int(
                data.get("last_read_inbox_message_id", 0)
            )
            data_class.last_read_outbox_message_id = int(
                data.get("last_read_outbox_message_id", 0)
            )
            data_class.last_message_id = int(data.get("last_message_id", 0))

        return data_class


class MessageReaction(TlObject):
    r"""Contains information about a reaction to a message

    Parameters:
        type (:class:`~pytdbot.types.ReactionType`):
            Type of the reaction

        total_count (:class:`int`):
            Number of times the reaction was added

        is_chosen (:class:`bool`):
            True, if the reaction is chosen by the current user

        used_sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the message sender used by the current user to add the reaction; may be null if unknown or the reaction isn't chosen

        recent_sender_ids (List[:class:`~pytdbot.types.MessageSender`]):
            Identifiers of at most 3 recent message senders, added the reaction; available in private, basic group and supergroup chats

    """

    def __init__(
        self,
        type: ReactionType = None,
        total_count: int = 0,
        is_chosen: bool = False,
        used_sender_id: MessageSender = None,
        recent_sender_ids: List[MessageSender] = None,
    ) -> None:
        self.type: Union[
            ReactionTypeEmoji, ReactionTypeCustomEmoji, ReactionTypePaid, None
        ] = type
        r"""Type of the reaction"""
        self.total_count: int = int(total_count)
        r"""Number of times the reaction was added"""
        self.is_chosen: bool = bool(is_chosen)
        r"""True, if the reaction is chosen by the current user"""
        self.used_sender_id: Union[MessageSenderUser, MessageSenderChat, None] = (
            used_sender_id
        )
        r"""Identifier of the message sender used by the current user to add the reaction; may be null if unknown or the reaction isn't chosen"""
        self.recent_sender_ids: List[MessageSender] = recent_sender_ids or []
        r"""Identifiers of at most 3 recent message senders, added the reaction; available in private, basic group and supergroup chats"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReaction"]:
        return "messageReaction"

    @classmethod
    def getClass(self) -> Literal["MessageReaction"]:
        return "MessageReaction"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "type": self.type,
            "total_count": self.total_count,
            "is_chosen": self.is_chosen,
            "used_sender_id": self.used_sender_id,
            "recent_sender_ids": self.recent_sender_ids,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReaction", None]:
        if data:
            data_class = cls()
            data_class.type = data.get("type", None)
            data_class.total_count = int(data.get("total_count", 0))
            data_class.is_chosen = data.get("is_chosen", False)
            data_class.used_sender_id = data.get("used_sender_id", None)
            data_class.recent_sender_ids = data.get("recent_sender_ids", None)

        return data_class


class MessageReactions(TlObject):
    r"""Contains a list of reactions added to a message

    Parameters:
        reactions (List[:class:`~pytdbot.types.MessageReaction`]):
            List of added reactions

        are_tags (:class:`bool`):
            True, if the reactions are tags and Telegram Premium users can filter messages by them

        paid_reactors (List[:class:`~pytdbot.types.PaidReactor`]):
            Information about top users that added the paid reaction

        can_get_added_reactions (:class:`bool`):
            True, if the list of added reactions is available using getMessageAddedReactions

    """

    def __init__(
        self,
        reactions: List[MessageReaction] = None,
        are_tags: bool = False,
        paid_reactors: List[PaidReactor] = None,
        can_get_added_reactions: bool = False,
    ) -> None:
        self.reactions: List[MessageReaction] = reactions or []
        r"""List of added reactions"""
        self.are_tags: bool = bool(are_tags)
        r"""True, if the reactions are tags and Telegram Premium users can filter messages by them"""
        self.paid_reactors: List[PaidReactor] = paid_reactors or []
        r"""Information about top users that added the paid reaction"""
        self.can_get_added_reactions: bool = bool(can_get_added_reactions)
        r"""True, if the list of added reactions is available using getMessageAddedReactions"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReactions"]:
        return "messageReactions"

    @classmethod
    def getClass(self) -> Literal["MessageReactions"]:
        return "MessageReactions"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "reactions": self.reactions,
            "are_tags": self.are_tags,
            "paid_reactors": self.paid_reactors,
            "can_get_added_reactions": self.can_get_added_reactions,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReactions", None]:
        if data:
            data_class = cls()
            data_class.reactions = data.get("reactions", None)
            data_class.are_tags = data.get("are_tags", False)
            data_class.paid_reactors = data.get("paid_reactors", None)
            data_class.can_get_added_reactions = data.get(
                "can_get_added_reactions", False
            )

        return data_class


class MessageInteractionInfo(TlObject):
    r"""Contains information about interactions with a message

    Parameters:
        view_count (:class:`int`):
            Number of times the message was viewed

        forward_count (:class:`int`):
            Number of times the message was forwarded

        reply_info (:class:`~pytdbot.types.MessageReplyInfo`):
            Information about direct or indirect replies to the message; may be null\. Currently, available only in channels with a discussion supergroup and discussion supergroups for messages, which are not replies itself

        reactions (:class:`~pytdbot.types.MessageReactions`):
            The list of reactions or tags added to the message; may be null

    """

    def __init__(
        self,
        view_count: int = 0,
        forward_count: int = 0,
        reply_info: MessageReplyInfo = None,
        reactions: MessageReactions = None,
    ) -> None:
        self.view_count: int = int(view_count)
        r"""Number of times the message was viewed"""
        self.forward_count: int = int(forward_count)
        r"""Number of times the message was forwarded"""
        self.reply_info: Union[MessageReplyInfo, None] = reply_info
        r"""Information about direct or indirect replies to the message; may be null\. Currently, available only in channels with a discussion supergroup and discussion supergroups for messages, which are not replies itself"""
        self.reactions: Union[MessageReactions, None] = reactions
        r"""The list of reactions or tags added to the message; may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageInteractionInfo"]:
        return "messageInteractionInfo"

    @classmethod
    def getClass(self) -> Literal["MessageInteractionInfo"]:
        return "MessageInteractionInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "view_count": self.view_count,
            "forward_count": self.forward_count,
            "reply_info": self.reply_info,
            "reactions": self.reactions,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageInteractionInfo", None]:
        if data:
            data_class = cls()
            data_class.view_count = int(data.get("view_count", 0))
            data_class.forward_count = int(data.get("forward_count", 0))
            data_class.reply_info = data.get("reply_info", None)
            data_class.reactions = data.get("reactions", None)

        return data_class


class UnreadReaction(TlObject):
    r"""Contains information about an unread reaction to a message

    Parameters:
        type (:class:`~pytdbot.types.ReactionType`):
            Type of the reaction

        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the sender, added the reaction

        is_big (:class:`bool`):
            True, if the reaction was added with a big animation

    """

    def __init__(
        self,
        type: ReactionType = None,
        sender_id: MessageSender = None,
        is_big: bool = False,
    ) -> None:
        self.type: Union[
            ReactionTypeEmoji, ReactionTypeCustomEmoji, ReactionTypePaid, None
        ] = type
        r"""Type of the reaction"""
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of the sender, added the reaction"""
        self.is_big: bool = bool(is_big)
        r"""True, if the reaction was added with a big animation"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["unreadReaction"]:
        return "unreadReaction"

    @classmethod
    def getClass(self) -> Literal["UnreadReaction"]:
        return "UnreadReaction"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "type": self.type,
            "sender_id": self.sender_id,
            "is_big": self.is_big,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["UnreadReaction", None]:
        if data:
            data_class = cls()
            data_class.type = data.get("type", None)
            data_class.sender_id = data.get("sender_id", None)
            data_class.is_big = data.get("is_big", False)

        return data_class


class MessageTopicThread(TlObject, MessageTopic):
    r"""A topic in a non\-forum supergroup chat

    Parameters:
        message_thread_id (:class:`int`):
            Unique identifier of the message thread

    """

    def __init__(self, message_thread_id: int = 0) -> None:
        self.message_thread_id: int = int(message_thread_id)
        r"""Unique identifier of the message thread"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageTopicThread"]:
        return "messageTopicThread"

    @classmethod
    def getClass(self) -> Literal["MessageTopic"]:
        return "MessageTopic"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "message_thread_id": self.message_thread_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageTopicThread", None]:
        if data:
            data_class = cls()
            data_class.message_thread_id = int(data.get("message_thread_id", 0))

        return data_class


class MessageTopicForum(TlObject, MessageTopic):
    r"""A topic in a forum supergroup chat or a chat with a bot

    Parameters:
        forum_topic_id (:class:`int`):
            Unique identifier of the forum topic

    """

    def __init__(self, forum_topic_id: int = 0) -> None:
        self.forum_topic_id: int = int(forum_topic_id)
        r"""Unique identifier of the forum topic"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageTopicForum"]:
        return "messageTopicForum"

    @classmethod
    def getClass(self) -> Literal["MessageTopic"]:
        return "MessageTopic"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "forum_topic_id": self.forum_topic_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageTopicForum", None]:
        if data:
            data_class = cls()
            data_class.forum_topic_id = int(data.get("forum_topic_id", 0))

        return data_class


class MessageTopicDirectMessages(TlObject, MessageTopic):
    r"""A topic in a channel direct messages chat administered by the current user

    Parameters:
        direct_messages_chat_topic_id (:class:`int`):
            Unique identifier of the topic

    """

    def __init__(self, direct_messages_chat_topic_id: int = 0) -> None:
        self.direct_messages_chat_topic_id: int = int(direct_messages_chat_topic_id)
        r"""Unique identifier of the topic"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageTopicDirectMessages"]:
        return "messageTopicDirectMessages"

    @classmethod
    def getClass(self) -> Literal["MessageTopic"]:
        return "MessageTopic"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "direct_messages_chat_topic_id": self.direct_messages_chat_topic_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageTopicDirectMessages", None]:
        if data:
            data_class = cls()
            data_class.direct_messages_chat_topic_id = int(
                data.get("direct_messages_chat_topic_id", 0)
            )

        return data_class


class MessageTopicSavedMessages(TlObject, MessageTopic):
    r"""A topic in Saved Messages chat

    Parameters:
        saved_messages_topic_id (:class:`int`):
            Unique identifier of the Saved Messages topic

    """

    def __init__(self, saved_messages_topic_id: int = 0) -> None:
        self.saved_messages_topic_id: int = int(saved_messages_topic_id)
        r"""Unique identifier of the Saved Messages topic"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageTopicSavedMessages"]:
        return "messageTopicSavedMessages"

    @classmethod
    def getClass(self) -> Literal["MessageTopic"]:
        return "MessageTopic"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "saved_messages_topic_id": self.saved_messages_topic_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageTopicSavedMessages", None]:
        if data:
            data_class = cls()
            data_class.saved_messages_topic_id = int(
                data.get("saved_messages_topic_id", 0)
            )

        return data_class


class MessageEffectTypeEmojiReaction(TlObject, MessageEffectType):
    r"""An effect from an emoji reaction

    Parameters:
        select_animation (:class:`~pytdbot.types.Sticker`):
            Select animation for the effect in TGS format

        effect_animation (:class:`~pytdbot.types.Sticker`):
            Effect animation for the effect in TGS format

    """

    def __init__(
        self, select_animation: Sticker = None, effect_animation: Sticker = None
    ) -> None:
        self.select_animation: Union[Sticker, None] = select_animation
        r"""Select animation for the effect in TGS format"""
        self.effect_animation: Union[Sticker, None] = effect_animation
        r"""Effect animation for the effect in TGS format"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageEffectTypeEmojiReaction"]:
        return "messageEffectTypeEmojiReaction"

    @classmethod
    def getClass(self) -> Literal["MessageEffectType"]:
        return "MessageEffectType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "select_animation": self.select_animation,
            "effect_animation": self.effect_animation,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageEffectTypeEmojiReaction", None]:
        if data:
            data_class = cls()
            data_class.select_animation = data.get("select_animation", None)
            data_class.effect_animation = data.get("effect_animation", None)

        return data_class


class MessageEffectTypePremiumSticker(TlObject, MessageEffectType):
    r"""An effect from a premium sticker

    Parameters:
        sticker (:class:`~pytdbot.types.Sticker`):
            The premium sticker\. The effect can be found at sticker\.full\_type\.premium\_animation

    """

    def __init__(self, sticker: Sticker = None) -> None:
        self.sticker: Union[Sticker, None] = sticker
        r"""The premium sticker\. The effect can be found at sticker\.full\_type\.premium\_animation"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageEffectTypePremiumSticker"]:
        return "messageEffectTypePremiumSticker"

    @classmethod
    def getClass(self) -> Literal["MessageEffectType"]:
        return "MessageEffectType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sticker": self.sticker}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageEffectTypePremiumSticker", None]:
        if data:
            data_class = cls()
            data_class.sticker = data.get("sticker", None)

        return data_class


class MessageEffect(TlObject):
    r"""Contains information about an effect added to a message

    Parameters:
        id (:class:`int`):
            Unique identifier of the effect

        static_icon (:class:`~pytdbot.types.Sticker`):
            Static icon for the effect in WEBP format; may be null if none

        emoji (:class:`str`):
            Emoji corresponding to the effect that can be used if static icon isn't available

        is_premium (:class:`bool`):
            True, if Telegram Premium subscription is required to use the effect

        type (:class:`~pytdbot.types.MessageEffectType`):
            Type of the effect

    """

    def __init__(
        self,
        id: int = 0,
        static_icon: Sticker = None,
        emoji: str = "",
        is_premium: bool = False,
        type: MessageEffectType = None,
    ) -> None:
        self.id: int = int(id)
        r"""Unique identifier of the effect"""
        self.static_icon: Union[Sticker, None] = static_icon
        r"""Static icon for the effect in WEBP format; may be null if none"""
        self.emoji: Union[str, None] = emoji
        r"""Emoji corresponding to the effect that can be used if static icon isn't available"""
        self.is_premium: bool = bool(is_premium)
        r"""True, if Telegram Premium subscription is required to use the effect"""
        self.type: Union[
            MessageEffectTypeEmojiReaction, MessageEffectTypePremiumSticker, None
        ] = type
        r"""Type of the effect"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageEffect"]:
        return "messageEffect"

    @classmethod
    def getClass(self) -> Literal["MessageEffect"]:
        return "MessageEffect"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "static_icon": self.static_icon,
            "emoji": self.emoji,
            "is_premium": self.is_premium,
            "type": self.type,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageEffect", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.static_icon = data.get("static_icon", None)
            data_class.emoji = data.get("emoji", "")
            data_class.is_premium = data.get("is_premium", False)
            data_class.type = data.get("type", None)

        return data_class


class MessageSendingStatePending(TlObject, MessageSendingState):
    r"""The message is being sent now, but has not yet been delivered to the server

    Parameters:
        sending_id (:class:`int`):
            Non\-persistent message sending identifier, specified by the application

    """

    def __init__(self, sending_id: int = 0) -> None:
        self.sending_id: int = int(sending_id)
        r"""Non\-persistent message sending identifier, specified by the application"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSendingStatePending"]:
        return "messageSendingStatePending"

    @classmethod
    def getClass(self) -> Literal["MessageSendingState"]:
        return "MessageSendingState"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "sending_id": self.sending_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSendingStatePending", None]:
        if data:
            data_class = cls()
            data_class.sending_id = int(data.get("sending_id", 0))

        return data_class


class MessageSendingStateFailed(TlObject, MessageSendingState):
    r"""The message failed to be sent

    Parameters:
        error (:class:`~pytdbot.types.Error`):
            The cause of the message sending failure

        can_retry (:class:`bool`):
            True, if the message can be re\-sent using resendMessages or readdQuickReplyShortcutMessages

        need_another_sender (:class:`bool`):
            True, if the message can be re\-sent only on behalf of a different sender

        need_another_reply_quote (:class:`bool`):
            True, if the message can be re\-sent only if another quote is chosen in the message that is replied by the given message

        need_drop_reply (:class:`bool`):
            True, if the message can be re\-sent only if the message to be replied is removed\. This will be done automatically by resendMessages

        required_paid_message_star_count (:class:`int`):
            The number of Telegram Stars that must be paid to send the message; 0 if the current amount is correct

        retry_after (:class:`float`):
            Time left before the message can be re\-sent, in seconds\. No update is sent when this field changes

    """

    def __init__(
        self,
        error: Error = None,
        can_retry: bool = False,
        need_another_sender: bool = False,
        need_another_reply_quote: bool = False,
        need_drop_reply: bool = False,
        required_paid_message_star_count: int = 0,
        retry_after: float = 0.0,
    ) -> None:
        self.error: Union[Error, None] = error
        r"""The cause of the message sending failure"""
        self.can_retry: bool = bool(can_retry)
        r"""True, if the message can be re\-sent using resendMessages or readdQuickReplyShortcutMessages"""
        self.need_another_sender: bool = bool(need_another_sender)
        r"""True, if the message can be re\-sent only on behalf of a different sender"""
        self.need_another_reply_quote: bool = bool(need_another_reply_quote)
        r"""True, if the message can be re\-sent only if another quote is chosen in the message that is replied by the given message"""
        self.need_drop_reply: bool = bool(need_drop_reply)
        r"""True, if the message can be re\-sent only if the message to be replied is removed\. This will be done automatically by resendMessages"""
        self.required_paid_message_star_count: int = int(
            required_paid_message_star_count
        )
        r"""The number of Telegram Stars that must be paid to send the message; 0 if the current amount is correct"""
        self.retry_after: float = float(retry_after)
        r"""Time left before the message can be re\-sent, in seconds\. No update is sent when this field changes"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSendingStateFailed"]:
        return "messageSendingStateFailed"

    @classmethod
    def getClass(self) -> Literal["MessageSendingState"]:
        return "MessageSendingState"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "error": self.error,
            "can_retry": self.can_retry,
            "need_another_sender": self.need_another_sender,
            "need_another_reply_quote": self.need_another_reply_quote,
            "need_drop_reply": self.need_drop_reply,
            "required_paid_message_star_count": self.required_paid_message_star_count,
            "retry_after": self.retry_after,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSendingStateFailed", None]:
        if data:
            data_class = cls()
            data_class.error = data.get("error", None)
            data_class.can_retry = data.get("can_retry", False)
            data_class.need_another_sender = data.get("need_another_sender", False)
            data_class.need_another_reply_quote = data.get(
                "need_another_reply_quote", False
            )
            data_class.need_drop_reply = data.get("need_drop_reply", False)
            data_class.required_paid_message_star_count = int(
                data.get("required_paid_message_star_count", 0)
            )
            data_class.retry_after = data.get("retry_after", 0.0)

        return data_class


class TextQuote(TlObject):
    r"""Describes manually or automatically chosen quote from another message

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            Text of the quote\. Only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities can be present in the text

        position (:class:`int`):
            Approximate quote position in the original message in UTF\-16 code units as specified by the message sender

        is_manual (:class:`bool`):
            True, if the quote was manually chosen by the message sender

    """

    def __init__(
        self, text: FormattedText = None, position: int = 0, is_manual: bool = False
    ) -> None:
        self.text: Union[FormattedText, None] = text
        r"""Text of the quote\. Only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities can be present in the text"""
        self.position: int = int(position)
        r"""Approximate quote position in the original message in UTF\-16 code units as specified by the message sender"""
        self.is_manual: bool = bool(is_manual)
        r"""True, if the quote was manually chosen by the message sender"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["textQuote"]:
        return "textQuote"

    @classmethod
    def getClass(self) -> Literal["TextQuote"]:
        return "TextQuote"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "text": self.text,
            "position": self.position,
            "is_manual": self.is_manual,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["TextQuote", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.position = int(data.get("position", 0))
            data_class.is_manual = data.get("is_manual", False)

        return data_class


class InputTextQuote(TlObject):
    r"""Describes manually chosen quote from another message

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            Text of the quote; 0\-getOption\(\"message\_reply\_quote\_length\_max\"\) characters\. Only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities are allowed to be kept and must be kept in the quote

        position (:class:`int`):
            Quote position in the original message in UTF\-16 code units

    """

    def __init__(self, text: FormattedText = None, position: int = 0) -> None:
        self.text: Union[FormattedText, None] = text
        r"""Text of the quote; 0\-getOption\(\"message\_reply\_quote\_length\_max\"\) characters\. Only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities are allowed to be kept and must be kept in the quote"""
        self.position: int = int(position)
        r"""Quote position in the original message in UTF\-16 code units"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputTextQuote"]:
        return "inputTextQuote"

    @classmethod
    def getClass(self) -> Literal["InputTextQuote"]:
        return "InputTextQuote"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "text": self.text, "position": self.position}

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputTextQuote", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.position = int(data.get("position", 0))

        return data_class


class MessageReplyToMessage(TlObject, MessageReplyTo):
    r"""Describes a message replied by a given message

    Parameters:
        chat_id (:class:`int`):
            The identifier of the chat to which the message belongs; may be 0 if the replied message is in unknown chat

        message_id (:class:`int`):
            The identifier of the message; may be 0 if the replied message is in unknown chat

        quote (:class:`~pytdbot.types.TextQuote`):
            Chosen quote from the replied message; may be null if none

        checklist_task_id (:class:`int`):
            Identifier of the checklist task in the original message that was replied; 0 if none

        origin (:class:`~pytdbot.types.MessageOrigin`):
            Information about origin of the message if the message was from another chat or topic; may be null for messages from the same chat

        origin_send_date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was sent if the message was from another chat or topic; 0 for messages from the same chat

        content (:class:`~pytdbot.types.MessageContent`):
            Media content of the message if the message was from another chat or topic; may be null for messages from the same chat and messages without media\. Can be only one of the following types: messageAnimation, messageAudio, messageChecklist, messageContact, messageDice, messageDocument, messageGame, messageGiveaway, messageGiveawayWinners, messageInvoice, messageLocation, messagePaidMedia, messagePhoto, messagePoll, messageSticker, messageStory, messageText \(for link preview\), messageVenue, messageVideo, messageVideoNote, or messageVoiceNote

    """

    def __init__(
        self,
        chat_id: int = 0,
        message_id: int = 0,
        quote: TextQuote = None,
        checklist_task_id: int = 0,
        origin: MessageOrigin = None,
        origin_send_date: int = 0,
        content: MessageContent = None,
    ) -> None:
        self.chat_id: int = int(chat_id)
        r"""The identifier of the chat to which the message belongs; may be 0 if the replied message is in unknown chat"""
        self.message_id: int = int(message_id)
        r"""The identifier of the message; may be 0 if the replied message is in unknown chat"""
        self.quote: Union[TextQuote, None] = quote
        r"""Chosen quote from the replied message; may be null if none"""
        self.checklist_task_id: int = int(checklist_task_id)
        r"""Identifier of the checklist task in the original message that was replied; 0 if none"""
        self.origin: Union[
            MessageOriginUser,
            MessageOriginHiddenUser,
            MessageOriginChat,
            MessageOriginChannel,
            None,
        ] = origin
        r"""Information about origin of the message if the message was from another chat or topic; may be null for messages from the same chat"""
        self.origin_send_date: int = int(origin_send_date)
        r"""Point in time \(Unix timestamp\) when the message was sent if the message was from another chat or topic; 0 for messages from the same chat"""
        self.content: Union[
            MessageText,
            MessageAnimation,
            MessageAudio,
            MessageDocument,
            MessagePaidMedia,
            MessagePhoto,
            MessageSticker,
            MessageVideo,
            MessageVideoNote,
            MessageVoiceNote,
            MessageExpiredPhoto,
            MessageExpiredVideo,
            MessageExpiredVideoNote,
            MessageExpiredVoiceNote,
            MessageLocation,
            MessageVenue,
            MessageContact,
            MessageAnimatedEmoji,
            MessageDice,
            MessageGame,
            MessagePoll,
            MessageStory,
            MessageChecklist,
            MessageInvoice,
            MessageCall,
            MessageGroupCall,
            MessageVideoChatScheduled,
            MessageVideoChatStarted,
            MessageVideoChatEnded,
            MessageInviteVideoChatParticipants,
            MessageBasicGroupChatCreate,
            MessageSupergroupChatCreate,
            MessageChatChangeTitle,
            MessageChatChangePhoto,
            MessageChatDeletePhoto,
            MessageChatAddMembers,
            MessageChatJoinByLink,
            MessageChatJoinByRequest,
            MessageChatDeleteMember,
            MessageChatUpgradeTo,
            MessageChatUpgradeFrom,
            MessagePinMessage,
            MessageScreenshotTaken,
            MessageChatSetBackground,
            MessageChatSetTheme,
            MessageChatSetMessageAutoDeleteTime,
            MessageChatBoost,
            MessageForumTopicCreated,
            MessageForumTopicEdited,
            MessageForumTopicIsClosedToggled,
            MessageForumTopicIsHiddenToggled,
            MessageSuggestProfilePhoto,
            MessageSuggestBirthdate,
            MessageCustomServiceAction,
            MessageGameScore,
            MessagePaymentSuccessful,
            MessagePaymentSuccessfulBot,
            MessagePaymentRefunded,
            MessageGiftedPremium,
            MessagePremiumGiftCode,
            MessageGiveawayCreated,
            MessageGiveaway,
            MessageGiveawayCompleted,
            MessageGiveawayWinners,
            MessageGiftedStars,
            MessageGiftedTon,
            MessageGiveawayPrizeStars,
            MessageGift,
            MessageUpgradedGift,
            MessageRefundedUpgradedGift,
            MessagePaidMessagesRefunded,
            MessagePaidMessagePriceChanged,
            MessageDirectMessagePriceChanged,
            MessageChecklistTasksDone,
            MessageChecklistTasksAdded,
            MessageSuggestedPostApprovalFailed,
            MessageSuggestedPostApproved,
            MessageSuggestedPostDeclined,
            MessageSuggestedPostPaid,
            MessageSuggestedPostRefunded,
            MessageContactRegistered,
            MessageUsersShared,
            MessageChatShared,
            MessageBotWriteAccessAllowed,
            MessageWebAppDataSent,
            MessageWebAppDataReceived,
            MessagePassportDataSent,
            MessagePassportDataReceived,
            MessageProximityAlertTriggered,
            MessageUnsupported,
            None,
        ] = content
        r"""Media content of the message if the message was from another chat or topic; may be null for messages from the same chat and messages without media\. Can be only one of the following types: messageAnimation, messageAudio, messageChecklist, messageContact, messageDice, messageDocument, messageGame, messageGiveaway, messageGiveawayWinners, messageInvoice, messageLocation, messagePaidMedia, messagePhoto, messagePoll, messageSticker, messageStory, messageText \(for link preview\), messageVenue, messageVideo, messageVideoNote, or messageVoiceNote"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReplyToMessage"]:
        return "messageReplyToMessage"

    @classmethod
    def getClass(self) -> Literal["MessageReplyTo"]:
        return "MessageReplyTo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_id": self.message_id,
            "quote": self.quote,
            "checklist_task_id": self.checklist_task_id,
            "origin": self.origin,
            "origin_send_date": self.origin_send_date,
            "content": self.content,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReplyToMessage", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.quote = data.get("quote", None)
            data_class.checklist_task_id = int(data.get("checklist_task_id", 0))
            data_class.origin = data.get("origin", None)
            data_class.origin_send_date = int(data.get("origin_send_date", 0))
            data_class.content = data.get("content", None)

        return data_class


class MessageReplyToStory(TlObject, MessageReplyTo):
    r"""Describes a story replied by a given message

    Parameters:
        story_poster_chat_id (:class:`int`):
            The identifier of the poster of the story

        story_id (:class:`int`):
            The identifier of the story

    """

    def __init__(self, story_poster_chat_id: int = 0, story_id: int = 0) -> None:
        self.story_poster_chat_id: int = int(story_poster_chat_id)
        r"""The identifier of the poster of the story"""
        self.story_id: int = int(story_id)
        r"""The identifier of the story"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageReplyToStory"]:
        return "messageReplyToStory"

    @classmethod
    def getClass(self) -> Literal["MessageReplyTo"]:
        return "MessageReplyTo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "story_poster_chat_id": self.story_poster_chat_id,
            "story_id": self.story_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageReplyToStory", None]:
        if data:
            data_class = cls()
            data_class.story_poster_chat_id = int(data.get("story_poster_chat_id", 0))
            data_class.story_id = int(data.get("story_id", 0))

        return data_class


class InputMessageReplyToMessage(TlObject, InputMessageReplyTo):
    r"""Describes a message to be replied in the same chat and forum topic

    Parameters:
        message_id (:class:`int`):
            The identifier of the message to be replied in the same chat and forum topic\. A message can be replied in the same chat and forum topic only if messageProperties\.can\_be\_replied

        quote (:class:`~pytdbot.types.InputTextQuote`):
            Quote from the message to be replied; pass null if none\. Must always be null for replies in secret chats

        checklist_task_id (:class:`int`):
            Identifier of the checklist task in the message to be replied; pass 0 to reply to the whole message

    """

    def __init__(
        self,
        message_id: int = 0,
        quote: InputTextQuote = None,
        checklist_task_id: int = 0,
    ) -> None:
        self.message_id: int = int(message_id)
        r"""The identifier of the message to be replied in the same chat and forum topic\. A message can be replied in the same chat and forum topic only if messageProperties\.can\_be\_replied"""
        self.quote: Union[InputTextQuote, None] = quote
        r"""Quote from the message to be replied; pass null if none\. Must always be null for replies in secret chats"""
        self.checklist_task_id: int = int(checklist_task_id)
        r"""Identifier of the checklist task in the message to be replied; pass 0 to reply to the whole message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputMessageReplyToMessage"]:
        return "inputMessageReplyToMessage"

    @classmethod
    def getClass(self) -> Literal["InputMessageReplyTo"]:
        return "InputMessageReplyTo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "message_id": self.message_id,
            "quote": self.quote,
            "checklist_task_id": self.checklist_task_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputMessageReplyToMessage", None]:
        if data:
            data_class = cls()
            data_class.message_id = int(data.get("message_id", 0))
            data_class.quote = data.get("quote", None)
            data_class.checklist_task_id = int(data.get("checklist_task_id", 0))

        return data_class


class InputMessageReplyToExternalMessage(TlObject, InputMessageReplyTo):
    r"""Describes a message to be replied that is from a different chat or a forum topic; not supported in secret chats

    Parameters:
        chat_id (:class:`int`):
            The identifier of the chat to which the message to be replied belongs

        message_id (:class:`int`):
            The identifier of the message to be replied in the specified chat\. A message can be replied in another chat or forum topic only if messageProperties\.can\_be\_replied\_in\_another\_chat

        quote (:class:`~pytdbot.types.InputTextQuote`):
            Quote from the message to be replied; pass null if none

        checklist_task_id (:class:`int`):
            Identifier of the checklist task in the message to be replied; pass 0 to reply to the whole message

    """

    def __init__(
        self,
        chat_id: int = 0,
        message_id: int = 0,
        quote: InputTextQuote = None,
        checklist_task_id: int = 0,
    ) -> None:
        self.chat_id: int = int(chat_id)
        r"""The identifier of the chat to which the message to be replied belongs"""
        self.message_id: int = int(message_id)
        r"""The identifier of the message to be replied in the specified chat\. A message can be replied in another chat or forum topic only if messageProperties\.can\_be\_replied\_in\_another\_chat"""
        self.quote: Union[InputTextQuote, None] = quote
        r"""Quote from the message to be replied; pass null if none"""
        self.checklist_task_id: int = int(checklist_task_id)
        r"""Identifier of the checklist task in the message to be replied; pass 0 to reply to the whole message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputMessageReplyToExternalMessage"]:
        return "inputMessageReplyToExternalMessage"

    @classmethod
    def getClass(self) -> Literal["InputMessageReplyTo"]:
        return "InputMessageReplyTo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "chat_id": self.chat_id,
            "message_id": self.message_id,
            "quote": self.quote,
            "checklist_task_id": self.checklist_task_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputMessageReplyToExternalMessage", None]:
        if data:
            data_class = cls()
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.quote = data.get("quote", None)
            data_class.checklist_task_id = int(data.get("checklist_task_id", 0))

        return data_class


class InputMessageReplyToStory(TlObject, InputMessageReplyTo):
    r"""Describes a story to be replied

    Parameters:
        story_poster_chat_id (:class:`int`):
            The identifier of the poster of the story\. Currently, stories can be replied only in the chat that posted the story; channel stories can't be replied

        story_id (:class:`int`):
            The identifier of the story

    """

    def __init__(self, story_poster_chat_id: int = 0, story_id: int = 0) -> None:
        self.story_poster_chat_id: int = int(story_poster_chat_id)
        r"""The identifier of the poster of the story\. Currently, stories can be replied only in the chat that posted the story; channel stories can't be replied"""
        self.story_id: int = int(story_id)
        r"""The identifier of the story"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["inputMessageReplyToStory"]:
        return "inputMessageReplyToStory"

    @classmethod
    def getClass(self) -> Literal["InputMessageReplyTo"]:
        return "InputMessageReplyTo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "story_poster_chat_id": self.story_poster_chat_id,
            "story_id": self.story_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["InputMessageReplyToStory", None]:
        if data:
            data_class = cls()
            data_class.story_poster_chat_id = int(data.get("story_poster_chat_id", 0))
            data_class.story_id = int(data.get("story_id", 0))

        return data_class


class FactCheck(TlObject):
    r"""Describes a fact\-check added to the message by an independent checker

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            Text of the fact\-check

        country_code (:class:`str`):
            A two\-letter ISO 3166\-1 alpha\-2 country code of the country for which the fact\-check is shown

    """

    def __init__(self, text: FormattedText = None, country_code: str = "") -> None:
        self.text: Union[FormattedText, None] = text
        r"""Text of the fact\-check"""
        self.country_code: Union[str, None] = country_code
        r"""A two\-letter ISO 3166\-1 alpha\-2 country code of the country for which the fact\-check is shown"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["factCheck"]:
        return "factCheck"

    @classmethod
    def getClass(self) -> Literal["FactCheck"]:
        return "FactCheck"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "text": self.text,
            "country_code": self.country_code,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FactCheck", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.country_code = data.get("country_code", "")

        return data_class


class Message(TlObject, MessageBoundMethods):
    r"""Describes a message

    Parameters:
        id (:class:`int`):
            Message identifier; unique for the chat to which the message belongs

        sender_id (:class:`~pytdbot.types.MessageSender`):
            Identifier of the sender of the message

        chat_id (:class:`int`):
            Chat identifier

        sending_state (:class:`~pytdbot.types.MessageSendingState`):
            The sending state of the message; may be null if the message isn't being sent and didn't fail to be sent

        scheduling_state (:class:`~pytdbot.types.MessageSchedulingState`):
            The scheduling state of the message; may be null if the message isn't scheduled

        is_outgoing (:class:`bool`):
            True, if the message is outgoing

        is_pinned (:class:`bool`):
            True, if the message is pinned

        is_from_offline (:class:`bool`):
            True, if the message was sent because of a scheduled action by the message sender, for example, as away, or greeting service message

        can_be_saved (:class:`bool`):
            True, if content of the message can be saved locally

        has_timestamped_media (:class:`bool`):
            True, if media timestamp entities refers to a media in this message as opposed to a media in the replied message

        is_channel_post (:class:`bool`):
            True, if the message is a channel post\. All messages to channels are channel posts, all other messages are not channel posts

        is_paid_star_suggested_post (:class:`bool`):
            True, if the message is a suggested channel post which was paid in Telegram Stars; a warning must be shown if the message is deleted in less than getOption\(\"suggested\_post\_lifetime\_min\"\) seconds after sending

        is_paid_ton_suggested_post (:class:`bool`):
            True, if the message is a suggested channel post which was paid in Toncoins; a warning must be shown if the message is deleted in less than getOption\(\"suggested\_post\_lifetime\_min\"\) seconds after sending

        contains_unread_mention (:class:`bool`):
            True, if the message contains an unread mention for the current user

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was sent; 0 for scheduled messages

        edit_date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was last edited; 0 for scheduled messages

        forward_info (:class:`~pytdbot.types.MessageForwardInfo`):
            Information about the initial message sender; may be null if none or unknown

        import_info (:class:`~pytdbot.types.MessageImportInfo`):
            Information about the initial message for messages created with importMessages; may be null if the message isn't imported

        interaction_info (:class:`~pytdbot.types.MessageInteractionInfo`):
            Information about interactions with the message; may be null if none

        unread_reactions (List[:class:`~pytdbot.types.UnreadReaction`]):
            Information about unread reactions added to the message

        fact_check (:class:`~pytdbot.types.FactCheck`):
            Information about fact\-check added to the message; may be null if none

        suggested_post_info (:class:`~pytdbot.types.SuggestedPostInfo`):
            Information about the suggested post; may be null if the message isn't a suggested post

        reply_to (:class:`~pytdbot.types.MessageReplyTo`):
            Information about the message or the story this message is replying to; may be null if none

        topic_id (:class:`~pytdbot.types.MessageTopic`):
            Identifier of the topic within the chat to which the message belongs; may be null if none

        self_destruct_type (:class:`~pytdbot.types.MessageSelfDestructType`):
            The message's self\-destruct type; may be null if none

        self_destruct_in (:class:`float`):
            Time left before the message self\-destruct timer expires, in seconds; 0 if self\-destruction isn't scheduled yet

        auto_delete_in (:class:`float`):
            Time left before the message will be automatically deleted by message\_auto\_delete\_time setting of the chat, in seconds; 0 if never

        via_bot_user_id (:class:`int`):
            If non\-zero, the user identifier of the inline bot through which this message was sent

        sender_business_bot_user_id (:class:`int`):
            If non\-zero, the user identifier of the business bot that sent this message

        sender_boost_count (:class:`int`):
            Number of times the sender of the message boosted the supergroup at the time the message was sent; 0 if none or unknown\. For messages sent by the current user, supergroupFullInfo\.my\_boost\_count must be used instead

        paid_message_star_count (:class:`int`):
            The number of Telegram Stars the sender paid to send the message

        author_signature (:class:`str`):
            For channel posts and anonymous group messages, optional author signature

        media_album_id (:class:`int`):
            Unique identifier of an album this message belongs to; 0 if none\. Only audios, documents, photos and videos can be grouped together in albums

        effect_id (:class:`int`):
            Unique identifier of the effect added to the message; 0 if none

        restriction_info (:class:`~pytdbot.types.RestrictionInfo`):
            Information about the restrictions that must be applied to the message content; may be null if none

        content (:class:`~pytdbot.types.MessageContent`):
            Content of the message

        reply_markup (:class:`~pytdbot.types.ReplyMarkup`):
            Reply markup for the message; may be null if none

    """

    def __init__(
        self,
        id: int = 0,
        sender_id: MessageSender = None,
        chat_id: int = 0,
        sending_state: MessageSendingState = None,
        scheduling_state: MessageSchedulingState = None,
        is_outgoing: bool = False,
        is_pinned: bool = False,
        is_from_offline: bool = False,
        can_be_saved: bool = False,
        has_timestamped_media: bool = False,
        is_channel_post: bool = False,
        is_paid_star_suggested_post: bool = False,
        is_paid_ton_suggested_post: bool = False,
        contains_unread_mention: bool = False,
        date: int = 0,
        edit_date: int = 0,
        forward_info: MessageForwardInfo = None,
        import_info: MessageImportInfo = None,
        interaction_info: MessageInteractionInfo = None,
        unread_reactions: List[UnreadReaction] = None,
        fact_check: FactCheck = None,
        suggested_post_info: SuggestedPostInfo = None,
        reply_to: MessageReplyTo = None,
        topic_id: MessageTopic = None,
        self_destruct_type: MessageSelfDestructType = None,
        self_destruct_in: float = 0.0,
        auto_delete_in: float = 0.0,
        via_bot_user_id: int = 0,
        sender_business_bot_user_id: int = 0,
        sender_boost_count: int = 0,
        paid_message_star_count: int = 0,
        author_signature: str = "",
        media_album_id: int = 0,
        effect_id: int = 0,
        restriction_info: RestrictionInfo = None,
        content: MessageContent = None,
        reply_markup: ReplyMarkup = None,
    ) -> None:
        self.id: int = int(id)
        r"""Message identifier; unique for the chat to which the message belongs"""
        self.sender_id: Union[MessageSenderUser, MessageSenderChat, None] = sender_id
        r"""Identifier of the sender of the message"""
        self.chat_id: int = int(chat_id)
        r"""Chat identifier"""
        self.sending_state: Union[
            MessageSendingStatePending, MessageSendingStateFailed, None
        ] = sending_state
        r"""The sending state of the message; may be null if the message isn't being sent and didn't fail to be sent"""
        self.scheduling_state: Union[
            MessageSchedulingStateSendAtDate,
            MessageSchedulingStateSendWhenOnline,
            MessageSchedulingStateSendWhenVideoProcessed,
            None,
        ] = scheduling_state
        r"""The scheduling state of the message; may be null if the message isn't scheduled"""
        self.is_outgoing: bool = bool(is_outgoing)
        r"""True, if the message is outgoing"""
        self.is_pinned: bool = bool(is_pinned)
        r"""True, if the message is pinned"""
        self.is_from_offline: bool = bool(is_from_offline)
        r"""True, if the message was sent because of a scheduled action by the message sender, for example, as away, or greeting service message"""
        self.can_be_saved: bool = bool(can_be_saved)
        r"""True, if content of the message can be saved locally"""
        self.has_timestamped_media: bool = bool(has_timestamped_media)
        r"""True, if media timestamp entities refers to a media in this message as opposed to a media in the replied message"""
        self.is_channel_post: bool = bool(is_channel_post)
        r"""True, if the message is a channel post\. All messages to channels are channel posts, all other messages are not channel posts"""
        self.is_paid_star_suggested_post: bool = bool(is_paid_star_suggested_post)
        r"""True, if the message is a suggested channel post which was paid in Telegram Stars; a warning must be shown if the message is deleted in less than getOption\(\"suggested\_post\_lifetime\_min\"\) seconds after sending"""
        self.is_paid_ton_suggested_post: bool = bool(is_paid_ton_suggested_post)
        r"""True, if the message is a suggested channel post which was paid in Toncoins; a warning must be shown if the message is deleted in less than getOption\(\"suggested\_post\_lifetime\_min\"\) seconds after sending"""
        self.contains_unread_mention: bool = bool(contains_unread_mention)
        r"""True, if the message contains an unread mention for the current user"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the message was sent; 0 for scheduled messages"""
        self.edit_date: int = int(edit_date)
        r"""Point in time \(Unix timestamp\) when the message was last edited; 0 for scheduled messages"""
        self.forward_info: Union[MessageForwardInfo, None] = forward_info
        r"""Information about the initial message sender; may be null if none or unknown"""
        self.import_info: Union[MessageImportInfo, None] = import_info
        r"""Information about the initial message for messages created with importMessages; may be null if the message isn't imported"""
        self.interaction_info: Union[MessageInteractionInfo, None] = interaction_info
        r"""Information about interactions with the message; may be null if none"""
        self.unread_reactions: List[UnreadReaction] = unread_reactions or []
        r"""Information about unread reactions added to the message"""
        self.fact_check: Union[FactCheck, None] = fact_check
        r"""Information about fact\-check added to the message; may be null if none"""
        self.suggested_post_info: Union[SuggestedPostInfo, None] = suggested_post_info
        r"""Information about the suggested post; may be null if the message isn't a suggested post"""
        self.reply_to: Union[MessageReplyToMessage, MessageReplyToStory, None] = (
            reply_to
        )
        r"""Information about the message or the story this message is replying to; may be null if none"""
        self.topic_id: Union[
            MessageTopicThread,
            MessageTopicForum,
            MessageTopicDirectMessages,
            MessageTopicSavedMessages,
            None,
        ] = topic_id
        r"""Identifier of the topic within the chat to which the message belongs; may be null if none"""
        self.self_destruct_type: Union[
            MessageSelfDestructTypeTimer, MessageSelfDestructTypeImmediately, None
        ] = self_destruct_type
        r"""The message's self\-destruct type; may be null if none"""
        self.self_destruct_in: float = float(self_destruct_in)
        r"""Time left before the message self\-destruct timer expires, in seconds; 0 if self\-destruction isn't scheduled yet"""
        self.auto_delete_in: float = float(auto_delete_in)
        r"""Time left before the message will be automatically deleted by message\_auto\_delete\_time setting of the chat, in seconds; 0 if never"""
        self.via_bot_user_id: int = int(via_bot_user_id)
        r"""If non\-zero, the user identifier of the inline bot through which this message was sent"""
        self.sender_business_bot_user_id: int = int(sender_business_bot_user_id)
        r"""If non\-zero, the user identifier of the business bot that sent this message"""
        self.sender_boost_count: int = int(sender_boost_count)
        r"""Number of times the sender of the message boosted the supergroup at the time the message was sent; 0 if none or unknown\. For messages sent by the current user, supergroupFullInfo\.my\_boost\_count must be used instead"""
        self.paid_message_star_count: int = int(paid_message_star_count)
        r"""The number of Telegram Stars the sender paid to send the message"""
        self.author_signature: Union[str, None] = author_signature
        r"""For channel posts and anonymous group messages, optional author signature"""
        self.media_album_id: int = int(media_album_id)
        r"""Unique identifier of an album this message belongs to; 0 if none\. Only audios, documents, photos and videos can be grouped together in albums"""
        self.effect_id: int = int(effect_id)
        r"""Unique identifier of the effect added to the message; 0 if none"""
        self.restriction_info: Union[RestrictionInfo, None] = restriction_info
        r"""Information about the restrictions that must be applied to the message content; may be null if none"""
        self.content: Union[
            MessageText,
            MessageAnimation,
            MessageAudio,
            MessageDocument,
            MessagePaidMedia,
            MessagePhoto,
            MessageSticker,
            MessageVideo,
            MessageVideoNote,
            MessageVoiceNote,
            MessageExpiredPhoto,
            MessageExpiredVideo,
            MessageExpiredVideoNote,
            MessageExpiredVoiceNote,
            MessageLocation,
            MessageVenue,
            MessageContact,
            MessageAnimatedEmoji,
            MessageDice,
            MessageGame,
            MessagePoll,
            MessageStory,
            MessageChecklist,
            MessageInvoice,
            MessageCall,
            MessageGroupCall,
            MessageVideoChatScheduled,
            MessageVideoChatStarted,
            MessageVideoChatEnded,
            MessageInviteVideoChatParticipants,
            MessageBasicGroupChatCreate,
            MessageSupergroupChatCreate,
            MessageChatChangeTitle,
            MessageChatChangePhoto,
            MessageChatDeletePhoto,
            MessageChatAddMembers,
            MessageChatJoinByLink,
            MessageChatJoinByRequest,
            MessageChatDeleteMember,
            MessageChatUpgradeTo,
            MessageChatUpgradeFrom,
            MessagePinMessage,
            MessageScreenshotTaken,
            MessageChatSetBackground,
            MessageChatSetTheme,
            MessageChatSetMessageAutoDeleteTime,
            MessageChatBoost,
            MessageForumTopicCreated,
            MessageForumTopicEdited,
            MessageForumTopicIsClosedToggled,
            MessageForumTopicIsHiddenToggled,
            MessageSuggestProfilePhoto,
            MessageSuggestBirthdate,
            MessageCustomServiceAction,
            MessageGameScore,
            MessagePaymentSuccessful,
            MessagePaymentSuccessfulBot,
            MessagePaymentRefunded,
            MessageGiftedPremium,
            MessagePremiumGiftCode,
            MessageGiveawayCreated,
            MessageGiveaway,
            MessageGiveawayCompleted,
            MessageGiveawayWinners,
            MessageGiftedStars,
            MessageGiftedTon,
            MessageGiveawayPrizeStars,
            MessageGift,
            MessageUpgradedGift,
            MessageRefundedUpgradedGift,
            MessagePaidMessagesRefunded,
            MessagePaidMessagePriceChanged,
            MessageDirectMessagePriceChanged,
            MessageChecklistTasksDone,
            MessageChecklistTasksAdded,
            MessageSuggestedPostApprovalFailed,
            MessageSuggestedPostApproved,
            MessageSuggestedPostDeclined,
            MessageSuggestedPostPaid,
            MessageSuggestedPostRefunded,
            MessageContactRegistered,
            MessageUsersShared,
            MessageChatShared,
            MessageBotWriteAccessAllowed,
            MessageWebAppDataSent,
            MessageWebAppDataReceived,
            MessagePassportDataSent,
            MessagePassportDataReceived,
            MessageProximityAlertTriggered,
            MessageUnsupported,
            None,
        ] = content
        r"""Content of the message"""
        self.reply_markup: Union[
            ReplyMarkupRemoveKeyboard,
            ReplyMarkupForceReply,
            ReplyMarkupShowKeyboard,
            ReplyMarkupInlineKeyboard,
            None,
        ] = reply_markup
        r"""Reply markup for the message; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["message"]:
        return "message"

    @classmethod
    def getClass(self) -> Literal["Message"]:
        return "Message"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "sender_id": self.sender_id,
            "chat_id": self.chat_id,
            "sending_state": self.sending_state,
            "scheduling_state": self.scheduling_state,
            "is_outgoing": self.is_outgoing,
            "is_pinned": self.is_pinned,
            "is_from_offline": self.is_from_offline,
            "can_be_saved": self.can_be_saved,
            "has_timestamped_media": self.has_timestamped_media,
            "is_channel_post": self.is_channel_post,
            "is_paid_star_suggested_post": self.is_paid_star_suggested_post,
            "is_paid_ton_suggested_post": self.is_paid_ton_suggested_post,
            "contains_unread_mention": self.contains_unread_mention,
            "date": self.date,
            "edit_date": self.edit_date,
            "forward_info": self.forward_info,
            "import_info": self.import_info,
            "interaction_info": self.interaction_info,
            "unread_reactions": self.unread_reactions,
            "fact_check": self.fact_check,
            "suggested_post_info": self.suggested_post_info,
            "reply_to": self.reply_to,
            "topic_id": self.topic_id,
            "self_destruct_type": self.self_destruct_type,
            "self_destruct_in": self.self_destruct_in,
            "auto_delete_in": self.auto_delete_in,
            "via_bot_user_id": self.via_bot_user_id,
            "sender_business_bot_user_id": self.sender_business_bot_user_id,
            "sender_boost_count": self.sender_boost_count,
            "paid_message_star_count": self.paid_message_star_count,
            "author_signature": self.author_signature,
            "media_album_id": self.media_album_id,
            "effect_id": self.effect_id,
            "restriction_info": self.restriction_info,
            "content": self.content,
            "reply_markup": self.reply_markup,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Message", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.sender_id = data.get("sender_id", None)
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.sending_state = data.get("sending_state", None)
            data_class.scheduling_state = data.get("scheduling_state", None)
            data_class.is_outgoing = data.get("is_outgoing", False)
            data_class.is_pinned = data.get("is_pinned", False)
            data_class.is_from_offline = data.get("is_from_offline", False)
            data_class.can_be_saved = data.get("can_be_saved", False)
            data_class.has_timestamped_media = data.get("has_timestamped_media", False)
            data_class.is_channel_post = data.get("is_channel_post", False)
            data_class.is_paid_star_suggested_post = data.get(
                "is_paid_star_suggested_post", False
            )
            data_class.is_paid_ton_suggested_post = data.get(
                "is_paid_ton_suggested_post", False
            )
            data_class.contains_unread_mention = data.get(
                "contains_unread_mention", False
            )
            data_class.date = int(data.get("date", 0))
            data_class.edit_date = int(data.get("edit_date", 0))
            data_class.forward_info = data.get("forward_info", None)
            data_class.import_info = data.get("import_info", None)
            data_class.interaction_info = data.get("interaction_info", None)
            data_class.unread_reactions = data.get("unread_reactions", None)
            data_class.fact_check = data.get("fact_check", None)
            data_class.suggested_post_info = data.get("suggested_post_info", None)
            data_class.reply_to = data.get("reply_to", None)
            data_class.topic_id = data.get("topic_id", None)
            data_class.self_destruct_type = data.get("self_destruct_type", None)
            data_class.self_destruct_in = data.get("self_destruct_in", 0.0)
            data_class.auto_delete_in = data.get("auto_delete_in", 0.0)
            data_class.via_bot_user_id = int(data.get("via_bot_user_id", 0))
            data_class.sender_business_bot_user_id = int(
                data.get("sender_business_bot_user_id", 0)
            )
            data_class.sender_boost_count = int(data.get("sender_boost_count", 0))
            data_class.paid_message_star_count = int(
                data.get("paid_message_star_count", 0)
            )
            data_class.author_signature = data.get("author_signature", "")
            data_class.media_album_id = int(data.get("media_album_id", 0))
            data_class.effect_id = int(data.get("effect_id", 0))
            data_class.restriction_info = data.get("restriction_info", None)
            data_class.content = data.get("content", None)
            data_class.reply_markup = data.get("reply_markup", None)

        return data_class


class Messages(TlObject):
    r"""Contains a list of messages

    Parameters:
        total_count (:class:`int`):
            Approximate total number of messages found

        messages (List[:class:`~pytdbot.types.Message`]):
            List of messages; messages may be null

    """

    def __init__(self, total_count: int = 0, messages: List[Message] = None) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of messages found"""
        self.messages: List[Message] = messages or []
        r"""List of messages; messages may be null"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messages"]:
        return "messages"

    @classmethod
    def getClass(self) -> Literal["Messages"]:
        return "Messages"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "messages": self.messages,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["Messages", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.messages = data.get("messages", None)

        return data_class


class FoundMessages(TlObject):
    r"""Contains a list of messages found by a search

    Parameters:
        total_count (:class:`int`):
            Approximate total number of messages found; \-1 if unknown

        messages (List[:class:`~pytdbot.types.Message`]):
            List of messages

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        total_count: int = 0,
        messages: List[Message] = None,
        next_offset: str = "",
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of messages found; \-1 if unknown"""
        self.messages: List[Message] = messages or []
        r"""List of messages"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundMessages"]:
        return "foundMessages"

    @classmethod
    def getClass(self) -> Literal["FoundMessages"]:
        return "FoundMessages"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "messages": self.messages,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundMessages", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.messages = data.get("messages", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class FoundChatMessages(TlObject):
    r"""Contains a list of messages found by a search in a given chat

    Parameters:
        total_count (:class:`int`):
            Approximate total number of messages found; \-1 if unknown

        messages (List[:class:`~pytdbot.types.Message`]):
            List of messages

        next_from_message_id (:class:`int`):
            The offset for the next request\. If 0, there are no more results

    """

    def __init__(
        self,
        total_count: int = 0,
        messages: List[Message] = None,
        next_from_message_id: int = 0,
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Approximate total number of messages found; \-1 if unknown"""
        self.messages: List[Message] = messages or []
        r"""List of messages"""
        self.next_from_message_id: int = int(next_from_message_id)
        r"""The offset for the next request\. If 0, there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundChatMessages"]:
        return "foundChatMessages"

    @classmethod
    def getClass(self) -> Literal["FoundChatMessages"]:
        return "FoundChatMessages"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "messages": self.messages,
            "next_from_message_id": self.next_from_message_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundChatMessages", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.messages = data.get("messages", None)
            data_class.next_from_message_id = int(data.get("next_from_message_id", 0))

        return data_class


class FoundPublicPosts(TlObject):
    r"""Contains a list of messages found by a public post search

    Parameters:
        messages (List[:class:`~pytdbot.types.Message`]):
            List of found public posts

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

        search_limits (:class:`~pytdbot.types.PublicPostSearchLimits`):
            Updated public post search limits after the query; repeated requests with the same query will be free; may be null if they didn't change

        are_limits_exceeded (:class:`bool`):
            True, if the query has failed because search limits are exceeded\. In this case search\_limits\.daily\_free\_query\_count will be equal to 0

    """

    def __init__(
        self,
        messages: List[Message] = None,
        next_offset: str = "",
        search_limits: PublicPostSearchLimits = None,
        are_limits_exceeded: bool = False,
    ) -> None:
        self.messages: List[Message] = messages or []
        r"""List of found public posts"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""
        self.search_limits: Union[PublicPostSearchLimits, None] = search_limits
        r"""Updated public post search limits after the query; repeated requests with the same query will be free; may be null if they didn't change"""
        self.are_limits_exceeded: bool = bool(are_limits_exceeded)
        r"""True, if the query has failed because search limits are exceeded\. In this case search\_limits\.daily\_free\_query\_count will be equal to 0"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundPublicPosts"]:
        return "foundPublicPosts"

    @classmethod
    def getClass(self) -> Literal["FoundPublicPosts"]:
        return "FoundPublicPosts"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "messages": self.messages,
            "next_offset": self.next_offset,
            "search_limits": self.search_limits,
            "are_limits_exceeded": self.are_limits_exceeded,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundPublicPosts", None]:
        if data:
            data_class = cls()
            data_class.messages = data.get("messages", None)
            data_class.next_offset = data.get("next_offset", "")
            data_class.search_limits = data.get("search_limits", None)
            data_class.are_limits_exceeded = data.get("are_limits_exceeded", False)

        return data_class


class MessagePosition(TlObject):
    r"""Contains information about a message in a specific position

    Parameters:
        position (:class:`int`):
            0\-based message position in the full list of suitable messages

        message_id (:class:`int`):
            Message identifier

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the message was sent

    """

    def __init__(self, position: int = 0, message_id: int = 0, date: int = 0) -> None:
        self.position: int = int(position)
        r"""0\-based message position in the full list of suitable messages"""
        self.message_id: int = int(message_id)
        r"""Message identifier"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the message was sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messagePosition"]:
        return "messagePosition"

    @classmethod
    def getClass(self) -> Literal["MessagePosition"]:
        return "MessagePosition"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "position": self.position,
            "message_id": self.message_id,
            "date": self.date,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessagePosition", None]:
        if data:
            data_class = cls()
            data_class.position = int(data.get("position", 0))
            data_class.message_id = int(data.get("message_id", 0))
            data_class.date = int(data.get("date", 0))

        return data_class


class MessagePositions(TlObject):
    r"""Contains a list of message positions

    Parameters:
        total_count (:class:`int`):
            Total number of messages found

        positions (List[:class:`~pytdbot.types.MessagePosition`]):
            List of message positions

    """

    def __init__(
        self, total_count: int = 0, positions: List[MessagePosition] = None
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Total number of messages found"""
        self.positions: List[MessagePosition] = positions or []
        r"""List of message positions"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messagePositions"]:
        return "messagePositions"

    @classmethod
    def getClass(self) -> Literal["MessagePositions"]:
        return "MessagePositions"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "positions": self.positions,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessagePositions", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.positions = data.get("positions", None)

        return data_class


class MessageCalendarDay(TlObject):
    r"""Contains information about found messages sent on a specific day

    Parameters:
        total_count (:class:`int`):
            Total number of found messages sent on the day

        message (:class:`~pytdbot.types.Message`):
            First message sent on the day

    """

    def __init__(self, total_count: int = 0, message: Message = None) -> None:
        self.total_count: int = int(total_count)
        r"""Total number of found messages sent on the day"""
        self.message: Union[Message, None] = message
        r"""First message sent on the day"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageCalendarDay"]:
        return "messageCalendarDay"

    @classmethod
    def getClass(self) -> Literal["MessageCalendarDay"]:
        return "MessageCalendarDay"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "message": self.message,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageCalendarDay", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.message = data.get("message", None)

        return data_class


class MessageCalendar(TlObject):
    r"""Contains information about found messages, split by days according to the option \"utc\_time\_offset\"

    Parameters:
        total_count (:class:`int`):
            Total number of found messages

        days (List[:class:`~pytdbot.types.MessageCalendarDay`]):
            Information about messages sent

    """

    def __init__(
        self, total_count: int = 0, days: List[MessageCalendarDay] = None
    ) -> None:
        self.total_count: int = int(total_count)
        r"""Total number of found messages"""
        self.days: List[MessageCalendarDay] = days or []
        r"""Information about messages sent"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageCalendar"]:
        return "messageCalendar"

    @classmethod
    def getClass(self) -> Literal["MessageCalendar"]:
        return "MessageCalendar"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_count": self.total_count,
            "days": self.days,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageCalendar", None]:
        if data:
            data_class = cls()
            data_class.total_count = int(data.get("total_count", 0))
            data_class.days = data.get("days", None)

        return data_class


class BusinessMessage(TlObject):
    r"""Describes a message from a business account as received by a bot

    Parameters:
        message (:class:`~pytdbot.types.Message`):
            The message

        reply_to_message (:class:`~pytdbot.types.Message`):
            Message that is replied by the message in the same chat; may be null if none

    """

    def __init__(
        self, message: Message = None, reply_to_message: Message = None
    ) -> None:
        self.message: Union[Message, None] = message
        r"""The message"""
        self.reply_to_message: Union[Message, None] = reply_to_message
        r"""Message that is replied by the message in the same chat; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessMessage"]:
        return "businessMessage"

    @classmethod
    def getClass(self) -> Literal["BusinessMessage"]:
        return "BusinessMessage"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "message": self.message,
            "reply_to_message": self.reply_to_message,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessMessage", None]:
        if data:
            data_class = cls()
            data_class.message = data.get("message", None)
            data_class.reply_to_message = data.get("reply_to_message", None)

        return data_class


class BusinessMessages(TlObject):
    r"""Contains a list of messages from a business account as received by a bot

    Parameters:
        messages (List[:class:`~pytdbot.types.BusinessMessage`]):
            List of business messages

    """

    def __init__(self, messages: List[BusinessMessage] = None) -> None:
        self.messages: List[BusinessMessage] = messages or []
        r"""List of business messages"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["businessMessages"]:
        return "businessMessages"

    @classmethod
    def getClass(self) -> Literal["BusinessMessages"]:
        return "BusinessMessages"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "messages": self.messages}

    @classmethod
    def from_dict(cls, data: dict) -> Union["BusinessMessages", None]:
        if data:
            data_class = cls()
            data_class.messages = data.get("messages", None)

        return data_class


class MessageSourceChatHistory(TlObject, MessageSource):
    r"""The message is from a chat history"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceChatHistory"]:
        return "messageSourceChatHistory"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceChatHistory", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceMessageThreadHistory(TlObject, MessageSource):
    r"""The message is from history of a message thread"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceMessageThreadHistory"]:
        return "messageSourceMessageThreadHistory"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceMessageThreadHistory", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceForumTopicHistory(TlObject, MessageSource):
    r"""The message is from history of a forum topic"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceForumTopicHistory"]:
        return "messageSourceForumTopicHistory"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceForumTopicHistory", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceDirectMessagesChatTopicHistory(TlObject, MessageSource):
    r"""The message is from history of a topic in a channel direct messages chat administered by the current user"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceDirectMessagesChatTopicHistory"]:
        return "messageSourceDirectMessagesChatTopicHistory"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["MessageSourceDirectMessagesChatTopicHistory", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceHistoryPreview(TlObject, MessageSource):
    r"""The message is from chat, message thread or forum topic history preview"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceHistoryPreview"]:
        return "messageSourceHistoryPreview"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceHistoryPreview", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceChatList(TlObject, MessageSource):
    r"""The message is from a chat list or a forum topic list"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceChatList"]:
        return "messageSourceChatList"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceChatList", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceSearch(TlObject, MessageSource):
    r"""The message is from search results, including file downloads, local file list, outgoing document messages, calendar"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceSearch"]:
        return "messageSourceSearch"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceSearch", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceChatEventLog(TlObject, MessageSource):
    r"""The message is from a chat event log"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceChatEventLog"]:
        return "messageSourceChatEventLog"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceChatEventLog", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceNotification(TlObject, MessageSource):
    r"""The message is from a notification"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceNotification"]:
        return "messageSourceNotification"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceNotification", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceScreenshot(TlObject, MessageSource):
    r"""The message was screenshotted; the source must be used only if the message content was visible during the screenshot"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceScreenshot"]:
        return "messageSourceScreenshot"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceScreenshot", None]:
        if data:
            data_class = cls()

        return data_class


class MessageSourceOther(TlObject, MessageSource):
    r"""The message is from some other source"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["messageSourceOther"]:
        return "messageSourceOther"

    @classmethod
    def getClass(self) -> Literal["MessageSource"]:
        return "MessageSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["MessageSourceOther", None]:
        if data:
            data_class = cls()

        return data_class


class AdvertisementSponsor(TlObject):
    r"""Information about the sponsor of an advertisement

    Parameters:
        url (:class:`str`):
            URL of the sponsor to be opened when the advertisement is clicked

        photo (:class:`~pytdbot.types.Photo`):
            Photo of the sponsor; may be null if must not be shown

        info (:class:`str`):
            Additional optional information about the sponsor to be shown along with the advertisement

    """

    def __init__(self, url: str = "", photo: Photo = None, info: str = "") -> None:
        self.url: Union[str, None] = url
        r"""URL of the sponsor to be opened when the advertisement is clicked"""
        self.photo: Union[Photo, None] = photo
        r"""Photo of the sponsor; may be null if must not be shown"""
        self.info: Union[str, None] = info
        r"""Additional optional information about the sponsor to be shown along with the advertisement"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["advertisementSponsor"]:
        return "advertisementSponsor"

    @classmethod
    def getClass(self) -> Literal["AdvertisementSponsor"]:
        return "AdvertisementSponsor"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "url": self.url,
            "photo": self.photo,
            "info": self.info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["AdvertisementSponsor", None]:
        if data:
            data_class = cls()
            data_class.url = data.get("url", "")
            data_class.photo = data.get("photo", None)
            data_class.info = data.get("info", "")

        return data_class


class SponsoredMessage(TlObject):
    r"""Describes a sponsored message

    Parameters:
        message_id (:class:`int`):
            Message identifier; unique for the chat to which the sponsored message belongs among both ordinary and sponsored messages

        is_recommended (:class:`bool`):
            True, if the message needs to be labeled as \"recommended\" instead of \"sponsored\"

        can_be_reported (:class:`bool`):
            True, if the message can be reported to Telegram moderators through reportChatSponsoredMessage

        content (:class:`~pytdbot.types.MessageContent`):
            Content of the message\. Currently, can be only of the types messageText, messageAnimation, messagePhoto, or messageVideo\. Video messages can be viewed fullscreen

        sponsor (:class:`~pytdbot.types.AdvertisementSponsor`):
            Information about the sponsor of the message

        title (:class:`str`):
            Title of the sponsored message

        button_text (:class:`str`):
            Text for the message action button

        accent_color_id (:class:`int`):
            Identifier of the accent color for title, button text and message background

        background_custom_emoji_id (:class:`int`):
            Identifier of a custom emoji to be shown on the message background; 0 if none

        additional_info (:class:`str`):
            If non\-empty, additional information about the sponsored message to be shown along with the message

    """

    def __init__(
        self,
        message_id: int = 0,
        is_recommended: bool = False,
        can_be_reported: bool = False,
        content: MessageContent = None,
        sponsor: AdvertisementSponsor = None,
        title: str = "",
        button_text: str = "",
        accent_color_id: int = 0,
        background_custom_emoji_id: int = 0,
        additional_info: str = "",
    ) -> None:
        self.message_id: int = int(message_id)
        r"""Message identifier; unique for the chat to which the sponsored message belongs among both ordinary and sponsored messages"""
        self.is_recommended: bool = bool(is_recommended)
        r"""True, if the message needs to be labeled as \"recommended\" instead of \"sponsored\""""
        self.can_be_reported: bool = bool(can_be_reported)
        r"""True, if the message can be reported to Telegram moderators through reportChatSponsoredMessage"""
        self.content: Union[
            MessageText,
            MessageAnimation,
            MessageAudio,
            MessageDocument,
            MessagePaidMedia,
            MessagePhoto,
            MessageSticker,
            MessageVideo,
            MessageVideoNote,
            MessageVoiceNote,
            MessageExpiredPhoto,
            MessageExpiredVideo,
            MessageExpiredVideoNote,
            MessageExpiredVoiceNote,
            MessageLocation,
            MessageVenue,
            MessageContact,
            MessageAnimatedEmoji,
            MessageDice,
            MessageGame,
            MessagePoll,
            MessageStory,
            MessageChecklist,
            MessageInvoice,
            MessageCall,
            MessageGroupCall,
            MessageVideoChatScheduled,
            MessageVideoChatStarted,
            MessageVideoChatEnded,
            MessageInviteVideoChatParticipants,
            MessageBasicGroupChatCreate,
            MessageSupergroupChatCreate,
            MessageChatChangeTitle,
            MessageChatChangePhoto,
            MessageChatDeletePhoto,
            MessageChatAddMembers,
            MessageChatJoinByLink,
            MessageChatJoinByRequest,
            MessageChatDeleteMember,
            MessageChatUpgradeTo,
            MessageChatUpgradeFrom,
            MessagePinMessage,
            MessageScreenshotTaken,
            MessageChatSetBackground,
            MessageChatSetTheme,
            MessageChatSetMessageAutoDeleteTime,
            MessageChatBoost,
            MessageForumTopicCreated,
            MessageForumTopicEdited,
            MessageForumTopicIsClosedToggled,
            MessageForumTopicIsHiddenToggled,
            MessageSuggestProfilePhoto,
            MessageSuggestBirthdate,
            MessageCustomServiceAction,
            MessageGameScore,
            MessagePaymentSuccessful,
            MessagePaymentSuccessfulBot,
            MessagePaymentRefunded,
            MessageGiftedPremium,
            MessagePremiumGiftCode,
            MessageGiveawayCreated,
            MessageGiveaway,
            MessageGiveawayCompleted,
            MessageGiveawayWinners,
            MessageGiftedStars,
            MessageGiftedTon,
            MessageGiveawayPrizeStars,
            MessageGift,
            MessageUpgradedGift,
            MessageRefundedUpgradedGift,
            MessagePaidMessagesRefunded,
            MessagePaidMessagePriceChanged,
            MessageDirectMessagePriceChanged,
            MessageChecklistTasksDone,
            MessageChecklistTasksAdded,
            MessageSuggestedPostApprovalFailed,
            MessageSuggestedPostApproved,
            MessageSuggestedPostDeclined,
            MessageSuggestedPostPaid,
            MessageSuggestedPostRefunded,
            MessageContactRegistered,
            MessageUsersShared,
            MessageChatShared,
            MessageBotWriteAccessAllowed,
            MessageWebAppDataSent,
            MessageWebAppDataReceived,
            MessagePassportDataSent,
            MessagePassportDataReceived,
            MessageProximityAlertTriggered,
            MessageUnsupported,
            None,
        ] = content
        r"""Content of the message\. Currently, can be only of the types messageText, messageAnimation, messagePhoto, or messageVideo\. Video messages can be viewed fullscreen"""
        self.sponsor: Union[AdvertisementSponsor, None] = sponsor
        r"""Information about the sponsor of the message"""
        self.title: Union[str, None] = title
        r"""Title of the sponsored message"""
        self.button_text: Union[str, None] = button_text
        r"""Text for the message action button"""
        self.accent_color_id: int = int(accent_color_id)
        r"""Identifier of the accent color for title, button text and message background"""
        self.background_custom_emoji_id: int = int(background_custom_emoji_id)
        r"""Identifier of a custom emoji to be shown on the message background; 0 if none"""
        self.additional_info: Union[str, None] = additional_info
        r"""If non\-empty, additional information about the sponsored message to be shown along with the message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sponsoredMessage"]:
        return "sponsoredMessage"

    @classmethod
    def getClass(self) -> Literal["SponsoredMessage"]:
        return "SponsoredMessage"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "message_id": self.message_id,
            "is_recommended": self.is_recommended,
            "can_be_reported": self.can_be_reported,
            "content": self.content,
            "sponsor": self.sponsor,
            "title": self.title,
            "button_text": self.button_text,
            "accent_color_id": self.accent_color_id,
            "background_custom_emoji_id": self.background_custom_emoji_id,
            "additional_info": self.additional_info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["SponsoredMessage", None]:
        if data:
            data_class = cls()
            data_class.message_id = int(data.get("message_id", 0))
            data_class.is_recommended = data.get("is_recommended", False)
            data_class.can_be_reported = data.get("can_be_reported", False)
            data_class.content = data.get("content", None)
            data_class.sponsor = data.get("sponsor", None)
            data_class.title = data.get("title", "")
            data_class.button_text = data.get("button_text", "")
            data_class.accent_color_id = int(data.get("accent_color_id", 0))
            data_class.background_custom_emoji_id = int(
                data.get("background_custom_emoji_id", 0)
            )
            data_class.additional_info = data.get("additional_info", "")

        return data_class


class SponsoredMessages(TlObject):
    r"""Contains a list of sponsored messages

    Parameters:
        messages (List[:class:`~pytdbot.types.SponsoredMessage`]):
            List of sponsored messages

        messages_between (:class:`int`):
            The minimum number of messages between shown sponsored messages, or 0 if only one sponsored message must be shown after all ordinary messages

    """

    def __init__(
        self, messages: List[SponsoredMessage] = None, messages_between: int = 0
    ) -> None:
        self.messages: List[SponsoredMessage] = messages or []
        r"""List of sponsored messages"""
        self.messages_between: int = int(messages_between)
        r"""The minimum number of messages between shown sponsored messages, or 0 if only one sponsored message must be shown after all ordinary messages"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sponsoredMessages"]:
        return "sponsoredMessages"

    @classmethod
    def getClass(self) -> Literal["SponsoredMessages"]:
        return "SponsoredMessages"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "messages": self.messages,
            "messages_between": self.messages_between,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["SponsoredMessages", None]:
        if data:
            data_class = cls()
            data_class.messages = data.get("messages", None)
            data_class.messages_between = int(data.get("messages_between", 0))

        return data_class


class SponsoredChat(TlObject):
    r"""Describes a sponsored chat

    Parameters:
        unique_id (:class:`int`):
            Unique identifier of this result

        chat_id (:class:`int`):
            Chat identifier

        sponsor_info (:class:`str`):
            Additional optional information about the sponsor to be shown along with the chat

        additional_info (:class:`str`):
            If non\-empty, additional information about the sponsored chat to be shown along with the chat

    """

    def __init__(
        self,
        unique_id: int = 0,
        chat_id: int = 0,
        sponsor_info: str = "",
        additional_info: str = "",
    ) -> None:
        self.unique_id: int = int(unique_id)
        r"""Unique identifier of this result"""
        self.chat_id: int = int(chat_id)
        r"""Chat identifier"""
        self.sponsor_info: Union[str, None] = sponsor_info
        r"""Additional optional information about the sponsor to be shown along with the chat"""
        self.additional_info: Union[str, None] = additional_info
        r"""If non\-empty, additional information about the sponsored chat to be shown along with the chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sponsoredChat"]:
        return "sponsoredChat"

    @classmethod
    def getClass(self) -> Literal["SponsoredChat"]:
        return "SponsoredChat"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "unique_id": self.unique_id,
            "chat_id": self.chat_id,
            "sponsor_info": self.sponsor_info,
            "additional_info": self.additional_info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["SponsoredChat", None]:
        if data:
            data_class = cls()
            data_class.unique_id = int(data.get("unique_id", 0))
            data_class.chat_id = int(data.get("chat_id", 0))
            data_class.sponsor_info = data.get("sponsor_info", "")
            data_class.additional_info = data.get("additional_info", "")

        return data_class


class SponsoredChats(TlObject):
    r"""Contains a list of sponsored chats

    Parameters:
        chats (List[:class:`~pytdbot.types.SponsoredChat`]):
            List of sponsored chats

    """

    def __init__(self, chats: List[SponsoredChat] = None) -> None:
        self.chats: List[SponsoredChat] = chats or []
        r"""List of sponsored chats"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["sponsoredChats"]:
        return "sponsoredChats"

    @classmethod
    def getClass(self) -> Literal["SponsoredChats"]:
        return "SponsoredChats"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "chats": self.chats}

    @classmethod
    def from_dict(cls, data: dict) -> Union["SponsoredChats", None]:
        if data:
            data_class = cls()
            data_class.chats = data.get("chats", None)

        return data_class


class VideoMessageAdvertisement(TlObject):
    r"""Describes an advertisent to be shown while a video from a message is watched

    Parameters:
        unique_id (:class:`int`):
            Unique identifier of this result

        text (:class:`str`):
            Text of the advertisement

        min_display_duration (:class:`int`):
            The minimum amount of time the advertisement must be displayed before it can be hidden by the user, in seconds

        max_display_duration (:class:`int`):
            The maximum amount of time the advertisement must be displayed before it must be automatically hidden, in seconds

        can_be_reported (:class:`bool`):
            True, if the advertisement can be reported to Telegram moderators through reportVideoMessageAdvertisement

        sponsor (:class:`~pytdbot.types.AdvertisementSponsor`):
            Information about the sponsor of the advertisement

        title (:class:`str`):
            Title of the sponsored message

        additional_info (:class:`str`):
            If non\-empty, additional information about the sponsored message to be shown along with the message

    """

    def __init__(
        self,
        unique_id: int = 0,
        text: str = "",
        min_display_duration: int = 0,
        max_display_duration: int = 0,
        can_be_reported: bool = False,
        sponsor: AdvertisementSponsor = None,
        title: str = "",
        additional_info: str = "",
    ) -> None:
        self.unique_id: int = int(unique_id)
        r"""Unique identifier of this result"""
        self.text: Union[str, None] = text
        r"""Text of the advertisement"""
        self.min_display_duration: int = int(min_display_duration)
        r"""The minimum amount of time the advertisement must be displayed before it can be hidden by the user, in seconds"""
        self.max_display_duration: int = int(max_display_duration)
        r"""The maximum amount of time the advertisement must be displayed before it must be automatically hidden, in seconds"""
        self.can_be_reported: bool = bool(can_be_reported)
        r"""True, if the advertisement can be reported to Telegram moderators through reportVideoMessageAdvertisement"""
        self.sponsor: Union[AdvertisementSponsor, None] = sponsor
        r"""Information about the sponsor of the advertisement"""
        self.title: Union[str, None] = title
        r"""Title of the sponsored message"""
        self.additional_info: Union[str, None] = additional_info
        r"""If non\-empty, additional information about the sponsored message to be shown along with the message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["videoMessageAdvertisement"]:
        return "videoMessageAdvertisement"

    @classmethod
    def getClass(self) -> Literal["VideoMessageAdvertisement"]:
        return "VideoMessageAdvertisement"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "unique_id": self.unique_id,
            "text": self.text,
            "min_display_duration": self.min_display_duration,
            "max_display_duration": self.max_display_duration,
            "can_be_reported": self.can_be_reported,
            "sponsor": self.sponsor,
            "title": self.title,
            "additional_info": self.additional_info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["VideoMessageAdvertisement", None]:
        if data:
            data_class = cls()
            data_class.unique_id = int(data.get("unique_id", 0))
            data_class.text = data.get("text", "")
            data_class.min_display_duration = int(data.get("min_display_duration", 0))
            data_class.max_display_duration = int(data.get("max_display_duration", 0))
            data_class.can_be_reported = data.get("can_be_reported", False)
            data_class.sponsor = data.get("sponsor", None)
            data_class.title = data.get("title", "")
            data_class.additional_info = data.get("additional_info", "")

        return data_class


class VideoMessageAdvertisements(TlObject):
    r"""Contains a list of advertisements to be shown while a video from a message is watched

    Parameters:
        advertisements (List[:class:`~pytdbot.types.VideoMessageAdvertisement`]):
            List of advertisements

        start_delay (:class:`int`):
            Delay before the first advertisement is shown, in seconds

        between_delay (:class:`int`):
            Delay between consecutive advertisements, in seconds

    """

    def __init__(
        self,
        advertisements: List[VideoMessageAdvertisement] = None,
        start_delay: int = 0,
        between_delay: int = 0,
    ) -> None:
        self.advertisements: List[VideoMessageAdvertisement] = advertisements or []
        r"""List of advertisements"""
        self.start_delay: int = int(start_delay)
        r"""Delay before the first advertisement is shown, in seconds"""
        self.between_delay: int = int(between_delay)
        r"""Delay between consecutive advertisements, in seconds"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["videoMessageAdvertisements"]:
        return "videoMessageAdvertisements"

    @classmethod
    def getClass(self) -> Literal["VideoMessageAdvertisements"]:
        return "VideoMessageAdvertisements"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "advertisements": self.advertisements,
            "start_delay": self.start_delay,
            "between_delay": self.between_delay,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["VideoMessageAdvertisements", None]:
        if data:
            data_class = cls()
            data_class.advertisements = data.get("advertisements", None)
            data_class.start_delay = int(data.get("start_delay", 0))
            data_class.between_delay = int(data.get("between_delay", 0))

        return data_class


class ReportOption(TlObject):
    r"""Describes an option to report an entity to Telegram

    Parameters:
        id (:class:`bytes`):
            Unique identifier of the option

        text (:class:`str`):
            Text of the option

    """

    def __init__(self, id: bytes = b"", text: str = "") -> None:
        self.id: Union[bytes, None] = id
        r"""Unique identifier of the option"""
        self.text: Union[str, None] = text
        r"""Text of the option"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reportOption"]:
        return "reportOption"

    @classmethod
    def getClass(self) -> Literal["ReportOption"]:
        return "ReportOption"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "id": self.id, "text": self.text}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReportOption", None]:
        if data:
            data_class = cls()
            data_class.id = b64decode(data.get("id", b""))
            data_class.text = data.get("text", "")

        return data_class


class ReportSponsoredResultOk(TlObject, ReportSponsoredResult):
    r"""The message was reported successfully"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reportSponsoredResultOk"]:
        return "reportSponsoredResultOk"

    @classmethod
    def getClass(self) -> Literal["ReportSponsoredResult"]:
        return "ReportSponsoredResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReportSponsoredResultOk", None]:
        if data:
            data_class = cls()

        return data_class


class ReportSponsoredResultFailed(TlObject, ReportSponsoredResult):
    r"""The sponsored message is too old or not found"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reportSponsoredResultFailed"]:
        return "reportSponsoredResultFailed"

    @classmethod
    def getClass(self) -> Literal["ReportSponsoredResult"]:
        return "ReportSponsoredResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReportSponsoredResultFailed", None]:
        if data:
            data_class = cls()

        return data_class


class ReportSponsoredResultOptionRequired(TlObject, ReportSponsoredResult):
    r"""The user must choose an option to report the message and repeat request with the chosen option

    Parameters:
        title (:class:`str`):
            Title for the option choice

        options (List[:class:`~pytdbot.types.ReportOption`]):
            List of available options

    """

    def __init__(self, title: str = "", options: List[ReportOption] = None) -> None:
        self.title: Union[str, None] = title
        r"""Title for the option choice"""
        self.options: List[ReportOption] = options or []
        r"""List of available options"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reportSponsoredResultOptionRequired"]:
        return "reportSponsoredResultOptionRequired"

    @classmethod
    def getClass(self) -> Literal["ReportSponsoredResult"]:
        return "ReportSponsoredResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "title": self.title, "options": self.options}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["ReportSponsoredResultOptionRequired", None]:
        if data:
            data_class = cls()
            data_class.title = data.get("title", "")
            data_class.options = data.get("options", None)

        return data_class


class ReportSponsoredResultAdsHidden(TlObject, ReportSponsoredResult):
    r"""Sponsored messages were hidden for the user in all chats"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reportSponsoredResultAdsHidden"]:
        return "reportSponsoredResultAdsHidden"

    @classmethod
    def getClass(self) -> Literal["ReportSponsoredResult"]:
        return "ReportSponsoredResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReportSponsoredResultAdsHidden", None]:
        if data:
            data_class = cls()

        return data_class


class ReportSponsoredResultPremiumRequired(TlObject, ReportSponsoredResult):
    r"""The user asked to hide sponsored messages, but Telegram Premium is required for this"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reportSponsoredResultPremiumRequired"]:
        return "reportSponsoredResultPremiumRequired"

    @classmethod
    def getClass(self) -> Literal["ReportSponsoredResult"]:
        return "ReportSponsoredResult"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["ReportSponsoredResultPremiumRequired", None]:
        if data:
            data_class = cls()

        return data_class


class FileDownload(TlObject):
    r"""Describes a file added to file download list

    Parameters:
        file_id (:class:`int`):
            File identifier

        message (:class:`~pytdbot.types.Message`):
            The message with the file

        add_date (:class:`int`):
            Point in time \(Unix timestamp\) when the file was added to the download list

        complete_date (:class:`int`):
            Point in time \(Unix timestamp\) when the file downloading was completed; 0 if the file downloading isn't completed

        is_paused (:class:`bool`):
            True, if downloading of the file is paused

    """

    def __init__(
        self,
        file_id: int = 0,
        message: Message = None,
        add_date: int = 0,
        complete_date: int = 0,
        is_paused: bool = False,
    ) -> None:
        self.file_id: int = int(file_id)
        r"""File identifier"""
        self.message: Union[Message, None] = message
        r"""The message with the file"""
        self.add_date: int = int(add_date)
        r"""Point in time \(Unix timestamp\) when the file was added to the download list"""
        self.complete_date: int = int(complete_date)
        r"""Point in time \(Unix timestamp\) when the file downloading was completed; 0 if the file downloading isn't completed"""
        self.is_paused: bool = bool(is_paused)
        r"""True, if downloading of the file is paused"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["fileDownload"]:
        return "fileDownload"

    @classmethod
    def getClass(self) -> Literal["FileDownload"]:
        return "FileDownload"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "file_id": self.file_id,
            "message": self.message,
            "add_date": self.add_date,
            "complete_date": self.complete_date,
            "is_paused": self.is_paused,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FileDownload", None]:
        if data:
            data_class = cls()
            data_class.file_id = int(data.get("file_id", 0))
            data_class.message = data.get("message", None)
            data_class.add_date = int(data.get("add_date", 0))
            data_class.complete_date = int(data.get("complete_date", 0))
            data_class.is_paused = data.get("is_paused", False)

        return data_class


class DownloadedFileCounts(TlObject):
    r"""Contains number of being downloaded and recently downloaded files found

    Parameters:
        active_count (:class:`int`):
            Number of active file downloads found, including paused

        paused_count (:class:`int`):
            Number of paused file downloads found

        completed_count (:class:`int`):
            Number of completed file downloads found

    """

    def __init__(
        self, active_count: int = 0, paused_count: int = 0, completed_count: int = 0
    ) -> None:
        self.active_count: int = int(active_count)
        r"""Number of active file downloads found, including paused"""
        self.paused_count: int = int(paused_count)
        r"""Number of paused file downloads found"""
        self.completed_count: int = int(completed_count)
        r"""Number of completed file downloads found"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["downloadedFileCounts"]:
        return "downloadedFileCounts"

    @classmethod
    def getClass(self) -> Literal["DownloadedFileCounts"]:
        return "DownloadedFileCounts"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "active_count": self.active_count,
            "paused_count": self.paused_count,
            "completed_count": self.completed_count,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["DownloadedFileCounts", None]:
        if data:
            data_class = cls()
            data_class.active_count = int(data.get("active_count", 0))
            data_class.paused_count = int(data.get("paused_count", 0))
            data_class.completed_count = int(data.get("completed_count", 0))

        return data_class


class FoundFileDownloads(TlObject):
    r"""Contains a list of downloaded files, found by a search

    Parameters:
        total_counts (:class:`~pytdbot.types.DownloadedFileCounts`):
            Total number of suitable files, ignoring offset

        files (List[:class:`~pytdbot.types.FileDownload`]):
            The list of files

        next_offset (:class:`str`):
            The offset for the next request\. If empty, then there are no more results

    """

    def __init__(
        self,
        total_counts: DownloadedFileCounts = None,
        files: List[FileDownload] = None,
        next_offset: str = "",
    ) -> None:
        self.total_counts: Union[DownloadedFileCounts, None] = total_counts
        r"""Total number of suitable files, ignoring offset"""
        self.files: List[FileDownload] = files or []
        r"""The list of files"""
        self.next_offset: Union[str, None] = next_offset
        r"""The offset for the next request\. If empty, then there are no more results"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["foundFileDownloads"]:
        return "foundFileDownloads"

    @classmethod
    def getClass(self) -> Literal["FoundFileDownloads"]:
        return "FoundFileDownloads"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "total_counts": self.total_counts,
            "files": self.files,
            "next_offset": self.next_offset,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["FoundFileDownloads", None]:
        if data:
            data_class = cls()
            data_class.total_counts = data.get("total_counts", None)
            data_class.files = data.get("files", None)
            data_class.next_offset = data.get("next_offset", "")

        return data_class


class NotificationSettingsScopePrivateChats(TlObject, NotificationSettingsScope):
    r"""Notification settings applied to all private and secret chats when the corresponding chat setting has a default value"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["notificationSettingsScopePrivateChats"]:
        return "notificationSettingsScopePrivateChats"

    @classmethod
    def getClass(self) -> Literal["NotificationSettingsScope"]:
        return "NotificationSettingsScope"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["NotificationSettingsScopePrivateChats", None]:
        if data:
            data_class = cls()

        return data_class


class NotificationSettingsScopeGroupChats(TlObject, NotificationSettingsScope):
    r"""Notification settings applied to all basic group and supergroup chats when the corresponding chat setting has a default value"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["notificationSettingsScopeGroupChats"]:
        return "notificationSettingsScopeGroupChats"

    @classmethod
    def getClass(self) -> Literal["NotificationSettingsScope"]:
        return "NotificationSettingsScope"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["NotificationSettingsScopeGroupChats", None]:
        if data:
            data_class = cls()

        return data_class


class NotificationSettingsScopeChannelChats(TlObject, NotificationSettingsScope):
    r"""Notification settings applied to all channel chats when the corresponding chat setting has a default value"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["notificationSettingsScopeChannelChats"]:
        return "notificationSettingsScopeChannelChats"

    @classmethod
    def getClass(self) -> Literal["NotificationSettingsScope"]:
        return "NotificationSettingsScope"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(
        cls, data: dict
    ) -> Union["NotificationSettingsScopeChannelChats", None]:
        if data:
            data_class = cls()

        return data_class


class ChatNotificationSettings(TlObject):
    r"""Contains information about notification settings for a chat or a forum topic

    Parameters:
        use_default_mute_for (:class:`bool`):
            If true, the value for the relevant type of chat or the forum chat is used instead of mute\_for

        mute_for (:class:`int`):
            Time left before notifications will be unmuted, in seconds

        use_default_sound (:class:`bool`):
            If true, the value for the relevant type of chat or the forum chat is used instead of sound\_id

        sound_id (:class:`int`):
            Identifier of the notification sound to be played for messages; 0 if sound is disabled

        use_default_show_preview (:class:`bool`):
            If true, the value for the relevant type of chat or the forum chat is used instead of show\_preview

        show_preview (:class:`bool`):
            True, if message content must be displayed in notifications

        use_default_mute_stories (:class:`bool`):
            If true, the value for the relevant type of chat is used instead of mute\_stories

        mute_stories (:class:`bool`):
            True, if story notifications are disabled for the chat

        use_default_story_sound (:class:`bool`):
            If true, the value for the relevant type of chat is used instead of story\_sound\_id

        story_sound_id (:class:`int`):
            Identifier of the notification sound to be played for stories; 0 if sound is disabled

        use_default_show_story_poster (:class:`bool`):
            If true, the value for the relevant type of chat is used instead of show\_story\_poster

        show_story_poster (:class:`bool`):
            True, if the chat that posted a story must be displayed in notifications

        use_default_disable_pinned_message_notifications (:class:`bool`):
            If true, the value for the relevant type of chat or the forum chat is used instead of disable\_pinned\_message\_notifications

        disable_pinned_message_notifications (:class:`bool`):
            If true, notifications for incoming pinned messages will be created as for an ordinary unread message

        use_default_disable_mention_notifications (:class:`bool`):
            If true, the value for the relevant type of chat or the forum chat is used instead of disable\_mention\_notifications

        disable_mention_notifications (:class:`bool`):
            If true, notifications for messages with mentions will be created as for an ordinary unread message

    """

    def __init__(
        self,
        use_default_mute_for: bool = False,
        mute_for: int = 0,
        use_default_sound: bool = False,
        sound_id: int = 0,
        use_default_show_preview: bool = False,
        show_preview: bool = False,
        use_default_mute_stories: bool = False,
        mute_stories: bool = False,
        use_default_story_sound: bool = False,
        story_sound_id: int = 0,
        use_default_show_story_poster: bool = False,
        show_story_poster: bool = False,
        use_default_disable_pinned_message_notifications: bool = False,
        disable_pinned_message_notifications: bool = False,
        use_default_disable_mention_notifications: bool = False,
        disable_mention_notifications: bool = False,
    ) -> None:
        self.use_default_mute_for: bool = bool(use_default_mute_for)
        r"""If true, the value for the relevant type of chat or the forum chat is used instead of mute\_for"""
        self.mute_for: int = int(mute_for)
        r"""Time left before notifications will be unmuted, in seconds"""
        self.use_default_sound: bool = bool(use_default_sound)
        r"""If true, the value for the relevant type of chat or the forum chat is used instead of sound\_id"""
        self.sound_id: int = int(sound_id)
        r"""Identifier of the notification sound to be played for messages; 0 if sound is disabled"""
        self.use_default_show_preview: bool = bool(use_default_show_preview)
        r"""If true, the value for the relevant type of chat or the forum chat is used instead of show\_preview"""
        self.show_preview: bool = bool(show_preview)
        r"""True, if message content must be displayed in notifications"""
        self.use_default_mute_stories: bool = bool(use_default_mute_stories)
        r"""If true, the value for the relevant type of chat is used instead of mute\_stories"""
        self.mute_stories: bool = bool(mute_stories)
        r"""True, if story notifications are disabled for the chat"""
        self.use_default_story_sound: bool = bool(use_default_story_sound)
        r"""If true, the value for the relevant type of chat is used instead of story\_sound\_id"""
        self.story_sound_id: int = int(story_sound_id)
        r"""Identifier of the notification sound to be played for stories; 0 if sound is disabled"""
        self.use_default_show_story_poster: bool = bool(use_default_show_story_poster)
        r"""If true, the value for the relevant type of chat is used instead of show\_story\_poster"""
        self.show_story_poster: bool = bool(show_story_poster)
        r"""True, if the chat that posted a story must be displayed in notifications"""
        self.use_default_disable_pinned_message_notifications: bool = bool(
            use_default_disable_pinned_message_notifications
        )
        r"""If true, the value for the relevant type of chat or the forum chat is used instead of disable\_pinned\_message\_notifications"""
        self.disable_pinned_message_notifications: bool = bool(
            disable_pinned_message_notifications
        )
        r"""If true, notifications for incoming pinned messages will be created as for an ordinary unread message"""
        self.use_default_disable_mention_notifications: bool = bool(
            use_default_disable_mention_notifications
        )
        r"""If true, the value for the relevant type of chat or the forum chat is used instead of disable\_mention\_notifications"""
        self.disable_mention_notifications: bool = bool(disable_mention_notifications)
        r"""If true, notifications for messages with mentions will be created as for an ordinary unread message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatNotificationSettings"]:
        return "chatNotificationSettings"

    @classmethod
    def getClass(self) -> Literal["ChatNotificationSettings"]:
        return "ChatNotificationSettings"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "use_default_mute_for": self.use_default_mute_for,
            "mute_for": self.mute_for,
            "use_default_sound": self.use_default_sound,
            "sound_id": self.sound_id,
            "use_default_show_preview": self.use_default_show_preview,
            "show_preview": self.show_preview,
            "use_default_mute_stories": self.use_default_mute_stories,
            "mute_stories": self.mute_stories,
            "use_default_story_sound": self.use_default_story_sound,
            "story_sound_id": self.story_sound_id,
            "use_default_show_story_poster": self.use_default_show_story_poster,
            "show_story_poster": self.show_story_poster,
            "use_default_disable_pinned_message_notifications": self.use_default_disable_pinned_message_notifications,
            "disable_pinned_message_notifications": self.disable_pinned_message_notifications,
            "use_default_disable_mention_notifications": self.use_default_disable_mention_notifications,
            "disable_mention_notifications": self.disable_mention_notifications,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatNotificationSettings", None]:
        if data:
            data_class = cls()
            data_class.use_default_mute_for = data.get("use_default_mute_for", False)
            data_class.mute_for = int(data.get("mute_for", 0))
            data_class.use_default_sound = data.get("use_default_sound", False)
            data_class.sound_id = int(data.get("sound_id", 0))
            data_class.use_default_show_preview = data.get(
                "use_default_show_preview", False
            )
            data_class.show_preview = data.get("show_preview", False)
            data_class.use_default_mute_stories = data.get(
                "use_default_mute_stories", False
            )
            data_class.mute_stories = data.get("mute_stories", False)
            data_class.use_default_story_sound = data.get(
                "use_default_story_sound", False
            )
            data_class.story_sound_id = int(data.get("story_sound_id", 0))
            data_class.use_default_show_story_poster = data.get(
                "use_default_show_story_poster", False
            )
            data_class.show_story_poster = data.get("show_story_poster", False)
            data_class.use_default_disable_pinned_message_notifications = data.get(
                "use_default_disable_pinned_message_notifications", False
            )
            data_class.disable_pinned_message_notifications = data.get(
                "disable_pinned_message_notifications", False
            )
            data_class.use_default_disable_mention_notifications = data.get(
                "use_default_disable_mention_notifications", False
            )
            data_class.disable_mention_notifications = data.get(
                "disable_mention_notifications", False
            )

        return data_class


class ScopeNotificationSettings(TlObject):
    r"""Contains information about notification settings for several chats

    Parameters:
        mute_for (:class:`int`):
            Time left before notifications will be unmuted, in seconds

        sound_id (:class:`int`):
            Identifier of the notification sound to be played; 0 if sound is disabled

        show_preview (:class:`bool`):
            True, if message content must be displayed in notifications

        use_default_mute_stories (:class:`bool`):
            If true, story notifications are received only for the first 5 chats from topChatCategoryUsers regardless of the value of mute\_stories

        mute_stories (:class:`bool`):
            True, if story notifications are disabled

        story_sound_id (:class:`int`):
            Identifier of the notification sound to be played for stories; 0 if sound is disabled

        show_story_poster (:class:`bool`):
            True, if the chat that posted a story must be displayed in notifications

        disable_pinned_message_notifications (:class:`bool`):
            True, if notifications for incoming pinned messages will be created as for an ordinary unread message

        disable_mention_notifications (:class:`bool`):
            True, if notifications for messages with mentions will be created as for an ordinary unread message

    """

    def __init__(
        self,
        mute_for: int = 0,
        sound_id: int = 0,
        show_preview: bool = False,
        use_default_mute_stories: bool = False,
        mute_stories: bool = False,
        story_sound_id: int = 0,
        show_story_poster: bool = False,
        disable_pinned_message_notifications: bool = False,
        disable_mention_notifications: bool = False,
    ) -> None:
        self.mute_for: int = int(mute_for)
        r"""Time left before notifications will be unmuted, in seconds"""
        self.sound_id: int = int(sound_id)
        r"""Identifier of the notification sound to be played; 0 if sound is disabled"""
        self.show_preview: bool = bool(show_preview)
        r"""True, if message content must be displayed in notifications"""
        self.use_default_mute_stories: bool = bool(use_default_mute_stories)
        r"""If true, story notifications are received only for the first 5 chats from topChatCategoryUsers regardless of the value of mute\_stories"""
        self.mute_stories: bool = bool(mute_stories)
        r"""True, if story notifications are disabled"""
        self.story_sound_id: int = int(story_sound_id)
        r"""Identifier of the notification sound to be played for stories; 0 if sound is disabled"""
        self.show_story_poster: bool = bool(show_story_poster)
        r"""True, if the chat that posted a story must be displayed in notifications"""
        self.disable_pinned_message_notifications: bool = bool(
            disable_pinned_message_notifications
        )
        r"""True, if notifications for incoming pinned messages will be created as for an ordinary unread message"""
        self.disable_mention_notifications: bool = bool(disable_mention_notifications)
        r"""True, if notifications for messages with mentions will be created as for an ordinary unread message"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["scopeNotificationSettings"]:
        return "scopeNotificationSettings"

    @classmethod
    def getClass(self) -> Literal["ScopeNotificationSettings"]:
        return "ScopeNotificationSettings"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "mute_for": self.mute_for,
            "sound_id": self.sound_id,
            "show_preview": self.show_preview,
            "use_default_mute_stories": self.use_default_mute_stories,
            "mute_stories": self.mute_stories,
            "story_sound_id": self.story_sound_id,
            "show_story_poster": self.show_story_poster,
            "disable_pinned_message_notifications": self.disable_pinned_message_notifications,
            "disable_mention_notifications": self.disable_mention_notifications,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ScopeNotificationSettings", None]:
        if data:
            data_class = cls()
            data_class.mute_for = int(data.get("mute_for", 0))
            data_class.sound_id = int(data.get("sound_id", 0))
            data_class.show_preview = data.get("show_preview", False)
            data_class.use_default_mute_stories = data.get(
                "use_default_mute_stories", False
            )
            data_class.mute_stories = data.get("mute_stories", False)
            data_class.story_sound_id = int(data.get("story_sound_id", 0))
            data_class.show_story_poster = data.get("show_story_poster", False)
            data_class.disable_pinned_message_notifications = data.get(
                "disable_pinned_message_notifications", False
            )
            data_class.disable_mention_notifications = data.get(
                "disable_mention_notifications", False
            )

        return data_class


class ReactionNotificationSourceNone(TlObject, ReactionNotificationSource):
    r"""Notifications for reactions are disabled"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionNotificationSourceNone"]:
        return "reactionNotificationSourceNone"

    @classmethod
    def getClass(self) -> Literal["ReactionNotificationSource"]:
        return "ReactionNotificationSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionNotificationSourceNone", None]:
        if data:
            data_class = cls()

        return data_class


class ReactionNotificationSourceContacts(TlObject, ReactionNotificationSource):
    r"""Notifications for reactions are shown only for reactions from contacts"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionNotificationSourceContacts"]:
        return "reactionNotificationSourceContacts"

    @classmethod
    def getClass(self) -> Literal["ReactionNotificationSource"]:
        return "ReactionNotificationSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionNotificationSourceContacts", None]:
        if data:
            data_class = cls()

        return data_class


class ReactionNotificationSourceAll(TlObject, ReactionNotificationSource):
    r"""Notifications for reactions are shown for all reactions"""

    def __init__(self) -> None:
        pass

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionNotificationSourceAll"]:
        return "reactionNotificationSourceAll"

    @classmethod
    def getClass(self) -> Literal["ReactionNotificationSource"]:
        return "ReactionNotificationSource"

    def to_dict(self) -> dict:
        return {"@type": self.getType()}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionNotificationSourceAll", None]:
        if data:
            data_class = cls()

        return data_class


class ReactionNotificationSettings(TlObject):
    r"""Contains information about notification settings for reactions

    Parameters:
        message_reaction_source (:class:`~pytdbot.types.ReactionNotificationSource`):
            Source of message reactions for which notifications are shown

        story_reaction_source (:class:`~pytdbot.types.ReactionNotificationSource`):
            Source of story reactions for which notifications are shown

        sound_id (:class:`int`):
            Identifier of the notification sound to be played; 0 if sound is disabled

        show_preview (:class:`bool`):
            True, if reaction sender and emoji must be displayed in notifications

    """

    def __init__(
        self,
        message_reaction_source: ReactionNotificationSource = None,
        story_reaction_source: ReactionNotificationSource = None,
        sound_id: int = 0,
        show_preview: bool = False,
    ) -> None:
        self.message_reaction_source: Union[
            ReactionNotificationSourceNone,
            ReactionNotificationSourceContacts,
            ReactionNotificationSourceAll,
            None,
        ] = message_reaction_source
        r"""Source of message reactions for which notifications are shown"""
        self.story_reaction_source: Union[
            ReactionNotificationSourceNone,
            ReactionNotificationSourceContacts,
            ReactionNotificationSourceAll,
            None,
        ] = story_reaction_source
        r"""Source of story reactions for which notifications are shown"""
        self.sound_id: int = int(sound_id)
        r"""Identifier of the notification sound to be played; 0 if sound is disabled"""
        self.show_preview: bool = bool(show_preview)
        r"""True, if reaction sender and emoji must be displayed in notifications"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["reactionNotificationSettings"]:
        return "reactionNotificationSettings"

    @classmethod
    def getClass(self) -> Literal["ReactionNotificationSettings"]:
        return "ReactionNotificationSettings"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "message_reaction_source": self.message_reaction_source,
            "story_reaction_source": self.story_reaction_source,
            "sound_id": self.sound_id,
            "show_preview": self.show_preview,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ReactionNotificationSettings", None]:
        if data:
            data_class = cls()
            data_class.message_reaction_source = data.get(
                "message_reaction_source", None
            )
            data_class.story_reaction_source = data.get("story_reaction_source", None)
            data_class.sound_id = int(data.get("sound_id", 0))
            data_class.show_preview = data.get("show_preview", False)

        return data_class


class DraftMessage(TlObject):
    r"""Contains information about a message draft

    Parameters:
        reply_to (:class:`~pytdbot.types.InputMessageReplyTo`):
            Information about the message to be replied; inputMessageReplyToStory is unsupported; may be null if none

        date (:class:`int`):
            Point in time \(Unix timestamp\) when the draft was created

        input_message_text (:class:`~pytdbot.types.InputMessageContent`):
            Content of the message draft; must be of the type inputMessageText, inputMessageVideoNote, or inputMessageVoiceNote

        effect_id (:class:`int`):
            Identifier of the effect to apply to the message when it is sent; 0 if none

        suggested_post_info (:class:`~pytdbot.types.InputSuggestedPostInfo`):
            Information about the suggested post; may be null if none

    """

    def __init__(
        self,
        reply_to: InputMessageReplyTo = None,
        date: int = 0,
        input_message_text: InputMessageContent = None,
        effect_id: int = 0,
        suggested_post_info: InputSuggestedPostInfo = None,
    ) -> None:
        self.reply_to: Union[
            InputMessageReplyToMessage,
            InputMessageReplyToExternalMessage,
            InputMessageReplyToStory,
            None,
        ] = reply_to
        r"""Information about the message to be replied; inputMessageReplyToStory is unsupported; may be null if none"""
        self.date: int = int(date)
        r"""Point in time \(Unix timestamp\) when the draft was created"""
        self.input_message_text: Union[
            InputMessageText,
            InputMessageAnimation,
            InputMessageAudio,
            InputMessageDocument,
            InputMessagePaidMedia,
            InputMessagePhoto,
            InputMessageSticker,
            InputMessageVideo,
            InputMessageVideoNote,
            InputMessageVoiceNote,
            InputMessageLocation,
            InputMessageVenue,
            InputMessageContact,
            InputMessageDice,
            InputMessageGame,
            InputMessageInvoice,
            InputMessagePoll,
            InputMessageStory,
            InputMessageChecklist,
            InputMessageForwarded,
            None,
        ] = input_message_text
        r"""Content of the message draft; must be of the type inputMessageText, inputMessageVideoNote, or inputMessageVoiceNote"""
        self.effect_id: int = int(effect_id)
        r"""Identifier of the effect to apply to the message when it is sent; 0 if none"""
        self.suggested_post_info: Union[InputSuggestedPostInfo, None] = (
            suggested_post_info
        )
        r"""Information about the suggested post; may be null if none"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["draftMessage"]:
        return "draftMessage"

    @classmethod
    def getClass(self) -> Literal["DraftMessage"]:
        return "DraftMessage"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "reply_to": self.reply_to,
            "date": self.date,
            "input_message_text": self.input_message_text,
            "effect_id": self.effect_id,
            "suggested_post_info": self.suggested_post_info,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["DraftMessage", None]:
        if data:
            data_class = cls()
            data_class.reply_to = data.get("reply_to", None)
            data_class.date = int(data.get("date", 0))
            data_class.input_message_text = data.get("input_message_text", None)
            data_class.effect_id = int(data.get("effect_id", 0))
            data_class.suggested_post_info = data.get("suggested_post_info", None)

        return data_class


class ChatTypePrivate(TlObject, ChatType):
    r"""An ordinary chat with a user

    Parameters:
        user_id (:class:`int`):
            User identifier

    """

    def __init__(self, user_id: int = 0) -> None:
        self.user_id: int = int(user_id)
        r"""User identifier"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatTypePrivate"]:
        return "chatTypePrivate"

    @classmethod
    def getClass(self) -> Literal["ChatType"]:
        return "ChatType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "user_id": self.user_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatTypePrivate", None]:
        if data:
            data_class = cls()
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class ChatTypeBasicGroup(TlObject, ChatType):
    r"""A basic group \(a chat with 0\-200 other users\)

    Parameters:
        basic_group_id (:class:`int`):
            Basic group identifier

    """

    def __init__(self, basic_group_id: int = 0) -> None:
        self.basic_group_id: int = int(basic_group_id)
        r"""Basic group identifier"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatTypeBasicGroup"]:
        return "chatTypeBasicGroup"

    @classmethod
    def getClass(self) -> Literal["ChatType"]:
        return "ChatType"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "basic_group_id": self.basic_group_id}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatTypeBasicGroup", None]:
        if data:
            data_class = cls()
            data_class.basic_group_id = int(data.get("basic_group_id", 0))

        return data_class


class ChatTypeSupergroup(TlObject, ChatType):
    r"""A supergroup or channel \(with unlimited members\)

    Parameters:
        supergroup_id (:class:`int`):
            Supergroup or channel identifier

        is_channel (:class:`bool`):
            True, if the supergroup is a channel

    """

    def __init__(self, supergroup_id: int = 0, is_channel: bool = False) -> None:
        self.supergroup_id: int = int(supergroup_id)
        r"""Supergroup or channel identifier"""
        self.is_channel: bool = bool(is_channel)
        r"""True, if the supergroup is a channel"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatTypeSupergroup"]:
        return "chatTypeSupergroup"

    @classmethod
    def getClass(self) -> Literal["ChatType"]:
        return "ChatType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "supergroup_id": self.supergroup_id,
            "is_channel": self.is_channel,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatTypeSupergroup", None]:
        if data:
            data_class = cls()
            data_class.supergroup_id = int(data.get("supergroup_id", 0))
            data_class.is_channel = data.get("is_channel", False)

        return data_class


class ChatTypeSecret(TlObject, ChatType):
    r"""A secret chat with a user

    Parameters:
        secret_chat_id (:class:`int`):
            Secret chat identifier

        user_id (:class:`int`):
            User identifier of the other user in the secret chat

    """

    def __init__(self, secret_chat_id: int = 0, user_id: int = 0) -> None:
        self.secret_chat_id: int = int(secret_chat_id)
        r"""Secret chat identifier"""
        self.user_id: int = int(user_id)
        r"""User identifier of the other user in the secret chat"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatTypeSecret"]:
        return "chatTypeSecret"

    @classmethod
    def getClass(self) -> Literal["ChatType"]:
        return "ChatType"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "secret_chat_id": self.secret_chat_id,
            "user_id": self.user_id,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatTypeSecret", None]:
        if data:
            data_class = cls()
            data_class.secret_chat_id = int(data.get("secret_chat_id", 0))
            data_class.user_id = int(data.get("user_id", 0))

        return data_class


class ChatFolderIcon(TlObject):
    r"""Represents an icon for a chat folder

    Parameters:
        name (:class:`str`):
            The chosen icon name for short folder representation; one of \"All\", \"Unread\", \"Unmuted\", \"Bots\", \"Channels\", \"Groups\", \"Private\", \"Custom\", \"Setup\", \"Cat\", \"Crown\", \"Favorite\", \"Flower\", \"Game\", \"Home\", \"Love\", \"Mask\", \"Party\", \"Sport\", \"Study\", \"Trade\", \"Travel\", \"Work\", \"Airplane\", \"Book\", \"Light\", \"Like\", \"Money\", \"Note\", \"Palette\"

    """

    def __init__(self, name: str = "") -> None:
        self.name: Union[str, None] = name
        r"""The chosen icon name for short folder representation; one of \"All\", \"Unread\", \"Unmuted\", \"Bots\", \"Channels\", \"Groups\", \"Private\", \"Custom\", \"Setup\", \"Cat\", \"Crown\", \"Favorite\", \"Flower\", \"Game\", \"Home\", \"Love\", \"Mask\", \"Party\", \"Sport\", \"Study\", \"Trade\", \"Travel\", \"Work\", \"Airplane\", \"Book\", \"Light\", \"Like\", \"Money\", \"Note\", \"Palette\""""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatFolderIcon"]:
        return "chatFolderIcon"

    @classmethod
    def getClass(self) -> Literal["ChatFolderIcon"]:
        return "ChatFolderIcon"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "name": self.name}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatFolderIcon", None]:
        if data:
            data_class = cls()
            data_class.name = data.get("name", "")

        return data_class


class ChatFolderName(TlObject):
    r"""Describes name of a chat folder

    Parameters:
        text (:class:`~pytdbot.types.FormattedText`):
            The text of the chat folder name; 1\-12 characters without line feeds\. May contain only CustomEmoji entities

        animate_custom_emoji (:class:`bool`):
            True, if custom emoji in the name must be animated

    """

    def __init__(
        self, text: FormattedText = None, animate_custom_emoji: bool = False
    ) -> None:
        self.text: Union[FormattedText, None] = text
        r"""The text of the chat folder name; 1\-12 characters without line feeds\. May contain only CustomEmoji entities"""
        self.animate_custom_emoji: bool = bool(animate_custom_emoji)
        r"""True, if custom emoji in the name must be animated"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatFolderName"]:
        return "chatFolderName"

    @classmethod
    def getClass(self) -> Literal["ChatFolderName"]:
        return "ChatFolderName"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "text": self.text,
            "animate_custom_emoji": self.animate_custom_emoji,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatFolderName", None]:
        if data:
            data_class = cls()
            data_class.text = data.get("text", None)
            data_class.animate_custom_emoji = data.get("animate_custom_emoji", False)

        return data_class


class ChatFolder(TlObject):
    r"""Represents a folder for user chats

    Parameters:
        name (:class:`~pytdbot.types.ChatFolderName`):
            The name of the folder

        icon (:class:`~pytdbot.types.ChatFolderIcon`):
            The chosen icon for the chat folder; may be null\. If null, use getChatFolderDefaultIconName to get default icon name for the folder

        color_id (:class:`int`):
            The identifier of the chosen color for the chat folder icon; from \-1 to 6\. If \-1, then color is disabled\. Can't be changed if folder tags are disabled or the current user doesn't have Telegram Premium subscription

        is_shareable (:class:`bool`):
            True, if at least one link has been created for the folder

        pinned_chat_ids (List[:class:`int`]):
            The chat identifiers of pinned chats in the folder\. There can be up to getOption\(\"chat\_folder\_chosen\_chat\_count\_max\"\) pinned and always included non\-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium

        included_chat_ids (List[:class:`int`]):
            The chat identifiers of always included chats in the folder\. There can be up to getOption\(\"chat\_folder\_chosen\_chat\_count\_max\"\) pinned and always included non\-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium

        excluded_chat_ids (List[:class:`int`]):
            The chat identifiers of always excluded chats in the folder\. There can be up to getOption\(\"chat\_folder\_chosen\_chat\_count\_max\"\) always excluded non\-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium

        exclude_muted (:class:`bool`):
            True, if muted chats need to be excluded

        exclude_read (:class:`bool`):
            True, if read chats need to be excluded

        exclude_archived (:class:`bool`):
            True, if archived chats need to be excluded

        include_contacts (:class:`bool`):
            True, if contacts need to be included

        include_non_contacts (:class:`bool`):
            True, if non\-contact users need to be included

        include_bots (:class:`bool`):
            True, if bots need to be included

        include_groups (:class:`bool`):
            True, if basic groups and supergroups need to be included

        include_channels (:class:`bool`):
            True, if channels need to be included

    """

    def __init__(
        self,
        name: ChatFolderName = None,
        icon: ChatFolderIcon = None,
        color_id: int = 0,
        is_shareable: bool = False,
        pinned_chat_ids: List[int] = None,
        included_chat_ids: List[int] = None,
        excluded_chat_ids: List[int] = None,
        exclude_muted: bool = False,
        exclude_read: bool = False,
        exclude_archived: bool = False,
        include_contacts: bool = False,
        include_non_contacts: bool = False,
        include_bots: bool = False,
        include_groups: bool = False,
        include_channels: bool = False,
    ) -> None:
        self.name: Union[ChatFolderName, None] = name
        r"""The name of the folder"""
        self.icon: Union[ChatFolderIcon, None] = icon
        r"""The chosen icon for the chat folder; may be null\. If null, use getChatFolderDefaultIconName to get default icon name for the folder"""
        self.color_id: int = int(color_id)
        r"""The identifier of the chosen color for the chat folder icon; from \-1 to 6\. If \-1, then color is disabled\. Can't be changed if folder tags are disabled or the current user doesn't have Telegram Premium subscription"""
        self.is_shareable: bool = bool(is_shareable)
        r"""True, if at least one link has been created for the folder"""
        self.pinned_chat_ids: List[int] = pinned_chat_ids or []
        r"""The chat identifiers of pinned chats in the folder\. There can be up to getOption\(\"chat\_folder\_chosen\_chat\_count\_max\"\) pinned and always included non\-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium"""
        self.included_chat_ids: List[int] = included_chat_ids or []
        r"""The chat identifiers of always included chats in the folder\. There can be up to getOption\(\"chat\_folder\_chosen\_chat\_count\_max\"\) pinned and always included non\-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium"""
        self.excluded_chat_ids: List[int] = excluded_chat_ids or []
        r"""The chat identifiers of always excluded chats in the folder\. There can be up to getOption\(\"chat\_folder\_chosen\_chat\_count\_max\"\) always excluded non\-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium"""
        self.exclude_muted: bool = bool(exclude_muted)
        r"""True, if muted chats need to be excluded"""
        self.exclude_read: bool = bool(exclude_read)
        r"""True, if read chats need to be excluded"""
        self.exclude_archived: bool = bool(exclude_archived)
        r"""True, if archived chats need to be excluded"""
        self.include_contacts: bool = bool(include_contacts)
        r"""True, if contacts need to be included"""
        self.include_non_contacts: bool = bool(include_non_contacts)
        r"""True, if non\-contact users need to be included"""
        self.include_bots: bool = bool(include_bots)
        r"""True, if bots need to be included"""
        self.include_groups: bool = bool(include_groups)
        r"""True, if basic groups and supergroups need to be included"""
        self.include_channels: bool = bool(include_channels)
        r"""True, if channels need to be included"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatFolder"]:
        return "chatFolder"

    @classmethod
    def getClass(self) -> Literal["ChatFolder"]:
        return "ChatFolder"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "name": self.name,
            "icon": self.icon,
            "color_id": self.color_id,
            "is_shareable": self.is_shareable,
            "pinned_chat_ids": self.pinned_chat_ids,
            "included_chat_ids": self.included_chat_ids,
            "excluded_chat_ids": self.excluded_chat_ids,
            "exclude_muted": self.exclude_muted,
            "exclude_read": self.exclude_read,
            "exclude_archived": self.exclude_archived,
            "include_contacts": self.include_contacts,
            "include_non_contacts": self.include_non_contacts,
            "include_bots": self.include_bots,
            "include_groups": self.include_groups,
            "include_channels": self.include_channels,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatFolder", None]:
        if data:
            data_class = cls()
            data_class.name = data.get("name", None)
            data_class.icon = data.get("icon", None)
            data_class.color_id = int(data.get("color_id", 0))
            data_class.is_shareable = data.get("is_shareable", False)
            data_class.pinned_chat_ids = data.get("pinned_chat_ids", None)
            data_class.included_chat_ids = data.get("included_chat_ids", None)
            data_class.excluded_chat_ids = data.get("excluded_chat_ids", None)
            data_class.exclude_muted = data.get("exclude_muted", False)
            data_class.exclude_read = data.get("exclude_read", False)
            data_class.exclude_archived = data.get("exclude_archived", False)
            data_class.include_contacts = data.get("include_contacts", False)
            data_class.include_non_contacts = data.get("include_non_contacts", False)
            data_class.include_bots = data.get("include_bots", False)
            data_class.include_groups = data.get("include_groups", False)
            data_class.include_channels = data.get("include_channels", False)

        return data_class


class ChatFolderInfo(TlObject):
    r"""Contains basic information about a chat folder

    Parameters:
        id (:class:`int`):
            Unique chat folder identifier

        name (:class:`~pytdbot.types.ChatFolderName`):
            The name of the folder

        icon (:class:`~pytdbot.types.ChatFolderIcon`):
            The chosen or default icon for the chat folder

        color_id (:class:`int`):
            The identifier of the chosen color for the chat folder icon; from \-1 to 6\. If \-1, then color is disabled

        is_shareable (:class:`bool`):
            True, if at least one link has been created for the folder

        has_my_invite_links (:class:`bool`):
            True, if the chat folder has invite links created by the current user

    """

    def __init__(
        self,
        id: int = 0,
        name: ChatFolderName = None,
        icon: ChatFolderIcon = None,
        color_id: int = 0,
        is_shareable: bool = False,
        has_my_invite_links: bool = False,
    ) -> None:
        self.id: int = int(id)
        r"""Unique chat folder identifier"""
        self.name: Union[ChatFolderName, None] = name
        r"""The name of the folder"""
        self.icon: Union[ChatFolderIcon, None] = icon
        r"""The chosen or default icon for the chat folder"""
        self.color_id: int = int(color_id)
        r"""The identifier of the chosen color for the chat folder icon; from \-1 to 6\. If \-1, then color is disabled"""
        self.is_shareable: bool = bool(is_shareable)
        r"""True, if at least one link has been created for the folder"""
        self.has_my_invite_links: bool = bool(has_my_invite_links)
        r"""True, if the chat folder has invite links created by the current user"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatFolderInfo"]:
        return "chatFolderInfo"

    @classmethod
    def getClass(self) -> Literal["ChatFolderInfo"]:
        return "ChatFolderInfo"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "id": self.id,
            "name": self.name,
            "icon": self.icon,
            "color_id": self.color_id,
            "is_shareable": self.is_shareable,
            "has_my_invite_links": self.has_my_invite_links,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatFolderInfo", None]:
        if data:
            data_class = cls()
            data_class.id = int(data.get("id", 0))
            data_class.name = data.get("name", None)
            data_class.icon = data.get("icon", None)
            data_class.color_id = int(data.get("color_id", 0))
            data_class.is_shareable = data.get("is_shareable", False)
            data_class.has_my_invite_links = data.get("has_my_invite_links", False)

        return data_class


class ChatFolderInviteLink(TlObject):
    r"""Contains a chat folder invite link

    Parameters:
        invite_link (:class:`str`):
            The chat folder invite link

        name (:class:`str`):
            Name of the link

        chat_ids (List[:class:`int`]):
            Identifiers of chats, included in the link

    """

    def __init__(
        self, invite_link: str = "", name: str = "", chat_ids: List[int] = None
    ) -> None:
        self.invite_link: Union[str, None] = invite_link
        r"""The chat folder invite link"""
        self.name: Union[str, None] = name
        r"""Name of the link"""
        self.chat_ids: List[int] = chat_ids or []
        r"""Identifiers of chats, included in the link"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatFolderInviteLink"]:
        return "chatFolderInviteLink"

    @classmethod
    def getClass(self) -> Literal["ChatFolderInviteLink"]:
        return "ChatFolderInviteLink"

    def to_dict(self) -> dict:
        return {
            "@type": self.getType(),
            "invite_link": self.invite_link,
            "name": self.name,
            "chat_ids": self.chat_ids,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatFolderInviteLink", None]:
        if data:
            data_class = cls()
            data_class.invite_link = data.get("invite_link", "")
            data_class.name = data.get("name", "")
            data_class.chat_ids = data.get("chat_ids", None)

        return data_class


class ChatFolderInviteLinks(TlObject):
    r"""Represents a list of chat folder invite links

    Parameters:
        invite_links (List[:class:`~pytdbot.types.ChatFolderInviteLink`]):
            List of the invite links

    """

    def __init__(self, invite_links: List[ChatFolderInviteLink] = None) -> None:
        self.invite_links: List[ChatFolderInviteLink] = invite_links or []
        r"""List of the invite links"""

    def __str__(self):
        return str(pytdbot.utils.obj_to_json(self, indent=4))

    @classmethod
    def getType(self) -> Literal["chatFolderInviteLinks"]:
        return "chatFolderInviteLinks"

    @classmethod
    def getClass(self) -> Literal["ChatFolderInviteLinks"]:
        return "ChatFolderInviteLinks"

    def to_dict(self) -> dict:
        return {"@type": self.getType(), "invite_links": self.invite_links}

    @classmethod
    def from_dict(cls, data: dict) -> Union["ChatFolderInviteLinks", None]:
        if data:
            data_class = cls()
            data_class.invite_links = data.get("invite_links", None)

        return data_class


class ChatFolderInviteLinkInfo(TlObject):
    r"""Contains information about an invite link to a chat folder

    Parameters:
        chat_folder_info (:class:`~pytdbot.types.ChatFolderInfo`):
            Basic information about the chat folder; chat folder identifier will be 0 if the user didn't have the chat folder yet

        missing_chat_ids (List[:class:`int`]):
            Identifiers of the chats from the link, which aren't added to the folder yet

        added_chat_ids (List[:class:`int`]):
            Identifiers of the chats from the link, which are 