# Math Bridge 🌉

Zaawansowana biblioteka matematyczna do obliczeń z funkcjami gamma, rozkładem Poissona i Kelly Criterion.

## ✨ Cechy

- ✅ Pełna kompatybilność z scipy, numpy, keras, xgboost
- ✅ Funkcje gamma (dodatnie, ujemne, zespolone)
- ✅ Rozkład Poissona z pełnymi statystykami
- ✅ Kelly Criterion i zarządzanie bankrollem
- ✅ Bezpieczne operacje matematyczne
- ✅ Brak błędów stringowych!

## 📦 Instalacja
```bash
pip install math-bridge
```

## 🚀 Szybki start
```python
from math_bridge import MathBridge

mb = MathBridge()

# Funkcja Gamma
gamma_val = mb.gamma_positive(5)
print(f"Γ(5) = {gamma_val}")  # 24.0

# Rozkład Poissona
prob = mb.poisson_probability(k=17, lmbda=12)
print(f"P(17; 12) = {prob:.6f}")

# Kelly Criterion
confidence = mb.kelly_confidence(errors=0, fraction_str='1/2')
print(f"Confidence = {confidence:.4f}")
```

## 📚 Dokumentacja

### Funkcje Gamma

#### `gamma_positive(z)`
Funkcja gamma dla wartości dodatnich.
```python
mb.gamma_positive(5)  # 24.0
mb.gamma_positive([1, 2, 3])  # array([1., 1., 2.])
```

#### `gamma_negative(x)`
Funkcja gamma dla wartości ujemnych.
```python
mb.gamma_negative(-1.5)  # 2.363271801207355
```

#### `gamma_complex(z_real, z_imag)`
Funkcja gamma dla liczb zespolonych.
```python
result = mb.gamma_complex(1, 2)
print(result['modulus'])    # Moduł
print(result['argument'])   # Argument
```

### Rozkład Poissona

#### `poisson_probability(k, lmbda)`
Prawdopodobieństwo P(k; λ).
```python
mb.poisson_probability(17, 12)  # 0.038325
```

#### `poisson_full_stats(lmbda, k=None)`
Pełne statystyki rozkładu.
```python
stats = mb.poisson_full_stats(12, 17)
print(stats)
# {
#   'lambda': 12,
#   'expected_value': 12,
#   'median': 12.0,
#   'variance': 12,
#   'entropy': 2.154,
#   'probability': 0.038325,
#   'cdf': 0.937
# }
```

### Kelly Criterion

#### `kelly_confidence(errors, fraction_str)`
Oblicza confidence % dla danej liczby błędów.
```python
mb.kelly_confidence(0, '1/2')   # 0.66 (66%)
mb.kelly_confidence(3, '1/3')   # 0.1875 (18.75%)
```

#### `combine_fractions(frac1, frac2)`
Łączy zbliżone frakcje Kelly.
```python
mb.combine_fractions('1/3', '1/2')  # '5/6'
mb.combine_fractions('1/4', '2/4')  # '3/4'
```

### Macierz wyników

#### `interpret_result(goals_home, goals_away)`
Interpretuje wynik meczu.
```python
result = mb.interpret_result(2, 0)
print(result['matrix_code'])  # 'A8'
print(result['result_1x2'])   # '1'
```

### Wzory złożone

#### `formula_3_plus(a, b, delta, gamma_val, beta_val)`
Wzór 3: [B + (δ + γ)]³ / [B + γ]³
```python
result = mb.formula_3_plus(2, 3, 1.5, 2.5, 3.0)
```

#### `delta_discriminant(a, b, c)`
Delta równania kwadratowego.
```python
delta = mb.delta_discriminant(1, -3, 2)
print(delta['delta'])         # 1
print(delta['solutions'])     # 2
```

## 🔧 Funkcje pomocnicze

### Bezpieczne operacje
```python
# Bezpieczne dzielenie (nie crashuje przy dzieleniu przez 0)
mb.safe_divide(10, 0)  # nan zamiast błędu

# Bezpieczne potęgowanie
mb.safe_power(2, 1000)  # Obsługuje bardzo duże liczby

# Bezpieczny logarytm
mb.safe_log(-5)  # nan zamiast błędu
```

### Walidacja danych
```python
result = mb.validate_input(0.5, 'probability')
print(result['valid'])  # True

result = mb.validate_input(-0.5, 'positive')
print(result['valid'])  # False
print(result['error'])  # 'Wartość musi być dodatnia'
```

### Konwersja danych
```python
# String → float
mb.from_string('1/2')     # 0.5
mb.from_string('3.14')    # 3.14

# Do numpy array
mb.to_numpy_array([1, 2, 3])  # array([1., 2., 3.])

# Formatowanie wyjścia
mb.format_output(3.14159, precision=2)  # "3.14"
```

## 📊 Przykłady użycia

### Przykład 1: Analiza rozkładu Poissona
```python
from math_bridge import MathBridge

mb = MathBridge()

# Parametry
lambda_param = 12
k_values = range(5, 20)

# Obliczenia
for k in k_values:
    prob = mb.poisson_probability(k, lambda_param)
    print(f"P({k}; {lambda_param}) = {prob:.6f}")
```

### Przykład 2: Kelly Criterion dla zakładów
```python
from math_bridge import MathBridge

mb = MathBridge()

# Sprawdź wszystkie frakcje dla 0 błędów
fractions = ['1/2', '1/3', '2/3', '1/4', '1/5']

for frac in fractions:
    confidence = mb.kelly_confidence(errors=0, fraction_str=frac)
    print(f"Frakcja {frac}: {confidence:.4f} ({confidence*100:.2f}%)")
```

### Przykład 3: Łączenie podobnych frakcji
```python
from math_bridge import MathBridge

mb = MathBridge()

# Sprawdź podobieństwo
similarity = mb.check_fraction_similarity(
    errors1=5, frac1='1/3',
    errors2=6, frac2='1/2',
    tolerance=0.005
)

if similarity['similar']:
    print(f"Frakcje są podobne! Łączę je:")
    print(f"Wynik: {similarity['combined']}")
```

### Przykład 4: Analiza wyników meczów
```python
from math_bridge import MathBridge

mb = MathBridge()

# Lista wyników
matches = [
    (3, 1),  # 3-1
    (2, 2),  # 2-2
    (0, 2),  # 0-2
]

for home, away in matches:
    result = mb.interpret_result(home, away)
    print(f"{home}-{away}: {result['matrix_code']} (typ: {result['result_1x2']})")
```

## 🧪 Testy
```bash
# Instalacja z opcją dev
pip install math-bridge[dev]

# Uruchomienie testów
pytest tests/

# Z pokryciem kodu
pytest --cov=math_bridge tests/
```

## 🤝 Integracja z innymi bibliotekami

### Z NumPy
```python
import numpy as np
from math_bridge import MathBridge

mb = MathBridge()

# Konwersja do numpy
data = [1, 2, 3, 4, 5]
np_array = mb.to_numpy_array(data)

# Obliczenia wektorowe
gammas = mb.gamma_positive(np_array)
print(gammas)  # array([  1.,   1.,   2.,   6.,  24.])
```

### Z Pandas
```python
import pandas as pd
from math_bridge import MathBridge

mb = MathBridge()

# DataFrame z danymi
df = pd.DataFrame({
    'k': [10, 12, 14, 16, 18],
    'lambda': [12, 12, 12, 12, 12]
})

# Oblicz prawdopodobieństwa
df['probability'] = df.apply(
    lambda row: mb.poisson_probability(row['k'], row['lambda']),
    axis=1
)
```

### Z XGBoost/Keras
```python
from math_bridge import MathBridge
import numpy as np

mb = MathBridge()

# Przygotuj features z matematycznych funkcji
def create_features(x):
    return np.array([
        mb.gamma_positive(x),
        mb.safe_log(x),
        mb.safe_sqrt(x)
    ])

# Użyj w modelu ML
X_train = np.array([create_features(x) for x in range(1, 100)])
# Teraz możesz użyć X_train w XGBoost/Keras!
```

## ⚠️ Uwagi

- **Brak błędów stringowych**: Wszystkie funkcje zwracają `np.nan` zamiast crashować
- **Bezpieczne operacje**: Dzielenie przez zero, logarytmy z ujemnych liczb - wszystko obsłużone
- **Kompatybilność**: Działa z scipy 1.5+, numpy 1.19+, Python 3.7+

## 📄 Licencja

MIT License - możesz używać komercyjnie!

## 🐛 Zgłaszanie błędów

GitHub Issues: https://github.com/twoj-username/math-bridge/issues

## 👨‍💻 Autor

**Twoje Imię**
- Email: twoj@email.com
- GitHub: @twoj-username

## 🌟 Wsparcie

Jeśli biblioteka Ci pomogła, zostaw ⭐ na GitHub!