"""
Math Bridge Core Module
Główny moduł z wszystkimi funkcjami matematycznymi
"""

import numpy as np
from scipy.special import gamma, loggamma, factorial
from scipy import integrate
import warnings
warnings.filterwarnings('ignore')


class MathBridge:
    """
    Główna klasa obsługująca zaawansowane obliczenia matematyczne
    Kompatybilna z scipy, numpy, keras, xgboost
    """
    
    # Stałe matematyczne
    EULER_CONSTANT = 0.5772156649
    GOLDEN_RATIO = 1.618033988749895
    
    def __init__(self):
        """Inicjalizacja biblioteki z cache"""
        self.cache = {}
        self._version = "1.0.0"
    
    # ============= FUNKCJE GAMMA =============
    
    @staticmethod
    def gamma_positive(z):
        """
        Funkcja gamma dla wartości dodatnich
        
        Args:
            z: wartość lub lista wartości
            
        Returns:
            float lub numpy.array: wartość funkcji gamma
            
        Example:
            >>> mb = MathBridge()
            >>> mb.gamma_positive(5)
            24.0
        """
        if isinstance(z, (list, tuple)):
            return np.array([MathBridge.gamma_positive(x) for x in z])
        try:
            return float(gamma(z))
        except:
            return np.nan
    
    @staticmethod
    def gamma_negative(x):
        """
        Funkcja gamma dla wartości ujemnych
        Wzór: Γ(x) = π / [sin(πx) * Γ(1-x)]
        
        Args:
            x: wartość ujemna
            
        Returns:
            float: wartość funkcji gamma
            
        Example:
            >>> mb = MathBridge()
            >>> mb.gamma_negative(-1.5)
            2.363271801207355
        """
        if x > 0:
            return MathBridge.gamma_positive(x)
        try:
            return np.pi / (np.sin(np.pi * x) * gamma(1 - x))
        except:
            return np.nan
    
    @staticmethod
    def gamma_complex(z_real, z_imag):
        """
        Funkcja gamma dla liczb zespolonych z = x + iy
        
        Args:
            z_real: część rzeczywista
            z_imag: część urojona
            
        Returns:
            dict: {'modulus', 'argument', 'real', 'imag'}
            
        Example:
            >>> mb = MathBridge()
            >>> result = mb.gamma_complex(1, 2)
            >>> print(result['modulus'])
        """
        z = complex(z_real, z_imag)
        try:
            result = gamma(z)
            return {
                'modulus': abs(result),
                'argument': np.angle(result),
                'real': result.real,
                'imag': result.imag,
                'complex': result
            }
        except:
            return {
                'modulus': np.nan,
                'argument': np.nan,
                'real': np.nan,
                'imag': np.nan,
                'complex': complex(np.nan, np.nan)
            }
    
    @staticmethod
    def gamma_factorial(k, lmbda):
        """
        Wzór: Γ([k+1]λ) / k!
        
        Args:
            k: parametr całkowity
            lmbda: parametr lambda
            
        Returns:
            float: wynik wzoru
        """
        try:
            numerator = gamma((k + 1) * lmbda)
            denominator = factorial(k)
            return numerator / denominator
        except:
            return np.nan
    
    @staticmethod
    def gamma_table_values():
        """
        Zwraca tabelę wartości gamma dla x od -2.5 do 15
        
        Returns:
            dict: słownik {x: gamma(x)}
        """
        special_values = {
            -2.5: -0.945309,
            -1.5: 2.363271,
            -0.5: -3.544907
        }
        
        # Dodaj wartości od 0.5 do 15
        for x in np.arange(0.5, 15.5, 0.5):
            try:
                special_values[x] = float(gamma(x))
            except:
                special_values[x] = np.nan
        
        return special_values
    
    # ============= FUNKCJE BETA =============
    
    @staticmethod
    def beta_function(x, y):
        """
        Funkcja beta B(x,y) = Γ(x)Γ(y) / Γ(x+y)
        
        Args:
            x, y: parametry
            
        Returns:
            float: wartość funkcji beta
        """
        try:
            return gamma(x) * gamma(y) / gamma(x + y)
        except:
            return np.nan
    
    @staticmethod
    def beta_incomplete(x, a, b):
        """
        Niekompletna funkcja beta B(x; a, b) = ∫₀ˣ t^(a-1) (1-t)^(b-1) dt
        
        Args:
            x: górna granica całki
            a, b: parametry
            
        Returns:
            float: wartość niekompletnej funkcji beta
        """
        try:
            result, _ = integrate.quad(
                lambda t: t**(a-1) * (1-t)**(b-1),
                0, x
            )
            return result
        except:
            return np.nan
    
    @staticmethod
    def beta_incomplete_normalized(x, a, b):
        """
        Znormalizowana niekompletna funkcja beta I_x(a,b) = B(x;a,b) / B(a,b)
        
        Args:
            x: punkt
            a, b: parametry
            
        Returns:
            float: znormalizowana wartość
        """
        try:
            return MathBridge.beta_incomplete(x, a, b) / MathBridge.beta_function(a, b)
        except:
            return np.nan
    
    @staticmethod
    def alpha_function(x, a, b):
        """
        Funkcja Alpha: A(x; a, b) = ∫₀ˣ t^(a+1) (1+t)^(b+1) dt
        
        Args:
            x: górna granica
            a, b: parametry
            
        Returns:
            float: wartość funkcji alpha
        """
        try:
            result, _ = integrate.quad(
                lambda t: t**(a+1) * (1+t)**(b+1),
                0, x
            )
            return result
        except:
            return np.nan
    
    # ============= WZORY ZŁOŻONE =============
    
    @staticmethod
    def formula_1(a, b, delta, gamma_val, beta_val):
        """
        Wzór 1: f(x) = [(B + (δ - γ)) / (a + γβ)]³
        
        Args:
            a, b: parametry podstawowe
            delta: wartość delta
            gamma_val: wartość gamma
            beta_val: wartość beta
            
        Returns:
            float: wynik wzoru
        """
        try:
            numerator = beta_val + (delta - gamma_val)
            denominator = a + gamma_val * beta_val
            if denominator == 0:
                return np.nan
            return (numerator / denominator) ** 3
        except:
            return np.nan
    
    @staticmethod
    def formula_3_plus(a, b, delta, gamma_val, beta_val):
        """
        Wzór 3: f(x) a+b = [B + (δ + γ)]³ / [B + γ]³
        
        Args:
            a, b: parametry
            delta, gamma_val, beta_val: wartości
            
        Returns:
            float: wynik wzoru
        """
        try:
            numerator = (beta_val + (delta + gamma_val)) ** 3
            denominator = (beta_val + gamma_val) ** 3
            if denominator == 0:
                return np.nan
            return numerator / denominator
        except:
            return np.nan
    
    @staticmethod
    def formula_3_minus(a, b, delta, gamma_val):
        """
        Wzór 3: f(x) a-b = [A - (δ - γ)]³ / [A - γ]³
        
        Args:
            a, b: parametry
            delta, gamma_val: wartości
            
        Returns:
            float: wynik wzoru
        """
        try:
            numerator = (a - (delta - gamma_val)) ** 3
            denominator = (a - gamma_val) ** 3
            if denominator == 0:
                return np.nan
            return numerator / denominator
        except:
            return np.nan
    
    @staticmethod
    def formula_sigma_ratio(sigma_val, gamma_vals, delta_vals):
        """
        Wzór 3: (Σ × γ) / (Σ × δ)
        
        Args:
            sigma_val: wartość sigma
            gamma_vals: lista/wartość gamma
            delta_vals: lista/wartość delta
            
        Returns:
            float: iloraz sigma
        """
        try:
            if not isinstance(gamma_vals, (list, tuple, np.ndarray)):
                gamma_vals = [gamma_vals]
            if not isinstance(delta_vals, (list, tuple, np.ndarray)):
                delta_vals = [delta_vals]
            
            sigma_gamma = sigma_val * np.sum(gamma_vals)
            sigma_delta = sigma_val * np.sum(delta_vals)
            
            if sigma_delta == 0:
                return np.nan
            
            ratio = sigma_gamma / sigma_delta
            
            # Interpretacja wyniku (wzór 4)
            if ratio > 1:
                return {'value': ratio, 'solutions': 2, 'type': 'cayley_octaves'}
            else:
                return {'value': ratio, 'solutions': 1, 'type': 'cayley_octaves'}
        except:
            return np.nan
    
    # ============= ROZKŁAD POISSONA =============
    
    @staticmethod
    def poisson_probability(k, lmbda):
        """
        Prawdopodobieństwo Poissona P(k; λ) = e^(-λ) * λ^k / k!
        
        Args:
            k: liczba zdarzeń
            lmbda: parametr λ (średnia)
            
        Returns:
            float: prawdopodobieństwo
        """
        try:
            return (np.exp(-lmbda) * (lmbda ** k)) / factorial(k)
        except:
            return np.nan
    
    @staticmethod
    def poisson_cdf(k, lmbda):
        """
        Dystrybuanta rozkładu Poissona F(k; λ)
        
        Args:
            k: liczba zdarzeń
            lmbda: parametr λ
            
        Returns:
            float: wartość dystrybuanty
        """
        try:
            return np.sum([MathBridge.poisson_probability(i, lmbda) for i in range(int(k) + 1)])
        except:
            return np.nan
    
    @staticmethod
    def poisson_expected_value(lmbda):
        """Wartość oczekiwana = λ"""
        return lmbda
    
    @staticmethod
    def poisson_variance(lmbda):
        """Wariancja = λ"""
        return lmbda
    
    @staticmethod
    def poisson_median(lmbda):
        """Mediana ≈ ⌊λ + 1/3 - 0.02/λ⌋"""
        return np.floor(lmbda + 1/3 - 0.02/lmbda)
    
    @staticmethod
    def poisson_entropy(lmbda):
        """
        Entropia rozkładu Poissona
        H ≈ ½log(2πλ) - 1/(12λ) - 1/(24λ²) - 19/(360λ³)
        """
        try:
            term1 = 0.5 * np.log(2 * np.pi * lmbda)
            term2 = -1 / (12 * lmbda)
            term3 = -1 / (24 * lmbda**2)
            term4 = -19 / (360 * lmbda**3)
            return term1 + term2 + term3 + term4
        except:
            return np.nan
    
    @staticmethod
    def poisson_full_stats(lmbda, k=None):
        """
        Pełne statystyki rozkładu Poissona
        
        Args:
            lmbda: parametr λ
            k: opcjonalnie - konkretna wartość k
            
        Returns:
            dict: wszystkie statystyki
        """
        stats = {
            'lambda': lmbda,
            'expected_value': lmbda,
            'median': MathBridge.poisson_median(lmbda),
            'variance': lmbda,
            'entropy': MathBridge.poisson_entropy(lmbda)
        }
        
        if k is not None:
            stats['k'] = k
            stats['probability'] = MathBridge.poisson_probability(k, lmbda)
            stats['cdf'] = MathBridge.poisson_cdf(k, lmbda)
        
        return stats
    
    # ============= WZORY POISSON Z DOKUMENTU =============
    
    @staticmethod
    def poisson_changing_formula(x, y, z):
        """
        Wzór 7: Zmieniający się Poisson
        z = x + iy * z - [Γ(z)] - arg Γ(z)
        
        Args:
            x, y, z: parametry
            
        Returns:
            dict: wyniki obliczeń
        """
        try:
            z_complex = complex(x, y * z)
            gamma_z = gamma(z_complex)
            
            return {
                'z': z_complex,
                'gamma_modulus': abs(gamma_z),
                'gamma_argument': np.angle(gamma_z),
                'formula_value': x - z + complex(x, y * z),
                'x_squared_times_sqrt_y': x**2 * np.sqrt(abs(y))
            }
        except:
            return None
    
    @staticmethod
    def calculate_omega_C(lmbda, k, gamma_value):
        """
        Obliczenie klucza Omega (C) - wzór 12
        C = P(k; λ) * γ³ * λ
        
        Args:
            lmbda: parametr λ
            k: liczba zdarzeń
            gamma_value: wartość gamma
            
        Returns:
            float: wartość C
        """
        try:
            poisson_prob = MathBridge.poisson_probability(k, lmbda)
            return poisson_prob * (gamma_value ** 3) * lmbda
        except:
            return np.nan
    
    # ============= WZORY ALFA/BETA Z DOKUMENTU =============
    
    @staticmethod
    def formula_11_alfa(x, y, sigma_val, gamma_val, delta_val):
        """
        Wzór 11 Alfa: (x,y) = σ²⁸ * γ²⁸ / δ²⁸
        
        Args:
            x, y: parametry (obecnie nieużywane w wzorze)
            sigma_val, gamma_val, delta_val: wartości
            
        Returns:
            float: wynik wzoru
        """
        try:
            if delta_val == 0:
                return np.nan
            result = (sigma_val ** 28) * ((gamma_val ** 28) / (delta_val ** 28))
            return result
        except:
            return np.nan
    
    @staticmethod
    def formula_11_beta(x, y, sigma_val, gamma_val, delta_val):
        """
        Wzór 11 Beta: (x,y) = σ²⁰ * γ²⁰ / δ²⁰
        
        Args:
            x, y: parametry (obecnie nieużywane w wzorze)
            sigma_val, gamma_val, delta_val: wartości
            
        Returns:
            float: wynik wzoru
        """
        try:
            if delta_val == 0:
                return np.nan
            result = (sigma_val ** 20) * ((gamma_val ** 20) / (delta_val ** 20))
            return result
        except:
            return np.nan
    
    @staticmethod
    def delta_discriminant(a, b, c):
        """
        Delta z wzoru kwadratowego: Δ = b² - 4ac
        
        Args:
            a, b, c: współczynniki równania ax² + bx + c = 0
            
        Returns:
            dict: {'delta': float, 'solutions': int, 'interpretation': str}
        """
        delta = b**2 - 4*a*c
        
        if delta > 0:
            interpretation = "Δ > 0: równanie ma dwa różne rozwiązania rzeczywiste"
            solutions = 2
        elif delta == 0:
            interpretation = "Δ = 0: równanie ma jedno rozwiązanie podwójne"
            solutions = 1
        else:
            interpretation = "Δ < 0: równanie ma dwa rozwiązania zespolone"
            solutions = 0
        
        return {
            'delta': delta,
            'solutions': solutions,
            'interpretation': interpretation
        }
    
    # ============= KELLY CRITERION =============
    
    @staticmethod
    def kelly_fraction(fraction_str, win_rate, bankroll_type='bookmaker'):
        """
        Obliczenie frakcji Kelly
        
        Args:
            fraction_str: '1/2', '1/3', '2/3', etc.
            win_rate: 0.75, 0.50, 0.33, etc.
            bankroll_type: 'bookmaker' lub 'personal'
            
        Returns:
            float: wartość frakcji Kelly
            
        Example:
            >>> mb = MathBridge()
            >>> mb.kelly_fraction('1/2', 0.75, 'personal')
            0.66
        """
        try:
            parts = fraction_str.split('/')
            numerator = float(parts[0])
            denominator = float(parts[1])
            fraction = numerator / denominator
            
            if bankroll_type == 'bookmaker':
                # Wzory dla bankrollu bukmachera (wzór 13)
                bookmaker_rates = {
                    (1, 2): 0.0088,
                    (1, 3): 0.0075,
                    (2, 3): 0.0058,
                    (1, 4): 0.0070,
                    (2, 4): 0.0053,
                    (3, 4): 0.0045,
                    (1, 5): 0.00367,
                    (2, 5): 0.00497,
                    (3, 5): 0.00413,
                    (4, 5): 0.00367
                }
                key = (int(numerator), int(denominator))
                return bookmaker_rates.get(key, 0.005)
            else:
                # Wzory dla osobistego bankrollu
                base_percentage = win_rate * fraction
                return base_percentage
        except:
            return np.nan
    
    @staticmethod
    def kelly_confidence(errors, fraction_str):
        """
        Zwraca confidence % na podstawie liczby błędów i frakcji Kelly (wzór 13)
        
        Args:
            errors: liczba błędów (0-6)
            fraction_str: frakcja Kelly np. '1/2'
            
        Returns:
            float: confidence % dla osobistego bankrollu
            
        Example:
            >>> mb = MathBridge()
            >>> mb.kelly_confidence(0, '1/2')
            0.66
        """
        error_rates = {
            0: 0.75,
            1: 0.50,
            2: 0.33,
            3: 0.25,
            4: 0.20,
            5: 0.167,
            6: 0.143
        }
        
        if errors not in error_rates:
            return np.nan
        
        win_rate = error_rates[errors]
        return MathBridge.kelly_fraction(fraction_str, win_rate, 'personal')
    
    @staticmethod
    def kelly_table(fraction_str):
        """
        Generuje pełną tabelę Kelly dla wszystkich poziomów błędów
        
        Args:
            fraction_str: frakcja Kelly np. '1/2'
            
        Returns:
            dict: {errors: {'win_rate', 'bookmaker', 'personal'}}
        """
        results = {}
        error_rates = {
            0: 0.75,
            1: 0.50,
            2: 0.33,
            3: 0.25,
            4: 0.20,
            5: 0.167,
            6: 0.143
        }
        
        for errors, win_rate in error_rates.items():
            results[errors] = {
                'win_rate': win_rate,
                'bookmaker': MathBridge.kelly_fraction(fraction_str, win_rate, 'bookmaker'),
                'personal': MathBridge.kelly_fraction(fraction_str, win_rate, 'personal')
            }
        
        return results
    
    @staticmethod
    def combine_fractions(frac1, frac2):
        """
        Łączenie frakcji Kelly gdy są zbliżone (wzór 13)
        Przykład: 1/3 (błędy 5) + 1/2 (błędy 6) = 5/6
        
        Args:
            frac1: pierwsza frakcja '1/3'
            frac2: druga frakcja '1/2'
            
        Returns:
            str: połączona frakcja '5/6'
            
        Example:
            >>> mb = MathBridge()
            >>> mb.combine_fractions('1/3', '1/2')
            '5/6'
        """
        try:
            # Parsowanie pierwszej frakcji
            p1 = frac1.split('/')
            num1, den1 = int(p1[0]), int(p1[1])
            
            # Parsowanie drugiej frakcji
            p2 = frac2.split('/')
            num2, den2 = int(p2[0]), int(p2[1])
            
            # Sprowadzenie do wspólnego mianownika
            if den1 == den2:
                new_num = num1 + num2
                new_den = den1
            else:
                # Znajdź najmniejszy wspólny mianownik
                from math import gcd
                lcm = (den1 * den2) // gcd(den1, den2)
                new_num = num1 * (lcm // den1) + num2 * (lcm // den2)
                new_den = lcm
            
            # Uproszczenie
            gcd_val = gcd(new_num, new_den)
            return f"{new_num // gcd_val}/{new_den // gcd_val}"
        except:
            return None
    
    @staticmethod
    def check_fraction_similarity(errors1, frac1, errors2, frac2, tolerance=0.005):
        """
        Sprawdza czy frakcje są zbliżone i można je połączyć
        
        Args:
            errors1, errors2: liczby błędów
            frac1, frac2: frakcje Kelly
            tolerance: tolerancja różnicy
            
        Returns:
            dict: {'similar': bool, 'diff': float, 'combined': str or None}
        """
        val1 = MathBridge.kelly_confidence(errors1, frac1)
        val2 = MathBridge.kelly_confidence(errors2, frac2)
        
        diff = abs(val1 - val2)
        similar = diff <= tolerance
        
        result = {
            'similar': similar,
            'diff': diff,
            'value1': val1,
            'value2': val2,
            'combined': None
        }
        
        if similar:
            result['combined'] = MathBridge.combine_fractions(frac1, frac2)
        
        return result
    
    # ============= MACIERZ WYNIKÓW =============
    
    @staticmethod
    def result_matrix_lookup(row, col):
        """
        Sprawdzenie wyniku z macierzy (wzór 10)
        
        Args:
            row: -5 do 5 (różnica bramek)
            col: 'A', 'B', 'C' (kategoria)
            
        Returns:
            str: identyfikator wyniku np. 'C4' (wynik 0-2), 'A3' (wynik 3-0)
            
        Example:
            >>> mb = MathBridge()
            >>> mb.result_matrix_lookup(-2, 'C')
            'C4'
        """
        matrix = {
            -5: {'A': 'A1', 'B': 'B1', 'C': 'C1'},
            -4: {'A': 'A2', 'B': 'B2', 'C': 'C2'},
            -3: {'A': 'A3', 'B': 'B3', 'C': 'C3'},
            -2: {'A': 'A4', 'B': 'B4', 'C': 'C4'},
            -1: {'A': 'A5', 'B': 'B5', 'C': 'C5'},
            0: {'A': 'A6', 'B': 'B6', 'C': 'C6'},
            1: {'A': 'A7', 'B': 'B7', 'C': 'C7'},
            2: {'A': 'A8', 'B': 'B8', 'C': 'C8'},
            3: {'A': 'A9', 'B': 'B9', 'C': 'C9'},
            4: {'A': 'A10', 'B': 'B10', 'C': 'C10'},
            5: {'A': 'A11', 'B': 'B11', 'C': 'C11'}
        }
        return matrix.get(row, {}).get(col, None)
    
    @staticmethod
    def interpret_result(goals_home, goals_away):
        """
        Interpretacja wyniku meczu i mapowanie na macierz
        
        Args:
            goals_home: bramki gospodarzy
            goals_away: bramki gości
            
        Returns:
            dict: {'diff', 'row', 'result_1x2', 'matrix_code'}
        """
        diff = goals_home - goals_away
        
        # Ogranicz do zakresu -5 do 5
        row = max(-5, min(5, diff))
        
        if diff > 0:
            result_1x2 = '1'  # Wygrana gospodarzy
            col = 'A'
        elif diff < 0:
            result_1x2 = '2'  # Wygrana gości
            col = 'C'
        else:
            result_1x2 = 'X'  # Remis
            col = 'B'
        
        matrix_code = MathBridge.result_matrix_lookup(row, col)
        
        return {
            'goals_home': goals_home,
            'goals_away': goals_away,
            'difference': diff,
            'row': row,
            'result_1x2': result_1x2,
            'column': col,
            'matrix_code': matrix_code
        }
    
    # ============= FUNKCJE POMOCNICZE =============
    
    @staticmethod
    def safe_divide(a, b, default=np.nan):
        """
        Bezpieczne dzielenie z obsługą dzielenia przez zero
        
        Args:
            a: licznik
            b: mianownik
            default: wartość domyślna przy dzieleniu przez zero
            
        Returns:
            float: wynik dzielenia lub default
        """
        try:
            if b == 0:
                return default
            return a / b
        except:
            return default
    
    @staticmethod
    def safe_power(base, exp, default=np.nan):
        """
        Bezpieczne potęgowanie
        
        Args:
            base: podstawa
            exp: wykładnik
            default: wartość domyślna przy błędzie
            
        Returns:
            float: wynik potęgowania lub default
        """
        try:
            return base ** exp
        except:
            return default
    
    @staticmethod
    def safe_sqrt(x, default=np.nan):
        """Bezpieczny pierwiastek kwadratowy"""
        try:
            if x < 0:
                return default
            return np.sqrt(x)
        except:
            return default
    
    @staticmethod
    def safe_log(x, base=np.e, default=np.nan):
        """Bezpieczny logarytm"""
        try:
            if x <= 0:
                return default
            if base == np.e:
                return np.log(x)
            else:
                return np.log(x) / np.log(base)
        except:
            return default
    
    # ============= NARZĘDZIA DIAGNOSTYCZNE =============
    
    def validate_input(self, value, value_type='float', min_val=None, max_val=None):
        """
        Walidacja danych wejściowych
        
        Args:
            value: wartość do sprawdzenia
            value_type: 'float', 'int', 'positive', 'probability'
            min_val, max_val: opcjonalne ograniczenia
            
        Returns:
            dict: {'valid': bool, 'error': str or None, 'value': converted_value}
        """
        try:
            if value_type == 'float':
                val = float(value)
            elif value_type == 'int':
                val = int(value)
            elif value_type == 'positive':
                val = float(value)
                if val <= 0:
                    return {'valid': False, 'error': 'Wartość musi być dodatnia', 'value': None}
            elif value_type == 'probability':
                val = float(value)
                if val < 0 or val > 1:
                    return {'valid': False, 'error': 'Prawdopodobieństwo musi być w [0,1]', 'value': None}
            else:
                val = value
            
            if min_val is not None and val < min_val:
                return {'valid': False, 'error': f'Wartość musi być >= {min_val}', 'value': None}
            if max_val is not None and val > max_val:
                return {'valid': False, 'error': f'Wartość musi być <= {max_val}', 'value': None}
            
            return {'valid': True, 'error': None, 'value': val}
        except:
            return {'valid': False, 'error': 'Nieprawidłowy format danych', 'value': None}
    
    def get_version(self):
        """Zwraca wersję biblioteki"""
        return self._version
    
    def clear_cache(self):
        """Czyści cache obliczeń"""
        self.cache = {}
        return True
    
    # ============= FUNKCJE INTEGRACYJNE =============
    
    @staticmethod
    def to_numpy_array(data, dtype=np.float64):
        """
        Konwersja danych do numpy array
        
        Args:
            data: lista, tuple lub pojedyncza wartość
            dtype: typ danych numpy
            
        Returns:
            numpy.ndarray: przekonwertowane dane
        """
        try:
            if isinstance(data, np.ndarray):
                return data.astype(dtype)
            elif isinstance(data, (list, tuple)):
                return np.array(data, dtype=dtype)
            else:
                return np.array([data], dtype=dtype)
        except:
            return np.array([np.nan])
    
    @staticmethod
    def from_string(value_str):
        """
        Parsowanie wartości ze stringa
        Obsługuje: liczby, frakcje, wyrażenia matematyczne
        
        Args:
            value_str: string z wartością
            
        Returns:
            float: sparsowana wartość
        """
        try:
            # Obsługa frakcji
            if '/' in value_str:
                parts = value_str.split('/')
                return float(parts[0]) / float(parts[1])
            # Obsługa liczb
            else:
                return float(value_str)
        except:
            return np.nan
    
    @staticmethod
    def format_output(value, precision=4, scientific=False):
        """
        Formatowanie wyniku do wyświetlenia
        
        Args:
            value: wartość do sformatowania
            precision: liczba miejsc po przecinku
            scientific: czy użyć notacji naukowej
            
        Returns:
            str: sformatowana wartość
        """
        try:
            if np.isnan(value):
                return "NaN"
            if np.isinf(value):
                return "∞" if value > 0 else "-∞"
            if scientific:
                return f"{value:.{precision}e}"
            else:
                return f"{value:.{precision}f}"
        except:
            return str(value)


# ============= PRZYKŁADY UŻYCIA =============

def example_usage():
    """Kompleksowe przykłady użycia biblioteki"""
    
    mb = MathBridge()
    
    print("=" * 60)
    print("MATH BRIDGE - PRZYKŁADY UŻYCIA")
    print(f"Wersja: {mb.get_version()}")
    print("=" * 60)
    
    # Przykład 1: Funkcje Gamma
    print("\n1. FUNKCJE GAMMA:")
    print(f"   Γ(5) = {mb.gamma_positive(5)}")
    print(f"   Γ(-1.5) = {mb.gamma_negative(-1.5):.6f}")
    gamma_complex = mb.gamma_complex(1, 2)
    print(f"   Γ(1+2i) moduł = {gamma_complex['modulus']:.6f}")
    
    # Przykład 2: Rozkład Poissona
    print("\n2. ROZKŁAD POISSONA (λ=12, k=17):")
    stats = mb.poisson_full_stats(12, 17)
    print(f"   P(k=17) = {stats['probability']:.6f}")
    print(f"   E[X] = {stats['expected_value']}")
    print(f"   Var(X) = {stats['variance']}")
    print(f"   H(X) = {stats['entropy']:.6f}")
    
    # Przykład 3: Wzory złożone
    print("\n3. WZORY ZŁOŻONE:")
    result3 = mb.formula_3_plus(2, 3, 1.5, 2.5, 3.0)
    print(f"   Wzór 3 (plus) = {result3:.6f}")
    
    # Przykład 4: Kelly Criterion
    print("\n4. KELLY CRITERION:")
    print("   Frakcja 1/2, 0 błędów:")
    kelly_val = mb.kelly_confidence(0, '1/2')
    print(f"   Confidence = {kelly_val:.4f} ({kelly_val*100:.2f}%)")
    
    # Przykład 5: Łączenie frakcji
    print("\n5. ŁĄCZENIE FRAKCJI:")
    combined = mb.combine_fractions('1/3', '1/2')
    print(f"   1/3 + 1/2 = {combined}")
    
    # Przykład 6: Macierz wyników
    print("\n6. MACIERZ WYNIKÓW:")
    result = mb.interpret_result(2, 0)
    print(f"   Wynik 2-0:")
    print(f"   - Różnica: {result['difference']}")
    print(f"   - Rezultat: {result['result_1x2']}")
    print(f"   - Kod macierzy: {result['matrix_code']}")
    
    # Przykład 7: Delta
    print("\n7. DELTA (równanie kwadratowe):")
    delta_result = mb.delta_discriminant(1, -3, 2)
    print(f"   Δ = {delta_result['delta']}")
    print(f"   {delta_result['interpretation']}")
    
    # Przykład 8: Wzór 11
    print("\n8. WZÓR 11 (Alfa/Beta):")
    alfa_result = mb.formula_11_alfa(0, 0, 1.5, 2.0, 1.0)
    print(f"   Alfa(σ=1.5, γ=2.0, δ=1.0) = {mb.format_output(alfa_result)}")
    
    print("\n" + "=" * 60)
    print("Wszystkie funkcje działają poprawnie!")
    print("=" * 60)


if __name__ == "__main__":
    example_usage()