"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'services';
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = props.type ?? ServiceType.CLUSTER_IP;
        }
        this._externalIPs = props.externalIPs ?? [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        if (props.selector) {
            this.select(props.selector);
        }
        for (const port of props.ports ?? []) {
            this.bind(port.port, port);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        const ingr = options.ingress ?? new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Ports for this service.
     *
     * Use `bind()` to bind additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Return the first port of the service.
     */
    get port() {
        return [...this._ports][0].port;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    bind(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * Require this service to select pods matching the selector.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    select(selector) {
        const labels = selector.toPodSelectorConfig().labelSelector._toKube().matchLabels ?? {};
        for (const [key, value] of Object.entries(labels)) {
            this._selector[key] = value;
        }
    }
    /**
     * Require this service to select pods with this label.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    selectLabel(key, value) {
        this._selector[key] = value;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-25.Service", version: "2.4.21" };
//# sourceMappingURL=data:application/json;base64,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