# coding=utf-8
"""Module for parsing EnergyPlus Result Data Dictionary (RDD) files."""
from __future__ import division

from honeybee.search import filter_array_by_keywords
import os


class RDD(object):
    """Object for parsing EnergyPlus Result Data Dictionary (RDD) files.

    Args:
        file_path: Full path to a RDD file that was generated by EnergyPlus.

    Properties:
        * file_path
        * output_names
    """

    def __init__(self, file_path):
        """Initialize RDD"""
        assert os.path.isfile(file_path), 'No file was found at {}'.format(file_path)
        assert file_path.endswith('.rdd'), \
            '{} is not an RDD file ending in .rdd.'.format(file_path)
        self._file_path = file_path
        self._output_names = None

    @property
    def file_path(self):
        """Get the path to the .rdd file."""
        return self._file_path

    @property
    def output_names(self):
        """Get a list of all output names in the .rdd file."""
        if not self._output_names:
            self._parse_outputs()
        return self._output_names

    def filter_outputs_by_keywords(self, keywords):
        """Get a list of outputs in the RDD file filtered by keyword.

        Args:
            keywords: A list of keywords that will be used to filter the output names.
        """
        return filter_array_by_keywords(self.output_names, keywords, True)

    def _parse_outputs(self):
        """Parse all of the outputs from the file."""
        with open(self._file_path) as rdd_result:
            self._output_names = tuple(line.split(',')[-2] for line in rdd_result
                                       if not line.startswith('!'))

    def ToString(self):
        """Overwrite .NET ToString."""
        return self.__repr__()

    def __repr__(self):
        return 'Energy RDD Result: {}'.format(self.file_path)
