import pandas as pd


def getSimultaneosChargeDischarge(esM, compName, threshold=0.0):
    """Get operation time series in which simultaneous charging and discharging occurs.

    :param esM: EnergySystemModel instance representing the energy system in which the component should be modeled.
    :type esM: EnergySystemModel class instance

    :param compName: component name
    :type compName: string

    :param threshold: threshold for check of simultaneous operation
        |br| * the default value is 0.0
    :type threshold: float

    :return: simultaneousOperation: Dictionary with region as keys and pd.DataFrame as value, in which timesteps
        with simultaneous charge and discharge are listed.
    :rtype simultaneousChargeDischarge: dict
    """
    simultaneousOperation = {}
    for ip in esM.investmentPeriodNames:
        storageModel = esM.componentModelingDict["StorageModel"]
        tsCharge = storageModel.chargeOperationVariablesOptimum[
            esM.investmentPeriodNames[ip]
        ].loc[compName]
        tsDischarge = storageModel.dischargeOperationVariablesOptimum[
            esM.investmentPeriodNames[ip]
        ].loc[compName]
        simultaneousOperation = dict()
        for region in tsCharge.index:
            _tsCharge = tsCharge.loc[region]
            _tsDischarge = tsDischarge.loc[region]
            _tsCharge = _tsCharge.rename("Charge")
            _tsDischarge = _tsDischarge.rename("Discharge")
            simultaneousOperation[ip][region] = pd.concat(
                [_tsCharge, _tsDischarge], axis=1
            )
            simultaneousOperation[ip][region] = simultaneousOperation[ip][region][
                simultaneousOperation[region] > threshold
            ].dropna()
        # If no simultaneous charge and discharge occurs ts[region][ts[region] > 0] will only return nan values. After
        # dropping them

    if len(esM.investmentPeriodsList) == 1:
        # for single year optimization
        return simultaneousOperation[esM.investmentPeriodNames[0]]
    return simultaneousOperation
