import numpy as np
from spglib import get_symmetry_dataset

__author__ = "Artem Dembitskiy"

class Mesh:
    """
    This class implements mesh generation and finding irreducible mesh
    points given the rotations and translations symmetry operations.

    Example
    -------
    
    mesh = Mesh.from_atoms(atoms, resolution = 0.2, symprec = 0.1)
    ir_grid_points, mapping = mesh.get_irreducible_mapping()
    
    """
    
    def __init__(self, mesh_shape, rotations, translations, symmetry_dataset=None):
        """
        Initialization.

        
        Parameters
        ----------
        
        mesh_shape: tuple of integers, (nx, ny, nz)
            shape of the mesh

        rotations: list of numpy arrays of shape (3, 3)
            rotation matrices

        translations: list of numpy arrays of shape (3, 1)
            translation vector
        
        symmetry_dataset: dict generated by spglib library, None by default (Optional)
            dataset containing symmetry operations, used for tests
        """
        
        self.mesh_shape = mesh_shape
        self.rotations = rotations
        self.translations = translations
        self.symmetry_dataset = symmetry_dataset
        self.mesh = self._generate_mesh()

    @property
    def addresses(self):
        return self.mesh
        
    
    @classmethod
    def from_atoms(cls, atoms, resolution=0.2, symprec=None):
        """
        Create Mesh object  form ASE's Atoms object.

        
        Parameters
        ----------
        
        atoms: ASE's Atoms object
            atomic structure
        
        resolution: float, 0.2 by default
            distance between points of the mesh in angstroms
        symprec: None or float, None by default
            symmetry precision
            used by spglib' get_symmetry_dataset to find symmetry operations
        """
  
        rotations = [np.eye(3, dtype=int)]
        translations = [np.zeros(3)]
        symmetry_dataset = None
        a, b, c = np.linalg.norm(atoms.cell, axis = 1)
        nx, ny, nz = int(a // resolution), int(b // resolution), int(c // resolution)
        nx, ny, nz = [n + (n % 2 + 1) % 2 for n in (nx, ny, nz)] # make odd numbers
        mesh_shape = (nx, ny, nz)
        
        if symprec:
            spglib_cell = (atoms.cell, atoms.get_scaled_positions(), atoms.numbers)
            symmetry_dataset = get_symmetry_dataset(spglib_cell, symprec=symprec)
            rotations = symmetry_dataset.rotations
            translations = symmetry_dataset.translations
        
        return cls(mesh_shape, rotations, translations, symmetry_dataset=symmetry_dataset)
        
    def _generate_mesh(self):
        """Generate integer grid points (i,j,k) for a 3D mesh, including edges and faces."""
        grid = np.mgrid[
            0:self.mesh_shape[0],
            0:self.mesh_shape[1],
            0:self.mesh_shape[2]
        ].reshape(3, -1).T
        return grid  # shape (N,3), integer

    
    def _index_to_linear(self, indices):
        """Convert 3D integer indices -> 1D linear indices for fast lookup."""
        return np.ravel_multi_index(indices.T, self.mesh_shape)

    
    def _apply_sym_ops(self):
        """Apply symmetry ops to integer grid points, wrapping by mesh_shape."""
        all_images = []
        mesh_shape = np.array(self.mesh_shape)
        for R, t in zip(self.rotations, self.translations):
            t_mesh = np.rint(t * (mesh_shape - 1)).astype(int)
            transformed = (self.mesh @ R.T + t_mesh) % (mesh_shape - 1)
            all_images.append(transformed)
        return np.stack(all_images, axis=0)  # (nsym, npoints, 3)

    
    def get_irreducible_mapping(self):
        """
        Find symmetry-equivalent points on an integer 3D mesh.
        
        Returns
        -------
        indices of the irreducible points of the mesh and 
        mapping indices to restore full mesh from irreducible points
        
        """
        
        npoints = len(self.mesh)
        nsym = len(self.rotations)
        
        lin_indices = self._index_to_linear(self.mesh)
        index_map = np.arange(npoints, dtype=int)
    
        all_images = self._apply_sym_ops()
        
        transformed_lin = self._index_to_linear(all_images.reshape(-1, 3))
        transformed_lin = transformed_lin.reshape(nsym, npoints)
        
        eq_ids = index_map[transformed_lin]
        eq_ids = np.concatenate([eq_ids, index_map[None, :]], axis=0)
        irreducible_ids = eq_ids.min(axis=0)
        
        unique, inverse = np.unique(irreducible_ids, return_inverse=True)
        return unique, inverse