"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CREATE_FAILED_PHYSICAL_ID_MARKER = void 0;
exports.cfnResponseSuccess = cfnResponseSuccess;
exports.cfnResponseFailed = cfnResponseFailed;
exports.startExecution = startExecution;
const client_sfn_1 = require("@aws-sdk/client-sfn"); // eslint-disable-line import/no-extraneous-dependencies
const http_1 = require("./http");
exports.CREATE_FAILED_PHYSICAL_ID_MARKER = 'AWSCDK::StateMachineProvider::CREATE_FAILED';
async function cfnResponseSuccess(event) {
    console.log('Event: %j', event);
    await (0, http_1.respond)('SUCCESS', {
        ...event.Input,
        PhysicalResourceId: event.Output?.PhysicalResourceId ?? event.Input.PhysicalResourceId ?? event.Input.RequestId,
        Data: event.Output?.Data ?? {},
        NoEcho: event.Output?.NoEcho,
    });
}
async function cfnResponseFailed(event) {
    console.log('Event: %j', event);
    const parsedCause = JSON.parse(event.Cause);
    const executionResult = {
        ...parsedCause,
        Input: JSON.parse(parsedCause.Input),
    };
    console.log('Execution result: %j', executionResult);
    let physicalResourceId = executionResult.Output?.PhysicalResourceId ?? executionResult.Input.PhysicalResourceId;
    if (!physicalResourceId) {
        // special case: if CREATE fails, which usually implies, we usually don't
        // have a physical resource id. in this case, the subsequent DELETE
        // operation does not have any meaning, and will likely fail as well. to
        // address this, we use a marker so the provider framework can simply
        // ignore the subsequent DELETE.
        if (executionResult.Input.RequestType === 'Create') {
            console.log('CREATE failed, responding with a marker physical resource id so that the subsequent DELETE will be ignored');
            physicalResourceId = exports.CREATE_FAILED_PHYSICAL_ID_MARKER;
        }
        else {
            console.log(`ERROR: Malformed event. "PhysicalResourceId" is required: ${JSON.stringify(event)}`);
        }
    }
    await (0, http_1.respond)('FAILED', {
        ...executionResult.Input,
        Reason: `${event.Error}: ${event.Cause}`,
        PhysicalResourceId: physicalResourceId,
    });
}
const stepFunctionsClient = new client_sfn_1.SFNClient({});
async function startExecution(event) {
    try {
        console.log('Event: %j', event);
        if (!process.env.STATE_MACHINE_ARN) {
            throw new Error('Missing STATE_MACHINE_ARN.');
        }
        // ignore DELETE event when the physical resource ID is the marker that
        // indicates that this DELETE is a subsequent DELETE to a failed CREATE
        // operation.
        if (event.RequestType === 'Delete' && event.PhysicalResourceId === exports.CREATE_FAILED_PHYSICAL_ID_MARKER) {
            console.log('ignoring DELETE event caused by a failed CREATE event');
            await (0, http_1.respond)('SUCCESS', event);
            return;
        }
        await stepFunctionsClient.send(new client_sfn_1.StartExecutionCommand({
            stateMachineArn: process.env.STATE_MACHINE_ARN,
            input: JSON.stringify(event),
        }));
    }
    catch (err) {
        console.log(err);
        await (0, http_1.respond)('FAILED', {
            ...event,
            Reason: err instanceof Error ? err.message : undefined,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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