"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const iam = require("aws-cdk-lib/aws-iam");
const route53 = require("aws-cdk-lib/aws-route53");
const patterns = require("aws-cdk-lib/aws-route53-patterns");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const origin_request_function_1 = require("./origin-request-function");
/**
 * A CloudFront static website hosted on S3
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName: props.domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(this.bucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: props.edgeLambdas ?? [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: new origin_request_function_1.OriginRequestFunction(this, 'OriginRequest'),
                    },
                ],
                responseHeadersPolicy: props.responseHeadersPolicy ?? new cloudfront.ResponseHeadersPolicy(this, 'ResponseHeadersPolicy', {
                    securityHeadersBehavior: StaticWebsite.defaultSecurityHeadersBehavior,
                }),
                cachePolicy: props.cachePolicy,
            },
            defaultRootObject: 'index.html',
            domainNames: [props.domainName],
            certificate,
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2021,
            httpVersion: 'http2and3',
        });
        new route53.ARecord(this, 'ARecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        new route53.RecordSet(this, 'HttpsRecord', {
            recordType: 'HTTPS',
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        if (props.backendConfiguration) {
            // Save backend config to bucket, can be queried by the frontend
            new cr.AwsCustomResource(this, 'PutConfig', {
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['s3:PutObject'],
                        resources: [this.bucket.arnForObjects('config.json')],
                    }),
                ]),
                onUpdate: {
                    service: 'S3',
                    action: 'putObject',
                    parameters: {
                        Bucket: this.bucket.bucketName,
                        Key: 'config.json',
                        Body: aws_cdk_lib_1.Stack.of(this).toJsonString(props.backendConfiguration),
                        ContentType: 'application/json',
                        CacheControl: 'max-age=0, no-cache, no-store, must-revalidate',
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('config'),
                },
            });
        }
        if (shouldAddRedirect(props)) {
            const httpsRedirect = new patterns.HttpsRedirect(this, 'HttpsRedirect', {
                targetDomain: props.domainName,
                zone: props.hostedZone,
                recordNames: props.redirects,
            });
            // Force minimum protocol version
            const redirectDistribution = httpsRedirect.node.tryFindChild('RedirectDistribution');
            const cfnDistribution = redirectDistribution.node.tryFindChild('CFDistribution');
            cfnDistribution.addPropertyOverride('DistributionConfig.ViewerCertificate.MinimumProtocolVersion', 'TLSv1.2_2021');
            cfnDistribution.addPropertyOverride('DistributionConfig.HttpVersion', 'http2and3');
        }
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "cloudstructs.StaticWebsite", version: "0.9.38" };
/**
* Best practice security headers used as default
*/
StaticWebsite.defaultSecurityHeadersBehavior = {
    contentTypeOptions: {
        override: true,
    },
    frameOptions: {
        frameOption: cloudfront.HeadersFrameOption.DENY,
        override: true,
    },
    referrerPolicy: {
        referrerPolicy: cloudfront.HeadersReferrerPolicy.STRICT_ORIGIN_WHEN_CROSS_ORIGIN,
        override: true,
    },
    strictTransportSecurity: {
        accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(31536000),
        includeSubdomains: true,
        preload: true,
        override: true,
    },
    xssProtection: {
        protection: true,
        modeBlock: true,
        override: true,
    },
};
function shouldAddRedirect(props) {
    if (props.redirects && props.redirects.length === 0) {
        return false;
    }
    if (!props.redirects && !aws_cdk_lib_1.Token.isUnresolved(props.domainName)
        && !aws_cdk_lib_1.Token.isUnresolved(props.hostedZone.zoneName)
        && props.domainName === props.hostedZone.zoneName) {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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