"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ToolkitCleaner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const clean_images_function_1 = require("./clean-images-function");
const clean_objects_function_1 = require("./clean-objects-function");
const extract_template_hashes_function_1 = require("./extract-template-hashes-function");
const get_stack_names_function_1 = require("./get-stack-names-function");
/**
 * Clean unused S3 and ECR assets from your CDK Toolkit.
 */
class ToolkitCleaner extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        // Dummy assets to reference S3 bucket and ECR repository
        const fileAsset = new aws_s3_assets_1.Asset(this, 'FileAsset', {
            path: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker', 'dummy.txt'),
        });
        const dockerImageAsset = new aws_ecr_assets_1.DockerImageAsset(this, 'DockerImageAsset', {
            directory: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker'),
        });
        const getStackNamesFunction = new get_stack_names_function_1.GetStackNamesFunction(this, 'GetStackNamesFunction', {
            timeout: aws_cdk_lib_1.Duration.seconds(30),
        });
        getStackNamesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:DescribeStacks', 'cloudformation:ListStacks'],
            resources: ['*'],
        }));
        const getStackNames = new tasks.LambdaInvoke(this, 'GetStackNames', {
            lambdaFunction: getStackNamesFunction,
            payloadResponseOnly: true,
        });
        const stacksMap = new sfn.Map(this, 'StacksMap', {
            maxConcurrency: 1, // Avoid "Rate exceeded" error from CloudFormation
            resultSelector: {
                AssetHashes: sfn.JsonPath.stringAt('$'),
            },
        });
        const extractTemplateHashesFunction = new extract_template_hashes_function_1.ExtractTemplateHashesFunction(this, 'ExtractTemplateHashesFunction', {
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: {
                DOCKER_IMAGE_ASSET_HASH: dockerImageAsset.assetHash,
            },
        });
        extractTemplateHashesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:GetTemplate'],
            resources: ['*'],
        }));
        const extractTemplateHashes = new tasks.LambdaInvoke(this, 'ExtractTemplateHashes', {
            lambdaFunction: extractTemplateHashesFunction,
            payloadResponseOnly: true,
        }).addRetry({
            errors: ['Throttling'], // Avoid "Rate exceeded" error from CloudFormation
        });
        const flattenHashes = new tasks.EvaluateExpression(this, 'FlattenHashes', {
            expression: '[...new Set(($.AssetHashes).flat())]',
        });
        const cleanObjectsFunction = new clean_objects_function_1.CleanObjectsFunction(this, 'CleanObjectsFunction', {
            timeout: props.cleanAssetsTimeout ?? aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanObjectsFunction.addEnvironment('BUCKET_NAME', fileAsset.bucket.bucketName);
        fileAsset.bucket.grantRead(cleanObjectsFunction);
        fileAsset.bucket.grantDelete(cleanObjectsFunction);
        const cleanObjects = new tasks.LambdaInvoke(this, 'CleanObjects', {
            lambdaFunction: cleanObjectsFunction,
            payloadResponseOnly: true,
        });
        const cleanImagesFunction = new clean_images_function_1.CleanImagesFunction(this, 'CleanImagesFunction', {
            timeout: props.cleanAssetsTimeout ?? aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanImagesFunction.addEnvironment('REPOSITORY_NAME', dockerImageAsset.repository.repositoryName);
        dockerImageAsset.repository.grant(cleanImagesFunction, 'ecr:DescribeImages', 'ecr:BatchDeleteImage');
        const cleanImages = new tasks.LambdaInvoke(this, 'CleanImages', {
            lambdaFunction: cleanImagesFunction,
            payloadResponseOnly: true,
        });
        if (!props.dryRun) {
            cleanObjectsFunction.addEnvironment('RUN', 'true');
            cleanImagesFunction.addEnvironment('RUN', 'true');
        }
        if (props.retainAssetsNewerThan) {
            const retainMilliseconds = props.retainAssetsNewerThan.toMilliseconds().toString();
            cleanObjectsFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
            cleanImagesFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
        }
        const sumReclaimed = new tasks.EvaluateExpression(this, 'SumReclaimed', {
            expression: '({ Deleted: $[0].Deleted + $[1].Deleted, Reclaimed: $[0].Reclaimed + $[1].Reclaimed })',
        });
        const stateMachine = new sfn.StateMachine(this, 'Resource', {
            definitionBody: sfn.DefinitionBody.fromChainable(getStackNames
                .next(stacksMap.itemProcessor(extractTemplateHashes))
                .next(flattenHashes)
                .next(new sfn.Parallel(this, 'Clean')
                .branch(cleanObjects)
                .branch(cleanImages))
                .next(sumReclaimed)),
        });
        const rule = new aws_events_1.Rule(this, 'Rule', {
            enabled: props.scheduleEnabled ?? true,
            schedule: props.schedule ?? aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.days(1)),
        });
        rule.addTarget(new aws_events_targets_1.SfnStateMachine(stateMachine));
    }
}
exports.ToolkitCleaner = ToolkitCleaner;
_a = JSII_RTTI_SYMBOL_1;
ToolkitCleaner[_a] = { fqn: "cloudstructs.ToolkitCleaner", version: "0.9.38" };
//# sourceMappingURL=data:application/json;base64,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