import type { IncomingHttpHeaders } from 'node:http';
import type { AxiosResponse } from 'axios';
import type { WebAPICallResult } from './WebClient';
/**
 * All errors produced by this package adhere to this interface
 */
export interface CodedError extends NodeJS.ErrnoException {
    code: ErrorCode;
}
/**
 * A dictionary of codes for errors produced by this package
 */
export declare enum ErrorCode {
    RequestError = "slack_webapi_request_error",
    HTTPError = "slack_webapi_http_error",
    PlatformError = "slack_webapi_platform_error",
    RateLimitedError = "slack_webapi_rate_limited_error",
    FileUploadInvalidArgumentsError = "slack_webapi_file_upload_invalid_args_error",
    FileUploadReadFileDataError = "slack_webapi_file_upload_read_file_data_error"
}
export type WebAPICallError = WebAPIPlatformError | WebAPIRequestError | WebAPIHTTPError | WebAPIRateLimitedError;
export type WebAPIFilesUploadError = WebAPIFileUploadInvalidArgumentsError;
export interface WebAPIFileUploadInvalidArgumentsError extends CodedError {
    code: ErrorCode.FileUploadInvalidArgumentsError;
    data: WebAPICallResult & {
        error: string;
    };
}
export interface WebAPIPlatformError extends CodedError {
    code: ErrorCode.PlatformError;
    data: WebAPICallResult & {
        error: string;
    };
}
export interface WebAPIRequestError extends CodedError {
    code: ErrorCode.RequestError;
    original: Error;
}
export interface WebAPIHTTPError extends CodedError {
    code: ErrorCode.HTTPError;
    statusCode: number;
    statusMessage: string;
    headers: IncomingHttpHeaders;
    body?: any;
}
export interface WebAPIRateLimitedError extends CodedError {
    code: ErrorCode.RateLimitedError;
    retryAfter: number;
}
/**
 * Factory for producing a {@link CodedError} from a generic error
 */
export declare function errorWithCode(error: Error, code: ErrorCode): CodedError;
/**
 * A factory to create WebAPIRequestError objects
 * @param original - original error
 * @param attachOriginal - config indicating if 'original' property should be added on the error object
 */
export declare function requestErrorWithOriginal(original: Error, attachOriginal: boolean): WebAPIRequestError;
/**
 * A factory to create WebAPIHTTPError objects
 * @param response - original error
 */
export declare function httpErrorFromResponse(response: AxiosResponse): WebAPIHTTPError;
/**
 * A factory to create WebAPIPlatformError objects
 * @param result - Web API call result
 */
export declare function platformErrorFromResult(result: WebAPICallResult & {
    error: string;
}): WebAPIPlatformError;
/**
 * A factory to create WebAPIRateLimitedError objects
 * @param retrySec - Number of seconds that the request can be retried in
 */
export declare function rateLimitedErrorWithDelay(retrySec: number): WebAPIRateLimitedError;
//# sourceMappingURL=errors.d.ts.map