"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ErrorCode = void 0;
exports.errorWithCode = errorWithCode;
exports.requestErrorWithOriginal = requestErrorWithOriginal;
exports.httpErrorFromResponse = httpErrorFromResponse;
exports.platformErrorFromResult = platformErrorFromResult;
exports.rateLimitedErrorWithDelay = rateLimitedErrorWithDelay;
/**
 * A dictionary of codes for errors produced by this package
 */
var ErrorCode;
(function (ErrorCode) {
    // general error
    ErrorCode["RequestError"] = "slack_webapi_request_error";
    ErrorCode["HTTPError"] = "slack_webapi_http_error";
    ErrorCode["PlatformError"] = "slack_webapi_platform_error";
    ErrorCode["RateLimitedError"] = "slack_webapi_rate_limited_error";
    // file uploads errors
    ErrorCode["FileUploadInvalidArgumentsError"] = "slack_webapi_file_upload_invalid_args_error";
    ErrorCode["FileUploadReadFileDataError"] = "slack_webapi_file_upload_read_file_data_error";
})(ErrorCode || (exports.ErrorCode = ErrorCode = {}));
/**
 * Factory for producing a {@link CodedError} from a generic error
 */
function errorWithCode(error, code) {
    // NOTE: might be able to return something more specific than a CodedError with conditional typing
    const codedError = error;
    codedError.code = code;
    return codedError;
}
/**
 * A factory to create WebAPIRequestError objects
 * @param original - original error
 * @param attachOriginal - config indicating if 'original' property should be added on the error object
 */
function requestErrorWithOriginal(original, attachOriginal) {
    const error = errorWithCode(new Error(`A request error occurred: ${original.message}`), ErrorCode.RequestError);
    if (attachOriginal) {
        error.original = original;
    }
    return error;
}
/**
 * A factory to create WebAPIHTTPError objects
 * @param response - original error
 */
function httpErrorFromResponse(response) {
    const error = errorWithCode(new Error(`An HTTP protocol error occurred: statusCode = ${response.status}`), ErrorCode.HTTPError);
    error.statusCode = response.status;
    error.statusMessage = response.statusText;
    const nonNullHeaders = {};
    for (const k of Object.keys(response.headers)) {
        if (k && response.headers[k]) {
            nonNullHeaders[k] = response.headers[k];
        }
    }
    error.headers = nonNullHeaders;
    error.body = response.data;
    return error;
}
/**
 * A factory to create WebAPIPlatformError objects
 * @param result - Web API call result
 */
function platformErrorFromResult(result) {
    const error = errorWithCode(new Error(`An API error occurred: ${result.error}`), ErrorCode.PlatformError);
    error.data = result;
    return error;
}
/**
 * A factory to create WebAPIRateLimitedError objects
 * @param retrySec - Number of seconds that the request can be retried in
 */
function rateLimitedErrorWithDelay(retrySec) {
    const error = errorWithCode(new Error(`A rate-limit has been reached, you may retry this request in ${retrySec} seconds`), ErrorCode.RateLimitedError);
    error.retryAfter = retrySec;
    return error;
}
//# sourceMappingURL=errors.js.map