import base64
import os
from io import BytesIO

import httpx
from PIL import Image

from xproject.xheaders import get_default_headers
from xproject.xurl import is_valid


def image_base64_to_base64(
        image_base64: str,
        image_base64_prefix: str | None = None,
        use_prefix: bool = True
) -> str:
    if image_base64_prefix is None:
        image_base64_prefix = "data:image/png;base64,"

    if use_prefix:
        if "," not in image_base64:
            image_base64 = image_base64_prefix + image_base64
    else:
        if "," in image_base64:
            image_base64 = image_base64.split(",", maxsplit=1)[-1]  # data:image/png;base64,...
    return image_base64


def image_bytes_to_base64(
        image_bytes: bytes,
        image_base64_prefix: str | None = None,
        use_prefix: bool = True
) -> str:
    image_base64 = base64.b64encode(image_bytes).decode()
    image_base64 = image_base64_to_base64(image_base64, image_base64_prefix, use_prefix)
    return image_base64


def image_base64_to_bytes(image_base64: str, image_file_path: str | None = None) -> bytes:
    image_base64 = image_base64_to_base64(image_base64, use_prefix=False)
    image_bytes = base64.b64decode(image_base64)

    if image_file_path is not None:
        image_file_path = os.path.abspath(image_file_path)
        image_dir_path = os.path.dirname(image_file_path)
        if not os.path.exists(image_dir_path):
            os.makedirs(image_dir_path)
        with open(image_file_path, "wb") as file:
            file.write(image_bytes)

    return image_bytes


def image_base64_to_file_ext(image_base64: str, default: str | None = None) -> str | None:
    file_ext = None

    if "," in image_base64:
        prefix, data = image_base64.split(",", maxsplit=1)  # data:image/png;base64,...
        file_ext = prefix.split(";", maxsplit=1)[0].split("/", maxsplit=1)[-1]

    if file_ext is None:
        image_bytes = image_base64_to_bytes(image_base64)
        image = Image.open(BytesIO(image_bytes))
        image_format = image.format
        file_ext = image_format.lower()

    if file_ext is None and default is not None:
        file_ext = default

    return file_ext


def image_url_to_base64(
        image_url: str,
        image_base64_prefix: str | None = None,
        use_prefix: bool = True
) -> str | None:
    """
    >>> image_url_to_base64("https://www.baidu.com/favicon.ico")
    'data:image/x-icon;base64,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'

    :param image_url:
    :param image_base64_prefix:
    :param use_prefix:
    :return:
    """
    if not is_valid(image_url):
        return None

    headers = get_default_headers()
    response = httpx.get(image_url, headers=headers)
    content = response.content

    if image_base64_prefix is None and use_prefix is True:
        content_type = response.headers.get("content-type", "image/png")
        image_base64_prefix = f"data:{content_type};base64,"

    image_base64 = image_bytes_to_base64(content, image_base64_prefix, use_prefix)
    return image_base64


def image_file_path_to_base64(
        image_file_path: str,
        image_base64_prefix: str | None = None,
        use_prefix: bool = True
) -> str:
    with open(image_file_path, "rb") as f:
        image_bytes = f.read()

    if image_base64_prefix is None and use_prefix is True:
        image = Image.open(image_file_path)
        image_format = image.format
        image_base64_prefix = f"data:image/{image_format.lower()};base64,"

    image_base64 = image_bytes_to_base64(image_bytes, image_base64_prefix, use_prefix)
    return image_base64
