"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const apputils_1 = require("@jupyterlab/apputils");
const coreutils_1 = require("@jupyterlab/coreutils");
const algorithm_1 = require("@phosphor/algorithm");
const signaling_1 = require("@phosphor/signaling");
const docmanager_1 = require("@jupyterlab/docmanager");
/**
 * The maximum upload size (in bytes) for notebook version < 5.1.0
 */
exports.LARGE_FILE_SIZE = 15 * 1024 * 1024;
exports.CHUNK_SIZE = 1024 * 1024;
class Uploader extends apputils_1.ToolbarButton {
    constructor(options) {
        super({
            iconClassName: "jp-FileUploadIcon jp-Icon jp-Icon-16",
            onClick: () => {
                this.context = "";
                this._input.click();
            },
            tooltip: "Upload Files",
        });
        // tslint:disable: variable-name
        this._input = Private.createUploadInput();
        this._uploads = [];
        this._uploadChanged = new signaling_1.Signal(this);
        this._onInputChanged = () => {
            const files = Array.prototype.slice.call(this._input.files);
            const pending = files.map((file) => this.upload(file, this.context));
            this.context = "";
            Promise.all(pending).catch((error) => {
                apputils_1.showErrorMessage("Upload Error", error);
            });
        };
        this._onInputClicked = () => {
            // In order to allow repeated uploads of the same file (with delete in between),
            // we need to clear the input value to trigger a change event.
            this._input.value = "";
        };
        this.basepath = options.widget.basepath;
        this.manager = options.manager;
        this.widget = options.widget;
        this._input.onclick = this._onInputClicked;
        this.context = "";
        this._input.onchange = this._onInputChanged;
        this.addClass("filetree-upload");
        this.addClass(options.widget.filetree_id);
    }
    contextClick(path) {
        this.context = path;
        this._input.click();
    }
    /**
     * Upload a `File` object.
     *
     * @param file - The `File` object to upload.
     *
     * @returns A promise containing the new file contents model.
     *
     * #### Notes
     * On Notebook version < 5.1.0, this will fail to upload files that are too
     * big to be sent in one request to the server. On newer versions, it will
     * ask for confirmation then upload the file in 1 MB chunks.
     */
    upload(file, path) {
        return __awaiter(this, void 0, void 0, function* () {
            const supportsChunked = coreutils_1.PageConfig.getNotebookVersion() >= [5, 1, 0];
            const largeFile = file.size > exports.LARGE_FILE_SIZE;
            if (largeFile && !supportsChunked) {
                const msg = `Cannot upload file (>${exports.LARGE_FILE_SIZE / (1024 * 1024)} MB). ${file.name}`;
                // tslint:disable-next-line: no-console
                console.warn(msg);
                throw msg;
            }
            const err = "File not uploaded";
            if (largeFile && !(yield this._shouldUploadLarge(file))) {
                throw new Error("Cancelled large file upload");
            }
            yield this._uploadCheckDisposed();
            yield this.widget.refresh();
            yield this._uploadCheckDisposed();
            const contents = yield this.widget.cm.get(this.context);
            contents.content.forEach((entry) => __awaiter(this, void 0, void 0, function* () {
                if ((entry.name === file.name) && !(yield docmanager_1.shouldOverwrite(file.name))) {
                    throw err;
                }
            }));
            yield this._uploadCheckDisposed();
            const chunkedUpload = supportsChunked && file.size > exports.CHUNK_SIZE;
            return yield this._upload(file, path, chunkedUpload);
        });
    }
    _uploadCheckDisposed() {
        if (this.isDisposed) {
            return Promise.reject("Filemanager disposed. File upload canceled");
        }
        return Promise.resolve();
    }
    _shouldUploadLarge(file) {
        return __awaiter(this, void 0, void 0, function* () {
            const { button } = yield apputils_1.showDialog({
                body: `The file size is ${Math.round(file.size / (1024 * 1024))} MB. Do you still want to upload it?`,
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.warnButton({ label: "UPLOAD" })],
                title: "Large file size warning",
            });
            return button.accept;
        });
    }
    /**
     * Perform the actual upload.
     */
    _upload(file, path_arg, chunked) {
        return __awaiter(this, void 0, void 0, function* () {
            // Gather the file model parameters.
            let path = path_arg || "";
            path = path ? path + "/" + file.name : file.name;
            const name = file.name;
            const type = "file";
            const format = "base64";
            const uploadInner = (blob, chunk) => __awaiter(this, void 0, void 0, function* () {
                yield this._uploadCheckDisposed();
                const reader = new FileReader();
                reader.readAsDataURL(blob);
                yield new Promise((resolve, reject) => {
                    reader.onload = resolve;
                    reader.onerror = (event) => reject(`Failed to upload "${file.name}":` + event);
                });
                yield this._uploadCheckDisposed();
                // remove header https://stackoverflow.com/a/24289420/907060
                const content = reader.result.split(",")[1];
                const model = {
                    chunk,
                    content,
                    format,
                    name,
                    type,
                };
                return yield this.manager.services.contents.save(this.basepath + path, model);
            });
            if (!chunked) {
                try {
                    return yield uploadInner(file);
                }
                catch (err) {
                    algorithm_1.ArrayExt.removeFirstWhere(this._uploads, (uploadIndex) => {
                        return file.name === uploadIndex.path;
                    });
                    throw err;
                }
            }
            let finalModel;
            let upload = { path, progress: 0 };
            this._uploadChanged.emit({
                name: "start",
                newValue: upload,
                oldValue: null,
            });
            for (let start = 0; !finalModel; start += exports.CHUNK_SIZE) {
                const end = start + exports.CHUNK_SIZE;
                const lastChunk = end >= file.size;
                const chunk = lastChunk ? -1 : end / exports.CHUNK_SIZE;
                const newUpload = { path, progress: start / file.size };
                this._uploads.splice(this._uploads.indexOf(upload));
                this._uploads.push(newUpload);
                this._uploadChanged.emit({
                    name: "update",
                    newValue: newUpload,
                    oldValue: upload,
                });
                upload = newUpload;
                let currentModel;
                try {
                    currentModel = yield uploadInner(file.slice(start, end), chunk);
                }
                catch (err) {
                    algorithm_1.ArrayExt.removeFirstWhere(this._uploads, (uploadIndex) => {
                        return file.name === uploadIndex.path;
                    });
                    this._uploadChanged.emit({
                        name: "failure",
                        newValue: upload,
                        oldValue: null,
                    });
                    throw err;
                }
                if (lastChunk) {
                    finalModel = currentModel;
                }
            }
            this._uploads.splice(this._uploads.indexOf(upload));
            this._uploadChanged.emit({
                name: "finish",
                newValue: null,
                oldValue: upload,
            });
            return finalModel;
        });
    }
}
exports.Uploader = Uploader;
// tslint:disable-next-line: no-namespace
var Private;
(function (Private) {
    function createUploadInput() {
        const input = document.createElement("input");
        input.type = "file";
        input.multiple = true;
        return input;
    }
    Private.createUploadInput = createUploadInput;
})(Private || (Private = {}));
