"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const YAML = require("yaml");
const aws_credentials_1 = require("./private/aws-credentials");
const github = require("./workflows-model");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.gitHubActionRoleArn = props.gitHubActionRoleArn;
        this.useGitHubActionRole = this.gitHubActionRoleArn ? true : false;
        this.jobSettings = props.jobSettings;
        this.awsCredentials = props.awsCredentials ?? {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        const yaml = YAML.stringify(workflow, {
            indent: 2,
        });
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const diffProtection = this.node.tryGetContext('cdk-pipelines-github:diffProtection') ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || yaml !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        fs_1.writeFileSync(this.workflowPath, yaml);
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Worflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -x'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '::set-output name=${ASSET_HASH_NAME}::${assetId}'`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.jobSettings,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region: this.publishAssetsAuthRegion }),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.jobSettings,
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                ...(this.stackProperties[stack.stackArtifactId]?.environment ? {
                    environment: this.stackProperties[stack.stackArtifactId].environment,
                } : {}),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region, assumeRoleArn }),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.jobSettings,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    stepsToConfigureAws(openId, { region, assumeRoleArn }) {
        function getDeployRole(arn) {
            return arn.replace('cfn-exec', 'deploy');
        }
        let steps = [];
        if (openId) {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via OIDC Role', {
                region,
                gitHubActionRoleArn: this.gitHubActionRoleArn,
            }));
            if (assumeRoleArn) {
                // Result of initial credentials with GitHub Action role are these environment variables
                steps.push(aws_credentials_1.awsCredentialStep('Assume CDK Deploy Role', {
                    region,
                    accessKeyId: '${{ env.AWS_ACCESS_KEY_ID }}',
                    secretAccessKey: '${{ env.AWS_SECRET_ACCESS_KEY }}',
                    sessionToken: '${{ env.AWS_SESSION_TOKEN }}',
                    roleToAssume: getDeployRole(assumeRoleArn),
                }));
            }
        }
        else {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via GitHub Secrets', {
                region,
                accessKeyId: `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
                secretAccessKey: `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
                ...(this.awsCredentials.sessionToken ? {
                    sessionToken: `\${{ secrets.${this.awsCredentials.sessionToken} }}`,
                } : undefined),
                roleToAssume: assumeRoleArn,
            }));
        }
        return steps;
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.2.173" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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