"""Environment configuration utilities"""
from pathlib import Path
from typing import Dict, Optional
import secrets
import string


def generate_env_file(
    project_path: Path,
    env_name: str,
    framework: Optional[str] = None,
    database: Optional[str] = None
) -> Path:
    """
    Generate environment configuration file.
    
    Args:
        project_path: Path to project root
        env_name: Environment name (development, staging, production, test)
        framework: Detected framework name
        database: Database type
        
    Returns:
        Path to created .env file
    """
    # Generate secure secret key
    secret_key = ''.join(secrets.choice(string.ascii_letters + string.digits) for _ in range(50))
    
    # Base configuration
    env_vars = {
        'DEBUG': 'True' if env_name == 'development' else 'False',
        'ENVIRONMENT': env_name,
        'SECRET_KEY': secret_key,
    }
    
    # Framework-specific variables
    if framework == 'django':
        env_vars.update({
            'ALLOWED_HOSTS': 'localhost,127.0.0.1' if env_name == 'development' else 'api.example.com',
            'DJANGO_SETTINGS_MODULE': 'config.settings',
        })
    elif framework == 'fastapi':
        env_vars.update({
            'APP_NAME': 'FastAPI Application',
            'API_V1_PREFIX': '/api/v1',
            'CORS_ORIGINS': '["http://localhost:3000", "http://localhost:8080"]' if env_name == 'development' else '[]',
        })
    elif framework == 'flask':
        env_vars.update({
            'FLASK_ENV': env_name,
            'FLASK_APP': 'app',
        })
    
    # Database configuration
    if database == 'postgresql':
        if env_name == 'development':
            env_vars['DATABASE_URL'] = 'postgresql://user:password@localhost:5432/dev_db'
        elif env_name == 'staging':
            env_vars['DATABASE_URL'] = 'postgresql://staging_user:password@staging-db:5432/staging_db'
        elif env_name == 'production':
            env_vars['DATABASE_URL'] = 'postgresql://prod_user:password@prod-db:5432/prod_db'
        else:  # test
            env_vars['DATABASE_URL'] = 'postgresql://test_user:password@localhost:5432/test_db'
    
    elif database == 'mysql':
        if env_name == 'development':
            env_vars['DATABASE_URL'] = 'mysql://user:password@localhost:3306/dev_db'
        elif env_name == 'staging':
            env_vars['DATABASE_URL'] = 'mysql://staging_user:password@staging-db:3306/staging_db'
        elif env_name == 'production':
            env_vars['DATABASE_URL'] = 'mysql://prod_user:password@prod-db:3306/prod_db'
        else:  # test
            env_vars['DATABASE_URL'] = 'mysql://test_user:password@localhost:3306/test_db'
    
    elif database == 'mongodb':
        if env_name == 'development':
            env_vars['DATABASE_URL'] = 'mongodb://localhost:27017/dev_db'
        elif env_name == 'staging':
            env_vars['DATABASE_URL'] = 'mongodb://staging-db:27017/staging_db'
        elif env_name == 'production':
            env_vars['DATABASE_URL'] = 'mongodb://prod-db:27017/prod_db'
        else:  # test
            env_vars['DATABASE_URL'] = 'mongodb://localhost:27017/test_db'
    
    elif database == 'sqlite':
        if env_name == 'test':
            env_vars['DATABASE_URL'] = 'sqlite:///./test.db'
        else:
            env_vars['DATABASE_URL'] = f'sqlite:///./{env_name}.db'
    
    elif database == 'redis':
        if env_name == 'development':
            env_vars['REDIS_URL'] = 'redis://localhost:6379/0'
        elif env_name == 'staging':
            env_vars['REDIS_URL'] = 'redis://staging-redis:6379/0'
        elif env_name == 'production':
            env_vars['REDIS_URL'] = 'redis://prod-redis:6379/0'
        else:  # test
            env_vars['REDIS_URL'] = 'redis://localhost:6379/1'
    
    # Logging configuration
    if env_name == 'production':
        env_vars['LOG_LEVEL'] = 'INFO'
    else:
        env_vars['LOG_LEVEL'] = 'DEBUG'
    
    # Build .env file content
    env_content = f"""# {env_name.upper()} Environment Configuration
# Generated by Projex
# DO NOT commit this file to version control (add to .gitignore)

"""
    
    for key, value in env_vars.items():
        env_content += f"{key}={value}\n"
    
    # Add comment section for additional variables
    env_content += """
# Add your custom environment variables below
# EXAMPLE_VAR=value
"""
    
    # Determine file name
    if env_name == 'development':
        env_file = project_path / '.env.development'
    else:
        env_file = project_path / f'.env.{env_name}'
    
    env_file.write_text(env_content)
    
    return env_file


def list_env_files(project_path: Path) -> list:
    """
    List all environment files in the project.
    
    Args:
        project_path: Path to project root
        
    Returns:
        List of environment file paths
    """
    env_files = []
    patterns = ['.env', '.env.*']
    
    for pattern in patterns:
        for env_file in project_path.glob(pattern):
            if env_file.is_file() and not env_file.name.endswith('.example'):
                env_files.append(env_file)
    
    return sorted(env_files)


def get_env_variables(env_file: Path) -> Dict[str, str]:
    """
    Parse environment variables from .env file.
    
    Args:
        env_file: Path to .env file
        
    Returns:
        Dictionary of environment variables
    """
    env_vars = {}
    
    if not env_file.exists():
        return env_vars
    
    try:
        content = env_file.read_text(encoding='utf-8')
        
        for line in content.split('\n'):
            line = line.strip()
            
            # Skip comments and empty lines
            if not line or line.startswith('#'):
                continue
            
            # Parse KEY=VALUE
            if '=' in line:
                key, value = line.split('=', 1)
                env_vars[key.strip()] = value.strip()
    except Exception:
        pass
    
    return env_vars

