"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources
 */
class NagSuppressions {
    /**
     * Apply cdk-nag suppressions to a Stack and optionally nested stacks
     * @param stack The Stack to apply the suppression to
     * @param suppressions A list of suppressions to apply to the stack
     * @param applyToNestedStacks Apply the suppressions to children stacks (default:false)
     */
    static addStackSuppressions(stack, suppressions, applyToNestedStacks = false) {
        const stacks = applyToNestedStacks
            ? stack.node.findAll().filter((x) => x instanceof aws_cdk_lib_1.Stack)
            : [stack];
        stacks.forEach((s) => {
            nag_suppression_helper_1.NagSuppressionHelper.assertSuppressionsAreValid(s.node.id, suppressions);
            let metadata = s.templateOptions.metadata?.cdk_nag ?? {};
            metadata = nag_suppression_helper_1.NagSuppressionHelper.addRulesToMetadata(metadata, suppressions);
            if (!s.templateOptions.metadata) {
                s.templateOptions.metadata = {};
            }
            s.templateOptions.metadata.cdk_nag = metadata;
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children
     * @param construct The IConstruct(s) to apply the suppression to
     * @param suppressions A list of suppressions to apply to the resource
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false)
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        const constructArray = Array.isArray(construct)
            ? construct
            : [construct];
        if (constructArray.length === 0) {
            throw new Error(`Construct cannot be an empty array.`);
        }
        constructArray.forEach((c) => {
            nag_suppression_helper_1.NagSuppressionHelper.assertSuppressionsAreValid(c.node.id, suppressions);
            const constructs = applyToChildren ? c.node.findAll() : [c];
            for (const child of constructs) {
                const possibleL1 = child.node.defaultChild
                    ? child.node.defaultChild
                    : child;
                if (possibleL1 instanceof aws_cdk_lib_1.CfnResource) {
                    const resource = possibleL1;
                    let metadata = resource.getMetadata('cdk_nag');
                    metadata = nag_suppression_helper_1.NagSuppressionHelper.addRulesToMetadata(metadata, suppressions);
                    resource.addMetadata('cdk_nag', metadata);
                }
            }
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children via its path
     * @param stack The Stack the construct belongs to
     * @param path The path(s) to the construct in the provided stack
     * @param suppressions A list of suppressions to apply to the resource
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false)
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        const pathArray = Array.isArray(path) ? path : [path];
        if (pathArray.length === 0) {
            throw new Error(`Path cannot be an empty array.`);
        }
        pathArray.forEach((p) => {
            let added = false;
            for (const child of stack.node.findAll()) {
                const fixedPath = p.replace(/^\//, '');
                if (child.node.path === fixedPath ||
                    child.node.path + '/Resource' === fixedPath) {
                    NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
                    added = true;
                }
            }
            if (!added) {
                throw new Error(`Suppression path "${p}" did not match any resource. This can occur when a resource does not exist or if a suppression is applied before a resource is created.`);
            }
        });
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "cdk-nag.NagSuppressions", version: "2.28.183" };
//# sourceMappingURL=data:application/json;base64,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