"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NIST80053R4Checks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_pack_1 = require("../nag-pack");
const nag_rules_1 = require("../nag-rules");
const apigw_1 = require("../rules/apigw");
const autoscaling_1 = require("../rules/autoscaling");
const cloudtrail_1 = require("../rules/cloudtrail");
const cloudwatch_1 = require("../rules/cloudwatch");
const codebuild_1 = require("../rules/codebuild");
const dms_1 = require("../rules/dms");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const efs_1 = require("../rules/efs");
const elasticache_1 = require("../rules/elasticache");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const iam_1 = require("../rules/iam");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const opensearch_1 = require("../rules/opensearch");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const sns_1 = require("../rules/sns");
const vpc_1 = require("../rules/vpc");
const waf_1 = require("../rules/waf");
/**
 * Check for NIST 800-53 rev 4 compliance.
 * Based on the NIST 800-53 rev 4 AWS operational best practices: https://docs.aws.amazon.com/config/latest/developerguide/operational-best-practices-for-nist-800-53_rev_4.html
 */
class NIST80053R4Checks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'NIST.800.53.R4';
    }
    visit(node) {
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            this.checkAPIGW(node);
            this.checkAutoScaling(node);
            this.checkCloudTrail(node);
            this.checkCloudWatch(node);
            this.checkCodeBuild(node);
            this.checkDMS(node);
            this.checkDynamoDB(node);
            this.checkEC2(node);
            this.checkEFS(node);
            this.checkElastiCache(node);
            this.checkELB(node);
            this.checkEMR(node);
            this.checkIAM(node);
            this.checkKMS(node);
            this.checkLambda(node);
            this.checkOpenSearch(node);
            this.checkRDS(node);
            this.checkRedshift(node);
            this.checkS3(node);
            this.checkSageMaker(node);
            this.checkSNS(node);
            this.checkVPC(node);
            this.checkWAF(node);
        }
    }
    /**
     * Check API Gateway Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAPIGW(node) {
        this.applyRule({
            info: 'The API Gateway stage does not have caching enabled and encrypted for all methods - (Control IDs: SC-13, SC-28).',
            explanation: "To help protect data at rest, ensure encryption is enabled for your API Gateway stage's cache. Because sensitive data can be captured for the API method, enable encryption at rest to help protect that data.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWCacheEnabledAndEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The API Gateway stage does not have execution logging enabled for all methods - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'API Gateway logging displays detailed views of users who accessed the API and the way they accessed the API. This insight enables visibility of user activities.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Auto Scaling Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAutoScaling(node) {
        this.applyRule({
            info: 'The Auto Scaling group utilizes a load balancer and does not have an ELB health check configured - (Control IDs: SC-5).',
            explanation: 'The Elastic Load Balancer (ELB) health checks for Amazon Elastic Compute Cloud (Amazon EC2) Auto Scaling groups support maintenance of adequate capacity and availability.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupELBHealthCheckRequired,
            node: node,
        });
    }
    /**
     * Check CloudTrail Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudTrail(node) {
        this.applyRule({
            info: 'The trail does not have CloudWatch logs enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-6(1)(3), AU-7(1), AU-12(a)(c), CA-7(a)(b), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Use Amazon CloudWatch to centrally collect and manage log event activity. Inclusion of AWS CloudTrail data provides details of API call activity within your AWS account.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailCloudWatchLogsEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have a KMS key ID or have encryption enabled - (Control ID: AU-9).',
            explanation: 'Because sensitive data may exist and to help protect data at rest, ensure encryption is enabled for your AWS CloudTrail trails.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailEncryptionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The trail does not have log file validation enabled - (Control ID: AC-6).',
            explanation: 'Utilize AWS CloudTrail log file validation to check the integrity of CloudTrail logs. Log file validation helps determine if a log file was modified or deleted or unchanged after CloudTrail delivered it. This feature is built using industry standard algorithms: SHA-256 for hashing and SHA-256 with RSA for digital signing. This makes it computationally infeasible to modify, delete or forge CloudTrail log files without detection.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudtrail_1.CloudTrailLogFileValidationEnabled,
            node: node,
        });
    }
    /**
     * Check CloudWatch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCloudWatch(node) {
        this.applyRule({
            info: 'The CloudWatch alarm does not have at least one alarm action, one INSUFFICIENT_DATA action, or one OK action enabled - (Control IDs: AC-2(4), AU-6(1)(3), AU-7(1), CA-7(a)(b), IR-4(1), SI-4(2), SI-4(4), SI-4(5), SI-4(a)(b)(c)).',
            explanation: 'Amazon CloudWatch alarms alert when a metric breaches the threshold for a specified number of evaluation periods. The alarm performs one or more actions based on the value of the metric or expression relative to a threshold over a number of time periods.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchAlarmAction,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group is not encrypted with an AWS KMS key - (Control IDs: AU-9, SC-13, SC-28).',
            explanation: 'To help protect sensitive data at rest, ensure encryption is enabled for your Amazon CloudWatch Log Groups.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupEncrypted,
            node: node,
        });
        this.applyRule({
            info: 'The CloudWatch Log Group does not have an explicit retention period configured - (Control IDs: AU-11, SI-12).',
            explanation: 'Ensure a minimum duration of event log data is retained for your log groups to help with troubleshooting and forensics investigations. The lack of available past event log data makes it difficult to reconstruct and identify potentially malicious events.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudwatch_1.CloudWatchLogGroupRetentionPeriod,
            node: node,
        });
    }
    /**
     * Check CodeBuild Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCodeBuild(node) {
        this.applyRule({
            info: 'The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables - (Control IDs: AC-6, IA-5(7), SA-3(a)).',
            explanation: 'Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectEnvVarAwsCred,
            node: node,
        });
        this.applyRule({
            info: 'The CodeBuild project which utilizes either a GitHub or BitBucket source repository does not utilize OAuth - (Control ID: SA-3(a)).',
            explanation: 'OAuth is the most secure method of authenticating your CodeBuild application. Use OAuth instead of personal access tokens or a user name and password to grant authorization for accessing GitHub or Bitbucket repositories.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectSourceRepoUrl,
            node: node,
        });
    }
    /**
     * Check DMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDMS(node) {
        this.applyRule({
            info: 'The DMS replication instance is public - (Control IDs: AC-3).',
            explanation: 'DMS replication instances can contain sensitive information and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dms_1.DMSReplicationNotPublic,
            node: node,
        });
    }
    /**
     * Check DynamoDB Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDynamoDB(node) {
        this.applyRule({
            info: "The provisioned capacity DynamoDB table does not have Auto Scaling enabled on it's indexes - (Control IDs: CP-10, SC-5).",
            explanation: 'Amazon DynamoDB auto scaling uses the AWS Application Auto Scaling service to adjust provisioned throughput capacity that automatically responds to actual traffic patterns. This enables a table or a global secondary index to increase its provisioned read/write capacity to handle sudden increases in traffic, without throttling.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBAutoScalingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon DynamoDB tables are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The recovery maintains continuous backups of your table for the last 35 days.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
    }
    /**
     * Check EC2 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEC2(node) {
        this.applyRule({
            info: 'The EBS volume is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic Block Store (Amazon EBS) volumes are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have IMDSV2 (Instance Metadata Service Version 2) enabled - (Control ID: AC-6).',
            explanation: 'Instance Metadata Service Version 2 (IMDSv2) helps protect access and control of Amazon Elastic Compute Cloud (Amazon EC2) instance metadata. The IMDSv2 method uses session-based controls. With IMDSv2, controls can be implemented to restrict changes to instance metadata.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2IMDSv2Enabled,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance does not have detailed monitoring enabled - (Control IDs: CA-7(a)(b), SI-4(2), SI-4(a)(b)(c)).',
            explanation: 'Detailed monitoring provides additional monitoring information (such as 1-minute period graphs) on the AWS console.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is not within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstancesInVPC,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance is associated with a public IP address - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)). ',
            explanation: 'Amazon EC2 instances can contain sensitive information and access control is required for such resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceNoPublicIp,
            node: node,
        });
        this.applyRule({
            info: 'The EC2 instance allows unrestricted inbound IPv4 TCP traffic on common ports (20, 21, 3389, 3306, 4333) - (Control IDs: AC-4, CM-2, SC-7, SC-7(3)).',
            explanation: 'Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedCommonPorts,
            node: node,
        });
        this.applyRule({
            info: 'The Security Group allows unrestricted SSH access - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedSSH,
            node: node,
        });
    }
    /**
     * Check EFS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEFS(node) {
        this.applyRule({
            info: 'The EFS is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon Elastic File System (Amazon EFS) file systems are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The EFS does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check ElastiCache Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkElastiCache(node) {
        this.applyRule({
            info: 'The ElastiCache Redis cluster does not retain automatic backups for at least 15 days - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'Automatic backups can help guard against data loss. If a failure occurs, you can create a new cluster, which restores your data from the most recent backup.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterAutomaticBackup,
            node: node,
        });
    }
    /**
     * Check Elastic Load Balancer Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkELB(node) {
        this.applyRule({
            info: 'The ALB does not have invalid HTTP header dropping enabled - (Control ID: AC-17(2)).',
            explanation: 'Ensure that your Application Load Balancers (ALB) are configured to drop http headers. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpDropInvalidHeaderEnabled,
            node: node,
        });
        this.applyRule({
            info: "The ALB's HTTP listeners are not configured to redirect to HTTPS - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13, SC-23).",
            explanation: 'To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBHttpToHttpsRedirection,
            node: node,
        });
        this.applyRule({
            info: 'The ALB is not associated with AWS WAFv2 web ACL - (Control IDs: SC-7, SI-4(a)(b)(c)).',
            explanation: 'A WAF helps to protect your web applications or APIs against common web exploits. These web exploits may affect availability, compromise security, or consume excessive resources within your environment.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ALBWAFEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not utilize an SSL certificate provided by ACM (Amazon Certificate Manager) - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'Because sensitive data can exist and to help protect data at transit, ensure encryption is enabled for your Elastic Load Balancing. Use AWS Certificate Manager to manage, provision and deploy public and private SSL/TLS certificates with AWS services and internal resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBACMCertificateRequired,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not balance traffic between at least 2 Availability Zones - (Control IDs: SC-5, CP-10).',
            explanation: 'The cross-zone load balancing reduces the need to maintain equivalent numbers of instances in each enabled availability zone.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ALB, NLB, or GLB does not have deletion protection enabled - (Control IDs: CM-2, CP-10).',
            explanation: 'Use this feature to prevent your load balancer from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The ELB does not have logging enabled - (Control ID: AU-2(a)(d)).',
            explanation: "Elastic Load Balancing activity is a central point of communication within an environment. Ensure ELB logging is enabled. The collected data provides detailed information about requests sent to The ELB. Each log contains information such as the time the request was received, the client's IP address, latencies, request paths, and server responses.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The CLB does not restrict its listeners to only the SSL and HTTPS protocols - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-23).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check EMR Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkEMR(node) {
        this.applyRule({
            info: 'The EMR cluster does not have Kerberos enabled - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'The access permissions and authorizations can be managed and incorporated with the principles of least privilege and separation of duties, by enabling Kerberos for Amazon EMR clusters.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRKerberosEnabled,
            node: node,
        });
    }
    /**
     * Check IAM Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkIAM(node) {
        this.applyRule({
            info: 'The IAM Group does not have at least one IAM User - (Control ID: AC-2(j)).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMGroupHasUsers,
            node: node,
        });
        this.applyRule({
            info: 'The IAM Group, User, or Role contains an inline policy - (Control ID: AC-6).',
            explanation: 'AWS recommends to use managed policies instead of inline policies. The managed policies allow reusability, versioning and rolling back, and delegating permissions management.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoInlinePolicy,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy grants admin access, meaning the policy allows a principal to perform all actions on all resources - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, restricting policies from containing "Effect": "Allow" with "Action": "*" over "Resource": "*". Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMPolicyNoStatementsWithAdminAccess,
            node: node,
        });
        this.applyRule({
            info: 'The IAM user does not belong to any group(s) - (Control IDs: AC-2(1), AC-2(j), AC-3, AC-6).',
            explanation: 'AWS Identity and Access Management (IAM) can help you restrict access permissions and authorizations, by ensuring IAM users are members of at least one group. Allowing users more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserGroupMembership,
            node: node,
        });
        this.applyRule({
            info: 'The IAM policy is attached at the user level - (Control IDs: AC-2(j), AC-3, AC-5c, AC-6).',
            explanation: 'Assigning privileges at the group or the role level helps to reduce opportunity for an identity to receive or retain excessive privileges.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMUserNoPolicies,
            node: node,
        });
    }
    /**
     * Check KMS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkKMS(node) {
        this.applyRule({
            info: 'The KMS Symmetric key does not have automatic key rotation enabled - (Control ID: SC-12).',
            explanation: 'Enable key rotation to ensure that keys are rotated once they have reached the end of their crypto period.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Lambda Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            info: 'The Lambda function is not VPC enabled - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaInsideVPC,
            node: node,
        });
        this.applyRule({
            info: 'The Lambda function permission grants public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaFunctionPublicAccessProhibited,
            node: node,
        });
    }
    /**
     * Check OpenSearch Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkOpenSearch(node) {
        this.applyRule({
            info: 'The OpenSearch Service domain does not have encryption at rest enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain is not running within a VPC - (Control IDs: AC-4, SC-7, SC-7(3)).',
            explanation: 'VPCs help secure your AWS resources and provide an extra layer of protection.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled - (Control IDs: SC-7, SC-8, SC-8(1)).',
            explanation: 'Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
    }
    /**
     * Check Amazon RDS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRDS(node) {
        this.applyRule({
            info: 'The RDS DB instance does not have enhanced monitoring enabled - (Control ID: CA-7(a)(b)).',
            explanation: 'Enable enhanced monitoring to help monitor Amazon RDS availability. This provides detailed visibility into the health of your Amazon RDS database instances.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSEnhancedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance is not in an AWS Backup plan - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'To help with data back-up processes, ensure your Amazon Relational Database Service (Amazon RDS) instances are a part of an AWS Backup plan. AWS Backup is a fully managed backup service with a policy-based backup solution. This solution simplifies your backup management and enables you to meet your business and regulatory backup compliance requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInBackupPlan,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance does not have backups enabled - (Control IDs: CP-9(b), CP-10, SI-12).',
            explanation: 'The backup feature of Amazon RDS creates backups of your databases and transaction logs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance or Aurora DB cluster does not have deletion protection enabled - (Control ID: SC-5).',
            explanation: 'Ensure Amazon Relational Database Service (Amazon RDS) instances and clusters have deletion protection enabled. Use deletion protection to prevent your Amazon RDS DB instances and clusters from being accidentally or maliciously deleted, which can lead to loss of availability for your applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance allows public access - (Control IDs: AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon RDS database instances can contain sensitive information, and principles and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstancePublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The non-Aurora RDS DB instance or Aurora cluster does not have all CloudWatch log types exported - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'To help with logging and monitoring within your environment, ensure Amazon Relational Database Service (Amazon RDS) logging is enabled. With Amazon RDS logging, you can capture events such as connections, disconnections, queries, or tables queried.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled - (Control IDs: CP-10, SC-5, SC-36).',
            explanation: 'Multi-AZ support in Amazon Relational Database Service (Amazon RDS) provides enhanced availability and durability for database instances. When you provision a Multi-AZ database instance, Amazon RDS automatically creates a primary database instance, and synchronously replicates the data to a standby instance in a different Availability Zone. In case of an infrastructure failure, Amazon RDS performs an automatic failover to the standby so that you can resume database operations as soon as the failover is complete.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            info: 'The RDS DB instance or Aurora DB cluster does not have storage encrypted - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in Amazon RDS DB instances and clusters, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
    }
    /**
     * Check Redshift Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkRedshift(node) {
        this.applyRule({
            info: 'The Redshift cluster does not have encryption or audit logging enabled - (Control IDs: AC-2(4), AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c), SC-13).',
            explanation: 'To protect data at rest, ensure that encryption is enabled for your Amazon Redshift clusters. You must also ensure that required configurations are deployed on Amazon Redshift clusters. The audit logging should be enabled to provide information about connections and user activities in the database.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterConfiguration,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster allows public access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'Amazon Redshift clusters can contain sensitive information and principles and access control is required for such accounts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            info: 'The Redshift cluster does not require TLS/SSL encryption - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
    }
    /**
     * Check Amazon S3 Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkS3(node) {
        this.applyRule({
            info: 'The S3 Bucket does not have object lock enabled - (Control ID: SC-28).',
            explanation: 'Because sensitive data can exist at rest in S3 buckets, enforce object locks at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketDefaultLockEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have server access logs enabled - (Control IDs: AC-2(g), AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicReadProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'The management of access should be consistent with the classification of the data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketPublicWriteProhibited,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have replication enabled - (Control IDs: AU-9(2), CP-9(b), CP-10, SC-5, SC-36).',
            explanation: 'Amazon Simple Storage Service (Amazon S3) Cross-Region Replication (CRR) supports maintaining adequate capacity and availability. CRR enables automatic, asynchronous copying of objects across Amazon S3 buckets to help ensure that data availability is maintained.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketReplicationEnabled,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket or bucket policy does not require requests to use SSL - (Control IDs: AC-17(2), SC-7, SC-8, SC-8(1), SC-13).',
            explanation: 'To help protect data in transit, ensure that your Amazon Simple Storage Service (Amazon S3) buckets require requests to use Secure Socket Layer (SSL). Because sensitive data can exist, enable encryption in transit to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            info: 'The S3 Bucket does not have versioning enabled - (Control IDs: CP-10, SI-12).',
            explanation: 'Use versioning to preserve, retrieve, and restore every version of every object stored in your Amazon S3 bucket. Versioning helps you to easily recover from unintended user actions and application failures.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketVersioningEnabled,
            node: node,
        });
    }
    /**
     * Check SageMaker Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSageMaker(node) {
        this.applyRule({
            info: 'The SageMaker endpoint is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker endpoint, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerEndpointConfigurationKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook is not encrypted with a KMS key - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in SageMaker notebook, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            info: 'The SageMaker notebook does not disable direct internet access - (Control IDs: AC-3, AC-4, AC-6, AC-21(b), SC-7, SC-7(3)).',
            explanation: 'By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Amazon SNS Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSNS(node) {
        this.applyRule({
            info: 'The SNS topic does not have KMS encryption enabled - (Control IDs: SC-13, SC-28).',
            explanation: 'Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSEncryptedKMS,
            node: node,
        });
    }
    /**
     * Check VPC Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkVPC(node) {
        this.applyRule({
            info: "The VPC's default security group allows inbound or outbound traffic - (Control IDs: AC-4, SC-7, SC-7(3)).",
            explanation: 'When creating a VPC through CloudFormation, the default security group will always be open. Therefore it is important to always close the default security group after stack creation whenever a VPC is created. Restricting all the traffic on the default security group helps in restricting remote access to your AWS resources.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCDefaultSecurityGroupClosed,
            node: node,
        });
        this.applyRule({
            info: 'The VPC does not have an associated Flow Log - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c)).',
            explanation: 'The VPC flow logs provide detailed records for information about the IP traffic going to and from network interfaces in your Amazon Virtual Private Cloud (Amazon VPC). By default, the flow log record includes values for the different components of the IP flow, including the source, destination, and protocol.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
    }
    /**
     * Check WAF Resources
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkWAF(node) {
        this.applyRule({
            info: 'The WAFv2 web ACL does not have logging enabled - (Control IDs: AU-2(a)(d), AU-3, AU-12(a)(c), SC-7, SI-4(a)(b)(c)).',
            explanation: 'AWS WAF logging provides detailed information about the traffic that is analyzed by your web ACL. The logs record the time that AWS WAF received the request from your AWS resource, information about the request, and an action for the rule that each request matched.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: waf_1.WAFv2LoggingEnabled,
            node: node,
        });
    }
}
exports.NIST80053R4Checks = NIST80053R4Checks;
_a = JSII_RTTI_SYMBOL_1;
NIST80053R4Checks[_a] = { fqn: "cdk-nag.NIST80053R4Checks", version: "2.28.183" };
//# sourceMappingURL=data:application/json;base64,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