"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
/**
 * EBS volumes have encryption enabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnVolume) {
        const encryption = nag_rules_1.NagRules.resolveIfPrimitive(node, node.encrypted);
        if (encryption !== true) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ec2_1.CfnInstance) {
        const instanceEBSState = InstanceEBSState(node);
        const launchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                if (isMatchingLaunchTemplate(child, launchTemplate.launchTemplateName, launchTemplate.launchTemplateId)) {
                    const instanceLaunchTemplateState = InstanceLaunchTemplateState(child);
                    if (instanceEBSState === BlockDevicesState.Absent &&
                        instanceLaunchTemplateState === BlockDevicesState.Absent) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    if (instanceEBSState === BlockDevicesState.Unencrypted ||
                        instanceLaunchTemplateState === BlockDevicesState.Unencrypted) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    return nag_rules_1.NagRuleCompliance.COMPLIANT;
                }
            }
        }
        if (instanceEBSState === BlockDevicesState.Encrypted) {
            return nag_rules_1.NagRuleCompliance.COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
    }
    else if (node instanceof aws_autoscaling_1.CfnLaunchConfiguration) {
        const blockDeviceMappings = aws_cdk_lib_1.Stack.of(node).resolve(node.blockDeviceMappings);
        if (blockDeviceMappings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        else {
            for (const blockDeviceMapping of blockDeviceMappings) {
                const encryption = nag_rules_1.NagRules.resolveIfPrimitive(node, blockDeviceMapping.ebs.encrypted);
                if (encryption !== true) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_autoscaling_1.CfnAutoScalingGroup) {
        const launchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                if (isMatchingLaunchTemplate(child, launchTemplate.launchTemplateName, launchTemplate.launchTemplateId)) {
                    if (InstanceLaunchTemplateState(child) === BlockDevicesState.Encrypted) {
                        return nag_rules_1.NagRuleCompliance.COMPLIANT;
                    }
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
var BlockDevicesState;
(function (BlockDevicesState) {
    BlockDevicesState[BlockDevicesState["Encrypted"] = 0] = "Encrypted";
    BlockDevicesState[BlockDevicesState["Unencrypted"] = 1] = "Unencrypted";
    BlockDevicesState[BlockDevicesState["Absent"] = 2] = "Absent";
})(BlockDevicesState || (BlockDevicesState = {}));
function isMatchingLaunchTemplate(node, launchTemplateName, launchTemplateId) {
    return (launchTemplateName === node.launchTemplateName ||
        launchTemplateId === nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref));
}
function InstanceLaunchTemplateState(launchTemplate) {
    const launchTemplateData = aws_cdk_lib_1.Stack.of(launchTemplate).resolve(launchTemplate.launchTemplateData);
    if (launchTemplateData.blockDeviceMappings === undefined ||
        launchTemplateData.blockDeviceMappings.length === 0) {
        return BlockDevicesState.Absent;
    }
    else {
        const launchTemplateBlockDeviceMappings = aws_cdk_lib_1.Stack.of(launchTemplate).resolve(launchTemplateData.blockDeviceMappings);
        const devicesAllEncrypted = launchTemplateBlockDeviceMappings.every((blockDeviceMapping) => {
            const encryption = nag_rules_1.NagRules.resolveIfPrimitive(launchTemplate, blockDeviceMapping.ebs.encrypted);
            return encryption === true;
        });
        return devicesAllEncrypted
            ? BlockDevicesState.Encrypted
            : BlockDevicesState.Unencrypted;
    }
}
function InstanceEBSState(node) {
    const blockDeviceMappings = aws_cdk_lib_1.Stack.of(node).resolve(node.blockDeviceMappings);
    if (blockDeviceMappings === undefined || blockDeviceMappings.length === 0) {
        return BlockDevicesState.Absent;
    }
    else {
        const devicesAllEncrypted = blockDeviceMappings.every((blockDeviceMapping) => {
            const encryption = nag_rules_1.NagRules.resolveIfPrimitive(node, blockDeviceMapping.ebs.encrypted);
            return encryption === true;
        });
        return devicesAllEncrypted
            ? BlockDevicesState.Encrypted
            : BlockDevicesState.Unencrypted;
    }
}
//# sourceMappingURL=data:application/json;base64,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