"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
/**
 * The EC2 Instance requires IMDsv2
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnInstance) {
        if (node.launchTemplate === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const instanceLaunchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                if (isMatchingLaunchTemplate(child, instanceLaunchTemplate.launchTemplateName, instanceLaunchTemplate.launchTemplateId)) {
                    return hasHttpTokens(child)
                        ? nag_rules_1.NagRuleCompliance.COMPLIANT
                        : nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
    }
    else if (node instanceof aws_autoscaling_1.CfnAutoScalingGroup) {
        if (node.launchTemplate) {
            const nodeLaunchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                    if (isMatchingLaunchTemplate(child, nodeLaunchTemplate.launchTemplateName, nodeLaunchTemplate.launchTemplateId)) {
                        return hasHttpTokens(child)
                            ? nag_rules_1.NagRuleCompliance.COMPLIANT
                            : nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        else if (node.launchConfigurationName) {
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_autoscaling_1.CfnLaunchConfiguration) {
                    if (isMatchingLaunchConfiguration(child, node.launchConfigurationName)) {
                        return hasHttpTokens(child)
                            ? nag_rules_1.NagRuleCompliance.COMPLIANT
                            : nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
function isMatchingLaunchTemplate(node, launchTemplateName, launchTemplateId) {
    return (launchTemplateName === node.launchTemplateName ||
        launchTemplateId === nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref));
}
function isMatchingLaunchConfiguration(node, launchConfigurationName) {
    return (launchConfigurationName === node.launchConfigurationName ||
        nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, launchConfigurationName) ===
            nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref));
}
function hasHttpTokens(node) {
    let meta;
    if (node instanceof aws_ec2_1.CfnLaunchTemplate) {
        const launchTemplateData = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplateData);
        meta = aws_cdk_lib_1.Stack.of(node).resolve(launchTemplateData.metadataOptions);
    }
    else if (node instanceof aws_autoscaling_1.CfnLaunchConfiguration) {
        meta = aws_cdk_lib_1.Stack.of(node).resolve(node.metadataOptions);
    }
    return meta?.httpTokens === 'required';
}
//# sourceMappingURL=data:application/json;base64,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