"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
const BLOCKED_PORTS = [20, 21, 3389, 3309, 3306, 4333];
/**
 * EC2 instances have all common TCP ports restricted for ingress IPv4 traffic
 * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/default-custom-security-groups.html
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnSecurityGroup) {
        const ingressRules = aws_cdk_lib_1.Stack.of(node).resolve(node.securityGroupIngress);
        if (ingressRules != undefined) {
            //For each ingress rule, ensure that it does not allow unrestricted SSH traffic.
            for (const rule of ingressRules) {
                const resolvedRule = aws_cdk_lib_1.Stack.of(node).resolve(rule);
                for (const portNum of BLOCKED_PORTS) {
                    if (!testPort(node, resolvedRule, portNum)) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_ec2_1.CfnSecurityGroupIngress) {
        for (const portNum of BLOCKED_PORTS) {
            if (!testPort(node, node, portNum)) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to identify if the given port number is unrestricted
 * @param node the CfnResource to check
 * @param rule the CfnSecurityGroupIngress rule to check
 * @param portNum the number of the port to check
 */
function testPort(node, rule, portNum) {
    //Does this rule apply to TCP traffic?
    const ipProtocol = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.ipProtocol);
    const cidrIp = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.cidrIp);
    const fromPort = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.fromPort);
    const toPort = nag_rules_1.NagRules.resolveIfPrimitive(node, rule.toPort);
    if (ipProtocol === 'tcp') {
        //Does this rule allow all IPv4 addresses (unrestricted access)?
        if (cidrIp != undefined && cidrIp.includes('/0')) {
            //Is a port range specified?
            if (fromPort != undefined && toPort != undefined) {
                if ((fromPort <= portNum && toPort >= portNum) ||
                    fromPort == -1 ||
                    toPort == -1) {
                    return false;
                }
            }
            else {
                if (fromPort == portNum) {
                    return false;
                }
            }
        }
    }
    //Are all ports allowed?
    if (ipProtocol === '-1') {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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