"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const nag_rules_1 = require("../../nag-rules");
/**
 * EKS Cluster Kubernetes API server endpoints have public access disabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_eks_1.CfnCluster) {
        const resourcesVpcConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.resourcesVpcConfig);
        const endpointPublicAccess = nag_rules_1.NagRules.resolveIfPrimitive(node, resourcesVpcConfig.endpointPublicAccess);
        if (endpointPublicAccess !== false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node.cfnResourceType === 'Custom::AWSCDK-EKS-Cluster') {
        // The CDK uses a Custom Resource with AWS SDK calls to create EKS Clusters
        const props = aws_cdk_lib_1.Stack.of(node).resolve(node._cfnProperties);
        const endpointPublicAccess = nag_rules_1.NagRules.resolveIfPrimitive(node, props?.Config?.resourcesVpcConfig?.endpointPublicAccess);
        if (endpointPublicAccess !== false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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