"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticsearch_1 = require("aws-cdk-lib/aws-elasticsearch");
const aws_opensearchservice_1 = require("aws-cdk-lib/aws-opensearchservice");
const nag_rules_1 = require("../../nag-rules");
/**
 * OpenSearch Service domains only grant access via allowlisted IP addresses
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticsearch_1.CfnDomain || node instanceof aws_opensearchservice_1.CfnDomain) {
        const accessPolicies = getPolicyFromCR() ?? aws_cdk_lib_1.Stack.of(node).resolve(node.accessPolicies);
        if (accessPolicies === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const statements = accessPolicies?.Statement;
        if (statements === undefined || statements.length === 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        for (const statement of statements) {
            if (statement.Effect === 'Allow') {
                const allowList = statement?.Condition?.IpAddress?.['aws:sourceIp'];
                if (allowList === undefined || allowList.length === 0) {
                    return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
    function getPolicyFromCR() {
        const crPath = node.node.path.split('/Resource').slice(0, -1).join('/Resource') +
            (node instanceof aws_opensearchservice_1.CfnDomain ? '/AccessPolicy' : '/ESAccessPolicy') +
            '/Resource/Default';
        const cr = aws_cdk_lib_1.Stack.of(node)
            .node.findAll()
            .find((c) => c.node.path === crPath);
        if (cr) {
            const props = aws_cdk_lib_1.Stack.of(cr).resolve(cr._cfnProperties);
            const update = props?.Update?.['Fn::Join']?.[1];
            return JSON.parse(JSON.parse(update.join('')).parameters.AccessPolicies);
        }
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiT3BlblNlYXJjaEFsbG93bGlzdGVkSVBzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3J1bGVzL29wZW5zZWFyY2gvT3BlblNlYXJjaEFsbG93bGlzdGVkSVBzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7OztFQUdFO0FBQ0YsK0JBQTZCO0FBQzdCLDZDQUFpRDtBQUNqRCxxRUFBNkU7QUFDN0UsNkVBQThEO0FBQzlELCtDQUFvRDtBQUVwRDs7O0dBR0c7QUFDSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQXFCLEVBQUU7SUFDdkMsSUFBSSxJQUFJLFlBQVksNkJBQWUsSUFBSSxJQUFJLFlBQVksaUNBQVMsRUFBRTtRQUNoRSxNQUFNLGNBQWMsR0FDbEIsZUFBZSxFQUFFLElBQUksbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUNuRSxJQUFJLGNBQWMsS0FBSyxTQUFTLEVBQUU7WUFDaEMsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7U0FDeEM7UUFDRCxNQUFNLFVBQVUsR0FBRyxjQUFjLEVBQUUsU0FBUyxDQUFDO1FBQzdDLElBQUksVUFBVSxLQUFLLFNBQVMsSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN2RCxPQUFPLDZCQUFpQixDQUFDLGFBQWEsQ0FBQztTQUN4QztRQUNELEtBQUssTUFBTSxTQUFTLElBQUksVUFBVSxFQUFFO1lBQ2xDLElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxPQUFPLEVBQUU7Z0JBQ2hDLE1BQU0sU0FBUyxHQUFHLFNBQVMsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7Z0JBQ3BFLElBQUksU0FBUyxLQUFLLFNBQVMsSUFBSSxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtvQkFDckQsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7aUJBQ3hDO2FBQ0Y7U0FDRjtRQUNELE9BQU8sNkJBQWlCLENBQUMsU0FBUyxDQUFDO0tBQ3BDO1NBQU07UUFDTCxPQUFPLDZCQUFpQixDQUFDLGNBQWMsQ0FBQztLQUN6QztJQUNELFNBQVMsZUFBZTtRQUN0QixNQUFNLE1BQU0sR0FDVixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7WUFDaEUsQ0FBQyxJQUFJLFlBQVksaUNBQVMsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQztZQUNqRSxtQkFBbUIsQ0FBQztRQUN0QixNQUFNLEVBQUUsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUM7YUFDdEIsSUFBSSxDQUFDLE9BQU8sRUFBRTthQUNkLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssTUFBTSxDQUFDLENBQUM7UUFDdkMsSUFBSSxFQUFFLEVBQUU7WUFDTixNQUFNLEtBQUssR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQU8sRUFBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQzdELE1BQU0sTUFBTSxHQUFHLEtBQUssRUFBRSxNQUFNLEVBQUUsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FDZixJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUN0RCxDQUFDO1NBQ0g7SUFDSCxDQUFDO0FBQ0gsQ0FBQyxFQUNELE1BQU0sRUFDTixFQUFFLEtBQUssRUFBRSxZQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsSUFBSSxFQUFFLENBQ2xDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuKi9cbmltcG9ydCB7IHBhcnNlIH0gZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5Eb21haW4gYXMgTGVnYWN5Q2ZuRG9tYWluIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWVsYXN0aWNzZWFyY2gnO1xuaW1wb3J0IHsgQ2ZuRG9tYWluIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLW9wZW5zZWFyY2hzZXJ2aWNlJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlIH0gZnJvbSAnLi4vLi4vbmFnLXJ1bGVzJztcblxuLyoqXG4gKiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWlucyBvbmx5IGdyYW50IGFjY2VzcyB2aWEgYWxsb3dsaXN0ZWQgSVAgYWRkcmVzc2VzXG4gKiBAcGFyYW0gbm9kZSB0aGUgQ2ZuUmVzb3VyY2UgdG8gY2hlY2tcbiAqL1xuZXhwb3J0IGRlZmF1bHQgT2JqZWN0LmRlZmluZVByb3BlcnR5KFxuICAobm9kZTogQ2ZuUmVzb3VyY2UpOiBOYWdSdWxlQ29tcGxpYW5jZSA9PiB7XG4gICAgaWYgKG5vZGUgaW5zdGFuY2VvZiBMZWdhY3lDZm5Eb21haW4gfHwgbm9kZSBpbnN0YW5jZW9mIENmbkRvbWFpbikge1xuICAgICAgY29uc3QgYWNjZXNzUG9saWNpZXMgPVxuICAgICAgICBnZXRQb2xpY3lGcm9tQ1IoKSA/PyBTdGFjay5vZihub2RlKS5yZXNvbHZlKG5vZGUuYWNjZXNzUG9saWNpZXMpO1xuICAgICAgaWYgKGFjY2Vzc1BvbGljaWVzID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICB9XG4gICAgICBjb25zdCBzdGF0ZW1lbnRzID0gYWNjZXNzUG9saWNpZXM/LlN0YXRlbWVudDtcbiAgICAgIGlmIChzdGF0ZW1lbnRzID09PSB1bmRlZmluZWQgfHwgc3RhdGVtZW50cy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICB9XG4gICAgICBmb3IgKGNvbnN0IHN0YXRlbWVudCBvZiBzdGF0ZW1lbnRzKSB7XG4gICAgICAgIGlmIChzdGF0ZW1lbnQuRWZmZWN0ID09PSAnQWxsb3cnKSB7XG4gICAgICAgICAgY29uc3QgYWxsb3dMaXN0ID0gc3RhdGVtZW50Py5Db25kaXRpb24/LklwQWRkcmVzcz8uWydhd3M6c291cmNlSXAnXTtcbiAgICAgICAgICBpZiAoYWxsb3dMaXN0ID09PSB1bmRlZmluZWQgfHwgYWxsb3dMaXN0Lmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuQ09NUExJQU5UO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gTmFnUnVsZUNvbXBsaWFuY2UuTk9UX0FQUExJQ0FCTEU7XG4gICAgfVxuICAgIGZ1bmN0aW9uIGdldFBvbGljeUZyb21DUigpOiBhbnkge1xuICAgICAgY29uc3QgY3JQYXRoID1cbiAgICAgICAgbm9kZS5ub2RlLnBhdGguc3BsaXQoJy9SZXNvdXJjZScpLnNsaWNlKDAsIC0xKS5qb2luKCcvUmVzb3VyY2UnKSArXG4gICAgICAgIChub2RlIGluc3RhbmNlb2YgQ2ZuRG9tYWluID8gJy9BY2Nlc3NQb2xpY3knIDogJy9FU0FjY2Vzc1BvbGljeScpICtcbiAgICAgICAgJy9SZXNvdXJjZS9EZWZhdWx0JztcbiAgICAgIGNvbnN0IGNyID0gU3RhY2sub2Yobm9kZSlcbiAgICAgICAgLm5vZGUuZmluZEFsbCgpXG4gICAgICAgIC5maW5kKChjKSA9PiBjLm5vZGUucGF0aCA9PT0gY3JQYXRoKTtcbiAgICAgIGlmIChjcikge1xuICAgICAgICBjb25zdCBwcm9wcyA9IFN0YWNrLm9mKGNyKS5yZXNvbHZlKCg8YW55PmNyKS5fY2ZuUHJvcGVydGllcyk7XG4gICAgICAgIGNvbnN0IHVwZGF0ZSA9IHByb3BzPy5VcGRhdGU/LlsnRm46OkpvaW4nXT8uWzFdO1xuICAgICAgICByZXR1cm4gSlNPTi5wYXJzZShcbiAgICAgICAgICBKU09OLnBhcnNlKHVwZGF0ZS5qb2luKCcnKSkucGFyYW1ldGVycy5BY2Nlc3NQb2xpY2llc1xuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cbiAgfSxcbiAgJ25hbWUnLFxuICB7IHZhbHVlOiBwYXJzZShfX2ZpbGVuYW1lKS5uYW1lIH1cbik7XG4iXX0=