"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_rds_1 = require("aws-cdk-lib/aws-rds");
const nag_rules_1 = require("../../nag-rules");
/**
 * Non-Aurora RDS DB instances and Aurora clusters are configured to export all possible log types to CloudWatch
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_rds_1.CfnDBInstance) {
        const dbType = String(nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine));
        const exports = aws_cdk_lib_1.Stack.of(node).resolve(node.enableCloudwatchLogsExports) ?? [];
        const needed = [];
        if (!dbType.includes('aurora')) {
            if (dbType.includes('mariadb') || dbType.includes('mysql')) {
                needed.push('audit', 'error', 'general', 'slowquery');
            }
            else if (dbType.includes('postgres')) {
                needed.push('postgresql', 'upgrade');
            }
            else if (dbType.includes('oracle')) {
                needed.push('audit', 'alert', 'listener', 'oemagent', 'trace');
            }
            else if (dbType.includes('sqlserver')) {
                needed.push('agent', 'error');
            }
            const findings = needed
                .filter((log) => !exports.includes(log))
                .map((log) => `LogExport::${log}`);
            return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
        }
        else {
            return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
        }
    }
    else if (node instanceof aws_rds_1.CfnDBCluster) {
        const engine = nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine).toLowerCase();
        const needed = [];
        const exports = aws_cdk_lib_1.Stack.of(node).resolve(node.enableCloudwatchLogsExports) ?? [];
        if (engine.toLowerCase() === 'aurora' ||
            engine.toLowerCase() === 'aurora-mysql') {
            needed.push('audit', 'error', 'general', 'slowquery');
        }
        else if (engine.toLowerCase() === 'aurora-postgresql') {
            needed.push('postgresql');
        }
        const findings = needed
            .filter((log) => !exports.includes(log))
            .map((log) => `LogExport::${log}`);
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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