"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const nag_rules_1 = require("../../nag-rules");
const flatten_cfn_reference_1 = require("../../utils/flatten-cfn-reference");
/**
 * SQS queues have a dead-letter queue enabled if they are not used as a dead-letter queue
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_sqs_1.CfnQueue) {
        const redrivePolicy = aws_cdk_lib_1.Stack.of(node).resolve(node.redrivePolicy);
        if (redrivePolicy === undefined) {
            const queueLogicalId = nag_rules_1.NagRules.resolveResourceFromInstrinsic(node, node.ref);
            const queueName = aws_cdk_lib_1.Stack.of(node).resolve(node.queueName);
            let found = false;
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_sqs_1.CfnQueue) {
                    if (isMatchingQueue(child, queueLogicalId, queueName)) {
                        found = true;
                        break;
                    }
                }
                else if (child instanceof aws_lambda_1.CfnFunction) {
                    if (isMatchingLambdaFunction(child, queueLogicalId, queueName)) {
                        found = true;
                        break;
                    }
                }
            }
            if (!found) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether a given SQS Queue uses the target SQS queue as a DLQ
 * @param node the CfnQueue to check
 * @param queueLogicalId the Cfn Logical ID of the target queue
 * @param queueName the name of the target queue
 * returns whether the CfnQueue uses the target SQS queue as a DLQ
 */
function isMatchingQueue(node, queueLogicalId, queueName) {
    const redrivePolicy = aws_cdk_lib_1.Stack.of(node).resolve(node.redrivePolicy);
    const deadLetterTargetArn = flatten_cfn_reference_1.flattenCfnReference(redrivePolicy?.deadLetterTargetArn ?? '');
    if (new RegExp(`${queueLogicalId}(?![\\w])`).test(deadLetterTargetArn) ||
        (queueName !== undefined &&
            new RegExp(`:${queueName}(?![\\w\\-_\\.])`).test(deadLetterTargetArn))) {
        return true;
    }
    return false;
}
/**
 * Helper function to check whether a given Lambda Function uses the target SQS queue as a DLQ
 * @param node the CfnFunction to check
 * @param queueLogicalId the Cfn Logical ID of the target queue
 * @param queueName the name of the target queue
 * returns whether the CfnFunction uses the target SQS queue as a DLQ
 */
function isMatchingLambdaFunction(node, queueLogicalId, queueName) {
    const deadLetterConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.deadLetterConfig);
    const targetArn = flatten_cfn_reference_1.flattenCfnReference(aws_cdk_lib_1.Stack.of(node).resolve(deadLetterConfig?.targetArn) ?? '');
    if (new RegExp(`${queueLogicalId}(?![\\w])`).test(targetArn) ||
        (queueName !== undefined &&
            new RegExp(`:${queueName}(?![\\w\\-_\\.])`).test(targetArn))) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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