from __future__ import annotations

from typing import Union

from typing_extensions import Literal

from abqpy.decorators import abaqus_class_doc, abaqus_method_doc

from ..Datum.DatumAxis import DatumAxis
from ..Region.Region import Region
from ..UtilityAndView.abaqusConstants import (
    COMPUTED,
    DEFAULT,
    KINEMATIC,
    OFF,
    OMIT,
    Boolean,
    SymbolicConstant,
)
from ..UtilityAndView.abaqusConstants import abaqusConstants as C
from .Interaction import Interaction


@abaqus_class_doc
class SurfaceToSurfaceContactExp(Interaction):
    """The SurfaceToSurfaceContactExp object defines surface-to-surface contact during an Abaqus/Explicit
    analysis. The SurfaceToSurfaceContactExp object is derived from the Interaction object.

    .. note::
        This object can be accessed by::

            import interaction
            mdb.models[name].interactions[name]
    """

    #: A String specifying the repository key.
    name: str

    #: A String specifying the name of the step in which the SurfaceToSurfaceContactExp object
    #: is created.
    createStepName: str

    #: A Region object specifying the master surface.
    master: Region

    #: A Region object specifying the slave surface.
    slave: Region

    #: A SymbolicConstant specifying the contact formulation. Possible values are FINITE and
    #: SMALL.
    sliding: SymbolicConstant

    #: A String specifying the name of the ContactProperty object associated with this
    #: interaction.
    interactionProperty: str

    #: A SymbolicConstant specifying the mechanical constraint formulation. Possible values are
    #: KINEMATIC and PENALTY. The default value is KINEMATIC.
    mechanicalConstraint: SymbolicConstant = KINEMATIC

    #: A SymbolicConstant specifying the weighting for node-to-face contact. Possible values
    #: are DEFAULT and SPECIFIED. The default value is DEFAULT.
    weightingFactorType: SymbolicConstant = DEFAULT

    #: A Float specifying the weighting factor for the contact surfaces when
    #: **weightingFactorType** = SPECIFIED. The default value is 0.0.
    weightingFactor: float = 0

    #: A String specifying the name of the ContactControl object associated with this
    #: interaction. An empty string indicates that the default contact controls will be used.
    #: The default value is an empty string.
    contactControls: str = ""

    #: A SymbolicConstant or a Float specifying the initial clearance at regions of contact.
    #: Possible values are OMIT and COMPUTED. The default value is OMIT.
    initialClearance: Union[SymbolicConstant, float] = OMIT

    #: None or a sequence of Floats specifying the half thread angle used for bolt clearance.
    #: The default value is None.
    halfThreadAngle: str | None = None

    #: None or a sequence of Floats specifying the pitch used for bolt clearance. The default
    #: value is None.
    pitch: str | None = None

    #: The SymbolicConstant COMPUTED or a Float specifying the major diameter of the bolt used
    #: for bolt clearance. The default value is COMPUTED.
    majorBoltDiameter: Union[SymbolicConstant, float] = COMPUTED

    #: The SymbolicConstant COMPUTED or a Float specifying the mean diameter of the bolt used
    #: for bolt clearance. The default value is COMPUTED.
    meanBoltDiameter: Union[SymbolicConstant, float] = COMPUTED

    #: A DatumAxis object specifying the orientation of the bolt hole when specifying bolt
    #: clearance.
    datumAxis: DatumAxis | None = None

    #: A Boolean specifying whether to reverse the bolt clearance direction given by the datum
    #: axis. The default value is OFF.
    useReverseDatumAxis: Boolean = OFF

    #: A Region object specifying the contact region for which clearance is specified.
    clearanceRegion: Region | None = None

    @abaqus_method_doc
    def __init__(
        self,
        name: str,
        createStepName: str,
        master: Region,
        slave: Region,
        sliding: Literal[C.SMALL, C.FINITE],
        interactionProperty: str,
        mechanicalConstraint: Literal[C.PENALTY, C.KINEMATIC] = KINEMATIC,
        weightingFactorType: Literal[C.DEFAULT, C.SPECIFIED] = DEFAULT,
        weightingFactor: float = 0,
        contactControls: str = "",
        initialClearance: Union[Literal[C.OMIT, C.COMPUTED], float] = OMIT,
        halfThreadAngle: str | None = None,
        pitch: str | None = None,
        majorBoltDiameter: Union[Literal[C.COMPUTED], float] = COMPUTED,
        meanBoltDiameter: Union[Literal[C.COMPUTED], float] = COMPUTED,
        datumAxis: DatumAxis | None = None,
        useReverseDatumAxis: Boolean = OFF,
        clearanceRegion: Region | None = None,
    ):
        """This method creates a SurfaceToSurfaceContactExp object.

        .. note::
            This function can be accessed by::

                mdb.models[name].SurfaceToSurfaceContactExp

        Parameters
        ----------
        name
            A String specifying the repository key.
        createStepName
            A String specifying the name of the step in which the SurfaceToSurfaceContactExp object
            is created.
        master
            A Region object specifying the master surface.
        slave
            A Region object specifying the slave surface.
        sliding
            A SymbolicConstant specifying the contact formulation. Possible values are FINITE and
            SMALL.
        interactionProperty
            A String specifying the name of the ContactProperty object associated with this
            interaction.
        mechanicalConstraint
            A SymbolicConstant specifying the mechanical constraint formulation. Possible values are
            KINEMATIC and PENALTY. The default value is KINEMATIC.
        weightingFactorType
            A SymbolicConstant specifying the weighting for node-to-face contact. Possible values
            are DEFAULT and SPECIFIED. The default value is DEFAULT.
        weightingFactor
            A Float specifying the weighting factor for the contact surfaces when
            **weightingFactorType** = SPECIFIED. The default value is 0.0.
        contactControls
            A String specifying the name of the ContactControl object associated with this
            interaction. An empty string indicates that the default contact controls will be used.
            The default value is an empty string.
        initialClearance
            A SymbolicConstant or a Float specifying the initial clearance at regions of contact.
            Possible values are OMIT and COMPUTED. The default value is OMIT.
        halfThreadAngle
            None or a sequence of Floats specifying the half thread angle used for bolt clearance.
            The default value is None.
        pitch
            None or a sequence of Floats specifying the pitch used for bolt clearance. The default
            value is None.
        majorBoltDiameter
            The SymbolicConstant COMPUTED or a Float specifying the major diameter of the bolt used
            for bolt clearance. The default value is COMPUTED.
        meanBoltDiameter
            The SymbolicConstant COMPUTED or a Float specifying the mean diameter of the bolt used
            for bolt clearance. The default value is COMPUTED.
        datumAxis
            A DatumAxis object specifying the orientation of the bolt hole when specifying bolt
            clearance.
        useReverseDatumAxis
            A Boolean specifying whether to reverse the bolt clearance direction given by the datum
            axis. The default value is OFF.
        clearanceRegion
            A Region object specifying the contact region for which clearance is specified.

        Returns
        -------
        SurfaceToSurfaceContactExp
            A SurfaceToSurfaceContactExp object.
        """
        super().__init__()

    @abaqus_method_doc
    def swapSurfaces(self):
        """This method switches the master and slave surfaces of a surface-to-surface contact pair.

        This command is valid only during the step in which the interaction is created.
        """
        ...

    @abaqus_method_doc
    def setValues(
        self,
        mechanicalConstraint: Literal[C.PENALTY, C.KINEMATIC] = KINEMATIC,
        weightingFactorType: Literal[C.DEFAULT, C.SPECIFIED] = DEFAULT,
        weightingFactor: float = 0,
        contactControls: str = "",
        initialClearance: Union[Literal[C.OMIT, C.COMPUTED], float] = OMIT,
        halfThreadAngle: str | None = None,
        pitch: str | None = None,
        majorBoltDiameter: Union[Literal[C.COMPUTED], float] = COMPUTED,
        meanBoltDiameter: Union[Literal[C.COMPUTED], float] = COMPUTED,
        datumAxis: DatumAxis | None = None,
        useReverseDatumAxis: Boolean = OFF,
        clearanceRegion: Region | None = None,
    ):
        """This method modifies the data for an existing SurfaceToSurfaceContactExp object in the step where it
        is created.

        Parameters
        ----------
        mechanicalConstraint
            A SymbolicConstant specifying the mechanical constraint formulation. Possible values are
            KINEMATIC and PENALTY. The default value is KINEMATIC.
        weightingFactorType
            A SymbolicConstant specifying the weighting for node-to-face contact. Possible values
            are DEFAULT and SPECIFIED. The default value is DEFAULT.
        weightingFactor
            A Float specifying the weighting factor for the contact surfaces when
            **weightingFactorType** = SPECIFIED. The default value is 0.0.
        contactControls
            A String specifying the name of the ContactControl object associated with this
            interaction. An empty string indicates that the default contact controls will be used.
            The default value is an empty string.
        initialClearance
            A SymbolicConstant or a Float specifying the initial clearance at regions of contact.
            Possible values are OMIT and COMPUTED. The default value is OMIT.
        halfThreadAngle
            None or a sequence of Floats specifying the half thread angle used for bolt clearance.
            The default value is None.
        pitch
            None or a sequence of Floats specifying the pitch used for bolt clearance. The default
            value is None.
        majorBoltDiameter
            The SymbolicConstant COMPUTED or a Float specifying the major diameter of the bolt used
            for bolt clearance. The default value is COMPUTED.
        meanBoltDiameter
            The SymbolicConstant COMPUTED or a Float specifying the mean diameter of the bolt used
            for bolt clearance. The default value is COMPUTED.
        datumAxis
            A DatumAxis object specifying the orientation of the bolt hole when specifying bolt
            clearance.
        useReverseDatumAxis
            A Boolean specifying whether to reverse the bolt clearance direction given by the datum
            axis. The default value is OFF.
        clearanceRegion
            A Region object specifying the contact region for which clearance is specified.
        """
        ...

    @abaqus_method_doc
    def setValuesInStep(self, stepName: str, interactionProperty: str = "", contactControls: str = ""):
        """This method modifies the propagating data for an existing SurfaceToSurfaceContactExp object in the
        specified step.

        Parameters
        ----------
        stepName
            A String specifying the name of the step in which the interaction is modified.
        interactionProperty
            A String specifying the name of the ContactProperty object associated with this
            interaction.
        contactControls
            A String specifying the name of the ContactControl object associated with this
            interaction. An empty string indicates that the default contact controls will be used.
            The default value is an empty string.
        """
        ...
