from __future__ import annotations

from enum import Enum

from abqpy.decorators import abaqus_class_doc, abaqus_method_doc

from .AbaqusBoolean import AbaqusBoolean

value = "SymbolicConstant"
name = "SymbolicConstant"


@abaqus_class_doc
class SymbolicConstant(str):
    """The SymbolicConstant object represents a string in a way that can be stored in a replay file and used as
    an argument to Abaqus Scripting Interface methods and functions. By convention the string representation of
    the SymbolicConstant object is the same as its variable name. If you pass a SymbolicConstant object to the
    Python repr() function, the function returns the text without quotes. In effect, the text is the variable
    that, by convention, refers to the SymbolicConstant object. Two SymbolicConstant objects with the same text
    are the same Python object, although you can assign them to different variables. All of the SymbolicConstant
    objects that are required in Abaqus Scripting Interface methods are defined in the abaqusConstants module.
    Some SymbolicConstant objects and the SymbolicConstant constructor are defined in the abaqus module. The
    SymbolicConstant constructor is also defined in the symbolicConstants module.

    .. note::
        This object can be accessed by::

            from symbolicConstants import *
            from abaqusConstants import *
    """

    @abaqus_method_doc
    def __init__(self, text: str) -> None:
        """The SymbolicConstant method creates a SymbolicConstant object.

        .. note::
            This function can be accessed by::

                SymbolicConstant

        Parameters
        ----------
        text
            A String specifying the text of the SymbolicConstant object. The String must contain
            only capital letters, digits, or underscores and must not start with a digit.

        Returns
        -------
        SymbolicConstant
            A SymbolicConstant object.
        """
        if not text.isupper() or not text.isidentifier():
            raise ValueError(f"SymbolicConstant name {text} may only contain upper case, digit or underscore")
        self.text = text

    def getId(self) -> int:
        return id(self)

    def getText(self) -> str:
        return self.text

    def __new__(cls, name: str) -> SymbolicConstant:
        if name == "ON":
            return AbaqusBoolean(1)
        elif name == "OFF":
            return AbaqusBoolean(0)
        else:
            return super().__new__(cls, name)


class abaqusConstants(SymbolicConstant, Enum):
    def __str__(self) -> str:
        return self.name

    def __repr__(self) -> str:
        return self.name

    YES = "YES"
    NO = "NO"

    ALL_TYPES = "ALL_TYPES"
    ALL_METHODS = "ALL_METHODS"
    BOTH = "BOTH"
    EARLIEST = "EARLIEST"
    GUI = "GUI"
    KERNEL = "KERNEL"
    LATEST = "LATEST"

    A0 = "A0"
    A1 = "A1"
    A2 = "A2"
    A3 = "A3"
    A4 = "A4"
    A5 = "A5"
    ABAQUS = "ABAQUS"
    ABAQUS_AQUA = "ABAQUS_AQUA"
    ABAQUS_BIORID = "ABAQUS_BIORID"
    ABAQUS_CEL = "ABAQUS_CEL"
    ABAQUS_CFD = "ABAQUS_CFD"
    ABAQUS_DESIGN = "ABAQUS_DESIGN"
    ABAQUS_EXPLICIT = "ABAQUS_EXPLICIT"
    ABAQUS_STANDARD = "ABAQUS_STANDARD"
    ABORTED = "ABORTED"
    ABS = "ABS"
    ABSCISSA = "ABSCISSA"
    ABSOLUTE = "ABSOLUTE"
    ABSOLUTE_DIFFERENCE = "ABSOLUTE_DIFFERENCE"
    ABSOLUTE_DISTANCE = "ABSOLUTE_DISTANCE"
    ABSOLUTE_EQUAL = "ABSOLUTE_EQUAL"
    ABSOLUTE_GREATER_THAN_EQUAL = "ABSOLUTE_GREATER_THAN_EQUAL"
    ABSOLUTE_GROWTH_MOVEMENT = "ABSOLUTE_GROWTH_MOVEMENT"
    ABSOLUTE_LESS_THAN_EQUAL = "ABSOLUTE_LESS_THAN_EQUAL"
    ABSOLUTE_SHRINK_MOVEMENT = "ABSOLUTE_SHRINK_MOVEMENT"
    ABSOLUTE_VALUE = "ABSOLUTE_VALUE"
    ABS_DEFAULT = "ABS_DEFAULT"
    AC1D2 = "AC1D2"
    AC1D3 = "AC1D3"
    AC2D3 = "AC2D3"
    AC2D4 = "AC2D4"
    AC2D4R = "AC2D4R"
    AC2D6 = "AC2D6"
    AC2D8 = "AC2D8"
    AC3D10 = "AC3D10"
    AC3D15 = "AC3D15"
    AC3D20 = "AC3D20"
    AC3D4 = "AC3D4"
    AC3D6 = "AC3D6"
    AC3D8 = "AC3D8"
    AC3D8R = "AC3D8R"
    ACAX3 = "ACAX3"
    ACAX4 = "ACAX4"
    ACAX4R = "ACAX4R"
    ACAX6 = "ACAX6"
    ACAX8 = "ACAX8"
    ACCELERATION = "ACCELERATION"
    ACCELEROMETER = "ACCELEROMETER"
    ACIN2D2 = "ACIN2D2"
    ACIN2D3 = "ACIN2D3"
    ACIN3D3 = "ACIN3D3"
    ACIN3D4 = "ACIN3D4"
    ACIN3D6 = "ACIN3D6"
    ACIN3D8 = "ACIN3D8"
    ACINAX2 = "ACINAX2"
    ACINAX3 = "ACINAX3"
    ACOUSTIC = "ACOUSTIC"
    ACOUSTICS = "ACOUSTICS"
    ACOUSTIC_INTENSITY = "ACOUSTIC_INTENSITY"
    ACROSS = "ACROSS"
    ACTIVE_CUT_RANGE = "ACTIVE_CUT_RANGE"
    AC_OFF = "AC_OFF"
    AC_ON = "AC_ON"
    AC_PROJECTION = "AC_PROJECTION"
    ADAPTIVE_MESHING = "ADAPTIVE_MESHING"
    ADAPTIVE_MESH_SMOOTHING_FAILED = "ADAPTIVE_MESH_SMOOTHING_FAILED"
    ADD = "ADD"
    ADJUST = "ADJUST"
    ADJUST_LENGTH = "ADJUST_LENGTH"
    ADMITTANCE = "ADMITTANCE"
    ADVANCING_FRONT = "ADVANCING_FRONT"
    AFTER = "AFTER"
    AGGRESSIVE = "AGGRESSIVE"
    AIR_BLAST = "AIR_BLAST"
    ALGORITHM = "ALGORITHM"
    ALIGN = "ALIGN"
    ALL = "ALL"
    ALLISO = "ALLISO"
    ALLOW_SUBCYCLING = "ALLOW_SUBCYCLING"
    ALLSTAR = "ALLSTAR"
    ALL_DAMAGE_STEPS = "ALL_DAMAGE_STEPS"
    ALL_DATUMS = "ALL_DATUMS"
    ALL_DIRECT = "ALL_DIRECT"
    ALL_DIRECT_COMPONENTS = "ALL_DIRECT_COMPONENTS"
    ALL_DISTRIB_CONTINUUM_COUPLING = "ALL_DISTRIB_CONTINUUM_COUPLING"
    ALL_DISTRIB_COUPLING = "ALL_DISTRIB_COUPLING"
    ALL_DISTRIB_STRUCTURAL_COUPLING = "ALL_DISTRIB_STRUCTURAL_COUPLING"
    ALL_EDGES = "ALL_EDGES"
    ALL_ELEMENTS = "ALL_ELEMENTS"
    ALL_FRAMES = "ALL_FRAMES"
    ALL_FREQUENCIES = "ALL_FREQUENCIES"
    ALL_GEOMETRY = "ALL_GEOMETRY"
    ALL_INCREMENTS = "ALL_INCREMENTS"
    ALL_KINEM_COUPLING = "ALL_KINEM_COUPLING"
    ALL_LOCATIONS = "ALL_LOCATIONS"
    ALL_MODAL_STEPS = "ALL_MODAL_STEPS"
    ALL_MPCS = "ALL_MPCS"
    ALL_NODAL_DIAMETER = "ALL_NODAL_DIAMETER"
    ALL_NODES = "ALL_NODES"
    ALL_NONLINEAR_STEPS = "ALL_NONLINEAR_STEPS"
    ALL_PRINCIPALS = "ALL_PRINCIPALS"
    ALL_PRINCIPAL_COMPONENTS = "ALL_PRINCIPAL_COMPONENTS"
    ALL_RIGID_BODY = "ALL_RIGID_BODY"
    ALL_SHELL_SOLID_COUPLING = "ALL_SHELL_SOLID_COUPLING"
    ALL_STATIC_STEPS = "ALL_STATIC_STEPS"
    ALL_SURFACES = "ALL_SURFACES"
    ALL_TIES = "ALL_TIES"
    ALWAYS = "ALWAYS"
    AMBIENT = "AMBIENT"
    AMG = "AMG"
    AMPLITUDE = "AMPLITUDE"
    AMS = "AMS"
    ANALYSIS = "ANALYSIS"
    ANALYSIS_CHECKS = "ANALYSIS_CHECKS"
    ANALYSIS_DEFAULT = "ANALYSIS_DEFAULT"
    ANALYSIS_PRODUCT_DEFAULT = "ANALYSIS_PRODUCT_DEFAULT"
    ANALYTICAL_FIELD = "ANALYTICAL_FIELD"
    ANALYTIC_RIGID_SURFACE = "ANALYTIC_RIGID_SURFACE"
    ANGLE = "ANGLE"
    ANGLE_0 = "ANGLE_0"
    ANGLE_45 = "ANGLE_45"
    ANGLE_90 = "ANGLE_90"
    ANGLE_NEG45 = "ANGLE_NEG45"
    ANGULAR = "ANGULAR"
    ANGULAR_DEVIATION = "ANGULAR_DEVIATION"
    ANGULAR_MOMENTUM = "ANGULAR_MOMENTUM"
    ANIMATION = "ANIMATION"
    ANISOTROPIC = "ANISOTROPIC"
    ANNEAL = "ANNEAL"
    ANNEALING = "ANNEALING"
    ANTIALIASING = "ANTIALIASING"
    ANY_JOB = "ANY_JOB"
    ANY_MESSAGE_TYPE = "ANY_MESSAGE_TYPE"
    APPLY_FORCE = "APPLY_FORCE"
    AQUA = "AQUA"
    ARC = "ARC"
    ARC_LENGTH = "ARC_LENGTH"
    AREA = "AREA"
    AREA_VELOCITY_SQUARED = "AREA_VELOCITY_SQUARED"
    ARRAY = "ARRAY"
    ARRAY_1D = "ARRAY_1D"
    ARRHENIUS = "ARRHENIUS"
    ARROW = "ARROW"
    ARRUDA_BOYCE = "ARRUDA_BOYCE"
    ASCENDING = "ASCENDING"
    ASI2D2 = "ASI2D2"
    ASI2D3 = "ASI2D3"
    ASI3D3 = "ASI3D3"
    ASI3D4 = "ASI3D4"
    ASI3D6 = "ASI3D6"
    ASI3D8 = "ASI3D8"
    ASIAX2 = "ASIAX2"
    ASIAX3 = "ASIAX3"
    ASPECT_RATIO = "ASPECT_RATIO"
    ASSEMBLY = "ASSEMBLY"
    ASSEMBLY_LOAD = "ASSEMBLY_LOAD"
    ASSEMBLY_MAP = "ASSEMBLY_MAP"
    ASSEMBLY_MAP_COLORS = "ASSEMBLY_MAP_COLORS"
    ASSEMBLY_SET = "ASSEMBLY_SET"
    AS_DISPLAYED = "AS_DISPLAYED"
    AS_IS = "AS_IS"
    ATTACH_TO_POINT = "ATTACH_TO_POINT"
    ATTACH_TO_REGION = "ATTACH_TO_REGION"
    AT_BEGINNING = "AT_BEGINNING"
    AUGMENTED_LAGRANGE = "AUGMENTED_LAGRANGE"
    AUGMENTED_LAGRANGE_INCOMPATIBILITIES = "AUGMENTED_LAGRANGE_INCOMPATIBILITIES"
    AUTO = "AUTO"
    AUTOCAD = "AUTOCAD"
    AUTOCOMPUTE = "AUTOCOMPUTE"
    AUTOMATIC = "AUTOMATIC"
    AUTOMATIC_EBE = "AUTOMATIC_EBE"
    AUTOMATIC_GLOBAL = "AUTOMATIC_GLOBAL"
    AUTO_ALIGN = "AUTO_ALIGN"
    AUTO_FIT = "AUTO_FIT"
    AUTO_FIT_PTS = "AUTO_FIT_PTS"
    AUTO_INCREMENTATION = "AUTO_INCREMENTATION"
    AUTO_TIGHT = "AUTO_TIGHT"
    AVERAGE = "AVERAGE"
    AVERAGE_EDGE_LENGTH = "AVERAGE_EDGE_LENGTH"
    AVERAGE_SIZE = "AVERAGE_SIZE"
    AVERAGE_STRAIN = "AVERAGE_STRAIN"
    AVERAGE_TRANSLATION = "AVERAGE_TRANSLATION"
    AVERAGING_REGION_MAP = "AVERAGING_REGION_MAP"
    AVERAGING_REGION_MAP_COLORS = "AVERAGING_REGION_MAP_COLORS"
    AVI = "AVI"
    AXIAL = "AXIAL"
    AXIAL_FORCE_CONSTRAINT = "AXIAL_FORCE_CONSTRAINT"
    AXISYM = "AXISYM"
    AXISYMMETRIC = "AXISYMMETRIC"
    AXIS_1 = "AXIS_1"
    AXIS_2 = "AXIS_2"
    AXIS_3 = "AXIS_3"
    B21 = "B21"
    B21H = "B21H"
    B22 = "B22"
    B22H = "B22H"
    B23 = "B23"
    B23H = "B23H"
    B31 = "B31"
    B31H = "B31H"
    B31OS = "B31OS"
    B31OSH = "B31OSH"
    B32 = "B32"
    B32H = "B32H"
    B32OS = "B32OS"
    B32OSH = "B32OSH"
    B33 = "B33"
    B33H = "B33H"
    BACKWARD_EULER = "BACKWARD_EULER"
    BALANCED = "BALANCED"
    BANDED = "BANDED"
    BAR = "BAR"
    BASE = "BASE"
    BATCHPRE_PHASE = "BATCHPRE_PHASE"
    BCGS = "BCGS"
    BC_MAP = "BC_MAP"
    BC_MAP_COLORS = "BC_MAP_COLORS"
    BEAM = "BEAM"
    BEAM_MPC = "BEAM_MPC"
    BEFORE = "BEFORE"
    BEFORE_ANALYSIS = "BEFORE_ANALYSIS"
    BELOW_MIN = "BELOW_MIN"
    BENDING = "BENDING"
    BEST_FIT = "BEST_FIT"
    BETWEEN_CAVITIES = "BETWEEN_CAVITIES"
    BIASED = "BIASED"
    BIAS_MAX_SIZE = "BIAS_MAX_SIZE"
    BIAS_METHOD = "BIAS_METHOD"
    BIAS_MIN_SIZE = "BIAS_MIN_SIZE"
    BIAS_RATIO = "BIAS_RATIO"
    BIAXIAL = "BIAXIAL"
    BIDIRECTIONAL = "BIDIRECTIONAL"
    BILINEAR = "BILINEAR"
    BIMOMENT = "BIMOMENT"
    BK = "BK"
    BLACK_AND_WHITE = "BLACK_AND_WHITE"
    BLACK_TO_WHITE = "BLACK_TO_WHITE"
    BLEND = "BLEND"
    BLOCKING_ALL = "BLOCKING_ALL"
    BLUE_TO_RED = "BLUE_TO_RED"
    BOLT = "BOLT"
    BOOLEAN = "BOOLEAN"
    # BOTH = 'BOTH'
    BOTHSIDES = "BOTHSIDES"
    BOTH_SIDES = "BOTH_SIDES"
    BOTH_SURFACES = "BOTH_SURFACES"
    BOTTOM = "BOTTOM"
    BOTTOM_CENTER = "BOTTOM_CENTER"
    BOTTOM_LEFT = "BOTTOM_LEFT"
    BOTTOM_RIGHT = "BOTTOM_RIGHT"
    BOTTOM_SURFACE = "BOTTOM_SURFACE"
    BOTTOM_UP = "BOTTOM_UP"
    BOUNDARY_CONDITION = "BOUNDARY_CONDITION"
    BOUNDARY_ONLY = "BOUNDARY_ONLY"
    BOX_OFF = "BOX_OFF"
    BOX_ON = "BOX_ON"
    BUCKLE = "BUCKLE"
    BUCKLING_MODES = "BUCKLING_MODES"
    BUILT_INTO_BASE_STATE = "BUILT_INTO_BASE_STATE"
    BUILT_INTO_MODES = "BUILT_INTO_MODES"
    BULK_VISCOSITY = "BULK_VISCOSITY"
    BUSHING = "BUSHING"
    BY_NUMBER = "BY_NUMBER"
    BY_SPACING = "BY_SPACING"
    C3D10 = "C3D10"
    C3D10E = "C3D10E"
    C3D10H = "C3D10H"
    C3D10I = "C3D10I"
    C3D10M = "C3D10M"
    C3D10MH = "C3D10MH"
    C3D10MHT = "C3D10MHT"
    C3D10MP = "C3D10MP"
    C3D10MPH = "C3D10MPH"
    C3D10MPT = "C3D10MPT"
    C3D10MT = "C3D10MT"
    C3D15 = "C3D15"
    C3D15E = "C3D15E"
    C3D15H = "C3D15H"
    C3D20 = "C3D20"
    C3D20E = "C3D20E"
    C3D20H = "C3D20H"
    C3D20HT = "C3D20HT"
    C3D20P = "C3D20P"
    C3D20PH = "C3D20PH"
    C3D20R = "C3D20R"
    C3D20RE = "C3D20RE"
    C3D20RH = "C3D20RH"
    C3D20RHT = "C3D20RHT"
    C3D20RP = "C3D20RP"
    C3D20RPH = "C3D20RPH"
    C3D20RT = "C3D20RT"
    C3D20T = "C3D20T"
    C3D4 = "C3D4"
    C3D4E = "C3D4E"
    C3D4H = "C3D4H"
    C3D4P = "C3D4P"
    C3D4T = "C3D4T"
    C3D6 = "C3D6"
    C3D6E = "C3D6E"
    C3D6H = "C3D6H"
    C3D6P = "C3D6P"
    C3D6T = "C3D6T"
    C3D8 = "C3D8"
    C3D8E = "C3D8E"
    C3D8H = "C3D8H"
    C3D8HT = "C3D8HT"
    C3D8I = "C3D8I"
    C3D8IH = "C3D8IH"
    C3D8P = "C3D8P"
    C3D8PH = "C3D8PH"
    C3D8PHT = "C3D8PHT"
    C3D8PT = "C3D8PT"
    C3D8R = "C3D8R"
    C3D8RH = "C3D8RH"
    C3D8RHT = "C3D8RHT"
    C3D8RP = "C3D8RP"
    C3D8RPH = "C3D8RPH"
    C3D8RPHT = "C3D8RPHT"
    C3D8RPT = "C3D8RPT"
    C3D8RT = "C3D8RT"
    C3D8T = "C3D8T"
    CALCULATE = "CALCULATE"
    CALCULATOR_PHASE = "CALCULATOR_PHASE"
    CAMERA = "CAMERA"
    CARDAN = "CARDAN"
    CARMAN_KOZENY = "CARMAN_KOZENY"
    CARTESIAN = "CARTESIAN"
    CATEGORY_BASED = "CATEGORY_BASED"
    CAVITY = "CAVITY"
    CAVITY_RADIATION = "CAVITY_RADIATION"
    CAVITY_RADIATION_EMMISIVITY_TOO_LARGE = "CAVITY_RADIATION_EMMISIVITY_TOO_LARGE"
    CAVITY_RADIATION_PARALLEL_DECOMPOSITION = "CAVITY_RADIATION_PARALLEL_DECOMPOSITION"
    CAX3 = "CAX3"
    CAX3E = "CAX3E"
    CAX3H = "CAX3H"
    CAX3T = "CAX3T"
    CAX4 = "CAX4"
    CAX4E = "CAX4E"
    CAX4H = "CAX4H"
    CAX4HT = "CAX4HT"
    CAX4I = "CAX4I"
    CAX4IH = "CAX4IH"
    CAX4P = "CAX4P"
    CAX4PH = "CAX4PH"
    CAX4R = "CAX4R"
    CAX4RH = "CAX4RH"
    CAX4RHT = "CAX4RHT"
    CAX4RP = "CAX4RP"
    CAX4RPH = "CAX4RPH"
    CAX4RT = "CAX4RT"
    CAX4T = "CAX4T"
    CAX6 = "CAX6"
    CAX6E = "CAX6E"
    CAX6H = "CAX6H"
    CAX6M = "CAX6M"
    CAX6MH = "CAX6MH"
    CAX6MHT = "CAX6MHT"
    CAX6MP = "CAX6MP"
    CAX6MPH = "CAX6MPH"
    CAX6MT = "CAX6MT"
    CAX8 = "CAX8"
    CAX8E = "CAX8E"
    CAX8H = "CAX8H"
    CAX8HT = "CAX8HT"
    CAX8P = "CAX8P"
    CAX8PH = "CAX8PH"
    CAX8R = "CAX8R"
    CAX8RE = "CAX8RE"
    CAX8RH = "CAX8RH"
    CAX8RHT = "CAX8RHT"
    CAX8RP = "CAX8RP"
    CAX8RPH = "CAX8RPH"
    CAX8RT = "CAX8RT"
    CAX8T = "CAX8T"
    CAXA41 = "CAXA41"
    CAXA42 = "CAXA42"
    CAXA43 = "CAXA43"
    CAXA44 = "CAXA44"
    CAXA4H1 = "CAXA4H1"
    CAXA4H2 = "CAXA4H2"
    CAXA4H3 = "CAXA4H3"
    CAXA4H4 = "CAXA4H4"
    CAXA4R1 = "CAXA4R1"
    CAXA4R2 = "CAXA4R2"
    CAXA4R3 = "CAXA4R3"
    CAXA4R4 = "CAXA4R4"
    CAXA4RH1 = "CAXA4RH1"
    CAXA4RH2 = "CAXA4RH2"
    CAXA4RH3 = "CAXA4RH3"
    CAXA4RH4 = "CAXA4RH4"
    CAXA81 = "CAXA81"
    CAXA82 = "CAXA82"
    CAXA83 = "CAXA83"
    CAXA84 = "CAXA84"
    CAXA8H1 = "CAXA8H1"
    CAXA8H2 = "CAXA8H2"
    CAXA8H3 = "CAXA8H3"
    CAXA8H4 = "CAXA8H4"
    CAXA8P1 = "CAXA8P1"
    CAXA8P2 = "CAXA8P2"
    CAXA8P3 = "CAXA8P3"
    CAXA8P4 = "CAXA8P4"
    CAXA8R1 = "CAXA8R1"
    CAXA8R2 = "CAXA8R2"
    CAXA8R3 = "CAXA8R3"
    CAXA8R4 = "CAXA8R4"
    CAXA8RH1 = "CAXA8RH1"
    CAXA8RH2 = "CAXA8RH2"
    CAXA8RH3 = "CAXA8RH3"
    CAXA8RH4 = "CAXA8RH4"
    CAXA8RP1 = "CAXA8RP1"
    CAXA8RP2 = "CAXA8RP2"
    CAXA8RP3 = "CAXA8RP3"
    CAXA8RP4 = "CAXA8RP4"
    CCL12 = "CCL12"
    CCL12H = "CCL12H"
    CCL18 = "CCL18"
    CCL18H = "CCL18H"
    CCL24 = "CCL24"
    CCL24H = "CCL24H"
    CCL24R = "CCL24R"
    CCL24RH = "CCL24RH"
    CCL9 = "CCL9"
    CCL9H = "CCL9H"
    CCW = "CCW"
    CENTER = "CENTER"
    CENTER_LEFT = "CENTER_LEFT"
    CENTER_OF_MASS = "CENTER_OF_MASS"
    CENTER_RIGHT = "CENTER_RIGHT"
    CENTROID = "CENTROID"
    CFD = "CFD"
    CFD_ANALYSIS = "CFD_ANALYSIS"
    CFD_PHASE = "CFD_PHASE"
    CG = "CG"
    CGAX3 = "CGAX3"
    CGAX3H = "CGAX3H"
    CGAX3HT = "CGAX3HT"
    CGAX3T = "CGAX3T"
    CGAX4 = "CGAX4"
    CGAX4H = "CGAX4H"
    CGAX4HT = "CGAX4HT"
    CGAX4R = "CGAX4R"
    CGAX4RH = "CGAX4RH"
    CGAX4RHT = "CGAX4RHT"
    CGAX4RT = "CGAX4RT"
    CGAX4T = "CGAX4T"
    CGAX6 = "CGAX6"
    CGAX6H = "CGAX6H"
    CGAX6M = "CGAX6M"
    CGAX6MH = "CGAX6MH"
    CGAX6MHT = "CGAX6MHT"
    CGAX6MT = "CGAX6MT"
    CGAX8 = "CGAX8"
    CGAX8H = "CGAX8H"
    CGAX8HT = "CGAX8HT"
    CGAX8R = "CGAX8R"
    CGAX8RH = "CGAX8RH"
    CGAX8RHT = "CGAX8RHT"
    CGAX8RT = "CGAX8RT"
    CGAX8T = "CGAX8T"
    CGPE10 = "CGPE10"
    CGPE10H = "CGPE10H"
    CGPE10R = "CGPE10R"
    CGPE10RH = "CGPE10RH"
    CGPE5 = "CGPE5"
    CGPE5H = "CGPE5H"
    CGPE6 = "CGPE6"
    CGPE6H = "CGPE6H"
    CGPE6I = "CGPE6I"
    CGPE6IH = "CGPE6IH"
    CGPE6MH = "CGPE6MH"
    CGPE6R = "CGPE6R"
    CGPE6RH = "CGPE6RH"
    CGPE8 = "CGPE8"
    CGPE8H = "CGPE8H"
    CHEBYCHEV = "CHEBYCHEV"
    CHECKER = "CHECKER"
    CHECK_COMPLETED = "CHECK_COMPLETED"
    CHECK_RUNNING = "CHECK_RUNNING"
    CHECK_SUBMITTED = "CHECK_SUBMITTED"
    CINAX4 = "CINAX4"
    CINAX5R = "CINAX5R"
    CINPE4 = "CINPE4"
    CINPE5R = "CINPE5R"
    CINPS4 = "CINPS4"
    CINPS5R = "CINPS5R"
    CIRCLE = "CIRCLE"
    CIRCLE_RADIUS = "CIRCLE_RADIUS"
    CIRCULAR = "CIRCULAR"
    CIRCUM = "CIRCUM"
    CIRCUMF = "CIRCUMF"
    CIRCUMFERENTIAL = "CIRCUMFERENTIAL"
    CIRC_MIRROR_RECT = "CIRC_MIRROR_RECT"
    CIRC_RECT_MIRROR = "CIRC_RECT_MIRROR"
    CLEARANCE = "CLEARANCE"
    CLOCKWISE = "CLOCKWISE"
    CLOSEST = "CLOSEST"
    CLOSEST_POINT_FRACTION = "CLOSEST_POINT_FRACTION"
    CLOSURE_VALUE = "CLOSURE_VALUE"
    COARSE = "COARSE"
    COD = "COD"
    CODEC = "CODEC"
    COEFFICIENT = "COEFFICIENT"
    COEFFICIENTS = "COEFFICIENTS"
    COH2D4 = "COH2D4"
    COH2D4P = "COH2D4P"
    COH3D6 = "COH3D6"
    COH3D6P = "COH3D6P"
    COH3D8 = "COH3D8"
    COH3D8P = "COH3D8P"
    COHAX4 = "COHAX4"
    COHAX4P = "COHAX4P"
    COLOR = "COLOR"
    COMBINED = "COMBINED"
    COMMA_SEPARATED_VALUES = "COMMA_SEPARATED_VALUES"
    COMPLETED = "COMPLETED"
    COMPLEX = "COMPLEX"
    COMPLEX_EIGENSOLVER = "COMPLEX_EIGENSOLVER"
    COMPLEX_FREQUENCY = "COMPLEX_FREQUENCY"
    COMPLEX_MAGNITUDE = "COMPLEX_MAGNITUDE"
    COMPLEX_PHASE = "COMPLEX_PHASE"
    COMPLEX_VAL_AT_ANGLE = "COMPLEX_VAL_AT_ANGLE"
    COMPONENT = "COMPONENT"
    COMPONENT_NUMBER = "COMPONENT_NUMBER"
    COMPRESSEDINDEX = "COMPRESSEDINDEX"
    COMPRESSED_VRML = "COMPRESSED_VRML"
    COMPRESSIBLE = "COMPRESSIBLE"
    COMPRESSION = "COMPRESSION"
    COMPUTE = "COMPUTE"
    COMPUTED = "COMPUTED"
    COMPUTED_TOLERANCE = "COMPUTED_TOLERANCE"
    COMP_DEFAULT = "COMP_DEFAULT"
    CONCENTRIC = "CONCENTRIC"
    CONDITION_BASED_OPTIMIZATION = "CONDITION_BASED_OPTIMIZATION"
    CONN2D2 = "CONN2D2"
    CONN3D2 = "CONN3D2"
    CONNECTOR = "CONNECTOR"
    CONNECTOR_MAP = "CONNECTOR_MAP"
    CONNECTOR_MAP_COLORS = "CONNECTOR_MAP_COLORS"
    CONNECTOR_PROP_MAP = "CONNECTOR_PROP_MAP"
    CONNECTOR_PROP_MAP_COLORS = "CONNECTOR_PROP_MAP_COLORS"
    CONNECTOR_TYPE_MAP = "CONNECTOR_TYPE_MAP"
    CONNECTOR_TYPE_MAP_COLORS = "CONNECTOR_TYPE_MAP_COLORS"
    CONSERVATIVE = "CONSERVATIVE"
    CONSOLIDATION = "CONSOLIDATION"
    CONSTANT = "CONSTANT"
    CONSTANTPRESSURE = "CONSTANTPRESSURE"
    CONSTANTS = "CONSTANTS"
    CONSTANTVOLUME = "CONSTANTVOLUME"
    CONSTANT_RATIO = "CONSTANT_RATIO"
    CONSTANT_THROUGH_THICKNESS = "CONSTANT_THROUGH_THICKNESS"
    CONSTANT_VELOCITY = "CONSTANT_VELOCITY"
    CONSTRAINED_LAPLACIAN = "CONSTRAINED_LAPLACIAN"
    CONSTRAINT = "CONSTRAINT"
    CONSTRAINT_MAP = "CONSTRAINT_MAP"
    CONSTRAINT_MAP_COLORS = "CONSTRAINT_MAP_COLORS"
    CONSTRAINT_TYPE_MAP = "CONSTRAINT_TYPE_MAP"
    CONSTRAINT_TYPE_MAP_COLORS = "CONSTRAINT_TYPE_MAP_COLORS"
    CONSTRUCTION = "CONSTRUCTION"
    CONTACT = "CONTACT"
    CONTACT_ALL = "CONTACT_ALL"
    CONTACT_EXPLICIT = "CONTACT_EXPLICIT"
    CONTACT_PRESELECT = "CONTACT_PRESELECT"
    CONTACT_STANDARD = "CONTACT_STANDARD"
    CONTINUE = "CONTINUE"
    CONTINUOUS = "CONTINUOUS"
    CONTINUUM = "CONTINUUM"
    CONTINUUM_SHELL = "CONTINUUM_SHELL"
    CONTOURS_ON_DEF = "CONTOURS_ON_DEF"
    CONTOURS_ON_UNDEF = "CONTOURS_ON_UNDEF"
    CONTROL_POINTS = "CONTROL_POINTS"
    CONVERGENCE = "CONVERGENCE"
    CONVERT_SDI_OFF = "CONVERT_SDI_OFF"
    CONVERT_SDI_ON = "CONVERT_SDI_ON"
    CONWEP = "CONWEP"
    COORDINATE = "COORDINATE"
    COPLANAR_EDGES = "COPLANAR_EDGES"
    CORIOLIS_LOAD = "CORIOLIS_LOAD"
    CORRECTION_ACCEPTED = "CORRECTION_ACCEPTED"
    CORRECTION_ACCEPTED_ESTIMATED_CORRECTION = "CORRECTION_ACCEPTED_ESTIMATED_CORRECTION"
    CORRECTION_ACCEPTED_SMALL_INCREMENT = "CORRECTION_ACCEPTED_SMALL_INCREMENT"
    CORRECTION_NOT_ACCEPTED = "CORRECTION_NOT_ACCEPTED"
    CORRELATED = "CORRELATED"
    COSINE = "COSINE"
    COUNT = "COUNT"
    COUNTERCLOCKWISE = "COUNTERCLOCKWISE"
    COUPLED = "COUPLED"
    COUPLED_MOTION = "COUPLED_MOTION"
    COUPLED_POSITION = "COUPLED_POSITION"
    COUPLED_TEMP_DISPLACEMENT = "COUPLED_TEMP_DISPLACEMENT"
    COUPLED_THERMAL_ELECTRIC = "COUPLED_THERMAL_ELECTRIC"
    COUPLED_THERMAL_ELECTRICAL = "COUPLED_THERMAL_ELECTRICAL"
    COUPLED_THERMAL_ELECTRICAL_STRUCTURAL = "COUPLED_THERMAL_ELECTRICAL_STRUCTURAL"
    COUPLED_THERMAL_STRESS = "COUPLED_THERMAL_STRESS"
    COUPLED_TRACTION = "COUPLED_TRACTION"
    COUPLING_FORCE_CONSTRAINT = "COUPLING_FORCE_CONSTRAINT"
    COUPLING_MOMENT_CONSTRAINT = "COUPLING_MOMENT_CONSTRAINT"
    CPE3 = "CPE3"
    CPE3E = "CPE3E"
    CPE3H = "CPE3H"
    CPE3T = "CPE3T"
    CPE4 = "CPE4"
    CPE4E = "CPE4E"
    CPE4H = "CPE4H"
    CPE4HT = "CPE4HT"
    CPE4I = "CPE4I"
    CPE4IH = "CPE4IH"
    CPE4P = "CPE4P"
    CPE4PH = "CPE4PH"
    CPE4R = "CPE4R"
    CPE4RH = "CPE4RH"
    CPE4RHT = "CPE4RHT"
    CPE4RP = "CPE4RP"
    CPE4RPH = "CPE4RPH"
    CPE4RT = "CPE4RT"
    CPE4T = "CPE4T"
    CPE6 = "CPE6"
    CPE6E = "CPE6E"
    CPE6H = "CPE6H"
    CPE6M = "CPE6M"
    CPE6MH = "CPE6MH"
    CPE6MHT = "CPE6MHT"
    CPE6MP = "CPE6MP"
    CPE6MPH = "CPE6MPH"
    CPE6MT = "CPE6MT"
    CPE8 = "CPE8"
    CPE8E = "CPE8E"
    CPE8H = "CPE8H"
    CPE8HT = "CPE8HT"
    CPE8P = "CPE8P"
    CPE8PH = "CPE8PH"
    CPE8R = "CPE8R"
    CPE8RE = "CPE8RE"
    CPE8RH = "CPE8RH"
    CPE8RHT = "CPE8RHT"
    CPE8RP = "CPE8RP"
    CPE8RPH = "CPE8RPH"
    CPE8RT = "CPE8RT"
    CPE8T = "CPE8T"
    CPEG3 = "CPEG3"
    CPEG3H = "CPEG3H"
    CPEG3HT = "CPEG3HT"
    CPEG3T = "CPEG3T"
    CPEG4 = "CPEG4"
    CPEG4H = "CPEG4H"
    CPEG4HT = "CPEG4HT"
    CPEG4I = "CPEG4I"
    CPEG4IH = "CPEG4IH"
    CPEG4R = "CPEG4R"
    CPEG4RH = "CPEG4RH"
    CPEG4RT = "CPEG4RT"
    CPEG4RHT = "CPEG4RHT"
    CPEG4T = "CPEG4T"
    CPEG6 = "CPEG6"
    CPEG6H = "CPEG6H"
    CPEG6M = "CPEG6M"
    CPEG6MH = "CPEG6MH"
    CPEG6MHT = "CPEG6MHT"
    CPEG6MT = "CPEG6MT"
    CPEG8 = "CPEG8"
    CPEG8H = "CPEG8H"
    CPEG8HT = "CPEG8HT"
    CPEG8R = "CPEG8R"
    CPEG8RH = "CPEG8RH"
    CPEG8RT = "CPEG8RT"
    CPEG8RHT = "CPEG8RHT"
    CPEG8T = "CPEG8T"
    CPS3 = "CPS3"
    CPS3E = "CPS3E"
    CPS3T = "CPS3T"
    CPS4 = "CPS4"
    CPS4E = "CPS4E"
    CPS4I = "CPS4I"
    CPS4R = "CPS4R"
    CPS4RT = "CPS4RT"
    CPS4T = "CPS4T"
    CPS6 = "CPS6"
    CPS6E = "CPS6E"
    CPS6M = "CPS6M"
    CPS6MT = "CPS6MT"
    CPS8 = "CPS8"
    CPS8E = "CPS8E"
    CPS8R = "CPS8R"
    CPS8RE = "CPS8RE"
    CPS8RT = "CPS8RT"
    CPS8T = "CPS8T"
    CPU_TIME_XPL = "CPU_TIME_XPL"
    CQC = "CQC"
    CRACKTIP = "CRACKTIP"
    CRACK_GROWTH = "CRACK_GROWTH"
    CRACK_LENGTH = "CRACK_LENGTH"
    CRACK_NORMAL = "CRACK_NORMAL"
    CREATED = "CREATED"
    CREEP = "CREEP"
    CREEP_OFF = "CREEP_OFF"
    CREEP_TEST_DATA = "CREEP_TEST_DATA"
    CRITICAL_DAMPING_FRACTION = "CRITICAL_DAMPING_FRACTION"
    CRITICAL_STRESS = "CRITICAL_STRESS"
    CROSS = "CROSS"
    CROSSED_SURFACES = "CROSSED_SURFACES"
    CROSSING_VALUE = "CROSSING_VALUE"
    CRUSHABLE_FOAM = "CRUSHABLE_FOAM"
    CSYS = "CSYS"
    CUBIC = "CUBIC"
    CUBIC_SPLINE = "CUBIC_SPLINE"
    CURRENT = "CURRENT"
    CURRENT_DISPLAY_GROUP = "CURRENT_DISPLAY_GROUP"
    CURRENT_FRAME = "CURRENT_FRAME"
    CURVATURE = "CURVATURE"
    CURVATURE_BASED_BY_SIZE = "CURVATURE_BASED_BY_SIZE"
    CURVE_LEGEND = "CURVE_LEGEND"
    CURVE_NAME = "CURVE_NAME"
    CURVE_NAME_LEGEND = "CURVE_NAME_LEGEND"
    CUSTOM = "CUSTOM"
    CUT = "CUT"
    CUT_OFF = "CUT_OFF"
    CVJOINT = "CVJOINT"
    CW = "CW"
    CYCLIC_SYMMETRY = "CYCLIC_SYMMETRY"
    CYLINDER = "CYLINDER"
    CYLINDRICAL = "CYLINDRICAL"
    C_INTEGRAL = "C_INTEGRAL"
    DAMAGE = "DAMAGE"
    DAMAGE_CRITERION = "DAMAGE_CRITERION"
    DAMPING_COEFFICIENT = "DAMPING_COEFFICIENT"
    DAMPING_FACTOR = "DAMPING_FACTOR"
    DASHED = "DASHED"
    DASHPOT1 = "DASHPOT1"
    DASHPOT2 = "DASHPOT2"
    DASHPOTA = "DASHPOTA"
    DATACHECK = "DATACHECK"
    DATUM = "DATUM"
    DB = "DB"
    DB2 = "DB2"
    DC1D2 = "DC1D2"
    DC1D2E = "DC1D2E"
    DC1D3 = "DC1D3"
    DC1D3E = "DC1D3E"
    DC2D3 = "DC2D3"
    DC2D3E = "DC2D3E"
    DC2D4 = "DC2D4"
    DC2D4E = "DC2D4E"
    DC2D6 = "DC2D6"
    DC2D6E = "DC2D6E"
    DC2D8 = "DC2D8"
    DC2D8E = "DC2D8E"
    DC3D10 = "DC3D10"
    DC3D10E = "DC3D10E"
    DC3D15 = "DC3D15"
    DC3D15E = "DC3D15E"
    DC3D20 = "DC3D20"
    DC3D20E = "DC3D20E"
    DC3D4 = "DC3D4"
    DC3D4E = "DC3D4E"
    DC3D6 = "DC3D6"
    DC3D6E = "DC3D6E"
    DC3D8 = "DC3D8"
    DC3D8E = "DC3D8E"
    DCAX3 = "DCAX3"
    DCAX3E = "DCAX3E"
    DCAX4 = "DCAX4"
    DCAX4E = "DCAX4E"
    DCAX6 = "DCAX6"
    DCAX6E = "DCAX6E"
    DCAX8 = "DCAX8"
    DCAX8E = "DCAX8E"
    DCC1D2 = "DCC1D2"
    DCC1D2D = "DCC1D2D"
    DCC2D4 = "DCC2D4"
    DCC2D4D = "DCC2D4D"
    DCC3D8 = "DCC3D8"
    DCC3D8D = "DCC3D8D"
    DCCAX2 = "DCCAX2"
    DCCAX2D = "DCCAX2D"
    DCCAX4 = "DCCAX4"
    DCCAX4D = "DCCAX4D"
    DCOUP2D = "DCOUP2D"
    DCOUP3D = "DCOUP3D"
    DDM_ITERATIVE = "DDM_ITERATIVE"
    DEACTIVATED = "DEACTIVATED"
    DEACTIVATED_FROM_BASE_STATE = "DEACTIVATED_FROM_BASE_STATE"
    DEBONDING = "DEBONDING"
    DECIMAL = "DECIMAL"
    DEFAULT = "DEFAULT"
    DEFAULTFORMAT = "DEFAULTFORMAT"
    DEFAULT_COLORS = "DEFAULT_COLORS"
    DEFAULT_LIMIT = "DEFAULT_LIMIT"
    DEFAULT_MODEL = "DEFAULT_MODEL"
    DEFAULT_SIZE = "DEFAULT_SIZE"
    DEFINE = "DEFINE"
    DEFORMABLE_BODY = "DEFORMABLE_BODY"
    DEFORMED = "DEFORMED"
    DELAYED = "DELAYED"
    DELETE = "DELETE"
    DELTA_OVER_1_ITERATION = "DELTA_OVER_1_ITERATION"
    DELTA_OVER_2_ITERATIONS = "DELTA_OVER_2_ITERATIONS"
    DELTA_OVER_3_ITERATIONS = "DELTA_OVER_3_ITERATIONS"
    DELTA_OVER_4_ITERATIONS = "DELTA_OVER_4_ITERATIONS"
    DELTA_OVER_5_ITERATIONS = "DELTA_OVER_5_ITERATIONS"
    DELTA_OVER_6_ITERATIONS = "DELTA_OVER_6_ITERATIONS"
    DEMOLD_REGION = "DEMOLD_REGION"
    DENSITY = "DENSITY"
    DENSITY_ROTATIONAL_ACCELERATION = "DENSITY_ROTATIONAL_ACCELERATION"
    DERIVED_COMPONENT = "DERIVED_COMPONENT"
    DESCENDING = "DESCENDING"
    DESIGN_SENSITIVITY = "DESIGN_SENSITIVITY"
    DEVIATION_FACTOR = "DEVIATION_FACTOR"
    DGAP = "DGAP"
    DIFFERENCE = "DIFFERENCE"
    DIFFUSE = "DIFFUSE"
    DIRECT = "DIRECT"
    DIRECTION = "DIRECTION"
    DIRECTIONAL = "DIRECTIONAL"
    DIRECTION_COSINE = "DIRECTION_COSINE"
    DIRECT_COMPONENT = "DIRECT_COMPONENT"
    DIRECT_CYCLIC = "DIRECT_CYCLIC"
    DIRECT_INCREMENTATION = "DIRECT_INCREMENTATION"
    DIRECT_NO_STOP_INCREMENTATION = "DIRECT_NO_STOP_INCREMENTATION"
    DIRECT_SOLVER = "DIRECT_SOLVER"
    DIRECT_STEADY_STATE_DYNAMIC = "DIRECT_STEADY_STATE_DYNAMIC"
    DIRECT_SYMMETRIC = "DIRECT_SYMMETRIC"
    DIRECT_UNSYMMETRIC = "DIRECT_UNSYMMETRIC"
    DISABLED_BY_SYSTEM = "DISABLED_BY_SYSTEM"
    DISABLED_BY_USER = "DISABLED_BY_USER"
    DISABLE_THROUGHOUT_STEP = "DISABLE_THROUGHOUT_STEP"
    DISCONTINUITIES = "DISCONTINUITIES"
    DISCRETE = "DISCRETE"
    DISCRETE_FIELD = "DISCRETE_FIELD"
    DISCRETE_RIGID_SURFACE = "DISCRETE_RIGID_SURFACE"
    DISK = "DISK"
    DISPLACEMENT = "DISPLACEMENT"
    DISPLACEMENT_FIELD = "DISPLACEMENT_FIELD"
    DISPLAY_GROUPS = "DISPLAY_GROUPS"
    DISPLAY_GRP_MAP = "DISPLAY_GRP_MAP"
    DISPLAY_GRP_MAP_COLORS = "DISPLAY_GRP_MAP_COLORS"
    DISSIPATED_ENERGY_FRACTION = "DISSIPATED_ENERGY_FRACTION"
    DISTRIBUTING = "DISTRIBUTING"
    DISTRIBUTING_COUPLING = "DISTRIBUTING_COUPLING"
    DIVIDE = "DIVIDE"
    DMASS_XPL = "DMASS_XPL"
    DOF_MODE = "DOF_MODE"
    DOF_MODE_MPC = "DOF_MODE_MPC"
    DOMAIN = "DOMAIN"
    DOTTED = "DOTTED"
    DOT_DASH = "DOT_DASH"
    DOUBLE = "DOUBLE"
    DOUBLE_CONSTRAINT_ONLY = "DOUBLE_CONSTRAINT_ONLY"
    DOUBLE_PLUS_PACK = "DOUBLE_PLUS_PACK"
    DOUBLE_PRECISION = "DOUBLE_PRECISION"
    DOUBLE_SIDED = "DOUBLE_SIDED"
    DPI_1200 = "DPI_1200"
    DPI_150 = "DPI_150"
    DPI_300 = "DPI_300"
    DPI_450 = "DPI_450"
    DPI_600 = "DPI_600"
    DPI_75 = "DPI_75"
    DRAG2D = "DRAG2D"
    DRAG3D = "DRAG3D"
    DS3 = "DS3"
    DS4 = "DS4"
    DS6 = "DS6"
    DS8 = "DS8"
    DSAX1 = "DSAX1"
    DSAX2 = "DSAX2"
    DSC = "DSC"
    DUPLICATE_NODES = "DUPLICATE_NODES"
    DURING_ANALYSIS = "DURING_ANALYSIS"
    DYNAMIC = "DYNAMIC"
    DYNAMIC_EXPLICIT = "DYNAMIC_EXPLICIT"
    DYNAMIC_IMPLICIT = "DYNAMIC_IMPLICIT"
    DYNAMIC_SUBSPACE = "DYNAMIC_SUBSPACE"
    DYNAMIC_TEMP_DISPLACEMENT = "DYNAMIC_TEMP_DISPLACEMENT"
    EC3D8R = "EC3D8R"
    EC3D8RT = "EC3D8RT"
    ECURRENT_AREA_TIME = "ECURRENT_AREA_TIME"
    EDGE = "EDGE"
    EDGE1 = "EDGE1"
    EDGE2 = "EDGE2"
    EDGE3 = "EDGE3"
    EDGE4 = "EDGE4"
    EDGES = "EDGES"
    EDGETOEDGE = "EDGETOEDGE"
    EDGETOFACE = "EDGETOFACE"
    EDGE_LIST = "EDGE_LIST"
    EDGE_SEEDING_METHOD = "EDGE_SEEDING_METHOD"
    EIGENFREQUENCY = "EIGENFREQUENCY"
    EIGENVALUE_BUCKLING = "EIGENVALUE_BUCKLING"
    EITHER = "EITHER"
    ELASTIC = "ELASTIC"
    ELASTIC_PLASTIC = "ELASTIC_PLASTIC"
    ELBOW31 = "ELBOW31"
    ELBOW31B = "ELBOW31B"
    ELBOW31C = "ELBOW31C"
    ELBOW32 = "ELBOW32"
    ELBOW_MPC = "ELBOW_MPC"
    ELECTRICAL = "ELECTRICAL"
    ELECTRICAL_CONTACT = "ELECTRICAL_CONTACT"
    ELECTRICAL_POTENTIAL_FIELD = "ELECTRICAL_POTENTIAL_FIELD"
    ELECTRIC_CHARGE = "ELECTRIC_CHARGE"
    ELECTRIC_CURRENT = "ELECTRIC_CURRENT"
    ELECTRIC_CURRENT_AREA = "ELECTRIC_CURRENT_AREA"
    ELECTRIC_POTENTIAL = "ELECTRIC_POTENTIAL"
    ELECTROMAGNETIC_TIME_HARMONIC = "ELECTROMAGNETIC_TIME_HARMONIC"
    ELEMENT = "ELEMENT"
    ELEMENTS = "ELEMENTS"
    ELEMENT_ALL = "ELEMENT_ALL"
    ELEMENT_BY_ELEMENT_INCREMENTATION = "ELEMENT_BY_ELEMENT_INCREMENTATION"
    ELEMENT_CENTER_PROJECTION = "ELEMENT_CENTER_PROJECTION"
    ELEMENT_CENTROID = "ELEMENT_CENTROID"
    ELEMENT_FACE = "ELEMENT_FACE"
    ELEMENT_FACE_INTEGRATION_POINT = "ELEMENT_FACE_INTEGRATION_POINT"
    ELEMENT_NODAL = "ELEMENT_NODAL"
    ELEMENT_NODES = "ELEMENT_NODES"
    ELEMENT_PRESELECT = "ELEMENT_PRESELECT"
    ELEMENT_SET = "ELEMENT_SET"
    ELEM_SHAPE = "ELEM_SHAPE"
    ELLIPSE = "ELLIPSE"
    ELLIPTICAL = "ELLIPTICAL"
    ELSET_MAP = "ELSET_MAP"
    ELSET_MAP_COLORS = "ELSET_MAP_COLORS"
    ELTYPE_MAP = "ELTYPE_MAP"
    ELTYPE_MAP_COLORS = "ELTYPE_MAP_COLORS"
    EMAG = "EMAG"
    EMBEDDED_COEFF = "EMBEDDED_COEFF"
    EMBEDDED_ELEMENT = "EMBEDDED_ELEMENT"
    EMC2D3 = "EMC2D3"
    EMC2D4 = "EMC2D4"
    EMC3D4 = "EMC3D4"
    EMC3D6 = "EMC3D6"
    EMC3D8 = "EMC3D8"
    EMF = "EMF"
    EMPTY_FIELD = "EMPTY_FIELD"
    EMPTY_LEAF = "EMPTY_LEAF"
    ENABLED = "ENABLED"
    ENCASTRE = "ENCASTRE"
    END = "END"
    END1 = "END1"
    END2 = "END2"
    END3 = "END3"
    END_FRAME_TIME = "END_FRAME_TIME"
    END_RELEASE = "END_RELEASE"
    END_STEP = "END_STEP"
    ENERGY = "ENERGY"
    ENERGY_ALL = "ENERGY_ALL"
    ENERGY_DENSITY = "ENERGY_DENSITY"
    ENERGY_NONE = "ENERGY_NONE"
    ENERGY_PRESELECT = "ENERGY_PRESELECT"
    ENERGY_RELEASE_RATE = "ENERGY_RELEASE_RATE"
    ENERGY_TYPE = "ENERGY_TYPE"
    ENFORCEMENT_OFF = "ENFORCEMENT_OFF"
    ENFORCEMENT_ON = "ENFORCEMENT_ON"
    ENGINEERING = "ENGINEERING"
    ENGINEERING_CONSTANTS = "ENGINEERING_CONSTANTS"
    ENHANCED = "ENHANCED"
    ENHANCED_VCCT = "ENHANCED_VCCT"
    ENTEREDCOORD = "ENTEREDCOORD"
    ENUMERATION = "ENUMERATION"
    ENVELOPE = "ENVELOPE"
    EPOTENTIAL_GRADIENT = "EPOTENTIAL_GRADIENT"
    EPS = "EPS"
    EQUAL = "EQUAL"
    EQUALRADIUS = "EQUALRADIUS"
    EQUATION = "EQUATION"
    EQUILIBRIUM = "EQUILIBRIUM"
    EQUIV_STRESS = "EQUIV_STRESS"
    ERROR = "ERROR"
    ETOTAL_XPL = "ETOTAL_XPL"
    EULER = "EULER"
    EULERIAN = "EULERIAN"
    EVENT_ACCELERATION = "EVENT_ACCELERATION"
    EVENT_DISPLACEMENT = "EVENT_DISPLACEMENT"
    EVENT_GRAVITY = "EVENT_GRAVITY"
    EVENT_VELOCITY = "EVENT_VELOCITY"
    EVERY_CYCLE = "EVERY_CYCLE"
    EVERY_NCYCLES = "EVERY_NCYCLES"
    EVERY_TIME_INCREMENT = "EVERY_TIME_INCREMENT"
    EXACT = "EXACT"
    EXACT_TARGETS = "EXACT_TARGETS"
    EXCESSIVE_DISTORTION = "EXCESSIVE_DISTORTION"
    EXCESSIVE_ELEMENT_DISTORTION = "EXCESSIVE_ELEMENT_DISTORTION"
    EXCESSIVE_STRAIN_INCREMENT = "EXCESSIVE_STRAIN_INCREMENT"
    EXCLUDE = "EXCLUDE"
    EXPLICIT = "EXPLICIT"
    EXPLICIT_ANALYSIS = "EXPLICIT_ANALYSIS"
    EXPLICIT_DYNAMIC = "EXPLICIT_DYNAMIC"
    EXPLICIT_ONLY = "EXPLICIT_ONLY"
    EXPLICIT_PHASE = "EXPLICIT_PHASE"
    EXPONENTIAL = "EXPONENTIAL"
    EXPONENTIAL_DECAY = "EXPONENTIAL_DECAY"
    EXPONENTIAL_LAW = "EXPONENTIAL_LAW"
    EXPORT_STEP_SIZE = "EXPORT_STEP_SIZE"
    EXTERIOR = "EXTERIOR"
    EXTERNAL = "EXTERNAL"
    EXTRAPOLATE_AVERAGE_COMPUTE = "EXTRAPOLATE_AVERAGE_COMPUTE"
    EXTRAPOLATE_COMPUTE = "EXTRAPOLATE_COMPUTE"
    EXTRAPOLATE_COMPUTE_AVERAGE = "EXTRAPOLATE_COMPUTE_AVERAGE"
    EXTRAPOLATE_COMPUTE_DISCONTINUITIES = "EXTRAPOLATE_COMPUTE_DISCONTINUITIES"
    EXTRA_COARSE = "EXTRA_COARSE"
    EXTRA_FINE = "EXTRA_FINE"
    F2D2 = "F2D2"
    F3D3 = "F3D3"
    F3D4 = "F3D4"
    FACE1 = "FACE1"
    FACE2 = "FACE2"
    FACE3 = "FACE3"
    FACE4 = "FACE4"
    FACE5 = "FACE5"
    FACE6 = "FACE6"
    FACETOEDGE = "FACETOEDGE"
    FACETOFACE = "FACETOFACE"
    FACETS = "FACETS"
    FACE_CENTERED = "FACE_CENTERED"
    FACE_UNKNOWN = "FACE_UNKNOWN"
    FACTOR = "FACTOR"
    FARTHEST_POINT_FRACTION = "FARTHEST_POINT_FRACTION"
    FAST = "FAST"
    FASTENER = "FASTENER"
    FATIGUE = "FATIGUE"
    FAX2 = "FAX2"
    FC3D4 = "FC3D4"
    FC3D5 = "FC3D5"
    FC3D6 = "FC3D6"
    FC3D8 = "FC3D8"
    FEATURE = "FEATURE"
    FE_SAFE = "FE_SAFE"
    FGMRES = "FGMRES"
    FICK = "FICK"
    FIELD = "FIELD"
    FIELDREPORTFORMAT = "FIELDREPORTFORMAT"
    FIELD_OUTPUT = "FIELD_OUTPUT"
    FIELD_THICKNESS = "FIELD_THICKNESS"
    FILE = "FILE"
    FILL = "FILL"
    FILLED = "FILLED"
    FILLED_ARROW = "FILLED_ARROW"
    FILLED_CIRCLE = "FILLED_CIRCLE"
    FILLED_DIAMOND = "FILLED_DIAMOND"
    FILLED_SQUARE = "FILLED_SQUARE"
    FILLED_TRI = "FILLED_TRI"
    FILTER = "FILTER"
    FINE = "FINE"
    FINER = "FINER"
    FINITE = "FINITE"
    FIRST = "FIRST"
    FIRST_AND_LAST = "FIRST_AND_LAST"
    FIRST_CYCLE = "FIRST_CYCLE"
    FIRST_FRAME = "FIRST_FRAME"
    FIRST_ORDER_ADVECTION = "FIRST_ORDER_ADVECTION"
    FIRST_STEP = "FIRST_STEP"
    FITTED_VALUE = "FITTED_VALUE"
    FIT_HEIGHT = "FIT_HEIGHT"
    FIT_TO_PAGE = "FIT_TO_PAGE"
    FIT_TO_VIEWPORT = "FIT_TO_VIEWPORT"
    FIT_WIDTH = "FIT_WIDTH"
    FIXED = "FIXED"
    FIXED_CFL = "FIXED_CFL"
    FIXED_DOF = "FIXED_DOF"
    FIXED_EBE = "FIXED_EBE"
    FIXED_INCREMENTATION = "FIXED_INCREMENTATION"
    FIXED_TIME = "FIXED_TIME"
    FIXED_USER_DEFINED_INC = "FIXED_USER_DEFINED_INC"
    FIX_LENGTH = "FIX_LENGTH"
    FIX_NONE = "FIX_NONE"
    FLD = "FLD"
    FLEXIBLE = "FLEXIBLE"
    FLEXION_TORSION = "FLEXION_TORSION"
    FLINK = "FLINK"
    FLOAT = "FLOAT"
    FLOW = "FLOW"
    FLOW_CONVERTER = "FLOW_CONVERTER"
    FLUID = "FLUID"
    FLUID_PRESSURE_FIELD = "FLUID_PRESSURE_FIELD"
    FOLLOW = "FOLLOW"
    FONT = "FONT"
    FORCE = "FORCE"
    FORCE_SINGLE = "FORCE_SINGLE"
    FORCE_VOLUME = "FORCE_VOLUME"
    FORMULA = "FORMULA"
    FORWARD = "FORWARD"
    FRACTION = "FRACTION"
    FRACTIONAL = "FRACTIONAL"
    FRACTURE_MECHANICS = "FRACTURE_MECHANICS"
    FRAME2D = "FRAME2D"
    FRAME3D = "FRAME3D"
    FRAME_BASED = "FRAME_BASED"
    FRAME_VALUE = "FRAME_VALUE"
    FREE = "FREE"
    FREED = "FREED"
    FREE_FORM = "FREE_FORM"
    FREE_TASK_REGION_EQUIV_STRESS = "FREE_TASK_REGION_EQUIV_STRESS"
    FREQUENCY = "FREQUENCY"
    FREQUENCY_DATA = "FREQUENCY_DATA"
    FREQUENCY_RANGE = "FREQUENCY_RANGE"
    FRICTIONLESS = "FRICTIONLESS"
    FROM_ASCII_FILE = "FROM_ASCII_FILE"
    FROM_FILE = "FROM_FILE"
    FROM_FILE_AND_USER_DEFINED = "FROM_FILE_AND_USER_DEFINED"
    FROM_GEOMETRY = "FROM_GEOMETRY"
    FROM_KEYBOARD = "FROM_KEYBOARD"
    FROM_ODB = "FROM_ODB"
    FROM_OPERATION = "FROM_OPERATION"
    FROM_SECTION = "FROM_SECTION"
    FROM_USER_DEFINED = "FROM_USER_DEFINED"
    FULL = "FULL"
    FULLY = "FULLY"
    FULL_CYCLE = "FULL_CYCLE"
    FULL_FIELD = "FULL_FIELD"
    FULL_NEWTON = "FULL_NEWTON"
    FUNG_ANISOTROPIC = "FUNG_ANISOTROPIC"
    FUNG_ORTHOTROPIC = "FUNG_ORTHOTROPIC"
    G = "G"
    GALERKIN = "GALERKIN"
    GAPCYL = "GAPCYL"
    GAPSPHER = "GAPSPHER"
    GAPUNI = "GAPUNI"
    GAPUNIT = "GAPUNIT"
    GASKET = "GASKET"
    GAUSS = "GAUSS"
    GAUSS_COUPLING = "GAUSS_COUPLING"
    GENERAL = "GENERAL"
    GENERALIZED = "GENERALIZED"
    GENERALIZED_BEAM = "GENERALIZED_BEAM"
    GENERALIZED_DECAY = "GENERALIZED_DECAY"
    GENERALIZED_SHELL = "GENERALIZED_SHELL"
    GENERAL_OPTIMIZATION = "GENERAL_OPTIMIZATION"
    GENERAL_PARTICLE = "GENERAL_PARTICLE"
    GEOMETRY = "GEOMETRY"
    GEOMETRYFORMAT = "GEOMETRYFORMAT"
    GEOMETRY_ENHANCED = "GEOMETRY_ENHANCED"
    GEOM_DEVIATION_FACTOR = "GEOM_DEVIATION_FACTOR"
    GEOSTATIC = "GEOSTATIC"
    GFI = "GFI"
    GIGA_BYTES = "GIGA_BYTES"
    GK2D2 = "GK2D2"
    GK2D2N = "GK2D2N"
    GK3D12M = "GK3D12M"
    GK3D12MN = "GK3D12MN"
    GK3D18 = "GK3D18"
    GK3D18N = "GK3D18N"
    GK3D2 = "GK3D2"
    GK3D2N = "GK3D2N"
    GK3D4L = "GK3D4L"
    GK3D4LN = "GK3D4LN"
    GK3D6 = "GK3D6"
    GK3D6L = "GK3D6L"
    GK3D6LN = "GK3D6LN"
    GK3D6N = "GK3D6N"
    GK3D8 = "GK3D8"
    GK3D8N = "GK3D8N"
    GKAX2 = "GKAX2"
    GKAX2N = "GKAX2N"
    GKAX4 = "GKAX4"
    GKAX4N = "GKAX4N"
    GKAX6 = "GKAX6"
    GKAX6N = "GKAX6N"
    GKPE4 = "GKPE4"
    GKPE6 = "GKPE6"
    GKPS4 = "GKPS4"
    GKPS4N = "GKPS4N"
    GKPS6 = "GKPS6"
    GKPS6N = "GKPS6N"
    GLOBAL = "GLOBAL"
    GLOBAL_NONE = "GLOBAL_NONE"
    GLOBAL_X = "GLOBAL_X"
    GLOBAL_Y = "GLOBAL_Y"
    GLOBAL_Z = "GLOBAL_Z"
    GOURAUD = "GOURAUD"
    GRADED = "GRADED"
    GRADIENT = "GRADIENT"
    GRADIENTS = "GRADIENTS"
    GRADIENTS_THROUGH_BEAM_CS = "GRADIENTS_THROUGH_BEAM_CS"
    GRADIENTS_THROUGH_SHELL_CS = "GRADIENTS_THROUGH_SHELL_CS"
    GRAVITY = "GRAVITY"
    GREATER_THAN = "GREATER_THAN"
    GREATER_THAN_EQUAL = "GREATER_THAN_EQUAL"
    GREYSCALE = "GREYSCALE"
    GRID = "GRID"
    GROUND = "GROUND"
    GROUP_BY_MATERIAL = "GROUP_BY_MATERIAL"
    GROWTH_MOVEMENT = "GROWTH_MOVEMENT"
    GRP = "GRP"
    HALF = "HALF"
    HALF_CYCLE = "HALF_CYCLE"
    HALF_INDEX_SHIFT = "HALF_INDEX_SHIFT"
    HALF_OF_AVERAGE = "HALF_OF_AVERAGE"
    HARD = "HARD"
    HARDWARE_OVERLAY = "HARDWARE_OVERLAY"
    HARMONIC = "HARMONIC"
    HEADING = "HEADING"
    HEALER_TYPE = "HEALER_TYPE"
    HEATCAP = "HEATCAP"
    HEAT_FLUX = "HEAT_FLUX"
    HEAT_FLUX_AREA = "HEAT_FLUX_AREA"
    HEAT_FLUX_RATE = "HEAT_FLUX_RATE"
    HEAT_FLUX_VOLUME = "HEAT_FLUX_VOLUME"
    HEAT_TRANSFER = "HEAT_TRANSFER"
    HEIGHT = "HEIGHT"
    HEX = "HEX"
    HEX20 = "HEX20"
    HEX8 = "HEX8"
    HEX_DOMINATED = "HEX_DOMINATED"
    HIDDEN = "HIDDEN"
    HIGH = "HIGH"
    HINGE = "HINGE"
    HISTORY = "HISTORY"
    HOLLOW_CIRCLE = "HOLLOW_CIRCLE"
    HOLLOW_DIAMOND = "HOLLOW_DIAMOND"
    HOLLOW_SQUARE = "HOLLOW_SQUARE"
    HOLLOW_TRI = "HOLLOW_TRI"
    HOLZAPFEL = "HOLZAPFEL"
    HOME = "HOME"
    HORIZONTAL = "HORIZONTAL"
    HYBRID = "HYBRID"
    HYDRAULIC = "HYDRAULIC"
    HYDROSTATIC = "HYDROSTATIC"
    HYDROSTATIC_FLUID_MODELING = "HYDROSTATIC_FLUID_MODELING"
    HYPERBOLIC = "HYPERBOLIC"
    HYPERBOLIC_SINE = "HYPERBOLIC_SINE"
    HYSTERESIS_INITIAL_GUESSES_EXHAUSTED = "HYSTERESIS_INITIAL_GUESSES_EXHAUSTED"
    HYSTERESIS_JACOBIAN_CANNOT_BE_INVERTED = "HYSTERESIS_JACOBIAN_CANNOT_BE_INVERTED"
    ICC = "ICC"
    IDEALGAS = "IDEALGAS"
    IDENTICAL = "IDENTICAL"
    IDENTITY = "IDENTITY"
    IGNITIONANDGROWTH = "IGNITIONANDGROWTH"
    IMAGINARY = "IMAGINARY"
    IMMEDIATE = "IMMEDIATE"
    IMPEDANCE = "IMPEDANCE"
    IMPERFECTION = "IMPERFECTION"
    IMPLICIT = "IMPLICIT"
    IMPLICIT_DYNAMIC = "IMPLICIT_DYNAMIC"
    IMPLICIT_EXPLICIT = "IMPLICIT_EXPLICIT"
    IMPORT = "IMPORT"
    IMPORT_STEP_SIZE = "IMPORT_STEP_SIZE"
    IMPRINT = "IMPRINT"
    IMPROVED = "IMPROVED"
    INCHES = "INCHES"
    INCLUDE = "INCLUDE"
    INCOMPRESSIBLE = "INCOMPRESSIBLE"
    INCREMENT = "INCREMENT"
    INCREMENTAL = "INCREMENTAL"
    INCREMENTATION = "INCREMENTATION"
    INCREMENTATION_ALL = "INCREMENTATION_ALL"
    INCREMENTATION_PRESELECT = "INCREMENTATION_PRESELECT"
    INC_SIZE = "INC_SIZE"
    INDEPENDENT = "INDEPENDENT"
    INDEX = "INDEX"
    INERTIA_RELIEF = "INERTIA_RELIEF"
    INFILTRATION = "INFILTRATION"
    INFINITE = "INFINITE"
    INFLOW = "INFLOW"
    INITIAL = "INITIAL"
    INITIAL_AND_LAST = "INITIAL_AND_LAST"
    INITIAL_CONDITION = "INITIAL_CONDITION"
    INITIAL_NODES = "INITIAL_NODES"
    INITIAL_OPENINGS = "INITIAL_OPENINGS"
    INITIAL_OVERCLOSURES = "INITIAL_OVERCLOSURES"
    INITIAL_OVERCLOSURES_EXPLICIT = "INITIAL_OVERCLOSURES_EXPLICIT"
    INPUT = "INPUT"
    INPUT_FILE = "INPUT_FILE"
    INSIDE = "INSIDE"
    INSTANCE = "INSTANCE"
    INSTANCE_FROM_INSTANCE = "INSTANCE_FROM_INSTANCE"
    INSTANCE_FROM_PART = "INSTANCE_FROM_PART"
    INSTANCE_MAP = "INSTANCE_MAP"
    INSTANCE_MAP_COLORS = "INSTANCE_MAP_COLORS"
    INSTANCE_NOT_APPLICABLE = "INSTANCE_NOT_APPLICABLE"
    INSTANCE_TYPE_MAP = "INSTANCE_TYPE_MAP"
    INSTANCE_TYPE_MAP_COLORS = "INSTANCE_TYPE_MAP_COLORS"
    INSTANTANEOUS = "INSTANTANEOUS"
    INTEGER = "INTEGER"
    INTEGRATED_ALL = "INTEGRATED_ALL"
    INTEGRATED_PRESELECT = "INTEGRATED_PRESELECT"
    INTEGRATION_POINT = "INTEGRATION_POINT"
    INTEGRATION_POINTS = "INTEGRATION_POINTS"
    INTERACTION_MAP = "INTERACTION_MAP"
    INTERACTION_MAP_COLORS = "INTERACTION_MAP_COLORS"
    INTERACTION_PROP_MAP = "INTERACTION_PROP_MAP"
    INTERACTION_PROP_MAP_COLORS = "INTERACTION_PROP_MAP_COLORS"
    INTERACTION_TYPE_MAP = "INTERACTION_TYPE_MAP"
    INTERACTION_TYPE_MAP_COLORS = "INTERACTION_TYPE_MAP_COLORS"
    INTERFERENCE = "INTERFERENCE"
    INTERNAL = "INTERNAL"
    INTERNAL_SET_MAP = "INTERNAL_SET_MAP"
    INTERNAL_SET_MAP_COLORS = "INTERNAL_SET_MAP_COLORS"
    INTERNAL_SURFACE_MAP = "INTERNAL_SURFACE_MAP"
    INTERNAL_SURFACE_MAP_COLORS = "INTERNAL_SURFACE_MAP_COLORS"
    INTERPOLATED = "INTERPOLATED"
    INTERPOLATE_OFF = "INTERPOLATE_OFF"
    INTERPOLATE_ON = "INTERPOLATE_ON"
    INTERRUPTED = "INTERRUPTED"
    INTERSECTION = "INTERSECTION"
    INTERSECTIONS = "INTERSECTIONS"
    INV3 = "INV3"
    INVALID = "INVALID"
    INVALID_SURF = "INVALID_SURF"
    INVARIANT = "INVARIANT"
    INWARD = "INWARD"
    ISL21A = "ISL21A"
    ISL22A = "ISL22A"
    ISOLINES = "ISOLINES"
    ISOSURFACE = "ISOSURFACE"
    ISOTROPIC = "ISOTROPIC"
    ISOTROPIC_CFD = "ISOTROPIC_CFD"
    ITERATION = "ITERATION"
    ITERATIVE = "ITERATIVE"
    ITERATIVE_SOLVER = "ITERATIVE_SOLVER"
    ITSCYL = "ITSCYL"
    ITSUNI = "ITSUNI"
    ITT21 = "ITT21"
    ITT31 = "ITT31"
    JACOBI_COUPLING = "JACOBI_COUPLING"
    JAMA = "JAMA"
    JOB_ABORTED = "JOB_ABORTED"
    JOB_COMPLETED = "JOB_COMPLETED"
    JOB_INTERRUPTED = "JOB_INTERRUPTED"
    JOB_STATUS_COMPLETED_SUCCESSFULLY = "JOB_STATUS_COMPLETED_SUCCESSFULLY"
    JOB_STATUS_EXITED_WITH_ERROR = "JOB_STATUS_EXITED_WITH_ERROR"
    JOB_STATUS_TERMINATED_BY_EXTERNAL_REQUEST = "JOB_STATUS_TERMINATED_BY_EXTERNAL_REQUEST"
    JOB_STATUS_UNKNOWN = "JOB_STATUS_UNKNOWN"
    JOB_SUBMITTED = "JOB_SUBMITTED"
    JOHNSON_COOK = "JOHNSON_COOK"
    JOIN = "JOIN"
    JOINT2D = "JOINT2D"
    JOINT3D = "JOINT3D"
    JOINTC = "JOINTC"
    JUSTIFY_LEFT = "JUSTIFY_LEFT"
    JWL = "JWL"
    J_INTEGRAL = "J_INTEGRAL"
    KEPS_RNG = "KEPS_RNG"
    KII0 = "KII0"
    KINEMATIC = "KINEMATIC"
    KINEMATIC_COUPLING = "KINEMATIC_COUPLING"
    KINEMATIC_HARDENING = "KINEMATIC_HARDENING"
    KINEMATIC_VIOLATIONS = "KINEMATIC_VIOLATIONS"
    KINETIC_ENERGY_XPL = "KINETIC_ENERGY_XPL"
    K_FACTORS = "K_FACTORS"
    LAGRANGE = "LAGRANGE"
    LAGRANGIAN = "LAGRANGIAN"
    LAG_ANALYSIS = "LAG_ANALYSIS"
    LAMINA = "LAMINA"
    LANCZOS = "LANCZOS"
    LANCZOS_EIGENSOLVER = "LANCZOS_EIGENSOLVER"
    LANDSCAPE = "LANDSCAPE"
    LARGE = "LARGE"
    LARGE_ANGLE = "LARGE_ANGLE"
    LARGE_ANGLE_QUAD_FACE = "LARGE_ANGLE_QUAD_FACE"
    LARGE_ANGLE_TRI_FACE = "LARGE_ANGLE_TRI_FACE"
    LARGE_STRAIN_INCREMENT = "LARGE_STRAIN_INCREMENT"
    LAST = "LAST"
    LAST_FRAME = "LAST_FRAME"
    LAST_INCREMENT = "LAST_INCREMENT"
    LAST_STEP = "LAST_STEP"
    LATERAL_EXTENSION_RATIO_NOT_FOUND = "LATERAL_EXTENSION_RATIO_NOT_FOUND"
    LATERAL_NOMINAL_STRAIN = "LATERAL_NOMINAL_STRAIN"
    LAYUP = "LAYUP"
    LAYUP_MAP = "LAYUP_MAP"
    LAYUP_MAP_COLORS = "LAYUP_MAP_COLORS"
    LAYUP_ORIENTATION = "LAYUP_ORIENTATION"
    LEAD_ANALYSIS = "LEAD_ANALYSIS"
    LEAF_COLORS = "LEAF_COLORS"
    LEDGER = "LEDGER"
    LEFT = "LEFT"
    LEGAL = "LEGAL"
    LENGTH = "LENGTH"
    LESS_THAN = "LESS_THAN"
    LESS_THAN_EQUAL = "LESS_THAN_EQUAL"
    LETTER = "LETTER"
    LETTERS = "LETTERS"
    LICENSING = "LICENSING"
    LIFT_EQUATION = "LIFT_EQUATION"
    LINE = "LINE"
    LINE2 = "LINE2"
    LINE3 = "LINE3"
    LINEAR = "LINEAR"
    LINEAR_EXTRAPOLATION = "LINEAR_EXTRAPOLATION"
    LINEAR_LEAST_SQUARES = "LINEAR_LEAST_SQUARES"
    LINEAR_PRESSURE = "LINEAR_PRESSURE"
    LINELINEDIST = "LINELINEDIST"
    LINES = "LINES"
    LINE_LOOP = "LINE_LOOP"
    LINE_SEARCH = "LINE_SEARCH"
    LINE_STRIP = "LINE_STRIP"
    LINK = "LINK"
    LINK_MPC = "LINK_MPC"
    LINUX = "LINUX"
    LIST_OF_MODES = "LIST_OF_MODES"
    LOAD = "LOAD"
    LOAD_CASE = "LOAD_CASE"
    LOAD_MAP = "LOAD_MAP"
    LOAD_MAP_COLORS = "LOAD_MAP_COLORS"
    LOCAL = "LOCAL"
    LOCAL_GRADIENT = "LOCAL_GRADIENT"
    LOCKSTEP = "LOCKSTEP"
    LOG = "LOG"
    LOGARITHMIC = "LOGARITHMIC"
    LONG = "LONG"
    LONGEST_EDGE = "LONGEST_EDGE"
    LONG_TERM = "LONG_TERM"
    LOOP = "LOOP"
    LOOP_BACKWARD = "LOOP_BACKWARD"
    LOOSE_COUPLING = "LOOSE_COUPLING"
    LOW = "LOW"
    LS3S = "LS3S"
    LS6 = "LS6"
    LUMIN = "LUMIN"
    M3D3 = "M3D3"
    M3D4 = "M3D4"
    M3D4R = "M3D4R"
    M3D6 = "M3D6"
    M3D8 = "M3D8"
    M3D8R = "M3D8R"
    MACAULEY = "MACAULEY"
    MAGNITUDE = "MAGNITUDE"
    MAGNITUDE_AND_PHASE = "MAGNITUDE_AND_PHASE"
    MAINTAIN_CURRENT = "MAINTAIN_CURRENT"
    MAIN_REGION = "MAIN_REGION"
    MANUAL = "MANUAL"
    MAP = "MAP"
    MARLOW = "MARLOW"
    MASS = "MASS"
    MASS_DIFFUSION = "MASS_DIFFUSION"
    MASS_FLOW_AREA = "MASS_FLOW_AREA"
    MASS_FLOW_AREA_RATE = "MASS_FLOW_AREA_RATE"
    MASS_FLOW_RATE = "MASS_FLOW_RATE"
    MASS_FLUX = "MASS_FLUX"
    MASS_PER_AREA = "MASS_PER_AREA"
    MASS_PER_LENGTH = "MASS_PER_LENGTH"
    MASS_PER_VOLUME = "MASS_PER_VOLUME"
    MASS_PROPORTIONAL = "MASS_PROPORTIONAL"
    MASS_RATE_LEAK = "MASS_RATE_LEAK"
    MASS_SCALING = "MASS_SCALING"
    MASTER = "MASTER"
    MATCH = "MATCH"
    MATERIAL = "MATERIAL"
    MATERIAL_FLOW_FIELD = "MATERIAL_FLOW_FIELD"
    MATERIAL_INSTABILITY = "MATERIAL_INSTABILITY"
    MATERIAL_MAP = "MATERIAL_MAP"
    MATERIAL_MAP_COLORS = "MATERIAL_MAP_COLORS"
    MATERIAL_POINT_CALCULATIONS = "MATERIAL_POINT_CALCULATIONS"
    MATRIX = "MATRIX"
    MAX = "MAX"
    MAX1 = "MAX1"
    MAX2 = "MAX2"
    MAXIMIZE = "MAXIMIZE"
    MAXIMIZED = "MAXIMIZED"
    MAXIMUM = "MAXIMUM"
    MAXIMUM_NUMBER_OF_CONTACT_STRESS_AUGMENTATIONS = "MAXIMUM_NUMBER_OF_CONTACT_STRESS_AUGMENTATIONS"
    MAXIMUM_NUMBER_OF_EQUILIBRIUM_ITERATIONS = "MAXIMUM_NUMBER_OF_EQUILIBRIUM_ITERATIONS"
    MAXIMUM_NUMBER_OF_SEVERE_DISCONTINUITY_ITERATIONS = "MAXIMUM_NUMBER_OF_SEVERE_DISCONTINUITY_ITERATIONS"
    MAXIMUM_SLIDE_DISTANCE_EXCEEDED = "MAXIMUM_SLIDE_DISTANCE_EXCEEDED"
    MAX_ABS_VALUE = "MAX_ABS_VALUE"
    MAX_CORRECTION = "MAX_CORRECTION"
    MAX_EDGE = "MAX_EDGE"
    MAX_ERROR = "MAX_ERROR"
    MAX_ELASTOPLASTIC_STRAIN = "MAX_ELASTOPLASTIC_STRAIN"
    MAX_FREQUENCY = "MAX_FREQUENCY"
    MAX_INCREMENT = "MAX_INCREMENT"
    MAX_INPLANE_PRINCIPAL = "MAX_INPLANE_PRINCIPAL"
    MAX_PRINCIPAL = "MAX_PRINCIPAL"
    MAX_RESIDUAL = "MAX_RESIDUAL"
    MAX_SEPARATION = "MAX_SEPARATION"
    MAX_SHEAR_STRAIN = "MAX_SHEAR_STRAIN"
    MAX_STEP_SIZE = "MAX_STEP_SIZE"
    MAX_STRESS = "MAX_STRESS"
    MAX_VALUE = "MAX_VALUE"
    MCL6 = "MCL6"
    MCL9 = "MCL9"
    MECHANICAL = "MECHANICAL"
    MECHANICAL_CONTACT = "MECHANICAL_CONTACT"
    MEDIAL_AXIS = "MEDIAL_AXIS"
    MEDIUM = "MEDIUM"
    MEGA_BYTES = "MEGA_BYTES"
    MEMBRANE = "MEMBRANE"
    MEMORY = "MEMORY"
    MEMORY_ESTIMATE = "MEMORY_ESTIMATE"
    MERGE = "MERGE"
    MERR = "MERR"
    MESH = "MESH"
    MESHED_BEAM_SECTION = "MESHED_BEAM_SECTION"
    MESH_MAP = "MESH_MAP"
    MESH_MAP_COLORS = "MESH_MAP_COLORS"
    MESH_TIE = "MESH_TIE"
    MGAX1 = "MGAX1"
    MGAX2 = "MGAX2"
    MIDDLE = "MIDDLE"
    MIDDLE_SURFACE = "MIDDLE_SURFACE"
    MIDSIDE_ONLY = "MIDSIDE_ONLY"
    MID_PRINCIPAL = "MID_PRINCIPAL"
    MILLING_REGION = "MILLING_REGION"
    MIN = "MIN"
    MINIMIZE = "MINIMIZE"
    MINIMIZED = "MINIMIZED"
    MINIMIZE_MAXIMUM = "MINIMIZE_MAXIMUM"
    MINIMUM = "MINIMUM"
    MINIMUM_MAXIMUM = "MINIMUM_MAXIMUM"
    MINIMUM_MOVE = "MINIMUM_MOVE"
    MIN_EDGE = "MIN_EDGE"
    MIN_INPLANE_PRINCIPAL = "MIN_INPLANE_PRINCIPAL"
    MIN_MAX_EDGE = "MIN_MAX_EDGE"
    MIN_MAX_LABEL = "MIN_MAX_LABEL"
    MIN_PRINCIPAL = "MIN_PRINCIPAL"
    MIN_PRINCIPAL_STRAIN = "MIN_PRINCIPAL_STRAIN"
    MIN_SIZE_FACTOR = "MIN_SIZE_FACTOR"
    MIN_STEP_SIZE = "MIN_STEP_SIZE"
    MIN_TRANSITION = "MIN_TRANSITION"
    MIN_VALUE = "MIN_VALUE"
    MIRROR = "MIRROR"
    MIRROR_CIRC_RECT = "MIRROR_CIRC_RECT"
    MIRROR_RECT_CIRC = "MIRROR_RECT_CIRC"
    MISES = "MISES"
    MIXED = "MIXED"
    MM = "MM"
    MODAL = "MODAL"
    MODAL_ALL = "MODAL_ALL"
    MODAL_DYNAMICS = "MODAL_DYNAMICS"
    MODEL = "MODEL"
    MODEL_CHANGE = "MODEL_CHANGE"
    MODEL_SIZE = "MODEL_SIZE"
    MODERATE = "MODERATE"
    MODERATE_DISSIPATION = "MODERATE_DISSIPATION"
    MODES = "MODES"
    MODE_BASED_DYNAMIC = "MODE_BASED_DYNAMIC"
    MODE_BASED_STEADY_STATE_DYNAMIC = "MODE_BASED_STEADY_STATE_DYNAMIC"
    MODE_INDEPENDENT = "MODE_INDEPENDENT"
    MODE_NUMBER = "MODE_NUMBER"
    MODE_RANGE = "MODE_RANGE"
    MODIFIED = "MODIFIED"
    MODIFIED_FROM_BASE_STATE = "MODIFIED_FROM_BASE_STATE"
    MOMENT = "MOMENT"
    MONITOR_DATA = "MONITOR_DATA"
    MOONEY_RIVLIN = "MOONEY_RIVLIN"
    MOTION = "MOTION"
    MOTION_TYPE = "MOTION_TYPE"
    MOVEMENT = "MOVEMENT"
    MOVING_NOISE = "MOVING_NOISE"
    MPC = "MPC"
    MPI = "MPI"
    MSBO = "MSBO"
    MSFLD = "MSFLD"
    MSPEI_XPL = "MSPEI_XPL"
    MT = "MT"
    MTS = "MTS"
    MULTIPLE_DIRECTION_ABSOLUTE_SUM = "MULTIPLE_DIRECTION_ABSOLUTE_SUM"
    MULTIPLE_DIRECTION_FORTY_PERCENT_RULE = "MULTIPLE_DIRECTION_FORTY_PERCENT_RULE"
    MULTIPLE_DIRECTION_SRSS_SUM = "MULTIPLE_DIRECTION_SRSS_SUM"
    MULTIPLE_DIRECTION_THIRTY_PERCENT_RULE = "MULTIPLE_DIRECTION_THIRTY_PERCENT_RULE"
    MULTIPLE_SURFACE = "MULTIPLE_SURFACE"
    MULTIPLICATIVE = "MULTIPLICATIVE"
    MULTIPLY = "MULTIPLY"
    N1 = "N1"
    N1_COSINES = "N1_COSINES"
    N2 = "N2"
    NATURAL_FREQUENCY_EXTRACTION = "NATURAL_FREQUENCY_EXTRACTION"
    NATURAL_LOG = "NATURAL_LOG"
    NEAREST_INTEGER = "NEAREST_INTEGER"
    NEGATE = "NEGATE"
    NEGATIVE = "NEGATIVE"
    NEO_HOOKE = "NEO_HOOKE"
    NEVER = "NEVER"
    NEWTONIAN = "NEWTONIAN"
    NEW_CONTACT_PATCH = "NEW_CONTACT_PATCH"
    NMORI = "NMORI"
    # NO = 'NO'
    NODAL = "NODAL"
    NODAL_ANALYTICAL_FIELD = "NODAL_ANALYTICAL_FIELD"
    NODAL_AVERAGE = "NODAL_AVERAGE"
    NODAL_DISCRETE_FIELD = "NODAL_DISCRETE_FIELD"
    NODAL_LINE = "NODAL_LINE"
    NODE = "NODE"
    NODES = "NODES"
    NODE_ALL = "NODE_ALL"
    NODE_CENTERED = "NODE_CENTERED"
    NODE_LIST = "NODE_LIST"
    NODE_MODE = "NODE_MODE"
    NODE_MODE_MPC = "NODE_MODE_MPC"
    NODE_PRESELECT = "NODE_PRESELECT"
    NODE_TO_SURFACE = "NODE_TO_SURFACE"
    NONACCUMULATEDENERGY = "NONACCUMULATEDENERGY"
    NONE = "NONE"
    NONLINEAR = "NONLINEAR"
    NONREFLECTING = "NONREFLECTING"
    NONUNIFORM = "NONUNIFORM"
    NON_DEFAULT = "NON_DEFAULT"
    NON_REFLECTING = "NON_REFLECTING"
    NORM = "NORM"
    NORMAL = "NORMAL"
    NORMALIZED_CONCENTRATION_FIELD = "NORMALIZED_CONCENTRATION_FIELD"
    NORMALS = "NORMALS"
    NORMAL_ANNOTATED = "NORMAL_ANNOTATED"
    NORMAL_TANGENTIAL = "NORMAL_TANGENTIAL"
    NORMAL_VECTOR = "NORMAL_VECTOR"
    NORM_DISTANCE = "NORM_DISTANCE"
    NORM_FIRST = "NORM_FIRST"
    NOT_ALLOWED = "NOT_ALLOWED"
    NOT_APPLICABLE = "NOT_APPLICABLE"
    NOT_SET = "NOT_SET"
    NOT_YET_ACTIVE = "NOT_YET_ACTIVE"
    NO_BLOCKING = "NO_BLOCKING"
    NO_EXTRAPOLATION = "NO_EXTRAPOLATION"
    NO_HEAD = "NO_HEAD"
    NO_IDEALIZATION = "NO_IDEALIZATION"
    NO_INDEPENDENT_COMPONENTS = "NO_INDEPENDENT_COMPONENTS"
    NO_INITIAL_INTERSECTION = "NO_INITIAL_INTERSECTION"
    NO_LIMIT = "NO_LIMIT"
    NO_LINE = "NO_LINE"
    NO_LONGER_ACTIVE = "NO_LONGER_ACTIVE"
    NO_OUTPUT = "NO_OUTPUT"
    NO_OUTPUT_VARIABLES = "NO_OUTPUT_VARIABLES"
    NO_REFINEMENT = "NO_REFINEMENT"
    NO_SIMPLIFICATION = "NO_SIMPLIFICATION"
    NO_SLIP = "NO_SLIP"
    NRL = "NRL"
    NSET_MAP = "NSET_MAP"
    NT = "NT"
    NTH_POWER = "NTH_POWER"
    NTH_ROOT = "NTH_ROOT"
    NUMBER = "NUMBER"
    NUMBERS = "NUMBERS"
    NUMBER_INTERVALS = "NUMBER_INTERVALS"
    NUMBER_OF_LAYERS = "NUMBER_OF_LAYERS"
    NUMBER_OF_VALUES = "NUMBER_OF_VALUES"
    NUMERICAL_PROBLEM = "NUMERICAL_PROBLEM"
    NUM_ATTEMPTS = "NUM_ATTEMPTS"
    NUM_DATA_TYPE = "NUM_DATA_TYPE"
    NUM_ELEMENTS_EXCEEDING = "NUM_ELEMENTS_EXCEEDING"
    NUM_EQI = "NUM_EQI"
    NUM_ITERS = "NUM_ITERS"
    NUM_PTS_ALONG_DIR = "NUM_PTS_ALONG_DIR"
    NUM_PTS_BETWEEN_PTS = "NUM_PTS_BETWEEN_PTS"
    NUM_SDI = "NUM_SDI"
    OBLIQUE_DOWN = "OBLIQUE_DOWN"
    OBLIQUE_UP = "OBLIQUE_UP"
    ODB = "ODB"
    ODB_FILE = "ODB_FILE"
    ODB_FRAME = "ODB_FRAME"
    ODB_REGIONS = "ODB_REGIONS"
    ODB_VALUES = "ODB_VALUES"
    OFFSET_FIELD = "OFFSET_FIELD"
    OGDEN = "OGDEN"
    OGDEN_N1 = "OGDEN_N1"
    OGDEN_N2 = "OGDEN_N2"
    OGDEN_N3 = "OGDEN_N3"
    OGDEN_N4 = "OGDEN_N4"
    OGDEN_N5 = "OGDEN_N5"
    OGDEN_N6 = "OGDEN_N6"
    OMIT = "OMIT"
    ONE_CONFIG = "ONE_CONFIG"
    ONLY = "ONLY"
    OPENINGS = "OPENINGS"
    OPEN_GL = "OPEN_GL"
    OPTIMIZATION_DISPLACEMENT = "OPTIMIZATION_DISPLACEMENT"
    OPT_DATASAVE_EVERY_CYCLE = "OPT_DATASAVE_EVERY_CYCLE"
    OPT_DATASAVE_FIRST_AND_LAST_CYCLE = "OPT_DATASAVE_FIRST_AND_LAST_CYCLE"
    OPT_DATASAVE_SPECIFY_CYCLE = "OPT_DATASAVE_SPECIFY_CYCLE"
    OPT_EXTRACT_SMOOTH_ABAQUS_INPUT_FILE = "OPT_EXTRACT_SMOOTH_ABAQUS_INPUT_FILE"
    OPT_EXTRACT_SMOOTH_NONE = "OPT_EXTRACT_SMOOTH_NONE"
    OPT_EXTRACT_SMOOTH_STL = "OPT_EXTRACT_SMOOTH_STL"
    ORDINATE = "ORDINATE"
    ORIENTATION = "ORIENTATION"
    ORIENT_FIELD = "ORIENT_FIELD"
    ORIENT_ON_DEF = "ORIENT_ON_DEF"
    ORIENT_ON_UNDEF = "ORIENT_ON_UNDEF"
    ORIGINAL = "ORIGINAL"
    ORIGINAL_MODEL = "ORIGINAL_MODEL"
    ORIGIN_AXIS = "ORIGIN_AXIS"
    ORTHOGONAL = "ORTHOGONAL"
    ORTHOTROPIC = "ORTHOTROPIC"
    OTHER = "OTHER"
    OTHER_BC = "OTHER_BC"
    OUTFLOW = "OUTFLOW"
    OUTOFPLANE_PRINCIPAL = "OUTOFPLANE_PRINCIPAL"
    OUTPUT = "OUTPUT"
    OUTSIDE = "OUTSIDE"
    OUTWARD = "OUTWARD"
    OUT_OF_PLANE_EXTENSION_RATIO_NOT_FOUND = "OUT_OF_PLANE_EXTENSION_RATIO_NOT_FOUND"
    OVERCLOSED = "OVERCLOSED"
    OVERCLOSURES = "OVERCLOSURES"
    OVERCONSTRAINT = "OVERCONSTRAINT"
    OVERHANG_REGION = "OVERHANG_REGION"
    OVERLAY = "OVERLAY"
    OVERWRITE = "OVERWRITE"
    PACKAGER_PHASE = "PACKAGER_PHASE"
    PARABOLA = "PARABOLA"
    PARABOLIC = "PARABOLIC"
    PARABOLIC_EXTRAPOLATION = "PARABOLIC_EXTRAPOLATION"
    PARALLEL = "PARALLEL"
    PARAMETERS = "PARAMETERS"
    PARAMETRIC_DATA = "PARAMETRIC_DATA"
    PART = "PART"
    PARTIAL_BLOCKING = "PARTIAL_BLOCKING"
    PARTIAL_FIELD = "PARTIAL_FIELD"
    PART_ASSEMBLY = "PART_ASSEMBLY"
    PART_GEOM_MAP = "PART_GEOM_MAP"
    PART_GEOM_MAP_COLORS = "PART_GEOM_MAP_COLORS"
    PART_MAP = "PART_MAP"
    PART_MAP_COLORS = "PART_MAP_COLORS"
    PATH = "PATH"
    PATH_POINTS = "PATH_POINTS"
    PATH_X = "PATH_X"
    PATH_Y = "PATH_Y"
    PATH_Z = "PATH_Z"
    PATTERN_ALONG_DIRECTION = "PATTERN_ALONG_DIRECTION"
    PATTERN_INVALID = "PATTERN_INVALID"
    PATTERN_ORTHOGONALLY = "PATTERN_ORTHOGONALLY"
    PENALTY = "PENALTY"
    PENETRATION_TOLERANCE_EXCEEDED = "PENETRATION_TOLERANCE_EXCEEDED"
    PERCENTAGE = "PERCENTAGE"
    PERIMETER = "PERIMETER"
    PERIOD = "PERIOD"
    PERP = "PERP"
    PERPENDICULAR = "PERPENDICULAR"
    PERSPECTIVE = "PERSPECTIVE"
    PERTURBATION_AND_BUCKLING = "PERTURBATION_AND_BUCKLING"
    PHASE = "PHASE"
    PHONG = "PHONG"
    PICKED = "PICKED"
    PICKEDPOINTS = "PICKEDPOINTS"
    PIEZOELECTRIC = "PIEZOELECTRIC"
    PIEZO_ELECTRICAL_POTENTIAL_FIELD = "PIEZO_ELECTRICAL_POTENTIAL_FIELD"
    PINNED = "PINNED"
    PIN_MPC = "PIN_MPC"
    PIPE21 = "PIPE21"
    PIPE21H = "PIPE21H"
    PIPE22 = "PIPE22"
    PIPE22H = "PIPE22H"
    PIPE31 = "PIPE31"
    PIPE31H = "PIPE31H"
    PIPE32 = "PIPE32"
    PIPE32H = "PIPE32H"
    PLANAR = "PLANAR"
    PLANE = "PLANE"
    PLANE12 = "PLANE12"
    PLANE13 = "PLANE13"
    PLANE21 = "PLANE21"
    PLANE23 = "PLANE23"
    PLANE31 = "PLANE31"
    PLANE32 = "PLANE32"
    PLASTIC_MOTION = "PLASTIC_MOTION"
    PLAY = "PLAY"
    PLAY_ONCE = "PLAY_ONCE"
    PLOT_MAP = "PLOT_MAP"
    PLOT_MAP_COLORS = "PLOT_MAP_COLORS"
    PLOT_OPTIONS = "PLOT_OPTIONS"
    PLOT_STATE = "PLOT_STATE"
    PLY = "PLY"
    PLY_BASED = "PLY_BASED"
    PLY_MAP = "PLY_MAP"
    PLY_MAP_COLORS = "PLY_MAP_COLORS"
    PNEUMATIC = "PNEUMATIC"
    PNG = "PNG"
    POINT = "POINT"
    POINTLINEDIST = "POINTLINEDIST"
    POINTPOINTDIST = "POINTPOINTDIST"
    POINTS = "POINTS"
    POINTS_NOW_SLIPPING = "POINTS_NOW_SLIPPING"
    POINTS_NOW_STICKING = "POINTS_NOW_STICKING"
    POINTS_THROUGH_SECTION = "POINTS_THROUGH_SECTION"
    POINTWISE = "POINTWISE"
    POINT_ARC = "POINT_ARC"
    POINT_LIST = "POINT_LIST"
    POISSON = "POISSON"
    POISSON_RATIO = "POISSON_RATIO"
    POLAR = "POLAR"
    POLYNOMIAL = "POLYNOMIAL"
    POLY_N1 = "POLY_N1"
    POLY_N2 = "POLY_N2"
    POLY_N3 = "POLY_N3"
    POLY_N4 = "POLY_N4"
    POLY_N5 = "POLY_N5"
    POLY_N6 = "POLY_N6"
    PORE_FLUID_CONTACT = "PORE_FLUID_CONTACT"
    PORE_FLUID_DIFFUSION = "PORE_FLUID_DIFFUSION"
    PORE_LIQUID_PRESSURE_FIELD = "PORE_LIQUID_PRESSURE_FIELD"
    PORTRAIT = "PORTRAIT"
    POSITION = "POSITION"
    POSITIONS = "POSITIONS"
    POSITIVE = "POSITIVE"
    POWER = "POWER"
    POWER_LAW = "POWER_LAW"
    PREDEFINED = "PREDEFINED"
    PREDEFINED_FIELD = "PREDEFINED_FIELD"
    PREDEFINED_PATH = "PREDEFINED_PATH"
    PRESCRIBEDCONDITION_DOF = "PRESCRIBEDCONDITION_DOF"
    PRESELECT = "PRESELECT"
    PRESERVE_SECTION = "PRESERVE_SECTION"
    PRESS = "PRESS"
    PRESSURE = "PRESSURE"
    PRESSURE_GRADIENT = "PRESSURE_GRADIENT"
    PRESSURE_PENETRATION = "PRESSURE_PENETRATION"
    PRESSURE_STRESS_CONSTRAINT = "PRESSURE_STRESS_CONSTRAINT"
    PREVIOUS = "PREVIOUS"
    PREVIOUS_STEP = "PREVIOUS_STEP"
    PRIMARY_VECTOR = "PRIMARY_VECTOR"
    PRINCIPAL = "PRINCIPAL"
    PRINCIPAL_COMPONENT = "PRINCIPAL_COMPONENT"
    PRINTER = "PRINTER"
    PROJECTION_CARTESIAN = "PROJECTION_CARTESIAN"
    PROJECTION_FLEXION_TORSION = "PROJECTION_FLEXION_TORSION"
    PROJECT_BY_DIRECTION = "PROJECT_BY_DIRECTION"
    PROJECT_BY_DISTANCE = "PROJECT_BY_DISTANCE"
    PROJECT_BY_NUMBER = "PROJECT_BY_NUMBER"
    PROJECT_BY_PROXIMITY = "PROJECT_BY_PROXIMITY"
    PROLATE = "PROLATE"
    PRONY = "PRONY"
    PROPAGATED = "PROPAGATED"
    PROPAGATED_FROM_BASE_STATE = "PROPAGATED_FROM_BASE_STATE"
    PROPAGATED_FROM_COMPUTED = "PROPAGATED_FROM_COMPUTED"
    PROPAGATED_FROM_FREQUENCY = "PROPAGATED_FROM_FREQUENCY"
    PROPERTY_CHANGE = "PROPERTY_CHANGE"
    PROPERTY_MAP = "PROPERTY_MAP"
    PROPERTY_MAP_COLORS = "PROPERTY_MAP_COLORS"
    PROPERTY_REF = "PROPERTY_REF"
    PS = "PS"
    PSI24 = "PSI24"
    PSI26 = "PSI26"
    PSI34 = "PSI34"
    PSI36 = "PSI36"
    PS_ALWAYS = "PS_ALWAYS"
    PS_IF_AVAILABLE = "PS_IF_AVAILABLE"
    PYR5 = "PYR5"
    Q3D10M = "Q3D10M"
    Q3D10MH = "Q3D10MH"
    Q3D20 = "Q3D20"
    Q3D20H = "Q3D20H"
    Q3D20R = "Q3D20R"
    Q3D20RH = "Q3D20RH"
    Q3D4 = "Q3D4"
    Q3D6 = "Q3D6"
    Q3D8 = "Q3D8"
    Q3D8H = "Q3D8H"
    Q3D8R = "Q3D8R"
    Q3D8RH = "Q3D8RH"
    QUAD = "QUAD"
    QUAD4 = "QUAD4"
    QUAD8 = "QUAD8"
    QUADRATIC = "QUADRATIC"
    QUADS = "QUADS"
    QUAD_DOMINATED = "QUAD_DOMINATED"
    QUAD_SEPARATION = "QUAD_SEPARATION"
    QUAD_STRIP = "QUAD_STRIP"
    QUAD_TRACTION = "QUAD_TRACTION"
    QUASI_NEWTON = "QUASI_NEWTON"
    QUASI_NEWTON_METHOD = "QUASI_NEWTON_METHOD"
    QUASI_STATIC = "QUASI_STATIC"
    QUATERNION_2D = "QUATERNION_2D"
    QUATERNION_3D = "QUATERNION_3D"
    QUICKTIME = "QUICKTIME"
    QUILT = "QUILT"
    QUINTIC = "QUINTIC"
    Q_VECTORS = "Q_VECTORS"
    R2D2 = "R2D2"
    R3D3 = "R3D3"
    R3D4 = "R3D4"
    RADIAL = "RADIAL"
    RADIAL_THRUST = "RADIAL_THRUST"
    RADIATION_ALL = "RADIATION_ALL"
    RADIUS = "RADIUS"
    RAINBOW = "RAINBOW"
    RAMP = "RAMP"
    RANDOM_RESPONSE = "RANDOM_RESPONSE"
    RANGE = "RANGE"
    RASTER = "RASTER"
    RATE = "RATE"
    RATE_OF_CONVERGENCE_IS_SLOW = "RATE_OF_CONVERGENCE_IS_SLOW"
    RAW24 = "RAW24"
    RAW32 = "RAW32"
    RAW8 = "RAW8"
    RAW_DATA = "RAW_DATA"
    RAX2 = "RAX2"
    RB2D2 = "RB2D2"
    RB3D2 = "RB3D2"
    REAL = "REAL"
    REAL_AND_IMAGINARY = "REAL_AND_IMAGINARY"
    REAL_ONLY = "REAL_ONLY"
    REBAR = "REBAR"
    RECIEVE_PREDICTION = "RECIEVE_PREDICTION"
    RECOMPUTE_EACH_FRAME = "RECOMPUTE_EACH_FRAME"
    RECOMPUTE_GEOMETRY = "RECOMPUTE_GEOMETRY"
    RECOVER = "RECOVER"
    RECT_CIRC_MIRROR = "RECT_CIRC_MIRROR"
    RECT_MIRROR_CIRC = "RECT_MIRROR_CIRC"
    RECURSIVE = "RECURSIVE"
    REDUCED_POLYNOMIAL = "REDUCED_POLYNOMIAL"
    REDUCED_POLYNOMIAL_ALL = "REDUCED_POLYNOMIAL_ALL"
    REDUCED_POLYNOMIAL_N1 = "REDUCED_POLYNOMIAL_N1"
    REDUCED_POLYNOMIAL_N2 = "REDUCED_POLYNOMIAL_N2"
    REDUCED_POLYNOMIAL_N3 = "REDUCED_POLYNOMIAL_N3"
    REDUCED_POLYNOMIAL_N4 = "REDUCED_POLYNOMIAL_N4"
    REDUCED_POLYNOMIAL_N5 = "REDUCED_POLYNOMIAL_N5"
    REDUCED_POLYNOMIAL_N6 = "REDUCED_POLYNOMIAL_N6"
    RED_TO_BLUE = "RED_TO_BLUE"
    REEDER = "REEDER"
    REFERENCE = "REFERENCE"
    REGION = "REGION"
    REGULAR = "REGULAR"
    REGULAR_BEAM = "REGULAR_BEAM"
    REGULAR_SHELL = "REGULAR_SHELL"
    REINFORCEMENT = "REINFORCEMENT"
    REINITIALIZE = "REINITIALIZE"
    RELATIVE = "RELATIVE"
    RELATIVE_EQUAL = "RELATIVE_EQUAL"
    RELATIVE_GREATER_THAN_EQUAL = "RELATIVE_GREATER_THAN_EQUAL"
    RELATIVE_LESS_THAN_EQUAL = "RELATIVE_LESS_THAN_EQUAL"
    RELATIVE_SLOPE_DROP = "RELATIVE_SLOPE_DROP"
    RELATIVE_VALUE = "RELATIVE_VALUE"
    RELAXATION = "RELAXATION"
    RELAXATION_TEST_DATA = "RELAXATION_TEST_DATA"
    RELAX_STIFFNESS = "RELAX_STIFFNESS"
    REMOVE = "REMOVE"
    REPETITIVE_SDI_PATTERN = "REPETITIVE_SDI_PATTERN"
    REPLACE = "REPLACE"
    RESET_TO_INITIAL = "RESET_TO_INITIAL"
    RESIDUAL_ACCEPTED = "RESIDUAL_ACCEPTED"
    RESIDUAL_NOT_ACCEPTED = "RESIDUAL_NOT_ACCEPTED"
    RESPONSE_SPECTRUM = "RESPONSE_SPECTRUM"
    RESTART = "RESTART"
    RESTRICTED_TASK_REGION_EQUIV_STRESS = "RESTRICTED_TASK_REGION_EQUIV_STRESS"
    RESULTANT = "RESULTANT"
    RETENTION_FACTOR = "RETENTION_FACTOR"
    RETRACTOR = "RETRACTOR"
    REVERSE = "REVERSE"
    REVERSED_RAINBOW = "REVERSED_RAINBOW"
    REVERSE_MIRROR = "REVERSE_MIRROR"
    REVOLUTE = "REVOLUTE"
    REVOLUTION = "REVOLUTION"
    RIGHT = "RIGHT"
    RIGID = "RIGID"
    RIGID_BODY = "RIGID_BODY"
    RIGID_SPRING_RIGID = "RIGID_SPRING_RIGID"
    RIKS = "RIKS"
    RLE24 = "RLE24"
    RLE8 = "RLE8"
    ROLLING = "ROLLING"
    ROTARYI = "ROTARYI"
    ROTARY_INERTIA = "ROTARY_INERTIA"
    ROTATE = "ROTATE"
    ROTATION = "ROTATION"
    ROTATIONAL_ACCELERATION = "ROTATIONAL_ACCELERATION"
    ROTATIONAL_VELOCITY = "ROTATIONAL_VELOCITY"
    ROTATION_ACCELEROMETER = "ROTATION_ACCELEROMETER"
    ROTATION_ANGLE = "ROTATION_ANGLE"
    ROTATION_FIELD = "ROTATION_FIELD"
    ROTATION_NONE = "ROTATION_NONE"
    ROUGH = "ROUGH"
    RSH = "RSH"
    RSS = "RSS"
    RUNNING = "RUNNING"
    S1 = "S1"
    S2 = "S2"
    S3 = "S3"
    S3R = "S3R"
    S3RS = "S3RS"
    S3RT = "S3RT"
    S3T = "S3T"
    S4 = "S4"
    S4R = "S4R"
    S4R5 = "S4R5"
    S4RS = "S4RS"
    S4RSW = "S4RSW"
    S4RT = "S4RT"
    S4T = "S4T"
    S5 = "S5"
    S6 = "S6"
    S8R = "S8R"
    S8R5 = "S8R5"
    S8RT = "S8RT"
    SAX1 = "SAX1"
    SAX2 = "SAX2"
    SAX2T = "SAX2T"
    SAXA11 = "SAXA11"
    SAXA12 = "SAXA12"
    SAXA13 = "SAXA13"
    SAXA14 = "SAXA14"
    SAXA21 = "SAXA21"
    SAXA22 = "SAXA22"
    SAXA23 = "SAXA23"
    SAXA24 = "SAXA24"
    SC6R = "SC6R"
    SC6RT = "SC6RT"
    SC8R = "SC8R"
    SC8RT = "SC8RT"
    SCALAR = "SCALAR"
    SCALE = "SCALE"
    SCALE_FACTOR = "SCALE_FACTOR"
    SCATTERED = "SCATTERED"
    SCIENTIFIC = "SCIENTIFIC"
    SCREEN = "SCREEN"
    SCREEN_SIZE = "SCREEN_SIZE"
    SECOND = "SECOND"
    SECOND_ORDER_ADVECTION = "SECOND_ORDER_ADVECTION"
    SECTION = "SECTION"
    SECTION_MAP = "SECTION_MAP"
    SECTION_MAP_COLORS = "SECTION_MAP_COLORS"
    SECTION_PTS = "SECTION_PTS"
    SEGMENTS = "SEGMENTS"
    SELECTED = "SELECTED"
    SELECTION_GRP_MAP = "SELECTION_GRP_MAP"
    SELECTION_GRP_MAP_COLORS = "SELECTION_GRP_MAP_COLORS"
    SELECTIVE = "SELECTIVE"
    SELECT_ALL = "SELECT_ALL"
    SELECT_BY_ANGLE = "SELECT_BY_ANGLE"
    SELECT_BY_NUMBER = "SELECT_BY_NUMBER"
    SELF = "SELF"
    SEMI_AUTOMATIC = "SEMI_AUTOMATIC"
    SEND_PREDICTION = "SEND_PREDICTION"
    SEPARATE = "SEPARATE"
    SEPARATED = "SEPARATED"
    SEPARATED_SOLUTION = "SEPARATED_SOLUTION"
    SEPARATE_TABLES = "SEPARATE_TABLES"
    SEQUENTIAL_THERMAL_STRESS = "SEQUENTIAL_THERMAL_STRESS"
    SEQ_ID = "SEQ_ID"
    SET = "SET"
    SET_EQUAL_DT = "SET_EQUAL_DT"
    SET_MAP = "SET_MAP"
    SET_MAP_COLORS = "SET_MAP_COLORS"
    SEVERE_CONTACT_OVERCLOSURES = "SEVERE_CONTACT_OVERCLOSURES"
    SEVERE_OVERCLOSURES = "SEVERE_OVERCLOSURES"
    SFM3D3 = "SFM3D3"
    SFM3D4 = "SFM3D4"
    SFM3D4R = "SFM3D4R"
    SFM3D6 = "SFM3D6"
    SFM3D8 = "SFM3D8"
    SFM3D8R = "SFM3D8R"
    SFMAX1 = "SFMAX1"
    SFMAX2 = "SFMAX2"
    SFMCL6 = "SFMCL6"
    SFMCL9 = "SFMCL9"
    SFMGAX1 = "SFMGAX1"
    SFMGAX2 = "SFMGAX2"
    SHADED = "SHADED"
    SHAPE_FACTOR = "SHAPE_FACTOR"
    SHARED_LM_PRESSURE_FIELD = "SHARED_LM_PRESSURE_FIELD"
    SHARED_LM_VOLUME_FIELD = "SHARED_LM_VOLUME_FIELD"
    SHEAR = "SHEAR"
    SHEARCREEP = "SHEARCREEP"
    SHEARRELAXATION = "SHEARRELAXATION"
    SHELL = "SHELL"
    SHELL_TO_SOLID_COUPLING = "SHELL_TO_SOLID_COUPLING"
    SHORT = "SHORT"
    SHORTEST_EDGE = "SHORTEST_EDGE"
    SHORTEST_PATH = "SHORTEST_PATH"
    SHORT_FIBER = "SHORT_FIBER"
    SHRINK_FIT = "SHRINK_FIT"
    SHRINK_MOVEMENT = "SHRINK_MOVEMENT"
    SIDE1 = "SIDE1"
    SIDE2 = "SIDE2"
    SIGCONT = "SIGCONT"
    SIGINT = "SIGINT"
    SIGN = "SIGN"
    SIGTERM = "SIGTERM"
    SIGTSTP = "SIGTSTP"
    SIGUSR1 = "SIGUSR1"
    SIGUSR2 = "SIGUSR2"
    SIM = "SIM"
    SIMP = "SIMP"
    SIMPLE = "SIMPLE"
    SIMPLESHEAR = "SIMPLESHEAR"
    SIMPSON = "SIMPSON"
    SIMULATION_ABORTED = "SIMULATION_ABORTED"
    SIMULATION_COMPLETED = "SIMULATION_COMPLETED"
    SIMULATION_INTERRUPTED = "SIMULATION_INTERRUPTED"
    SIMULATION_STARTED = "SIMULATION_STARTED"
    SINE = "SINE"
    SINGHM = "SINGHM"
    SINGLE = "SINGLE"
    SINGLE_DIRECTION = "SINGLE_DIRECTION"
    SINGLE_NODE = "SINGLE_NODE"
    SINGLE_PRECISION = "SINGLE_PRECISION"
    SINGLE_TABLE = "SINGLE_TABLE"
    SINGLE_VALUE = "SINGLE_VALUE"
    SIZE = "SIZE"
    SIZE_ON_SCREEN = "SIZE_ON_SCREEN"
    SKINS = "SKINS"
    SKIN_MAP = "SKIN_MAP"
    SKIN_MAP_COLORS = "SKIN_MAP_COLORS"
    SLASH = "SLASH"
    SLAVE = "SLAVE"
    SLIDE_DISTANCE_EXCEEDED = "SLIDE_DISTANCE_EXCEEDED"
    SLIDE_PLANE = "SLIDE_PLANE"
    SLIDING = "SLIDING"
    SLIPPED_OFF_PATCH = "SLIPPED_OFF_PATCH"
    SLIPRING = "SLIPRING"
    SLOT = "SLOT"
    SMALL = "SMALL"
    SMALLEST_ELEM_AT_CENTER = "SMALLEST_ELEM_AT_CENTER"
    SMALLEST_ELEM_AT_ENDS = "SMALLEST_ELEM_AT_ENDS"
    SMALLEST_ELEM_LOCATION = "SMALLEST_ELEM_LOCATION"
    SMALL_ANGLE = "SMALL_ANGLE"
    SMALL_ANGLE_QUAD_FACE = "SMALL_ANGLE_QUAD_FACE"
    SMALL_ANGLE_TRI_FACE = "SMALL_ANGLE_TRI_FACE"
    SMEAR_ABOUT_CORE = "SMEAR_ABOUT_CORE"
    SMEAR_ALL_LAYERS = "SMEAR_ALL_LAYERS"
    SMOOTHING_AS_IS = "SMOOTHING_AS_IS"
    SMOOTHING_OFF = "SMOOTHING_OFF"
    SMOOTHING_ON = "SMOOTHING_ON"
    SNEG = "SNEG"
    SOFTWARE_OVERLAY = "SOFTWARE_OVERLAY"
    SOFT_CONTACT_INCOMPATIBILITIES = "SOFT_CONTACT_INCOMPATIBILITIES"
    SOILS = "SOILS"
    SOLID = "SOLID"
    SOLIDWORKS = "SOLIDWORKS"
    SOLUTION_APPEARS_TO_BE_DIVERGING = "SOLUTION_APPEARS_TO_BE_DIVERGING"
    SOLVER_DEFAULT = "SOLVER_DEFAULT"
    SPALART = "SPALART"
    SPECIFIED = "SPECIFIED"
    SPECIFIED_LIMIT = "SPECIFIED_LIMIT"
    SPECIFIED_MODES = "SPECIFIED_MODES"
    SPECIFIED_NODAL_DIAMETER = "SPECIFIED_NODAL_DIAMETER"
    SPECIFY = "SPECIFY"
    SPECIFY_NUM_PTS = "SPECIFY_NUM_PTS"
    SPECIFY_ORIENT = "SPECIFY_ORIENT"
    SPECIFY_PATH = "SPECIFY_PATH"
    SPECIFY_STEP_SIZE = "SPECIFY_STEP_SIZE"
    SPECIFY_THICKNESS = "SPECIFY_THICKNESS"
    SPECIFY_TOLERANCE = "SPECIFY_TOLERANCE"
    SPECTRUM = "SPECTRUM"
    SPHERE = "SPHERE"
    SPHERICAL = "SPHERICAL"
    SPIN = "SPIN"
    SPLINE = "SPLINE"
    SPLINE_APPROXIMATION = "SPLINE_APPROXIMATION"
    SPLINE_INTERPOLATION = "SPLINE_INTERPOLATION"
    SPOS = "SPOS"
    SPOT_WELD_CONTACT = "SPOT_WELD_CONTACT"
    SPRING1 = "SPRING1"
    SPRING2 = "SPRING2"
    SPRINGA = "SPRINGA"
    SPRING_RIGID_SPRING = "SPRING_RIGID_SPRING"
    SQUARE_ROOT = "SQUARE_ROOT"
    SRSS = "SRSS"
    SSH = "SSH"
    SSOR = "SSOR"
    STABILIZATION = "STABILIZATION"
    STABILIZED = "STABILIZED"
    STABLE_INC_XPL = "STABLE_INC_XPL"
    STABLE_TIME_INCREMENT = "STABLE_TIME_INCREMENT"
    STACK_1 = "STACK_1"
    STACK_2 = "STACK_2"
    STACK_3 = "STACK_3"
    STACK_ORIENTATION = "STACK_ORIENTATION"
    STAGNATION = "STAGNATION"
    STAMP = "STAMP"
    STANDALONE = "STANDALONE"
    STANDALONENOSHOW = "STANDALONENOSHOW"
    STANDARD = "STANDARD"
    STANDARD_ANALYSIS = "STANDARD_ANALYSIS"
    STANDARD_EXPLICIT = "STANDARD_EXPLICIT"
    STANDARD_PHASE = "STANDARD_PHASE"
    START = "START"
    STARTED = "STARTED"
    STATE_FRAME = "STATE_FRAME"
    STATIC = "STATIC"
    STATIC_GENERAL = "STATIC_GENERAL"
    STATIC_LINEAR_PERTURBATION = "STATIC_LINEAR_PERTURBATION"
    STATIC_PERTURBATION = "STATIC_PERTURBATION"
    STATIC_RIKS = "STATIC_RIKS"
    STATUS = "STATUS"
    STEADY_STATE = "STEADY_STATE"
    STEADY_STATE_DIRECT = "STEADY_STATE_DIRECT"
    STEADY_STATE_MODAL = "STEADY_STATE_MODAL"
    STEADY_STATE_SUBSPACE = "STEADY_STATE_SUBSPACE"
    STEADY_STATE_TRANSPORT = "STEADY_STATE_TRANSPORT"
    STEP = "STEP"
    STEP_END = "STEP_END"
    STEP_START = "STEP_START"
    STEP_TIME = "STEP_TIME"
    STEP_TIME_XPL = "STEP_TIME_XPL"
    STIFFNESS = "STIFFNESS"
    STIFFNESS_OPTIMIZATION = "STIFFNESS_OPTIMIZATION"
    STIPPLED = "STIPPLED"
    STOP = "STOP"
    STRAIN = "STRAIN"
    STRAIN_FREE_CORRECTIONS = "STRAIN_FREE_CORRECTIONS"
    STRAIN_RATE = "STRAIN_RATE"
    STRESS = "STRESS"
    STRESS_INTENS_FACTOR = "STRESS_INTENS_FACTOR"
    STRESS_PERTURBATION = "STRESS_PERTURBATION"
    STRI3 = "STRI3"
    STRI65 = "STRI65"
    STRINGERS = "STRINGERS"
    STRINGER_MAP = "STRINGER_MAP"
    STRINGER_MAP_COLORS = "STRINGER_MAP_COLORS"
    STRUCTURAL = "STRUCTURAL"
    STRUCTURED = "STRUCTURED"
    SUBMISSION_ABORTED = "SUBMISSION_ABORTED"
    SUBMITTED = "SUBMITTED"
    SUBMODELING = "SUBMODELING"
    SUBSPACE = "SUBSPACE"
    SUBSPACE_DYNAMIC = "SUBSPACE_DYNAMIC"
    SUBSPACE_EIGENSOLVER = "SUBSPACE_EIGENSOLVER"
    SUBSPACE_STEADY_STATE_DYNAMIC = "SUBSPACE_STEADY_STATE_DYNAMIC"
    SUBSTANCE = "SUBSTANCE"
    SUBSTRUCTURE_GENERATE = "SUBSTRUCTURE_GENERATE"
    SUBSTRUCTURING = "SUBSTRUCTURING"
    SUBTRACT = "SUBTRACT"
    SUM = "SUM"
    SUPERIMPOSE = "SUPERIMPOSE"
    SUPPRESS = "SUPPRESS"
    SURFACE = "SURFACE"
    SURFACE_BLAST = "SURFACE_BLAST"
    SURFACE_INTEGRATION_POINT = "SURFACE_INTEGRATION_POINT"
    SURFACE_MAP = "SURFACE_MAP"
    SURFACE_MAP_COLORS = "SURFACE_MAP_COLORS"
    SURFACE_NODAL = "SURFACE_NODAL"
    SURFACE_POINT_EQUIV_STRESS = "SURFACE_POINT_EQUIV_STRESS"
    SURFACE_TO_SURFACE = "SURFACE_TO_SURFACE"
    SVG = "SVG"
    SWEEP = "SWEEP"
    SWING = "SWING"
    SYMBOL = "SYMBOL"
    SYMBOLS_ON_DEF = "SYMBOLS_ON_DEF"
    SYMBOLS_ON_UNDEF = "SYMBOLS_ON_UNDEF"
    SYMMETRIC = "SYMMETRIC"
    SYMMETRIC_MODEL_GENERATION = "SYMMETRIC_MODEL_GENERATION"
    SYNTAXCHECK = "SYNTAXCHECK"
    SYSTEM = "SYSTEM"
    SYSTEM_ASSIGN = "SYSTEM_ASSIGN"
    SYSTEM_DEFINED = "SYSTEM_DEFINED"
    T2D2 = "T2D2"
    T2D2E = "T2D2E"
    T2D2H = "T2D2H"
    T2D2T = "T2D2T"
    T2D3 = "T2D3"
    T2D3E = "T2D3E"
    T2D3H = "T2D3H"
    T2D3T = "T2D3T"
    T3D2 = "T3D2"
    T3D2E = "T3D2E"
    T3D2H = "T3D2H"
    T3D2T = "T3D2T"
    T3D3 = "T3D3"
    T3D3E = "T3D3E"
    T3D3H = "T3D3H"
    T3D3T = "T3D3T"
    TABULAR = "TABULAR"
    TANGENT = "TANGENT"
    TANGENTIAL = "TANGENTIAL"
    TAPERED = "TAPERED"
    TASK_REGION_EQUIV_STRESS = "TASK_REGION_EQUIV_STRESS"
    TASK_REGION_LAYERS = "TASK_REGION_LAYERS"
    TECHNIQUE = "TECHNIQUE"
    TEMPERATURE = "TEMPERATURE"
    TEMPERATURE_FALLEN_BELOW_ABSOLUTE_ZERO = "TEMPERATURE_FALLEN_BELOW_ABSOLUTE_ZERO"
    TEMPERATURE_FIELD = "TEMPERATURE_FIELD"
    TENP = "TENP"
    TENSION = "TENSION"
    TENSOR = "TENSOR"
    TENSOR_2D_PLANAR = "TENSOR_2D_PLANAR"
    TENSOR_2D_SURFACE = "TENSOR_2D_SURFACE"
    TENSOR_3D_FULL = "TENSOR_3D_FULL"
    TENSOR_3D_PLANAR = "TENSOR_3D_PLANAR"
    TENSOR_3D_SURFACE = "TENSOR_3D_SURFACE"
    TERMINATED = "TERMINATED"
    TESSELLATED = "TESSELLATED"
    TEST_DATA = "TEST_DATA"
    TET = "TET"
    TET10 = "TET10"
    TET4 = "TET4"
    TEXTURE_MAPPED = "TEXTURE_MAPPED"
    THERMAL = "THERMAL"
    THERMAL_CONTACT = "THERMAL_CONTACT"
    THERMOMECHANICAL = "THERMOMECHANICAL"
    THICK = "THICK"
    THICKNESS = "THICKNESS"
    THICKNESS_DISCRETE_FIELD = "THICKNESS_DISCRETE_FIELD"
    THICKNESS_MAGNITUDE = "THICKNESS_MAGNITUDE"
    THICK_WALL = "THICK_WALL"
    THIN = "THIN"
    THINNING = "THINNING"
    THIN_WALL = "THIN_WALL"
    THREADS = "THREADS"
    THREED_DATA = "THREED_DATA"
    THREE_D = "THREE_D"
    THROUGHOUT_STEP = "THROUGHOUT_STEP"
    TIE = "TIE"
    TIE_MPC = "TIE_MPC"
    TIFF = "TIFF"
    TIGHTEN_GAPS = "TIGHTEN_GAPS"
    TIME = "TIME"
    TIME_AVERAGE = "TIME_AVERAGE"
    TIME_BASED = "TIME_BASED"
    TIME_HEAT_FLUX = "TIME_HEAT_FLUX"
    TIME_HEAT_FLUX_AREA = "TIME_HEAT_FLUX_AREA"
    TIME_HISTORY = "TIME_HISTORY"
    TIME_INCREMENT = "TIME_INCREMENT"
    TIME_INTEGRATION = "TIME_INTEGRATION"
    TIME_INTEGRATION_ACCURACY_LIMIT_EXCEEDED = "TIME_INTEGRATION_ACCURACY_LIMIT_EXCEEDED"
    TIME_INTEGRATION_STABILITY_LIMIT_EXCEEDED = "TIME_INTEGRATION_STABILITY_LIMIT_EXCEEDED"
    TIME_INTERVAL = "TIME_INTERVAL"
    TIME_INTERVAL_VALUE = "TIME_INTERVAL_VALUE"
    TIME_POINT = "TIME_POINT"
    TIME_VOLUME = "TIME_VOLUME"
    TIME_VOLUME_FLUX = "TIME_VOLUME_FLUX"
    TMORI = "TMORI"
    TOLERANCE = "TOLERANCE"
    TOP = "TOP"
    TOP_AND_BOTTOM = "TOP_AND_BOTTOM"
    TOP_CENTER = "TOP_CENTER"
    TOP_LEFT = "TOP_LEFT"
    TOP_RIGHT = "TOP_RIGHT"
    TOP_SURFACE = "TOP_SURFACE"
    TOROIDAL = "TOROIDAL"
    TOTAL = "TOTAL"
    TOTAL_ABSOLUTE_MOVEMENT = "TOTAL_ABSOLUTE_MOVEMENT"
    TOTAL_FORCE = "TOTAL_FORCE"
    TOTAL_MASS = "TOTAL_MASS"
    TOTAL_MASS_XPL = "TOTAL_MASS_XPL"
    TOTAL_NUM = "TOTAL_NUM"
    TOTAL_NUMBER = "TOTAL_NUMBER"
    TOTAL_TIME_XPL = "TOTAL_TIME_XPL"
    TO_ENVIRONMENT = "TO_ENVIRONMENT"
    TRACTION = "TRACTION"
    TRACTION_SEPARATION = "TRACTION_SEPARATION"
    TRANSFORMATION = "TRANSFORMATION"
    TRANSIENT = "TRANSIENT"
    TRANSIENT_FIDELITY = "TRANSIENT_FIDELITY"
    TRANSLATE = "TRANSLATE"
    TRANSLATOR = "TRANSLATOR"
    TRANSPARENT = "TRANSPARENT"
    TRANSVERSE = "TRANSVERSE"
    TRANSVERSE_SHEAR_FORCE_CONSTRAINT = "TRANSVERSE_SHEAR_FORCE_CONSTRAINT"
    TRAPEZOID = "TRAPEZOID"
    TRESCA = "TRESCA"
    TRI = "TRI"
    TRIGGER = "TRIGGER"
    TRI3 = "TRI3"
    TRI6 = "TRI6"
    TRIANGLES = "TRIANGLES"
    TRIANGLE_FAN = "TRIANGLE_FAN"
    TRIANGLE_STRIP = "TRIANGLE_STRIP"
    TRIDIRECTIONAL = "TRIDIRECTIONAL"
    TRUE_DISTANCE = "TRUE_DISTANCE"
    TRUE_DISTANCE_X = "TRUE_DISTANCE_X"
    TRUE_DISTANCE_Y = "TRUE_DISTANCE_Y"
    TRUE_DISTANCE_Z = "TRUE_DISTANCE_Z"
    TRUSS = "TRUSS"
    TURB_NONE = "TURB_NONE"
    TURN = "TURN"
    TWIST = "TWIST"
    TWO_CONFIG = "TWO_CONFIG"
    TWO_D_PLANAR = "TWO_D_PLANAR"
    TYPED_IN = "TYPED_IN"
    TYPE_NOT_APPLICABLE = "TYPE_NOT_APPLICABLE"
    T_STRESS = "T_STRESS"
    U1 = "U1"
    U2 = "U2"
    U3 = "U3"
    UJOINT = "UJOINT"
    UNCHANGED = "UNCHANGED"
    UNCONVERGED_CAP_PLASTICITY = "UNCONVERGED_CAP_PLASTICITY"
    UNCONVERGED_CLAY_PLASTICITY = "UNCONVERGED_CLAY_PLASTICITY"
    UNCONVERGED_CONCRETE_PLASTICITY = "UNCONVERGED_CONCRETE_PLASTICITY"
    UNCONVERGED_CREEP_PLASTICITY = "UNCONVERGED_CREEP_PLASTICITY"
    UNCONVERGED_DRUCKER_PRAGER_PLASTICITY = "UNCONVERGED_DRUCKER_PRAGER_PLASTICITY"
    UNCONVERGED_FOAM_PLASTICITY = "UNCONVERGED_FOAM_PLASTICITY"
    UNCONVERGED_HYPERELASTICITY = "UNCONVERGED_HYPERELASTICITY"
    UNCONVERGED_METAL_PLASTICITY = "UNCONVERGED_METAL_PLASTICITY"
    UNCONVERGED_MOHR_COULOMB_PLASTICITY = "UNCONVERGED_MOHR_COULOMB_PLASTICITY"
    UNCORRELATED = "UNCORRELATED"
    UNCOUPLED = "UNCOUPLED"
    UNDEFINED_ANALYSIS = "UNDEFINED_ANALYSIS"
    UNDEFINED_INVARIANT = "UNDEFINED_INVARIANT"
    UNDEFINED_POSITION = "UNDEFINED_POSITION"
    UNDEFORMED = "UNDEFORMED"
    UNDEX = "UNDEX"
    UNDEX_CHARGE = "UNDEX_CHARGE"
    UNIAXIAL = "UNIAXIAL"
    UNIAXIAL_VOLUMETRIC = "UNIAXIAL_VOLUMETRIC"
    UNIDIRECTIONAL = "UNIDIRECTIONAL"
    UNIFORM = "UNIFORM"
    UNIFORM_BY_NUMBER = "UNIFORM_BY_NUMBER"
    UNIFORM_BY_SIZE = "UNIFORM_BY_SIZE"
    UNIFORM_ERROR = "UNIFORM_ERROR"
    UNIFORM_SPACING = "UNIFORM_SPACING"
    UNION = "UNION"
    UNIVERSAL = "UNIVERSAL"
    UNIX = "UNIX"
    UNKNOWN = "UNKNOWN"
    UNKNOWNAXIS = "UNKNOWNAXIS"
    UNKNOWNPLANE = "UNKNOWNPLANE"
    UNKNOWN_ANALYSIS_CODE = "UNKNOWN_ANALYSIS_CODE"
    UNKNOWN_DIMENSION = "UNKNOWN_DIMENSION"
    UNKNOWN_HEX = "UNKNOWN_HEX"
    UNKNOWN_HOURGLASS_CONTROL = "UNKNOWN_HOURGLASS_CONTROL"
    UNKNOWN_KINEMATIC_SPLIT = "UNKNOWN_KINEMATIC_SPLIT"
    UNKNOWN_PHASE = "UNKNOWN_PHASE"
    UNKNOWN_QUAD = "UNKNOWN_QUAD"
    UNKNOWN_SHAPE = "UNKNOWN_SHAPE"
    UNKNOWN_STRESS_RATE = "UNKNOWN_STRESS_RATE"
    UNKNOWN_TET = "UNKNOWN_TET"
    UNKNOWN_TRI = "UNKNOWN_TRI"
    UNKNOWN_WEDGE = "UNKNOWN_WEDGE"
    UNLIMITED = "UNLIMITED"
    UNMESHABLE = "UNMESHABLE"
    UNRESOLVED_INITIAL_OVERCLOSURES = "UNRESOLVED_INITIAL_OVERCLOSURES"
    UNSET = "UNSET"
    UNSPECIFIED = "UNSPECIFIED"
    UNSYMMETRIC = "UNSYMMETRIC"
    UR1 = "UR1"
    UR2 = "UR2"
    UR3 = "UR3"
    USA_ADDED_MASS_GENERATION = "USA_ADDED_MASS_GENERATION"
    USER = "USER"
    USER_CUSTOMIZED = "USER_CUSTOMIZED"
    USER_DEFINED = "USER_DEFINED"
    USER_MPC = "USER_MPC"
    USER_SPECIFIED = "USER_SPECIFIED"
    USER_SUB = "USER_SUB"
    USER_SUBROUTINE = "USER_SUBROUTINE"
    USER_SUBROUTINE_REQUEST = "USER_SUBROUTINE_REQUEST"
    USE_BOTTOM = "USE_BOTTOM"
    USE_BOTTOM_AND_TOP = "USE_BOTTOM_AND_TOP"
    USE_ENVELOPE = "USE_ENVELOPE"
    USE_GEOMETRY = "USE_GEOMETRY"
    USE_MESH = "USE_MESH"
    USE_TOP = "USE_TOP"
    USUP = "USUP"
    VALUE = "VALUE"
    VALUES = "VALUES"
    VALUES_AND_HISTORY = "VALUES_AND_HISTORY"
    VAN_DER_WAALS = "VAN_DER_WAALS"
    VAN_DER_WALLS_STRETCHES_LOCKING = "VAN_DER_WALLS_STRETCHES_LOCKING"
    VARIABLE_RATIO = "VARIABLE_RATIO"
    VCCT = "VCCT"
    VECTOR = "VECTOR"
    VECTOR_COMPONENT = "VECTOR_COMPONENT"
    VELOCITY = "VELOCITY"
    VELOCITY_PARABOLIC = "VELOCITY_PARABOLIC"
    VELOCITY_SQUARED = "VELOCITY_SQUARED"
    VERTEX_ADJ_TO_SMALLEST_ELEM = "VERTEX_ADJ_TO_SMALLEST_ELEM"
    VERTICAL = "VERTICAL"
    VERY_SMALL = "VERY_SMALL"
    VERY_THIN = "VERY_THIN"
    VIEW_MANIP = "VIEW_MANIP"
    VISCO = "VISCO"
    VISCOUS = "VISCOUS"
    VMS = "VMS"
    VOID = "VOID"
    VOLUME = "VOLUME"
    VOLUMETRIC = "VOLUMETRIC"
    VOLUMETRICCREEP = "VOLUMETRICCREEP"
    VOLUMETRICRELAXATION = "VOLUMETRICRELAXATION"
    VOLUMETRIC_DATA = "VOLUMETRIC_DATA"
    VOLUME_COMPRESSION = "VOLUME_COMPRESSION"
    VOLUME_FLUX = "VOLUME_FLUX"
    VOLUME_FLUX_AREA = "VOLUME_FLUX_AREA"
    VOLUME_FRACTION = "VOLUME_FRACTION"
    VOLUME_PROPORTIONAL = "VOLUME_PROPORTIONAL"
    VOL_FLUX = "VOL_FLUX"
    VOL_RATE_LEAK = "VOL_RATE_LEAK"
    VRML = "VRML"
    WARNING = "WARNING"
    WARP2D3 = "WARP2D3"
    WARP2D4 = "WARP2D4"
    WEDGE = "WEDGE"
    WEDGE15 = "WEDGE15"
    WEDGE6 = "WEDGE6"
    WEIGHTED = "WEIGHTED"
    WEIGHTED_ADD = "WEIGHTED_ADD"
    WELD = "WELD"
    WHITE_TO_BLACK = "WHITE_TO_BLACK"
    WHOLE_ELEMENT = "WHOLE_ELEMENT"
    WHOLE_MODEL = "WHOLE_MODEL"
    WHOLE_PART_INSTANCE = "WHOLE_PART_INSTANCE"
    WHOLE_REGION = "WHOLE_REGION"
    WHOLE_SURFACE = "WHOLE_SURFACE"
    WIDTH = "WIDTH"
    WINDOWS = "WINDOWS"
    WIRE = "WIRE"
    WIREFRAME = "WIREFRAME"
    WLF = "WLF"
    WRAP_AROUND = "WRAP_AROUND"
    X11 = "X11"
    XASYMM = "XASYMM"
    XAXIS = "XAXIS"
    XMARKER = "XMARKER"
    XOR = "XOR"
    XSYMM = "XSYMM"
    XYPLANE = "XYPLANE"
    XYZ = "XYZ"
    XZPLANE = "XZPLANE"
    YASYMM = "YASYMM"
    YAXIS = "YAXIS"
    YEOH = "YEOH"
    YIELD_RATIO = "YIELD_RATIO"
    YSYMM = "YSYMM"
    YZPLANE = "YZPLANE"
    ZASYMM = "ZASYMM"
    ZAXIS = "ZAXIS"
    ZERO = "ZERO"
    ZERO_PRESSURE = "ZERO_PRESSURE"
    ZSYMM = "ZSYMM"
    X_COORDINATE = "X_COORDINATE"
    Y_COORDINATE = "Y_COORDINATE"
    Z_COORDINATE = "Z_COORDINATE"
    AVERAGED_AT_NODES = "AVERAGED_AT_NODES"
    CENTROIDAL = "CENTROIDAL"
    K110 = "K110"
    UNARY_NEGATIVE = "UNARY_NEGATIVE"
    HYPERBOLIC_COSINE = "HYPERBOLIC_COSINE"
    INVERSE_COSINE = "INVERSE_COSINE"
    INVERSE_SINE = "INVERSE_SINE"
    HYPERBOLIC_TANGENT = "HYPERBOLIC_TANGENT"
    INVERSE_TANGENT = "INVERSE_TANGENT"
    NORMALIZE = "NORMALIZE"
    DEG2RAD = "DEG2RAD"
    RAD2DEG = "RAD2DEG"
    SMOOTH = "SMOOTH"
    SWAP = "SWAP"
    AVERAGE_ALL = "AVERAGE_ALL"
    MAXIMUM_ENVELOPE = "MAXIMUM_ENVELOPE"
    MINIMUM_ENVELOPE = "MINIMUM_ENVELOPE"
    RANGE_ALL = "RANGE_ALL"
    SS = "SS"
    TOKEN = "TOKEN"
    CREDIT = "CREDIT"
    REUSS = "REUSS"
    VOIGT = "VOIGT"
    INVERSED_MT = "INVERSED_MT"
    EISO = "EISO"
    PISO = "PISO"
    IGES = "IGES"
    ANALYTICAL_FIELD_THICKNESS = "ANALYTICAL_FIELD_THICKNESS"
    NORMAL_DATUM = "NORMAL_DATUM"
    PRIMARY_DATUM = "PRIMARY_DATUM"
    COMPLEX_MAG_AT_ANGLE = "COMPLEX_MAG_AT_ANGLE"
    VALANIS_LANDEL = "VALANIS_LANDEL"
    ELLIPTIC_CYLINDER = "ELLIPTIC_CYLINDER"
    PENNY = "PENNY"
    ORIENTATION_TENSOR = "ORIENTATION_TENSOR"
    RANDOM3D = "RANDOM3D"
    OBLATE = "OBLATE"
    UMAT = "UMAT"
    BRITTLE = "BRITTLE"
    DUCTILE = "DUCTILE"
    NT11 = "NT11"
    FLUID_PRESSURE = "FLUID_PRESSURE"
    WARP = "WARP"
    ELECTRICAL_POTENTIAL = "ELECTRICAL_POTENTIAL"
    NN30 = "NN30"
    NN11 = "NN11"
    NT30 = "NT30"
    STRING = "STRING"
    ELECTROMAGNETIC = "ELECTROMAGNETIC"
    JUSTIFY_CENTER = "JUSTIFY_CENTER"
    JUSTIFY_RIGHT = "JUSTIFY_RIGHT"
    TWO_SIDED = "TWO_SIDED"
    SMALL_SLIDING = "SMALL_SLIDING"
    NO_CRUSH = "NO_CRUSH"
    NO_TRIGGER = "NO_TRIGGER"
    LOCATION = "LOCATION"
    DEPENDENT = "DEPENDENT"
    MAIN = "MAIN"
    SECONDARY = "SECONDARY"
    ELEMENT_ORDER_SMOOTHING = "ELEMENT_ORDER_SMOOTHING"
    LINEAR_SMOOTHING = "LINEAR_SMOOTHING"
    QUADRATIC_SMOOTHING = "QUADRATIC_SMOOTHING"
    NO_VERTICES = "NO_VERTICES"
    ALL_VERTICES = "ALL_VERTICES"
    SIMULATION_SUBMITTED = "SIMULATION_SUBMITTED"
    HEALER_JOB = "HEALER_JOB"
    COMPLEX_ENVELOPE_MAX_ABS = "COMPLEX_ENVELOPE_MAX_ABS"
    COMPLEX_ENVELOPE_MIN = "COMPLEX_ENVELOPE_MIN"
    COMPLEX_ENVELOPE_MAX = "COMPLEX_ENVELOPE_MAX"
    TRANSFORM = "TRANSFORM"
    WIRE_FRAME = "WIRE_FRAME"
    FIELD_ANALYTICAL = "FIELD_ANALYTICAL"
    EXTRA = "EXTRA"
    API = "API"
    ALLOW = "ALLOW"
    BYPASS = "BYPASS"
    FACET = "FACET"
    S11 = "S11"
    STEEL = "STEEL"
    S22 = "S22"
