import time
from contextlib import suppress
from logging import Logger
from pypomes_core import timestamp_duration
from typing import Any

from .db_common import (
    DbEngine, _bind_columns, _bind_marks, _except_msg, _remove_nulls
)
from .db_pomes import db_connect, db_close, db_count


# ruff: noqa: C901 (function is too complex)
# ruff: noqa: PLR0912 (checks for functions or methods with too many statements)
# ruff: noqa: PLR0915 (too many statements)
def db_migrate_data(source_engine: DbEngine,
                    source_table: str,
                    source_columns: list[str],
                    target_engine: DbEngine,
                    target_table: str,
                    target_columns: list[str] = None,
                    source_conn: Any = None,
                    target_conn: Any = None,
                    source_committable: bool = None,
                    target_committable: bool = None,
                    where_clause: str = None,
                    orderby_clause: str = None,
                    offset_count: int = None,
                    limit_count: int = None,
                    batch_size_in: int = None,
                    batch_size_out: int = None,
                    identity_column: str = None,
                    has_nulls: bool = None,
                    errors: list[str] = None,
                    logger: Logger = None) -> int | None:
    """
    Migrate data from one database to another database.

    The origin and destination databases must be in the list of databases configured and
    supported by this package. The specified database columns must not be type LOB
    (large binary object), and must have the same cardinality, and be of respectively
    equivalent types, in both the origin and the destination databases.

    Specific handling is required for identity columns (i.e., columns whose values are generated
    directly by the database engine - typically, they are also primary keys), and thus they must be
    identified by *identity_column*. For target Oracle databases, *GENERATED ALWAYS*
    identity columns should be ommited, as attempting to insert values therein raises an error,
    whereas *GENERATED BY DEFAULT AS IDENTITY* columns are acceptable.

    The parameter *has_nulls* indicates that one or more columns holding string content in the source
    table have embedded NULL characters. Some databases, particularly PostgreSQL, do not accept NULLs
    embedded in text, so they must be removed.

    If not positive integers, *limit_count*, *batch_size_in*, and *chunk_size_out* are ignored.
    Care should be exercised when specifying *offset_count*, so as not to skip wanted tuples, as it is used
    to offset the retrieval of tuples. If it is set to -1, its value is set from a *COUNT* operation in
    *target_table*, with *where_clause* if specified. All other negative values are ignored.

    The parameters *batch_size_in* and *limit_count* are used to limit the maximum number of tuples
    to retrieve in each SELECT operation, and *batch_size_out* indicates the maximum number of tuples
    to write in each INSERT operation.

    Further, if *batch_size_in* and *limit_count* are both defined, and *batch_size_in* is set to a larger
    value, it is reduced accordingly. Likewise, if *batch_size_in* and *batch_size_out* are both defined,
    and *batch_size_out* is set to a larger value, it is reduced accordingly. Finally, for both
    *offset_count* and *batch_size_in* or *limit_count* to be used together with SQLServer, an *ORDER BY*
    clause must have been specifed, otherwise a runtime error is raised.

    The parameters *source_committable* and *target_committable* are relevant only if
    *source_conn* and *target_conn* are provided, respectively, and are otherwise ignored.
    A rollback is always attempted, if an error occurs.

    :param source_engine: the source database engine type
    :param source_table: the possibly schema-qualified table to read the data from
    :param source_columns: the columns to be migrated
    :param target_engine: the destination database engine type
    :param target_table: the possibly schema-qualified table to write the data to
    :param target_columns: the columns to write the data to (defaults to the source columns)
    :param source_conn: the connection to the source database (obtains a new one, if not provided)
    :param target_conn: the connection to the destination database (obtains a new one, if not provided)
    :param source_committable: whether to commit on *source_conn* upon errorless completion
    :param target_committable: whether to commit on *target_conn* upon errorless completion
    :param where_clause: optional criteria for tuple selection
    :param orderby_clause: optional retrieval order
    :param offset_count: number of tuples to skip in source table (defaults to none)
    :param limit_count: maximum number of tuples to migrate (defaults to no limit)
    :param batch_size_in: maximum number of tuples to read in each batch (defaults to no limit)
    :param batch_size_out: maximum number of tuples to write in each batch (defaults to no limit)
    :param identity_column: column whose values are generated by the database
    :param has_nulls: indicates that one or more string-type source columns have embedded nulls
    :param errors: incidental error messages
    :param logger: optional logger
    :return: the number of tuples effectively migrated, or *None* if error
    """
    # initialize the return variable
    result: int | None = 0

    # make sure to have an errors list
    if not isinstance(errors, list):
        errors = []

    # make sure to have connections to the source and destination databases
    curr_source_conn: Any = source_conn or db_connect(engine=source_engine,
                                                      errors=errors,
                                                      logger=logger)
    curr_target_conn: Any = target_conn or db_connect(engine=target_engine,
                                                      errors=errors,
                                                      logger=logger)
    if not errors:

        # define the number of rows to skip
        if offset_count == -1:
            offset_count = db_count(table=target_table,
                                    where_clause=where_clause,
                                    engine=target_engine,
                                    connection=target_conn,
                                    committable=target_committable,
                                    errors=errors,
                                    logger=logger)
        else:
            offset_count = __normalize_value(value=offset_count)

        # normalize these parameters
        limit_count = __normalize_value(value=limit_count)
        batch_size_in = __normalize_value(value=batch_size_in)
        batch_size_out = __normalize_value(value=batch_size_out)
        if 0 < limit_count < batch_size_in:
            batch_size_in = limit_count
            if logger:
                logger.debug(msg=f"Value of input batch size changed to {batch_size_in}")
        if 0 < batch_size_in < batch_size_out:
            batch_size_out = batch_size_in
            if logger:
                logger.debug(msg=f"Value of output batch size changed to {batch_size_out}")

        # make sure to have target columns
        if not target_columns:
            target_columns = source_columns

        # buid the SELECT query:
        #   SELECT * FROM <table>
        #     WHERE <where-clause>
        #     ORDER BY <column>
        #     FETCH FIRST <n> ROWS ONLY
        #   or
        #   SELECT * FROM <table>
        #     WHERE <where-clause>
        #     ORDER BY <column>
        #     OFFSET <n> ROWS
        #     FETCH NEXT <n> ROWS ONLY
        cols: str = ", ".join(source_columns)
        sel_stmt: str = f"SELECT {cols} FROM {source_table}"
        if where_clause:
            sel_stmt += f" WHERE {where_clause}"
        if orderby_clause:
            sel_stmt += f" ORDER BY {orderby_clause}"
        sel_stmt += " OFFSET @offset ROWS"

        if limit_count or batch_size_in:
            if source_engine == DbEngine.POSTGRES:
                sel_stmt += " LIMIT @limit"
            elif source_engine in [DbEngine.ORACLE, DbEngine.SQLSERVER]:
                sel_stmt += " FETCH @limit ROWS ONLY"

        # build the INSERT query
        if target_engine == DbEngine.POSTGRES:
            values: str = "VALUES %s"
            # pre-insert handling of identity columns on PostgreSQL
            if identity_column:
                values = " OVERRIDING SYSTEM VALUE " + values
        else:
            bind_marks: str = _bind_marks(engine=target_engine,
                                          start=1,
                                          finish=len(target_columns) + 1)
            values: str = f"VALUES({bind_marks})"
        cols = ", ".join(target_columns)
        insert_stmt: str = f"INSERT INTO {target_table} ({cols}) {values}"
        row_count: int = 0

        # log the migration start
        if logger:
            logger.debug(msg=f"Started migrating data, "
                             f"from {source_engine}.{source_table} to {target_engine}.{target_table}, "
                             f"limit {limit_count}, offset {offset_count + row_count}, "
                             f"input batch size {batch_size_in}, output batch size {batch_size_out}")

        # build the template to handle inserts of null values in PostgreSQL
        bulk_template: str | None = None
        if target_engine == DbEngine.POSTGRES:
            from . import postgres_pomes
            bulk_template = postgres_pomes.build_typified_template(insert_stmt=insert_stmt,
                                                                   nullable_only=True,
                                                                   conn=curr_target_conn,
                                                                   errors=errors,
                                                                   logger=logger)
        # pre-insert handling of identity columns on SQLServer
        if not errors and identity_column and target_engine == DbEngine.SQLSERVER:
            from . import sqlserver_pomes
            sqlserver_pomes.identity_pre_insert(insert_stmt=insert_stmt,
                                                conn=target_conn,
                                                errors=errors,
                                                logger=logger)
        err_msg: str | None = None
        # errors ?
        if not errors:
            # no, migrate the data
            try:
                source_cursor: Any = curr_source_conn.cursor()
                target_cursor: Any = curr_target_conn.cursor()
                if target_engine == DbEngine.SQLSERVER:
                    target_cursor.fast_executemany = True

                # adjust the offset and limit
                curr_stmt: str = sel_stmt
                curr_limit: int = min(batch_size_in, limit_count)
                if curr_limit == 0:
                    curr_limit = max(batch_size_in, limit_count)
                if curr_limit:
                    curr_stmt = curr_stmt.replace("@limit", str(curr_limit), 1)
                else:
                    curr_stmt = curr_stmt.replace(" LIMIT @limit", "", 1)
                    curr_stmt = curr_stmt.replace(" FETCH @limit ROWS ONLY", "", 1)
                if offset_count:
                    curr_stmt = curr_stmt.replace("@offset", str(offset_count), 1)\
                                         .replace(" FETCH ", " FETCH NEXT ", 1)
                else:
                    curr_stmt = curr_stmt.replace(" OFFSET @offset ROWS", "", 1)\
                                         .replace(" FETCH ", " FETCH FIRST ", 1)

                # execute the query and retrieve the result set
                source_cursor.execute(statement=curr_stmt)
                rows_in: list[tuple] = source_cursor.fetchall()

                # traverse the result set
                while rows_in:
                    # log the retrieval operation
                    if logger:
                        logger.debug(msg=f"Read {len(rows_in)} tuples "
                                         f"from {source_engine}.{source_table}, "
                                         f"offset {offset_count + row_count}, "
                                         f"connection {id(curr_source_conn)}")
                    pos_from: int = 0

                    # migrate the tuples
                    while pos_from < len(rows_in):
                        pos_to: int = min(pos_from + batch_size_out, len(rows_in)) \
                                      if batch_size_out else len(rows_in)
                        rows_out: list[tuple] = rows_in[pos_from:pos_to]
                        pos_from = pos_to
                        match target_engine:
                            case DbEngine.MYSQL:
                                pass
                            case DbEngine.ORACLE:
                                target_cursor.executemany(statement=insert_stmt,
                                                          parameters=rows_out)
                            case DbEngine.POSTGRES:
                                from psycopg2.extras import execute_values
                                if has_nulls:
                                    rows_out = _remove_nulls(rows=rows_out)
                                execute_values(cur=target_cursor,
                                               sql=insert_stmt,
                                               template=bulk_template,
                                               argslist=rows_out)
                            case DbEngine.SQLSERVER:
                                target_cursor.executemany(insert_stmt,
                                                          rows_out)

                        # increment the tuple migration counter and log the partial migration
                        row_count += len(rows_out)
                        if logger:
                            logger.debug(msg=f"Migrated {len(rows_out)} tuples, "
                                             f"from {source_engine}.{source_table}, "
                                             f"connection {id(curr_source_conn)} "
                                             f"to {target_engine}.{target_table}, "
                                             f"connection {id(curr_target_conn)}")
                    # read the next batch
                    if limit_count > row_count or (batch_size_in and not limit_count):
                        curr_limit = min(batch_size_in, limit_count - row_count)
                        if curr_limit <= 0:
                            curr_limit = max(batch_size_in, limit_count - row_count)
                        curr_stmt = sel_stmt.replace("@offset", str(offset_count + row_count), 1)\
                                            .replace("@limit", str(curr_limit), 1)\
                                            .replace(" FETCH ", " FETCH NEXT ", 1)
                        source_cursor.execute(statement=curr_stmt)
                        rows_in = source_cursor.fetchall()
                    else:
                        # signal end of migration
                        rows_in = []

                # close the cursors
                source_cursor.close()
                target_cursor.close()

                # post-insert handling of identity columns
                if identity_column and row_count > 0:
                    if target_engine == DbEngine.POSTGRES:
                        from . import postgres_pomes
                        postgres_pomes.identity_post_insert(insert_stmt=insert_stmt,
                                                            conn=target_conn,
                                                            committable=False,
                                                            identity_column=identity_column,
                                                            errors=errors,
                                                            logger=logger)
                    elif target_engine == DbEngine.SQLSERVER:
                        from . import sqlserver_pomes
                        sqlserver_pomes.identity_post_insert(insert_stmt=insert_stmt,
                                                             conn=target_conn,
                                                             committable=False,
                                                             identity_column=identity_column,
                                                             errors=errors,
                                                             logger=logger)
                if errors:
                    if curr_source_conn:
                        with suppress(Exception):
                            curr_source_conn.rollback()
                else:
                    # commit the transactions
                    if source_committable or not source_conn:
                        curr_source_conn.commit()
                    if target_committable or not target_conn:
                        curr_target_conn.commit()
            except Exception as e:
                # rollback the transactions
                if curr_source_conn:
                    with suppress(Exception):
                        curr_source_conn.rollback()
                if curr_target_conn:
                    with suppress(Exception):
                        curr_target_conn.rollback()
                err_msg = _except_msg(exception=e,
                                      engine=source_engine)
            finally:
                # close the connections, if locally acquired
                if curr_source_conn and not source_conn:
                    db_close(connection=curr_source_conn,
                             logger=logger)
                if curr_target_conn and not target_conn:
                    db_close(connection=curr_source_conn,
                             logger=logger)

        # log the migration finish
        if err_msg:
            errors.append(err_msg)
            if logger:
                logger.error(msg=err_msg)

        if not errors:
            if logger:
                logger.debug(msg=f"Finished migrating {row_count} tuples, "
                                 f"from {source_engine}.{source_table} "
                                 f"to {target_engine}.{target_table}, offset {offset_count}")
            result = row_count

    return result


# ruff: noqa: PLR0912 (checks for functions or methods with too many branches)
def db_migrate_lobs(source_engine: DbEngine,
                    source_table: str,
                    source_lob_column: str,
                    source_pk_columns: list[str],
                    target_engine: DbEngine,
                    target_table: str,
                    target_lob_column: str = None,
                    target_pk_columns: list[str] = None,
                    source_conn: Any = None,
                    target_conn: Any = None,
                    source_committable: bool = None,
                    target_committable: bool = None,
                    where_clause: str = None,
                    orderby_clause: str = None,
                    offset_count: int = None,
                    limit_count: int = None,
                    batch_size: int = None,
                    chunk_size: int = None,
                    errors: list[str] = None,
                    logger: Logger = None,
                    log_trigger: int = 10000) -> tuple[int, int] | None:
    """
    Migrate large binary objects (LOBs) between databases.

    The origin and destination databases must be in the list of databases configured and
    supported by this package. One or more columns making up a primary key, or a unique row identifier,
    must exist on *source_table* and *target_table*, and be provided in *source_pk_columns*.
    It is assumed that the primary key columns have the same cardinality, and are of respectively
    equivalent types, in both the origin and the destination databases.

    The parameter *offset_count* is used to offset the retrieval of tuples. If it is set to -1,
    its value is set from a *COUNT* operation in *target_table*, with *where_clause* if specified.
    It is ignored for all other values different from a positive integer.

    The parameters *batch_size* and *limit_count* are used to limit the maximum number of tuples
    to retrieve in each SELECT operation. They are ignored if not positive integers.
    If *batch_size* or *offset_count* has been specified, but *orderby_clause* has not,
    then an ORDER BY clause is constructed from the data in *source_pk_columns*.

    Further, if *batch_size* and *limit_count* are both defined, and *batch_size* is set to a larger value,
    it is reduced accordingly. Finally, for both *offset_count* and *batch_size* or *limit_count* to be used
    together with SQLServer, an *ORDER BY* clause must have been specifed, otherwise a runtime error is raised.

    The parameters *source_committable* and *target_committable* are relevant only
    if *source_conn* and *target_conn* are provided, respectively, and are otherwise ignored.
    A rollback is always attempted, if an error occurs.

    :param source_engine: the source database engine type
    :param source_table: the possibly schema-qualified table to read the LOBs from
    :param source_lob_column: the column holding the LOB
    :param source_pk_columns: columns making up a primary key, or a unique identifier for a tuple, in source table
    :param target_engine: the destination database engine type
    :param target_table: the possibly schema-qualified table to write the LOB to
    :param target_lob_column: the column to write the LOB to (defaults to the source column)
    :param target_pk_columns: columns making up a primary key, or a unique identifier for a tuple, in target table
    :param source_conn: the connection to the source database (obtains a new one, if not provided)
    :param target_conn: the connection to the destination database (obtains a new one, if not provided)
    :param source_committable: whether to commit on *source_conn* upon errorless completion
    :param target_committable: whether to commit on *target_conn* upon errorless completion
    :param where_clause: optional criteria for tuple selection
    :param orderby_clause: optional retrieval order
    :param offset_count: number of tuples to skip in source table (defaults to none)
    :param limit_count: maximum number of tuples to migrate (defaults to no limit)
    :param batch_size: maximum number of tuples to read in each batch (defaults to no limit)
    :param chunk_size: size in bytes of the data chunk to read/write (defaults to no limit)
    :param errors: incidental error messages
    :param logger: optional logger
    :param log_trigger: number of LOBs to trigger logging info on migration (defaults to 10000 LOBs)
    :return: the number of LOBs effectively migrated and their total size, or *None* if error
    """
    # initialize the return variable
    result: tuple[int, int] | None = None

    # make sure to have an errors list
    if not isinstance(errors, list):
        errors = []

    # make sure to have connections to the source and destination databases
    curr_source_conn: Any = source_conn or db_connect(engine=source_engine,
                                                      errors=errors,
                                                      logger=logger)
    curr_target_conn: Any = target_conn or db_connect(engine=target_engine,
                                                      errors=errors,
                                                      logger=logger)
    if not errors:
        # make sure to have a target column
        if not target_lob_column:
            target_lob_column = source_lob_column

        # make sure to have target PK columns
        if not target_pk_columns:
            target_pk_columns = source_pk_columns

        # define the number of rows to skip
        if offset_count == -1:
            offset_count = db_count(table=target_table,
                                    where_clause=where_clause,
                                    engine=target_engine,
                                    connection=target_conn,
                                    committable=target_committable,
                                    errors=errors,
                                    logger=logger)
        else:
            offset_count = __normalize_value(value=offset_count)

        # normalize these parameters
        limit_count = __normalize_value(value=limit_count)
        batch_size = __normalize_value(value=batch_size)
        chunk_size = __normalize_value(value=chunk_size) or -1
        log_trigger = __normalize_value(value=log_trigger)
        if 0 < limit_count < batch_size:
            batch_size = limit_count
            if logger:
                logger.debug(msg=f"Value of input batch size changed to {batch_size}")

        # buid the SELECT query
        source_pks: str = ", ".join(source_pk_columns)
        sel_stmt: str = f"SELECT {source_pks}, {source_lob_column} FROM {source_table}"
        if where_clause:
            sel_stmt += f" WHERE {where_clause}"
        if not orderby_clause and (offset_count or batch_size):
            orderby_clause = ", ".join(source_pk_columns)
        if orderby_clause:
            sel_stmt += f" ORDER BY {orderby_clause}"
        sel_stmt += " OFFSET @offset ROWS"

        if limit_count or batch_size:
            if source_engine == DbEngine.POSTGRES:
                sel_stmt += " LIMIT @limit"
            elif source_engine in [DbEngine.ORACLE, DbEngine.SQLSERVER]:
                sel_stmt += " FETCH @limit ROWS ONLY"

        # build the UPDATE query
        lob_index: int = len(source_pk_columns)
        target_where_columns: str = _bind_columns(engine=target_engine,
                                                  columns=target_pk_columns,
                                                  concat=" AND ",
                                                  start_index=2)
        if target_engine == DbEngine.ORACLE:
            update_stmt: str = (f"UPDATE {target_table} "
                                f"SET {target_lob_column} = empty_blob() "
                                f"WHERE {target_where_columns} "
                                f"RETURNING {target_lob_column} INTO :{len(target_where_columns) + 2}")
        else:
            lob_bind: str = _bind_columns(engine=target_engine,
                                          columns=[target_lob_column],
                                          concat=", ",
                                          start_index=1)
            update_stmt: str = (f"UPDATE {target_table} "
                                f"SET {target_lob_column} = "
                                f"{target_lob_column} || {lob_bind[len(target_lob_column) + 3:]} "
                                f"WHERE {target_where_columns}")

        # log the migration start
        if logger:
            logger.debug(msg=f"Started migrating LOBs, "
                             f"from {source_engine}.{source_table}.{source_lob_column},"
                             f" connection {id(curr_source_conn)}, "
                             f"to {target_engine}.{target_table}.{target_lob_column}, "
                             f" connection {id(curr_target_conn)}, "
                             f"input batch size {batch_size}, offset {offset_count}, "
                             f"limit {limit_count}, chunk size {chunk_size}")
        # migrate the LOBs
        byte_count: int = 0
        byte_step: int = 0
        row_count: int = 0
        row_step: int = 0
        start_count: float = time.time()
        start_step: float = time.time()
        err_msg: str | None = None
        try:
            source_cursor: Any = curr_source_conn.cursor()
            target_cursor: Any = curr_target_conn.cursor()

            # adjust the offset and limit
            curr_stmt: str = sel_stmt
            curr_limit: int = min(batch_size, limit_count)
            if curr_limit == 0:
                curr_limit = max(batch_size, limit_count)
            if curr_limit:
                curr_stmt = curr_stmt.replace("@limit", str(curr_limit), 1)
            else:
                curr_stmt = curr_stmt.replace(" LIMIT @limit", "", 1)
                curr_stmt = curr_stmt.replace(" FETCH @limit ROWS ONLY", "", 1)
            if offset_count:
                curr_stmt = curr_stmt.replace("@offset", str(offset_count), 1)\
                                     .replace(" FETCH ", " FETCH NEXT ", 1)
            else:
                curr_stmt = curr_stmt.replace(" OFFSET @offset ROWS", "", 1)\
                                     .replace(" FETCH ", " FETCH FIRST ", 1)
            # go for the data
            next_rs: bool = True
            while next_rs:
                next_rs = False

                # execute the query
                # (parameter name is 'statement' for oracle, 'query' for postgres, 'sql' for sqlserver)
                source_cursor.execute(curr_stmt)

                # traverse the result set
                row: tuple = source_cursor.fetchone()
                while row:
                    next_rs = True
                    # retrieve the values of the primary key columns (leave LOB column out)
                    pk_vals: tuple = tuple([row[inx] for inx in range(lob_index)])

                    ora_lob: Any = None
                    if target_engine == DbEngine.ORACLE:
                        # initialize the LOB column with an empty LOB
                        import oracledb
                        lob_var = target_cursor.var(oracledb.DB_TYPE_BLOB)
                        target_cursor.execute(statement=update_stmt,
                                              parameters=(*pk_vals, lob_var))
                        ora_lob = lob_var.getValue()

                    # access the LOB in chunks and write it to database
                    first: bool = True
                    offset: int = 1
                    is_migrated: bool = False
                    lob: Any = row[lob_index]
                    lob_data: bytes | str = lob.read(offset=offset,
                                                     amount=chunk_size) if lob is not None else None
                    # make sure to skip null LOBs, and to migrate empty ones
                    while lob_data is not None and (first or len(lob_data) > 0):
                        first = False
                        is_migrated = True
                        match target_engine:
                            case DbEngine.MYSQL:
                                pass
                            case DbEngine.ORACLE:
                                ora_lob.write(data=lob_data,
                                              offset=offset)
                            case DbEngine.POSTGRES:
                                from psycopg2 import Binary
                                # remove append indication on initial update
                                update_pg: str = update_stmt if offset > 1 else \
                                    update_stmt.replace(f"{target_lob_column} || ", "", 1)

                                # string data may come from a LOB (Oracle's CLOB is a good example)
                                col_data: str | Binary = Binary(lob_data) if isinstance(lob_data, bytes) else lob_data
                                target_cursor.execute(query=update_pg,
                                                      vars=(col_data, *pk_vals))
                            case DbEngine.SQLSERVER:
                                from pyodbc import Binary
                                # remove append indication on initial update
                                update_sqls: str = update_stmt if offset > 1 else \
                                    update_stmt.replace(f"{target_lob_column} || ", "", 1)

                                # string data may come from a LOB (Oracle's NCLOB is a good example)
                                col_data: str | Binary = Binary(lob_data) if isinstance(lob_data, bytes) else lob_data
                                target_cursor.execute(sql=update_sqls,
                                                      params=(col_data, *pk_vals))
                        if len(lob_data) > 0:
                            # read the next chunk
                            offset += len(lob_data)
                            lob_data = lob.read(offset=offset,
                                                amount=chunk_size)

                    # increment the LOB migration counter, if applicable
                    if is_migrated:
                        byte_count += offset - 1
                        byte_step += offset - 1
                        row_count += 1
                        row_step += 1

                    # log partial result at each 'log_trigger' LOBs migrated
                    if logger and row_step >= log_trigger:
                        finish_step: float = time.time()
                        mins: float = (finish_step - start_step) / 60
                        duration: str = timestamp_duration(start=start_step,
                                                           finish=finish_step)
                        logger.debug(msg=f"Migrated {row_step} LOBs, {byte_step} bytes, in {duration} "
                                         f"({row_step/mins:.2f} LOBs/min, "
                                         f"{byte_step/(mins * 1024 ** 2):.2f} MBytes/min),  "
                                         f"from {source_engine}.{source_table}.{source_lob_column}, "
                                         f" connection {id(curr_source_conn)}, "
                                         f"to {target_engine}.{target_table}.{target_lob_column}, "
                                         f" connection {id(curr_target_conn)}, "
                                         f"offset {offset_count + row_count - row_step}")
                        byte_step = 0
                        row_step = 0
                        start_step = finish_step

                    # retrieve the next row
                    row = source_cursor.fetchone()

                # adjust the new offset and limit
                if next_rs and (limit_count > row_count or (batch_size and not limit_count)):
                    curr_limit = min(batch_size, limit_count - row_count)
                    if curr_limit <= 0:
                        curr_limit = max(batch_size, limit_count - row_count)
                    curr_stmt = sel_stmt.replace("@offset", str(offset_count + row_count), 1)\
                                        .replace("@limit", str(curr_limit), 1)\
                                        .replace(" FETCH ", " FETCH NEXT ", 1)
                else:
                    # signal end of migration
                    next_rs = False

            # close the cursors and commit the transactions
            source_cursor.close()
            if source_committable or not source_conn:
                curr_source_conn.commit()
            target_cursor.close()
            if target_committable or not target_conn:
                curr_target_conn.commit()
        except Exception as e:
            # rollback the transactions
            if curr_source_conn:
                with suppress(Exception):
                    curr_source_conn.rollback()
            if curr_target_conn:
                with suppress(Exception):
                    curr_target_conn.rollback()
            err_msg = _except_msg(exception=e,
                                  engine=source_engine)
        finally:
            # close the connections, if locally acquired
            if curr_source_conn and not source_conn:
                db_close(connection=curr_source_conn,
                         logger=logger)
            if curr_target_conn and not target_conn:
                db_close(connection=curr_target_conn,
                         logger=logger)

        # log the migration finish
        if err_msg:
            errors.append(err_msg)
            if logger:
                logger.error(msg=err_msg)
        elif not errors and logger:
            finish_count: float = time.time()
            mins = (finish_count - start_count) / 60
            duration: str = timestamp_duration(start=start_count,
                                               finish=finish_count)
            logger.debug(msg=f"Finished streaming {row_count} LOBs, {byte_count} bytes, in {duration} "
                             f"({row_count/mins:.2f} LOBs/min, {byte_count/(mins * 1024 ** 2):.2f} MBytes/min), "
                             f"from {source_engine}.{source_table}.{source_lob_column} "
                             f"to {target_engine}.{target_table}.{target_lob_column}, offset {offset_count}")
        if not errors:
            result = (row_count, byte_count)

    return result


def __normalize_value(value: int) -> int:
    """
    Normalize *value* to a consistent value.

    :param value: the value to normalized
    :return: the normalized value
    """
    if isinstance(value, int) and \
            not isinstance(value, bool) and value > 0:
        result: int = value
    else:
        result: int = 0

    return result
