import React from 'react';
import { Timeline, Typography, Button } from '@material-tailwind/react';
import { HiExternalLink } from 'react-icons/hi';

import { useTicketContext } from '@/contexts/TicketsContext';
import { formatDateString } from '@/utils';
import { Link } from 'react-router';

/**
 * Sample ticket object:
created: "2025-07-14T07:43:53.007000-04:00"
description: "Please convert Users_truhlara_dev_fileglancer/fused-timeseries.zarr to a ZARR file.\nDestination folder: /Users/truhlara/dev/fileglancer/"
fsp_name: "Users_truhlara_dev_fileglancer"
key: "FT-54"
link: "https://issues.hhmi.org/issues/browse/FT-54"
path: "fused-timeseries.zarr"
resolution: "Unresolved"
status: "Open"
updated: "2025-07-14T07:43:53.007000-04:00"
username: "User(username='cb2e8bf0bc374ecbab87c53572cbba6f', name='Anonymous Eukelade', display_name='Anonymous Eukelade', initials='AE', avatar_url=None, color=None)"
 */

const possibleStatuses = ['Open', 'Pending', 'Work in progress', 'Done'];

function TimelineSegment({ step, index }: { step: string; index: number }) {
  const { ticket } = useTicketContext();
  const ticketStatus = ticket?.status;
  const isCurrentStep = ticketStatus === step;
  const isFirstStep = index === 0;
  const isLastStep = index === possibleStatuses.length - 1;
  const isFutureStep =
    possibleStatuses.indexOf(ticketStatus as string) <
    possibleStatuses.indexOf(step);

  return (
    <Timeline.Item>
      <Timeline.Header>
        {isLastStep ? null : (
          <Timeline.Separator
            className={isFutureStep ? '!bg-surface' : '!bg-primary'}
          />
        )}

        <Timeline.Icon
          className={`h-3 w-3 ${isFutureStep ? '!bg-surface' : '!bg-primary'}`}
        />
      </Timeline.Header>

      <Timeline.Body className="-translate-y-1.5">
        <Typography
          className={`${isFutureStep ? 'text-foreground' : 'text-primary font-bold'}`}
        >
          {step}
        </Typography>
        {isCurrentStep || isFirstStep ? (
          <Typography
            type="small"
            className={`mt-2 ${isFirstStep && !isCurrentStep ? 'text-foreground' : 'text-primary'}`}
          >
            {isFirstStep && !isCurrentStep
              ? `Opened: ${formatDateString(ticket?.created || '')}`
              : `Last updated: ${formatDateString(ticket?.updated || '')}`}
          </Typography>
        ) : null}
      </Timeline.Body>
    </Timeline.Item>
  );
}

export default function TicketDetails() {
  const { ticket } = useTicketContext();

  // Format description to properly display line breaks
  const formattedDescription = ticket?.description
    ? ticket.description.split('\\n').map((line, index) => (
        <React.Fragment key={index}>
          {line}
          {index < ticket.description.split('\\n').length - 1 && <br />}
        </React.Fragment>
      ))
    : null;

  return (
    <div className="mt-4 flex flex-col gap-6 min-w-max w-full">
      {ticket?.link && (
        <Button
          as={Link}
          to={ticket.link}
          target="_blank"
          rel="noopener noreferrer"
          variant="outline"
          size="sm"
          className="flex items-center justify-center gap-1 text-primary px-2 py-1 !self-start"
        >
          View ticket in JIRA
          <HiExternalLink className="h-3.5 w-3.5" />
        </Button>
      )}
      <div className="overflow-hidden rounded-lg border border-surface">
        <div className="p-2 border-b border-surface bg-surface-light text-sm text-foreground dark:bg-surface-dark font-semibold">
          Description
        </div>
        <div className="p-2 text-sm whitespace-pre-line">
          {formattedDescription}
        </div>
      </div>
      <Timeline orientation="vertical">
        {possibleStatuses.map((step, index) => (
          <TimelineSegment key={step} step={step} index={index} />
        ))}
      </Timeline>
    </div>
  );
}
