"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("@aws-cdk/assets");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new assets.Staging(this, 'Stage', Object.assign(Object.assign({}, props), { sourcePath: path.resolve(props.path) }));
        this.sourceHash = staging.sourceHash;
        this.assetPath = staging.stagedPath;
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === AssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        // add parameters for s3 bucket and s3 key. those will be set by
        // the toolkit or by CI/CD when the stack is deployed and will include
        // the name of the bucket and the S3 key where the code lives.
        const bucketParam = new cdk.CfnParameter(this, 'S3Bucket', {
            type: 'String',
            description: `S3 bucket for asset "${this.node.path}"`,
        });
        const keyParam = new cdk.CfnParameter(this, 'S3VersionKey', {
            type: 'String',
            description: `S3 key for asset version "${this.node.path}"`
        });
        const hashParam = new cdk.CfnParameter(this, 'ArtifactHash', {
            description: `Artifact hash for asset "${this.node.path}"`,
            type: 'String',
        });
        this.s3BucketName = bucketParam.valueAsString;
        this.s3Prefix = cdk.Fn.select(0, cdk.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, keyParam.valueAsString)).toString();
        const s3Filename = cdk.Fn.select(1, cdk.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, keyParam.valueAsString)).toString();
        this.s3ObjectKey = `${this.s3Prefix}${s3Filename}`;
        this.bucket = s3.Bucket.fromBucketName(this, 'AssetBucket', this.s3BucketName);
        // form the s3 URL of the object key
        this.s3Url = this.bucket.urlForObject(this.s3ObjectKey);
        // attach metadata to the lambda function which includes information
        // for tooling to be able to package and upload a directory to the
        // s3 bucket and plug in the bucket name and key in the correct
        // parameters.
        const asset = {
            path: this.assetPath,
            id: this.node.uniqueId,
            packaging,
            sourceHash: this.sourceHash,
            s3BucketParameter: bucketParam.logicalId,
            s3KeyParameter: keyParam.logicalId,
            artifactHashParameter: hashParam.logicalId,
        };
        this.node.addMetadata(cxapi.ASSET_METADATA, asset);
        for (const reader of (props.readers || [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the asset's S3 object.
     */
    grantRead(grantee) {
        // We give permissions on all files with the same prefix. Presumably
        // different versions of the same file will have the same prefix
        // and we don't want to accidentally revoke permission on old versions
        // when deploying a new version.
        this.bucket.grantRead(grantee, `${this.s3Prefix}*`);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return AssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return AssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
/**
 * Defines the way an asset is packaged before it is uploaded to S3.
 */
var AssetPackaging;
(function (AssetPackaging) {
    /**
     * Path refers to a directory on disk, the contents of the directory is
     * archived into a .zip.
     */
    AssetPackaging["ZIP_DIRECTORY"] = "zip";
    /**
     * Path refers to a single file on disk. The file is uploaded as-is.
     */
    AssetPackaging["FILE"] = "file";
})(AssetPackaging || (AssetPackaging = {}));
//# sourceMappingURL=data:application/json;base64,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