"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const asset_1 = require("../lib/asset");
const SAMPLE_ASSET_DIR = path.join(__dirname, 'sample-asset-directory');
function mkdtempSync() {
    return fs.mkdtempSync(path.join(os.tmpdir(), 'test.assets'));
}
module.exports = {
    'simple use case'(test) {
        const app = new cdk.App({
            context: {
                [cxapi.DISABLE_ASSET_STAGING_CONTEXT]: 'true'
            }
        });
        const stack = new cdk.Stack(app, 'MyStack');
        const asset = new asset_1.Asset(stack, 'MyAsset', {
            path: SAMPLE_ASSET_DIR
        });
        // verify that metadata contains an "aws:cdk:asset" entry with
        // the correct information
        const entry = asset.node.metadata.find(m => m.type === 'aws:cdk:asset');
        test.ok(entry, 'found metadata entry');
        // verify that now the template contains parameters for this asset
        const session = app.synth();
        test.deepEqual(stack.resolve(entry.data), {
            path: SAMPLE_ASSET_DIR,
            id: 'MyStackMyAssetBDDF29E3',
            packaging: 'zip',
            sourceHash: '6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2',
            s3BucketParameter: 'MyAssetS3Bucket68C9B344',
            s3KeyParameter: 'MyAssetS3VersionKey68E1A45D',
            artifactHashParameter: 'MyAssetArtifactHashF518BDDE',
        });
        const template = JSON.parse(fs.readFileSync(path.join(session.directory, 'MyStack.template.json'), 'utf-8'));
        test.equal(template.Parameters.MyAssetS3Bucket68C9B344.Type, 'String');
        test.equal(template.Parameters.MyAssetS3VersionKey68E1A45D.Type, 'String');
        test.done();
    },
    'verify that the app resolves tokens in metadata'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'my-stack');
        const dirPath = path.resolve(__dirname, 'sample-asset-directory');
        new asset_1.Asset(stack, 'MyAsset', {
            path: dirPath
        });
        const synth = app.synth().getStack(stack.stackName);
        const meta = synth.manifest.metadata || {};
        test.ok(meta['/my-stack/MyAsset']);
        test.ok(meta['/my-stack/MyAsset'][0]);
        test.deepEqual(meta['/my-stack/MyAsset'][0].data, {
            path: 'asset.6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2',
            id: "mystackMyAssetD6B1B593",
            packaging: "zip",
            sourceHash: '6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2',
            s3BucketParameter: "MyAssetS3Bucket68C9B344",
            s3KeyParameter: "MyAssetS3VersionKey68E1A45D",
            artifactHashParameter: 'MyAssetArtifactHashF518BDDE',
        });
        test.done();
    },
    '"file" assets'(test) {
        const stack = new cdk.Stack();
        const filePath = path.join(__dirname, 'file-asset.txt');
        const asset = new asset_1.Asset(stack, 'MyAsset', { path: filePath });
        const entry = asset.node.metadata.find(m => m.type === 'aws:cdk:asset');
        test.ok(entry, 'found metadata entry');
        // synthesize first so "prepare" is called
        const template = assert_1.SynthUtils.synthesize(stack).template;
        test.deepEqual(stack.resolve(entry.data), {
            path: 'asset.78add9eaf468dfa2191da44a7da92a21baba4c686cf6053d772556768ef21197.txt',
            packaging: 'file',
            id: 'MyAsset',
            sourceHash: '78add9eaf468dfa2191da44a7da92a21baba4c686cf6053d772556768ef21197',
            s3BucketParameter: 'MyAssetS3Bucket68C9B344',
            s3KeyParameter: 'MyAssetS3VersionKey68E1A45D',
            artifactHashParameter: 'MyAssetArtifactHashF518BDDE',
        });
        // verify that now the template contains parameters for this asset
        test.equal(template.Parameters.MyAssetS3Bucket68C9B344.Type, 'String');
        test.equal(template.Parameters.MyAssetS3VersionKey68E1A45D.Type, 'String');
        test.done();
    },
    '"readers" or "grantRead" can be used to grant read permissions on the asset to a principal'(test) {
        const stack = new cdk.Stack();
        const user = new iam.User(stack, 'MyUser');
        const group = new iam.Group(stack, 'MyGroup');
        const asset = new asset_1.Asset(stack, 'MyAsset', {
            path: path.join(__dirname, 'sample-asset-directory'),
            readers: [user]
        });
        asset.grantRead(group);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: ["s3:GetObject*", "s3:GetBucket*", "s3:List*"],
                        Effect: 'Allow',
                        Resource: [
                            { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":s3:::", { Ref: "MyAssetS3Bucket68C9B344" }]] },
                            { "Fn::Join": ["",
                                    [
                                        "arn:", { Ref: "AWS::Partition" }, ":s3:::", { Ref: "MyAssetS3Bucket68C9B344" },
                                        "/",
                                        { "Fn::Select": [0, { "Fn::Split": ["||", { Ref: "MyAssetS3VersionKey68E1A45D" }] }] },
                                        "*"
                                    ]
                                ] }
                        ]
                    }
                ]
            }
        }));
        test.done();
    },
    'fails if directory not found'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new asset_1.Asset(stack, 'MyDirectory', {
            path: '/path/not/found/' + Math.random() * 999999
        }));
        test.done();
    },
    'multiple assets under the same parent'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new asset_1.Asset(stack, 'MyDirectory1', { path: path.join(__dirname, 'sample-asset-directory') });
        new asset_1.Asset(stack, 'MyDirectory2', { path: path.join(__dirname, 'sample-asset-directory') });
        // THEN: no error
        test.done();
    },
    'isZipArchive indicates if the asset represents a .zip file (either explicitly or via ZipDirectory packaging)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const nonZipAsset = new asset_1.Asset(stack, 'NonZipAsset', {
            path: path.join(__dirname, 'sample-asset-directory', 'sample-asset-file.txt')
        });
        const zipDirectoryAsset = new asset_1.Asset(stack, 'ZipDirectoryAsset', {
            path: path.join(__dirname, 'sample-asset-directory')
        });
        const zipFileAsset = new asset_1.Asset(stack, 'ZipFileAsset', {
            path: path.join(__dirname, 'sample-asset-directory', 'sample-zip-asset.zip')
        });
        const jarFileAsset = new asset_1.Asset(stack, 'JarFileAsset', {
            path: path.join(__dirname, 'sample-asset-directory', 'sample-jar-asset.jar')
        });
        // THEN
        test.equal(nonZipAsset.isZipArchive, false);
        test.equal(zipDirectoryAsset.isZipArchive, true);
        test.equal(zipFileAsset.isZipArchive, true);
        test.equal(jarFileAsset.isZipArchive, true);
        test.done();
    },
    'addResourceMetadata can be used to add CFN metadata to resources'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
        const location = path.join(__dirname, 'sample-asset-directory');
        const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
        const asset = new asset_1.Asset(stack, 'MyAsset', { path: location });
        // WHEN
        asset.addResourceMetadata(resource, 'PropName');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('My::Resource::Type', {
            Metadata: {
                "aws:asset:path": 'asset.6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2',
                "aws:asset:property": "PropName"
            }
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'asset metadata is only emitted if ASSET_RESOURCE_METADATA_ENABLED_CONTEXT is defined'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
        const asset = new asset_1.Asset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
        // WHEN
        asset.addResourceMetadata(resource, 'PropName');
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('My::Resource::Type', {
            Metadata: {
                "aws:asset:path": SAMPLE_ASSET_DIR,
                "aws:asset:property": "PropName"
            }
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'staging': {
        'copy file assets under <outdir>/${fingerprint}.ext'(test) {
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            // GIVEN
            const app = new core_1.App({ outdir: tempdir });
            const stack = new core_1.Stack(app, 'stack');
            // WHEN
            new asset_1.Asset(stack, 'ZipFile', {
                path: path.join(SAMPLE_ASSET_DIR, 'sample-zip-asset.zip')
            });
            new asset_1.Asset(stack, 'TextFile', {
                path: path.join(SAMPLE_ASSET_DIR, 'sample-asset-file.txt')
            });
            // THEN
            app.synth();
            test.ok(fs.existsSync(tempdir));
            test.ok(fs.existsSync(path.join(tempdir, 'asset.a7a79cdf84b802ea8b198059ff899cffc095a1b9606e919f98e05bf80779756b.zip')));
            test.done();
        },
        'copy directory under .assets/fingerprint/**'(test) {
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            // GIVEN
            const app = new core_1.App({ outdir: tempdir });
            const stack = new core_1.Stack(app, 'stack');
            // WHEN
            new asset_1.Asset(stack, 'ZipDirectory', {
                path: SAMPLE_ASSET_DIR
            });
            // THEN
            app.synth();
            test.ok(fs.existsSync(tempdir));
            const hash = 'asset.6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2';
            test.ok(fs.existsSync(path.join(tempdir, hash, 'sample-asset-file.txt')));
            test.ok(fs.existsSync(path.join(tempdir, hash, 'sample-jar-asset.jar')));
            fs.readdirSync(tempdir);
            test.done();
        },
        'staging path is relative if the dir is below the working directory'(test) {
            // GIVEN
            const tempdir = mkdtempSync();
            process.chdir(tempdir); // change current directory to somewhere in /tmp
            const staging = '.my-awesome-staging-directory';
            const app = new core_1.App({
                outdir: staging,
                context: {
                    [cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT]: 'true',
                }
            });
            const stack = new core_1.Stack(app, 'stack');
            const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
            const asset = new asset_1.Asset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
            // WHEN
            asset.addResourceMetadata(resource, 'PropName');
            const template = assert_1.SynthUtils.synthesize(stack).template;
            test.deepEqual(template.Resources.MyResource.Metadata, {
                "aws:asset:path": `asset.6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2`,
                "aws:asset:property": "PropName"
            });
            test.done();
        },
        'if staging is disabled, asset path is absolute'(test) {
            // GIVEN
            const staging = path.resolve(mkdtempSync());
            const app = new core_1.App({
                outdir: staging,
                context: {
                    [cxapi.DISABLE_ASSET_STAGING_CONTEXT]: 'true',
                    [cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT]: 'true',
                }
            });
            const stack = new core_1.Stack(app, 'stack');
            const resource = new cdk.CfnResource(stack, 'MyResource', { type: 'My::Resource::Type' });
            const asset = new asset_1.Asset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
            // WHEN
            asset.addResourceMetadata(resource, 'PropName');
            const template = assert_1.SynthUtils.synthesize(stack).template;
            test.deepEqual(template.Resources.MyResource.Metadata, {
                "aws:asset:path": SAMPLE_ASSET_DIR,
                "aws:asset:property": "PropName"
            });
            test.done();
        },
        'cdk metadata points to staged asset'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'stack');
            new asset_1.Asset(stack, 'MyAsset', { path: SAMPLE_ASSET_DIR });
            // WHEN
            const session = app.synth();
            const artifact = session.getStack(stack.stackName);
            const metadata = artifact.manifest.metadata || {};
            const md = Object.values(metadata)[0][0].data;
            test.deepEqual(md.path, 'asset.6b84b87243a4a01c592d78e1fd3855c4bfef39328cd0a450cc97e81717fea2a2');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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