##############################################################################
#
# Copyright (c) 2014 Zope Foundation and Contributors.
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
##############################################################################
"""Real test for file-upload and beginning of a better internal test framework
"""
from __future__ import print_function

from __future__ import absolute_import
from builtins import str
from builtins import object
import io
import sys
import doctest

from p01.testbrowser.browser import Browser
import p01.testbrowser.tests.helper

try:
    unicode # p01.checker.ignore
except Exception as e:
    unicode = str

PY2 = sys.version_info[0] == 2
if PY2:
    import __builtin__ as builtins
    text_type = unicode
    native_str_type = builtins.str
    binary_types = (native_str_type, bytearray)
    bytes_type = str
else:
    import builtins
    text_type = str
    native_str_type = str
    text_type = str
    binary_types = (bytes, bytearray)
    bytes_type = bytes

def _latin1(s):
    if PY2:
        if isinstance(s, text_type):
            return s.encode('latin-1')
        if isinstance(s, binary_types):
            return s
        return builtins.str(s)
    else:
        if isinstance(s, binary_types):
            return s.decode('latin-1')
        if isinstance(s, str):
            return s
        return str(s)

def _wsgi_status(status):
    if PY2:
        if type(status) is native_str_type:
            return status
        if isinstance(status, text_type):
            return status.encode('latin-1', 'strict')
        try:
            ba = bytearray(status)
            return ''.join(chr(c) for c in ba)
        except Exception:
            return native_str_type(status)
    else:
        return status if isinstance(status, str) else str(status)

def _wsgi_header_name(s):
    if PY2:
        if type(s) is native_str_type:
            return s
        if isinstance(s, text_type):
            return s.encode('latin-1', 'strict')
        try:
            return ''.join(chr(c) for c in bytearray(s))
        except Exception:
            return builtins.str(s)
    else:
        if isinstance(s, binary_types):
            return s.decode('latin-1', 'strict')
        return s if isinstance(s, str) else str(s)

def _wsgi_header_value(s):
    return _wsgi_header_name(s) if PY2 else (
        s.decode('latin-1', 'strict') if isinstance(s, binary_types)
        else (s if isinstance(s, str) else str(s))
    )

class TestApp(object):
    next_response_body = None
    next_response_headers = None
    next_response_status = '200 OK'

    def set_next_response(self, body, status='200 OK', headers=None,
        content_type='text/html; charset=utf-8'):
        body_bytes = body.encode('utf-8') if isinstance(body, str) else body
        hdrs = [
            (_wsgi_header_name('Content-Type'), _wsgi_header_value(content_type)),
            (_wsgi_header_name('Content-Length'), native_str_type(len(body_bytes))),
        ]
        if headers:
            for k, v in headers:
                hdrs.append((_wsgi_header_name(k), _wsgi_header_value(v)))
        self.next_response_status = _wsgi_status(status)
        self.next_response_headers = hdrs
        self.next_response_body = body_bytes

    def __call__(self, environ, start_response):
        qs = environ.get('QUERY_STRING')
        print("%s %s%s HTTP/1.1" % (environ['REQUEST_METHOD'],
            environ['PATH_INFO'], '?'+qs if qs else ""))
        for ek, ev in sorted(environ.items()):
            if ek.startswith('HTTP_'):
                print("%s: %s" % (ek[5:].title(), ev))
        print()
        inp = getattr(environ.get('wsgi.input'), 'input', None)
        if inp is not None:
            print(inp.getvalue().decode('utf-8', 'replace'))
        status = _wsgi_status(self.next_response_status)
        headers = [(_wsgi_header_name(k), _wsgi_header_value(v)) for (k, v) in self.next_response_headers]
        start_response(status, headers)
        return [self.next_response_body]


class YetAnotherTestApp(object):

    def __init__(self):
        self.requests = []
        self.responses = []
        self.last_environ = {}
        self.last_input = None

    def add_response(self, body, headers=None, status='200', reason='OK'):
        if headers is None:
            length = unicode(len(body)).encode('latin1')
            headers = [(_wsgi_header_name('Content-Type'), _wsgi_header_value('text/html; charset="UTF-8"')),
                       (_wsgi_header_name('Content-Length'), length)]
        resp = dict(body=body, headers=headers, status=status, reason=reason)
        self.responses.append(resp)

    def __call__(self, environ, start_response):
        self.requests.append(environ)
        next_response = self.responses.pop(0)
        self.last_environ = environ
        self.last_input = environ['wsgi.input'].input.getvalue().decode('utf-8')
        status = '%s %s' % (next_response['status'], next_response['reason'])
        status = _wsgi_status(status)
        # headers = [
        #     (k, v.decode('latin-1') if isinstance(v, bytes_type) else v)
        #     for k, v in next_response['headers']
        # ]
        headers = [
            (_wsgi_header_name(k), _wsgi_header_value(v))
            for (k, v) in next_response['headers']
        ]

        start_response(status, headers)
        return [next_response['body']]


def test_relative_redirect(self):
    """
    >>> app = YetAnotherTestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> body = b'redirecting'
    >>> content_type = ('Content-Type', 'text/html; charset=UTF-8')
    >>> length = unicode(len(body)).encode('latin1')
    >>> headers = [content_type,
    ...            ('Location', 'foundit'),
    ...            ('Content-Length', length)]
    >>> app.add_response(body, headers=headers, status=302, reason='Found')
    >>> app.add_response(b'found_it', headers=[content_type])
    >>> browser.open('https://localhost/foo/bar')
    >>> browser.contents
    'found_it'
    >>> browser.url
    'https://localhost/foo/foundit'
    """


def test_relative_open_allowed_after_non_html_page(self):
    """
    >>> app = YetAnotherTestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> content_type = ('Content-Type', 'text/csv')
    >>> app.add_response(b'have,some,csv', headers=[content_type])
    >>> content_type = ('Content-Type', 'text/html; charset=UTF-8')
    >>> app.add_response(b'have some html', headers=[content_type])
    >>> browser.open('https://localhost/foo/bar')
    >>> browser.open('/baz')
    >>> browser.contents
    'have some html'
    >>> browser.url
    'https://localhost/baz'
    """

def test_accept_language_header_non_us():
    """Regression test for Accept-Language header

    Once was hardcoded to us-US!

    >>> app = YetAnotherTestApp()
    >>> app.add_response(b'mundo')
    >>> browser = Browser(wsgi_app=app)
    >>> browser.addHeader('Accept-Language', 'es-ES')
    >>> browser.open("http://localhost/hello")
    >>> app.last_environ['HTTP_ACCEPT_LANGUAGE']
    'es-ES'
    """

def test_reload_after_redirect():
    """
    When browser is redirected after form submit, reload() will not resubmit
    oroginal form data.

    >>> app = YetAnotherTestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> html = (b'''\
    ... <html><body>
    ...   <form action="submit" method="post" enctype="multipart/form-data">
    ...      <input type='text' name='name' value='Linus' />
    ...      <button name="do" type="button">Do Stuff</button>
    ...   </form></body></html>
    ... ''')
    >>> content_type = ('Content-Type', 'text/html; charset=UTF-8')
    >>> app.add_response(html, headers=[content_type])

    >>> redirect = ('Location', 'http://localhost/processed')
    >>> app.add_response(b"Moved", headers=[redirect],
    ...                  status=302, reason='Found')
    >>> app.add_response(b"Processed", headers=[content_type])

    >>> app.add_response(b"Reloaded", headers=[content_type])

    Start conversation

    >>> browser.open("http://localhost/form")
    >>> browser.getControl(name="do").click()

    We should have followed the redirect with GET request
    >>> browser.url
    'http://localhost/processed'
    >>> browser.contents
    'Processed'
    >>> app.last_environ['REQUEST_METHOD']
    'GET'
    >>> print(app.last_input)
    <BLANKLINE>

    After reload, expect no form data to be submitted
    >>> browser.reload()
    >>> browser.url
    'http://localhost/processed'
    >>> browser.contents
    'Reloaded'
    >>> app.last_environ['REQUEST_METHOD']
    'GET'
    >>> print(app.last_input)
    <BLANKLINE>
    """

def test_reload_after_post():
    """
    If we reload page just after submitting the form, all form data should be
    submitted again (just as real browsers do).

    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)

    First response is a form:

    >>> html = (b'''\
    ... <html><body>
    ...   <form action="submit" method="post" enctype="multipart/form-data">
    ...      <input type='text' name='name' value='Linus' />
    ...      <button name="do" type="button">Do Stuff</button>
    ...   </form></body></html>
    ... ''')
    >>> content_type = ('Content-Type', 'text/html; charset=UTF-8')
    >>> app.set_next_response(html, headers=[content_type])

    >>> browser.open('https://localhost/foo/bar')
    GET /foo/bar HTTP/1.1
    ...

    After submit, show result page
    >>> app.set_next_response(b'OK', headers=[content_type])

    Form data is there in POST request
    >>> browser.getControl(name="do").click()
    POST /foo/submit HTTP/1.1
    ...
    Content-Disposition: form-data; name="name"
    <BLANKLINE>
    Linus
    ...

    >>> browser.contents
    'OK'

    >>> browser.url
    'https://localhost/foo/submit'

    POST data is still there after reload
    >>> browser.reload()
    POST /foo/submit HTTP/1.1
    ...
    Content-Disposition: form-data; name="name"
    <BLANKLINE>
    Linus
    ...

    """

def test_button_without_name(self):
    """
    This once blew up.

    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <button type="button">Do Stuff</button>
    ...   </form></body></html>
    ... ''')
    >>> browser.open('http://localhost/') # doctest: +ELLIPSIS
    GET / HTTP/1.1
    ...
    >>> browser.getControl('NotThere') # doctest: +ELLIPSIS
    Traceback (most recent call last):
        ...
    LookupError: ...
    ...
    """


def test_submit_duplicate_name():
    """
    This test was inspired by bug #723 as testbrowser would pick up the wrong
    button when having the same name twice in a form.

    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)


    When given a form with two submit buttons that have the same name:

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <input type="submit" name="submit_me" value="GOOD" />
    ...      <input type="submit" name="submit_me" value="BAD" />
    ...   </form></body></html>
    ... ''')
    >>> browser.open('http://localhost/') # doctest: +ELLIPSIS
    GET / HTTP/1.1
    ...

    We can specify the second button through it's label/value:

    >>> browser.getControl('BAD')
    <SubmitControl name='submit_me'>
    >>> browser.getControl('BAD').value
    'BAD'
    >>> browser.getControl('BAD').click() # doctest: +REPORT_NDIFF +ELLIPSIS
    POST / HTTP/1.1
    ...
    Content-disposition: form-data; name="submit_me"
    <BLANKLINE>
    BAD
    ...


    This also works if the labels have whitespace around them (this tests a
    regression caused by the original fix for the above):

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <input type="submit" name="submit_me" value=" GOOD " />
    ...      <input type="submit" name="submit_me" value=" BAD " />
    ...   </form></body></html>
    ... ''')
    >>> browser.open('http://localhost/') # doctest: +ELLIPSIS
    GET / HTTP/1.1
    ...

    >>> browser.getControl('BAD')
    <SubmitControl name='submit_me'>
    >>> browser.getControl('BAD').value
    ' BAD '
    >>> browser.getControl('BAD').click() # doctest: +REPORT_NDIFF +ELLIPSIS
    POST / HTTP/1.1
    ...
    Content-disposition: form-data; name="submit_me"
    <BLANKLINE>
     BAD
    ...
    """


def test_file_upload():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)

    When given a form with a file-upload

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <input name="foo" type="file" />
    ...      <input type="submit" value="OK" />
    ...   </form></body></html>
    ... ''') # doctest: +ELLIPSIS
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...

    Fill in the form value using add_file:

    >>> browser.getControl(name='foo').add_file(
    ...     io.BytesIO(b'sample_data'), 'text/foo', 'x.txt')
    >>> browser.getControl('OK').click() # doctest: +REPORT_NDIFF +ELLIPSIS
    POST / HTTP/1.1
    ...
    Content-Disposition: form-data; name="foo"; filename="x.txt"
    Content-Type: text/foo
    <BLANKLINE>
    sample_data
    ...

    You can pass a string to add_file:

    >>> browser.getControl(name='foo').add_file(
    ...     b'blah blah blah', 'text/csv', 'x.csv')
    >>> browser.getControl('OK').click() # doctest: +REPORT_NDIFF +ELLIPSIS
    POST / HTTP/1.1
    ...
    Content-disposition: form-data; name="foo"; filename="x.csv"
    Content-type: text/csv
    <BLANKLINE>
    blah blah blah
    ...

    You can assign a value

    >>> browser.getControl(name='foo').value = b'bluh bluh'
    >>> browser.getControl('OK').click() # doctest: +REPORT_NDIFF +ELLIPSIS
    POST / HTTP/1.1
    ...
    Content-disposition: form-data; name="foo"; filename=""
    Content-type: application/octet-stream
    <BLANKLINE>
    bluh bluh
    ...

    """


def test_submit_gets_referrer():
    """
    Test for bug #98437: No HTTP_REFERER was sent when submitting a form.

    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)

    A simple form for testing, like abobe.

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form id="form" action="." method="post"
    ...                   enctype="multipart/form-data">
    ...      <input type="submit" name="submit_me" value="GOOD" />
    ...   </form></body></html>
    ... ''') # doctest: +ELLIPSIS
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...

    Now submit the form, and see that we get an referrer along:

    >>> form = browser.getForm(id='form')
    >>> form.submit(name='submit_me') # doctest: +ELLIPSIS
    POST / HTTP/1.1
    ...
    Referer: http://localhost/
    ...
    """


def test_new_instance_no_contents_should_not_fail(self):
    """
    When first instantiated, the browser has no contents.
    (Regression test for <http://bugs.launchpad.net/zope3/+bug/419119>)

    >>> browser = Browser()
    >>> print(browser.contents)
    None
    """


def test_strip_linebreaks_from_textarea(self):
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)

    According to http://www.w3.org/TR/html4/appendix/notes.html#h-B.3.1 line
    break immediately after start tags or immediately before end tags must be
    ignored, but real browsers only ignore a line break after a start tag.
    So if we give the following form:

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <textarea name="textarea">
    ... Foo
    ... </textarea>
    ...   </form></body></html>
    ... ''')
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...

    The value of the textarea won't contain the first line break:

    >>> browser.getControl(name='textarea').value
    'Foo\\n'


    Of course, if we add line breaks, so that there are now two line breaks
    after the start tag, the textarea value will start and end with a line
    break.

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <textarea name="textarea">
    ...
    ... Foo
    ... </textarea>
    ...   </form></body></html>
    ... ''')
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...

    >>> browser.getControl(name='textarea').value
    '\\nFoo\\n'


    Also, if there is some other whitespace after the start tag, it will be
    preserved.

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <textarea name="textarea">  Foo  </textarea>
    ...   </form></body></html>
    ... ''')
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...

    >>> browser.getControl(name='textarea').value
    '  Foo  '
    """


def test_relative_link():
    """
    RFC 1808 specifies how relative URLs should be resolved, let's see
    that we conform to it. Let's start with a simple example.

    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <a href="foo">link</a>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/bar') # doctest: +ELLIPSIS
    GET /bar HTTP/1.1
    ...

    >>> link = browser.getLink('link')
    >>> link.url
    'http://localhost/foo'


    It's possible to have a relative URL consisting of only a query part. In
    that case it should simply be appended to the base URL.

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <a href="?key=value">link</a>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/bar') # doctest: +ELLIPSIS
    GET /bar HTTP/1.1
    ...

    >>> link = browser.getLink('link')
    >>> link.url
    'http://localhost/bar?key=value'


    In the example above, the base URL was the page URL, but we can also
    specify a base URL using a <base> tag.

    >>> app.set_next_response(b'''\
    ... <html><head><base href="http://localhost/base" /></head><body>
    ...     <a href="?key=value">link</a>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/base/bar') # doctest: +ELLIPSIS
    GET /base/bar HTTP/1.1
    ...

    >>> link = browser.getLink('link')
    >>> link.url
    'http://localhost/base?key=value'
    """


def test_relative_open():
    """
    Browser is capable of opening relative urls as well as relative links

    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)

    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <a href="foo">link</a>
    ... </body></html>
    ... ''')
    >>> browser.open('bar') # doctest: +ELLIPSIS
    Traceback (most recent call last):
    ...
    BrowserStateError: Can't fetch relative reference: not viewing any document

    >>> browser.open('http://localhost/hello/foo') # doctest: +ELLIPSIS
    GET /hello/foo HTTP/1.1
    ...

    >>> browser.open('bar') # doctest: +ELLIPSIS
    GET /hello/bar HTTP/1.1
    ...

    >>> browser.open('/bar') # doctest: +ELLIPSIS
    GET /bar HTTP/1.1
    ...

    """


def test_submit_button():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <button name='clickable' value='test' type='submit'>
    ...         Click Me</button>
    ...         <button name='simple' value='value'>
    ...         Don't Click</button>
    ...     </form>
    ...     <a href="foo">link</a>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo') # doctest: +ELLIPSIS
    GET /foo HTTP/1.1
    ...

    >>> browser.getControl('Click Me')
    <ButtonControl name='clickable'>

    >>> browser.getControl('Click Me').click()
    GET /action?clickable=test HTTP/1.1
    ...
    """


def test_repeated_button():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <input name='one' value='Button' type='submit'>
    ...         <input name='two' value='Button' type='submit'>
    ...         <input name='one' value='Button' type='submit'>
    ...     </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo') # doctest: +ELLIPSIS
    GET /foo HTTP/1.1
    ...

    >>> browser.getControl('Button')
    Traceback (most recent call last):
    ...
    AmbiguityError: label 'Button' matches:
      <SubmitControl name='one'>
      <SubmitControl name='two'>
      <SubmitControl name='one'>

    >>> browser.getControl('Button', index=0)
    <SubmitControl name='one'>
    >>> browser.getControl('Button', index=1)
    <SubmitControl name='two'>
    >>> browser.getControl('Button', index=2)
    <SubmitControl name='one'>
    """


def test_subcontrols_can_be_selected_by_value():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <input id="foo1" type='checkbox' name='foo' value="1">
    ...         <label for="foo1">One</label>
    ...         <input id="foo2" type='checkbox' name='foo' value="2">
    ...         <label for="foo2">Two</label>
    ...         <input id="foo3" type='checkbox' name='foo' value="3">
    ...         <label for="foo3">Three</label>
    ...         <br>
    ...         <input id="bar1" type='radio' name='bar' value="1">
    ...         <label for="bar1">First</label>
    ...         <input id="bar2" type='radio' name='bar' value="2">
    ...         <label for="bar2">Second</label>
    ...         <input id="bar3" type='radio' name='bar' value="3">
    ...         <label for="bar3">Third</label>
    ...         <br>
    ...         <select name="baz">
    ...             <option value="1">uno</option>
    ...             <option value="2">duos</option>
    ...             <option>tres</option>
    ...         </select>
    ...     </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo') # doctest: +ELLIPSIS
    GET /foo HTTP/1.1
    ...

    >>> checkboxes = browser.getControl(name='foo')
    >>> checkboxes
    <CheckboxListControl name='foo'>
    >>> checkboxes.getControl('One')
    <CheckboxControl name='foo' id='foo1' value='1'>
    >>> checkboxes.getControl(value='1')
    <CheckboxControl name='foo' id='foo1' value='1'>

    >>> radiobuttons = browser.getControl(name='bar')
    >>> radiobuttons
    <RadioListControl name='bar'>
    >>> radiobuttons.getControl('First')
    <RadioControl name='bar' id='bar1' value='1'>
    >>> radiobuttons.getControl(value='1')
    <RadioControl name='bar' id='bar1' value='1'>

    >>> listcontrol = browser.getControl(name='baz')
    >>> listcontrol
    <SelectListControl name='baz'>
    >>> listcontrol.getControl('uno')
    <SelectControl name='baz' value='1' selected='selected'>
    >>> listcontrol.getControl(value='1')
    <SelectControl name='baz' value='1' selected='selected'>

    >>> listcontrol.getControl('tres')
    <SelectControl name='baz' value='tres'>
    >>> listcontrol.getControl(value='tres')
    <SelectControl name='baz' value='tres'>

    """


def test_option_with_explicit_value_and_first_value_an_empty_string():
    """
    >>> app = YetAnotherTestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.add_response(b'''\
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <select name="baz">
    ...             <option value="">uno</option>
    ...             <option value="2" selected="selected">duos</option>
    ...             <option>tres</option>
    ...         </select>
    ...     </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo')
    >>> browser.getControl(name='baz').displayValue
    ['duos']
    """


def test_option_without_explicit_value():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <select name="baz">
    ...             <option value="1">uno</option>
    ...             <option value="2">duos</option>
    ...             <option>tres</option>
    ...         </select>
    ...     </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo') # doctest: +ELLIPSIS
    GET /foo HTTP/1.1
    ...
    >>> listcontrol = browser.getControl(name='baz')
    >>> listcontrol
    <SelectListControl name='baz'>
    >>> listcontrol.getControl('uno')
    <SelectControl name='baz' value='1' selected='selected'>
    >>> listcontrol.getControl(value='1')
    <SelectControl name='baz' value='1' selected='selected'>

    >>> listcontrol.getControl('uno').selected = True
    >>> listcontrol.value
    ['1']
    >>> listcontrol.displayValue
    ['uno']

    >>> listcontrol.getControl(value='2').selected = True
    >>> listcontrol.value
    ['2']
    >>> listcontrol.displayValue
    ['duos']

    >>> listcontrol.getControl('tres').selected = True
    >>> listcontrol.value
    ['tres']
    >>> listcontrol.displayValue
    ['tres']

    """


def test_multiselect_option_without_explicit_value():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <select name="baz" multiple>
    ...             <option>uno</option>
    ...             <option selected value="2">duos</option>
    ...             <option selected>tres</option>
    ...         </select>
    ...     </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo') # doctest: +ELLIPSIS
    GET /foo HTTP/1.1
    ...
    >>> listcontrol = browser.getControl(name='baz')
    >>> listcontrol
    <SelectListControl name='baz'>
    >>> listcontrol.getControl('uno')
    <SelectControl name='baz' value='uno'>
    >>> listcontrol.getControl('duos')
    <SelectControl name='baz' value='2' selected='selected'>
    >>> listcontrol.getControl('tres')
    <SelectControl name='baz' value='tres' selected='selected'>

    >>> listcontrol.value
    ['2', 'tres']

    >>> listcontrol.getControl(value='uno').selected = True
    >>> listcontrol.value
    ['uno', '2', 'tres']
    >>> listcontrol.displayValue
    ['uno', 'duos', 'tres']

    """


def test_subcontrols_can_be_selected_by_label_substring():
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''
    ... <html><body>
    ...     <form method='get' action='action'>
    ...         <select name="foo">
    ...             <option>one/two</option>
    ...             <option>three</option>
    ...         </select>
    ...     </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/foo') # doctest: +ELLIPSIS
    GET /foo HTTP/1.1
    ...
    >>> listcontrol = browser.getControl(name='foo')
    >>> listcontrol.getControl('one')
    <SelectControl name='foo' value='one/two' selected='selected'>

    """

UNICODE_TEST = u'\u4e2d\u6587\u7dad' # unicode in doctests is hard!

def test_non_ascii_in_input_field(self):
    """
    Test non-ascii chars in form postings.

    >>> app = YetAnotherTestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> body = u'''\
    ... <html><body>
    ...   <form action="." method="post" enctype="multipart/form-data">
    ...      <input name="text" type="text" value="{0}"/>
    ...      <button name="do" type="button">Do Stuff</button>
    ...   </form></body></html>
    ... '''.format(UNICODE_TEST).encode('utf-8')
    >>> length = unicode(len(body)).encode('latin1')
    >>> headers = [('Content-Type', 'text/html; charset="UTF-8"'),
    ...             ('Content-Length', length)]
    >>> app.add_response(body, headers=headers)
    >>> app.add_response(body, headers=headers)

    Getting a form with non-ascii form values should do something sane:

    >>> browser.open('http://localhost/')
    >>> non_ascii = browser.getControl(name='text').value
    >>> from .. import _compat
    >>> if _compat.PYTHON2:
    ...     non_ascii = non_ascii.decode('utf-8')
    >>> non_ascii == UNICODE_TEST
    True

    Posting a form with non-ascii values should give the server access to the
    real data:

    >>> browser.getControl("Do Stuff").click()
    >>> UNICODE_TEST in app.last_input
    True
"""


def test_post_encoding_doesnt_leak_between_requests(self):
    """
    >>> app = YetAnotherTestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> body = b'''\
    ... <html><body>
    ...   <form action="." method="POST" enctype="multipart/form-data">
    ...      <input name="text.other-e" type="text" value="somedata"/>
    ...      <button name="do" type="button">Do Stuff</button>
    ...   </form></body></html>
    ... '''
    >>> app.add_response(b'json')
    >>> app.add_response(body)
    >>> app.add_response(b'done')

    Post some JSON

    >>> browser.post('http://localhost/', '1', 'application/json')
    >>> browser.contents
    'json'

    then get a form and post it

    >>> browser.open('http://localhost/')
    >>> browser.getControl("Do Stuff").click()
    >>> browser.contents
    'done'

    The content_type of the last post should come from the form's enctype attr:

    >>> print(app.last_environ['CONTENT_TYPE'])
    multipart/form-data...
"""


def test_links_without_href(self):
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ... <a href="/foo">Foo</a>
    ... <a>Missing href</a>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...
    >>> browser.getLink(url='/foo').url
    'http://localhost/foo'
    """


def test_controls_without_value(self):
    """
    >>> app = TestApp()
    >>> browser = Browser(wsgi_app=app)
    >>> app.set_next_response(b'''\
    ... <html><body>
    ... <form action="." method="post">
    ... <label for="foo-field">Foo Label</label>
    ... <input type="text" id="foo-field" value="Foo"/>
    ... <button type="submit">Submit</button>
    ... </form>
    ... </body></html>
    ... ''')
    >>> browser.open('http://localhost/')
    GET / HTTP/1.1
    ...
    >>> browser.getControl('Foo Label').value
    'Foo'
    """


def test_suite():
    return doctest.DocTestSuite(
        checker=p01.testbrowser.tests.helper.checker,
        optionflags=doctest.NORMALIZE_WHITESPACE | doctest.ELLIPSIS)

# additional_tests is for setuptools "setup.py test" support
additional_tests = test_suite
