import * as kplus from 'cdk8s-plus-30';
import { Construct } from 'constructs';
import * as k8s from './imports/k8s';
import { PlonePDBOptions } from './pdb';
/**
 * Container image configuration options.
 */
export interface PloneImageOptions {
    /**
     * Container image name and tag.
     * @example 'plone/plone-backend:6.0.10'
     * @default 'plone/plone-backend:latest'
     */
    readonly image?: string;
    /**
     * Names of Kubernetes secrets for pulling private images.
     * @default []
     */
    readonly imagePullSecrets?: string[];
    /**
     * Image pull policy (Always, IfNotPresent, Never).
     * @default 'IfNotPresent'
     */
    readonly imagePullPolicy?: string;
}
/**
 * Configuration options for PloneDeployment.
 */
export interface PloneDeploymentOptions {
    /**
     * Container image configuration.
     * @default 'plone/plone-backend:latest'
     */
    readonly image?: PloneImageOptions;
    /**
     * Environment variables for the container.
     * Use cdk8s-plus-30 Env to define variables and sources.
     * @default - no additional environment variables
     */
    readonly environment?: kplus.Env;
    /**
     * Number of pod replicas to run.
     * @default 2
     */
    readonly replicas?: number;
    /**
     * CPU limit for the container.
     * @default '1000m'
     */
    readonly limitCpu?: string;
    /**
     * Memory limit for the container.
     * @default '1Gi'
     */
    readonly limitMemory?: string;
    /**
     * CPU request for the container.
     * @default '200m'
     */
    readonly requestCpu?: string;
    /**
     * Memory request for the container.
     * @default '300Mi'
     */
    readonly requestMemory?: string;
    /**
     * Container port number to expose.
     */
    readonly port: number;
    /**
     * Additional Kubernetes labels for the deployment.
     * @default - standard Plone labels only
     */
    readonly labels?: {
        [name: string]: string;
    };
    /**
     * Annotations to add to the Deployment metadata.
     * Common annotations include: deployment timestamps, change tracking, etc.
     * @example { 'deployment.kubernetes.io/revision': '1' }
     * @default - no additional annotations
     */
    readonly annotations?: {
        [name: string]: string;
    };
    /**
     * Annotations to add to the Pod template metadata.
     * Common annotations include: Prometheus scraping config, Istio config,
     * backup policies, logging configurations, etc.
     * @example { 'prometheus.io/scrape': 'true', 'prometheus.io/port': '8080' }
     * @default - no additional annotations
     */
    readonly podAnnotations?: {
        [name: string]: string;
    };
    /**
     * Additional container specification overrides.
     * Advanced use only - merges with generated container spec.
     * @default - undefined
     */
    readonly ploneContainer?: k8s.Container;
    /**
     * Sidecar containers to run alongside the main container.
     * @example [{ name: 'log-forwarder', image: 'fluentd:latest' }]
     * @default []
     */
    readonly sidecars?: k8s.Container[];
    /**
     * PodDisruptionBudget configuration for high availability.
     * If provided, creates a PDB with the specified constraints.
     * @default - no PDB created
     */
    readonly pdb?: PlonePDBOptions;
    /**
     * Liveness probe configuration for the container.
     * @default - undefined (no liveness probe)
     */
    livenessProbe?: k8s.Probe;
    /**
     * Readiness probe configuration for the container.
     * @default - undefined (no readiness probe)
     */
    readinessProbe?: k8s.Probe;
}
/**
 * PloneDeployment creates a Kubernetes Deployment for Plone containers.
 *
 * This is an internal construct used by the Plone class.
 * It creates a Deployment with configurable replicas, resources, probes,
 * and an optional PodDisruptionBudget.
 */
export declare class PloneDeployment extends Construct {
    constructor(scope: Construct, id: string, options: PloneDeploymentOptions);
}
